/* $Id: efosc_img_screen_flat.c,v 1.1 2009-07-08 13:53:39 cizzo Exp $
 *
 * This file is part of the EFOSC Data Reduction Pipeline
 * Copyright (C) 2009 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cizzo $
 * $Date: 2009-07-08 13:53:39 $
 * $Revision: 1.1 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <efosc_img_screen_flat_impl.h>
#include <efosc_dfs.h>
#include <efosc_utils.h>

#include <cpl.h>

static int efosc_img_screen_flat_create(cpl_plugin *);
static int efosc_img_screen_flat_exec(cpl_plugin *);
static int efosc_img_screen_flat_destroy(cpl_plugin *);

/**
 * @defgroup efosc_img_screen_flat efosc_recipe Recipe
 *
 * See recipe description for details.
 */

/**@{*/


/**
 * @brief    Build the list of available plugins, for this module. 
 *
 * @param    list    The plugin list
 *
 * @return   0 if everything is ok, -1 otherwise
 *
 * Create the recipe instance and make it available to the application 
 * using the interface. This function is exported.
 */

int cpl_plugin_get_info(cpl_pluginlist *list)
{
    cpl_recipe *recipe = cpl_calloc(1, sizeof *recipe );
    cpl_plugin *plugin = &recipe->interface;

    if (EFOSC_BINARY_VERSION != efosc_get_version_binary())
    {
        cpl_msg_error(cpl_func, 
              "I am efosc_img_screen_flat version %d, but I am linking "
              "against the EFOSC library version %d. "
              "This will not work. "
              "Please remove all previous installations "
              "of the " PACKAGE_NAME " and try again.",
              EFOSC_BINARY_VERSION, efosc_get_version_binary());
        return 1;
    }

    cpl_plugin_init(plugin,
                    CPL_PLUGIN_API,
                    EFOSC_BINARY_VERSION,
                    CPL_PLUGIN_TYPE_RECIPE,
                    efosc_img_screen_flat_name,
                    efosc_img_screen_flat_description_short,
                    efosc_img_screen_flat_description,
                    efosc_img_screen_flat_author,
                    efosc_img_screen_flat_email,
                    efosc_license,
                    efosc_img_screen_flat_create,
                    efosc_img_screen_flat_exec,
                    efosc_img_screen_flat_destroy);

    cpl_pluginlist_append(list, plugin);
    
    return 0;
}


/**
 * @brief    Setup the recipe options    
 *
 * @param    plugin  The plugin
 *
 * @return   0 if everything is ok
 *
 * Defining the command-line/configuration parameters for the recipe.
 */

static int efosc_img_screen_flat_create(cpl_plugin *plugin)
{
    cpl_recipe    *recipe;

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, 
                      "CPL error code is set (%s), "
                      "refusing to create recipe efosc_img_screen_flat", 
                      cpl_error_get_message());
        return 1;
    }
    
    /* 
     * Check that the plugin is part of a valid recipe 
     */

    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
        recipe = (cpl_recipe *)plugin;
    }
    else {
        return 1;
    }

    /* 
     * Create the parameters list in the cpl_recipe object 
     */

    recipe->parameters = cpl_parameterlist_new(); 
    
    efosc_img_screen_flat_define_parameters(recipe->parameters);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, 
                      "Could not create efosc_img_screen_flat parameters");
        return 1;
    }
        
    return 0;
}


/**
 * @brief    Execute the plugin instance given by the interface
 *
 * @param    plugin  the plugin
 *
 * @return   0 if everything is ok
 */

static int efosc_img_screen_flat_exec(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    cpl_errorstate initial_errorstate = cpl_errorstate_get();

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        cpl_msg_error(cpl_func, 
                      "CPL error code is set (%s), "
                      "refusing to execute recipe efosc_img_screen_flat", 
                      cpl_error_get_message());
        return 1;
    }
    
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
        recipe = (cpl_recipe *)plugin;
    }
    else {
        return 1;
    }

    if (recipe->frames == NULL) {
        cpl_msg_error(cpl_func, 
                      "Null frameset");
        return 1;
    }

    if (recipe->parameters == NULL) {
        cpl_msg_error(cpl_func, 
                      "Null parameter list");
        return 1;
    }

    efosc_begin(recipe->frames,
               efosc_img_screen_flat_description_short);

    efosc_img_screen_flat(recipe->frames, recipe->parameters);

    return efosc_end(recipe->frames, initial_errorstate);
}


/**
 * @brief    Destroy what has been created by the 'create' function
 *
 * @param    plugin  The plugin
 *
 * @return   0 if everything is ok
 */

static int efosc_img_screen_flat_destroy(cpl_plugin *plugin)
{
    cpl_recipe *recipe;
    
    if (cpl_plugin_get_type(plugin) == CPL_PLUGIN_TYPE_RECIPE) {
        recipe = (cpl_recipe *)plugin;
    }
    else {
        return -1;
    }

    cpl_parameterlist_delete(recipe->parameters); 

    return 0;
}

/**@}*/
