/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "eris_ifu_error.h"
#include "eris_ifu_utils.h"
#include "eris_ifu_functions.h"
#include "eris_ifu_dark_static.h"
#include "eris_ifu_dfs.h"
#include "eris_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_dark_static     IFU Dark Frame Processing Functions
 *
 * This module provides functions for processing dark frames, including master
 * dark creation, bad pixel map generation, noise estimation, and quality control
 * parameter calculation for IFU data.
 *
 * @par Synopsis:
 * @code
 *   #include "eris_ifu_dark_static.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Determine master dark and bad pixel maps
 * @param parlist           Input recipe parameters
 * @param darkImageList     Input HDRL dark images list
 * @param pdarkcollapse     Input collapsing method parameter for HDRL
 * @param masterDarkHdrlImg Output master dark HDRL image (pointer to image pointer)
 * @param qualityImage      Output data quality indicator image (pointer to image pointer)
 * @param masterBpm         Output master bad pixel map image (pointer to image pointer)
 * @param contribMap        Output contribution map showing number of contributing frames per pixel (pointer to image pointer)
 * @param bpm2dMask         Output bad pixel map from 2D analysis (pointer to mask pointer)
 * @param bpm3dMask         Output bad pixel map from 3D analysis (pointer to mask pointer)
 * @param qcParams          Quality control parameters property list
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * This function processes a list of dark frames to create a master dark image,
 * noise map, and bad pixel maps using both 2D and 3D analysis methods. It also
 * generates a data quality indicator (DQI) image that combines information from
 * saturated pixels, border pixels, and both bad pixel map methods.
 *
 * @note The output images and masks must be freed by the caller using appropriate
 *       deletion functions (hdrl_image_delete, cpl_image_delete, cpl_mask_delete).
 * @note Border pixels (4 pixels on each edge) are automatically flagged as bad pixels.
 * @note QC parameters are computed and added to qcParams if it is not NULL.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_dark_static(const cpl_parameterlist *parlist,
		hdrl_imagelist          *darkImageList,
		hdrl_parameter          *pdarkcollapse,
		hdrl_image              **masterDarkHdrlImg,
		cpl_image               **qualityImage,
		cpl_image               **masterBpm,
		cpl_image               **contribMap,
		cpl_mask                **bpm2dMask,
		cpl_mask                **bpm3dMask,
		cpl_propertylist        *qcParams)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	cpl_image *noiseImage = NULL;
	cpl_image *masterBpmImg = NULL;
	cpl_mask *borderSaturatedMask = NULL;
	const cpl_mask *masterBpmMask = NULL;
	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(darkImageList, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(pdarkcollapse, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(masterDarkHdrlImg, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qualityImage, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(masterBpm, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(contribMap, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(bpm2dMask, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(bpm3dMask, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qcParams, CPL_ERROR_NULL_INPUT);

	TRY
	{
		cpl_msg_info(cpl_func, "Generate master dark image");
		hdrl_image *tmpHdrlImg = NULL;
		hdrl_imagelist_collapse(darkImageList, pdarkcollapse,
				&tmpHdrlImg, contribMap);

		cpl_msg_info(cpl_func, "Generate noise image");
		noiseImage = eris_ifu_dark_noise(darkImageList);
		*masterDarkHdrlImg = hdrl_image_create(
				hdrl_image_get_image_const(tmpHdrlImg), noiseImage);
		eris_ifu_free_hdrl_image(&tmpHdrlImg);

		// fetch bad-pix-image: border pixels and saturated pixels are
		// set up-to-now
		borderSaturatedMask = cpl_mask_duplicate(
				hdrl_image_get_mask_const(*masterDarkHdrlImg));


		/* Create & apply bad-pixel-mask */
		eris_ifu_calc_bpm(parlist,
				REC_NAME_DARK,
				*masterDarkHdrlImg,       // needed for 2dbpm
				darkImageList,            // needed for 3dbpm
				bpm2dMask, bpm3dMask);   // return values

		// create quality type image
		(*qualityImage) = eris_ifu_dark_get_dqi(
				borderSaturatedMask, *bpm2dMask, *bpm3dMask);

		// fetch bad-pix-image: bpm2dMask and bpm3dMask are added now
		masterBpmMask = hdrl_image_get_mask_const(*masterDarkHdrlImg);

		masterBpmImg = cpl_image_new_from_mask(masterBpmMask);

		if (qcParams != NULL) {
			cpl_msg_info(cpl_func, "Generate QC parameters");
			eris_ifu_dark_qc(parlist, *masterDarkHdrlImg, darkImageList,
					masterBpmMask, *qualityImage, qcParams);
		}
	}
	CATCH
	{
		retVal = cpl_error_get_code();
		eris_ifu_free_mask(bpm2dMask);
		eris_ifu_free_mask(bpm3dMask);
	}

	(*masterBpm) = masterBpmImg;
	eris_ifu_free_image(&noiseImage);
	eris_ifu_free_mask(&borderSaturatedMask);
	eris_check_error_code("eris_ifu_dark_static");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Determine dark noise image from standard deviation across frames
 * @param imageList   Input HDRL dark images list
 * @return   Dark noise image, or NULL on error
 *
 * Calculates the pixel-by-pixel standard deviation across all input dark frames
 * to estimate the temporal noise characteristics. This noise image represents
 * the frame-to-frame variation at each pixel location.
 *
 * @note The returned image must be freed by the caller using cpl_image_delete().
 * @note Rejected pixels in the input images are included in the calculation.
 */
/*----------------------------------------------------------------------------*/
cpl_image *eris_ifu_dark_noise(hdrl_imagelist* imageList)
{
	cpl_image *noiseImage = NULL;
	cpl_size xDim,yDim,zDim;
	cpl_vector *data;
	int rejected;

	TRY
	{
		xDim = hdrl_imagelist_get_size_x(imageList);
		yDim = hdrl_imagelist_get_size_y(imageList);
		zDim = hdrl_imagelist_get_size(imageList);
		noiseImage = cpl_image_new(xDim, yDim, CPL_TYPE_DOUBLE);
		data = cpl_vector_new(zDim);
		for (int x = 1; x <= xDim; x++)
		{
			for (int y = 1; y <= yDim; y++)
			{
				for (int z = 0; z < zDim; z++)
				{
					cpl_vector_set(data, z, cpl_image_get(
						hdrl_image_get_image_const(
						hdrl_imagelist_get(imageList,z)), x, y, &rejected));
				}
				cpl_image_set(noiseImage, x, y, cpl_vector_get_stdev(data));
			}
		}
	}
	CATCH
	{
	}
	eris_ifu_free_vector(&data);
	eris_check_error_code("eris_ifu_dark_noise");
	return noiseImage;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Determine quality control parameters for master dark
 * @param parlist           Input recipe parameters
 * @param masterDarkHdrlImg Input HDRL master dark image
 * @param darkImageList     Input HDRL dark images list (raw frames)
 * @param masterBpm         Input master bad pixel map mask
 * @param qualityImage      Input data quality indicator image
 * @param qcParams          Output QC parameters property list
 * @return   CPL_ERROR_NONE on success, otherwise an error code
 *
 * Computes various quality control parameters for the master dark frame and
 * adds them to the provided property list. Parameters include:
 * - Number and fraction of bad pixels (total, saturated, 2D method, 3D method)
 * - Mean and standard deviation of the master dark (excluding bad pixels)
 * - Average and standard deviation of raw dark medians
 * - Fixed Pattern Noise (FPN) measured in a specified region
 * - Read-Out Noise (RON) measured in a specified region
 * - Individual RON measurements from consecutive frame differences
 *
 * @note The RON calculation uses the specified zone parameters from the input
 *       parameter list (qc_ron_xmin, qc_ron_xmax, qc_ron_ymin, qc_ron_ymax,
 *       qc_ron_hsize, qc_ron_nsamp).
 * @note The FPN calculation uses similar zone parameters (qc_fpn_*).
 * @note Border pixels (4 on each edge, total 2048*8 + 2040*8) are excluded
 *       from the bad pixel count.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code eris_ifu_dark_qc(
		const cpl_parameterlist *parlist,
		hdrl_image *masterDarkHdrlImg,
		hdrl_imagelist* darkImageList,
		const cpl_mask *masterBpm,
		const cpl_image *qualityImage,
		cpl_propertylist *qcParams)
{
	cpl_error_code retVal = CPL_ERROR_NONE;
	hdrl_image *qcMasterDarkHdrlImg = NULL;
	const cpl_image *masterDarkImage = NULL;
	cpl_image *tmpImg = NULL;
	cpl_vector *qcMedian = NULL;
	cpl_size nBadPixels = 0;
	cpl_size frameCnt;
	hdrl_value mean;
	double stdev;
	cpl_size zone_def[4];
	cpl_size hsize, nsamp;
	double qc_ron_val, qc_ron_err, qc_fpn_val;
	char *paramName;
	cpl_size sx = 0;
	cpl_size sy = 0;
	cpl_size npix = 0;
	double fracBadPix = 0;
	const int *qiData = NULL;
	cpl_size imgDim = 0;
	int satCnt = 0;
	int bp2dCnt = 0;
	int bp3dCnt = 0;

	cpl_ensure_code(parlist, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(masterDarkHdrlImg, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(darkImageList, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(masterBpm, CPL_ERROR_NULL_INPUT);
	cpl_ensure_code(qcParams, CPL_ERROR_NULL_INPUT);

	TRY
	{

		// count bad pixels but subtract "masked" border pixels
		nBadPixels = cpl_mask_count(masterBpm) - (2048*8 + 2040*8);

		//exit(0);
		sx = cpl_mask_get_size_x(masterBpm);
		sy = cpl_mask_get_size_y(masterBpm);

		npix = sx *  sy;
		fracBadPix = (double) ((int) nBadPixels) / (int) npix;

		eris_ifu_append_qc_int(qcParams,"DARK NBADPIX", (int) nBadPixels,
				"Total number of bad pixels but border pixels");
		eris_ifu_append_qc_float(qcParams, "DARK BPIXFRAC", fracBadPix,
				"Fraction of bad pixels to total");

		qiData = cpl_image_get_data_const(qualityImage);
		imgDim = cpl_image_get_size_x(qualityImage) *
				cpl_image_get_size_y(qualityImage);
		satCnt = 0;
		bp2dCnt = 0;
		bp3dCnt = 0;
		for (cpl_size ix=0; ix<imgDim; ix++) {
			if (qiData[ix] & ERIS_DQI_SAT) {
				satCnt++;
			}
			if (qiData[ix] & ERIS_DQI_BP_BPM2D) {
				bp2dCnt++;
			}
			if (qiData[ix] & ERIS_DQI_BP_BPM3D) {
				bp3dCnt++;
			}
		}
		eris_ifu_append_qc_int(qcParams,"DARK NBADPIXSAT", satCnt,
				"Saturated pixels");
		eris_ifu_append_qc_int(qcParams,"DARK NBADPIX2D", bp2dCnt,
				"Dark 2D bad pixels");

		eris_ifu_append_qc_int(qcParams,"DARK NBADPIX3D", bp3dCnt,
				"Dark 3D bad pixels");
		eris_ifu_append_qc_double(qcParams, "DARK BPIXFRAC2D",
				(double) ((int) bp2dCnt) / (int) npix,
				"Fraction of 2D bad pixels to total");

		eris_ifu_append_qc_double(qcParams, "DARK BPIXFRAC3D",
				(double) ((int) bp3dCnt) / (int) npix,
				"Fraction of 3D bad pixels to total");


		qcMasterDarkHdrlImg = hdrl_image_duplicate(masterDarkHdrlImg);

		eris_ifu_hdrl_image_reject_mask(qcMasterDarkHdrlImg, masterBpm);

		mean = hdrl_image_get_mean(qcMasterDarkHdrlImg);
		stdev = hdrl_image_get_stdev(qcMasterDarkHdrlImg);
		eris_ifu_free_hdrl_image(&qcMasterDarkHdrlImg);
		CHECK_ERROR_STATE();
		eris_ifu_append_qc_double(qcParams, "MASTERDARK MEAN", mean.data,
				"[ADU] Clean mean of master dark image");

		//	        eris_ifu_append_qc_double(qcParams, "MASTERDARK ERR", mean.error,
		//	            "Clean error of master dark image");
		eris_ifu_append_qc_double(qcParams, "MASTERDARK STDEV", stdev,
				"[ADU] Clean stdev of master dark image");

		frameCnt = hdrl_imagelist_get_size(darkImageList);

		qcMedian = cpl_vector_new(frameCnt);
		for (int i=0; i<frameCnt; i++) {
			cpl_vector_set(qcMedian, i,
					cpl_image_get_median(
							hdrl_image_get_image_const(
									hdrl_imagelist_get(darkImageList,i))));
		}

		eris_ifu_append_qc_double(qcParams,"DARKMED AVE",
				cpl_vector_get_mean(qcMedian), "[ADU] Average of raw darks medians");

		eris_ifu_append_qc_double(qcParams,"DARKMED STDEV",
				cpl_vector_get_stdev(qcMedian), "[ADU] Stdev Read Out Noise");
		eris_ifu_free_vector(&qcMedian);

		masterDarkImage = hdrl_image_get_image_const(masterDarkHdrlImg);
		/* -- FPN */
		zone_def[0] = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_fpn_xmin"));
		zone_def[1] = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_fpn_xmax"));
		zone_def[2] = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_fpn_ymin"));
		zone_def[3] = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_fpn_ymax"));
		hsize  = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_fpn_hsize"));
		nsamp = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_fpn_nsamp"));
		CHECK_ERROR_STATE();

		cpl_flux_get_noise_window(masterDarkImage, zone_def, hsize, nsamp,
				&qc_fpn_val, NULL);
		eris_ifu_append_qc_double(qcParams,"DARKFPN", qc_fpn_val,
				"Fixed Pattern Noise of combined frames");

		/* -- RON */
		zone_def[0] = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_ron_xmin"));
		zone_def[1] = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_ron_xmax"));
		zone_def[2] = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_ron_ymin"));
		zone_def[3] = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_ron_ymax"));
		hsize  = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_ron_hsize"));
		nsamp = cpl_parameter_get_int(
				cpl_parameterlist_find_const(parlist,
						"eris.eris_ifu_dark.qc_ron_nsamp"));
		CHECK_ERROR_STATE();

		cpl_flux_get_noise_window(masterDarkImage, zone_def, hsize,
				nsamp, &qc_ron_val, &qc_ron_err);
		eris_ifu_append_qc_double(qcParams,"RON", qc_ron_val,
				"[ADU] Read Out Noise");
		eris_ifu_append_qc_double(qcParams,"RONRMS", qc_ron_err,
				"[ADU] Error of Read Out Noise");
		/* --RONn */
		for (int i=0 ; i<frameCnt-1 ; i++) {
			tmpImg = cpl_image_subtract_create(
					hdrl_image_get_image_const(
							hdrl_imagelist_get(darkImageList,i)),
							hdrl_image_get_image_const(
									hdrl_imagelist_get(darkImageList,i+1))	);

			cpl_flux_get_noise_window(tmpImg, zone_def, hsize, nsamp,
					&qc_ron_val, NULL);
			qc_ron_val *= sqrt(1./2.);

			paramName = cpl_sprintf("RON%d",i+1);
			eris_ifu_append_qc_double(qcParams,paramName, qc_ron_val,
					"[ADU] Read Out Noise");

			eris_ifu_free_image(&tmpImg);
			eris_ifu_free_string(&paramName);
		}
		eris_ifu_append_qc_int(qcParams,"NRONS", (int) frameCnt-1,
				"Number of RON frames");

	}
	CATCH
	{
		retVal = cpl_error_get_code();
	}
	eris_check_error_code("eris_ifu_dark_qc");
	return retVal;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief    Create a data quality indicator (DQI) image from bad pixel masks
 * @param masterBpmMask   Master bad pixel mask (saturated and border pixels)
 * @param bpm2dMask       Bad pixel mask from 2D analysis method
 * @param bpm3dMask       Bad pixel mask from 3D analysis method
 * @return   DQI image with combined quality flags, or NULL on error
 *
 * Creates a data quality indicator image by combining information from multiple
 * bad pixel masks. The DQI image is an integer image where each pixel value is
 * a bitwise combination of quality flags:
 * - ERIS_DQI_SAT: Saturated pixels (from masterBpmMask)
 * - ERIS_DQI_BP: Border pixels (4-pixel border on all edges)
 * - ERIS_DQI_BP_BPM2D: Bad pixels identified by 2D analysis
 * - ERIS_DQI_BP_BPM3D: Bad pixels identified by 3D analysis
 *
 * @note The returned image must be freed by the caller using cpl_image_delete().
 * @note Multiple flags can be set for a single pixel if it fails multiple criteria.
 * @note Border pixels are explicitly set to ERIS_DQI_BP (4 pixels on each edge).
 */
/*----------------------------------------------------------------------------*/
cpl_image* eris_ifu_dark_get_dqi(
		const cpl_mask* masterBpmMask,
		const cpl_mask* bpm2dMask,
		const cpl_mask* bpm3dMask ) {

	cpl_image *dqi = NULL;
	cpl_image *tmpImg = NULL;

	TRY {
		dqi = cpl_image_new_from_mask(masterBpmMask);
		cpl_image_multiply_scalar(dqi, ERIS_DQI_SAT);
		for (cpl_size ix=1; ix <= ERIS_IFU_DETECTOR_SIZE; ix++) {
			for (cpl_size iy=1; iy <= 4; iy ++) {
				cpl_image_set(dqi,ix, iy, ERIS_DQI_BP);
			}
			for (cpl_size iy=ERIS_IFU_DETECTOR_SIZE-3;
					iy <= ERIS_IFU_DETECTOR_SIZE; iy ++) {
				cpl_image_set(dqi,ix, iy, ERIS_DQI_BP);
			}
		}
		for (cpl_size iy=1; iy <= ERIS_IFU_DETECTOR_SIZE; iy++) {
			for (cpl_size ix=1; ix <= 4; ix ++) {
				cpl_image_set(dqi,ix, iy, ERIS_DQI_BP);
			}
			for (cpl_size ix=ERIS_IFU_DETECTOR_SIZE-3;
					ix <= ERIS_IFU_DETECTOR_SIZE; ix ++) {
				cpl_image_set(dqi,ix, iy, ERIS_DQI_BP);
			}
		}

		tmpImg = cpl_image_new_from_mask(bpm2dMask);
		cpl_image_multiply_scalar(tmpImg, ERIS_DQI_BP_BPM2D);
		cpl_image_add(dqi, tmpImg);
		eris_ifu_free_image(&tmpImg);

		tmpImg = cpl_image_new_from_mask(bpm3dMask);
		cpl_image_multiply_scalar(tmpImg, ERIS_DQI_BP_BPM3D);
		cpl_image_add(dqi, tmpImg);
		eris_ifu_free_image(&tmpImg);

	}
	CATCH
	{
	}
	eris_check_error_code("eris_ifu_dark_get_dqi");
	return dqi;
}

/**@}*/
