/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef ERIS_IFU_FUNCTIONS_H
#define ERIS_IFU_FUNCTIONS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_ifu_functions    IFU Miscellaneous Functions
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <gsl/gsl_interp.h>
#include <gsl/gsl_spline.h>
#include <gsl/gsl_errno.h>
#include <cpl.h>
#include <hdrl.h>
#include "eris_ifu_constants.h"
#include "eris_ifu_dfs.h"

/*-----------------------------------------------------------------------------
                                   Define
 -----------------------------------------------------------------------------*/
#define LINE_EXPOSURE_CORRECTION 1
#define COLUMN_EXPOSURE_CORRECTION 2
#define COSMIC_RAY_EXPOSURE_CORRECTION 4
#define COSMIC_RAY_EXPOSURE_DETECTION 8

#define BAD_PIX  CPL_BINARY_1
#define GOOD_PIX CPL_BINARY_0

#define DBL_ZERO_TOLERANCE    1e-10

typedef enum {
    PD_SCIENCE,      // standard science products
    PD_AUXILLIARY,   // additional intermediate science products
    PD_ALL,          // all science products
    PD_DEBUG         // additional products for pipeline developers
} productDepthType;

struct stdParamStruct {
        ifsInstrument instrument;
        bool          line_corr;
        bool          col_corr;
        bool          crh_corr;
        bool          crh_det;
        int           rawImageCorrectionMask;
        productDepthType productDepth;
        hdrl_parameter *crh_detection;
};
/* Stack allocated object assign by value (copy) only */
#define stdParamStructInit {0,0,0,0,0,0,0,NULL}

#define GAUSS_PAR_RANGE_MAX 41
struct gaussParStruct {
        double x0;
        double sigma;
        double area;
        double offset;
        double peak;
        double mse;
        int errorCode;
        int range;
        double xdata[GAUSS_PAR_RANGE_MAX];
        double ydata[GAUSS_PAR_RANGE_MAX];
};

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/
ifsInstrument eris_ifu_get_instrument_frame(cpl_frame *frame);


hdrl_image * eris_ifu_raw_hdrl_image(const cpl_image *cplImage);

cpl_image* eris_ifu_calc_noise_map(const cpl_image *data,
                                   double gain,
                                   double readnoise);

cpl_error_code eris_ifu_add_badpix_border(cpl_image *data, cpl_boolean add_ones,
        cpl_image *dqi);

cpl_error_code eris_ifu_saturation_detection(cpl_image *data, cpl_image *dqi);

cpl_mask* eris_ifu_detect_crh(hdrl_image* image,
                              int exposureCorrectionMode,
                              hdrl_parameter *laCosmicParams,
                              bool maskImage);

hdrl_image* eris_ifu_load_exposure_file(const char *filename,
                                   int exposureCorrectionMode,
                                   cpl_image *dqi);

hdrl_image* eris_ifu_load_exposure_frame(const cpl_frame *frame,
                                   int exposureCorrectionMode,
                                   cpl_image *dqi);

hdrl_imagelist* eris_ifu_load_exposure_frameset(const cpl_frameset *frameset,
                                   int exposureCorrectionMode);

cpl_error_code eris_ifu_exposure_line_correction(cpl_image *image);

cpl_error_code eris_ifu_exposure_column_correction(cpl_image *image);

cpl_error_code eris_ifu_calc_bpm(const cpl_parameterlist *pl,
                                 const char              *recipe_name,
                                 hdrl_image              *master_img,
                                 const hdrl_imagelist    *imglist_on,
                                 cpl_mask                **bpm2dMask,
                                 cpl_mask                **bpm3dMask);

hdrl_image * eris_ifu_warp_polynomial_image(const hdrl_image *inImage,
        const cpl_polynomial *poly_u,
        const cpl_polynomial *poly_v);

cpl_error_code eris_ifu_add_std_params(cpl_parameterlist *pl,
        const char *recipename);

cpl_error_code eris_ifu_fetch_std_param(const cpl_parameterlist *parlist,
        const char *recipename,
        struct stdParamStruct *stdParams);

cpl_error_code eris_parlist_config_add_all_recipes(cpl_parameterlist *pl,
                                                   const char* recname);

cpl_error_code eris_parlist_config_add_bpm(cpl_parameterlist *pl,
                                           const char *recname);

cpl_error_code eris_parlist_config_add_flat(cpl_parameterlist *pl,
                                            const char *recname);

cpl_vector* eris_ifu_polyfit_1d(const cpl_vector *x,
                                const cpl_vector *y,
                                const int degree);

cpl_polynomial * eris_ifu_1d_polynomial_fit(
        int nPoints,
        double *xdata,
        double *ydata,
        int degree);

cpl_error_code eris_ifu_1d_interpolation(
        double *xIn,
        double *yIn,
        int     nIn,
        double *xOut,
        double *yOut,
        int     nOut,
        const int interType);

void remove_2nans(int size_in, double *xin, double *yin,
        int *size_out, double **xout, double **yout);

// now in eris_ifu_vector.h/.c
//double eris_ifu_vector_get_mean(const cpl_vector *vec);

double eris_ifu_image_get_mean(const cpl_image *img);

cpl_error_code eris_ifu_line_gauss_fit(const cpl_vector *yIn,
                                       int center,
                                       int range,
                                       struct gaussParStruct *gaussPar);

cpl_vector* eris_ifu_calc_centers_collapse_chunk(const cpl_image* img,
                                                      int chunk_center,
                                                      int height);

cpl_vector* eris_ifu_image_collapse(const cpl_image* img);

cpl_error_code eris_ifu_slitpos_gauss(const cpl_image *profile_x,
                                           double          *left_edge_pos,
                                           double          *right_edge_pos,
                                           int             llx,
                                           int             productDepth);
void eris_ifu_free_std_param(struct stdParamStruct * stdParams) ;

cpl_error_code eris_ifu_bpm_correction(hdrl_image *himg,
                                      hdrl_image *badPixelMaskImg);

double eris_ifu_auto_derot_corr(double alt, double rot); 
#endif
/**@}*/


