/* $Id$
 *
 * This file is part of the ERIS/NIX Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

 /*
 * $Author$
 * $Date$
 * $Rev$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                   Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>

#include "eris_nix_dfs.h"
#include "eris_nix_master_dark.h"
#include "eris_nix_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_master_dark      Utilities relating to MASTER_DARKs
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Create a new master_dark struct and initialise the contents
  @param    dark     The hdrl_image containing the dark
  @param    hot_bpm  The cpl_image containing the hot-pixel bpm
  @param    filename The name of the associated file, or NULL
  @param    plist    The associated property list
  @return   The new master_dark struct

  The function returns a pointer to a master_dark structure that is populated
  with copies of the given items.

  The object and its contents are deleted by a call to en_master_dark_delete.
 */
/*----------------------------------------------------------------------------*/

master_dark * en_master_dark_create(const hdrl_image * dark,
                                    const cpl_mask * hot_bpm,
                                    const char * filename,
                                    const cpl_propertylist * plist) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    /* check for NULL inputs */

    cpl_ensure(dark && hot_bpm, CPL_ERROR_NULL_INPUT, NULL);

    master_dark * result = cpl_malloc(sizeof(master_dark));
 
    result->dark = hdrl_image_duplicate(dark);
    result->hot_bpm = cpl_mask_duplicate(hot_bpm);
    if (filename) {
        result->filename = cpl_strdup(filename);
    } else {
        result->filename = NULL;
    }
    if (plist) {
        result->plist = cpl_propertylist_duplicate(plist);
    } else {
        result->plist = NULL;
    }

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Delete a 'master_dark' struct
  @param    target  The master_dark struct to be deleted

  The function deletes a master_dark struct and its contents. If the
  pointer is NULL then no action is taken but no error set.
 */
/*----------------------------------------------------------------------------*/

void en_master_dark_delete(master_dark * target) {
    
    if (target) {
        cpl_free(target->filename);
        hdrl_image_delete(target->dark);
        cpl_mask_delete(target->hot_bpm);
        cpl_propertylist_delete(target->plist);
        cpl_free(target);
    }
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Load a 'master_dark' struct from a frameset.
  @param    frameset  The list of input frames
  @param    tag       The frame tag to look for
  @param    used      The list of raw/calibration frames used for this product
  @return   Pointer to the master_dark struct

  The function loads a master_dark struct from a frameset.
 */
/*----------------------------------------------------------------------------*/

master_dark * en_master_dark_load_from_frameset(const cpl_frameset * frameset,
                                                const char * tag,
                                                cpl_frameset * used) {
    cpl_mask           * bpm = NULL;
    hdrl_image         * dark = NULL;
    const char         * filename = NULL;
    mef_extension_list * mef_extensions = NULL;
    cpl_propertylist   * plist = NULL;
    master_dark        * result = NULL;

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    /* check parameters */

    cpl_ensure(frameset, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(tag, CPL_ERROR_NULL_INPUT, NULL);
 
    /* Load elements from tagged file in frameset */

    const cpl_frame * target_frame = cpl_frameset_find_const(frameset, tag);
    enu_check(target_frame != NULL, CPL_ERROR_DATA_NOT_FOUND,
              "SoF has no file tagged %s", tag);

    filename = cpl_frame_get_filename(target_frame);

    /* Read the data in as an hdrl image, any mef extensions present */

    enu_himage_load_from_fits(filename, &dark, &mef_extensions,
                              &plist);
    enu_check_error_code("failed to read file %s", filename);

    bpm = enu_mef_extension_list_get_mask(mef_extensions,
                                          ERIS_NIX_HOT_BPM_EXTENSION);
    enu_check_error_code("failed to find bpm extension %s in %s",
                         ERIS_NIX_HOT_BPM_EXTENSION, filename);

    cpl_frameset_insert(used, cpl_frame_duplicate(target_frame));

    result = en_master_dark_create(dark, bpm, filename, plist);

 cleanup:

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        en_master_dark_delete(result);
        result = NULL;
    }

    cpl_mask_delete(bpm);
    hdrl_image_delete(dark);
    enu_mef_extension_list_delete(mef_extensions);
    cpl_propertylist_delete(plist);

    return result;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Create a test master_dark struct
  @param    nx    x dimension of detector
  @param    ny    y dimension of detector
  @param    dit   The DIT of the dark
  @param    ndit  The NDIT of the dark
  @param    mode  The detector read mode for the dark
  @return   The new master_dark struct

  The function returns a pointer to a dummy master_dark structure for use
  in simple tests.

  The object and its contents are deleted by a call to en_master_dark_delete.
 */
/*----------------------------------------------------------------------------*/

master_dark * en_master_dark_test(const cpl_size nx,
                                  const cpl_size ny,
                                  const double dit,
                                  const int ndit,
                                  const char * mode) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return NULL;

    cpl_image        * conf = NULL; 
    hdrl_image       * dark = NULL;
    cpl_image        * dark_data = NULL;
    cpl_image        * dark_error = NULL;
    cpl_mask         * hotbpm = NULL;
    cpl_propertylist * plist = NULL;
    master_dark      * result = NULL;

    /* make test master dark */
  
    dark_data = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(dark_data, 1, 1, nx, ny, 7.0);
    dark_error = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(dark_error, 1, 1, nx, ny, 3.0);
    dark = hdrl_image_create(dark_data, dark_error);
    hotbpm = cpl_mask_new(nx, ny);
    conf = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(conf, 1, 1, nx, ny, 5.0);

    plist = cpl_propertylist_new();
    cpl_propertylist_update_string(plist, CPL_DFS_PRO_CATG,
                                   ERIS_NIX_MASTER_DARK_IMG_PRO_CATG);
    cpl_propertylist_update_string(plist, "PRODCATG", "ANCILLARY.IMAGE");

    cpl_propertylist_update_double(plist, "ESO DET DIT", dit);
    cpl_propertylist_update_double(plist, "ESO DET NDIT", ndit);
    cpl_propertylist_update_string(plist, "ESO DET READ CURNAME", mode);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN ROT", 0);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN STRX", 1);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN STRY", 1);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN NX", nx);
    cpl_propertylist_update_int(plist, "ESO DET SEQ1 WIN NY", ny);

    cpl_propertylist_append_double(plist, "ESO QC READ NOISE", 6.2);
    cpl_propertylist_set_comment(plist, "ESO QC READ NOISE",
    		"Detector read out noise");
    cpl_propertylist_append_double(plist, "ESO QC READ NOISE VAR", 6.3);
    cpl_propertylist_set_comment(plist, "ESO QC READ NOISE VAR",
        		"variance on Detector read out noise");

    hdrl_value dark_median = hdrl_image_get_median(dark);
    cpl_propertylist_append_double(plist, "ESO QC DARK MED",
                                   (double) dark_median.data);
    cpl_propertylist_set_comment(plist, "ESO QC DARK MED",
            		"Dark median");

    hdrl_value dark_mean = hdrl_image_get_mean(dark);
    cpl_propertylist_append_double(plist, "ESO QC DARK MEAN",
                                   (double) dark_mean.data);
    cpl_propertylist_set_comment(plist, "ESO QC DARK MEAN",
                		"Dark mean");
    double dark_rms = hdrl_image_get_stdev(dark);
    cpl_propertylist_append_double(plist, "ESO QC DARK RMS", dark_rms);
    cpl_propertylist_set_comment(plist, "ESO QC DARK RMS",
                    		"Dark RMS");
    cpl_size nhot = cpl_mask_count(hotbpm); 
    cpl_propertylist_append_int(plist, "ESO QC NUMBER HOT PIXEL", (int)nhot);
    cpl_propertylist_set_comment(plist, "ESO QC NUMBER HOT PIXEL",
                        		"Number of detected hot pixels");
    cpl_propertylist_append_double(plist, "ESO QC PARTICLE_RATE", -1);
    cpl_propertylist_set_comment(plist, "ESO QC PARTICLE_RATE",
                            		"Particle rate");
    cpl_propertylist_append_double(plist, "ESO QC GAIN", 8.4);
    cpl_propertylist_set_comment(plist, "ESO QC GAIN",
                                		"Detector gain");

    result = en_master_dark_create(dark, hotbpm, NULL, plist);

    /* tidy up */

    cpl_image_delete(conf); 
    hdrl_image_delete(dark);
    cpl_image_delete(dark_data);
    cpl_image_delete(dark_error);
    cpl_mask_delete(hotbpm);
    cpl_propertylist_delete(plist);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        en_master_dark_delete(result);
        result = NULL;
    }

    return result;
}

/**@}*/
