/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "eris_nix_test_defs.h"
#include <eris_nix_dfs.h>
#include <eris_nix_utils.h>
#include <hdrl_utils.h>
#include <math.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup eris_nix_sky_subtract_test  Unit test of sky subtraction routines in
    eris_nix_utils
 *
 */
/*----------------------------------------------------------------------------*/

/*-----------------------------------------------------------------------------
                                       Prototypes
 -----------------------------------------------------------------------------*/

static void eris_nix_sky_subtract_test(void);

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static void eris_nix_sky_subtract_test(void)
{
    cpl_error_code          code;
    cpl_image             * data = NULL;
    cpl_image             * error = NULL;
    cpl_frameset          * frameset = NULL;
    cpl_propertylist      * plist = NULL;
    located_image         * target = NULL;
    located_imagelist     * target_data = NULL;
    cpl_image             * testconf = NULL;
    hdrl_image            * testimg = NULL;
    cpl_frameset          * used = NULL;

    frameset = cpl_frameset_new();
    used = cpl_frameset_new();

    /* create a test located_imagelist */

    data = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(data, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 20.0);
    error = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(error, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 4.0);
    testimg = hdrl_image_create(data, error);
    testconf = cpl_image_new(TEST_SIZE_X, TEST_SIZE_Y, CPL_TYPE_DOUBLE);
    cpl_image_fill_window(testconf, 1, 1, TEST_SIZE_X, TEST_SIZE_Y, 5.0);
    plist = cpl_propertylist_new();
    cpl_propertylist_prepend_double(plist, "ESO DET NDIT", 9.0);
    cpl_propertylist_prepend_double(plist, "ESO DET DIT", 15.0);
    cpl_propertylist_prepend_string(plist, "ESO DET READ CURNAME",
      "SLOW_UP_THE_RAMP");
    cpl_propertylist_prepend_string(plist, "ESO PRO CATG",
      "CALIB_JITTER");
    cpl_propertylist_prepend_double(plist, "MJD-OBS", 10.0);
    target_data = enu_located_imagelist_new(TEST_LIST_SIZE);

    /* located_image takes ownership of all input objects */ 

    for (cpl_size i=0; i<TEST_LIST_SIZE; i++) {
        hdrl_image * img_copy = hdrl_image_duplicate(testimg);
        cpl_image_fill_window(hdrl_image_get_image(img_copy), 1, 1, 
                              TEST_SIZE_X, TEST_SIZE_Y, (double) i * 20.0);
        cpl_image * conf_copy = cpl_image_duplicate(testconf);
        cpl_propertylist * plist_copy = cpl_propertylist_duplicate(plist);
        cpl_propertylist_set_double(plist_copy, "MJD-OBS",
                                    (double)i * 100.0 / (3600.0 * 24.0));

        cpl_frame * tframe = cpl_frame_new();
        cpl_frame_set_filename(tframe, "test.fits");
        cpl_frame_set_tag(tframe, "JITTER");
        cpl_frame_set_group(tframe, CPL_FRAME_GROUP_RAW);
        cpl_frame_set_level(tframe, CPL_FRAME_LEVEL_NONE);
        cpl_frame_set_type(tframe, CPL_FRAME_TYPE_IMAGE);
        cpl_frameset_insert(frameset, tframe);
        cpl_frame * tframe2 = cpl_frame_duplicate(tframe);
        cpl_frameset_insert(used, tframe2);
        cpl_frame * tframe3 = cpl_frame_duplicate(tframe); 

        located_image * newlimg = enu_located_image_new(img_copy,
                                                        NULL,
                                                        conf_copy,
                                                        NULL, NULL,
                                                        plist_copy,
                                                        NULL, NULL, NULL, NULL,
                                                        tframe3);
        enu_located_imagelist_insert(target_data, newlimg, i);
    }
    target = enu_located_image_duplicate(target_data->limages[0]);

    /* test enu_bracket_skys */

    cpl_vector * bracket_skys = enu_bracket_skys(NULL, 350.0, target_data,
                                                 CPL_TRUE);
    cpl_vector_delete(bracket_skys);
    code = cpl_error_get_code();  
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    bracket_skys = enu_bracket_skys(target, 0.0, target_data, CPL_TRUE);
    cpl_vector_delete(bracket_skys);
    code = cpl_error_get_code();  
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    bracket_skys = enu_bracket_skys(target, 350.0, NULL, CPL_TRUE);
    cpl_vector_delete(bracket_skys);
    code = cpl_error_get_code();  
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* answer should be a vector containing [0,1] */
    bracket_skys = enu_bracket_skys(target, 350.0, target_data, CPL_TRUE);
    code = cpl_error_get_code();  
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_vector * answer = cpl_vector_new(2);
    cpl_vector_set(answer, 0, (double)0);
    cpl_vector_set(answer, 1, (double)1);
    cpl_test_vector_abs(bracket_skys, answer, 0.01);
    cpl_vector_delete(bracket_skys);

    /* same with debug off */
    bracket_skys = enu_bracket_skys(target, 350.0, target_data, CPL_FALSE);
    code = cpl_error_get_code();  
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_vector_abs(bracket_skys, answer, 0.01);
    cpl_vector_delete(bracket_skys);
    cpl_vector_delete(answer);

    /* test enu_sky_backgrounds */

    cpl_imagelist * sky_conf_list = NULL;
    hdrl_imagelist * sky_himlist = NULL;
    cpl_size x_probe = -1;
    cpl_size y_probe = -1;
    code = enu_sky_backgrounds(NULL, "test", 0.0, target_data, target_data, 
                               x_probe, y_probe, &sky_himlist, 
                               &sky_conf_list);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = enu_sky_backgrounds("test", NULL, 0.0, target_data, target_data, 
                               x_probe, y_probe, &sky_himlist, &sky_conf_list);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = enu_sky_backgrounds("test", "test", 0.0, NULL, target_data, 
                               x_probe, y_probe, &sky_himlist, &sky_conf_list);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = enu_sky_backgrounds("test", "test", 0.0, target_data, NULL, 
                               x_probe, y_probe, &sky_himlist, &sky_conf_list);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = enu_sky_backgrounds("test", "bracket", 0.0, target_data,
                               target_data, x_probe, y_probe, &sky_himlist,
                               &sky_conf_list);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* test with no probe pixel set */
    code = enu_sky_backgrounds("collapse-median", "bracket", 350.0, target_data,
                               target_data,x_probe, y_probe, &sky_himlist,
                               &sky_conf_list);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    hdrl_image * testim = hdrl_imagelist_get(sky_himlist, 0);
    int reject = 0;
    double dval = 0.0;
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 10.0), DBL_EPSILON);
    testim = hdrl_imagelist_get(sky_himlist, 1);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 20.0), DBL_EPSILON);
    testim = hdrl_imagelist_get(sky_himlist, 2);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 40.0), DBL_EPSILON);
    testim = hdrl_imagelist_get(sky_himlist, 3);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 60.0), DBL_EPSILON);
    testim = hdrl_imagelist_get(sky_himlist, 4);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 70.0), DBL_EPSILON);

    cpl_imagelist_delete(sky_conf_list);
    hdrl_imagelist_delete(sky_himlist);

    /* same test with probe pixel */
    x_probe = 2;
    y_probe = 2;
    code = enu_sky_backgrounds("collapse-median", "bracket", 350.0, target_data,
                               target_data,x_probe, y_probe, &sky_himlist,
                               &sky_conf_list);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    testim = hdrl_imagelist_get(sky_himlist, 0);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 10.0), DBL_EPSILON);
    testim = hdrl_imagelist_get(sky_himlist, 1);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 20.0), DBL_EPSILON);
    testim = hdrl_imagelist_get(sky_himlist, 2);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 40.0), DBL_EPSILON);
    testim = hdrl_imagelist_get(sky_himlist, 3);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 60.0), DBL_EPSILON);
    testim = hdrl_imagelist_get(sky_himlist, 4);
    dval = cpl_image_get(hdrl_image_get_image(testim), 1, 1, &reject);
    cpl_test_leq(fabs(dval - 70.0), DBL_EPSILON);

    cpl_imagelist_delete(sky_conf_list);
    hdrl_imagelist_delete(sky_himlist);

    /* test enu_sky_subtract_limlist */

    code = enu_sky_subtract_limlist(NULL, "test", 0.0, NULL, 
                                    x_probe, y_probe, target_data);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = enu_sky_subtract_limlist("test", NULL, 0.0, NULL, 
                                    x_probe, y_probe, target_data);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = enu_sky_subtract_limlist("test", "test", 0.0, NULL, 
                                    x_probe, y_probe, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = enu_sky_subtract_limlist("collapse-median", "bracket", 350.0,
                                    target_data, x_probe, y_probe, 
                                    target_data);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    dval = cpl_image_get(hdrl_image_get_image(target_data->limages[0]->himage),
                         1, 1, &reject);
    cpl_test_leq(fabs(dval + 10.0), DBL_EPSILON);
    dval = cpl_image_get(hdrl_image_get_image(target_data->limages[1]->himage),
                         1, 1, &reject);
    cpl_test_leq(fabs(dval + 0.0), DBL_EPSILON);
    dval = cpl_image_get(hdrl_image_get_image(target_data->limages[2]->himage),
                         1, 1, &reject);
    cpl_test_leq(fabs(dval + 0.0), DBL_EPSILON);
    dval = cpl_image_get(hdrl_image_get_image(target_data->limages[3]->himage),
                         1, 1, &reject);
    cpl_test_leq(fabs(dval + 0.0), DBL_EPSILON);
    dval = cpl_image_get(hdrl_image_get_image(target_data->limages[4]->himage),
                         1, 1, &reject);
    cpl_test_leq(fabs(dval - 10.0), DBL_EPSILON);

    /* tidy up */

    cpl_image_delete(data);
    cpl_image_delete(error);
    cpl_frameset_delete(frameset);
    cpl_propertylist_delete(plist);
    enu_located_image_delete(target);
    enu_located_imagelist_delete(target_data);
    cpl_image_delete(testconf);
    hdrl_image_delete(testimg);
    cpl_frameset_delete(used);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of eris_nix_sky_subtract module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    eris_nix_sky_subtract_test();

    return cpl_test_end(0);

}

/**@}*/
