/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

//#include <libgen.h>
//#include <math.h>
#include <string.h>
#include <eris_utils.h>
//#include "casu_utils.h"
//#include "casu_mods.h"

//#include "eris_utils.h"
//#include "eris_nix_utils.h"
//#include "eris_pfits.h"
//#include "eris_dfs.h"
#include "eris_nix_dfs.h"
//#include "eris_nix_master_bpm.h"
//#include "eris_nix_master_dark.h"
//#include "eris_nix_gain_linearity.h"
//#include <hdrl.h>
#include "eris_nix_scired.h"
#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_nix_img_scired_description[] =
"This recipe removes the detector signature from "
ERIS_NIX_RAW_OBJECT_JITTER_DO_CATG",\n"
ERIS_NIX_RAW_SKY_JITTER_DO_CATG" and "
ERIS_NIX_RAW_STD_JITTER_DO_CATG" frames."
"This comprises dark subtraction,\n"
"linearization, flat-fielding, and association with the master \n"
"bad-pixel mask.\n"
"\n"
"Input files:\n"
"\n"
"  DO CATG                     Explanation             Req.  #Frames\n"
"  -------                     -----------             ---   -------\n"
"  "ERIS_NIX_RAW_OBJECT_JITTER_DO_CATG
               "               object jitter frames     N      0-n\n"
"  "ERIS_NIX_RAW_SKY_JITTER_DO_CATG
            "                  offset sky jitter        N      0-n\n"
"                              frames\n"
"  "ERIS_NIX_RAW_STD_JITTER_DO_CATG
            "                  standard jitter          N      0-n\n"
"                              frames\n"
"  "ERIS_NIX_RAW_OBJECT_LSS_JITTER_DO_CATG
                   "           object LSS jitter        N      0-n\n"
"                              frames\n"
"  "ERIS_NIX_RAW_SKY_LSS_JITTER_DO_CATG
                "              sky LSS jitter           N      0-n\n"
"                              frames\n"
"  "ERIS_NIX_RAW_STD_LSS_JITTER_DO_CATG
                "              standard LSS jitter      N      0-n\n"
"                              frames\n"
"  "ERIS_NIX_MASTER_DARK_IMG_PRO_CATG
             "                 a MASTER_DARK with       Y       1\n"
"                              matching detector \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_BPM_LAMP_PRO_CATG
                 "             a lamp MASTER_BPM        Y       1\n"
"  or\n"
"  "ERIS_NIX_MASTER_BPM_SKY_PRO_CATG
                "              a sky MASTER_BPM         Y       1\n"
"  "ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG
                         "     a lamp hifreq flat       N    0 or 1\n"
"                              matching observing \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_FLAT_SKY_HIFREQ_PRO_CATG
                        "      a sky hifreq flat        N    0 or 1\n"
"                              matching observing \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_FLAT_LAMP_LOFREQ_PRO_CATG
                         "     a lamp hifreq flat       N    0 or 1\n"
"                              matching observing \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_FLAT_SKY_LOFREQ_PRO_CATG
                        "      a sky hifreq flat        N    0 or 1\n"
"                              matching observing \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_FLAT_TWILIGHT_LOFREQ_PRO_CATG"\n"
"                              a twilight lofreq flat   N    0 or 1\n"
"                              matching observing \n"
"                              configuration (preferred\n"
"                              over the LAMP version\n"
"                              below)\n"
"  "ERIS_NIX_GAIN_PRO_CATG
           "                   DETMON gain information  N    0 or 1\n"
"                              e.g. in file \n"
"                              detmon_ir_lg_gain_table.fits\n"
"  "ERIS_NIX_COEFFS_CUBE_PRO_CATG
             "                 DETMON linearity curves  N    0 or 1\n"
"                              e.g. in file \n"
"                              detmon_ir_coeffs_cube.fits\n"
"  "ERIS_NIX_NL_BPM_PRO_CATG
           "                   DETMON non-linear bpm    N    0 or 1\n"
"                              e.g. in file \n"
"                              detmon_ir_lg_bpm.fits\n"
"\n"
"Output files:\n"
"\n"
"  DO CATG                     Explanation \n"
"  -------                     ----------- \n"
"  "ERIS_NIX_CAL_DET_OBJECT_JITTER_PRO_CATG
                       "       Calibrated results.\n"
"  or "ERIS_NIX_CAL_DET_SKY_JITTER_PRO_CATG"\n"
"  or "ERIS_NIX_CAL_DET_STD_JITTER_PRO_CATG"\n"
"  or "ERIS_NIX_CAL_DET_OBJECT_LSS_JITTER_PRO_CATG"\n"
"  or "ERIS_NIX_CAL_DET_SKY_LSS_JITTER_PRO_CATG"\n"
"  or "ERIS_NIX_CAL_DET_STD_LSS_JITTER_PRO_CATG"\n"
"\n"
"  The output will be a FITS file named 'cal_det_<inputname>.fits',\n"
"  with extensions:\n"
"  - DATA                      the result data.\n"
"  - ERR                       the result error plane.\n"
"  - DQ                        the result data quality plane.\n"
"  - CONFIDENCE                the result confidence plane.\n"
"\n";

#define RECIPE_NAME "eris_nix_img_scired"
#define CONTEXT "eris."RECIPE_NAME
/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_nix_img_scired, ERIS_BINARY_VERSION,
                  "John Lightfoot",
                  PACKAGE_BUGREPORT, "2017",
                  "Remove detector signature from frames",
                  eris_nix_img_scired_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_img_scired_fill_parameterlist(
  cpl_parameterlist * self) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();


    hdrl_parameter    * catalogue_defaults = NULL;
    cpl_parameterlist * catalogue_parlist = NULL;
    cpl_parameter* p;

    /*------------------------------------------------------------------------*/
    /*                           CALDET PARAMETERS                            */
    /*------------------------------------------------------------------------*/
    p = cpl_parameter_new_enum(CONTEXT".fill-rejected",
                               CPL_TYPE_STRING,
                               "how to treat reject pixel",
                               CONTEXT,
                               "set_value", 3,
                               "set_value",
                               "set_NaN",
                               "noop");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fill-rejected");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".fill-value",
                                CPL_TYPE_DOUBLE,
                                "value to use in 'set_value' case",
                                CONTEXT, 0.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fill-value");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Flag to decide whether to apply corrected CD matrices to data */

    p = cpl_parameter_new_value(CONTEXT".cd_matrix_modify",
    		CPL_TYPE_BOOL,
			"TRUE to insert corrected CD-matrix",
			CONTEXT, CPL_TRUE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cd-matrix-modify");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* coords of pixel to be used for diagnostics during reduction */

    p = cpl_parameter_new_value(CONTEXT".x_probe", CPL_TYPE_INT,
                                "x coord of diagnostic pixel",
                                CONTEXT, -1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "x-probe");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".y_probe", CPL_TYPE_INT,
                                "y coord of diagnostic pixel",
                                CONTEXT, -1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "y-probe");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_range(CONTEXT".collapse_cube", CPL_TYPE_INT,
       		"Collapse cube if DET.FRAM.FORMAT is cube. Possible values: "
       		"0, no collapse; 1, mean collapse; 2, median collapse",
   			CONTEXT, 0, 0, 2);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "collapse-cube");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /*------------------------------------------------------------------------*/
    /*                           SKYSUB PARAMETERS                            */
    /*------------------------------------------------------------------------*/
    /* generate the general parameter list for the catalogue */

    catalogue_defaults = hdrl_catalogue_parameter_create(20, 3.0, CPL_TRUE,
                         10.0, CPL_TRUE, 64, 2.0, 3.0,
                         ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);
    catalogue_parlist = hdrl_catalogue_parameter_create_parlist(CONTEXT,
                        "catalogue", catalogue_defaults);

    /* add the subset of parameters to be used */

    for (p = cpl_parameterlist_get_first(catalogue_parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(catalogue_parlist)) {

        const char * pname = cpl_parameter_get_name(p);
        if (strstr(pname, "min-pixels") ||
            strstr(pname, "threshold") ||
            strstr(pname, "mesh-size") ||
            strstr(pname, "threshold") ||
            strstr(pname, "smooth-gauss-fwhm")) {
            cpl_parameter * duplicate = cpl_parameter_duplicate(p);
            cpl_parameterlist_append(self, duplicate);
        }
    }





    /* now the recipe-specific parameters */

    /* generate the general parameter list for the catalogue */
        eris_nix_catalogue_param_set(CONTEXT, self, 20, 3.0, CPL_TRUE,
        		5.0, CPL_TRUE, 64, 2.0, 3.0, ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);

    p = cpl_parameter_new_enum(CONTEXT".sky-source", CPL_TYPE_STRING,
                               "data to be used for calculation of sky "
                               "background", CONTEXT,
                               "auto", 3, "auto", "target", "offset");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-source");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_enum(CONTEXT".sky-selector", CPL_TYPE_STRING,
                               "method for selecting sky frames",
                               CONTEXT, "bracket", 1, "bracket");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-selector");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_enum(CONTEXT".sky-method", CPL_TYPE_STRING,
                               "method for combining sky frames",
                               CONTEXT, "collapse-median", 2,
                               "collapse-median", "median-median");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-method");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_range(CONTEXT".sky-bracket-time", CPL_TYPE_DOUBLE,
                               "2 * max.time between target and sky measurement",
                               CONTEXT, 1800.0, 60.0, 18000.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "sky-bracket-time");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);


    p = cpl_parameter_new_value(CONTEXT".esoformat",
                                CPL_TYPE_BOOL,
                                "TRUE to output MEF file conforming to ESO standard",
                                CONTEXT, CPL_TRUE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "esoformat");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* coords of pixel to be used for diagnostics during reduction */

    p = cpl_parameter_new_value(CONTEXT".x_probe", CPL_TYPE_INT,
                                "x coord of diagnostic pixel",
                                CONTEXT, -1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "x-probe");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".y_probe", CPL_TYPE_INT,
                                "y coord of diagnostic pixel",
                                CONTEXT, -1);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "y-probe");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);


    
    
    /*------------------------------------------------------------------------*/
    /*                           WCS PARAMETERS                            */
    /*------------------------------------------------------------------------*/
    
    
    
    /* add the subset of parameters to be used */

    for (p = cpl_parameterlist_get_first(catalogue_parlist);
         p != NULL;
         p = cpl_parameterlist_get_next(catalogue_parlist)) {

        const char * pname = cpl_parameter_get_name(p);
        if (strstr(pname, "minmax") == NULL) {
            cpl_parameter * duplicate = cpl_parameter_duplicate(p);
            cpl_parameterlist_append(self, duplicate);
        }
    }

    /* Flag to decide how we get the astrometric standard star information.
       If "none", then use the local catalogues specified in the sof. If not,
       then use one of the selection of catalogues available from CDS -
       only gaia has sufficient accuracy */

    p = cpl_parameter_new_enum(CONTEXT".cdssearch_astrom",
                               CPL_TYPE_STRING,
                               "CDS astrometric catalogue",
                               CONTEXT,
                               "none", 3, "none", "2MASS", "gaiaedr3");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cdssearch-astrom");
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".pixel_radius",
                                CPL_TYPE_DOUBLE,
                                "Max. distance between object and catalogue "
                                "entry for association (pixels)",
                                CONTEXT, 5.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pixel-radius");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".strict_classification",
                                CPL_TYPE_BOOL,
                                "TRUE if objects to match must be classified "
                                "stellar and round",
                                CONTEXT, CPL_TRUE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "strict-classification");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);
    
    
    
    p = cpl_parameter_new_value(CONTEXT".edges-trim",
                                  CPL_TYPE_STRING,
  								"Trim image edges in the source detection: "
  								"[txmin,tymin,txmax,tymax]. t means trim "
  								"the user can trim differently from the min and "
  								"max image edge values along X and Y axes",
                                  CONTEXT, "30,30,30,30");
      cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "edges-trim");
      cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
      cpl_parameterlist_append(self, p);
    
    
    
    /*------------------------------------------------------------------------*/
    /*                           PHOTOM PARAMETERS                            */
    /*------------------------------------------------------------------------*/
    
    
    p = cpl_parameter_new_enum(CONTEXT".cdssearch_photom",
                                CPL_TYPE_STRING,
                                "CDS photometric catalogue",
                                CONTEXT,
                                "none", 2, "none", "2mass");
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cdssearch_photom");
     cpl_parameterlist_append(self, p);

     /* parameter list for  object catalogue generation */

     hdrl_parameter_delete(catalogue_defaults);
     cpl_parameterlist_delete(catalogue_parlist);
     catalogue_defaults = hdrl_catalogue_parameter_create(4, 2.5, CPL_TRUE,
                          10.0, CPL_TRUE, 64, 2.0, 3.0,
                          ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);
     catalogue_parlist = hdrl_catalogue_parameter_create_parlist(CONTEXT,
                         "catalogue", catalogue_defaults);
     for (p = cpl_parameterlist_get_first(catalogue_parlist);
          p != NULL;
          p = cpl_parameterlist_get_next(catalogue_parlist)) {

         const char * pname = cpl_parameter_get_name(p);
         if (strstr(pname, "minmax") == NULL) {
             cpl_parameter * duplicate = cpl_parameter_duplicate(p);
             cpl_parameterlist_append(self, duplicate);
         }
     }

     /* parameters for object-catalogue matching algorithm */

 /*  commented out for now. This method is not reliable at present,
     better to insist on a good wcs calibration then match single
     standard with brightest object within pixel-radius*/
 /*
     p = cpl_parameter_new_value(CONTEXT".assoc_radius",
                                 CPL_TYPE_DOUBLE,
                                 "Max distance between object and catalogue "
                                 "entry for association (arcsec)",
                                 CONTEXT, 0.5);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "assoc-radius");
     cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
     cpl_parameterlist_append(self, p);

     p = cpl_parameter_new_value(CONTEXT".isolation_radius",
                                 CPL_TYPE_DOUBLE,
                                 "Min distance between object and nearest "
                                 "neighbour for it to be 'isolated' (arcsec)",
                                 CONTEXT, 1.0);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "isolation-radius");
     cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
     cpl_parameterlist_append(self, p);
 */

     p = cpl_parameter_new_value(CONTEXT".pixel_radius",
                                 CPL_TYPE_DOUBLE,
                                 "Max. distance between object and catalogue "
                                 "entry for association (pixels)",
                                 CONTEXT, 5.0);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pixel-radius");
     cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
     cpl_parameterlist_append(self, p);

     p = cpl_parameter_new_range(CONTEXT".minphotom",
                                 CPL_TYPE_INT,
                                 "Min number of matched stars for photometric "
                                 "calibration", CONTEXT, 1, 1, 100000);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "minphotom");
     cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
     cpl_parameterlist_append(self, p);

     p = cpl_parameter_new_value(CONTEXT".magerrcut",
                                 CPL_TYPE_DOUBLE,
                                 "Matched stars with magnitude error above "
                                 "this cutoff will not be used.",
                                 CONTEXT, 0.5);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "magerrcut");
     cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
     cpl_parameterlist_append(self, p);


     p = cpl_parameter_new_value(CONTEXT".esoformat",
                                 CPL_TYPE_BOOL,
                                 "TRUE to output MEF file conforming to ESO standard",
                                 "background, catalogue components",
                                 CONTEXT, CPL_FALSE);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "esoformat");
     cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
     cpl_parameterlist_append(self, p);
     
     
     /*------------------------------------------------------------------------*/
     /*                       HDRL-STACK PARAMETERS                            */
     /*------------------------------------------------------------------------*/
     /* The lower rejection threshold to be used during stacking */
 /*
     p = cpl_parameter_new_range(CONTEXT".stk_lthr", CPL_TYPE_DOUBLE,
                                 "Low rejection threshold", CONTEXT,
                                 5.0, 1.0e-6, 1.0e10);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "stk-lthr");
     cpl_parameterlist_append(self, p);
 */
     /* The upper rejection threshold to be used during stacking */
 /*
     p = cpl_parameter_new_range(CONTEXT".stk_hthr", CPL_TYPE_DOUBLE,
                                 "Upper rejection threshold", CONTEXT,
                                 5.0, 1.0e-6, 1.0e10);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "stk-hthr");
     cpl_parameterlist_append(self, p);
 */
     /* Interpolation method */

     p = cpl_parameter_new_enum(CONTEXT".interpolation_method",
                                CPL_TYPE_STRING, "The interpolation method",
                                CONTEXT, "lanczos", 6, "nearest",
                                "linear", "quadratic", "renka", "drizzle",
                                "lanczos");
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI,
                             "interpolation-method");
     cpl_parameterlist_append(self, p);

     /* Loop distance */

     p = cpl_parameter_new_value(CONTEXT".loop_distance", CPL_TYPE_INT,
                                 "maximum pixel offset taken into account",
                                 CONTEXT, 1);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "loop-distance");
     cpl_parameterlist_append(self, p);

     /* Kernel size */

     p = cpl_parameter_new_value(CONTEXT".kernel_size", CPL_TYPE_INT,
                                 "(Lanczos method) size of kernel in pixels",
                                 CONTEXT, 2);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "kernel-size");
     cpl_parameterlist_append(self, p);

     /* Critical radius */

     p = cpl_parameter_new_value(CONTEXT".critical_radius", CPL_TYPE_DOUBLE,
                                 "(Renka method) distance beyond which weights "
                                 "set to 0",
                                 CONTEXT, 5.0);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "critical-radius");
     cpl_parameterlist_append(self, p);

     /* Drizzle frac parameters */

     p = cpl_parameter_new_value(CONTEXT".pix_frac_x", CPL_TYPE_DOUBLE,
                                 "(Drizzle method) percentage of flux "
                                 "to drizzle from original to target pixel",
                                 CONTEXT, 50.0);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pix-frac-x");
     cpl_parameterlist_append(self, p);
     p = cpl_parameter_new_value(CONTEXT".pix_frac_y", CPL_TYPE_DOUBLE,
                                 "(Drizzle method) percentage of flux "
                                 "to drizzle from original to target pixel",
                                 CONTEXT, 50.0);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "pix-frac-y");
     cpl_parameterlist_append(self, p);

     /* generate the general parameter list for the catalogue */

     hdrl_parameter_delete(catalogue_defaults);
     cpl_parameterlist_delete(catalogue_parlist);
     catalogue_defaults = hdrl_catalogue_parameter_create(4, 2.5, CPL_TRUE,
                          10.0, CPL_TRUE, 64, 2.0, 3.0,
                          ERIS_NIX_SATURATION_DEFAULT, HDRL_CATALOGUE_ALL);
     catalogue_parlist = hdrl_catalogue_parameter_create_parlist(CONTEXT,
                         "catalogue", catalogue_defaults);

     /* add the subset of parameters to be used */

     for (p = cpl_parameterlist_get_first(catalogue_parlist);
          p != NULL;
          p = cpl_parameterlist_get_next(catalogue_parlist)) {

         const char * pname = cpl_parameter_get_name(p);
         if (strstr(pname, "minmax") == NULL) {
             cpl_parameter * duplicate = cpl_parameter_duplicate(p);
             cpl_parameterlist_append(self, duplicate);
         }
     }

     p = cpl_parameter_new_value(CONTEXT".debug-data",
                                 CPL_TYPE_BOOL, "true to save interim results",
                                 CONTEXT, CPL_FALSE);
     cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "debug-data");
     cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
     cpl_parameterlist_append(self, p);
  
  
    /* tidy up */

    hdrl_parameter_delete(catalogue_defaults);
    cpl_parameterlist_delete(catalogue_parlist);
    

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int eris_nix_img_scired(cpl_frameset * frameset,
                            const cpl_parameterlist * parlist) {

	/* preliminary check to handle cube format: in that case skip following data
	 * reduction */
	cpl_frame* frame = NULL;
	frame = cpl_frameset_find(frameset,ERIS_NIX_RAW_OBJECT_JITTER_DO_CATG);
	cpl_boolean is_format_single = CPL_TRUE;
	if(frame != NULL) {
		cpl_propertylist* plist = NULL;
		const char* fname = cpl_frame_get_filename(frame);

		plist = cpl_propertylist_load(fname, 0);
		const char* key = "ESO DET FRAM FORMAT";
		if (cpl_propertylist_has(plist, key)) {
			const char* format = cpl_propertylist_get_string(plist, key);
			if(strcmp(format, "cube") == 0) {
				is_format_single = CPL_FALSE;
			}
		}

	}

	eris_nix_scired_cal_det(frameset, parlist, RECIPE_NAME, CONTEXT);
	if(is_format_single) {
		eris_nix_scired_skysub(frameset, parlist, RECIPE_NAME, CONTEXT);
		eris_nix_scired_cal_wcs(frameset, parlist, RECIPE_NAME, CONTEXT);
		eris_nix_scired_cal_phot(frameset, parlist, RECIPE_NAME, CONTEXT);
		eris_nix_scired_hdrl_stack(frameset, parlist, RECIPE_NAME, CONTEXT);
	}
	
    return (int) cpl_error_get_code();
}
