/* $Id$
 *
 * This file is part of the ERIS Pipeline
 * Copyright (C) 2017 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author$
 * $Date$
 * $Revision$
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

//#include <libgen.h>
//#include <math.h>
//#include <string.h>

//#include "casu_utils.h"
//#include "casu_mods.h"

//#include "eris_utils.h"
//#include "eris_nix_utils.h"
//#include "eris_pfits.h"
//#include "eris_dfs.h"
#include "eris_nix_dfs.h"
//#include "eris_nix_master_bpm.h"
//#include "eris_nix_master_dark.h"
//#include "eris_nix_gain_linearity.h"

#include "eris_nix_scired.h"
//#include <hdrl.h>

#include <cpl.h>

/*-----------------------------------------------------------------------------
                            Static variables
 -----------------------------------------------------------------------------*/

static const char eris_nix_pupil_description[] =
"This recipe performs QC on "
ERIS_NIX_RAW_PUPIL_LAMP_OPEN_DO_CATG",\n"
ERIS_NIX_RAW_PUPIL_LAMP_MASK_DO_CATG" and "
ERIS_NIX_RAW_PUPIL_SKY_OPEN_DO_CATG" and "
ERIS_NIX_RAW_PUPIL_SKY_MASK_DO_CATG" and "
ERIS_NIX_RAW_PUPIL_BKG_OPEN_DO_CATG" and "
ERIS_NIX_RAW_PUPIL_BKG_MASK_DO_CATG" and "
ERIS_NIX_RAW_PUPIL_DARK_OPEN_DO_CATG" and"
ERIS_NIX_RAW_PUPIL_DARK_MASK_DO_CATG" frames. "
"This comprises dark subtraction,\n"
"linearization, flat-fielding, association with the master \n"
"bad-pixel mask, and correction of image rotation.\n"
"\n"
"Input files:\n"
"\n"
"  DO CATG                     Explanation             Req.  #Frames\n"
"  -------                     -----------             ---   -------\n"
"  "ERIS_NIX_RAW_PUPIL_LAMP_DO_CATG
            "                  pupil lamp frames        N      0-n\n"
"  "ERIS_NIX_RAW_PUPIL_SKY_DO_CATG
            "                   pupil sky frames         N      0-n\n"
"  "ERIS_NIX_RAW_PUPIL_BKG_DO_CATG
            "                   pupil bkg frames         N      0-n\n"
"  "ERIS_NIX_RAW_PUPIL_DARK_DO_CATG
            "                  pupil dark frames        N      0-n\n"

"  "ERIS_NIX_MASTER_DARK_IMG_PRO_CATG
                 "             a MASTER_DARK with       Y       1\n"
"                              matching detector \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_BPM_LAMP_PRO_CATG
                 "             a lamp MASTER_BPM        Y       1\n"
"  or\n"
"  "ERIS_NIX_MASTER_BPM_SKY_PRO_CATG
                "              a sky MASTER_BPM         Y       1\n"
"  "ERIS_NIX_MASTER_FLAT_LAMP_HIFREQ_PRO_CATG
                         "     a lamp hifreq flat       N    0 or 1\n"
"                              matching observing \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_FLAT_SKY_HIFREQ_PRO_CATG
                        "      a sky hifreq flat        N    0 or 1\n"
"                              matching observing \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_FLAT_LAMP_LOFREQ_PRO_CATG
                         "     a lamp hifreq flat       N    0 or 1\n"
"                              matching observing \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_FLAT_SKY_LOFREQ_PRO_CATG
                        "      a sky hifreq flat        N    0 or 1\n"
"                              matching observing \n"
"                              configuration\n"
"  "ERIS_NIX_MASTER_FLAT_TWILIGHT_LOFREQ_PRO_CATG"\n"
"                              a twilight lofreq flat   N    0 or 1\n"
"                              matching observing \n"
"                              configuration (preferred\n"
"                              over the LAMP version\n"
"                              below)\n"
"  "ERIS_NIX_GAIN_PRO_CATG
           "                   DETMON gain information  N    0 or 1\n"
"                              e.g. in file \n"
"                              detmon_ir_lg_gain_table.fits\n"
"  "ERIS_NIX_COEFFS_CUBE_PRO_CATG
             "                 DETMON linearity curves  N    0 or 1\n"
"                              e.g. in file \n"
"                              detmon_ir_coeffs_cube.fits\n"
"  "ERIS_NIX_NL_BPM_PRO_CATG
           "                   DETMON non-linear bpm    N    0 or 1\n"
"                              e.g. in file \n"
"                              detmon_ir_lg_bpm.fits\n"
"  "ERIS_NIX_WCS_REFINE_PRO_CATG
            "                  NIX wcs refinement file  N    0 or 1\n"
"                              with updated CD-matrix\n"
"                              to correct image \n"
"                              rotation\n"
"\n"
"Output files:\n"
"\n"
"  DO CATG                     Explanation \n"
"  -------                     ----------- \n"
"  PREF_OPEN, PREF_JHK, PREF_CROSS, PREF_SAM (PREF=PUPIL_LAMP, PUPIL_SKY_PUPIL_DARK or PUPIL_BKG)"
                       "       Calibrated results (detector signature removed).\n"
"\n"
"  The output will be a FITS file named 'cal_det_<inputname>.fits',\n"
"  with extensions:\n"
"  - DATA                      the result data.\n"
"  - ERR                       the result error plane.\n"
"  - DQ                        the result data quality plane.\n"
"  - CONFIDENCE                the result confidence plane.\n"
"\n";

#define RECIPE_NAME "eris_nix_pupil"
#define CONTEXT "eris."RECIPE_NAME
/*-----------------------------------------------------------------------------
                            Private function prototypes
 -----------------------------------------------------------------------------*/

cpl_recipe_define(eris_nix_pupil, ERIS_BINARY_VERSION,
                  "Andrea Modigliani, John Lightfoot",
                  PACKAGE_BUGREPORT, "2017",
                  "Remove detector signature from PUPIL frames",
                  eris_nix_pupil_description);

/*-----------------------------------------------------------------------------
                                Function code
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Setup the recipe options    
  @param    self  the non-NULL parameterlist to fill
  @return   CPL_ERROR_NONE iff everything is ok

  Defining the command-line/configuration parameters for the recipe.
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code eris_nix_pupil_fill_parameterlist(
  cpl_parameterlist * self) {

    if (cpl_error_get_code() != CPL_ERROR_NONE) return cpl_error_get_code();

    cpl_parameter* p;

    p = cpl_parameter_new_enum(CONTEXT".fill-rejected",
                               CPL_TYPE_STRING,
                               "how to treat reject pixel",
                               CONTEXT, 
                               "set_value", 3,
                               "set_value",
                               "set_NaN",
                               "noop");
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fill-rejected");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    p = cpl_parameter_new_value(CONTEXT".fill-value",
                                CPL_TYPE_DOUBLE,
                                "value to use in 'set_value' case",
                                CONTEXT, 0.0);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "fill-value");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    /* Flag to decide whether to apply corrected CD matrices to data */

    p = cpl_parameter_new_value(CONTEXT".cd_matrix_modify",
                                CPL_TYPE_BOOL,
                                "TRUE to insert corrected CD-matrix",
                                CONTEXT, CPL_FALSE);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "cd-matrix-modify");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    // coords of pixel to be used for diagnostics during reduction
    eris_nix_pixel_coord_diagnostic_param_set(CONTEXT, self);

    p = cpl_parameter_new_range(CONTEXT".collapse_cube", CPL_TYPE_INT,
    		"May collapse cube if DET.FRAM.FORMAT is cube. Possible values: "
    		"0, no collapse; 1, mean collapse; 2, median collapse;"
    		"3, weighted mean",
			CONTEXT, 0, 0, 3);
    cpl_parameter_set_alias(p, CPL_PARAMETER_MODE_CLI, "collapse-cube");
    cpl_parameter_disable(p, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, p);

    return 0;
}


/*----------------------------------------------------------------------------*/
/**
  @brief    Interpret the command line options and execute the data processing
  @param    frameset   the frames list
  @param    parlist    the parameters list
  @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/

static int eris_nix_pupil(cpl_frameset * frameset,
                            const cpl_parameterlist * parlist) {

	eris_nix_scired_cal_det(frameset, parlist, RECIPE_NAME, CONTEXT);

    return (int) cpl_error_get_code();
}
