/*                                                                           a
 *   This file is part of the ESO X-shooter Pipeline                         *
 *   Copyright (C) 2006 European Southern Observatory                        *
 *                                                                           *
 *   This library is free software; you can redistribute it and/or modify    *
 *   it under the terms of the GNU General Public License as published by    *
 *   the Free Software Foundation; either version 2 of the License, or       *
 *   (at your option) any later version.                                     *
 *                                                                           *
 *   This program is distributed in the hope that it will be useful,         *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of          *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           *
 *   GNU General Public License for more details.                            *
 *                                                                           *
 *   You should have received a copy of the GNU General Public License       *
 *   along with this program; if not, write to the Free Software             *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA    *
 *                                                                           */

/*
 * $Author: dsosnows $
 * $Date: 2013-08-06 14:49:56 $
 * $Revision: 1.7 $
 * $Name: not supported by cvs2svn $
 */

/*----------------------------------------------------------------------------
 Includes
 ----------------------------------------------------------------------------*/

#include <espdr_CCD.h>
#include <espdr_parameters.h>
#include <espdr_utils.h>
#include <espdr_msg.h>
#include <espdr_instrument.h>

/*----------------------------------------------------------------------------
 Functions
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 AMO added
 @brief allocate memory and initialise the CCD_geom parameters structure
 @param parameters      parameters list
 @param CCD_geom_frame  CCD_geometry structure
 @return   CCD geometry structure
 */
/*----------------------------------------------------------------------------*/

espdr_CCD_geometry* espdr_CCD_geom_init(cpl_parameterlist *parameters,
                                        cpl_frame* CCD_geom_frame) {
    
    const char *CCD_geom_filename = NULL;
    //cpl_error_code my_error;
    espdr_CCD_geometry* CCD_geom = NULL;
    
    /* check input function parameters */
    cpl_ensure(parameters != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(CCD_geom_frame != NULL, CPL_ERROR_NULL_INPUT, NULL);
    
    CCD_geom = (espdr_CCD_geometry *)cpl_malloc( sizeof(espdr_CCD_geometry));
    

    /* From the CCD parameters tables set, we take always the first readable */
    CCD_geom_filename = cpl_frame_get_filename(CCD_geom_frame);
    //my_error = espdr_read_CCD_parameters(CCD_geom_filename, CCD_geom);
    espdr_read_CCD_parameters(CCD_geom_filename, CCD_geom);

    espdr_msg("Input parameters for %s:", CCD_geom->instrument);
    
    /* TODO: AMO: this printing should be optional */
    espdr_parameters_CCD_geometry_print(CCD_geom);
    
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        espdr_msg_error("Error during CCD_geom parameters initialisation");
        return NULL;
    }
    
    return CCD_geom;
}

/*---------------------------------------------------------------------------*/
/**
 @brief    Read the CCD geometry parameters from a FITS table
 @param    filename	the filename of the FITS table
 @param    CCD_geom	the data structure with all CCD geometry parameters
 @return   CPL_ERROR_NONE iff OK
 
 TODO: do one really need to distinguish between extensions? Isn't the
 geometry the same?
 --> DSO:   yes, it can happen, that extensions differ.
 */
/*--------------------------------------------------------------------------*/
cpl_error_code espdr_read_CCD_parameters(const char *filename,
                                         espdr_CCD_geometry *CCD_geom) {
	
    int i, j, exts_row_nb, outputs_row_nb;
    int ext_no, ext_nb, out_nb_x, out_nb_y, total_output_nb;
    int CCD_nx, CCD_ny, image_nx, image_ny;
    int out_x, out_y, out_no, out_phys_no;
    int raw_nx, raw_ny, real_nx, real_ny;
    int pscan_nx, pscan_ny, oscan_nx, oscan_ny;
    int llx, lly, urx, ury;
    const char *read_inst;
	
	cpl_table *CCD_geom_read_exts = NULL;
	cpl_table *CCD_geom_read_outputs = NULL;
	
	espdr_ensure(filename == NULL, CPL_ERROR_NULL_INPUT, 
			   "The input filename is NULL");
	
	espdr_msg("CCD geometry filename: %s", filename);
	
    CCD_geom_read_exts = cpl_table_load(filename, 1, 0);
    espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(),
                 "Error loading FITS table (extensions part) with CCD geometry parameters");
    
    read_inst = cpl_table_get_string(CCD_geom_read_exts, COL_NAME_INST_NAME, 0);
    strcpy(CCD_geom->instrument, read_inst);
    exts_row_nb = cpl_table_get_nrow(CCD_geom_read_exts);
    
    ext_nb = cpl_table_get_int(CCD_geom_read_exts, COL_NAME_EXT_NB, 0, NULL);
    CCD_geom->ext_nb = ext_nb;
    CCD_geom->exts = (espdr_CCD_extension *) cpl_malloc
                                        (ext_nb * sizeof(espdr_CCD_extension));
    
    CCD_nx = cpl_table_get_int(CCD_geom_read_exts, COL_NAME_CCD_NX, 0, NULL);
    CCD_ny = cpl_table_get_int(CCD_geom_read_exts, COL_NAME_CCD_NY, 0, NULL);
    image_nx = cpl_table_get_int(CCD_geom_read_exts, COL_NAME_IMAGE_NX, 0, NULL);
    image_ny = cpl_table_get_int(CCD_geom_read_exts, COL_NAME_IMAGE_NY, 0, NULL);
    
    total_output_nb = 0;
    for (i = 0; i < exts_row_nb; i++) {
        read_inst = cpl_table_get_string(CCD_geom_read_exts,
                                         COL_NAME_INST_NAME, i);
        espdr_ensure((strcmp(read_inst, CCD_geom->instrument) != 0),
                     CPL_ERROR_ILLEGAL_INPUT,
                     "The data is from wrong instrument: %s", read_inst);
        
        CCD_geom->exts[i].ext_nb = cpl_table_get_int(CCD_geom_read_exts,
                                                     COL_NAME_EXT_NB,
                                                     i, NULL);
        espdr_ensure(CCD_geom->exts[i].ext_nb != ext_nb,
                     CPL_ERROR_ILLEGAL_INPUT,
                     "Extensions number in row %d differs from the one in the first row", i);
        
        CCD_geom->exts[i].ext_no = cpl_table_get_int(CCD_geom_read_exts,
                                                     COL_NAME_EXT_NO,
                                                     i, NULL);
        out_nb_x = cpl_table_get_int(CCD_geom_read_exts,
                                     COL_NAME_OUT_NB_X, i, NULL);
        CCD_geom->exts[i].out_nb_x = out_nb_x;
        out_nb_y = cpl_table_get_int(CCD_geom_read_exts,
                                     COL_NAME_OUT_NB_Y, i, NULL);
        CCD_geom->exts[i].out_nb_y = out_nb_y;
        CCD_geom->exts[i].coll_or = cpl_table_get_int(CCD_geom_read_exts,
                                                      COL_NAME_COLL_OR,
                                                      i, NULL);
        CCD_geom->exts[i].corr_type = cpl_table_get_int(CCD_geom_read_exts,
                                                        COL_NAME_CORR_TYPE,
                                                        i, NULL);
        CCD_geom->exts[i].rot_angle = cpl_table_get_int(CCD_geom_read_exts,
                                                        COL_NAME_ROT_ANGLE,
                                                        i, NULL);
        CCD_geom->exts[i].ver_flip = cpl_table_get_int(CCD_geom_read_exts,
                                                       COL_NAME_VER_FLIP,
                                                       i, NULL);
        CCD_geom->exts[i].hor_flip = cpl_table_get_int(CCD_geom_read_exts,
                                                       COL_NAME_HOR_FLIP,
                                                       i, NULL);
        CCD_geom->exts[i].pxl_size = cpl_table_get_double(CCD_geom_read_exts,
                                                          COL_NAME_PXL_SIZE,
                                                          i, NULL);
        CCD_geom->exts[i].ver_flip = cpl_table_get_int(CCD_geom_read_exts,
                                                       COL_NAME_VER_FLIP,
                                                       i, NULL);
        
        CCD_geom->exts[i].CCD_nx = cpl_table_get_int(CCD_geom_read_exts,
                                                     COL_NAME_CCD_NX,
                                                     i, NULL);
        espdr_ensure(CCD_geom->exts[i].CCD_nx != CCD_nx,
                     CPL_ERROR_ILLEGAL_INPUT,
                     "The CCD x size in row %d differs from the first row", i);
        CCD_geom->exts[i].CCD_ny = cpl_table_get_int(CCD_geom_read_exts,
                                                     COL_NAME_CCD_NY,
                                                     i, NULL);
        espdr_ensure(CCD_geom->exts[i].CCD_ny != CCD_ny,
                     CPL_ERROR_ILLEGAL_INPUT,
                     "The CCD y size in row %d differs from the first row", i);
        CCD_geom->exts[i].image_nx = cpl_table_get_int(CCD_geom_read_exts,
                                                       COL_NAME_IMAGE_NX,
                                                       i, NULL);
        espdr_ensure(CCD_geom->exts[i].image_nx != image_nx,
                     CPL_ERROR_ILLEGAL_INPUT,
                     "The image x size in row %d differs from the first row",
                     i);
        CCD_geom->exts[i].image_ny = cpl_table_get_int(CCD_geom_read_exts,
                                                       COL_NAME_IMAGE_NY,
                                                       i, NULL);
        espdr_ensure(CCD_geom->exts[i].image_ny != image_ny,
                     CPL_ERROR_ILLEGAL_INPUT,
                     "The image y size in row %d differs from the first row",
                     i);
        
        CCD_geom->exts[i].outputs =
        (espdr_CCD_output **) cpl_malloc(out_nb_x *
                                         sizeof(espdr_CCD_output *));
        for (j = 0; j < out_nb_x; j++) {
            CCD_geom->exts[i].outputs[j] =
            (espdr_CCD_output *) cpl_malloc(out_nb_y *
                                            sizeof(espdr_CCD_output));
        }
        
        total_output_nb = total_output_nb + (out_nb_x * out_nb_y);
    }
    
    CCD_geom->total_output_nb = total_output_nb;
    
    
	CCD_geom_read_outputs = cpl_table_load(filename, 2, 0);
	espdr_ensure(cpl_error_get_code() != CPL_ERROR_NONE, cpl_error_get_code(),
                 "Error loading FITS table (outputs part) with CCD geometry parameters");
    
	outputs_row_nb = cpl_table_get_nrow(CCD_geom_read_outputs);
    for (i = 0; i < outputs_row_nb; i++) {
        read_inst = cpl_table_get_string(CCD_geom_read_outputs,
                                         COL_NAME_INST_NAME, i);
        espdr_ensure((strcmp(read_inst, CCD_geom->instrument) != 0),
                     CPL_ERROR_ILLEGAL_INPUT,
                     "The data is from wrong instrument: %s", read_inst);
        
        ext_no = cpl_table_get_int(CCD_geom_read_outputs,
                                   COL_NAME_EXT_NO, i, NULL);
        espdr_ensure(ext_no >= ext_nb, CPL_ERROR_ILLEGAL_INPUT,
                    "The extension number is bigger the the number of extensions");
        
        out_x = cpl_table_get_int(CCD_geom_read_outputs,
                                  COL_NAME_OUT_X, i, NULL);
        espdr_ensure(out_x >= out_nb_x, CPL_ERROR_ILLEGAL_INPUT,
                    "The output x coordinate is bigger the the number of outputs in X direction");
        
        out_y = cpl_table_get_int(CCD_geom_read_outputs,
                                  COL_NAME_OUT_Y, i, NULL);
        espdr_ensure(out_y >= out_nb_y, CPL_ERROR_ILLEGAL_INPUT,
                    "The output y coordinate is bigger the the number of outputs in Y direction");
        
        CCD_geom->exts[ext_no].outputs[out_x][out_y].ext_no = ext_no;
        CCD_geom->exts[ext_no].outputs[out_x][out_y].out_x = out_x;
        CCD_geom->exts[ext_no].outputs[out_x][out_y].out_y = out_y;
        
        out_no = cpl_table_get_int(CCD_geom_read_outputs,
                                   COL_NAME_OUT_NO, i, NULL);
        espdr_ensure(out_no >= total_output_nb, CPL_ERROR_ILLEGAL_INPUT,
                    "The output number is bigger the the number of outputs");
        CCD_geom->exts[ext_no].outputs[out_x][out_y].out_no = out_no;
        
        out_phys_no = cpl_table_get_int(CCD_geom_read_outputs,
                                        COL_NAME_OUT_PHYS_NO, i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].out_phys_no = out_phys_no;
        
        /* Raw image size X and Y */
        raw_nx = cpl_table_get_int(CCD_geom_read_outputs,
                                   COL_NAME_RAW_NX, i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].raw_nx = raw_nx;
        raw_ny = cpl_table_get_int(CCD_geom_read_outputs,
                                   COL_NAME_RAW_NY, i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].raw_ny = raw_ny;
        
        /* Real image size X and Y */
        real_nx = cpl_table_get_int(CCD_geom_read_outputs,
                                    COL_NAME_REAL_NX, i, NULL);
        espdr_ensure(real_nx > raw_nx, CPL_ERROR_ILLEGAL_INPUT,
                    "The real X size can't be greater then the raw X size");
        CCD_geom->exts[ext_no].outputs[out_x][out_y].real_nx = real_nx;
        real_ny = cpl_table_get_int(CCD_geom_read_outputs,
                                    COL_NAME_REAL_NY, i, NULL);
        espdr_ensure(real_nx > raw_nx, CPL_ERROR_ILLEGAL_INPUT,
                    "The real Y size can't be greater then the raw Y size");
        CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ny = real_ny;
        
        /* Real image coordinates: llx, lly, urx, ury */
        llx = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_REAL_LLX, i, NULL);
        espdr_ensure((llx >= real_nx) || (llx < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The real llx coordinate should be within 1 and %d",
                     real_nx);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].real_llx = llx;
        
        lly = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_REAL_LLY, i, NULL);
        espdr_ensure((lly >= real_ny) || (lly < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The real lly coordinate should be within 1 and %d",
                     real_ny);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].real_lly = lly;
        
        urx = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_REAL_URX, i, NULL);
        espdr_ensure((urx > real_nx) || (urx < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The real urx coordinate (%d) should be within 1 and %d",
                     urx, real_nx);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].real_urx = urx;
        
        ury = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_REAL_URY, i, NULL);
        espdr_ensure((ury > real_ny) || (ury < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The real ury coordinate (%d) should be within 1 and %d",
                     ury, real_ny);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].real_ury = ury;
        
        /* Prescan size X and Y */
        pscan_nx = cpl_table_get_int(CCD_geom_read_outputs,
                                     COL_NAME_PSCAN_NX, i, NULL);
        // This check is true iff coll_or != corr_type
        //espdr_ensure(pscan_nx > (raw_nx - real_nx), CPL_ERROR_ILLEGAL_INPUT,
        //             "pscan X size can't be greater than the difference between the raw X size and real X size");
        CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_nx = pscan_nx;
        pscan_ny = cpl_table_get_int(CCD_geom_read_outputs,
                                     COL_NAME_PSCAN_NY, i, NULL);
        // This check is true iff coll_or == corr_type
        //espdr_ensure(pscan_ny > (raw_ny - real_ny), CPL_ERROR_ILLEGAL_INPUT,
        //             "pscan Y size can't be greater than the difference between the raw Y size and real X size, %d, %d, %d",
        //             pscan_ny, raw_ny, real_ny);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_ny = pscan_ny;
        
        /* Prescan coordinates: llx, lly, urx, ury */
        llx = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_PSCAN_LLX, i, NULL);
        //espdr_ensure((llx >= pscan_nx) || (llx < 1), CPL_ERROR_ILLEGAL_INPUT,
        //             "The pscan llx coordinate should be within 1 and %d",
        //             pscan_nx);
        espdr_ensure(llx < 1, CPL_ERROR_ILLEGAL_INPUT,
                     "The pscan llx coordinate should be > 0");
        CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_llx = llx;
        
        lly = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_PSCAN_LLY, i, NULL);
        espdr_ensure((lly >= pscan_ny) || (lly < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The pscan lly coordinate should be within 1 and %d",
                     pscan_ny);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_lly = lly;
        
        urx = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_PSCAN_URX, i, NULL);
        //espdr_ensure((urx > pscan_nx) || (urx < 1), CPL_ERROR_ILLEGAL_INPUT,
        //             "The real urx coordinate should be within 1 and %d",
        //             pscan_nx);
        espdr_ensure(urx < 1, CPL_ERROR_ILLEGAL_INPUT,
                     "The pscan urx coordinate (%d) should be > 0", urx);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_urx = urx;
        
        ury = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_PSCAN_URY, i, NULL);
        espdr_ensure((ury > pscan_ny) || (ury < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The pscan ury coordinate (%d) should be within 1 and %d",
                     ury, pscan_ny);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].pscan_ury = ury;
        
        /* Overscan size X and Y */
        oscan_nx = cpl_table_get_int(CCD_geom_read_outputs,
                                     COL_NAME_OSCAN_NX, i, NULL);
        // This check is true iff coll_or != corr_type
        //espdr_ensure(oscan_nx > (raw_nx - real_nx), CPL_ERROR_ILLEGAL_INPUT,
        //             "oscan X size can't be greater than the difference between the raw X size and real X size");
        CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_nx = oscan_nx;
        oscan_ny = cpl_table_get_int(CCD_geom_read_outputs,
                                     COL_NAME_OSCAN_NY, i, NULL);
        // This check is true iff coll_or == corr_type
        //espdr_ensure(oscan_ny > (raw_ny - real_ny), CPL_ERROR_ILLEGAL_INPUT,
        //             "oscan Y size can't be greater than the difference between the raw Y size and real X size");
        CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_ny = oscan_ny;
        
        /* Overscan coordinates: llx, lly, urx, ury */
        llx = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_OSCAN_LLX, i, NULL);
        espdr_ensure((llx >= oscan_nx) || (llx < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The oscan llx coordinate should be within 1 and %d",
                     oscan_nx);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_llx = llx;
        
        lly = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_OSCAN_LLY, i, NULL);
        espdr_ensure((lly >= oscan_ny) || (lly < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The oscan lly coordinate should be within 1 and %d",
                     oscan_ny);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_lly = lly;
        
        urx = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_OSCAN_URX, i, NULL);
        espdr_ensure((urx > oscan_nx) || (urx < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The oscan urx coordinate (%d) should be within 1 and %d",
                     urx, oscan_nx);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_urx = urx;
        
        ury = cpl_table_get_int(CCD_geom_read_outputs,
                                COL_NAME_OSCAN_URY, i, NULL);
        espdr_ensure((ury > oscan_ny) || (ury < 1), CPL_ERROR_ILLEGAL_INPUT,
                     "The oscan ury coordinate (%d) should be within 1 and %d",
                     ury, oscan_ny);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].oscan_ury = ury;
        
        /* Perpendicular prescan & overscan sizes & coordinates, not used */
        CCD_geom->exts[ext_no].outputs[out_x][out_y].ppscan_nx =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_PPSCAN_NX,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].ppscan_ny =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_PPSCAN_NY,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].ppscan_llx =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_PPSCAN_LLX,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].ppscan_lly =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_PPSCAN_LLY,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].ppscan_urx =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_PPSCAN_URX,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].ppscan_ury =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_PPSCAN_URY,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].poscan_nx =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_POSCAN_NX,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].poscan_ny =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_POSCAN_NY,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].poscan_llx =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_POSCAN_LLX,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].poscan_lly =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_POSCAN_LLY,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].poscan_urx =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_POSCAN_URX,
                                                              i, NULL);
        CCD_geom->exts[ext_no].outputs[out_x][out_y].poscan_ury =
                                            cpl_table_get_int(CCD_geom_read_outputs,
                                                              COL_NAME_POSCAN_URY,
                                                              i, NULL);
    }
    
	cpl_table_delete(CCD_geom_read_exts);
	cpl_table_delete(CCD_geom_read_outputs);
	
	return (cpl_error_get_code());
}



/*---------------------------------------------------------------------------*/
/**
 @brief     Print the CCD geometry parameters
 @param     CCD_geometry    structure
 @return    CPL_ERROR_NONE iff OK
 */
/*---------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CCD_geometry_print(espdr_CCD_geometry *CCD_geom) {
	
	
    //int i, j, k;
    int i;
    
    /*
    espdr_msg("\tCCD geometry:");
    espdr_msg("\t\tExtensions:");
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        espdr_msg("\t\tExt number: %d", CCD_geom->exts[i].ext_no);
        espdr_msg("\t\t\tNumber of outputs in X: %d", CCD_geom->exts[i].out_nb_x);
        espdr_msg("\t\t\tNumber of outputs in Y: %d", CCD_geom->exts[i].out_nb_y);
        espdr_msg("\t\t\tCollumn orientation: %d", CCD_geom->exts[i].coll_or);
        espdr_msg("\t\t\tCorrection type: %d", CCD_geom->exts[i].corr_type);
        espdr_msg("\t\t\tRotation angle: %d", CCD_geom->exts[i].rot_angle);
        espdr_msg("\t\t\tVertical flip: %d", CCD_geom->exts[i].ver_flip);
        espdr_msg("\t\t\tHorizontal flip: %d", CCD_geom->exts[i].hor_flip);
        espdr_msg("\t\t\tPixel size: %.2f", CCD_geom->exts[i].pxl_size);
        espdr_msg("\t\t\tCCD X size: %d", CCD_geom->exts[i].CCD_nx);
        espdr_msg("\t\t\tCCD Y size: %d", CCD_geom->exts[i].CCD_ny);
        espdr_msg("\t\t\tRaw image X size: %d", CCD_geom->exts[i].image_nx);
        espdr_msg("\t\t\tRaw image Y size: %d", CCD_geom->exts[i].image_ny);
    
        espdr_msg("\t\tOutputs:");
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            for (k = 0; k < CCD_geom->exts[i].out_nb_y; k++) {
                espdr_msg("\t\tOutput number: %d",
                          CCD_geom->exts[i].outputs[j][k].out_no);
                espdr_msg("\t\tOutput physical number: %d",
                          CCD_geom->exts[i].outputs[j][k].out_phys_no);
                espdr_msg("\t\t\tExt number: %d",
                          CCD_geom->exts[i].outputs[j][k].ext_no);
                espdr_msg("\t\t\tOutput coordinates: (%d, %d)",
                          CCD_geom->exts[i].outputs[j][k].out_x,
                          CCD_geom->exts[i].outputs[j][k].out_y);
                espdr_msg("\t\t\tRaw size: %d x %d",
                          CCD_geom->exts[i].outputs[j][k].raw_nx,
                          CCD_geom->exts[i].outputs[j][k].raw_ny);
                espdr_msg("\t\t\tReal size: %d x %d",
                          CCD_geom->exts[i].outputs[j][k].real_nx,
                          CCD_geom->exts[i].outputs[j][k].real_ny);
                espdr_msg("\t\t\tReal coordinates: (%d, %d) & (%d, %d)",
                          CCD_geom->exts[i].outputs[j][k].real_llx,
                          CCD_geom->exts[i].outputs[j][k].real_lly,
                          CCD_geom->exts[i].outputs[j][k].real_urx,
                          CCD_geom->exts[i].outputs[j][k].real_ury);
                espdr_msg("\t\t\tPrescan size: %d x %d",
                          CCD_geom->exts[i].outputs[j][k].pscan_nx,
                          CCD_geom->exts[i].outputs[j][k].pscan_ny);
                espdr_msg("\t\t\tPrescan coordinates: (%d, %d) & (%d, %d)",
                          CCD_geom->exts[i].outputs[j][k].pscan_llx,
                          CCD_geom->exts[i].outputs[j][k].pscan_lly,
                          CCD_geom->exts[i].outputs[j][k].pscan_urx,
                          CCD_geom->exts[i].outputs[j][k].pscan_ury);
                espdr_msg("\t\t\tOverscan size: %d x %d",
                          CCD_geom->exts[i].outputs[j][k].oscan_nx,
                          CCD_geom->exts[i].outputs[j][k].oscan_ny);
                espdr_msg("\t\t\tOverscan coordinates: (%d, %d) & (%d, %d)",
                          CCD_geom->exts[i].outputs[j][k].oscan_llx,
                          CCD_geom->exts[i].outputs[j][k].oscan_lly,
                          CCD_geom->exts[i].outputs[j][k].oscan_urx,
                          CCD_geom->exts[i].outputs[j][k].oscan_ury);
            }
        }
    }
    */
    
    // Changed to the short version
    espdr_msg("\tCCD geometry:");
    espdr_msg("\t\tNumber of extensions: %d", CCD_geom->ext_nb);
    for (i = 0; i < CCD_geom->ext_nb; i++) {
        espdr_msg("\t\tExt number: %d", CCD_geom->exts[i].ext_no);
        espdr_msg("\t\t\tNumber of outputs in X: %d", CCD_geom->exts[i].out_nb_x);
        espdr_msg("\t\t\tNumber of outputs in Y: %d", CCD_geom->exts[i].out_nb_y);
        espdr_msg("\t\t\tCCD X size: %d", CCD_geom->exts[i].CCD_nx);
        espdr_msg("\t\t\tCCD Y size: %d", CCD_geom->exts[i].CCD_ny);
        espdr_msg("\t\t\tRaw image X size: %d", CCD_geom->exts[i].image_nx);
        espdr_msg("\t\t\tRaw image Y size: %d", CCD_geom->exts[i].image_ny);
    }
    
	return (CPL_ERROR_NONE);
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Delete the structure for CCD_geom
 @param    CCD_geom     CCD geometry structure
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_parameters_CCD_geometry_delete(espdr_CCD_geometry *CCD_geom) {
	
    int i, j;
    
    espdr_ensure(CCD_geom == NULL, CPL_ERROR_NULL_INPUT,
                 "CCD_geom structure is NULL");

    for (i = 0; i < CCD_geom->ext_nb; i++) {
        for (j = 0; j < CCD_geom->exts[i].out_nb_x; j++) {
            cpl_free(CCD_geom->exts[i].outputs[j]);
        }
        cpl_free(CCD_geom->exts[i].outputs);
    }
    if(CCD_geom->exts) cpl_free(CCD_geom->exts);
    if(CCD_geom) cpl_free(CCD_geom);
    CCD_geom = NULL;
    return cpl_error_get_code();
}

