/*
 * espdr_drift.c
 *
 *  Created on: Mar 4, 2016
 *      Author: Alex Segovia
 *	
 *	
 */

/*
 * $Author: asegovia $
 * $Date: 2016-03-04 14:48:13 $
 * $Revision: 1.16 $
 * $Name: not supported by cvs2svn $
 */


/*----------------------------------------------------------------------------
                                Includes
 ----------------------------------------------------------------------------*/
#include <espdr_drift.h>
#include <time.h>
/*----------------------------------------------------------------------------
                              Functions code
 ----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    parameters     the parameters list
 @param    frameset   the frames list

 In case of failure the cpl_error_code is set.
 */
/*---------------------------------------------------------------------------*/

int espdr_compu_drift(cpl_parameterlist *parameters, cpl_frameset *frameset,
		const char* recipe_id)
{

    cpl_error_code my_error = CPL_ERROR_NONE;

	cpl_frameset *wave_frames = NULL;
	cpl_frameset *dll_frames = NULL;
	cpl_frameset *s2d_frames = NULL;

	cpl_image *drift_matrix;

	cpl_imagelist *wave_images = NULL;
	cpl_imagelist *dll_images = NULL;
	cpl_imagelist *s2d_image_flux = NULL;
	cpl_imagelist *s2d_image_err = NULL;
	cpl_imagelist *s2d_image_qual = NULL;

	const char *pro_catg;
	char **pro_catg_array;

	int nx, ny, i, j;

    cpl_propertylist *keywords = NULL;
    cpl_propertylist *master_keywords = NULL;
	//const char *input_filename = NULL;

    const int rec_ntags = 19;
    const char* rec_tags[19] = {ESPDR_CCD_GEOM,
        ESPDR_PRO_CATG_WAVE_MATRIX_THAR_FP_A, ESPDR_PRO_CATG_WAVE_MATRIX_FP_THAR_B,
        ESPDR_PRO_CATG_DLL_MATRIX_THAR_FP_A, ESPDR_PRO_CATG_DLL_MATRIX_FP_THAR_B,
        ESPDR_PRO_CATG_S2D_BLAZE_FP_FP_A, ESPDR_PRO_CATG_S2D_BLAZE_FP_FP_B,
        ESPDR_PRO_CATG_S2D_BLAZE_THAR_FP_A, ESPDR_PRO_CATG_S2D_BLAZE_THAR_FP_B,
        ESPDR_PRO_CATG_S2D_BLAZE_FP_THAR_A, ESPDR_PRO_CATG_S2D_BLAZE_FP_THAR_B,
        ESPDR_PRO_CATG_S2D_BLAZE_THAR_THAR_A, ESPDR_PRO_CATG_S2D_BLAZE_THAR_THAR_B,
        ESPDR_PRO_CATG_S2D_BLAZE_LFC_FP_A, ESPDR_PRO_CATG_S2D_BLAZE_LFC_FP_B,
        ESPDR_PRO_CATG_S2D_BLAZE_FP_LFC_A, ESPDR_PRO_CATG_S2D_BLAZE_FP_LFC_B,
        ESPDR_PRO_CATG_S2D_BLAZE_LFC_LFC_A, ESPDR_PRO_CATG_S2D_BLAZE_LFC_LFC_B
    };
    int is_required[19] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

	/* input frames filenames */
    //const char *line_filename = NULL;
	cpl_msg_set_level(CPL_MSG_INFO);

    espdr_msg("Starting compute drift");

    /* Identify the RAW and CALIB frames in the input frameset */
    espdr_ensure(espdr_dfs_set_groups(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(),
                 "DFS setting groups failed");

    espdr_ensure(espdr_check_input_tags(frameset,
                                        rec_tags, is_required,
                                        rec_ntags) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");
    espdr_ensure(espdr_check_input_inst_config(frameset) != CPL_ERROR_NONE,
                 cpl_error_get_code(), "Wrong input tag!");

    cpl_frame* CCD_geom_frame = espdr_frame_find(frameset,
                                                 ESPDR_CCD_GEOM);
    espdr_CCD_geometry *CCD_geom = espdr_CCD_geom_init(parameters,
                                                       CCD_geom_frame);
    cpl_frame* inst_config_frame = espdr_get_inst_config(frameset);
    espdr_inst_config *inst_config = espdr_inst_config_init(parameters,
                                                            CCD_geom->ext_nb,
                                                            inst_config_frame);

    cpl_frameset *used_frames = cpl_frameset_new();

    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(CCD_geom_frame));
    my_error = cpl_frameset_insert(used_frames,
                                   cpl_frame_duplicate(inst_config_frame));

    /* Filling up the DRS QC KWs structure */
    espdr_qc_keywords *qc_kws = (espdr_qc_keywords *)
    cpl_malloc(sizeof(espdr_qc_keywords));
    my_error = espdr_fill_qc_keywords(inst_config, qc_kws);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "Filling QC KWs failed: %s",
                 cpl_error_get_message_default(my_error));
    espdr_ensure(qc_kws == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                 "QC KWs structure is NULL");

    cpl_table **drift_results_cum_slices = NULL;
	drift_results_cum_slices = (cpl_table **)
				   cpl_malloc(inst_config->slices_nb_per_phys_order*sizeof(cpl_table *));

    cpl_table **drift_results_slices = NULL;
	drift_results_slices = (cpl_table **)
				   cpl_malloc(inst_config->slices_nb_per_phys_order*sizeof(cpl_table *));

    /* Parameters overloading */

	const cpl_parameter *par_method_thar =
		cpl_parameterlist_find_const(parameters,"espdr.espdr_compu_drift.method_thar");
	int method_thar_set = cpl_parameter_get_default_flag(par_method_thar);

	if (method_thar_set) {
		strcpy(inst_config->drift_method_thar,
			cpl_parameter_get_string(par_method_thar));
	}

	const cpl_parameter *par_method_fp =
		cpl_parameterlist_find_const(parameters,"espdr.espdr_compu_drift.method_fp");
	int method_fp_set = cpl_parameter_get_default_flag(par_method_fp);

	if (method_fp_set) {
		strcpy(inst_config->drift_method_fp,
			cpl_parameter_get_string(par_method_fp));
	}

	const cpl_parameter *par_method_lfc =
		cpl_parameterlist_find_const(parameters,"espdr.espdr_compu_drift.method_lfc");
	int method_lfc_set = cpl_parameter_get_default_flag(par_method_lfc);

	if (method_lfc_set) {
		strcpy(inst_config->drift_method_lfc,
			cpl_parameter_get_string(par_method_lfc));
	}

	const cpl_parameter *par_drift_space =
		cpl_parameterlist_find_const(parameters,"espdr.espdr_compu_drift.space");
	int drift_space_set = cpl_parameter_get_default_flag(par_drift_space);

	if (drift_space_set) {
		strcpy(inst_config->drift_space,
			cpl_parameter_get_string(par_drift_space));
	}

	const cpl_parameter *par_drift_ksig =
		cpl_parameterlist_find_const(parameters,"espdr.espdr_compu_drift.ksig");
	int drift_ksig_set = cpl_parameter_get_default_flag(par_drift_ksig);

	if (drift_ksig_set) {
		inst_config->drift_ksigma = cpl_parameter_get_double(par_drift_ksig);
	}

	const cpl_parameter *par_drift_max_flux_threshold =
		cpl_parameterlist_find_const(parameters,"espdr.espdr_compu_drift.max_flux_threshold");
	int drift_max_flux_threshold_set = cpl_parameter_get_default_flag(par_drift_max_flux_threshold);

	if (drift_max_flux_threshold_set) {
		inst_config->drift_max_flux_threshold = cpl_parameter_get_double(par_drift_max_flux_threshold);
	}

	const cpl_parameter *par_drift_min_flux_threshold =
		cpl_parameterlist_find_const(parameters,"espdr.espdr_compu_drift.min_flux_threshold");
	int drift_min_flux_threshold_set = cpl_parameter_get_default_flag(par_drift_min_flux_threshold);

	if (drift_min_flux_threshold_set) {
		inst_config->drift_min_flux_threshold = cpl_parameter_get_double(par_drift_min_flux_threshold);
	}

	espdr_msg("Effective compute drift parameters:");
	espdr_msg("Drift method for THAR: %s", inst_config->drift_method_thar);
	espdr_msg("Drift method for FP: %s", inst_config->drift_method_fp);
	espdr_msg("Drift method for LFC: %s", inst_config->drift_method_lfc);
	espdr_msg("Drift space: %s", inst_config->drift_space);
	espdr_msg("Drift ksigma: %f", inst_config->drift_ksigma);
	espdr_msg("Drift max flux threshold: %f", inst_config->drift_max_flux_threshold);
	espdr_msg("Drift min flux threshold: %f", inst_config->drift_min_flux_threshold);

	/* Extract S2D frames */

	//char s2d_catg_tag[32];
	cpl_frame* frame = NULL;
	int nset = cpl_frameset_get_size(frameset);
	cpl_frameset_iterator* iter = cpl_frameset_iterator_new(frameset);
	frame = cpl_frameset_iterator_get(iter);
	//char *p;

	s2d_frames = cpl_frameset_new();

	for (int iter_frame = 0; iter_frame < nset; iter_frame++ ) {

		const char* curr_tag = cpl_frame_get_tag(frame);

		if (strstr(curr_tag,ESPDR_PRO_CATG_S2D_BLAZE) != NULL ) {
			cpl_frameset_insert(s2d_frames,
			cpl_frame_duplicate(frame));

		}
		cpl_frameset_iterator_advance(iter, 1);
		frame = cpl_frameset_iterator_get(iter);
	}
	cpl_frameset_iterator_delete(iter);

    int s2d_frameset_size = cpl_frameset_get_size(s2d_frames);

    espdr_ensure(s2d_frameset_size < 2, CPL_ERROR_NULL_INPUT,
                 "The input set of frames doesn't contain enough s2d");

	const char* s2d_name;
    const char* s2d_ref_name = cpl_frame_get_filename(
                                    cpl_frameset_get_position(s2d_frames,0));

	s2d_image_flux = cpl_imagelist_new();
	s2d_image_err = cpl_imagelist_new();
	s2d_image_qual = cpl_imagelist_new();

	pro_catg_array = cpl_malloc(s2d_frameset_size*sizeof(char*));
	double *mjd_array = cpl_calloc(s2d_frameset_size, sizeof(double));

    master_keywords = cpl_propertylist_new();

	/* Loading S2D images flux, err & qual */

	for ( int iter_image=0; iter_image < s2d_frameset_size; iter_image++ ) {

        s2d_name = cpl_frame_get_filename(
                            cpl_frameset_get_position(s2d_frames,iter_image));

        /* flux */
        my_error =
        cpl_imagelist_set(s2d_image_flux,cpl_image_load(
                                        s2d_name,CPL_TYPE_DOUBLE,0,1),iter_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "S2D flux image %s set failed.", s2d_name);

        /* err */
        my_error =
        cpl_imagelist_set(s2d_image_err,cpl_image_load(
                                        s2d_name,CPL_TYPE_DOUBLE,0,2),iter_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "S2D err image %s set failed.", s2d_name);

        /* qual */
        my_error =
        cpl_imagelist_set(s2d_image_qual,cpl_image_load(
                                        s2d_name,CPL_TYPE_INT,0,3),iter_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "S2D qual image %s set failed.", s2d_name);

        /* Load S2D keywords from the first image (primary header) */
        keywords = cpl_propertylist_load(s2d_name, 0);
        espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
                     "Keywords of %s are NULL", s2d_name);

        /* Getting the PRO.CATG key */
        pro_catg = cpl_propertylist_get_string(keywords,"ESO PRO CATG");
        pro_catg_array[iter_image] = cpl_malloc((strlen(pro_catg)+1)*sizeof(char));
        strcpy(pro_catg_array[iter_image],pro_catg);
        //espdr_msg("PRO_CATG[%d]: %s", iter_image,pro_catg_array[iter_image] );

        /* Getting MJD key*/
        mjd_array[iter_image] = cpl_propertylist_get_double(keywords,"MJD-OBS");

        my_error = cpl_frameset_insert(used_frames,
                        cpl_frame_duplicate(cpl_frameset_get_position(s2d_frames,
                                                                      iter_image)));

        cpl_propertylist_delete(keywords);
	}

	/* Extract wave frame */

    //char wave_catg_tag[32];
    nset = cpl_frameset_get_size(frameset);
    iter = cpl_frameset_iterator_new(frameset);
    frame = cpl_frameset_iterator_get(iter);

    wave_frames = cpl_frameset_new();

    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {

        const char* curr_tag = cpl_frame_get_tag(frame);

        if(strstr(curr_tag,ESPDR_PRO_CATG_WAVE_MATRIX) != NULL ) {
            cpl_frameset_insert(wave_frames,
                                cpl_frame_duplicate(frame));
        }

        cpl_frameset_iterator_advance(iter, 1);
        frame = cpl_frameset_iterator_get(iter);
    }
    cpl_frameset_iterator_delete(iter);

    int wave_frameset_size = cpl_frameset_get_size(wave_frames);
    espdr_ensure(wave_frameset_size < 1, CPL_ERROR_NULL_INPUT,
                 "The input set of frames doesn't contain enough wave");

    const char* wave_name;

    wave_images = cpl_imagelist_new();

	/* Loading wave images */

    for ( int iter_image=0;iter_image < wave_frameset_size;iter_image++ ) {
        wave_name = cpl_frame_get_filename(
                                cpl_frameset_get_position(wave_frames,iter_image));
        espdr_msg("Wave matrix file : %s", wave_name);
        my_error =
        cpl_imagelist_set(wave_images,cpl_image_load(
                                wave_name,CPL_TYPE_DOUBLE,0,1),iter_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "wave image added failed.");

        my_error = cpl_frameset_insert(used_frames,
                            cpl_frame_duplicate(cpl_frameset_get_position(wave_frames,
                                                                          iter_image)));
    }

	/* Extract dll frame */

    //char dll_catg_tag[32];
    nset = cpl_frameset_get_size(frameset);
    iter = cpl_frameset_iterator_new(frameset);
    frame = cpl_frameset_iterator_get(iter);

    dll_frames = cpl_frameset_new();

    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {

        const char* curr_tag = cpl_frame_get_tag(frame);

        if(strstr(curr_tag,ESPDR_PRO_CATG_DLL_MATRIX) != NULL ) {
            cpl_frameset_insert(dll_frames,
                                cpl_frame_duplicate(frame));
        }

        cpl_frameset_iterator_advance(iter, 1);
        frame = cpl_frameset_iterator_get(iter);
    }
    cpl_frameset_iterator_delete(iter);

    int dll_frameset_size = cpl_frameset_get_size(dll_frames);
    espdr_ensure(dll_frameset_size < 1, CPL_ERROR_NULL_INPUT,
                 "The input set of frames doesn't contain enough dll");

    const char* dll_name;

    dll_images = cpl_imagelist_new();

	/* Loading dll images */

    for ( int iter_image=0;iter_image < dll_frameset_size;iter_image++ ) {
        dll_name = cpl_frame_get_filename(
                                cpl_frameset_get_position(dll_frames,iter_image));
        espdr_msg("Wave matrix file : %s", dll_name);
        my_error =
        cpl_imagelist_set(dll_images,cpl_image_load(
                                dll_name,CPL_TYPE_DOUBLE,0,1),iter_image);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "dll image added failed.");

        my_error = cpl_frameset_insert(used_frames,
                        cpl_frame_duplicate(cpl_frameset_get_position(dll_frames,
                                                                      iter_image)));
    }

	/* Getting the flux, err & qual reference images  */

	cpl_image *s2d_ref_flux = cpl_imagelist_get(s2d_image_flux,0);
	cpl_image *s2d_ref_err = cpl_imagelist_get(s2d_image_err,0);
	cpl_image *s2d_ref_qual = cpl_imagelist_get(s2d_image_qual,0);

	nx = cpl_image_get_size_x(s2d_ref_flux);
	ny = cpl_image_get_size_y(s2d_ref_flux);


	/* Slicing the S2D reference */
	cpl_imagelist *s2d_ref_flux_sliced = cpl_imagelist_new();
	cpl_imagelist *s2d_ref_err_sliced = cpl_imagelist_new();
	cpl_imagelist *s2d_ref_qual_sliced = cpl_imagelist_new();
	cpl_image *s2d_ref_flux_slice;
	cpl_image *s2d_ref_err_slice;
	cpl_image *s2d_ref_qual_slice;

	for (int slice=0; slice < inst_config->slices_nb_per_phys_order; slice++) {

		s2d_ref_flux_slice =
		cpl_image_new(nx,
		ny/inst_config->slices_nb_per_phys_order,
		CPL_TYPE_DOUBLE);

		s2d_ref_err_slice =
		cpl_image_new(nx,
		ny/inst_config->slices_nb_per_phys_order,
		CPL_TYPE_DOUBLE);

		s2d_ref_qual_slice =
		cpl_image_new(nx,
		ny/inst_config->slices_nb_per_phys_order,
		CPL_TYPE_INT);

		cpl_imagelist_set(s2d_ref_flux_sliced,s2d_ref_flux_slice,slice);
		cpl_imagelist_set(s2d_ref_err_sliced,s2d_ref_err_slice,slice);
		cpl_imagelist_set(s2d_ref_qual_sliced,s2d_ref_qual_slice,slice);
	}

	espdr_split_by_slices(s2d_ref_flux,
			      inst_config->slices_nb_per_phys_order,
			      &s2d_ref_flux_sliced);

	espdr_split_by_slices(s2d_ref_err,
			      inst_config->slices_nb_per_phys_order,
			      &s2d_ref_err_sliced);

	espdr_split_by_slices(s2d_ref_qual,
			      inst_config->slices_nb_per_phys_order,
			      &s2d_ref_qual_sliced);

	/* Creating drift_matrix image */

	drift_matrix = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);

	/* Getting the S2D reference fiber */

	char *fibre;
	int pro_catg_len = strlen(pro_catg_array[0]);
	//espdr_msg("pro_catg_len = %d", pro_catg_len);
	// This code treats the drift on OBJECT,FP frames
	// Temporay workaround!!!
	if (pro_catg_len < 17) {
		strcpy(pro_catg_array[0], "S2D_BLAZE_OBJECT_FP_B");
		pro_catg_len = strlen(pro_catg_array[0]);
	}

	fibre = pro_catg_array[0] + pro_catg_len - 1;

	espdr_msg("Getting the S2D reference fibre : %s",fibre);
	//espdr_msg("pro_catg_array[0] = %s", pro_catg_array[0]);

	/* Getting the S2D wave source */

	char *spec_type = pro_catg_array[0];
	char *pro_catg_name;
	spec_type += strlen(ESPDR_PRO_CATG_S2D)+1+6;
	spec_type[strlen(spec_type)-2]=0;
	pro_catg_name = cpl_malloc((strlen(spec_type)+1)*sizeof(char));
	strcpy(pro_catg_name,spec_type);
	espdr_msg("Getting the S2D wave source: %s",spec_type);

	/* Getting the S2D sub wave source dependig on fibre */

    char src_a[24];
    char src_b[24];
    char src_selected[24];
    char *separator;

    separator = strtok((char*)spec_type,"_");
    strcpy(src_a, separator);
    separator = strtok(NULL,"_");
    strcpy(src_b, separator);

	if(strcmp(fibre,"A")==0) {
		strcpy(src_selected,src_a);
	} else if(strcmp(fibre,"B")==0) {
		strcpy(src_selected,src_b);
	} else {
		espdr_msg("ERROR: wrong fibre.");
		exit(EXIT_FAILURE);
	}

    espdr_msg( "------> src a = %s",src_a);
    espdr_msg( "------> src b = %s",src_b);
    espdr_msg( "------> src selected = %s",src_selected);

	/* Choosing the drift function based on wave source */

	char drift_method[64];

	if(strcmp(src_selected,"THAR")==0) {
		strcpy(drift_method,inst_config->drift_method_thar);
	} else if(strcmp(src_selected,"FP")==0) {
		strcpy(drift_method,inst_config->drift_method_fp);
	} else if(strcmp(src_selected,"LFC")==0) {
		strcpy(drift_method,inst_config->drift_method_lfc);
	} else {
		espdr_msg("ERROR: wrong wave source.");
		exit(EXIT_FAILURE);
	}

    espdr_msg( "Choosing drift_method: %s",drift_method);

	/* Getting the detectors number */

	espdr_msg("Detectors number: %d",CCD_geom->ext_nb);

	/* Creating table drif_results */

	int first_order = 0;
	int last_order = 0;
	int fibre_nb = getposition(fibre);

	for (int slice = 0; slice < inst_config->slices_nb_per_phys_order; slice++) {

        drift_results_slices[slice] = cpl_table_new(CCD_geom->ext_nb);

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "first_order",
                                        CPL_TYPE_INT);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for first_order: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "first_order",
                                             "");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "last_order",
                                        CPL_TYPE_INT);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for last_order: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "last_order",
                                             "");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "flux_ratio",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for flux_ratio: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "flux_ratio",
                                             "");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "flux_ratio_err",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for flux_ratio_err: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "flux_ratio_err",
                                             "");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "drift_mean",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for drift_mean: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "drift_mean",
                                             "[pix]");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "drift_mean_err",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for drift_mean_err: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "drift_mean_err",
                                             "[pix]");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "drift_slope_x",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for drift_slope_x: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "drift_slope_x",
                                             "[pix/pix]");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "drift_slope_x_err",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for drift_slope_x_err: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "drift_slope_x_err",
                                             "[pix/pix]");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "drift_slope_o",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for drift_slope_o: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "drift_slope_o",
                                             "[pix/order]");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "drift_slope_o_err",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for drift_slope_o_err: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "drift_slope_o_err",
                                             "[pix/order]");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "chisq",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for chisq: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "chisq",
                                             "");

        my_error = cpl_table_new_column(drift_results_slices[slice],
                                        "rejected",
                                        CPL_TYPE_INT);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for rejected: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_slices[slice],
                                             "rejected",
                                             "");

        /* Setting first order and last order columns per detector */

        first_order = 0;
        last_order = 0;
        for (j = 0; j < CCD_geom->ext_nb; j++) {
            last_order = last_order + inst_config->orders_nb[fibre_nb*CCD_geom->ext_nb+j];
            first_order = last_order - inst_config->orders_nb[fibre_nb*CCD_geom->ext_nb+j];
            cpl_table_set(drift_results_slices[slice],
                          "first_order",j,
                          first_order/inst_config->slices_nb_per_phys_order);
            cpl_table_set(drift_results_slices[slice],
                          "last_order",j,
                          (last_order-1)/inst_config->slices_nb_per_phys_order);
            espdr_msg("last_order=%d fibre=%d det=%d slice=%d",
                      (last_order-1)/inst_config->slices_nb_per_phys_order,fibre_nb,j,slice);
            espdr_msg("first_order=%d fibre=%d det=%d slice=%d",
                      first_order/inst_config->slices_nb_per_phys_order,fibre_nb,j,slice);
        }

	} // slices

	/* Getting the wave matrix */

	cpl_image *wave = cpl_imagelist_get(wave_images,0);

	/* Getting the dll */

	cpl_image *dll = cpl_imagelist_get(dll_images,0);

	/* Creating table drif_results_cum */

	char col_drift_mean[64];
	char col_drift_mean_err[64];
	char col_drift_flux_ratio[64];
	char col_drift_flux_ratio_err[64];
	char col_drift_slope_x[64];
	char col_drift_slope_x_err[64];
	char col_drift_slope_o[64];
	char col_drift_slope_o_err[64];
	char col_drift_chisq[64];
	char col_drift_rejected[64];

	for (int slice=0; slice < inst_config->slices_nb_per_phys_order; slice++) {

        drift_results_cum_slices[slice] = cpl_table_new(s2d_frameset_size-1);

        my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                        "mjd",
                                        CPL_TYPE_DOUBLE);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "cpl_table_new_column failed for mjd: %s",
                     cpl_error_get_message_default(my_error));
        my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                             "mjd",
                                             "[days]");

        /* Adding MJD to drift_results_cum_slices */

        for (j = 0; j < s2d_frameset_size-1; j++) {
            cpl_table_set(drift_results_cum_slices[slice],
                          "mjd",j,
                          mjd_array[j+1]);
        }

        for (j = 0; j < CCD_geom->ext_nb; j++) {

            strcpy(col_drift_mean,"drift_mean");
            sprintf(col_drift_mean,"%s_DET%c",col_drift_mean,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_mean,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_mean,
                                                 "[pix]");
            strcpy(col_drift_mean,"");

            strcpy(col_drift_mean_err,"drift_mean_err");
            sprintf(col_drift_mean_err,"%s_DET%c",col_drift_mean_err,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_mean_err,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_mean_err,
                                                 "[pix]");
            strcpy(col_drift_mean_err,"");

            strcpy(col_drift_flux_ratio,"flux_ratio");
            sprintf(col_drift_flux_ratio,"%s_DET%c",col_drift_flux_ratio,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_flux_ratio,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_flux_ratio,
                                                 "");
            strcpy(col_drift_flux_ratio,"");

            strcpy(col_drift_flux_ratio_err,"flux_ratio_err");
            sprintf(col_drift_flux_ratio_err,"%s_DET%c",col_drift_flux_ratio_err,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_flux_ratio_err,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_flux_ratio_err,
                                                 "");
            strcpy(col_drift_flux_ratio_err,"");

            strcpy(col_drift_slope_x,"drift_slope_x");
            sprintf(col_drift_slope_x,"%s_DET%c",col_drift_slope_x,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_slope_x,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_slope_x,
                                                 "[pix/pix]");
            strcpy(col_drift_slope_x,"");

            strcpy(col_drift_slope_x_err,"drift_slope_x_err");
            sprintf(col_drift_slope_x_err,"%s_DET%c",col_drift_slope_x_err,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_slope_x_err,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_slope_x_err,
                                                 "[pix/pix]");
            strcpy(col_drift_slope_x_err,"");

            strcpy(col_drift_slope_o,"drift_slope_o");
            sprintf(col_drift_slope_o,"%s_DET%c",col_drift_slope_o,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_slope_o,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_slope_o,
                                                 "[pix/order]");
            strcpy(col_drift_slope_o,"");

            strcpy(col_drift_slope_o_err,"drift_slope_o_err");
            sprintf(col_drift_slope_o_err,"%s_DET%c",col_drift_slope_o_err,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_slope_o_err,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_slope_o_err,
                                                 "[pix/order]");
            strcpy(col_drift_slope_o_err,"");

            strcpy(col_drift_chisq,"chisq");
            sprintf(col_drift_chisq,"%s_DET%c",col_drift_chisq,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_chisq,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_chisq,
                                                 "");
            strcpy(col_drift_chisq,"");

            strcpy(col_drift_rejected,"rejected");
            sprintf(col_drift_rejected,"%s_DET%c",col_drift_rejected,j+'0');
            my_error = cpl_table_new_column(drift_results_cum_slices[slice],
                                            col_drift_rejected,
                                            CPL_TYPE_DOUBLE);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "cpl_table_new_column failed: %s",
                         cpl_error_get_message_default(my_error));
            my_error = cpl_table_set_column_unit(drift_results_cum_slices[slice],
                                                 col_drift_rejected,
                                                 "");
            strcpy(col_drift_rejected,"");

        }

	} // slices

	/* Looping over the rest of S2D products */

	cpl_image *curr_s2d_flux;
	cpl_image *curr_s2d_err;
	cpl_image *curr_s2d_qual;

	char qc_key_prefix[64];
	const char* s2d_file_name;

	/* Getting S2D slices */

	cpl_imagelist *curr_s2d_flux_sliced = cpl_imagelist_new();
	cpl_imagelist *curr_s2d_err_sliced = cpl_imagelist_new();
	cpl_imagelist *curr_s2d_qual_sliced = cpl_imagelist_new();
	cpl_imagelist *wave_sliced = cpl_imagelist_new();
    cpl_imagelist *dll_sliced = cpl_imagelist_new();
	cpl_image *flux_slice;
	cpl_image *err_slice;
	cpl_image *qual_slice;
	cpl_image *wave_slice;
	cpl_image *dll_slice;

	for(i = 1; i < s2d_frameset_size; i++) {

		curr_s2d_flux=cpl_imagelist_get(s2d_image_flux,i);
		curr_s2d_err=cpl_imagelist_get(s2d_image_err,i);
		curr_s2d_qual=cpl_imagelist_get(s2d_image_qual,i);

        s2d_file_name = cpl_frame_get_filename(
                            cpl_frameset_get_position(s2d_frames,i));

		/* Load S2D keywords from the
		   first image (primary header) */

		keywords = cpl_propertylist_load(s2d_file_name, 0);
		espdr_ensure(keywords == NULL, CPL_ERROR_ILLEGAL_OUTPUT,
			     "keywords are NULL");

		/* Mem allocation for slicing imagelists */
		for (int slice=0; slice < inst_config->slices_nb_per_phys_order; slice++) {

            flux_slice = cpl_image_new(cpl_image_get_size_x(curr_s2d_flux),
                          cpl_image_get_size_y(curr_s2d_flux)/
                                       inst_config->slices_nb_per_phys_order,
                          CPL_TYPE_DOUBLE);

            err_slice = cpl_image_new(cpl_image_get_size_x(curr_s2d_err),
                          cpl_image_get_size_y(curr_s2d_err)/
                                      inst_config->slices_nb_per_phys_order,
                          CPL_TYPE_DOUBLE);

            qual_slice = cpl_image_new(cpl_image_get_size_x(curr_s2d_qual),
                          cpl_image_get_size_y(curr_s2d_qual)/
                                       inst_config->slices_nb_per_phys_order,
                          CPL_TYPE_INT);

            wave_slice = cpl_image_new(cpl_image_get_size_x(wave),
                          cpl_image_get_size_y(wave)/
                                       inst_config->slices_nb_per_phys_order,
                          CPL_TYPE_DOUBLE);

            dll_slice = cpl_image_new(cpl_image_get_size_x(dll),
                          cpl_image_get_size_y(dll)/
                                      inst_config->slices_nb_per_phys_order,
                          CPL_TYPE_DOUBLE);

			cpl_imagelist_set(curr_s2d_flux_sliced,flux_slice,slice);
			cpl_imagelist_set(curr_s2d_err_sliced,err_slice,slice);
			cpl_imagelist_set(curr_s2d_qual_sliced,qual_slice,slice);
			cpl_imagelist_set(wave_sliced,wave_slice,slice);
			cpl_imagelist_set(dll_sliced,dll_slice,slice);

		}

		/* Slicing input images for espdr_measure_drift */
        espdr_split_by_slices(curr_s2d_flux,
                              inst_config->slices_nb_per_phys_order,
                              &curr_s2d_flux_sliced);

        espdr_split_by_slices(curr_s2d_err,
                              inst_config->slices_nb_per_phys_order,
                              &curr_s2d_err_sliced);

        espdr_split_by_slices(curr_s2d_qual,
                              inst_config->slices_nb_per_phys_order,
                              &curr_s2d_qual_sliced);

        espdr_split_by_slices(wave,
                              inst_config->slices_nb_per_phys_order,
                              &wave_sliced);

        espdr_split_by_slices(dll,
                              inst_config->slices_nb_per_phys_order,
                              &dll_sliced);

		/* Making the drift calculation by slice */

        for (int slice=0; slice < inst_config->slices_nb_per_phys_order; slice++) {

            my_error = espdr_measure_drift(
                                cpl_imagelist_get(curr_s2d_flux_sliced,slice),
                                cpl_imagelist_get(curr_s2d_err_sliced,slice),
                                cpl_imagelist_get(curr_s2d_qual_sliced,slice),
                                cpl_imagelist_get(s2d_ref_flux_sliced,slice),
                                cpl_imagelist_get(s2d_ref_err_sliced,slice),
                                cpl_imagelist_get(s2d_ref_qual_sliced,slice),
                                cpl_imagelist_get(wave_sliced,slice),
                                cpl_imagelist_get(dll_sliced,slice),
                                drift_method,
                                inst_config->drift_space,
                                inst_config->drift_ksigma,
                                inst_config->drift_max_flux_threshold,
                                inst_config->drift_min_flux_threshold,
                                inst_config->image_cosmics_part,
                                &drift_results_slices[slice],
                                &drift_matrix);

            /* Adding ARCFILE of the reference file */

            strcpy(qc_key_prefix,"ESO QC DRIFT REF_ARC_FILE");
            if (cpl_propertylist_has(keywords, "ARCFILE")) {
                cpl_propertylist_append_string(master_keywords, qc_key_prefix,
                                               cpl_propertylist_get_string(keywords,
                                                                           "ARCFILE"));
            } else {
                //espdr_msg("REF FILE NAME:::::: %s", strrchr(s2d_ref_name, '/')+1);
                cpl_propertylist_append_string(master_keywords, qc_key_prefix,
                                               strrchr(s2d_ref_name, '/')+1);
            }

            strcpy(qc_key_prefix,"");

            /* Adding drift parameters */

            strcpy(qc_key_prefix,"ESO QC DRIFT METHOD");
            cpl_propertylist_append_string(master_keywords, qc_key_prefix,
                                           drift_method);
            strcpy(qc_key_prefix,"");

            strcpy(qc_key_prefix,"ESO QC DRIFT SPACE");
            cpl_propertylist_append_string(master_keywords, qc_key_prefix,
                                           inst_config->drift_space);
            strcpy(qc_key_prefix,"");

            /* Adding QC keywords per detector */

            for (j = 0; j < CCD_geom->ext_nb; j++) {

                /* Adding drift_mean per detector
                 to drift_results_cum_slices table*/

                strcpy(col_drift_mean,"drift_mean");
                sprintf(col_drift_mean,"%s_DET%c",col_drift_mean,j+'0');
                cpl_table_set(drift_results_cum_slices[slice], col_drift_mean, i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "drift_mean",j,NULL));
                strcpy(col_drift_mean,"");

                /* Adding drift_mean_err per detector
                 to drift_results_cum_slices table*/

                strcpy(col_drift_mean_err,"drift_mean_err");
                sprintf(col_drift_mean_err,"%s_DET%c",col_drift_mean_err,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_mean_err,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "drift_mean_err",j,NULL));
                strcpy(col_drift_mean_err,"");

                strcpy(col_drift_flux_ratio,"flux_ratio");
                sprintf(col_drift_flux_ratio,"%s_DET%c",col_drift_flux_ratio,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_flux_ratio,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "flux_ratio",j,NULL));
                strcpy(col_drift_flux_ratio,"");

                strcpy(col_drift_flux_ratio_err,"flux_ratio_err");
                sprintf(col_drift_flux_ratio_err,"%s_DET%c",col_drift_flux_ratio_err,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_flux_ratio_err,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "flux_ratio_err",j,NULL));
                strcpy(col_drift_flux_ratio_err,"");

                strcpy(col_drift_slope_x,"drift_slope_x");
                sprintf(col_drift_slope_x,"%s_DET%c",col_drift_slope_x,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_slope_x,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "drift_slope_x",j,NULL));
                strcpy(col_drift_slope_x,"");

                strcpy(col_drift_slope_x_err,"drift_slope_x_err");
                sprintf(col_drift_slope_x_err,"%s_DET%c",col_drift_slope_x_err,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_slope_x_err,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "drift_slope_x_err",j,NULL));
                strcpy(col_drift_slope_x_err,"");

                strcpy(col_drift_slope_o,"drift_slope_o");
                sprintf(col_drift_slope_o,"%s_DET%c",col_drift_slope_o,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_slope_o,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "drift_slope_o",j,NULL));
                strcpy(col_drift_slope_o,"");

                strcpy(col_drift_slope_o_err,"drift_slope_o_err");
                sprintf(col_drift_slope_o_err,"%s_DET%c",col_drift_slope_o_err,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_slope_o_err,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "drift_slope_o_err",j,NULL));
                strcpy(col_drift_slope_o_err,"");

                strcpy(col_drift_chisq,"chisq");
                sprintf(col_drift_chisq,"%s_DET%c",col_drift_chisq,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_chisq,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "chisq",j,NULL));
                strcpy(col_drift_chisq,"");

                strcpy(col_drift_rejected,"rejected");
                sprintf(col_drift_rejected,"%s_DET%c",col_drift_rejected,j+'0');
                cpl_table_set(drift_results_cum_slices[slice],
                              col_drift_rejected,i-1,
                              cpl_table_get(drift_results_slices[slice],
                                            "rejected",j,NULL));
                strcpy(col_drift_rejected,"");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "LAST_ORDER");
                cpl_propertylist_append_int(master_keywords, qc_key_prefix,
                                            cpl_table_get(drift_results_slices[slice],
                                                          "last_order", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "FLUX_RATIO");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "flux_ratio", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "FLUX_RATIO_ERR");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "flux_ratio_err", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "DRIFT_MEAN");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "drift_mean", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix,j+'0',"DRIFT_MEAN_ERR");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "drift_mean_err", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "DRIFT_SLOPE_X");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "drift_slope_x", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "DRIFT_SLOPE_X_ERR");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "drift_slope_x_err", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "DRIFT_SLOPE_O");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "drift_slope_o", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "DRIFT_SLOPE_O_ERR");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "drift_slope_o_err", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "CHISQ");
                cpl_propertylist_append_double(master_keywords, qc_key_prefix,
                                               cpl_table_get(drift_results_slices[slice],
                                                             "chisq", j, NULL));
                strcpy(qc_key_prefix, "");

                strcpy(qc_key_prefix, "ESO QC DRIFT DET");
                sprintf(qc_key_prefix, "%s%c %s", qc_key_prefix, j+'0', "REJECTED");
                cpl_propertylist_append_int(master_keywords, qc_key_prefix,
                                            cpl_table_get(drift_results_slices[slice],
                                                          "rejected", j, NULL));
                strcpy(qc_key_prefix, "");
            } // det
        } // slicess

		cpl_propertylist_delete(keywords);
	}

	char drift_result_cumul_name[128];
	char drift_result_cumul_txt_name[128];
    char drift_pro_catg_cumul_name[128];

	time_t rawtime;
	struct tm *timeinfo;
	char buffer_timestamp[64];

	time ( &rawtime );
	timeinfo = localtime(&rawtime);

	strftime(buffer_timestamp,64,"%Y-%m-%d_%H:%M:%S",timeinfo);

	for (int slice=0; slice < inst_config->slices_nb_per_phys_order; slice++) {

        sprintf(drift_pro_catg_cumul_name,
                "%s_%s_%s_%c",
                ESPDR_PRO_CATG_DRIFT,
                pro_catg_name, fibre, slice+'0');
        my_error = cpl_propertylist_update_string(master_keywords, PRO_CATG_KW,
                                                  drift_pro_catg_cumul_name);

        sprintf(drift_result_cumul_name,
                "%s_DRIFT_RESULTS_%s_%s_%c.fits",
                inst_config->instrument, pro_catg_name, fibre, slice+'0');

        my_error = cpl_dfs_save_table(frameset, NULL, parameters, used_frames,
                                      NULL, drift_results_cum_slices[slice], NULL,
                                      recipe_id, master_keywords, NULL,
                                      PACKAGE "/" PACKAGE_VERSION,
                                      drift_result_cumul_name);

        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_dfs_image_save failed: %s for file %s",
                     cpl_error_get_message_default(my_error),
                     drift_result_cumul_name);

        sprintf(drift_result_cumul_txt_name,
                "%s_DRIFT_RESULTS_%s_%s_%c.txt",
                inst_config->instrument,pro_catg_name,fibre,slice+'0');

		cpl_table_set_column_format(drift_results_cum_slices[slice],
					 cpl_table_get_column_name(drift_results_cum_slices[slice]),
					 "%.17g");

		for (int i = 1; i < cpl_table_get_ncol(drift_results_cum_slices[slice]); i++) {
			cpl_table_set_column_format(drift_results_cum_slices[slice],
					 cpl_table_get_column_name(NULL),
					 "%.17g");
		}

		FILE *table_dump;
		table_dump = fopen(drift_result_cumul_txt_name, "w");
		cpl_table_dump(
		drift_results_cum_slices[slice],0,s2d_frameset_size-1,table_dump);
		fclose(table_dump);

    } // slice


    /* cleaning memory */

    for ( int iter_image = 0; iter_image<s2d_frameset_size; iter_image++ ) {
        cpl_free(pro_catg_array[iter_image]);
    }

	cpl_free(pro_catg_array);
	cpl_imagelist_delete(wave_images);
	cpl_imagelist_delete(dll_images);
	cpl_imagelist_delete(s2d_image_flux);
	cpl_imagelist_delete(s2d_image_err);
	cpl_imagelist_delete(s2d_image_qual);
	cpl_imagelist_delete(curr_s2d_flux_sliced);
	cpl_imagelist_delete(curr_s2d_err_sliced);
	cpl_imagelist_delete(curr_s2d_qual_sliced);
	cpl_imagelist_delete(wave_sliced);
    cpl_imagelist_delete(dll_sliced);
	cpl_image_delete(drift_matrix);
	for (int slice=0; slice < inst_config->slices_nb_per_phys_order; slice++) {
		cpl_table_delete(drift_results_cum_slices[slice]);
		cpl_table_delete(drift_results_slices[slice]);
	}
	cpl_free(drift_results_cum_slices);
	cpl_free(drift_results_slices);
	cpl_imagelist_delete(s2d_ref_flux_sliced);
	cpl_imagelist_delete(s2d_ref_err_sliced);
	cpl_imagelist_delete(s2d_ref_qual_sliced);
	cpl_frameset_delete(wave_frames);
	cpl_frameset_delete(dll_frames);
	cpl_frameset_delete(s2d_frames);
	cpl_free(mjd_array);
	cpl_free(pro_catg_name);
	espdr_parameters_CCD_geometry_delete(CCD_geom);
	espdr_parameters_inst_config_delete(inst_config);

	return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief Init the drift table
 @param drift_table table to be initialized
 @return    CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_drift_table_init(cpl_table *drift_table,
                                      espdr_CCD_geometry *CCD_geom,
                                      espdr_inst_config *inst_config,
                                      int fibre,
                                      int *first_order_second_ext) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    
    my_error = cpl_table_new_column(drift_table,
                                    "first_order",
                                    CPL_TYPE_INT);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for first_order: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "last_order",
                                    CPL_TYPE_INT);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for last_order: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "flux_ratio",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for flux_ratio: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "flux_ratio_err",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for flux_ratio_err: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "drift_mean",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for drift_mean: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "drift_mean_err",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for drift_mean_err: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "drift_slope_x",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for drift_slope_x: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "drift_slope_x_err",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for drift_slope_x_err: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "drift_slope_o",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for drift_slope_o: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "drift_slope_o_err",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for drift_slope_o_err: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "chisq",
                                    CPL_TYPE_DOUBLE);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for chisq: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = cpl_table_new_column(drift_table,
                                    "rejected",
                                    CPL_TYPE_INT);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_table_new_column failed for rejected: %s",
                 cpl_error_get_message_default(my_error));
    
    /* Setting first order and last order columns per detector */
    
    int first_order = 0, last_order = 0;
    for (int j = 0; j < CCD_geom->ext_nb; j++) {
        last_order = last_order + inst_config->orders_nb[fibre*CCD_geom->ext_nb+j];
        first_order = last_order - inst_config->orders_nb[fibre*CCD_geom->ext_nb+j];
        cpl_table_set(drift_table, "first_order", j, first_order);
        cpl_table_set(drift_table, "last_order", j, (last_order-1));
        // Print in convention orders starting at 1 not at 0
        //espdr_msg("EXT %d: first order: %d, last order: %d",
        //          j, first_order+1, last_order);
    }
    if (CCD_geom->ext_nb == 1) {
        *first_order_second_ext = last_order + 1;
    } else {
        *first_order_second_ext = first_order + 1; // orders starting from 1, not from 0
    }
    //espdr_msg("first order 2nd ext: %d", *first_order_second_ext);
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Measure instrumental drift on wavelength calibration spectra THAR
            and FP, does not requires line tables parameters.
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_measure_drift(cpl_image *flux,
                                   cpl_image *err,
                                   cpl_image *qual,
                                   cpl_image *flux_ref,
                                   cpl_image *err_ref,
                                   cpl_image *qual_ref,
                                   cpl_image *ll,
                                   cpl_image *dll,
                                   char *drift_method,
                                   char *drift_space,
                                   double ksig,
                                   double max_flux_threshold,
                                   double min_flux_threshold,
                                   double cosmics_part,
                                   cpl_table **drift_results_RE,
                                   cpl_image **drift_matrix_RE) {

    int i, j, n, m, pir, det, neff, nx, ny, ndet, order, rejected;
    int *first_order, *last_order;
    double errtot2, chisq, lambda, dlambdadx;
    double drift, drift_mean, drift_mean_err, drift_slope_x, drift_slope_x_err, drift_slope_o, drift_slope_o_err;
    gsl_vector *y, *w, *p;
    gsl_matrix *A, *covar;
    double *fit, *res;
    gsl_multifit_linear_workspace *work;
    cpl_error_code my_error;
    
    nx = cpl_image_get_size_x(flux);
    ny = cpl_image_get_size_y(flux);
    ndet = cpl_table_get_nrow(*drift_results_RE);
    first_order = cpl_table_get_data_int(*drift_results_RE,"first_order");
    last_order = cpl_table_get_data_int(*drift_results_RE,"last_order");
    
    cpl_image *deriv;
    cpl_image *deriv_err;
    cpl_image *flux_norm = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);
    cpl_image *err_norm = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);
    cpl_image *qual_updated = cpl_image_new(nx,ny,CPL_TYPE_INT);
    double flux_ratio;
    double flux_ratio_err;
    
    my_error = espdr_discard_pixels(flux,
                                    err,
                                    qual,
                                    flux_ref,
                                    err_ref,
                                    qual_ref,
                                    max_flux_threshold,
                                    min_flux_threshold,
                                    cosmics_part,
                                    &qual_updated);
    
    if (my_error == CPL_ERROR_ILLEGAL_INPUT) {
        espdr_msg_warning("Caught ILLEGAL INPUT from espdr_discard_pixels(), skipping the drift computation");
        cpl_image_delete(flux_norm);
        cpl_image_delete(err_norm);
        return (CPL_ERROR_ILLEGAL_INPUT);
    }
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_discard_pixels failed: %s",
                 cpl_error_get_message_default(my_error));

    //espdr_msg("---> DSO DEBUG: method: %s, space: %s, ksig: %f, max flux: %f, min flux: %f",
    //          drift_method, drift_space, ksig, max_flux_threshold, min_flux_threshold);
    
    /*
    espdr_msg("---> ASE DEBUG : flux x = %lld",cpl_image_get_size_x(flux));
    espdr_msg("---> ASE DEBUG : flux y = %lld",cpl_image_get_size_y(flux));
    espdr_msg("---> ASE DEBUG : err x = %lld",cpl_image_get_size_x(err));
    espdr_msg("---> ASE DEBUG : err y = %lld",cpl_image_get_size_y(err));
    espdr_msg("---> ASE DEBUG : qual x = %lld",cpl_image_get_size_x(qual));
    espdr_msg("---> ASE DEBUG : qual y = %lld",cpl_image_get_size_y(qual));
    espdr_msg("---> ASE DEBUG : flux_ref x = %lld",cpl_image_get_size_x(flux_ref));
    espdr_msg("---> ASE DEBUG : flux_ref y = %lld",cpl_image_get_size_y(flux_ref));
    espdr_msg("---> ASE DEBUG : err_ref x = %lld",cpl_image_get_size_x(err_ref));
    espdr_msg("---> ASE DEBUG : err_ref y = %lld",cpl_image_get_size_y(err_ref));
    espdr_msg("---> ASE DEBUG : qual_ref x = %lld",cpl_image_get_size_x(qual_ref));
    espdr_msg("---> ASE DEBUG : qual_ref y = %lld",cpl_image_get_size_y(qual_ref));
    espdr_msg("---> ASE DEBUG : ll x = %lld",cpl_image_get_size_x(ll));
    espdr_msg("---> ASE DEBUG : ll y = %lld",cpl_image_get_size_y(ll));
    espdr_msg("---> ASE DEBUG : dll x = %lld",cpl_image_get_size_x(dll));
    espdr_msg("---> ASE DEBUG : dll y = %lld",cpl_image_get_size_y(dll));
    */
    
    deriv = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);
    deriv_err = cpl_image_new(nx,ny,CPL_TYPE_DOUBLE);
    my_error = espdr_compute_derivative(flux_ref,
                                        err_ref,
                                        qual_ref,
                                        &deriv,
                                        &deriv_err,
                                        &qual_updated);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_derivative failed: %s",
                 cpl_error_get_message_default(my_error));
    
    if (strcmp(drift_space,"velocity")==0)
        my_error = espdr_convert_derivative(&deriv,
                                            &deriv_err,
                                            ll,
                                            dll);
    
    my_error = espdr_normalize_spectrum(flux,
                                        err,
                                        flux_ref,
                                        err_ref,
                                        qual_updated,
                                        &flux_norm,
                                        &err_norm,
                                        &flux_ratio,
                                        &flux_ratio_err);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_normalize_spectrum failed: %s",
                 cpl_error_get_message_default(my_error));
    
    for (det = 0; det < ndet; det++) {
        
        if(strcmp(drift_method,"flux_global_drift_global_combined_fit")==0) {
            
            n = nx*(last_order[det]-first_order[det]+1);
            m = 2;
            y = gsl_vector_alloc (n);
            w = gsl_vector_alloc (n);
            p = gsl_vector_alloc (m);
            fit = (double *) malloc(n*sizeof(double));
            res = (double *) malloc(n*sizeof(double));
            A = gsl_matrix_alloc (n, m);
            covar = gsl_matrix_alloc (m, m);
            
            for (i = 0; i < n; i++) {
                order = (int)(i/nx)+first_order[det];
                j = i%nx;
                gsl_vector_set (y, i,
                                cpl_image_get(flux,j+1,order+1,&pir));
                errtot2 =
                pow(cpl_image_get(err,j+1,order+1,&pir),2.) +
                pow(cpl_image_get(err_ref,j+1,order+1,&pir),2.);
                
                if (cpl_image_get(qual_updated,j+1,order+1,&pir) != 0) {
                    gsl_vector_set (w, i, 0.);
                } else {
                    gsl_vector_set (w,i,1./errtot2);
                }
                
                gsl_matrix_set (A, i, 0,
                                cpl_image_get(flux_ref,j+1,order+1,&pir));
                gsl_matrix_set (A, i, 1, -1.*cpl_image_get(deriv,j+1,order+1,&pir));
            }
            
            work = gsl_multifit_linear_alloc (n, m);
            gsl_multifit_wlinear (A, w, y, p, covar, &chisq, work);
            gsl_multifit_linear_free (work);
            
            rejected = 0;
            neff = 0;
            
            for (i = 0; i < n; i++) {
                fit[i] = 0.;
                for (j = 0; j < m; j++)
                    fit[i] += gsl_matrix_get (A, i, j) * gsl_vector_get (p, j);
                res[i] = (gsl_vector_get(y, i)-fit[i])*sqrt(gsl_vector_get(w, i));
                if (fabs(res[i]) > ksig) {
                    gsl_vector_set (w, i, 0.);
                    rejected++;
                }
                if(gsl_vector_get(w,i) != 0.) neff++;
            }
            
            work = gsl_multifit_linear_alloc (n, m);
            gsl_multifit_wlinear (A, w, y, p, covar, &chisq, work);
            gsl_multifit_linear_free (work);
            
            chisq = chisq/(neff-m);
            
            drift_mean = gsl_vector_get(p, 1)/gsl_vector_get(p, 0);
            drift_mean_err = sqrt(gsl_matrix_get(covar, 1, 1)/
                                  pow(gsl_vector_get(p, 0),2.)+gsl_matrix_get(covar, 0, 0)*
                                  pow(gsl_vector_get(p, 1),2.)/pow(gsl_vector_get(p, 0),4.));
            
            cpl_table_set(*drift_results_RE,"flux_ratio",det,gsl_vector_get(p, 0));
            cpl_table_set(*drift_results_RE,"flux_ratio_err",det,sqrt(gsl_matrix_get(covar, 0, 0)));
            cpl_table_set(*drift_results_RE,"drift_mean",det,drift_mean);
            cpl_table_set(*drift_results_RE,"drift_mean_err",det,drift_mean_err);
            cpl_table_set(*drift_results_RE,"drift_slope_x",det,0.);
            cpl_table_set(*drift_results_RE,"drift_slope_x_err",det,0.);
            cpl_table_set(*drift_results_RE,"drift_slope_o",det,0.);
            cpl_table_set(*drift_results_RE,"drift_slope_o_err",det,0.);
            cpl_table_set(*drift_results_RE,"chisq",det,chisq);
            cpl_table_set_int(*drift_results_RE,"rejected",det,rejected);
            
            if (strcmp(drift_space,"velocity")==0) {
                for (i = 0; i < n; i++) {
                    order = (int)(i/nx)+first_order[det];
                    j = i%nx;
                    lambda = cpl_image_get(ll, j+1, order+1, &pir);
                    cpl_image_set(*drift_matrix_RE, j+1, order+1, drift_mean/2.99792458e8*lambda);
                }
                
            } else {
                for (i = 0; i < n; i++) {
                    order =
                    (int)(i/nx)+first_order[det];
                    j = i%nx;
                    dlambdadx = cpl_image_get(dll, j+1, order+1, &pir);
                    cpl_image_set(*drift_matrix_RE, j+1, order+1, drift_mean*dlambdadx);
                }
            }
            
            free(fit);
            free(res);
            gsl_vector_free(y); gsl_vector_free(w);
            gsl_vector_free(p);
            gsl_matrix_free(A); gsl_matrix_free(covar);
            
        } else if(strcmp(drift_method,"flux_global_drift_local_combined_fit")==0) {
            
            n = nx*(last_order[det]-first_order[det]+1);
            m = 4;
            y = gsl_vector_alloc (n);
            w = gsl_vector_alloc (n);
            p = gsl_vector_alloc (m);
            fit = (double *) malloc(n*sizeof(double));
            res = (double *) malloc(n*sizeof(double));
            A = gsl_matrix_alloc (n, m);
            covar = gsl_matrix_alloc (m, m);
            
            for (i = 0; i < n; i++) {
                order = (int)(i/nx)+first_order[det];
                j = i%nx;
                gsl_vector_set (y, i, cpl_image_get(flux,j+1,order+1,&pir));
                errtot2 =
                pow(cpl_image_get(err,j+1,order+1,&pir),2.)+
                pow(cpl_image_get(err_ref,j+1,order+1,&pir),2.);
                
                if (cpl_image_get(qual_updated,j+1,order+1,&pir) != 0) {
                    gsl_vector_set (w, i, 0.);
                } else {
                    gsl_vector_set (w, i, 1./errtot2);
                }
                
                gsl_matrix_set (A, i, 0, cpl_image_get(flux_ref,j+1,order+1,&pir));
                gsl_matrix_set (A, i, 1, -1.*cpl_image_get(deriv,j+1,order+1,&pir));
                gsl_matrix_set (A, i, 2, -1.*cpl_image_get(deriv,j+1,order+1,&pir)*(j-nx/2));
                gsl_matrix_set (A, i, 3, -1.*cpl_image_get(deriv,j+1,order+1,&pir)*(i/nx-n/nx/2));
            }
            
            work = gsl_multifit_linear_alloc (n, m);
            gsl_multifit_wlinear (A, w, y, p, covar, &chisq, work);
            gsl_multifit_linear_free (work);
            
            rejected = 0;
            neff = 0;
            
            for (i = 0; i < n; i++) {
                fit[i] = 0.;
                for (j = 0; j < m; j++)
                    fit[i] += gsl_matrix_get (A, i, j) * gsl_vector_get (p, j);
                res[i] = (gsl_vector_get(y, i)-fit[i])*sqrt(gsl_vector_get(w, i));
                if (fabs(res[i]) > ksig) {
                    gsl_vector_set (w, i, 0.);
                    rejected++;
                }
                if(gsl_vector_get(w,i) != 0.) neff++;
            }
            
            work = gsl_multifit_linear_alloc (n, m);
            gsl_multifit_wlinear (A, w, y, p, covar, &chisq, work);
            gsl_multifit_linear_free (work);
            
            chisq = chisq/(neff-m);
            
            drift_mean = gsl_vector_get(p, 1)/gsl_vector_get(p, 0);
            drift_mean_err = sqrt(gsl_matrix_get(covar, 1, 1)/
                                  pow(gsl_vector_get(p, 0),2.)+gsl_matrix_get(covar, 0, 0)*
                                  pow(gsl_vector_get(p, 1),2.)/pow(gsl_vector_get(p, 0),4.));
            drift_slope_x = gsl_vector_get(p, 2)/gsl_vector_get(p, 0);
            drift_slope_x_err = sqrt(gsl_matrix_get(covar, 2, 2)/
                                     pow(gsl_vector_get(p, 0),2.)+gsl_matrix_get(covar, 0, 0)*
                                     pow(gsl_vector_get(p, 2),2.)/pow(gsl_vector_get(p, 0),4.));
            drift_slope_o = gsl_vector_get(p, 3)/gsl_vector_get(p, 0);
            drift_slope_o_err = sqrt(gsl_matrix_get(covar, 3, 3)/
                                     pow(gsl_vector_get(p, 0),2.)+gsl_matrix_get(covar, 0, 0)*
                                     pow(gsl_vector_get(p, 3),2.)/pow(gsl_vector_get(p, 0),4.));
            
            /*espdr_msg("ASE DEBUG ---> n=%d",n);
             espdr_msg("ASE DEBUG ---> first_order=%d",first_order[det]);
             espdr_msg("ASE DEBUG ---> last_order=%d",last_order[det]);
             espdr_msg("ASE DEBUG ---> flux_ratio=%f",gsl_vector_get(p, 0));
             espdr_msg("ASE DEBUG ---> flux_ratio_err=%f",sqrt(gsl_matrix_get(covar, 0, 0)));
             espdr_msg("ASE DEBUG ---> drift_mean=%f",drift_mean);
             espdr_msg("ASE DEBUG ---> drift_mean_err=%f",drift_mean_err);
             espdr_msg("ASE DEBUG ---> drift_slope_x=%f",drift_slope_x);
             espdr_msg("ASE DEBUG ---> drift_slope_x_err=%f",drift_slope_x_err);
             espdr_msg("ASE DEBUG ---> drift_slope_o=%f",drift_slope_o);
             espdr_msg("ASE DEBUG ---> drift_slope_o_err=%f",drift_slope_o_err);
             espdr_msg("ASE DEBUG ---> chisq=%f",chisq);
             espdr_msg("ASE DEBUG ---> rejected=%f",rejected);*/
            
            cpl_table_set(*drift_results_RE,"flux_ratio",det,gsl_vector_get(p, 0));
            cpl_table_set(*drift_results_RE,"flux_ratio_err",det,sqrt(gsl_matrix_get(covar, 0, 0)));
            cpl_table_set(*drift_results_RE,"drift_mean",det,drift_mean);
            cpl_table_set(*drift_results_RE,"drift_mean_err",det,drift_mean_err);
            cpl_table_set(*drift_results_RE,"drift_slope_x",det,drift_slope_x);
            cpl_table_set(*drift_results_RE,"drift_slope_x_err",det,drift_slope_x_err);
            cpl_table_set(*drift_results_RE,"drift_slope_o",det,drift_slope_o);
            cpl_table_set(*drift_results_RE,"drift_slope_o_err",det,drift_slope_o_err);
            
            cpl_table_set(*drift_results_RE,"chisq",det,chisq);
            cpl_table_set_int(*drift_results_RE,"rejected",det,rejected);
            
            if (strcmp(drift_space,"velocity")==0) {
                for (i = 0; i < n; i++) {
                    order = (int)(i/nx)+first_order[det];
                    j = i%nx;
                    drift = drift_mean +
                    drift_slope_x*(j-nx/2) +
                    drift_slope_o*(i/nx-n/nx/2);
                    lambda = cpl_image_get(ll, j+1, order+1, &pir);
                    cpl_image_set(*drift_matrix_RE,
                                  j+1, order+1, drift/2.99792458e8*lambda);
                }
                
            } else {
                for (i = 0; i < n; i++) {
                    order = (int)(i/nx)+first_order[det];
                    j = i%nx;
                    drift = drift_mean +
                    drift_slope_x*(j-nx/2) +
                    drift_slope_o*(i/nx-n/nx/2);
                    dlambdadx = cpl_image_get(dll, j+1, order+1, &pir);
                    cpl_image_set(*drift_matrix_RE, j+1, order+1, drift*dlambdadx);
                }
            }
            
            free(fit);
            free(res);
            gsl_vector_free(y); gsl_vector_free(w);
            gsl_vector_free(p);
            gsl_matrix_free(A); gsl_matrix_free(covar);
            
        } else if(strcmp(drift_method,"flux_global_drift_global_sequential_fit")==0) {
            
            n = nx*(last_order[det]-first_order[det]+1);
            m = 1;
            y = gsl_vector_alloc (n);
            w = gsl_vector_alloc (n);
            p = gsl_vector_alloc (m);
            fit = (double *) malloc(n*sizeof(double));
            res = (double *) malloc(n*sizeof(double));
            A = gsl_matrix_alloc (n, m);
            covar = gsl_matrix_alloc (m, m);
            
            for (i = 0; i < n; i++) {
                order = (int)(i/nx)+first_order[det];
                j = i%nx;
                gsl_vector_set(y,i,cpl_image_get(flux_norm,j+1,order+1,&pir)-
                               cpl_image_get(flux_ref,j+1,order+1,&pir));
                errtot2 =
                pow(cpl_image_get(err_norm,j+1,order+1,&pir),2.)+
                pow(cpl_image_get(err_ref,j+1,order+1,&pir),2.);
                
                if (cpl_image_get(qual_updated,j+1,order+1,&pir) != 0) {
                    gsl_vector_set (w, i, 0.);
                } else {
                    gsl_vector_set (w, i, 1./errtot2);
                }
                
                gsl_matrix_set (A, i, 0, -1.*cpl_image_get(deriv,j+1,order+1,&pir));
            }
            
            work = gsl_multifit_linear_alloc (n, m);
            
            gsl_multifit_wlinear (A, w, y, p, covar, &chisq, work);
            gsl_multifit_linear_free (work);
           
            rejected = 0;
            neff = 0;
            
            for (i = 0; i < n; i++) {
                fit[i] = 0.;
                for (j = 0; j < m; j++)
                    fit[i] += gsl_matrix_get(A,i,j)*
                    gsl_vector_get(p,j);
                res[i] = (gsl_vector_get(y, i)-fit[i])*sqrt(gsl_vector_get(w, i));
                if (fabs(res[i]) > ksig) {
                    gsl_vector_set (w, i, 0.);
                    rejected++;
                }
                if(gsl_vector_get(w,i) != 0.) neff++;
            }
            
            work = gsl_multifit_linear_alloc (n, m);
            gsl_multifit_wlinear (A, w, y, p, covar, &chisq, work);
            gsl_multifit_linear_free (work);
            
            chisq = chisq/(neff-m);
            
            drift_mean = gsl_vector_get(p, 0);
            drift_mean_err = sqrt(gsl_matrix_get(covar, 0, 0));
            drift_slope_x = 0.;
            drift_slope_x_err = 0.;
            drift_slope_o = 0.;
            drift_slope_o_err = 0.;
           
            espdr_msg("Drift measured = %.4e",drift_mean);
 
            cpl_table_set(*drift_results_RE,"flux_ratio",det,flux_ratio);
            cpl_table_set(*drift_results_RE,"flux_ratio_err",det,flux_ratio_err);
            cpl_table_set(*drift_results_RE,"drift_mean",det,drift_mean);
            cpl_table_set(*drift_results_RE,"drift_mean_err",det,drift_mean_err);
            cpl_table_set(*drift_results_RE,"drift_slope_x",det,drift_slope_x);
            cpl_table_set(*drift_results_RE,"drift_slope_x_err",det,drift_slope_x_err);
            cpl_table_set(*drift_results_RE,"drift_slope_o",det,drift_slope_o);
            cpl_table_set(*drift_results_RE,"drift_slope_o_err",det,drift_slope_o_err);
            cpl_table_set(*drift_results_RE,"chisq",det,chisq);
            cpl_table_set_int(*drift_results_RE,"rejected",det,rejected);
            
            /*
             espdr_msg("----> DSO DEBUG: DET: %d", det);
             espdr_msg("----> DSO DEBUG: flux_ratio:       %f", flux_ratio);
             espdr_msg("----> DSO DEBUG: flux_ratio_err:   %f", flux_ratio_err);
             espdr_msg("----> DSO DEBUG: drift_mean:       %f", drift_mean);
             espdr_msg("----> DSO DEBUG: drift_mean_err:   %f", drift_mean_err);
             espdr_msg("----> DSO DEBUG: drift_slope_o:    %f", drift_slope_o);
             espdr_msg("----> DSO DEBUG: drift_slope_o_er: %f", drift_slope_o_err);
             espdr_msg("----> DSO DEBUG: drift_slope_x:    %f", drift_slope_x);
             espdr_msg("----> DSO DEBUG: drift_slope_x_er: %f", drift_slope_x_err);
             espdr_msg("----> DSO DEBUG: chi2:             %f", chisq);
             espdr_msg("----> DSO DEBUG: rejected:         %d", rejected);
            */
            
            if (strcmp(drift_space,"velocity")==0) {
                for (i = 0; i < n; i++) {
                    order = (int)(i/nx)+first_order[det];
                    j = i%nx;
                    drift = drift_mean +
                    drift_slope_x*(j-nx/2) +
                    drift_slope_o*(i/nx-n/nx/2);
                    lambda = cpl_image_get(ll, j+1, order+1, &pir);
                    cpl_image_set(*drift_matrix_RE, j+1, order+1, drift/2.99792458e8*lambda);
                }
                
            } else {
                for (i = 0; i < n; i++) {
                    order = (int)(i/nx)+first_order[det];
                    j = i%nx;
                    drift = drift_mean +
                    drift_slope_x*(j-nx/2) +
                    drift_slope_o*(i/nx-n/nx/2);
                    dlambdadx = cpl_image_get(dll, j+1, order+1, &pir);
                    cpl_image_set(*drift_matrix_RE, j+1, order+1, drift*dlambdadx);
                }
            }
            
            free(fit);
            free(res);
            gsl_vector_free(y); gsl_vector_free(w);
            gsl_vector_free(p);
            gsl_matrix_free(A); gsl_matrix_free(covar);
            
        } else if(strcmp(drift_method,"flux_global_drift_local_sequential_fit")==0) {
            
            n = nx*(last_order[det]-first_order[det]+1);
            m = 3;
            y = gsl_vector_alloc (n);
            w = gsl_vector_alloc (n);
            p = gsl_vector_alloc (m);
            fit = (double *) malloc(n*sizeof(double));
            res = (double *) malloc(n*sizeof(double));
            A = gsl_matrix_alloc (n, m);
            covar = gsl_matrix_alloc (m, m);
            
            for (i = 0; i < n; i++) {
                order = (int)(i/nx)+first_order[det];
                j = i%nx;
                gsl_vector_set (y, i,
                                cpl_image_get(flux_norm,j+1,order+1,&pir)-
                                cpl_image_get(flux_ref,j+1,order+1,&pir));
                errtot2 =
                pow(cpl_image_get(err_norm,j+1,order+1,&pir),2.)+
                pow(cpl_image_get(err_ref,j+1,order+1,&pir),2.);
                
                if(cpl_image_get(qual_updated,j+1,order+1,&pir) != 0) {
                    gsl_vector_set (w, i, 0.);
                } else {
                    gsl_vector_set (w, i, 1./errtot2);
                }
                
                gsl_matrix_set (A, i, 0, -1.*cpl_image_get(deriv,j+1,order+1,&pir));
                gsl_matrix_set (A, i, 1, -1.*cpl_image_get(deriv,j+1,order+1,&pir)*(j-nx/2));
                gsl_matrix_set (A, i, 2, -1.*cpl_image_get(deriv,j+1,order+1,&pir)*(i/nx-n/nx/2));
            }
            
            work = gsl_multifit_linear_alloc (n, m);
            gsl_multifit_wlinear (A, w, y, p, covar, &chisq, work);
            gsl_multifit_linear_free (work);
            
            rejected = 0;
            neff = 0;
            
            for (i = 0; i < n; i++) {
                fit[i] = 0.;
                for (j = 0; j < m; j++)
                    fit[i] += gsl_matrix_get (A, i, j) * gsl_vector_get (p, j);
                res[i] = (gsl_vector_get(y, i)-fit[i])*sqrt(gsl_vector_get(w, i));
                if (fabs(res[i]) > ksig) {
                    gsl_vector_set (w, i, 0.);
                    rejected++;
                }
                if(gsl_vector_get(w,i) != 0.) neff++;
            }
            
            work = gsl_multifit_linear_alloc (n, m);
            gsl_multifit_wlinear (A, w, y, p, covar, &chisq, work);
            gsl_multifit_linear_free (work);
            
            chisq = chisq/(neff-m);
            
            drift_mean = gsl_vector_get(p, 0);
            drift_mean_err = sqrt(gsl_matrix_get(covar, 0, 0));
            drift_slope_x = gsl_vector_get(p, 1);
            drift_slope_x_err = sqrt(gsl_matrix_get(covar, 1, 1));
            drift_slope_o = gsl_vector_get(p, 2);
            drift_slope_o_err = sqrt(gsl_matrix_get(covar, 2, 2));
            
            
            cpl_table_set(*drift_results_RE,"flux_ratio",det,flux_ratio);
            cpl_table_set(*drift_results_RE,"flux_ratio_err",det,flux_ratio_err);
            cpl_table_set(*drift_results_RE,"drift_mean",det,drift_mean);
            cpl_table_set(*drift_results_RE,"drift_mean_err",det,drift_mean_err);
            cpl_table_set(*drift_results_RE,"drift_slope_x",det,drift_slope_x);
            cpl_table_set(*drift_results_RE,"drift_slope_x_err",det,drift_slope_x_err);
            cpl_table_set(*drift_results_RE,"drift_slope_o",det,drift_slope_o);
            cpl_table_set(*drift_results_RE,"drift_slope_o_err",det,drift_slope_o_err);
            cpl_table_set(*drift_results_RE,"chisq",det,chisq);
            cpl_table_set_int(*drift_results_RE,"rejected",det,rejected);
            
            if (strcmp(drift_space,"velocity")==0) {
                for (i = 0; i < n; i++) {
                    order = (int)(i/nx)+first_order[det];
                    j = i%nx;
                    drift = drift_mean +
                    drift_slope_x*(j-nx/2) +
                    drift_slope_o*(i/nx-n/nx/2);
                    lambda = cpl_image_get(ll, j+1, order+1, &pir);
                    cpl_image_set(*drift_matrix_RE, j+1, order+1, drift/2.99792458e8*lambda);
                }
                
            } else {
                for (i = 0; i < n; i++) {
                    order =
                    (int)(i/nx)+first_order[det];
                    j = i%nx;
                    drift = drift_mean +
                    drift_slope_x*(j-nx/2) +
                    drift_slope_o*(i/nx-n/nx/2);
                    dlambdadx = cpl_image_get(dll, j+1, order+1, &pir);
                    cpl_image_set(*drift_matrix_RE, j+1, order+1, drift*dlambdadx);
                }
            }
            
            free(fit);
            free(res);
            gsl_vector_free(y); gsl_vector_free(w);
            gsl_vector_free(p);
            gsl_matrix_free(A); gsl_matrix_free(covar);
            
            
        } else if(strcmp(drift_method,"flux_local_drift_global_sequential_fit")==0) {
            
            
        } else if(strcmp(drift_method,"flux_local_drift_local_sequential_fit")==0) {
            
            
        } else {
            espdr_msg("ERROR : Wrong drift method -%s-",drift_method);
            exit(EXIT_FAILURE);
        }
    }
    
    cpl_image_delete(deriv);
    cpl_image_delete(deriv_err);
    cpl_image_delete(qual_updated);
    cpl_image_delete(flux_norm);
    cpl_image_delete(err_norm);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Normalise spectrum
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_normalize_spectrum(cpl_image *flux,
                                        cpl_image *err,
                                        cpl_image *flux_ref,
                                        cpl_image *err_ref,
                                        cpl_image *qual_updated,
                                        cpl_image **norm_flux_RE,
                                        cpl_image **norm_err_RE,
                                        double *flux_ratio,
                                        double *flux_ratio_err
                                        ) {
    
    int i,j,pir,nx,ny,k;
    double flux_sum = 0.;
    double flux_ref_sum = 0.;
    double flux_sum_order = 0.;
    double flux_ref_sum_order = 0.;
    double err2_ref_sum = 0.;
    double err2_sum = 0.;
    nx = cpl_image_get_size_x(flux);
    ny = cpl_image_get_size_y(flux);
    
    double *norm_flux_RE_data =
    cpl_image_get_data_double(*norm_flux_RE);
    double *norm_err_RE_data =
    cpl_image_get_data_double(*norm_err_RE);
    
    for (i = 0; i < ny; i++) {
        flux_sum_order = 0.;
        flux_ref_sum_order = 0.;
        for (j = 0; j < nx; j++) {
            if(cpl_image_get(qual_updated,j+1,i+1,&pir) == GOOD_PIXEL) {
                flux_sum += cpl_image_get(flux,j+1,i+1,&pir);
                flux_ref_sum += cpl_image_get(flux_ref,j+1,i+1,&pir);
                err2_sum += pow(cpl_image_get(err,j+1,i+1,&pir),2.0);
                err2_ref_sum += pow(cpl_image_get(err_ref,j+1,i+1,&pir),2.0);
                flux_sum_order += cpl_image_get(flux,j+1,i+1,&pir);
                flux_ref_sum_order += cpl_image_get(flux_ref,j+1,i+1,&pir);
            }
        }
        if (flux_ref_sum_order == 0.0) {
            espdr_msg_warning(ANSI_COLOR_RED"WARNING: The flux of the reference S2D spectrum is 0.0 for order %d, possibly no good pixels found"ANSI_COLOR_RESET, i+1);
            //return(CPL_ERROR_ILLEGAL_INPUT);
            }
        else {
            for (j = 0; j < nx; j++) {
                k = nx*i+j;
                norm_flux_RE_data[k] = cpl_image_get(flux,j+1,i+1,&pir)/
                                        flux_sum_order*flux_ref_sum_order;
                norm_err_RE_data[k] = cpl_image_get(err,j+1,i+1,&pir)/
                                        flux_sum_order*flux_ref_sum_order;
            }
        }
    }
    
    if (flux_ref_sum == 0.0) {
        espdr_msg_warning(ANSI_COLOR_RED"WARNING: The flux of the reference S2D spectrum is 0.0, possibly no good pixels found"ANSI_COLOR_RESET);
        return(CPL_ERROR_ILLEGAL_INPUT);
    }
    
    *flux_ratio = flux_sum/flux_ref_sum;
    *flux_ratio_err = sqrt(err2_sum/pow(flux_ref_sum,2.0)+
                           err2_ref_sum*pow(flux_sum,2.0)/pow(flux_ref_sum,4.0));
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief     Discard pixels
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_discard_pixels(cpl_image *flux,
                                    cpl_image *err,
                                    cpl_image *qual,
                                    cpl_image *flux_ref,
                                    cpl_image *err_ref,
                                    cpl_image *qual_ref,
                                    double max_flux_threshold,
                                    double min_flux_threshold,
                                    double cosmics_part,
                                    cpl_image **qual_updated_RE) {
    
    int i, j, pir, nx, ny, k, k_min, k_max;
    double max_flux_no_cosmics = 0.0;
    nx = cpl_image_get_size_x(flux);
    ny = cpl_image_get_size_y(flux);
    cpl_error_code my_error;
    int window = 12, bad_pix_nb = 0;
    
    my_error = cpl_image_or(*qual_updated_RE,qual,qual_ref);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "cpl_image_or failed: %s",
                 cpl_error_get_message_default(my_error));
    
    my_error = espdr_get_high_flux_rejection_bound(flux, qual,
                                                   cosmics_part*500.0,
                                                   &max_flux_no_cosmics);
    //espdr_msg("Max flux: %f, max flux no cosmics: %f, cosmics part: %f",
    //          cpl_image_get_max(flux),
    //          max_flux_no_cosmics, cosmics_part*500.0);
    
    espdr_ensure(max_flux_no_cosmics < min_flux_threshold,
                 CPL_ERROR_ILLEGAL_INPUT,
                 "Flux too low for drift computation, abandoning espdr_discard_pixels()");

    if (min_flux_threshold == 0.0) {
        espdr_msg("DRIFT: removing pixels neighbouring high flux ones (+/- %d pixels)", window);
    } else {
        espdr_msg("DRIFT: removing only high flux pixels, not their neighbours");
    }
    
    for (i = 0; i < ny; i++) {
        for (j = 0; j < nx; j++) {
            if (cpl_image_get(flux,j+1,i+1,&pir) > max_flux_threshold ||
                cpl_image_get(flux,j+1,i+1,&pir) < min_flux_threshold ||
                cpl_image_get(flux_ref,j+1,i+1,&pir) > max_flux_threshold ||
                cpl_image_get(flux_ref,j+1,i+1,&pir) < min_flux_threshold) {
                cpl_image_set(*qual_updated_RE,j+1,i+1,OTHER_BAD_PIXEL);
                bad_pix_nb++;
            }
            
            if (min_flux_threshold == 0.0) {
                if (cpl_image_get(flux,j+1,i+1,&pir) > max_flux_threshold ||
                    cpl_image_get(flux_ref,j+1,i+1,&pir) > max_flux_threshold) {
                    k_min = ESPDR_MAX(0, j-window);
                    k_max = ESPDR_MIN(j+window, nx);
                    for (k = k_min; k < k_max; k++) {
                        cpl_image_set(*qual_updated_RE,k+1,i+1,OTHER_BAD_PIXEL);
                        bad_pix_nb++;
                    }
                }
            }
        }
    }
    
    espdr_msg("DRIFT: number of bad pixels flagged: %d", bad_pix_nb);
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief     Compute derivative
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_derivative(cpl_image *flux,
                                        cpl_image *err,
                                        cpl_image *qual,
                                        cpl_image **derivative_RE,
                                        cpl_image **derivative_err_RE,
                                        cpl_image **qual_updated_RE) {
    
    int i,j,pir,nx,ny,k;
    nx = cpl_image_get_size_x(flux);
    ny = cpl_image_get_size_y(flux);
    
    double *derivative_RE_data = cpl_image_get_data_double(*derivative_RE);
    double *derivative_err_RE_data = cpl_image_get_data_double(*derivative_err_RE);
    int *qual_updated_RE_data = cpl_image_get_data_int(*qual_updated_RE);
    
    for (i = 0; i < ny; i++) {
        for (j = 1; j < nx-1; j++) {
            k = nx*i+j;
            
            derivative_RE_data[k] = (cpl_image_get(flux,j+2,i+1,&pir)-
                                     cpl_image_get(flux,j,i+1,&pir))/2.;
            
            derivative_err_RE_data[k] = sqrt(pow(cpl_image_get(err,j+2,i+1,&pir),2.0)
                                             +pow(cpl_image_get(err,j,i+1,&pir),2.0))/2.;
            
            if (cpl_image_get(qual,j+2,i+1,&pir)!=0 || cpl_image_get(qual,j,i+1,&pir)!=0) {
                qual_updated_RE_data[k] = OTHER_BAD_PIXEL;
            }
        }
        
        derivative_RE_data[nx*i] = derivative_RE_data[nx*i+1];
        derivative_RE_data[nx*(i+1)-1] = derivative_RE_data[nx*(i+1)-2];
        derivative_err_RE_data[nx*i] = derivative_err_RE_data[nx*i+1];
        derivative_err_RE_data[nx*(i+1)-1] = derivative_err_RE_data[nx*(i+1)-2];
        qual_updated_RE_data[nx*i] = OTHER_BAD_PIXEL;
        qual_updated_RE_data[nx*(i+1)-1] = OTHER_BAD_PIXEL;
    }
    
    return cpl_error_get_code();
}



/*----------------------------------------------------------------------------*/
/**
 @brief     Convert derivative
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_convert_derivative(cpl_image **deriv,
                                        cpl_image **deriv_err,
                                        cpl_image *ll,
                                        cpl_image *dll) {
    
    int i,j,pir,nx,ny;
    nx = cpl_image_get_size_x(*deriv);
    ny = cpl_image_get_size_y(*deriv);
    
    for (i = 0; i < ny; i++) {
        for (j = 0; j < nx; j++) {
            
            cpl_image_set(*deriv, j+1, i+1,
                          cpl_image_get(*deriv,j+1,i+1,&pir)/
                          cpl_image_get(dll,j+1,i+1,&pir)*
                          cpl_image_get(ll,j+1,i+1,&pir)/
                          2.99792458e8
                          );
            
            cpl_image_set(*deriv_err, j+1, i+1,
                          cpl_image_get(*deriv_err,j+1,i+1,&pir)/
                          cpl_image_get(dll,j+1,i+1,&pir)*
                          cpl_image_get(ll,j+1,i+1,&pir)/
                          2.99792458e8
                          );
            
        }
    }
    
    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Create drift matrices
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_create_drift_matrix(char *drift_space,
                                         int *nx,
                                         int *ny,
                                         int **physical_orders_id,
                                         espdr_inst_config *inst_config,
                                         espdr_CCD_geometry *CCD_geom,
                                         cpl_image **ll,
                                         cpl_image **dll,
                                         cpl_table **drift_results,
                                         cpl_image **drift_matrix_RE) {
    
    cpl_error_code my_error;
    
    int pir;
    int ndet = cpl_table_get_nrow(drift_results[0]);
    int *first_order_A = cpl_table_get_data_int(drift_results[0],"first_order");
    int *last_order_A = cpl_table_get_data_int(drift_results[0],"last_order");
    int *first_order_B = cpl_table_get_data_int(drift_results[1],"first_order");
    int *last_order_B = cpl_table_get_data_int(drift_results[1],"last_order");
    double *drift_mean = cpl_table_get_data_double(drift_results[1],"drift_mean");
    double *drift_slope_x = cpl_table_get_data_double(drift_results[1],"drift_slope_x");
    double *drift_slope_o = cpl_table_get_data_double(drift_results[1],"drift_slope_o");
    double *drift_A = NULL;
    double *drift_B = NULL;
    cpl_image *drift_order = NULL;
    int orders_nb_A[ndet];
    int orders_nb_B[ndet];
    double lambda, dlambdadx;
    bool found = false;
    int last_order_b_matched = -1, ext_a, ext_b;
    int fibre_A = 0;
    int fibre_B = 1;
    
    
    for (int det = 0; det < ndet; det++) {
        orders_nb_A[det] = last_order_A[det] - first_order_A[det] + 1;
        orders_nb_B[det] = last_order_B[det] - first_order_B[det] + 1;
        
        espdr_msg("fibre A: first order: %d, last order: %d, orders_nb: %d",
                  first_order_A[det]+1, last_order_A[det]+1, orders_nb_A[det]);
        espdr_msg("fibre B: first order: %d, last order: %d, orders_nb: %d",
                  first_order_B[det]+1, last_order_B[det]+1, orders_nb_B[det]);
    }
    
    espdr_msg("Sizes: A: %d x %d, B: %d x %d", nx[0], ny[0], nx[1], ny[1]);
    
    for (int det = 0; det < ndet; det++) {
        
        last_order_b_matched = -1;
        
        for (int order_a = first_order_A[det]; order_a < last_order_A[det]+1; order_a++) {
            ext_a = espdr_get_ext_index_for_order(order_a+1, fibre_A,
                                                  inst_config, CCD_geom);
            found = false;
            
            for (int order_b = first_order_B[det]; order_b < last_order_B[det]+1; order_b++) {
                ext_b = espdr_get_ext_index_for_order(order_b+1, fibre_B,
                                                      inst_config, CCD_geom);
                
                if ((physical_orders_id[fibre_A][order_a] == physical_orders_id[fibre_B][order_b]) &&
                    (ext_a == ext_b) && (!found) && (order_b > last_order_b_matched)) {
                    
                    last_order_b_matched = order_b; // in order not to much twice the same order from fibre B
                    found = true;                   // once order found, we don't look for another one
                    
                    // orders the same on A and B: copy from matrix B to matrix A
                    
                    drift_B = (double *)cpl_calloc(nx[1], sizeof(double));
                    for (int pixel = 0; pixel < nx[1]; pixel++) {
                        drift_B[pixel] = drift_mean[det] +
                                        drift_slope_x[det]*(pixel - nx[1]/2) +
                                        drift_slope_o[det]*(order_b - first_order_B[det] - orders_nb_B[det]/2);
                        if (strcmp(drift_space,"velocity") == 0) {
                            lambda = cpl_image_get(ll[1], pixel+1, order_b+1, &pir);
                            drift_B[pixel] = drift_B[pixel]/2.99792458e8*lambda;
                        } else {
                            dlambdadx = cpl_image_get(dll[1], pixel+1, order_b+1, &pir);
                            drift_B[pixel] = drift_B[pixel]*dlambdadx;
                        }
                    }
                    drift_order = cpl_image_wrap_double(nx[1], 1, drift_B);
                    my_error = cpl_image_copy(drift_matrix_RE[1], drift_order, 1, order_b + 1);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "cpl_image_copy with drift_order failed: %s",
                                 cpl_error_get_message_default(my_error));
                    cpl_image_unwrap(drift_order);
                    cpl_free(drift_B);
                    
                    drift_A = (double *)cpl_calloc(nx[0], sizeof(double));
                    for (int pixel = 0; pixel < nx[0]; pixel++) {
                        drift_A[pixel] = drift_mean[det] +
                                        drift_slope_x[det]*(pixel - nx[0]/2) +
                                        drift_slope_o[det]*(order_b - first_order_B[det] - orders_nb_B[det]/2);
                        if (strcmp(drift_space,"velocity") == 0) {
                            lambda = cpl_image_get(ll[0], pixel+1, order_a+1, &pir);
                            drift_A[pixel] = drift_A[pixel]/2.99792458e8*lambda;
                        } else {
                            dlambdadx = cpl_image_get(dll[0], pixel+1, order_a+1, &pir);
                            drift_A[pixel] = drift_A[pixel]*dlambdadx;
                        }
                    }
                    drift_order = cpl_image_wrap_double(nx[0], 1, drift_A);
                    my_error = cpl_image_copy(drift_matrix_RE[0], drift_order, 1, order_a + 1);
                    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                                 "cpl_image_copy with drift_order failed: %s",
                                 cpl_error_get_message_default(my_error));
                    cpl_image_unwrap(drift_order);
                    cpl_free(drift_A);
                }
            }
            
            if (!found) { /* order in A, but not in B fibre ==> unchanged */
                // No order on B, create A
                drift_A = (double *)cpl_calloc(nx[0], sizeof(double));
                if (last_order_b_matched == -1) { // these are orders at the beginning of the detector
                    for (int pixel = 0; pixel < nx[0]; pixel++) {
                        drift_A[pixel] = drift_mean[det] +
                                        drift_slope_x[det]*(pixel - nx[0]/2) +
                                        drift_slope_o[det]*(0 - first_order_B[det] - orders_nb_B[det]/2);
                        if (strcmp(drift_space,"velocity") == 0) {
                            lambda = cpl_image_get(ll[0], pixel+1, order_a+1, &pir);
                            drift_A[pixel] = drift_A[pixel]/2.99792458e8*lambda;
                        } else {
                            dlambdadx = cpl_image_get(dll[0], pixel+1, order_a+1, &pir);
                            drift_A[pixel] = drift_A[pixel]*dlambdadx;
                        }
                    }
                } else { // these are orders at the end of the detector
                    for (int pixel = 0; pixel < nx[0]; pixel++) {
                        drift_A[pixel] = drift_mean[det] +
                                        drift_slope_x[det]*(pixel - nx[0]/2) +
                                        drift_slope_o[det]*(last_order_B[det]);
                        if (strcmp(drift_space,"velocity") == 0) {
                            lambda = cpl_image_get(ll[0], pixel+1, order_a+1, &pir);
                            drift_A[pixel] = drift_A[pixel]/2.99792458e8*lambda;
                        } else {
                            dlambdadx = cpl_image_get(dll[0], pixel+1, order_a+1, &pir);
                            drift_A[pixel] = drift_A[pixel]*dlambdadx;
                        }
                    }
                }
                drift_order = cpl_image_wrap_double(nx[0], 1, drift_A);
                my_error = cpl_image_copy(drift_matrix_RE[0], drift_order, 1, order_a + 1);
                espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                             "cpl_image_copy with drift_order failed: %s",
                             cpl_error_get_message_default(my_error));
                cpl_image_unwrap(drift_order);
                cpl_free(drift_A);
            }
        }
    }
    
#if 0
    int j, n, order;
    for (int det = 0; det < ndet; det++) {
        n = nx[1]*(last_order[det]-first_order[det]+1);
        for (int i = 0; i < n; i++) {
            order = (int)(i/nx)+first_order[det];
            j = i%nx[1]; // == pixel
            drift = drift_mean + drift_slope_x*(j-nx[1]/2) + drift_slope_o*(i/nx[1]-n/nx[1]/2);
            if (strcmp(drift_space,"velocity") == 0) {
                lambda = cpl_image_get(ll, j+1, order+1, &pir);
                cpl_image_set(*drift_matrix_RE, j+1, order+1, drift/2.99792458e8*lambda);
            } else {
                dlambdadx = cpl_image_get(dll, j+1, order+1, &pir);
                cpl_image_set(*drift_matrix_RE, j+1, order+1, drift*dlambdadx);
            }
        }
    }
#endif
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Correct drift
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_drift(cpl_image *wave_matrix,
                                   cpl_image *drift_matrix,
                                   cpl_image **wave_matrix_RE) {
    
    int wnx = cpl_image_get_size_x(wave_matrix);
    int wny = cpl_image_get_size_y(wave_matrix);
    int dnx = cpl_image_get_size_x(drift_matrix);
    int dny = cpl_image_get_size_y(drift_matrix);
    
    //espdr_msg("Wave matrix size: %d x %d", wnx, wny);
    //espdr_msg("Drift matrix size: %d x %d", dnx, dny);
    
    espdr_ensure(((wnx != dnx) || (wny != dny)), CPL_ERROR_INCOMPATIBLE_INPUT,
                 "espdr_correct_drift: wave matrix and drift matrix have different sizes: %d x %d vs %d x %d",
                 wnx, wny, dnx, dny);
    
    *wave_matrix_RE = cpl_image_subtract_create(wave_matrix,drift_matrix);
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Check drift correction QC
 @return    CPL_ERROR_NONE if OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_check_drift_QC(cpl_table *drift_results,
                                    espdr_inst_config *inst_config,
                                    int ext_nb,
                                    int *drift_QC) {
    
    double drift_table_value_dbl;
    
    *drift_QC = 1;
    for (int det = 0; det < ext_nb; det++) {
        drift_table_value_dbl = cpl_table_get_double(drift_results, "flux_ratio", det, NULL);
        //espdr_msg("DRIFT flux ratio: %f", drift_table_value_dbl);
        if ((drift_table_value_dbl < inst_config->drift_min_flux_ratio) ||
            (drift_table_value_dbl > inst_config->drift_max_flux_ratio)) {
            espdr_msg(ANSI_COLOR_RED"WARNING: drift correction not applied: flux ratio beyond limits"ANSI_COLOR_RESET);
            *drift_QC = 0;
        }
        
        drift_table_value_dbl = cpl_table_get_double(drift_results, "chisq", det, NULL);
        //espdr_msg("DRIFT CHI2: %f", drift_table_value_dbl);
        if (drift_table_value_dbl > inst_config->drift_chi2_threshold) {
            espdr_msg(ANSI_COLOR_RED"WARNING: drift correction not applied: CHI2 too big"ANSI_COLOR_RESET);
            *drift_QC = 0;
        }
        
        drift_table_value_dbl = cpl_table_get_double(drift_results, "drift_mean", det, NULL);
        //espdr_msg("DRIFT mean: %f", drift_table_value_dbl);
        if (drift_table_value_dbl > inst_config->drift_mean_threshold) {
            espdr_msg(ANSI_COLOR_RED"WARNING: drift correction not applied: mean beyond limits"ANSI_COLOR_RESET);
            //espdr_msg("drift mean: %f\tmean limit: %f",
            //          drift_table_value_dbl, inst_config->drift_mean_threshold);
            *drift_QC = 0;
        }
        
        drift_table_value_dbl = cpl_table_get_double(drift_results, "drift_mean_err", det, NULL);
        //espdr_msg("DRIFT mean err: %f", drift_table_value_dbl);
        if (drift_table_value_dbl > inst_config->drift_mean_err_threshold) {
            espdr_msg(ANSI_COLOR_RED"WARNING: drift correction not applied: mean error beyond limits"ANSI_COLOR_RESET);
            *drift_QC = 0;
        }
    }
    
    return cpl_error_get_code();
}


/*----------------------------------------------------------------------------*/
/**
 @brief Save the drift QC KWs
 @param         drift_results       table with drift results
 @param         drift_chi2_check    returned check on the drift CHI2
 @param         inst_config         instrument config
 @param         qc_kws              QC KWs names
 @param[out]    keywords_RE         saved QC keywords
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_full_drift_QC(cpl_table *drift_results,
                                   espdr_inst_config *inst_config,
                                   espdr_CCD_geometry *CCD_geom,
                                   espdr_qc_keywords *qc_kws,
                                   int *drift_chi2_check,
                                   int *drift_flux_ratio_check,
                                   int *drift_mean_check,
                                   int *drift_mean_err_check,
                                   cpl_propertylist *keywords_RE) {

    char comment[COMMENT_LENGTH];
    char *drift_kw = NULL;
    
    if (drift_results != NULL) {
        
        double drift_table_value_dbl;
        int drift_table_value_int;
        sprintf(comment, " ");
        
        for (int i = 0; i < CCD_geom->ext_nb; i++) {
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_first_order_kw, i);
            
            drift_table_value_int = cpl_table_get_int(drift_results, "first_order", i, NULL);
            sprintf(comment, "first order nb on detector %d", i);
            espdr_keyword_add_int(drift_kw, drift_table_value_int+1, comment,
                                             &keywords_RE);
            cpl_free(drift_kw);
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_last_order_kw, i);
            
            drift_table_value_int = cpl_table_get_int(drift_results, "last_order", i, NULL);
            sprintf(comment, "last order nb on detector %d", i);
            espdr_keyword_add_int(drift_kw, drift_table_value_int+1, comment,
                                             &keywords_RE);
            cpl_free(drift_kw);
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_flux_ratio_kw, i);
            
            drift_table_value_dbl = cpl_table_get_double(drift_results, "flux_ratio", i, NULL);
            sprintf(comment, "flux ratio for detector %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            if ((drift_table_value_dbl < inst_config->drift_min_flux_ratio) ||
                (drift_table_value_dbl > inst_config->drift_max_flux_ratio)) {
                *drift_flux_ratio_check = 0;
            }
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_flux_ratio_err_kw, i);
            
            drift_table_value_dbl = cpl_table_get_double(drift_results, "flux_ratio_err", i, NULL);
            sprintf(comment, "flux ratio err for detector %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_mean_kw, i);
            
            drift_table_value_dbl = cpl_table_get_double(drift_results, "drift_mean", i, NULL);
            sprintf(comment, "mean drift [pxl] for det %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            if ((drift_table_value_dbl > inst_config->drift_mean_threshold) ||
                (drift_table_value_dbl == 0.0)) {
                *drift_mean_check = 0;
            }
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_mean_err_kw, i);
            drift_table_value_dbl = cpl_table_get_double(drift_results, "drift_mean_err", i, NULL);
            
            sprintf(comment, "mean drift err [pxl] for det %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            if ((drift_table_value_dbl > inst_config->drift_mean_err_threshold) ||
                (drift_table_value_dbl == 0.0)) {
                *drift_mean_err_check = 0;
            }
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_slope_x_kw, i);
            
            drift_table_value_dbl = cpl_table_get_double(drift_results, "drift_slope_x", i, NULL);
            sprintf(comment, "drift slope along the orders for det %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_slope_x_err_kw, i);
            drift_table_value_dbl = cpl_table_get_double(drift_results, "drift_slope_x_err", i, NULL);
            sprintf(comment, "drift slope err for det %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_slope_o_kw, i);
            drift_table_value_dbl = cpl_table_get_double(drift_results, "drift_slope_o", i, NULL);
            sprintf(comment, "drift slope across the orders for det %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_slope_o_err_kw, i);
            drift_table_value_dbl = cpl_table_get_double(drift_results, "drift_slope_o_err", i, NULL);
            sprintf(comment, "drift slope err for det %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_chisq_kw, i);
            drift_table_value_dbl = cpl_table_get_double(drift_results, "chisq", i, NULL);
            sprintf(comment, "drift fit CHI2 for det %d", i);
            espdr_keyword_add_double(drift_kw, drift_table_value_dbl, comment,
                                                &keywords_RE);
            cpl_free(drift_kw);
            
            if ((drift_table_value_dbl > inst_config->thar_drift_chi2_limit) ||
                (drift_table_value_dbl == 0.0)) {
                *drift_chi2_check = 0;
            }
            
            drift_kw = espdr_add_index_to_keyword(qc_kws->qc_drift_prefix_kw,
                                                  qc_kws->qc_drift_rejected_kw, i);
            
            drift_table_value_int = cpl_table_get_int(drift_results, "rejected", i, NULL);
            sprintf(comment, "nb of rejected pixels for det %d", i);
            espdr_keyword_add_int(drift_kw, drift_table_value_int, comment,
                                             &keywords_RE);
            cpl_free(drift_kw);
        }
    }
    
    return cpl_error_get_code();
}


