/*                                                                            *
 *   This file is part of the ESPRESSO Pipeline                               *
 *   Copyright (C) 2006 European Southern Observatory                         *
 *                                                                            *
 *   This library is free software; you can redistribute it and/or modify     *
 *   it under the terms of the GNU General Public License as published by     *
 *   the Free Software Foundation; either version 2 of the License, or        *
 *   (at your option) any later version.                                      *
 *                                                                            *
 *   This program is distributed in the hope that it will be useful,          *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 *   GNU General Public License for more details.                             *
 *                                                                            *
 *   You should have received a copy of the GNU General Public License        *
 *   along with this program; if not, write to the Free Software              *
 *   Foundation, 51 Franklin St, Fifth Floor, Boston, MA  02111-1307  USA     *
 *                                                                            */

/*
 * $Author: dsosnows $
 * $Date: 2023-05-22 16:52:15 $
 * $Revision: 1.6 $
 * $Name: not supported by cvs2svn $
 */

#include <espdr_tellurics.h>
#include <espdr_voigt.h>

/*----------------------------------------------------------------------------
 Functions code, functions to correct tellurics
 ----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 @brief     Compute and apply the tellurics correction
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static double espdr_compute_voigt_profile(double x,
                                          double hwmw,
                                          double gamma,
                                          double center) {
    
    double sigma = hwmw / (sqrt(2.0 * log(2.0)));
    
    return(voigt((x-center), sigma, gamma));
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Compute and apply the tellurics correction
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_correct_tellurics(const cpl_frameset* set,
                                       cpl_frameset* used_frames,
                                       espdr_inst_config *inst_config,
                                       cpl_image *s2d_blaze,
                                       cpl_image *s2d_error,
                                       cpl_image *s2d_qual,
                                       cpl_image *blaze,
                                       cpl_propertylist *keywords,
                                       cpl_image *wave_vacuum_shifted,
                                       cpl_image *dll_matrix,
                                       double rv_step,
                                       double berv,
                                       double berv_max,
                                       cpl_propertylist **tell_keywords_RE,
                                       cpl_image **telluric_spectrum_RE,
                                       cpl_image **s2d_tell_corr_RE,
                                       cpl_image **s2d_err_tell_corr_RE,
                                       cpl_image **s2d_qual_tell_corr_RE) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    cpl_image *res_map = NULL;
    cpl_frameset *hitran_qt_frames = cpl_frameset_new();
    cpl_frameset *hitran_strongest_frames = cpl_frameset_new();
    cpl_frameset *hitran_ccf_masks = cpl_frameset_new();
    espdr_hitran_line **hitran_tables = NULL;
    espdr_hitran_line **hitran_strongest_tables = NULL;
    espdr_qt_temp **hitran_qt = NULL;
    int *lines_nb = NULL, *qt_nb = NULL, *strongest_lines_nb = NULL;
    const char *telescope;
    char tel_sel[5];
    char* airm_start_kw;
    char* airm_end_kw;
    char* temp_K_kw;
    char* pres_start_kw;
    char* pres_end_kw;
    double airmass;
    double temp_K;
    double pressure_hPa, pressure_atm;
    double IWV, IWV_los;
    
    // Extract data from the FITS header
    
    if (strcmp(inst_config->instrument, "ESPRESSO") == 0) {
        
        if (cpl_propertylist_has(keywords, "TELESCOP")) {
            
            telescope = cpl_propertylist_get_string(keywords,"TELESCOP");
            sprintf(tel_sel,"TEL%c",telescope[9]);
            
            airm_start_kw = cpl_sprintf("%s %s %s", inst_config->prefix,
                    tel_sel, inst_config->airm_start_kw);
            airm_end_kw = cpl_sprintf("%s %s %s", inst_config->prefix,
                    tel_sel, inst_config->airm_end_kw);
            temp_K_kw = cpl_sprintf("%s %s %s",
                    inst_config->prefix, tel_sel, inst_config->temp_kw);
            pres_start_kw = cpl_sprintf("%s %s %s",
                    inst_config->prefix, tel_sel, inst_config->pres_start_kw);
            pres_end_kw = cpl_sprintf("%s %s %s",
                    inst_config->prefix, tel_sel, inst_config->pres_end_kw);
            
        } else {
            espdr_msg_error("No TELESCOPE KW found, exiting");
            cpl_frameset_delete(hitran_qt_frames);
            cpl_frameset_delete(hitran_strongest_frames);
            cpl_frameset_delete(hitran_ccf_masks);
            return (CPL_ERROR_NULL_INPUT);
        }
        
    } else {
    	temp_K_kw = cpl_sprintf("%s %s", inst_config->prefix, inst_config->temp_kw);
    	pres_start_kw = cpl_sprintf("%s %s", inst_config->prefix, inst_config->pres_start_kw);
    	pres_end_kw = cpl_sprintf("%s %s", inst_config->prefix, inst_config->pres_end_kw);
    	airm_start_kw = cpl_sprintf("%s", inst_config->airm_start_kw);
    	airm_end_kw = cpl_sprintf("%s", inst_config->airm_end_kw);
    }
    
    double airm_end = cpl_propertylist_get_double(keywords,airm_end_kw);
    double airm_start = cpl_propertylist_get_double(keywords,airm_start_kw);
    airmass = (airm_end + airm_start)/2.0;
    espdr_msg("airmass = %f", airmass);

    temp_K = cpl_propertylist_get_double(keywords, temp_K_kw) + 273.15;
    espdr_msg("Temp [K] = %f", temp_K);

    double pres_end = cpl_propertylist_get_double(keywords,pres_end_kw);
    double pres_start = cpl_propertylist_get_double(keywords,pres_start_kw);
    pressure_hPa = (pres_end + pres_start)/2.0;
    espdr_msg("pressure [hPa] = %f", pressure_hPa);
    pressure_atm = pressure_hPa / 1013.2501;
    espdr_msg("pressure [atm] = %f", pressure_atm);
    
    espdr_msg("KWs: %s %s %s", temp_K_kw, pres_start_kw, pres_end_kw);
    if (!cpl_propertylist_has(keywords, temp_K_kw) ||
        !cpl_propertylist_has(keywords, pres_start_kw) ||
        !cpl_propertylist_has(keywords, pres_end_kw)) {
        espdr_msg_error("No temperature or pressure KW found, please update your inst_config file, exiting");
        return (CPL_ERROR_NULL_INPUT);
    }
    
    // The measure from the telescope is not reliable enough and the initialiazation with 1.0 * airmass is working well
    IWV = 1.0;
    espdr_msg("IWV = %f", IWV);
    IWV_los = IWV * airmass;
    espdr_msg("IWV in the line of sight = %f", IWV_los);
    
    //espdr_msg("Getting resolution map");
    my_error = espdr_get_static_res_map_from_set(set, used_frames, &res_map);
    if (res_map == NULL) {
        espdr_msg_warning("No resolution map, skipping the telluric correction");
        cpl_frameset_delete(hitran_qt_frames);
        cpl_frameset_delete(hitran_strongest_frames);
        return (CPL_ERROR_NULL_INPUT);
    }
    espdr_msg("Res map size: %lld x %lld", cpl_image_get_size_x(res_map), cpl_image_get_size_y(res_map));
    espdr_msg("S2D size: %lld x %lld", cpl_image_get_size_x(s2d_blaze), cpl_image_get_size_y(s2d_blaze));
    
    //espdr_msg("Getting HITRAN lines tables");
    my_error = espdr_get_static_model_frames_from_set(set, used_frames, hitran_qt_frames,
                                                      hitran_strongest_frames,
                                                      hitran_ccf_masks);
    if (my_error == CPL_ERROR_NULL_INPUT) {
        espdr_msg_warning("No static mode (HITRAN LINES), skipping the telluric correction");
        cpl_frameset_delete(hitran_qt_frames);
        cpl_frameset_delete(hitran_strongest_frames);
        cpl_frameset_delete(hitran_ccf_masks);
        return (CPL_ERROR_NULL_INPUT);
    }
    
    //espdr_msg("HITRAN QT lines size: %lld", cpl_frameset_get_size(hitran_qt_frames));
    //espdr_msg("HITRAN strongest lines size: %lld", cpl_frameset_get_size(hitran_strongest_frames));
    
    //espdr_msg("Reading HITRAN lines tables");
    int qt_nset = cpl_frameset_get_size(hitran_qt_frames);
    int strongest_nset = cpl_frameset_get_size(hitran_strongest_frames);
    hitran_tables = (espdr_hitran_line **)cpl_malloc(qt_nset * sizeof(* hitran_tables));
    hitran_qt = (espdr_qt_temp **)cpl_malloc(qt_nset * sizeof(*hitran_qt));
    hitran_strongest_tables = (espdr_hitran_line **)cpl_malloc(strongest_nset * sizeof(* hitran_tables));
    lines_nb = (int *)cpl_calloc(qt_nset, sizeof(int));
    qt_nb = (int *)cpl_calloc(qt_nset, sizeof(int));
    strongest_lines_nb = (int *)cpl_calloc(strongest_nset, sizeof(int));
    char **molecule_name = (char **)cpl_malloc(qt_nset*sizeof(char *));
    for (int i = 0; i < qt_nset; i++) {
        molecule_name[i] = (char *)cpl_malloc(128 * sizeof(char));
    }
    my_error = espdr_read_hitran_from_frameset(hitran_qt_frames, hitran_tables, hitran_qt, lines_nb, qt_nb, molecule_name, 1);
    my_error = espdr_read_hitran_from_frameset(hitran_strongest_frames, hitran_strongest_tables, NULL, strongest_lines_nb, NULL, molecule_name, 0);
    
    cpl_table **telluric_mask = (cpl_table **)cpl_malloc(strongest_nset * sizeof(cpl_table *));
    int *mask_lines_nb = (int *)cpl_malloc(strongest_nset * sizeof(int));
    //espdr_msg("Reading telluric masks");
    my_error = espdr_get_telluric_mask(hitran_ccf_masks, telluric_mask, mask_lines_nb);
    
    //espdr_msg("Creating wavenumber grid");
    int ny_wave = cpl_image_get_size_y(wave_vacuum_shifted);
    double **wave = (double **)cpl_malloc(ny_wave * sizeof(double *));
    double **wavenumber = (double **)cpl_malloc(ny_wave * sizeof(double *));
    int *wavenumber_len = (int *)cpl_malloc(ny_wave * sizeof(int));
    double **wavenumber_selected = (double **)cpl_malloc(ny_wave * sizeof(double *));
    int *wavenumber_selected_len = (int *)cpl_malloc(ny_wave * sizeof(int));
    my_error = espdr_create_wavenumber_grid(wave_vacuum_shifted, wave, wavenumber, wavenumber_len,
                                            wavenumber_selected, wavenumber_selected_len);
    
    //espdr_msg("wavenumber");
    //for (int i = 0; i < ny_wave; i++) {
    //    espdr_msg("Order %d (len: %d): %f (%f) - %f (%f)", i+1, wavenumber_len[i], wavenumber[i][0], 100000000.0/wavenumber[i][0],
    //              wavenumber[i][wavenumber_len[i]-1], 100000000.0/wavenumber[i][wavenumber_len[i]-1]);
    //}
    
    //espdr_msg("wavenumber_selected");
    //for (int i = 0; i < ny_wave; i++) {
    //    espdr_msg("Order %d (len :%d): %f (%f) - %f (%f)", i+1, wavenumber_selected_len[i], wavenumber_selected[i][0], 100000000.0/wavenumber_selected[i][0],
    //              wavenumber_selected[i][wavenumber_selected_len[i]-1], 100000000.0/wavenumber_selected[i][wavenumber_selected_len[i]-1]);
    //}
    
    //espdr_msg("Setting temperature and QT");
    double temp_offset_molecule = 0.0;
    double Temp_offseted[qt_nset];
    double qt_used[qt_nset];
    for (int mol = 0; mol < qt_nset; mol++) {
        if (strcmp(hitran_tables[mol][0].molecule, "H2O") == 0) {
            temp_offset_molecule = 0.0;
        } else {
            temp_offset_molecule = 20.0;
        }
        Temp_offseted[mol] = temp_K - temp_offset_molecule;
        for (int i = 0; i < qt_nb[mol]; i++)
            if (round(Temp_offseted[mol]) == hitran_qt[mol][i].Temp) qt_used[mol] = hitran_qt[mol][i].Qt;
    }
    
    //espdr_msg("Setting N_mol and M_mol");
    double N_mol[qt_nset], M_mol[qt_nset];
    for (int mol = 0; mol < qt_nset; mol++) {
        if (strcmp(hitran_tables[mol][0].molecule, "H2O") == 0) {
            N_mol[mol] = N_H2O;
            M_mol[mol] = M_molH2O;
        } else {
            if (strcmp(hitran_tables[mol][0].molecule, "O2") == 0) {
                N_mol[mol] = N_O2;
                M_mol[mol] = M_molO2;
            } else {
                if (strcmp(hitran_tables[mol][0].molecule, "CO2") == 0) {
                    N_mol[mol] = N_CO2;
                    M_mol[mol] = M_molCO2;
                } else {
                    if (strcmp(hitran_tables[mol][0].molecule, "CH4") == 0) {
                        N_mol[mol] = N_CH4;
                        M_mol[mol] = M_molCH4;
                    } else {
                        espdr_msg_warning("No such molecule: %s",
                                          hitran_tables[mol][0].molecule);
                        return(CPL_ERROR_INCOMPATIBLE_INPUT);
                    }
                }
            }
        }
    }
    

#if 0
    int max_len = 0;
    for (int order = 0; order < ny_wave; order++) {
        if (max_len < wavenumber_len[order]) {
            max_len = wavenumber_len[order];
        }
    }
    double *wave_for_save = (double *)cpl_calloc(ny_wave*max_len, sizeof(double));
    double *wavenumber_for_save = (double *)cpl_calloc(ny_wave*max_len, sizeof(double));
    int index = 0;
    for (int order = 0; order < ny_wave; order++) {
        for (int j = 0; j < wavenumber_len[order]; j++) {
            wave_for_save[index] = wave[order][j];
            wavenumber_for_save[index] = wavenumber[order][j];
                index++;
            }
            for (int j = wavenumber_len[order]; j < max_len; j++) {
                wave_for_save[index] = 0.0;
                wavenumber_for_save[index] = 0.0;
                index++;
            }
    }
    
    cpl_image *wave_for_save_img = cpl_image_wrap_double(max_len, ny_wave, wave_for_save);
    cpl_image *wavenumber_for_save_img = cpl_image_wrap_double(max_len, ny_wave, wavenumber_for_save);
    
    char wave_filename[64];
    sprintf(wave_filename, "%s_wave_model.fits", inst_config->instrument);
    my_error = cpl_image_save(wave_for_save_img, wave_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    sprintf(wave_filename, "%s_wavenumber_model.fits", inst_config->instrument);
    my_error = cpl_image_save(wavenumber_for_save_img, wave_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    cpl_image_unwrap(wave_for_save_img);
    cpl_image_unwrap(wavenumber_for_save_img);
    cpl_free(wave_for_save);
    cpl_free(wavenumber_for_save);
    
    return(CPL_ERROR_NULL_INPUT);
#endif
        
    int **selected_orders_strongest = (int **)cpl_malloc(qt_nset * sizeof(int *));
    int **selected_orders = (int **)cpl_malloc(qt_nset * sizeof(int *));
    int *selected_orders_all = (int *)cpl_calloc(ny_wave, sizeof(int));
    
    //espdr_msg("Selecting orders");
    for (int i = 0; i < qt_nset; i++) {
        //espdr_msg("Selecting orders with strongest lines");
        selected_orders_strongest[i] = (int *)cpl_calloc(ny_wave, sizeof(int));
        my_error = espdr_select_orders(wavenumber, wavenumber_len, ny_wave,
                                       hitran_strongest_tables[i], strongest_lines_nb[i],
                                       selected_orders_strongest[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_select_orders failed on strongest lines for molecule: %s: %s",
                     hitran_tables[i][0].molecule, cpl_error_get_message_default(my_error));
        //for (int j = 0; j < ny_wave; j++) {
        //    espdr_msg("order %d - %d", j, selected_orders_strongest[i][j]);
        //}
        //espdr_msg("Selecting orders with all lines");
        selected_orders[i] = (int *)cpl_calloc(ny_wave, sizeof(int));
        my_error = espdr_select_orders(wavenumber, wavenumber_len, ny_wave,
                                       hitran_tables[i], lines_nb[i],
                                       selected_orders[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_select_orders failed on all lines for molecule: %s: %s",
                     hitran_tables[i][0].molecule, cpl_error_get_message_default(my_error));
        //for (int j = 0; j < ny_wave; j++) {
        //    espdr_msg("order %d - %d", j, selected_orders[i][j]);
        //}
        
        for (int order = 0; order < ny_wave; order++) {
            if (selected_orders[i][order] == 1) selected_orders_all[order] = 1;
        }
    }
    //for (int j = 0; j < ny_wave; j++) {
    //    espdr_msg("order %d - %d, CH4: %d, CO2: %d, H2O: %d, O2: %d",
    //              j, selected_orders_all[j], selected_orders[0][j],
    //              selected_orders[1][j], selected_orders[2][j], selected_orders[3][j]);
    //}

    
    cpl_image **spectrum_vs_model = (cpl_image **)cpl_malloc(qt_nset * sizeof(cpl_image *));
    int nx_s2d = cpl_image_get_size_x(s2d_blaze);
    int ny_s2d = cpl_image_get_size_y(s2d_blaze);
    cpl_array *RV_table = NULL;
    my_error = espdr_compute_rv_table(RV_RANGE_TELLURICS, rv_step, 0.0, &RV_table);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_rv_table failed");
    int RV_end = round(RV_RANGE_TELLURICS/rv_step);
    int RV_ini = - RV_end;
    int nx_ccf = RV_end - RV_ini +1;
    double *CCF_flux_data = (double *)cpl_calloc(nx_ccf, sizeof(double));
    double *CCF_model_flux_data = (double *)cpl_calloc(nx_ccf, sizeof(double));
    int RV_continuum_len = round(nx_ccf/4);
    
    //espdr_msg("Computing the CCF");
    // The CCF of the S2d_blaze is computed only once, it does not depend on the optimised parameters
    cpl_image **CCF_flux = (cpl_image **)cpl_malloc(qt_nset * sizeof(cpl_image *));
    cpl_image **CCF_error = (cpl_image **)cpl_malloc(qt_nset * sizeof(cpl_image *));
    cpl_image **CCF_qual = (cpl_image **)cpl_malloc(qt_nset * sizeof(cpl_image *));
    double CCF_median[qt_nset];
    
    for (int i = 0; i < qt_nset; i++) {
        CCF_flux[i]  = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
        CCF_error[i] = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
        CCF_qual[i]  = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_INT);
        
        my_error = espdr_compute_CCF(wave_vacuum_shifted, dll_matrix,
                                     s2d_blaze, s2d_error, blaze, s2d_qual,
                                     RV_table, telluric_mask[i], rv_step, berv, berv_max,
                                     &CCF_flux[i], &CCF_error[i], &CCF_qual[i]);
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_compute_CCF failed for molecule %s for s2d_blaze: %s",
                     hitran_tables[i][0].molecule, cpl_error_get_message_default(my_error));
        
        //char ccf_spectrum_filename[64];
        //sprintf(ccf_spectrum_filename, "%s_CCF_spectrum_%s_for_test.fits",
        //        inst_config->instrument, hitran_tables[i][0].molecule);
        //espdr_msg("Saving the CCF of %s in %s", hitran_tables[i][0].molecule, ccf_spectrum_filename);
        //my_error = cpl_image_save(CCF_flux[i], ccf_spectrum_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        
        int pxl;
        int CCF_continuum_length = 2*RV_continuum_len; // (0 .. 25 + 55 .. 80)
        cpl_vector *CCF_continuum_vector = cpl_vector_new(CCF_continuum_length);
        
        // Normalization of the sum
        for (int j = 0; j < RV_continuum_len; j++) {
            my_error = cpl_vector_set(CCF_continuum_vector, j, cpl_image_get(CCF_flux[i], j+1, ny_s2d+1, &pxl));
        }
        int rv_index_shift = nx_ccf - 2*RV_continuum_len;
        for (int j = nx_ccf - RV_continuum_len; j < nx_ccf; j++) {
            my_error = cpl_vector_set(CCF_continuum_vector, j-rv_index_shift,
                                      cpl_image_get(CCF_flux[i], j+1, ny_s2d+1, &pxl));
        }

        CCF_median[i] = cpl_vector_get_median(CCF_continuum_vector);
        //espdr_msg("CCF_median[%d]: %f", i, CCF_median[i]);
        my_error = cpl_image_divide_scalar(CCF_flux[i], CCF_median[i]);
        
        cpl_vector_delete(CCF_continuum_vector);
        
        //char ccf_norm_spectrum_filename[64];
        //sprintf(ccf_norm_spectrum_filename, "%s_CCF_norm_spectrum_%s_for_test.fits",
        //        inst_config->instrument, hitran_tables[i][0].molecule);
        //espdr_msg("Saving the norm CCF of %s in %s", hitran_tables[i][0].molecule, ccf_norm_spectrum_filename);
        //my_error = cpl_image_save(CCF_flux[i], ccf_norm_spectrum_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    }

#if 0
    clock_t fit_start, fit_end, fit_molecule_start, fit_molecule_end;
    double cpu_time_used_for_fit;
    fit_start = clock();
    //double pressure_atm_molecule[4] = {0.31106053, 0.32653213, 0.71535853, 0.40755797}; // optimal values from Romain
    //double PWV_molecule[4] = {3013855.30, 1310746.14, 1.07640845, 762447.326}; // optimal values from Romain
    //double pressure_atm_molecule_test[4] = {2.818255e-01, 3.333068e-01, 7.279501e-01, 4.230537e-01}; // optimal values from L-M and new masks
    //double PWV_molecule_test[4] = {2.977125e+06, 1.418249e+06, 1.062058e+00, 7.908802e+05}; // optimal values from L-M and new masks
    //double pressure_atm_molecule_test[4] = {3.376510e-01, 2.968796e-01, 6.289361e-01, 3.227631e-01}; // optimal values for Proxima, RV range 40
    //double PWV_molecule_test[4] = {3.508769e-01, 1.422709e-01, 4.202090e-01, 8.677341e-01}; // optimal values for Proxima, RV range 40
    //double pressure_atm_molecule_test[4] = {3.596612e-01, 2.856459e-01, 6.385294e-01, 3.947193e-01}; // optimal values for Proxima, RV range 30
    //double PWV_molecule_test[4] = {3.586304e-01, 1.405713e-01, 4.214562e-01, 8.420484e-01}; // optimal values for Proxima, RV range 30
    //double pressure_atm_molecule_test[4] = {1.036631e+00, 1.799762e+01, 1.474522e-01, 1.220059e+02}; // optimal values for TOI-4461, RV range 40
    //double PWV_molecule_test[4] = {-3.013124e-01, 3.798250e+01, 6.520922e-01, 9.535593e+02}; // optimal values for TOI-4461, RV range 40
    //double pressure_atm_molecule_test[4] = {9.522501e-01, 1.849670e+01, 4.342353e-01, -7.367396e+00}; // optimal values for TOI-4461, RV range 40, with bkgr
    //double PWV_molecule_test[4] = {-1.856368e-01, 2.634371e+01, 1.430019e-01, 1.522197e-01}; // optimal values for TOI-4461, RV range 40, with bkgr
    //double pressure_atm_molecule_test[4] = {4.191202e-01, 2.868862e-01, 6.221072e-01, 2.699562e-01}; // optimal values for Proxima, separate normalization
    //double PWV_molecule_test[4] = {3.470226e-01, 1.385896e-01, 4.182514e-01, 7.917900e-01}; // optimal values for Proxima, separate normalization
    double pressure_atm_molecule_test[4] = {1.043669e+00, 6.357389e-01, 1.199774e-01, 1.228674e-03}; // optimal values for TOI-4461, separate normalization
    double PWV_molecule_test[4] = {-7.157106e-0, 8.835831e-02, 7.247305e-01, -6.957967e-02l}; // optimal values for TOI-4166, separate normalization
    double fit_result;
    for (int i = 0; i < qt_nset; i++) {
    //int i = 0; // 0 == CH4, 1 == CO2, 2 == H2O, 3 == O2
        fit_molecule_start = clock();
        espdr_msg("Fitting telluric model for %s...", hitran_tables[i][0].molecule);
        fit_result = espdr_fit_telluric_model(pressure_atm_molecule_test[i], PWV_molecule_test[i],
                                              hitran_tables[i][0].molecule,
                                              Temp_offseted[i], qt_used[i], airmass,
                                              N_mol[i], M_mol[i], inst_config,
                                              wave, wavenumber, wavenumber_len,
                                              selected_orders[i],
                                              hitran_tables[i], lines_nb[i],
                                              res_map, wave_vacuum_shifted, dll_matrix, s2d_qual,
                                              telluric_mask[i], rv_step, berv, berv_max,
                                              nx_s2d, ny_s2d, nx_ccf, RV_table, CCF_flux[i], 1,
                                              CCF_flux_data, CCF_model_flux_data,
                                              &spectrum_vs_model[i]);
        my_error = cpl_error_get_code();
        espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                     "espdr_fit_telluric_model failed for molecule: %s: %s",
                     hitran_tables[i][0].molecule, cpl_error_get_message_default(my_error));
        
        //espdr_msg("spectrum vs model for %s size: %lld x %lld", hitran_tables[i][0].molecule,
        //          cpl_image_get_size_x(spectrum_vs_model[i]), cpl_image_get_size_y(spectrum_vs_model[i]));
        
        //char ccf_spectrum_vs_model_filename[64];
        //sprintf(ccf_spectrum_vs_model_filename, "%s_CCF_spectrum_vs_model_%s_for_test.fits",
        //        inst_config->instrument, hitran_tables[i][0].molecule);
        //my_error = cpl_image_save(spectrum_vs_model[i], ccf_spectrum_vs_model_filename,
        //                          CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        fit_molecule_end = clock();
        cpu_time_used_for_fit = ((double) (fit_molecule_end - fit_molecule_start)) / CLOCKS_PER_SEC;
        espdr_msg("CPU TIME used for molecule %s: %f sec", hitran_tables[i][0].molecule, cpu_time_used_for_fit);
    }
    fit_end = clock();
    cpu_time_used_for_fit = ((double) (fit_end - fit_start)) / CLOCKS_PER_SEC;
    espdr_msg("CPU TIME used by espdr_fit_telluric_model: %f sec", cpu_time_used_for_fit);
#endif
    //exit (0);
    
    //espdr_msg("Initializing input_params");
    espdr_fit_params input_params[qt_nset];
    for (int i = 0; i < qt_nset; i++) {
        strcpy(input_params[i].molecule, hitran_tables[i][0].molecule);
        input_params[i].Temp_offseted = Temp_offseted[i],
        input_params[i].qt_used = qt_used[i],
        input_params[i].airmass = airmass,
        input_params[i].N_mol = N_mol[i],
        input_params[i].M_mol = M_mol[i],
        input_params[i].inst_config = inst_config,
        input_params[i].wave = wave;
        input_params[i].wavenumber_grid = wavenumber;
        input_params[i].wavenumber_len = wavenumber_len;
        input_params[i].selected_orders = selected_orders_strongest[i];
        input_params[i].hitran_table = hitran_strongest_tables[i];
        input_params[i].hitran_lines_nb = strongest_lines_nb[i];
        input_params[i].res_map = res_map;
        input_params[i].wave_matrix = wave_vacuum_shifted;
        input_params[i].dll_matrix = dll_matrix;
        input_params[i].s2d_qual = s2d_qual;
        input_params[i].telluric_mask = telluric_mask[i];
        input_params[i].rv_step = rv_step;
        input_params[i].berv = berv;
        input_params[i].berv_max = berv_max;
        input_params[i].nx_s2d = nx_s2d;
        input_params[i].ny_s2d = ny_s2d;
        input_params[i].nx_ccf = nx_ccf;
        input_params[i].RV_table = RV_table;
        input_params[i].CCF_flux_norm = CCF_flux[i];
        input_params[i].save_flag = 0;
        input_params[i].CCF_flux_data_RE = CCF_flux_data;
        input_params[i].CCF_model_flux_data_RE = CCF_model_flux_data;
        input_params[i].spectrum_vs_model_RE = &spectrum_vs_model[i];
    }
    
    double pressure_atm_molecule[qt_nset];
    double pressure_atm_molecule_err[qt_nset];
    double PWV_molecule[qt_nset];
    double PWV_molecule_err[qt_nset];
    double chi2_final[qt_nset];
    int iterations_nb[qt_nset];

    //espdr_msg("Initialising the least squares structures");
    const gsl_multifit_nlinear_type *T = gsl_multifit_nlinear_trust;
    gsl_multifit_nlinear_fdf fdf;
    gsl_multifit_nlinear_parameters fdf_params = gsl_multifit_nlinear_default_parameters();
    const size_t n = nx_ccf;
    const size_t p = 2;
    
    gsl_vector *f;
    gsl_matrix *J;
    double t[nx_ccf];
    double y[nx_ccf];
    double weights[nx_ccf];
    double x_init[2];
    
    clock_t molecule_start, molecule_end;
    double cpu_time_used;
    
    for (int mol_index = 0; mol_index < qt_nset; mol_index++) {
        
        molecule_start = clock();
        
        espdr_msg("Starting optimisation for %s", hitran_tables[mol_index][0].molecule);
        
        double xtol = 1e-3;
        double gtol = 1e-8;
        double ftol = 0.0;
        
        if (strcmp(hitran_tables[mol_index][0].molecule, "CH4") == 0) {
            x_init[0] = pressure_atm/2.0;
            x_init[1] = PWV_CH4*airmass*1e-7;
        } else {
            if (strcmp(hitran_tables[mol_index][0].molecule, "CO2") == 0) {
                x_init[0] = pressure_atm/2.0;
                x_init[1] = PWV_CO2*airmass*1e-7;
            } else {
                if (strcmp(hitran_tables[mol_index][0].molecule, "H2O") == 0) {
                    x_init[0] = pressure_atm;
                    x_init[1] = IWV_los;
                } else {
                    if (strcmp(hitran_tables[mol_index][0].molecule, "O2") == 0) {
                        x_init[0] = pressure_atm/2.0;
                        x_init[1] = PWV_O2*airmass*1e-6;
                    } else {
                        espdr_msg_error("Wrong molecule: %s", hitran_tables[mol_index][0].molecule);
                        return(CPL_ERROR_INCOMPATIBLE_INPUT);
                    }
                }
            }
        }
        espdr_msg("Initial values set to: pressure: %e, PWV: %e", x_init[0], x_init[1]);
        
        gsl_vector_view x = gsl_vector_view_array (x_init, p);
        gsl_vector_view wts = gsl_vector_view_array(weights, n);
        gsl_rng * r;
        double chisq, chisq0;
        int status, info;
        //size_t i;

        espdr_msg("Stopping conditions: xtol (step tolerance): %e, gtol (gradient tolerance): %e", xtol, gtol);
        
        gsl_rng_env_setup();
        r = gsl_rng_alloc(gsl_rng_default);
        
        /* define the function to be minimized */
        //espdr_msg("Defining the function to be minimised");
        fdf.f = &espdr_telluric_fit_least_squares;
        fdf.df = NULL;   /* set to NULL for finite-difference Jacobian */
        fdf.fvv = NULL;     /* not using geodesic acceleration */
        fdf.n = n;
        fdf.p = p;
        fdf.params = &input_params[mol_index];
        
        /* Data to be fitted */
        //espdr_msg("Filling the data to be fitted");
        cpl_vector *CCF_flux_vector = cpl_vector_new_from_image_row(CCF_flux[mol_index], ny_s2d+1);
        double *CCF_flux_vector_data = cpl_vector_get_data(CCF_flux_vector);
        cpl_vector *CCF_error_vector = cpl_vector_new_from_image_row(CCF_error[mol_index], ny_s2d+1);
        double CCF_error_median = cpl_vector_get_median(CCF_error_vector);
        int el_null;
        
        for (int i = 0; i < nx_ccf; i ++) {
            t[i] = cpl_array_get(RV_table, i, &el_null);
            y[i] = CCF_flux_vector_data[i];
            //weights[i] = 1.0;
            weights[i] = (CCF_median[mol_index] * CCF_median[mol_index]) / (CCF_error_median * CCF_error_median);
        }
        cpl_vector_delete(CCF_flux_vector);
        
        //espdr_msg("Starting the minimisation");
        //espdr_msg("Allocating workspace");
        /* allocate workspace with default parameters */
        gsl_multifit_nlinear_workspace *w = gsl_multifit_nlinear_alloc(T, &fdf_params, n, p);
        //espdr_msg("w is a %s and %s solver",
        //          gsl_multifit_nlinear_name(w), gsl_multifit_nlinear_trs_name(w));
        
        //espdr_msg("Initializing solver");
        /* initialize solver with starting point and weights */
        gsl_multifit_nlinear_winit(&x.vector, &wts.vector, &fdf, w);

        //espdr_msg("Computing initial cost function");
        /* compute initial cost function */
        f = gsl_multifit_nlinear_residual(w);
        gsl_blas_ddot(f, f, &chisq0);
        //espdr_msg("Initial CHI2: %f", chisq0);
        
        espdr_msg("Solving the system, max %d iterations", FIT_ITERATIONS_MAX_NB);
        /* solve the system with a maximum of FIT_ITERATIONS_MAX_NB iterations */
        status = gsl_multifit_nlinear_driver(FIT_ITERATIONS_MAX_NB, xtol, gtol, ftol,
                                             callback, NULL, &info, w);
        
        //espdr_msg("Computing covariance");
        /* compute covariance of best fit parameters */
        gsl_matrix *covar = gsl_matrix_alloc (p, p);
        J = gsl_multifit_nlinear_jac(w);
        gsl_multifit_nlinear_covar (J, 0.0, covar);
        
        //espdr_msg("Computing the final cost");
        /* compute final cost */
        f = gsl_multifit_nlinear_residual(w);
        gsl_blas_ddot(f, f, &chisq);
        
    #define FIT(i) gsl_vector_get(w->x, i)
    #define ERR(i) sqrt(gsl_matrix_get(covar,i,i))

        espdr_msg("summary from method '%s/%s'",
                  gsl_multifit_nlinear_name(w), gsl_multifit_nlinear_trs_name(w));
        espdr_msg("number of iterations: %zu", gsl_multifit_nlinear_niter(w));
        espdr_msg("function evaluations: %zu", fdf.nevalf);
        espdr_msg("Jacobian evaluations: %zu", fdf.nevaldf);
        espdr_msg("reason for stopping: %s", (info == 1) ? "small step size" : "small gradient");
        espdr_msg("initial |f(x)| = %f", sqrt(chisq0));
        espdr_msg("final   |f(x)| = %f", sqrt(chisq));
        iterations_nb[mol_index] = gsl_multifit_nlinear_niter(w);
        
        double c;
        {
            double dof = n - p;
            c = GSL_MAX_DBL(1, sqrt(chisq / dof));

            espdr_msg("chisq/dof = %g", chisq / dof);

            espdr_msg("pressure = %e +/- %e", FIT(0), c*ERR(0));
            espdr_msg("PWV      = %e +/- %e", FIT(1), c*ERR(1));
        }

        espdr_msg("status = %s", gsl_strerror(status));
        
        espdr_msg("Best values: pressure %e, PWV: %e, nb of iterations: %zu",
                  FIT(0), FIT(1), gsl_multifit_nlinear_niter(w));
        
        pressure_atm_molecule[mol_index] = FIT(0);
        pressure_atm_molecule_err[mol_index] = c*ERR(0);
        PWV_molecule[mol_index] = FIT(1);
        PWV_molecule_err[mol_index] = c*ERR(1);
        chi2_final[mol_index] = chisq;
        
        gsl_multifit_nlinear_free(w);
        gsl_matrix_free(covar);
        gsl_rng_free(r);
        
        molecule_end = clock();
        cpu_time_used = ((double) (molecule_end - molecule_start)) / CLOCKS_PER_SEC;
        espdr_msg("CPU TIME used to optimize the parameters for %s: %f",
                  hitran_tables[mol_index][0].molecule, cpu_time_used);
    }
    
    espdr_msg("Compute the final spectrum");
    
    double PWV_fitted[qt_nset], PWV_fitted_err[qt_nset];
    for (int i = 0; i < qt_nset; i++) {
        if (strcmp(hitran_tables[i][0].molecule, "H2O") == 0) {
            PWV_fitted[i] = PWV_molecule[i];
            PWV_fitted_err[i] = PWV_molecule_err[i];
        } else {
            if (strcmp(hitran_tables[i][0].molecule, "O2") == 0) {
                PWV_fitted[i] = PWV_molecule[i] * 1e+06; // for O2
                PWV_fitted_err[i] = PWV_molecule_err[i] * 1e+06;
            } else {
                PWV_fitted[i] = PWV_molecule[i] * 1e+07; // for CH4 and CO2
                PWV_fitted_err[i] = PWV_molecule_err[i] * 1e+07;
            }
        }
    }
    
    cpl_image *telluric_spectrum = NULL;
    my_error = espdr_compute_telluric_model(pressure_atm_molecule, PWV_fitted,
                                            Temp_offseted, qt_used, airmass,
                                            N_mol, M_mol, qt_nset, inst_config,
                                            wave, wavenumber, wavenumber_len,
                                            selected_orders, selected_orders_all,
                                            hitran_tables, lines_nb,
                                            res_map, wave_vacuum_shifted, s2d_qual,
                                            nx_s2d, ny_s2d, 0,
                                            &telluric_spectrum);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_telluric_model failed: %s",
                 cpl_error_get_message_default(my_error));
    
    *telluric_spectrum_RE = cpl_image_duplicate(telluric_spectrum);
    
    double *s2d_blaze_data = cpl_image_get_data_double(s2d_blaze);
    double *s2d_error_data = cpl_image_get_data_double(s2d_error);
    int *s2d_qual_data = cpl_image_get_data_int(s2d_qual);
    double *telluric_spectrum_data = cpl_image_get_data_double(telluric_spectrum);
    double *s2d_tell_corr_data = (double *)cpl_calloc(nx_s2d*ny_s2d, sizeof(double));
    double *s2d_err_tell_corr_data = (double *)cpl_calloc(nx_s2d*ny_s2d, sizeof(double));
    int *s2d_qual_tell_corr_data = (int *)cpl_calloc(nx_s2d*ny_s2d, sizeof(int));
    for (int i = 0; i < nx_s2d*ny_s2d; i++) {
        if (telluric_spectrum_data[i] < 0.1) {
            s2d_tell_corr_data[i] = 0.0;
            s2d_err_tell_corr_data[i] = 0.0;
            //s2d_qual_tell_corr_data[i] = OTHER_BAD_PIXEL;
            s2d_qual_tell_corr_data[i] = LOW_QE_PIXEL;
        } else {
            s2d_tell_corr_data[i] = s2d_blaze_data[i] / telluric_spectrum_data[i];
            s2d_err_tell_corr_data[i] = s2d_error_data[i] / telluric_spectrum_data[i];
            s2d_qual_tell_corr_data[i] = s2d_qual_data[i];
        }
    }
    cpl_image *s2d_tell_corr = cpl_image_wrap_double(nx_s2d, ny_s2d, s2d_tell_corr_data);
    *s2d_tell_corr_RE = cpl_image_duplicate(s2d_tell_corr);
    cpl_image *s2d_err_tell_corr = cpl_image_wrap_double(nx_s2d, ny_s2d, s2d_err_tell_corr_data);
    *s2d_err_tell_corr_RE = cpl_image_duplicate(s2d_err_tell_corr);
    cpl_image *s2d_qual_tell_corr = cpl_image_wrap_int(nx_s2d, ny_s2d, s2d_qual_tell_corr_data);
    *s2d_qual_tell_corr_RE = cpl_image_duplicate(s2d_qual_tell_corr);

#if 0
    char tell_corr_filename[64];
    sprintf(tell_corr_filename, "%s_tell_corr_spectrum_all_for_test.fits", inst_config->instrument);
    espdr_msg("Saving s2d tell correctd in %s", tell_corr_filename);
    my_error = cpl_image_save(*s2d_tell_corr_RE, tell_corr_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    
    char tell_corr_err_filename[64];
    sprintf(tell_corr_err_filename, "%s_tell_corr_error_all_for_test.fits", inst_config->instrument);
    espdr_msg("Saving s2d error tell correctd in %s", tell_corr_filename);
    my_error = cpl_image_save(*s2d_err_tell_corr_RE, tell_corr_err_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
#endif
    
    my_error = espdr_fill_header(tell_keywords_RE, inst_config, qt_nset, molecule_name,
                                 PWV_fitted, PWV_fitted_err,
                                 pressure_atm_molecule, pressure_atm_molecule_err,
                                 Temp_offseted, iterations_nb, chi2_final);
    
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_fill_header failed: %s", cpl_error_get_message_default(my_error));
    
    espdr_msg("Spectrum corrected for tellurics with header keywords created");
    
    espdr_msg("Freeing memory");
    //for (int i = 0; i < qt_nset; i++) {
    //    cpl_image_delete(spectrum_vs_model[i]);
    //}
    //cpl_free(spectrum_vs_model);
    
    //espdr_msg("Freeing images");
    for (int i = 0; i < qt_nset; i++) {
        cpl_image_delete(CCF_flux[i]);
        cpl_image_delete(CCF_error[i]);
        cpl_image_delete(CCF_qual[i]);
    }
    cpl_free(CCF_flux);
    cpl_free(CCF_error);
    cpl_free(CCF_qual);
    
    cpl_image_delete(telluric_spectrum);
    cpl_image_unwrap(s2d_tell_corr);
    cpl_image_unwrap(s2d_err_tell_corr);
    cpl_image_unwrap(s2d_qual_tell_corr);
    cpl_free(s2d_tell_corr_data);
    cpl_free(s2d_err_tell_corr_data);
    cpl_free(s2d_qual_tell_corr_data);
    
    //espdr_msg("Freeing tables");
    for (int i = 0; i < qt_nset; i++) {
        cpl_free(hitran_tables[i]);
        cpl_free(hitran_qt[i]);
    }
    cpl_free(hitran_tables);
    cpl_free(hitran_qt);
    
    for (int i = 0; i < strongest_nset; i++) {
        cpl_free(hitran_strongest_tables[i]);
        cpl_free(telluric_mask[i]);
    }
    cpl_free(hitran_strongest_tables);
    cpl_free(telluric_mask);
    
    cpl_array_delete(RV_table);
    
    //espdr_msg("Freeing lines nb");
    cpl_free(lines_nb);
    cpl_free(qt_nb);
    cpl_free(strongest_lines_nb);
    cpl_free(mask_lines_nb);
    
    //espdr_msg("Freeing grid data");
    for (int i = 0; i < ny_wave; i++) {
        cpl_free(wave[i]);
        cpl_free(wavenumber[i]);
        cpl_free(wavenumber_selected[i]);
    }
    cpl_free(wave);
    cpl_free(wavenumber);
    cpl_free(wavenumber_selected);
    cpl_free(wavenumber_len);
    cpl_free(wavenumber_selected_len);
    
    //espdr_msg("Freeing input frames & images");
    cpl_image_delete(res_map);
    cpl_frameset_delete(hitran_qt_frames);
    cpl_frameset_delete(hitran_strongest_frames);


    //cpl_free(tel_sel);
    cpl_free(airm_start_kw);
    cpl_free(airm_end_kw);
    cpl_free(temp_K_kw);
    cpl_free(pres_start_kw);
    cpl_free(pres_end_kw);


    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Read HITRAN tables
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_read_hitran_from_frameset(cpl_frameset *hitran_frames,
                                               espdr_hitran_line **hitran_tables,
                                               espdr_qt_temp **hitran_qt,
                                               int *lines_nb_RE,
                                               int *qt_nb_RE,
                                               char **molecule_name,
                                               int qt_read_flag) {
    
    const char *hitran_filename = NULL;
    char *molecule = (char *)cpl_malloc(128 * sizeof(char));
    char *molecule_final = (char *)cpl_malloc(128 * sizeof(char));
    //char *temp_mol = (char *)cpl_malloc(128 * sizeof(char));
    char *temp_mol1 = (char *)cpl_malloc(128 * sizeof(char));
    cpl_table *line_t = NULL;
    
    /* Loop on frames */
    int nset = cpl_frameset_get_size(hitran_frames);
    cpl_frameset_iterator* iter = cpl_frameset_iterator_new(hitran_frames);
    cpl_frame *frame = cpl_frameset_iterator_get(iter);
    
    //espdr_msg("Got %d hitran lines tables", nset);
    for (int i = 0; i < nset; i++) {
        hitran_tables[i] = (espdr_hitran_line *)cpl_malloc(100000 * sizeof(espdr_hitran_line));
    }
    
    if (qt_read_flag == 1) {
        for (int i = 0; i < nset; i++) {
            hitran_qt[i] = (espdr_qt_temp *)cpl_malloc(10000 * sizeof(espdr_qt_temp));
        }
    }

    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
        
        hitran_filename = cpl_frame_get_filename(frame);
        //espdr_msg("Reading HITRAN lines from %s", hitran_filename);
        //strcpy(temp_mol, hitran_filename);
        molecule = strrchr(hitran_filename, '/')+1;
        strcpy(temp_mol1, molecule);
        molecule = strrchr(temp_mol1, '_')+1;
        molecule[strlen(molecule) - 5] = '\0';
        strcpy(molecule_final, molecule);
        strcpy(molecule_name[iter_frame], molecule);
        //espdr_msg("Molecule: %s", molecule_final);
        
        line_t = cpl_table_load(hitran_filename,1,0);
        
        //cpl_table_dump(line_t, 1, 10, NULL);
        
        int table_length = cpl_table_get_nrow(line_t);
        //espdr_msg("%s table length: %d", molecule_final, table_length);
        lines_nb_RE[iter_frame] = table_length;
        double *wave_number = cpl_table_get_data_double(line_t, COL_NAME_WAVE_NUMBER);
        double *intensity = cpl_table_get_data_double(line_t, COL_NAME_INTENSITY);
        double *gamma_air = cpl_table_get_data_double(line_t, COL_NAME_GAMMA_AIR);
        double *n_air = cpl_table_get_data_double(line_t, COL_NAME_N_AIR);
        double *delta_air = cpl_table_get_data_double(line_t, COL_NAME_DELTA_AIR);
        double *epp = cpl_table_get_data_double(line_t, COL_NAME_EPP);
        
        for (int i = 0; i < table_length; i++) {
            strcpy(hitran_tables[iter_frame][i].molecule, molecule_final);
            hitran_tables[iter_frame][i].wave_number = wave_number[i];
            hitran_tables[iter_frame][i].intensity = intensity[i];
            hitran_tables[iter_frame][i].gamma_air = gamma_air[i];
            hitran_tables[iter_frame][i].n_air = n_air[i];
            hitran_tables[iter_frame][i].delta_air = delta_air[i];
            hitran_tables[iter_frame][i].Epp = epp[i];
            //printf("%s\t%f\t%f\t%f\t%f\t%f\t%f\t%f\n", molecule_final,
            //       hitran_tables[iter_frame][i].wave_number,
            //       hitran_tables[iter_frame][i].intensity,
            //       hitran_tables[iter_frame][i].gamma_air,
            //       hitran_tables[iter_frame][i].gamma_self,
            //       hitran_tables[iter_frame][i].n_air,
            //       hitran_tables[iter_frame][i].delta_air,
            //       hitran_tables[iter_frame][i].Epp);
        }
        
        if (qt_read_flag == 1) {
            line_t = cpl_table_load(hitran_filename,2,0);
            //cpl_table_dump(line_t, 1, 20, NULL);

            double *Temp = cpl_table_get_data_double(line_t, COL_NAME_TEMP);
            double *Qt = cpl_table_get_data_double(line_t, COL_NAME_QT);
            
            int qt_table_length = cpl_table_get_nrow(line_t);
           //espdr_msg("%s Qt table length: %d", molecule_final, qt_table_length);
            qt_nb_RE[iter_frame] = qt_table_length;
            
            for (int i = 0; i < qt_table_length; i++) {
                strcpy(hitran_qt[iter_frame][i].molecule, molecule_final);
                hitran_qt[iter_frame][i].Temp = Temp[i];
                hitran_qt[iter_frame][i].Qt = Qt[i];
                //espdr_msg("%s: T %f - QT %f", molecule_final, Temp[i], Qt[i]);
            }
        }
        
        cpl_frameset_iterator_advance(iter, 1);
        frame = cpl_frameset_iterator_get(iter);
    }
    cpl_frameset_iterator_delete(iter);
    //cpl_free(temp_mol);
    
    
    return (cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Fit the telluric model
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_get_telluric_mask(cpl_frameset *hitran_ccf_frames,
                                       cpl_table **mask_RE,
                                       int *lines_nb_RE) {
    
    const char *hitran_filename = NULL;
    char *molecule = (char *)cpl_malloc(128 * sizeof(char));
    char *molecule_final = (char *)cpl_malloc(128 * sizeof(char));
    //char *temp_mol = (char *)cpl_malloc(128 * sizeof(char));
    char *temp_mol1 = (char *)cpl_malloc(128 * sizeof(char));
    cpl_table *line_t = NULL;
    
    /* Loop on frames */
    int nset = cpl_frameset_get_size(hitran_ccf_frames);
    cpl_frameset_iterator* iter = cpl_frameset_iterator_new(hitran_ccf_frames);
    cpl_frame *frame = cpl_frameset_iterator_get(iter);
    
    //espdr_msg("Got %d CCF mask tables", nset);
    
    for ( int iter_frame = 0; iter_frame < nset; iter_frame++ ) {
        
        hitran_filename = cpl_frame_get_filename(frame);
        //espdr_msg("Reading HITRAN CCF lines from %s", hitran_filename);
        //strcpy(temp_mol, hitran_filename);
        molecule = strrchr(hitran_filename, '/')+1;
        strcpy(temp_mol1, molecule);
        molecule = strrchr(temp_mol1, '_')+1;
        molecule[strlen(molecule) - 5] = '\0';
        strcpy(molecule_final, molecule);
        //espdr_msg("Molecule: %s", molecule_final);
        
//#if 0
        line_t = cpl_table_load(hitran_filename,1,0);

        //cpl_table_dump(line_t, 1, 10, NULL);
        //cpl_table_dump_structure(line_t, NULL);
        
        int table_length = cpl_table_get_nrow(line_t);
        //espdr_msg("%s table length: %d", molecule_final, table_length);
        lines_nb_RE[iter_frame] = table_length;
        
        mask_RE[iter_frame] = cpl_table_duplicate(line_t);
//#endif
#if 0
        cpl_error_code my_error = CPL_ERROR_NONE;
        // Allows ro read the telluric masks from the 2nd extension of teh strongest lines static calibrations
        line_t = cpl_table_load(hitran_filename,2,0);
        cpl_table_dump(line_t, 1, 10, NULL);
        //cpl_table_dump_structure(line_t, NULL);
        
        int table_length = cpl_table_get_nrow(line_t);
        //espdr_msg("%s table length: %d", molecule_final, table_length);
        lines_nb_RE[iter_frame] = table_length;
        
        mask_RE[iter_frame] = cpl_table_new(table_length);
        double *mask_lambda = cpl_table_get_data_double(line_t, "CCF_lines_position_wavelength");
        my_error = espdr_quicksort(mask_lambda, table_length);
        my_error = cpl_table_new_column(mask_RE[iter_frame], "lambda", CPL_TYPE_DOUBLE);
        my_error = cpl_table_copy_data_double(mask_RE[iter_frame], "lambda", mask_lambda);
        my_error = cpl_table_new_column(mask_RE[iter_frame], "contrast", CPL_TYPE_DOUBLE);
        my_error = cpl_table_fill_column_window_double(mask_RE[iter_frame], "contrast", 1, table_length, 1.0);
        cpl_table_dump(mask_RE[iter_frame], 1, 10, NULL);
#endif
        cpl_frameset_iterator_advance(iter, 1);
        frame = cpl_frameset_iterator_get(iter);
        cpl_table_delete(line_t);
    }
    
    cpl_frameset_iterator_delete(iter);
    //cpl_free(temp_mol);
    
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Create a grid of wavelengths
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_create_wavenumber_grid(cpl_image *wave_matrix,
                                            double **wave_RE,
                                            double **wavenumber_RE,
                                            int *wavenumber_len_RE,
                                            double **wavenumber_selected_RE,
                                            int *wavenumber_selected_len_RE) {
    
    int nx_wave = cpl_image_get_size_x(wave_matrix);
    int ny_wave = cpl_image_get_size_y(wave_matrix);
    double step, step_selected;
    double born;
    double middle_element, first_element, last_element;
    int wavenumber_len, wavenumber_selected_len;
    double wavenumber_val;
    cpl_vector *wave_vector = NULL, *wave_vector_diff = NULL;
    
    for (int order = 0; order < ny_wave; order++) {
        
        //espdr_msg("Creating wavenumber grid for order %d", order+1);
        wave_vector = cpl_vector_new_from_image_row(wave_matrix, order+1);
        middle_element = cpl_vector_get(wave_vector, nx_wave/2);
        first_element = cpl_vector_get(wave_vector, 0);
        last_element = cpl_vector_get(wave_vector, nx_wave-1);
        step = middle_element / MODEL_RESOLUTION;
        born = middle_element * 40.0 / 300000.0;
        //espdr_msg("for wavenumber: step - %e, born - %e", step, born);
        //espdr_msg("wave first: %.10f, last: %.10f", first_element, last_element);
        wavenumber_len = ((last_element+born) - (first_element-born)) / step + 1;
        //espdr_msg("wavenumber length: %d", wavenumber_len);
        wavenumber_len_RE[order] = wavenumber_len;
        wave_RE[order] = (double *)cpl_calloc(wavenumber_len, sizeof(double));
        wavenumber_RE[order] = (double *)cpl_calloc(wavenumber_len, sizeof(double));
        wavenumber_val = first_element - born;
        for (int i = 0; i < wavenumber_len; i++) {
            wave_RE[order][i] = wavenumber_val;
            wavenumber_val += step;
        }
        for (int i = 0; i < wavenumber_len; i++) {
            if (wave_RE[order][i] == 0.0) {
                espdr_msg_warning("wavenumber[%d][%d] is 0.0!!!", order, i);
            } else {
                wavenumber_RE[order][i] = 100000000.0 / wave_RE[order][i]; // [A] --> [1/cm]
            }
        }
        
        wave_vector_diff = cpl_vector_new(nx_wave);
        double diff_value = 0.0;
        for (int i = 1; i < nx_wave; i++) {
            diff_value = cpl_vector_get(wave_vector, i) - cpl_vector_get(wave_vector, i-1);
            cpl_vector_set(wave_vector_diff, i-1, diff_value);
        }
        cpl_vector_set(wave_vector_diff, nx_wave-1, diff_value);
        step_selected = cpl_vector_get_mean(wave_vector_diff);
        //espdr_msg("Difference mean in wave_vector: %e", step_selected);
        
        wavenumber_selected_len = ((last_element-4*born) - (first_element+4*born)) / step_selected + 1;
        wavenumber_selected_len_RE[order] = wavenumber_selected_len;
        //espdr_msg("wavenumber_selected length: %d", wavenumber_selected_len);
        
        wavenumber_selected_RE[order] = (double *)cpl_calloc(wavenumber_selected_len, sizeof(double));
        wavenumber_val = first_element + 4*born;
        for (int i = 0; i < wavenumber_selected_len; i++) {
            wavenumber_selected_RE[order][i] = wavenumber_val;
            wavenumber_val += step_selected;
        }
        for (int i = 0; i < wavenumber_selected_len; i++) {
            if (wavenumber_selected_RE[order][i] == 0.0) {
                espdr_msg_warning("wavenumber_selected[%d][%d] is 0.0!!!", order, i);
            } else {
                wavenumber_selected_RE[order][i] = 100000000.0 / wavenumber_selected_RE[order][i]; // [A] --> [1/cm]
            }
        }
        
        cpl_vector_delete(wave_vector_diff);
        cpl_vector_delete(wave_vector);
    }
    
    //espdr_msg("Wavenumber grid done");
    
    return (cpl_error_get_code());
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Select orders
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_select_orders(double **wavenumber_grid,
                                   int *wavenumber_len,
                                   int ny_s2d,
                                   espdr_hitran_line *hitran_table,
                                   int lines_nb,
                                   int *selected_orders) {
    
    //espdr_msg("Selecting orders for the molecule: %s", hitran_table[0].molecule);
    for (int order = 0; order < ny_s2d; order++) {
        
        double start_wn = wavenumber_grid[order][0];
        double end_wn = wavenumber_grid[order][wavenumber_len[order]-1];
        for (int i = 0; i < lines_nb; i++) {
            if ((hitran_table[i].wave_number > end_wn) &&
                (hitran_table[i].wave_number < start_wn)){
                selected_orders[order] = 1;
            }
        }
        //espdr_msg("Order %d selected? : %d", order, selected_orders[order]);
    }
    
    return (cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Fit the telluric model
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code espdr_compute_gradient(double *input,
                                             int length,
                                             double *output) {
    
    if (length < 1) {
        espdr_msg_warning("espdr_compute_gradient: the input array has length 0");
        return (CPL_ERROR_NULL_INPUT);
    }
    
    if (length < 2) {
        output[0] = input[0];
    } else {
        output[0] = input[1] - input[0];
        for (int i = 1; i < length -1; i++) {
            output[i] = (input[i+1] - input[i-1])/2.0;
        }
        output[length-1] = input[length-1] - input[length-2];
    }
    
    return (cpl_error_get_code());
}

/*----------------------------------------------------------------------------*/
/**
 @brief     Fit the telluric model
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

double espdr_fit_telluric_model(double Pressure,
                                double PWV_airmass,
                                char *molecule,
                                double Temp_offseted,
                                double qt_used,
                                double airmass,
                                double N_mol,
                                double M_mol,
                                espdr_inst_config *inst_config,
                                double **wave,
                                double **wavenumber_grid,
                                int *wavenumber_len,
                                int *selected_orders,
                                espdr_hitran_line *hitran_table,
                                int hitran_lines_nb,
                                cpl_image *res_map,
                                cpl_image *wave_matrix,
                                cpl_image *dll_matrix,
                                cpl_image *s2d_qual,
                                cpl_table *telluric_mask,
                                double rv_step,
                                double berv,
                                double berv_max,
                                int nx_s2d,
                                int ny_s2d,
                                int nx_ccf_in,
                                cpl_array *RV_table,
                                cpl_image *CCF_flux,
                                int save_flag,
                                double *CCF_flux_data_RE,
                                double *CCF_model_flux_data_RE,
                                cpl_image **spectrum_vs_model_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    double PWV_molecule = 0.0;
    double *hitran_wave_number_scaled = (double *)cpl_calloc(hitran_lines_nb, sizeof(double));
    double *hitran_intensity_scaled = (double *)cpl_calloc(hitran_lines_nb, sizeof(double));
    double *hitran_lorenzian_hwhm_scaled = (double *)cpl_calloc(hitran_lines_nb, sizeof(double));
    double *hitran_gaussian_hwhm_scaled = (double *)cpl_calloc(hitran_lines_nb, sizeof(double));
    int index;
    double fit_result = 0.0;
    //clock_t start_model, end_model, start_conv, end_conv, start_ccf, end_ccf;
    //clock_t start_order, end_order, start_profile, end_profile;
    double cpu_time_used;
    
    espdr_ensure(nx_s2d != cpl_image_get_size_x(res_map), CPL_ERROR_INCOMPATIBLE_INPUT,
                 "espdr_fit_telluric_model failed: x sizes of s2d and res_map differs: %s",
                 cpl_error_get_message_default(CPL_ERROR_INCOMPATIBLE_INPUT));
    espdr_ensure(ny_s2d != cpl_image_get_size_y(res_map), CPL_ERROR_INCOMPATIBLE_INPUT,
                 "espdr_fit_telluric_model failed: y sizes of s2d and res_map differs: %s",
                 cpl_error_get_message_default(CPL_ERROR_INCOMPATIBLE_INPUT));
    
    //espdr_msg("Starting espdr_fit_telluric_model for %s, pressure: %f, PWV: %f", molecule, Pressure, PWV_airmass);
    
    if (strcmp(molecule, "H2O") == 0) {
        PWV_molecule = PWV_airmass; // for H2O
    } else {
        if (strcmp(molecule, "O2") == 0) {
            PWV_molecule = PWV_airmass * 1e+06; // for O2
        } else {
            PWV_molecule = PWV_airmass * 1e+07; // for CH4 and CO2
        }
    }
    //espdr_msg("molecule %s: obs Temp = %f, Qt = %f, PWV = %f", molecule, Temp, qt_used, PWV_molecule);
    
    for (int i = 0 ; i < hitran_lines_nb; i++) {
        hitran_wave_number_scaled[i] = hitran_table[i].wave_number + hitran_table[i].delta_air * Pressure;
        hitran_intensity_scaled[i] = hitran_table[i].intensity * (Q_T_296K / qt_used) *
                                    (exp(-C2 * hitran_table[i].Epp/Temp_offseted)) / (exp(-C2 * hitran_table[i].Epp/296.0)) *
                                    (1.0 - exp(-C2 * hitran_table[i].wave_number/Temp_offseted)) / (1.0 - exp(-C2 * hitran_table[i].wave_number/296.0));
        hitran_lorenzian_hwhm_scaled[i] = pow(296.0/Temp_offseted, hitran_table[i].n_air) * hitran_table[i].gamma_air * Pressure; // assuming P_mol = 0.0
        
        if (strcmp(molecule, "H2O") != 0) {
            hitran_gaussian_hwhm_scaled[i] = hitran_table[i].wave_number / LIGHT_SPEED * sqrt(2.0 * NA * KB * Temp_offseted * log(2.0) / (0.001 * M_mol));
        }
    }
    
    int nb_of_orders_with_lines = 0;
    int *hitran_nb_per_order = (int *)cpl_calloc(ny_s2d, sizeof(int));
    double **hitran_intensity = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    double **hitran_wave_number = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    double **hitran_lorenzian_hwhm = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    double **hitran_gaussian_hwhm = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    double **spectrum = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    double **telluric_spectrum = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    
    /* Model computation */
    //espdr_msg("Starting to compute the model");
    //start_model = clock();
    int total_lines_nb = 0;
    for (int order = 0; order < ny_s2d; order++) {
        //start_order = clock();
        spectrum[order] = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
        telluric_spectrum[order] = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
        
        if (selected_orders[order] == 1) {
            //espdr_msg("Fitting order %d", order+1);
            
            double start_wavenumber_sel = wavenumber_grid[order][0];
            double end_wavenumber_sel = wavenumber_grid[order][wavenumber_len[order]-1];
            hitran_nb_per_order[order] = 0;
            for (int i = 0; i < hitran_lines_nb; i++) {
                if ((hitran_table[i].wave_number > end_wavenumber_sel) &&
                    (hitran_table[i].wave_number < start_wavenumber_sel)){
                    //espdr_msg("line %d for molecule: %s, wavenumber: %f, intesity: %e",
                    //          i, hitran_table[i].molecule, hitran_table[i].wave_number, hitran_table[i].intensity);
                    hitran_nb_per_order[order]++;
                }
            }
            //total_lines_nb += hitran_nb_per_order[order];
            if (hitran_nb_per_order[order] > 0) {
                nb_of_orders_with_lines++;
                //espdr_msg("For molecule %s, nb of hitran lines for order %d is %d",
                //          molecule, order+1, hitran_nb_per_order[order]);
            }
            
            hitran_intensity[order] = (double *)cpl_calloc(hitran_nb_per_order[order], sizeof(double));
            hitran_wave_number[order] = (double *)cpl_calloc(hitran_nb_per_order[order], sizeof(double));
            hitran_lorenzian_hwhm[order] = (double *)cpl_calloc(hitran_nb_per_order[order], sizeof(double));
            hitran_gaussian_hwhm[order] = (double *)cpl_calloc(hitran_nb_per_order[order], sizeof(double));
            index = 0;
            for (int i = 0; i < hitran_lines_nb; i++) {
                if ((hitran_table[i].wave_number > end_wavenumber_sel) &&
                    (hitran_table[i].wave_number < start_wavenumber_sel)){
                    hitran_intensity[order][index] = hitran_intensity_scaled[i];
                    hitran_wave_number[order][index] = hitran_wave_number_scaled[i];
                    hitran_lorenzian_hwhm[order][index] = hitran_lorenzian_hwhm_scaled[i];
                    if (strcmp(molecule, "H2O") != 0) {
                        hitran_gaussian_hwhm[order][index] = hitran_gaussian_hwhm_scaled[i];
                    }
                    index++;
                }
            }
            
            double line_profile = 0.0;
            /* line's profile is computed only between -1000 and +1000, instead of the whole order */
            for (int i = 0; i < hitran_nb_per_order[order]; i++) {
                int i_grid = 0;
                while ((i_grid < wavenumber_len[order]) && (hitran_wave_number[order][i] < wavenumber_grid[order][i_grid])) {
                    i_grid++;
                }
                if (i_grid == wavenumber_len[order]) {
                    espdr_msg_warning("The line %d with wavenumber %f is outside order %d (%f - %f), not using",
                                      i, hitran_wave_number[order][i], order, wavenumber_grid[order][0],
                                      wavenumber_grid[order][wavenumber_len[order]-1]);
                } else {
                    
                    int j_start = (i_grid - HITRAN_LINE_PROFILE_WIDTH) >= 0 ?
                                    i_grid - HITRAN_LINE_PROFILE_WIDTH : 0;
                    int j_end = (i_grid + HITRAN_LINE_PROFILE_WIDTH) < wavenumber_len[order] ?
                                    i_grid + HITRAN_LINE_PROFILE_WIDTH : wavenumber_len[order];
                    
                    for (int j = j_start; j < j_end; j++) {
                        if (strcmp(molecule, "H2O") == 0) {
                            line_profile = 1.0 / M_PI * hitran_lorenzian_hwhm[order][i] /
                                            (hitran_lorenzian_hwhm[order][i]*hitran_lorenzian_hwhm[order][i] +
                                             (wavenumber_grid[order][j] - hitran_wave_number[order][i]) *
                                             (wavenumber_grid[order][j] - hitran_wave_number[order][i]));
                        } else {
                            line_profile = espdr_compute_voigt_profile(wavenumber_grid[order][j],
                                                                       hitran_gaussian_hwhm[order][i],
                                                                       hitran_lorenzian_hwhm[order][i],
                                                                       hitran_wave_number[order][i]);
                        }
                        spectrum[order][j] += hitran_intensity[order][i] * line_profile;
                    }
                }
            }
            
            for (int j = 0; j < wavenumber_len[order]; j++) {
                telluric_spectrum[order][j] = exp(-PWV_molecule * N_mol * spectrum[order][j]);
            }
            
            cpl_free(hitran_intensity[order]);
            cpl_free(hitran_wave_number[order]);
            cpl_free(hitran_lorenzian_hwhm[order]);
            cpl_free(hitran_gaussian_hwhm[order]);

        } else {
            for (int j = 0; j < wavenumber_len[order]; j++) {
                telluric_spectrum[order][j] = 1.0;
            }
        }
        //end_order = clock();
        //cpu_time_used = ((double) (end_order - start_order)) / CLOCKS_PER_SEC;
        //espdr_msg("CPU TIME used to compute the model for order %d (%d lines): %f sec",
        //          order, hitran_nb_per_order[order], cpu_time_used);
    } // end order
    //end_model = clock();
    //cpu_time_used = ((double) (end_model - start_model)) / CLOCKS_PER_SEC;
    //espdr_msg("CPU TIME used to compute the model: %f sec", cpu_time_used);
    //espdr_msg("Total nb of hitran lines for %s: %d", hitran_table[0].molecule, total_lines_nb);
    
    cpl_free(hitran_wave_number_scaled);
    cpl_free(hitran_intensity_scaled);
    cpl_free(hitran_lorenzian_hwhm_scaled);
    cpl_free(hitran_gaussian_hwhm_scaled);
    
    cpl_free(hitran_intensity);
    cpl_free(hitran_wave_number);
    cpl_free(hitran_lorenzian_hwhm);
    cpl_free(hitran_gaussian_hwhm);
    
    //espdr_msg("Opacity & telluric spectra computed");
    
#if 0
    // Save the telluric spectrum for test
    
    int max_len = 0;
    for (int order = 0; order < ny_s2d; order++) {
        if (max_len < wavenumber_len[order]) {
            max_len = wavenumber_len[order];
        }
    }
    double *opacity_for_save = (double *)cpl_calloc(ny_s2d*max_len, sizeof(double));
    double *telluric_for_save = (double *)cpl_calloc(ny_s2d*max_len, sizeof(double));

    index = 0;
    for (int order = 0; order < ny_s2d; order++) {
        for (int j = 0; j < wavenumber_len[order]; j++) {
            opacity_for_save[index] = spectrum[order][j];
            telluric_for_save[index] = telluric_spectrum[order][j];
            index++;
        }
        for (int j = wavenumber_len[order]; j < max_len; j++) {
            opacity_for_save[index] = 0.0;
            telluric_for_save[index] = 0.0;
            index++;
        }
    }
    
    cpl_image *opacity_test = cpl_image_wrap_double(max_len, ny_s2d, opacity_for_save);
    char opacity_filename[64];
    sprintf(opacity_filename, "%s_opacity_spectrum_%s_for_test.fits", inst_config->instrument, molecule);
    espdr_msg("Saving opacity in %s", opacity_filename);
    my_error = cpl_image_save(opacity_test, opacity_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_unwrap(opacity_test);
    cpl_free(opacity_for_save);
    
    cpl_image *telluric_test = cpl_image_wrap_double(max_len, ny_s2d, telluric_for_save);
    char telluric_filename[64];
    sprintf(telluric_filename, "%s_telluric_spectrum_%s_for_test.fits", inst_config->instrument, molecule);
    espdr_msg("Saving telluric in %s", telluric_filename);
    my_error = cpl_image_save(telluric_test, telluric_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    cpl_image_unwrap(telluric_test);
    cpl_free(telluric_for_save);
#endif
    
    
    // Convolution
    
    //espdr_msg("Starting convolution");
    //start_conv = clock();
    double **telluric_spectrum_conv = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    double **telluric_spectrum_interp = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    for (int order = 0; order < ny_s2d; order++) {
        
        telluric_spectrum_interp[order] = (double *)cpl_calloc(nx_s2d, sizeof(double));
        telluric_spectrum_conv[order] = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
        if (selected_orders[order] == 1) {
            //espdr_msg("Convolution for order %d", order);
            
            // The non-extracted pixels, like shorter blue orders in ESPRESSO, are not convoluted
            cpl_image *s2d_qual_order_img = cpl_image_extract(s2d_qual, 1, order+1, nx_s2d, order+1);
            int *s2d_qual_order_data = cpl_image_get_data_int(s2d_qual_order_img);
            int left_i = 0;
            while ((left_i < nx_s2d) && (s2d_qual_order_data[left_i] == OTHER_BAD_PIXEL)) {
                //espdr_msg("s2d_qual[%d] = %d", left_i, s2d_qual_order_data[left_i]);
                left_i++;
            }
            left_i++; // for cpl functions pixels start at 1
            int right_i = nx_s2d-1;
            while ((right_i >= 0) && (s2d_qual_order_data[right_i] == OTHER_BAD_PIXEL)) {
                right_i--;
            }
            right_i++; // for cpl functions pixels start at 1
            int S2D_len = right_i - left_i + 1;
            
            cpl_image *res_map_order_img = cpl_image_extract(res_map, left_i, order+1, right_i, order+1);
            double *res_map_order_data = cpl_image_get_data_double(res_map_order_img);
            cpl_image *wave_matrix_order_img = cpl_image_extract(wave_matrix, left_i, order+1, right_i, order+1);
            double *wave_matrix_order_data = cpl_image_get_data_double(wave_matrix_order_img);
            double *res_interp_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            //espdr_msg("Interpolation between: %f and %f, n1 = %d, n2 = %d",
            //          wave_matrix_order_data[0], wave_matrix_order_data[nx_s2d-1], nx_s2d, wavenumber_len[order]);
            my_error = espdr_spline_bounded_interp_exterp(wave_matrix_order_data, res_map_order_data, S2D_len,
                                                          wave[order], res_interp_order, wavenumber_len[order]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_spline_interpolation failed for order %d: %s",
                         order+1, cpl_error_get_message_default(my_error));
           //espdr_msg("Interpolation done");

            double *fwhm_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            double *ww_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            double *sp2_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            double *ew_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            for (int i = 0; i < wavenumber_len[order]; i++) {
                fwhm_order[i] = (LIGHT_SPEED/1000.0) / res_interp_order[i];
                ew_order[i] = (fwhm_order[i] / 2.0) / sqrt(2.0 * log(2.0));
            }
            
            // Compute the range of the convolution to speed up the computation
            double *wave_order_gradient = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            my_error = espdr_compute_gradient(wave[order], wavenumber_len[order], wave_order_gradient);
            cpl_vector *fwhm_order_vector = cpl_vector_wrap(wavenumber_len[order], fwhm_order);
            cpl_vector *wave_order_vector = cpl_vector_wrap(wavenumber_len[order], wave[order]);
            cpl_vector *wave_order_gradient_vector = cpl_vector_wrap(wavenumber_len[order], wave_order_gradient);
            my_error = cpl_vector_divide(wave_order_gradient_vector, wave_order_vector);
            int range_scan = (int)ceil(cpl_vector_get_max(fwhm_order_vector) /
                                       (cpl_vector_get_median(wave_order_gradient_vector) * LIGHT_SPEED/1000.0) * 3.0);
            //espdr_msg("fwhm max: %f, gradient median: %f",
            //          cpl_vector_get_max(fwhm_order_vector), cpl_vector_get_median(wave_order_gradient_vector));
            cpl_vector_unwrap(fwhm_order_vector);
            cpl_vector_unwrap(wave_order_vector);
            cpl_vector_unwrap(wave_order_gradient_vector);
            cpl_free(wave_order_gradient);
            
            for (int i = 0; i < wavenumber_len[order]; i++) {
                int start_offset = 0;
                if (i < range_scan) start_offset = -i; else start_offset = -range_scan;
                int end_offset = 0;
                if (wavenumber_len[order] - i < range_scan) end_offset = wavenumber_len[order] - i; else end_offset = range_scan;
                for (int offset = start_offset; offset < end_offset; offset++) {
                    double dv = ((wave[order][i] / wave[order][i+offset]) - 1.0) * (LIGHT_SPEED/1000.0);
                    double w = exp(-0.5*(dv/ew_order[i])*(dv/ew_order[i]));
                    sp2_order[i] += telluric_spectrum[order][i+offset]*w;
                    ww_order[i] += w;
                }
                telluric_spectrum_conv[order][i] = sp2_order[i] / ww_order[i];
            }
            
            double *dl = (double *)cpl_calloc(nx_s2d, sizeof(double));
            /* one has to be sure the loop does not hit size of dl nor of wave_matrix_order_data */
            cpl_size nx_wave_order_data = cpl_image_get_size_x(wave_matrix_order_img);
            for (int i = 0; i < nx_s2d-1 && i < nx_wave_order_data-1; i++) {
                dl[i+1] = (wave_matrix_order_data[i+1] - wave_matrix_order_data[i]) / 2.0;
            }
            dl[0] = dl[1];

            for (int i = 0; i < left_i-1; i++) {
                telluric_spectrum_interp[order][i] = 1.0;
            }
            for (int i = left_i-1; i < right_i; i++) {
                int first_index = 0, last_index = wavenumber_len[order]-1, index = 0;
                double wave_first = wave_matrix_order_data[i-left_i+1] - dl[i-left_i+1];
                double wave_last = wave_matrix_order_data[i-left_i+1] + dl[i-left_i+1];
                while (wave[order][index] < wave_first) index++;
                first_index = index;
                while (wave[order][index] <= wave_last) index++;
                last_index = index;
                
                // take from telluric_spectrum_conv the values for wavelengths between <-dl, +dl> in wave_matrix
                for (int j = first_index; j < last_index; j++) {
                    telluric_spectrum_interp[order][i] += telluric_spectrum_conv[order][j];
                }
                if (last_index > first_index) {
                    telluric_spectrum_interp[order][i] = telluric_spectrum_interp[order][i] / (last_index - first_index);
                }
            }
            for (int i = right_i; i < nx_s2d; i++) {
                telluric_spectrum_interp[order][i] = 1.0;
            }
            
            cpl_free(dl);
            cpl_free(fwhm_order);
            cpl_free(ww_order);
            cpl_free(sp2_order);
            cpl_free(ew_order);
            cpl_free(res_interp_order);
            cpl_image_delete(res_map_order_img);
            cpl_image_delete(wave_matrix_order_img);
                
        } else {
            //espdr_msg("No convolution for order %d", order);
            for (int j = 0; j < nx_s2d; j++) {
                telluric_spectrum_interp[order][j] = 1.0;
            }
        }
    }
    //end_conv = clock();
    //cpu_time_used = ((double) (end_conv - start_conv)) / CLOCKS_PER_SEC;
    //espdr_msg("CPU TIME used to convolve: %f sec", cpu_time_used);

    double *telluric_interp_for_save = (double *)cpl_calloc(ny_s2d*nx_s2d, sizeof(double));
    index = 0;
    for (int order = 0; order < ny_s2d; order++) {
        for (int j = 0; j < nx_s2d; j++) {
            telluric_interp_for_save[index] = telluric_spectrum_interp[order][j];
            index++;
        }
    }
    
    cpl_image *telluric_interp_img = cpl_image_wrap_double(nx_s2d, ny_s2d, telluric_interp_for_save);
    
    if (save_flag == 1) {
        char telluric_interp_filename[64];
        sprintf(telluric_interp_filename, "%s_telluric_conv_spectrum_%s_for_test.fits", inst_config->instrument, molecule);
        espdr_msg("Saving telluric interp in %s", telluric_interp_filename);
        my_error = cpl_image_save(telluric_interp_img, telluric_interp_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    }
    
    /* Model CCFs computation */
    //start_ccf = clock();
    //espdr_msg("CCF computation");
    cpl_image *spectrum_error = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    cpl_image *spectrum_qual = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_INT);
    cpl_image *spectrum_blaze = cpl_image_new(nx_s2d, ny_s2d, CPL_TYPE_DOUBLE);
    my_error = cpl_image_fill_window(spectrum_error, 1, 1, nx_s2d, ny_s2d, 1.0);
    my_error = cpl_image_fill_window(spectrum_qual, 1, 1, nx_s2d, ny_s2d, 0);
    my_error = cpl_image_fill_window(spectrum_blaze, 1, 1, nx_s2d, ny_s2d, 1.0);
    
    int nx_ccf = cpl_array_get_size(RV_table);

    cpl_image *CCF_model_flux  = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
    cpl_image *CCF_model_error = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_DOUBLE);
    cpl_image *CCF_model_qual  = cpl_image_new(nx_ccf, ny_s2d+1, CPL_TYPE_INT);
    
    //espdr_msg("Computing the CCF on tellurics model for %s", hitran_table[0].molecule);
    //cpl_table_dump(telluric_mask, 0, 100, NULL);
    my_error = espdr_compute_CCF(wave_matrix, dll_matrix, telluric_interp_img,
                                 spectrum_error, spectrum_blaze, spectrum_qual,
                                 RV_table, telluric_mask, rv_step, berv, berv_max,
                                 &CCF_model_flux, &CCF_model_error, &CCF_model_qual);
    espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                 "espdr_compute_CCF failed for model (%s): %s",
                 hitran_table[0].molecule, cpl_error_get_message_default(my_error));

    if (save_flag == 1) {
        char ccf_model_filename[64];
        sprintf(ccf_model_filename, "%s_CCF_model_%s_for_test.fits", inst_config->instrument, molecule);
        my_error = cpl_image_save(CCF_model_flux, ccf_model_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    }
    
    int pxl;
    int RV_continuum_len = round(nx_ccf/4);
    int CCF_continuum_length = 2*RV_continuum_len; // (0 .. 20 + 60 .. 80)
    cpl_vector *CCF_continuum_model_vector = cpl_vector_new(CCF_continuum_length);
    // Normalization of the sum
    for (int i = 0; i < RV_continuum_len; i++) {
        my_error = cpl_vector_set(CCF_continuum_model_vector, i, cpl_image_get(CCF_model_flux, i+1, ny_s2d+1, &pxl));
    }
    
    int rv_index_shift = nx_ccf - 2*RV_continuum_len;
    for (int i = nx_ccf - RV_continuum_len; i < nx_ccf; i++) {
        my_error = cpl_vector_set(CCF_continuum_model_vector, i-rv_index_shift, cpl_image_get(CCF_model_flux, i+1, ny_s2d+1, &pxl));
    }
    
    double CCF_model_median = cpl_vector_get_median(CCF_continuum_model_vector);
    //espdr_msg("CCF_model_median: %f", CCF_model_median);
    my_error = cpl_image_divide_scalar(CCF_model_flux, CCF_model_median);
                                 
    cpl_vector_delete(CCF_continuum_model_vector);

    if (save_flag == 1) {
        char ccf_norm_model_filename[64];
        sprintf(ccf_norm_model_filename, "%s_CCF_norm_model_%s_for_test.fits", inst_config->instrument, molecule);
        my_error = cpl_image_save(CCF_model_flux, ccf_norm_model_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    }
    
    cpl_vector *CCF_flux_vector = cpl_vector_new_from_image_row(CCF_flux, ny_s2d+1);
    cpl_vector *CCF_model_flux_vector = cpl_vector_new_from_image_row(CCF_model_flux, ny_s2d+1);
    double *CCF_flux_data = cpl_vector_get_data(CCF_flux_vector);
    double *CCF_model_flux_data = cpl_vector_get_data(CCF_model_flux_vector);
    int ccf_flux_len = cpl_vector_get_size(CCF_flux_vector);
    fit_result = 0.0;
    for (int i = 0; i < ccf_flux_len; i++) {
        fit_result = fit_result + (CCF_flux_data[i] - CCF_model_flux_data[i]) *
                                    (CCF_flux_data[i] - CCF_model_flux_data[i]);
    }
    fit_result = fit_result / ccf_flux_len;
    
    for (int i = 0; i < ccf_flux_len; i++) {
        CCF_flux_data_RE[i] = CCF_flux_data[i];
        CCF_model_flux_data_RE[i] = CCF_model_flux_data[i];
    }
    
    //end_conv = clock();
    //cpu_time_used = ((double) (end_conv - start_conv)) / CLOCKS_PER_SEC;
    //espdr_msg("CPU TIME used for CCF: %f sec", cpu_time_used);
    
    cpl_image *spectrum_vs_model_img = cpl_image_divide_create(CCF_flux, CCF_model_flux);
    my_error = cpl_image_subtract_scalar(spectrum_vs_model_img, 1.0);
    
    *spectrum_vs_model_RE = cpl_image_duplicate(spectrum_vs_model_img);
    
    cpl_image_delete(spectrum_vs_model_img);
    cpl_image_delete(spectrum_error);
    cpl_image_delete(spectrum_qual);
    cpl_image_delete(spectrum_blaze);
    cpl_image_delete(CCF_model_flux);
    cpl_image_delete(CCF_model_error);
    cpl_image_delete(CCF_model_qual);
    cpl_image_unwrap(telluric_interp_img);
    cpl_free(telluric_interp_for_save);

    for (int order = 0; order < ny_s2d; order++) {
        cpl_free(spectrum[order]);
        cpl_free(telluric_spectrum[order]);
        cpl_free(telluric_spectrum_conv[order]);
        cpl_free(telluric_spectrum_interp[order]);
    }
    cpl_free(spectrum);
    cpl_free(telluric_spectrum);
    cpl_free(telluric_spectrum_conv);
    cpl_free(telluric_spectrum_interp);
    
    //espdr_msg("Fit done for molecule %s with result %f", molecule, fit_result);
    
    //espdr_msg("Fit finished");
    
    return (fit_result);
}


/*----------------------------------------------------------------------------*/
/**
 @brief     Function to be provided for the non-linear least-squares fitting
 @param
 @param
 @param
 @return   double
 */
/*----------------------------------------------------------------------------*/

int espdr_telluric_fit_least_squares(const gsl_vector *v,
                                     void *params,
                                     gsl_vector *f) {

    double fit_result;
    
    double pressure = gsl_vector_get(v, 0);
    double PWV_airmass = gsl_vector_get(v, 1);
    
    espdr_fit_params *input_params = (espdr_fit_params *)params;
    
    fit_result = espdr_fit_telluric_model(pressure,
                                          PWV_airmass,
                                          input_params->molecule,
                                          input_params->Temp_offseted,
                                          input_params->qt_used,
                                          input_params->airmass,
                                          input_params->N_mol,
                                          input_params->M_mol,
                                          input_params->inst_config,
                                          input_params->wave,
                                          input_params->wavenumber_grid,
                                          input_params->wavenumber_len,
                                          input_params->selected_orders,
                                          input_params->hitran_table,
                                          input_params->hitran_lines_nb,
                                          input_params->res_map,
                                          input_params->wave_matrix,
                                          input_params->dll_matrix,
                                          input_params->s2d_qual,
                                          input_params->telluric_mask,
                                          input_params->rv_step,
                                          input_params->berv,
                                          input_params->berv_max,
                                          input_params->nx_s2d,
                                          input_params->ny_s2d,
                                          input_params->nx_ccf,
                                          input_params->RV_table,
                                          input_params->CCF_flux_norm,
                                          input_params->save_flag,
                                          input_params->CCF_flux_data_RE,
                                          input_params->CCF_model_flux_data_RE,
                                          input_params->spectrum_vs_model_RE);
    
    for (int i = 0; i < input_params->nx_ccf; i++) {
        gsl_vector_set(f, i, input_params->CCF_flux_data_RE[i] - input_params->CCF_model_flux_data_RE[i]);
    }

    return GSL_SUCCESS;
}


void callback(const size_t iter,
              void *params,
              const gsl_multifit_nlinear_workspace *w) {
    
    gsl_vector *f = gsl_multifit_nlinear_residual(w);
    gsl_vector *x = gsl_multifit_nlinear_position(w);
    double rcond;

    /* compute reciprocal condition number of J(x) */
    gsl_multifit_nlinear_rcond(&rcond, w);

    espdr_msg("iter %2zu: pressure = %e, PWV = %e, cond(J) = %8.10f, |f(x)| = %.10f",
              iter, gsl_vector_get(x, 0), gsl_vector_get(x, 1), 1.0 / rcond, gsl_blas_dnrm2(f));
}



/*----------------------------------------------------------------------------*/
/**
 @brief     Compute the telluric model
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/

cpl_error_code espdr_compute_telluric_model(double *Pressure,
                                            double *PWV_molecule,
                                            double *Temp_offseted,
                                            double *qt_used,
                                            double airmass,
                                            double *N_mol,
                                            double *M_mol,
                                            int mol_nb,
                                            espdr_inst_config *inst_config,
                                            double **wave,
                                            double **wavenumber_grid,
                                            int *wavenumber_len,
                                            int **selected_orders,
                                            int *selected_orders_all,
                                            espdr_hitran_line **hitran_table,
                                            int *hitran_lines_nb,
                                            cpl_image *res_map,
                                            cpl_image *wave_matrix,
                                            cpl_image *s2d_qual,
                                            int nx_s2d,
                                            int ny_s2d,
                                            int save_flag,
                                            cpl_image **telluric_spectrum_RE) {

    cpl_error_code my_error = CPL_ERROR_NONE;
    int index;
    //clock_t start_model, end_model, start_conv, end_conv, start_ccf, end_ccf;
    //clock_t start_order, end_order, start_profile, end_profile;
    double cpu_time_used;
    char molecule[16];
    
    espdr_ensure(nx_s2d != cpl_image_get_size_x(res_map), CPL_ERROR_INCOMPATIBLE_INPUT,
                 "espdr_compute_telluric_model failed: x sizes of s2d and res_map differs: %s",
                 cpl_error_get_message_default(CPL_ERROR_INCOMPATIBLE_INPUT));
    espdr_ensure(ny_s2d != cpl_image_get_size_y(res_map), CPL_ERROR_INCOMPATIBLE_INPUT,
                 "espdr_compute_telluric_model failed: y sizes of s2d and res_map differs: %s",
                 cpl_error_get_message_default(CPL_ERROR_INCOMPATIBLE_INPUT));
    
    //espdr_msg("Starting espdr_fit_telluric_model for %s, pressure: %f, PWV: %f", molecule, Pressure, PWV_airmass);
    
    double **spectrum = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    for (int order = 0; order < ny_s2d; order++) {
        spectrum[order] = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
    }
    
    for (int mol = 0; mol < mol_nb; mol++) {
        
        strcpy(molecule, hitran_table[mol][0].molecule);
        espdr_msg("Computing profile for the molecule: %s", molecule);
        
        double *hitran_wave_number_scaled = (double *)cpl_calloc(hitran_lines_nb[mol], sizeof(double));
        double *hitran_intensity_scaled = (double *)cpl_calloc(hitran_lines_nb[mol], sizeof(double));
        double *hitran_lorenzian_hwhm_scaled = (double *)cpl_calloc(hitran_lines_nb[mol], sizeof(double));
        double *hitran_gaussian_hwhm_scaled = (double *)cpl_calloc(hitran_lines_nb[mol], sizeof(double));
        
        for (int i = 0 ; i < hitran_lines_nb[mol]; i++) {
            hitran_wave_number_scaled[i] = hitran_table[mol][i].wave_number + hitran_table[mol][i].delta_air * Pressure[mol];
            hitran_intensity_scaled[i] = hitran_table[mol][i].intensity * (Q_T_296K / qt_used[mol]) *
                                        (exp(-C2 * hitran_table[mol][i].Epp/Temp_offseted[mol])) / (exp(-C2 * hitran_table[mol][i].Epp/296.0)) *
                                        (1.0 - exp(-C2 * hitran_table[mol][i].wave_number/Temp_offseted[mol])) / (1.0 - exp(-C2 * hitran_table[mol][i].wave_number/296.0));
            hitran_lorenzian_hwhm_scaled[i] = pow(296.0/Temp_offseted[mol], hitran_table[mol][i].n_air) * hitran_table[mol][i].gamma_air * Pressure[mol]; // assuming P_mol = 0.0
            
            if (strcmp(molecule, "H2O") != 0) {
                hitran_gaussian_hwhm_scaled[i] = hitran_table[mol][i].wave_number / LIGHT_SPEED * sqrt(2.0 * NA * KB * Temp_offseted[mol] * log(2.0) / (0.001 * M_mol[mol]));
            }
        }
        
        int *hitran_nb_per_order = (int *)cpl_calloc(ny_s2d, sizeof(int));
        double **hitran_intensity = (double **)cpl_malloc(ny_s2d * sizeof(double *));
        double **hitran_wave_number = (double **)cpl_malloc(ny_s2d * sizeof(double *));
        double **hitran_lorenzian_hwhm = (double **)cpl_malloc(ny_s2d * sizeof(double *));
        double **hitran_gaussian_hwhm = (double **)cpl_malloc(ny_s2d * sizeof(double *));
        
        for (int order = 0; order < ny_s2d; order++) {
            
            if (selected_orders[mol][order] == 1) {
                //espdr_msg("Computing profile for order %d", order);
                double start_wavenumber_sel = wavenumber_grid[order][0];
                double end_wavenumber_sel = wavenumber_grid[order][wavenumber_len[order]-1];
                hitran_nb_per_order[order] = 0;
                for (int i = 0; i < hitran_lines_nb[mol]; i++) {
                    if ((hitran_table[mol][i].wave_number > end_wavenumber_sel) &&
                        (hitran_table[mol][i].wave_number < start_wavenumber_sel)){
                        hitran_nb_per_order[order]++;
                    }
                }
                hitran_intensity[order] = (double *)cpl_calloc(hitran_nb_per_order[order], sizeof(double));
                hitran_wave_number[order] = (double *)cpl_calloc(hitran_nb_per_order[order], sizeof(double));
                hitran_lorenzian_hwhm[order] = (double *)cpl_calloc(hitran_nb_per_order[order], sizeof(double));
                hitran_gaussian_hwhm[order] = (double *)cpl_calloc(hitran_nb_per_order[order], sizeof(double));
                index = 0;
                for (int i = 0; i < hitran_lines_nb[mol]; i++) {
                    if ((hitran_table[mol][i].wave_number > end_wavenumber_sel) &&
                        (hitran_table[mol][i].wave_number < start_wavenumber_sel)){
                        hitran_intensity[order][index] = hitran_intensity_scaled[i];
                        hitran_wave_number[order][index] = hitran_wave_number_scaled[i];
                        hitran_lorenzian_hwhm[order][index] = hitran_lorenzian_hwhm_scaled[i];
                        if (strcmp(molecule, "H2O") != 0) {
                            hitran_gaussian_hwhm[order][index] = hitran_gaussian_hwhm_scaled[i];
                        }
                        index++;
                    }
                }
                
                double line_profile = 0.0;
                /* line's profile is computed only between -1000 and +1000, instead of the whole order */
                for (int i = 0; i < hitran_nb_per_order[order]; i++) {
                    
                    int i_grid = 0;
                    while ((i_grid < wavenumber_len[order]) && (hitran_wave_number[order][i] < wavenumber_grid[order][i_grid])) {
                        i_grid++;
                    }
                    if (i_grid == wavenumber_len[order]) {
                        espdr_msg_warning("The line %d with wavenumber %f is outside order %d (%f - %f), not using",
                                          i, hitran_wave_number[order][i], order, wavenumber_grid[order][0],
                                          wavenumber_grid[order][wavenumber_len[order]-1]);
                    } else {
                        
                        int j_start = (i_grid - HITRAN_LINE_PROFILE_WIDTH) >= 0 ?
                                        i_grid - HITRAN_LINE_PROFILE_WIDTH : 0;
                        int j_end = (i_grid + HITRAN_LINE_PROFILE_WIDTH) < wavenumber_len[order] ?
                                        i_grid + HITRAN_LINE_PROFILE_WIDTH : wavenumber_len[order];
                        
                        for (int j = j_start; j < j_end; j++) {
                            if (strcmp(molecule, "H2O") == 0) {
                                line_profile = 1.0 / M_PI * hitran_lorenzian_hwhm[order][i] /
                                                (hitran_lorenzian_hwhm[order][i]*hitran_lorenzian_hwhm[order][i] +
                                                 (wavenumber_grid[order][j] - hitran_wave_number[order][i]) *
                                                 (wavenumber_grid[order][j] - hitran_wave_number[order][i]));
                            } else {
                                line_profile = espdr_compute_voigt_profile(wavenumber_grid[order][j],
                                                                           hitran_gaussian_hwhm[order][i],
                                                                           hitran_lorenzian_hwhm[order][i],
                                                                           hitran_wave_number[order][i]);
                            }
                            spectrum[order][j] += PWV_molecule[mol] * N_mol[mol] * hitran_intensity[order][i] * line_profile;
                        }
                    }
                    
                } // for hitran_nb_per_order
                
                cpl_free(hitran_intensity[order]);
                cpl_free(hitran_wave_number[order]);
                cpl_free(hitran_lorenzian_hwhm[order]);
                cpl_free(hitran_gaussian_hwhm[order]);

            }  else { // if selected_orders
                //espdr_msg("Not computing profile for order %d", order);
                //for (int j = 0; j < wavenumber_len[order]; j++) {
                //    spectrum[order][j] = 0.0;
                //}
            }
        } // for orders
        
        cpl_free(hitran_wave_number_scaled);
        cpl_free(hitran_intensity_scaled);
        cpl_free(hitran_lorenzian_hwhm_scaled);
        cpl_free(hitran_gaussian_hwhm_scaled);
        
        cpl_free(hitran_intensity);
        cpl_free(hitran_wave_number);
        cpl_free(hitran_lorenzian_hwhm);
        cpl_free(hitran_gaussian_hwhm);
        
    } // for molecule
    
    double **telluric_spectrum = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    for (int order = 0; order < ny_s2d; order++) {
        telluric_spectrum[order] = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
        for (int j = 0; j < wavenumber_len[order]; j++) {
            telluric_spectrum[order][j] = exp(-spectrum[order][j]);
        }
    }
    
    // Save the telluric spectrum for test
    
    if (save_flag == 1) {
        int max_len = 0;
        for (int order = 0; order < ny_s2d; order++) {
            if (max_len < wavenumber_len[order]) {
                max_len = wavenumber_len[order];
            }
        }
        double *opacity_for_save = (double *)cpl_calloc(ny_s2d*max_len, sizeof(double));
        double *telluric_for_save = (double *)cpl_calloc(ny_s2d*max_len, sizeof(double));

        index = 0;
        for (int order = 0; order < ny_s2d; order++) {
            for (int j = 0; j < wavenumber_len[order]; j++) {
                opacity_for_save[index] = spectrum[order][j];
                telluric_for_save[index] = telluric_spectrum[order][j];
                index++;
            }
            for (int j = wavenumber_len[order]; j < max_len; j++) {
                opacity_for_save[index] = 0.0;
                telluric_for_save[index] = 0.0;
                index++;
            }
        }
        
        cpl_image *opacity_test = cpl_image_wrap_double(max_len, ny_s2d, opacity_for_save);
        char opacity_filename[64];
        sprintf(opacity_filename, "%s_opacity_spectrum_all_for_test.fits", inst_config->instrument);
        espdr_msg("Saving opacity in %s", opacity_filename);
        cpl_image_save(opacity_test, opacity_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        cpl_image_unwrap(opacity_test);
        cpl_free(opacity_for_save);
        
        cpl_image *telluric_test = cpl_image_wrap_double(max_len, ny_s2d, telluric_for_save);
        char telluric_filename[64];
        sprintf(telluric_filename, "%s_telluric_spectrum_all_for_test.fits", inst_config->instrument);
        espdr_msg("Saving telluric in %s", telluric_filename);
        cpl_image_save(telluric_test, telluric_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
        cpl_image_unwrap(telluric_test);
        cpl_free(telluric_for_save);
    }
    
    
    // Convolution
    
    //espdr_msg("Starting convolution");
    //start_conv = clock();
    
    espdr_msg("Convolving the spectrum");
    double **telluric_spectrum_conv = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    double **telluric_spectrum_interp = (double **)cpl_malloc(ny_s2d * sizeof(double *));
    for (int order = 0; order < ny_s2d; order++) {
        
        telluric_spectrum_interp[order] = (double *)cpl_calloc(nx_s2d, sizeof(double));
        telluric_spectrum_conv[order] = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
        if (selected_orders_all[order] == 1) {
            //espdr_msg("Convolution for order %d", order);
            
            // The non-extracted pixels, like shorter blue orders in ESPRESSO, are not convoluted
            cpl_image *s2d_qual_order_img = cpl_image_extract(s2d_qual, 1, order+1, nx_s2d, order+1);
            int *s2d_qual_order_data = cpl_image_get_data_int(s2d_qual_order_img);
            int left_i = 0;
            while ((left_i < nx_s2d) && (s2d_qual_order_data[left_i] == OTHER_BAD_PIXEL)) {
                //espdr_msg("s2d_qual[%d] = %d", left_i, s2d_qual_order_data[left_i]);
                left_i++;
            }
            left_i++; // for cpl functions pixels start at 1
            int right_i = nx_s2d-1;
            while ((right_i >= 0) && (s2d_qual_order_data[right_i] == OTHER_BAD_PIXEL)) {
                right_i--;
            }
            right_i++; // for cpl functions pixels start at 1
            int S2D_len = right_i - left_i + 1;
            
            cpl_image *res_map_order_img = cpl_image_extract(res_map, left_i, order+1, right_i, order+1);
            double *res_map_order_data = cpl_image_get_data_double(res_map_order_img);
            cpl_image *wave_matrix_order_img = cpl_image_extract(wave_matrix, left_i, order+1, right_i, order+1);
            double *wave_matrix_order_data = cpl_image_get_data_double(wave_matrix_order_img);
            double *res_interp_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            
            my_error = espdr_spline_bounded_interp_exterp(wave_matrix_order_data, res_map_order_data, S2D_len,
                                                          wave[order], res_interp_order, wavenumber_len[order]);
            espdr_ensure(my_error != CPL_ERROR_NONE, my_error,
                         "espdr_spline_interpolation failed for order %d: %s",
                         order+1, cpl_error_get_message_default(my_error));

            double *fwhm_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            double *ww_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            double *sp2_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            double *ew_order = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            for (int i = 0; i < wavenumber_len[order]; i++) {
                fwhm_order[i] = (LIGHT_SPEED/1000.0) / res_interp_order[i];
                ew_order[i] = (fwhm_order[i] / 2.0) / sqrt(2.0 * log(2.0));
            }
            
            // Compute the range of the convolution to speed up the computation
            double *wave_order_gradient = (double *)cpl_calloc(wavenumber_len[order], sizeof(double));
            my_error = espdr_compute_gradient(wave[order], wavenumber_len[order], wave_order_gradient);
            cpl_vector *fwhm_order_vector = cpl_vector_wrap(wavenumber_len[order], fwhm_order);
            cpl_vector *wave_order_vector = cpl_vector_wrap(wavenumber_len[order], wave[order]);
            cpl_vector *wave_order_gradient_vector = cpl_vector_wrap(wavenumber_len[order], wave_order_gradient);
            cpl_vector_divide(wave_order_gradient_vector, wave_order_vector);
            int range_scan = (int)ceil(cpl_vector_get_max(fwhm_order_vector) /
                                       (cpl_vector_get_median(wave_order_gradient_vector) * LIGHT_SPEED/1000.0) * 3.0);
            cpl_vector_unwrap(fwhm_order_vector);
            cpl_vector_unwrap(wave_order_vector);
            cpl_vector_unwrap(wave_order_gradient_vector);
            cpl_free(wave_order_gradient);
            
            for (int i = 0; i < wavenumber_len[order]; i++) {
                int start_offset = 0;
                if (i < range_scan) start_offset = -i; else start_offset = -range_scan;
                int end_offset = 0;
                if (wavenumber_len[order] - i < range_scan) end_offset = wavenumber_len[order] - i; else end_offset = range_scan;
                for (int offset = start_offset; offset < end_offset; offset++) {
                    double dv = ((wave[order][i] / wave[order][i+offset]) - 1.0) * (LIGHT_SPEED/1000.0);
                    double w = exp(-0.5*(dv/ew_order[i])*(dv/ew_order[i]));
                    sp2_order[i] += telluric_spectrum[order][i+offset]*w;
                    ww_order[i] += w;
                }
                telluric_spectrum_conv[order][i] = sp2_order[i] / ww_order[i];
            }
            
            double *dl = (double *)cpl_calloc(S2D_len, sizeof(double));
            for (int i = 0; i < S2D_len-1; i++) {
                dl[i+1] = (wave_matrix_order_data[i+1] - wave_matrix_order_data[i]) / 2.0;
            }
            dl[0] = dl[1];

            for (int i = 0; i < left_i-1; i++) {
                telluric_spectrum_interp[order][i] = 1.0;
            }
            for (int i = left_i-1; i < right_i; i++) {
                int first_index = 0, last_index = wavenumber_len[order]-1, index = 0;
                double wave_first = wave_matrix_order_data[i-left_i+1] - dl[i-left_i+1];
                double wave_last = wave_matrix_order_data[i-left_i+1] + dl[i-left_i+1];
                while (wave[order][index] < wave_first) index++;
                first_index = index;
                while (wave[order][index] <= wave_last) index++;
                last_index = index;
                
                // take from telluric_spectrum_conv the values for wavelengths between <-dl, +dl> in wave_matrix
                for (int j = first_index; j < last_index; j++) {
                    telluric_spectrum_interp[order][i] += telluric_spectrum_conv[order][j];
                }
                if (last_index > first_index) {
                    telluric_spectrum_interp[order][i] = telluric_spectrum_interp[order][i] / (last_index - first_index);
                }
            }
            for (int i = right_i; i < nx_s2d; i++) {
                telluric_spectrum_interp[order][i] = 1.0;
            }
            
            cpl_free(dl);
            cpl_free(fwhm_order);
            cpl_free(ww_order);
            cpl_free(sp2_order);
            cpl_free(ew_order);
            cpl_free(res_interp_order);
            cpl_image_delete(res_map_order_img);
            cpl_image_delete(wave_matrix_order_img);
                
        } else {
            //espdr_msg("No convolution for order %d", order);
            for (int j = 0; j < nx_s2d; j++) {
                telluric_spectrum_interp[order][j] = 1.0;
            }
        }
    }
    //end_conv = clock();
    //cpu_time_used = ((double) (end_conv - start_conv)) / CLOCKS_PER_SEC;
    //espdr_msg("CPU TIME used to convolve: %f sec", cpu_time_used);

    double *telluric_interp_for_save = (double *)cpl_calloc(ny_s2d*nx_s2d, sizeof(double));
    index = 0;
    for (int order = 0; order < ny_s2d; order++) {
        for (int j = 0; j < nx_s2d; j++) {
            telluric_interp_for_save[index] = telluric_spectrum_interp[order][j];
            index++;
        }
    }
    
    cpl_image *telluric_interp_img = cpl_image_wrap_double(nx_s2d, ny_s2d, telluric_interp_for_save);
    
    if (save_flag == 1) {
        char telluric_interp_filename[64];
        sprintf(telluric_interp_filename, "%s_telluric_conv_spectrum_all_for_test.fits", inst_config->instrument);
        espdr_msg("Saving telluric interp in %s", telluric_interp_filename);
        cpl_image_save(telluric_interp_img, telluric_interp_filename, CPL_TYPE_DOUBLE, NULL, CPL_IO_CREATE);
    }
    
    *telluric_spectrum_RE = cpl_image_duplicate(telluric_interp_img);
    
    cpl_image_unwrap(telluric_interp_img);
    cpl_free(telluric_interp_for_save);

    for (int order = 0; order < ny_s2d; order++) {
        cpl_free(spectrum[order]);
        cpl_free(telluric_spectrum[order]);
        cpl_free(telluric_spectrum_conv[order]);
        cpl_free(telluric_spectrum_interp[order]);
    }
    cpl_free(spectrum);
    cpl_free(telluric_spectrum);
    cpl_free(telluric_spectrum_conv);
    cpl_free(telluric_spectrum_interp);
        
    return (cpl_error_get_code());
}



/*----------------------------------------------------------------------------*/
/**
 @brief     Save telluric keywords
 @param
 @param
 @param
 @return   CPL_ERROR_NONE iff OK
 */
/*----------------------------------------------------------------------------*/


cpl_error_code espdr_fill_header(cpl_propertylist **tell_keywords,
                                 espdr_inst_config *inst_config,
                                 int mol_nb,
                                 char **molecule_name,
                                 double *PWV_molecule,
                                 double *PWV_molecule_err,
                                 double *pressure_atm_molecule,
                                 double *pressure_atm_molecule_err,
                                 double *Temp_offseted,
                                 int *iterations_nb,
                                 double *chi2_final) {
    
    cpl_error_code my_error = CPL_ERROR_NONE;
    char *new_keyword = (char *)cpl_malloc(32 * sizeof(char));
    int QC_mol[mol_nb], global_qc = 1;
    
    for (int i = 0; i < mol_nb; i++) {
        sprintf(new_keyword, "%s QC TELL %s IWV", inst_config->prefix, molecule_name[i]);
        my_error = espdr_keyword_add_double(new_keyword, PWV_molecule[i]*10.0, "[mm]", tell_keywords);
        sprintf(new_keyword, "%s QC TELL %s IWV ERR", inst_config->prefix, molecule_name[i]);
        my_error = espdr_keyword_add_double(new_keyword, PWV_molecule_err[i]*10.0, "[mm]", tell_keywords);
        sprintf(new_keyword, "%s QC TELL %s PRESSURE", inst_config->prefix, molecule_name[i]);
        my_error = espdr_keyword_add_double(new_keyword, pressure_atm_molecule[i]*1013.2501, "[hPa]", tell_keywords);
        sprintf(new_keyword, "%s QC TELL %s PRESSURE ERR", inst_config->prefix, molecule_name[i]);
        my_error = espdr_keyword_add_double(new_keyword, pressure_atm_molecule_err[i]*1013.2501, "[hPa]", tell_keywords);
        sprintf(new_keyword, "%s QC TELL %s TEMP", inst_config->prefix, molecule_name[i]);
        my_error = espdr_keyword_add_double(new_keyword, Temp_offseted[i]-273.15, "[C]", tell_keywords);
        sprintf(new_keyword, "%s QC TELL %s ITER_NB", inst_config->prefix, molecule_name[i]);
        my_error = espdr_keyword_add_int(new_keyword, iterations_nb[i], "", tell_keywords);
        sprintf(new_keyword, "%s QC TELL %s CHI2", inst_config->prefix, molecule_name[i]);
        my_error = espdr_keyword_add_double(new_keyword, chi2_final[i], "", tell_keywords);
        if (PWV_molecule[i] <= 0.0) {
            QC_mol[i] = 0;
            global_qc = 0;
        } else {
            QC_mol[i] = 1;
        }
        // The quality control of the telluric correction is set to 0 (not valid) as soon as
        // the pressure fitted by the telluric model is significantly larger that
        // the maximum ambiant pressure seen in La Silla (maximum pressure 780, value set at 850).
        if ((pressure_atm_molecule[i] <= 0.0) || (pressure_atm_molecule[i]*1013.2501 > 850.0)) {
            QC_mol[i] = 0;
            global_qc = 0;
        }
        // It is not impossible to have errors higher than parameters values, so we don't apply this cryterium
        //if (PWV_molecule_err[i] > PWV_molecule[i]) {
        //    QC_mol[i] = 0;
        //    global_qc = 0;
        //}
        //if (pressure_atm_molecule_err[i] > pressure_atm_molecule[i]) {
        //    QC_mol[i] = 0;
        //    global_qc = 0;
        //}
        
        if (iterations_nb[i] >= FIT_ITERATIONS_MAX_NB) {
            QC_mol[i] = 0;
            global_qc = 0;
        }
        
        // When GSL obtain a NaN it put the value -9999.9, so we check againts these values
        if ((PWV_molecule[i] == GSL_NAN_NUMBER) || (PWV_molecule_err[i] == GSL_NAN_NUMBER) ||
            (pressure_atm_molecule[i] == GSL_NAN_NUMBER) || (pressure_atm_molecule_err[i] == GSL_NAN_NUMBER) ||
            (chi2_final[i] == GSL_NAN_NUMBER)) {
            QC_mol[i] = 0;
            global_qc = 0;
        }
        
        sprintf(new_keyword, "%s QC TELL %s CHECK", inst_config->prefix, molecule_name[i]);
        my_error = espdr_keyword_add_int(new_keyword, QC_mol[i], "", tell_keywords);
    }
    
    sprintf(new_keyword, "%s QC TELL CHECK", inst_config->prefix);
    my_error = espdr_keyword_add_int(new_keyword, global_qc, "", tell_keywords);
    
    cpl_free(new_keyword);
    
    return (cpl_error_get_code());
}
