/* $Id: espdr_dfs-test.c,v 1.2 2013-07-26 11:44:44 amodigli Exp $
 *
 * This file is part of the ESPDR Pipeline
 * Copyright (C) 2002,2003 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: amodigli $
 * $Date: 2013-07-26 11:44:44 $
 * $Revision: 1.2 $
 * $Name: not supported by cvs2svn $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include <stdlib.h>
#include <string.h>

#include <cpl.h>

#include <espdr_dfs.h>
#include <espdr_msg.h>

/*----------------------------------------------------------------------------*/
/**
 * @defgroup espdr_dfs_test  Unit test of espdr_dfs
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Textual representation of CPL frame group
  @param    group     to convert
  @return   textual representation
 */
/*----------------------------------------------------------------------------*/
static const char *
frame_group_to_string(cpl_frame_group group)
{
    const char * self = "???"; /* Only used if cpl_frame_group is extended */

    switch(group) {
    case CPL_FRAME_GROUP_RAW:
        self = CPL_FRAME_GROUP_RAW_ID;
        break;
    case CPL_FRAME_GROUP_NONE:
        self = "NONE";
        break;
    case CPL_FRAME_GROUP_CALIB:
        self = CPL_FRAME_GROUP_CALIB_ID;
        break;
    case CPL_FRAME_GROUP_PRODUCT:
        self = CPL_FRAME_GROUP_PRODUCT_ID;
        break;
    default:
        break;
    }

    return self;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of espdr_dfs_set_groups
 */
/*----------------------------------------------------------------------------*/
static void test_set_groups(void)
{
    const char *const fctid = "test_set_groups";
    const char *const test_subject = "espdr_dfs_set_groups";
    cpl_errorstate prestate = cpl_errorstate_get();
    espdr_msg_init();

    /* Test with invalid input */
    if (espdr_dfs_set_groups(NULL) == 0) {
        cpl_msg_error(fctid, "Function %s did not fail on NULL input",
                      test_subject);
        cpl_end();
        exit(EXIT_FAILURE);       
    }

    cpl_errorstate_set(prestate);

    /* Test with valid input */
    {
        /* Simulate data */
        const int N = 3;
        const char *const filename[] = {"raw1.fits", 
                                        "raw2.fits", 
                                        "calib.fits"};
        const char *const tag[] = {ESPDR_BIAS_RAW, 
                                   ESPDR_BIAS_RAW, 
                                   ESPDR_PRO_CATG_MBIAS};
        cpl_frame_group const expected_group[] = {CPL_FRAME_GROUP_RAW,
                                                  CPL_FRAME_GROUP_RAW,
                                                  CPL_FRAME_GROUP_CALIB};
        cpl_frameset *frames = cpl_frameset_new();
        int i;
    
        for (i = 0; i < N; i++) {
            cpl_frame *frame = cpl_frame_new();

            cpl_frame_set_filename(frame, filename[i]);
            cpl_frame_set_tag(frame, tag[i]);
            cpl_frameset_insert(frames, frame);
        }

        /* Call the function */
        if (espdr_dfs_set_groups(frames) != 0) {
            cpl_msg_error(fctid, "Function %s failed",
                          test_subject);
            cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
            cpl_frameset_delete(frames);
            cpl_end();
            exit(EXIT_FAILURE);       
        }
        
        /* Verify results */
        for (i = 0; i < N; i++) {
            cpl_frame *frame = cpl_frameset_get_position(frames, i);

            if (frame == NULL) {
                cpl_msg_error(fctid, "Missing frame number %d", i);
                cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
                cpl_frameset_delete(frames);
                cpl_end();
                exit(EXIT_FAILURE);       
            }

            if (cpl_frame_get_group(frame) != expected_group[i]) {
                cpl_msg_error(fctid, "Frame number %d has group %s, "
                              "%s expected",
                              i,
                              frame_group_to_string(cpl_frame_get_group(frame)),
                              frame_group_to_string(expected_group[i]));
                cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
                cpl_frameset_delete(frames);
                cpl_end();
                exit(EXIT_FAILURE);
            }
        }
        
        cpl_frameset_delete(frames);
    }
    cpl_test_abs(espdr_msg_get_warnings(), 0, 0.1);


    return;
}

void test_crea_ima(void) {
 cpl_image* input_image = cpl_image_new(5,5,CPL_TYPE_FLOAT);
    cpl_image_add_scalar(input_image,5000.);
    cpl_image_dump_structure(input_image,stdout);
    cpl_image_save(input_image, "uffa.fits", CPL_TYPE_FLOAT, NULL, CPL_IO_CREATE);
    return;

}
/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of espdr_dfs module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{

    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);
    //test_crea_ima();
    test_set_groups();
    return cpl_test_end(0);
}

/**@}*/
