from adari_core.data_libs.master_rawdisp import MasterRawdispReport
import os

from .espresso_utils import EspressoSetupInfo
from .espresso_utils import EspressoReportMixin


class EspressoRawdispReport(EspressoReportMixin, MasterRawdispReport):
    def __init__(self):
        super().__init__("espresso_rawdisp")
        self.extensions = []
        self.sof_tag = []
        self.tasks = {
            "BIAS": "bias",
            "DARK": "dark",
            "LED_FF": "led_ff",
            "DETMON_LAMP_OFF": "detmon",
            "DETMON_LAMP_ON": "detmon",
            "ORDERDEF_A": "orderdef",
            "ORDERDEF_B": "orderdef",
            "FLAT_A": "flat",
            "FLAT_B": "flat",
            "RAW_CONTAM_THAR": "contam_thar",
            "RAW_CONTAM_FP": "contam_fp",
            "FP_FP": "wave_fp_fp",
            "THAR_FP": "wave_thar_fp",
            "FP_THAR": "wave_fp_thar",
            "LFC_FP": "wave_lfc_fp",
            "FP_LFC": "wave_fp_lfc",
            "EFF_AB": "eff_ab",
            "FLUX": "flux",
        }
        self.task_scaling = {
            "bias": {
                "image": {"v_clip": "mad", "v_clip_kwargs": {"nmad": 4}},
                "hist": {"v_clip": "percentile", "v_clip_kwargs": {"percentile": 99.9}},
            },
            "dark": {
                "image": {"v_clip": "mad", "v_clip_kwargs": {"nmad": 4}},
                "hist": {"v_clip": "percentile", "v_clip_kwargs": {"percentile": 95}},
            },
            "led_ff": {"image": {"v_clip": "mad", "v_clip_kwargs": {"nmad": 4}}},
            "eff_ab": {"image": {"v_clip": "mad", "v_clip_kwargs": {"nmad": 4}}},
            "wave_fp_fp": {
                "image": {
                    "v_,min": 0,
                    "v_max": 3000,
                }
            },
            "wave_thar_fp": {
                "image": {
                    "v_,min": 0,
                    "v_max": 3000,
                }
            },
            "wave_fp_thar": {
                "image": {
                    "v_,min": 0,
                    "v_max": 3000,
                }
            },
            "wave_lfc_fp": {
                "image": {
                    "v_,min": 0,
                    "v_max": 3000,
                }
            },
            "wave_fp_lfc": {
                "image": {
                    "v_,min": 0,
                    "v_max": 3000,
                }
            },
        }
        self.setup_info = EspressoSetupInfo

    def parse_sof(self):
        tags = list(self.tasks.keys())
        # Since we want only the first raw file,
        # keep track of those already added
        added = {}
        file_lists = []
        for filename, catg in self.inputs:
            if catg in tags:
                if filename is not None and catg not in added:
                    file_lists.append({"filename": filename})
                    added[catg] = self.tasks[catg]
                    self.sof_tag.append(catg)
                    # Create specific scaling requirements for certain recipes
        return file_lists

    def generate_panels(self, **kwargs):
        panels = {}
        exts = ["CCD290blue", "CCD290red"]

        for ext in exts:
            new_panels = super().generate_panels(ext=ext)
            for i, (panel, panel_descr) in enumerate(new_panels.items()):
                # Alter the cut pos, or remove CutPlot(s) completely,
                # depending on task name
                try:
                    task_name = panel_descr["task_name"]
                except KeyError:
                    raise RuntimeError(
                        "A report has been created by "
                        "MasterRawdispReport that did "
                        "not come back with a task name "
                        "attached!"
                    )

                # Cuts along x (i.e. @ y = something)
                if task_name in [
                    "contam_thar",
                    "contam_fp",
                    "wave_fp_fp",
                    "wave_thar_fp",
                    "wave_fp_thar",
                    "wave_lfc_fp",
                    "wave_fp_lfc",
                ]:
                    # Remove
                    _ = panel.pop(8, 3)
                elif task_name in [
                    "led_ff",
                    "detmon",
                    "orderdef",
                    "flat",
                    "eff_ab",
                    "flux",
                ]:
                    cp = panel.retrieve(8, 3)
                    raw_plot = cp.retrieve_data("raw")
                    cp.update_cut_pos(
                        "raw", raw_plot.get_data_coord(raw_plot.data.shape[0] // 4, "y")
                    )

                # Cuts along y (i.e. @ x = something)
                if task_name in [
                    "orderdef",
                    "flat",
                    "eff_ab",
                    "flux",
                    "wave_fp_fp",
                    "wave_thar_fp",
                    "wave_fp_thar",
                    "wave_lfc_fp",
                    "wave_fp_lfc",
                ]:
                    # Remove
                    _ = panel.pop(8, 4)
                elif task_name in ["led_ff", "detmon"]:
                    cp = panel.retrieve(8, 4)
                    raw_plot = cp.retrieve_data("raw")
                    cp.update_cut_pos(
                        "raw",
                        raw_plot.get_data_coord(
                            raw_plot.data.shape[1] // 2 - raw_plot.data.shape[1] // 16,
                            "x",
                        ),
                    )

                # Make the description instrument-specfic
                panel_descr["report_description"] = (
                    f"ESPRESSO rawdisp panel - "
                    f"{panel_descr['task_name']}, "
                    f"{panel_descr['tag']}, "
                    f"{os.path.basename(panel_descr['filename'])}, "
                    f"{panel_descr['ext']}"
                )
            panels = {**panels, **new_panels}

        return panels


rep = EspressoRawdispReport()
