/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_blk.h"
#include "qmost_detect_and_trace_fib.h"
#include "qmost_fibtab.h"
#include "qmost_dfs.h"
#include "qmost_polynm.h"
#include "qmost_sort.h"
#include "qmost_stats.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_detect_and_trace_fib  qmost_detect_and_trace_fib
 * 
 * Detect and trace fibres.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_detect_and_trace_fib.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Calculation window used for FWHM determination. */

#define FWHMWIN 13

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/* Structure used to report features identified by findpeaks */

typedef struct {
    int iobj;
    float xpos;
    float tmax;
    float contrast;
} resstruct;

/* Structure used to store parameters of a detected object */

typedef struct {
    int objnum;
    int ihalf;
    int npos;
    double *xpos;
    double *ypos;
    float *fwhm;
    float *peak;
    float *contrast;
    int lastindex;
    int nord;
    cpl_polynomial *coefs;
    unsigned char skipme;
    double rms;
    double yref;
    double minx;
    double maxx;
    double miny;
    double maxy;
} objstruct;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code getmissing (
    cpl_table *in_fibinfo,
    int *nbad,
    int **bad);

static void findpeaks(
    int nx,
    float *indata,
    float *smoothed,
    float thr, 
    int width,
    int xbin,
    int *npeaks,
    resstruct **results);

static void sortbyx(
    objstruct **objects,
    int nobj,
    int **iorder);

static objstruct *objopen(
    int nblocks);

static void objclose(
    objstruct *o);

static void searchmatches (
    objstruct **objects,
    int nobj,
    resstruct *results,
    int npeaks,
    float ypos,
    float maxyoff,
    int **matches);

static void searchref (
    objstruct **objects,
    int nobj,
    int *iorder,
    qmost_traceinfo *trreflist,
    int ntrref,
    double y,
    double errlim,
    double searchgrid,
    int **matches,
    int **missing,
    int *nmissing);

static void badinterp(
    float *indata,
    unsigned char *inbpmdata,
    int nx);

static void findstfn(
    qmost_traceinfo *tr,
    objstruct *obj,
    cpl_binary *in_sm_buf,
    int nx,
    int ny,
    int ybin);

static void qmost_get_fwhm_spa (
    qmost_traceinfo *tr,
    float *img,
    cpl_binary *bpm,
    int nx,
    int ny,
    int xbin,
    int ybin);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Detect fibre images and trace them in the spectral
 *          direction.
 *
 * The positions of fibres on a frame are determined and the trace of
 * spatial positions versus spectral position is modelled with
 * polynomial fits of a given order.
 *
 * Block averaging in the spectral direction is used to improve SNR
 * and reduce sensitivity to outliers, starting at the given position
 * along the spectral axis.  The tracing proceeds first upwards from
 * this starting position, and then once the end of the region
 * specified by the slit mask or the end of the detector is reached,
 * the second part of the trace is computed by returning to the
 * starting position and tracing downwards.  The resulting set of 
 * trace samples are used to fit a polynomial to the trace and also
 * recorded into the FITS binary table output out_trace.  A binary
 * mask is also produced flagging pixels considered to be part of one
 * of the fibre profiles, to be used in scattered light correction.
 *
 * @param   in_image       (Given)    Input fibre flat image to
 *                                    process.  The data type must be
 *                                    CPL_TYPE_FLOAT.
 * @param   in_hdr         (Given)    The FITS header of the input
 *                                    with binning information written
 *                                    by qmost_ccdproc.
 * @param   in_fibinfo     (Given)    The corresponding fibinfo
 *                                    table, used to insert dummy
 *                                    entries to pad the trace table
 *                                    for bad fibres in conjunction
 *                                    with the replmissing flag.  If
 *                                    NULL, all fibres are assumed
 *                                    good and the trace table simply
 *                                    reflects what was detected in
 *                                    the image.
 * @param   in_sm          (Given)    Input slit mask.  Pixels set to
 *                                    false will be included in the
 *                                    trace, pixels set to true will
 *                                    be excluded from the trace (and
 *                                    consequently don't appear in the
 *                                    extracted spectrum, PSF,
 *                                    etc. when the trace is later
 *                                    used).  If NULL, all pixels are
 *                                    included.
 * @param   ref_trace_tbl  (Given)    Reference master trace table to
 *                                    use to obtain approximate
 *                                    expected positions for the
 *                                    fibres to identify any missing
 *                                    fibres.  Can be given as NULL,
 *                                    but this disables identification
 *                                    of missing fibres (in such cases
 *                                    it is imperative that the FIB_ST
 *                                    flags in the FIBINFO table are
 *                                    correct).
 * @param   ref_trace_hdr  (Given)    The corresponding FITS header
 *                                    for the reference trace table.
 * @param   iblock         (Given)    Window size for block averaging
 *                                    in the spectral direction prior
 *                                    to detection.
 * @param   detthr         (Given)    Threshold for profile detection
 *                                    in units of background sigma.
 * @param   nord           (Given)    The degree of the polynomial to
 *                                    be fit to the trace.
 * @param   width          (Given)    The nominal width of each
 *                                    profile.  Used to trap edge
 *                                    effects in detection and also
 *                                    to define the size of the region
 *                                    flagged in the output fibre
 *                                    mask.
 * @param   out_trace      (Returned) Output trace table.  This table
 *                                    (described in the Data
 *                                    Reduction Pipeline Description
 *                                    document, Section 6.5) contains
 *                                    a row per fibre with the trace
 *                                    polynomial coefficients, the
 *                                    trace samples, and diagnostic
 *                                    information such as measured
 *                                    spatial FWHM.
 * @param   out_trace_hdr  (Modified) Output trace table header to
 *                                    receive QC and internal
 *                                    keywords.
 * @param   out_mask       (Returned) Output fibre mask image flagging
 *                                    pixels that are part of a fibre
 *                                    image as true, and pixels that
 *                                    are not part of a fibre image as
 *                                    false.
 * @param   startline      (Given)    Start at this line (numbering
 *                                    from 1), or -1 to supply a
 *                                    suitable default. This should be
 *                                    somewhere near the middle of the
 *                                    detector axis with high SNR and
 *                                    well separated fibre traces. The
 *                                    default is to use the middle of
 *                                    the detector.
 * @param   replmissing    (Given)    If set, then any broken/unused
 *                                    fibres flagged by the FIBINFO
 *                                    table will be given dummy
 *                                    entries.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If one of the required input
 *                                    FITS header keywords or FIBINFO
 *                                    table columns was not found.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If sizes of input images and
 *                                    masks don't match.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required inputs or
 *                                    outputs was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input image data type was
 *                                    not float, or if an input FITS
 *                                    header keyword value or FIBINFO
 *                                    table column had an incorrect
 *                                    data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS SPATBIN</b>
 *   - <b>ESO DRS SPECBIN</b>
 *
 * @par Input FIBINFO Table Columns:
 *   - <b>FIB_ST</b>
 *
 * @par Output DRS Headers:
 *   - <b>MAXYFN</b>: The maximum end y-position out of all the traces
 *     reported in the trace table, numbering from 1.
 *   - <b>MINYST</b>: The minimum start y-position out of all the
 *     traces reported in the trace table, numbering from 1.
 *   - <b>NMISSING</b>: The number of fibres that were reported as bad
 *     in the FIBINFO table, or found to be bad during trace
 *     processing by comparison to the reference trace table, and were
 *     given dummy entries in the output trace table.
 *   - <b>NSPECDET</b>: The total number of fibres with trace
 *     information reported in the trace table.  This count includes
 *     bad fibres.
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_detect_and_trace_fib_lim (
    cpl_image *in_image,
    cpl_propertylist *in_hdr,
    cpl_table *in_fibinfo,
    cpl_mask *in_sm,
    cpl_table *ref_trace_tbl,
    cpl_propertylist *ref_trace_hdr,
    int iblock,
    float detthr,
    int nord,
    int width,
    cpl_table **out_trace,
    cpl_propertylist *out_trace_hdr,
    cpl_mask **out_mask,
    int startline,
    int replmissing)
{
    long naxis[2],naxis_bpm[2],naxis_sm[2],naxis_mask[2],npts;
    int specbin,spatbin,isbinned;
    int blkfac[2];
    int i,j,ib,ib1,ib2,ntab,yst1,yfn1;
    int nblocks,lookatzeroth,nobj_alloc,omatch,nx2,k;
    int loweredge,nblocks1,nblocks2,blockoffset,startblock,isign;
    int ix1,ix2;
    int nb,nmed,ngood,nmissing,minmin,maxmax,npeaks,nobj = 0;
    float val,wov2;
    double ypos,dsum,dsumsq,fitval,ydiff;
    objstruct *obj;
    qmost_traceinfo tr;

    cpl_mask *tmp_sm = NULL;
    cpl_mask *in_bpm;
    float *in_image_buf;
    cpl_binary *in_bpm_buf;
    cpl_binary *in_sm_buf;
    cpl_binary *tmp_sm_buf;

    float *pixtot;
    cpl_binary *smtot, *bpmtot, *mdata;

    cpl_errorstate prestate;

    /* These need to be garbage collected */
    float *indata = NULL;
    unsigned char *inbpmdata = NULL;
    float *smoothed = NULL;

    objstruct **objects = NULL;
    resstruct *results = NULL;
    int iobj;

    int *omatches = NULL;

    int *iord = NULL;
    int *missing = NULL;

    float *tmpdata = NULL;
    double *ytemp = NULL;

    qmost_traceinfo *trreflist = NULL;
    int ntrref = 0;
    int *ref_index = NULL;

    /* Check mandatory inputs and outputs */
    cpl_ensure_code(in_image, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(in_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_trace, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_trace_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(out_mask, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(tmp_sm != NULL) {                        \
        cpl_mask_delete(tmp_sm);                \
        tmp_sm = NULL;                          \
    }                                           \
    if(indata != NULL) {                        \
        cpl_free(indata);                       \
        indata = NULL;                          \
    }                                           \
    if(inbpmdata != NULL) {                     \
        cpl_free(inbpmdata);                    \
        inbpmdata = NULL;                       \
    }                                           \
    if(smoothed != NULL) {                      \
        cpl_free(smoothed);                     \
        smoothed = NULL;                        \
    }                                           \
    if(objects != NULL) {                       \
        for(iobj = 0; iobj < nobj; iobj++)      \
            objclose(objects[iobj]);            \
        cpl_free(objects);                      \
        objects = NULL;                         \
    }                                           \
    if(results != NULL) {                       \
        cpl_free(results);                      \
        results = NULL;                         \
    }                                           \
    if(omatches != NULL) {                      \
        cpl_free(omatches);                     \
        omatches = NULL;                        \
    }                                           \
    if(iord != NULL) {                          \
        cpl_free(iord);                         \
        iord = NULL;                            \
    }                                           \
    if(missing != NULL) {                       \
        cpl_free(missing);                      \
        missing = NULL;                         \
    }                                           \
    if(tmpdata != NULL) {                       \
        cpl_free(tmpdata);                      \
        tmpdata = NULL;                         \
    }                                           \
    if(ytemp != NULL) {                         \
        cpl_free(ytemp);                        \
        ytemp = NULL;                           \
    }                                           \
    if(trreflist != NULL) {                     \
        qmost_trclose(ntrref, &trreflist);      \
        trreflist = NULL;                       \
    }                                           \
    if(ref_index != NULL) {                     \
        cpl_free(ref_index);                    \
        ref_index = NULL;                       \
    }

    /* Get size of input image and check if binned */
    naxis[0] = cpl_image_get_size_x(in_image);
    naxis[1] = cpl_image_get_size_y(in_image);

    qmost_isbinned(in_hdr, &specbin, &spatbin, &isbinned);

    /* Adjust input parameters for binning, if used */
    if(isbinned) {
        iblock /= specbin;
        width /= spatbin;

        if(startline >= 0) {
            startline = qmost_nint((startline - 0.5) / specbin + 0.5);
        }
    }

    /* Supply default start line, if requested */
    if(startline < 0) {
        startline = naxis[1] / 2 + 1;
    }

    if(in_sm == NULL) {
        /* Create dummy slit mask if there isn't one */
        tmp_sm = cpl_mask_new(naxis[0], naxis[1]);
        if(tmp_sm == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't create dummy slitmask "
                                         "size (%ld, %ld)",
                                         naxis[0], naxis[1]);
        }
    }
    else {
        /* Check size of slit mask */
        naxis_sm[0] = cpl_mask_get_size_x(in_sm);
        naxis_sm[1] = cpl_mask_get_size_y(in_sm);
        
        if(naxis_sm[0] != naxis[0]*spatbin ||
           naxis_sm[1] != naxis[1]*specbin) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "image (%ld,%ld), "
                                         "slitmask (%ld, %ld) "
                                         "sizes don't match",
                                         naxis[0]*spatbin,naxis[1]*specbin,
                                         naxis_sm[0],naxis_sm[1]);
        }

        /* Bin down to match data */
        blkfac[0] = spatbin;
        blkfac[1] = specbin;
        
        tmp_sm = qmost_maskblk(in_sm, blkfac);
        if(tmp_sm == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "binning (%d,%d) of mask "
                                         "failed",
                                         blkfac[0], blkfac[1]);
        }
    }

    /* Get pointers to image, bpm and slitmask */
    in_image_buf = cpl_image_get_data_float(in_image);
    if(in_image_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input image");
    }

    in_bpm = cpl_image_get_bpm(in_image);  /* creates blank if none */
    if(in_bpm == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get input BPM");
    }

    /* Check size matches */
    naxis_bpm[0] = cpl_mask_get_size_x(in_bpm);
    naxis_bpm[1] = cpl_mask_get_size_y(in_bpm);

    if (naxis[0] != naxis_bpm[0] || naxis[1] != naxis_bpm[1]) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "image size (%ld,%ld) doesn't match "
                                     "BPM (%ld,%ld)",
                                     naxis[0],naxis[1],
                                     naxis_bpm[0],naxis_bpm[1]);
    }

    in_bpm_buf = cpl_mask_get_data(in_bpm);
    if(in_bpm_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get pointer to "
                                     "input BPM");
    }

    tmp_sm_buf = cpl_mask_get_data(tmp_sm);
    if(tmp_sm_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get pointer to "
                                     "input slitmask");
    }

    /* If you want to replace the missing fibres, then work out which ones
       are broken or unallocated */

    nmissing = 0;
    missing = NULL;
    if (replmissing && in_fibinfo != NULL) {
        if (getmissing(in_fibinfo,&nmissing,&missing) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get list of missing "
                                         "fibres from FIBINFO");
        }
    }
    
    /* Work out range to be considered */

    yst1 = 1;
    yfn1 = naxis[1];

    /* Lower edge of the block containing "startline".  Coerce to be
     * in range by moving it if it's within iblock/2 of one of the
     * ends of the image. */
    loweredge = startline - iblock / 2;

    if(loweredge < yst1) {
        loweredge = yst1;
    }
    
    if(loweredge + iblock - 1 > yfn1) {
        loweredge = yfn1 + 1 - iblock;
    }

    /* Number of blocks */
    nblocks1 = (yfn1 + 1 - loweredge) / iblock;
    nblocks2 = (loweredge - yst1) / iblock;

    nblocks = nblocks1 + nblocks2;

    if(nblocks < nord+1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                     "blocks of size %d are too large "
                                     "for image of size %ld and "
                                     "polynomial degree %d",
                                     iblock, naxis[1], nord);
    }

    blockoffset = loweredge - nblocks2 * iblock;
    startblock = nblocks2;

    /* Grab some memory for the block median profile and background
       estimation */
    nx2 = naxis[0];

    indata = cpl_malloc(nx2*sizeof(float));
    inbpmdata = cpl_malloc(nx2*sizeof(unsigned char));
    smoothed = cpl_malloc(nx2*sizeof(float));

    /* Set up object structures */

    nobj = 0;
    nobj_alloc = 64;
    objects = cpl_malloc(nobj_alloc*sizeof(objstruct *));

    /* Loop for each block and define the pixel limits. Work it so
       that you start at the designated startline of the spectrum and 
       work outwards */

    lookatzeroth = 0;
    isign = 1;
    nb = startblock;
    for (ib = 0; ib < nblocks; ib++) {
        npeaks = 0;

        if(nb == nblocks) {
            nb = startblock - 1;
            isign = -1;
            lookatzeroth = 1;
        }

        ib1 = blockoffset + nb*iblock;
        ib2 = ib1 + iblock - 1;
        nb += isign;

        /* Position in binned image */
        ypos = 0.5 * (ib1 + ib2);

        /* Position in unbinned image */
        ypos = (ypos - 0.5) * specbin + 0.5;

        /* Set pointers to the subset block */

        pixtot = in_image_buf + (ib1-1) * naxis[0];
        smtot = tmp_sm_buf + (ib1-1) * naxis[0];
        bpmtot = in_bpm_buf + (ib1-1) * naxis[0];

        /* Block median along the spectral direction */

        nmed = (ib2 - ib1 + 1);
        tmpdata = cpl_malloc(nmed*sizeof(float));
        for (i = 0; i < nx2; i++) {
            ngood = 0;
            for (j = 0; j < nmed; j++) {
                if(!(bpmtot[j*nx2+i] || smtot[j*nx2+i])) {
                    tmpdata[ngood] = pixtot[j*nx2+i];
                    ngood++;
                }
            }
            if (ngood >= nmed/2) {
                if(qmost_med(tmpdata, NULL, ngood, &val) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "couldn't compute block "
                                                 "median for y = (%d:%d)",
                                                 ib1, ib2);
                }
                inbpmdata[i] = 0;
                indata[i] = val;
            } else {
                indata[i] = 0.0;
                inbpmdata[i] = 1;
            }
        }
        cpl_free(tmpdata);
        tmpdata = NULL;

        /* Interpolate out bad pixels */

        badinterp(indata,inbpmdata,nx2);

        /* Hanning smooth the profile gently */

        smoothed[0] = indata[0];
        smoothed[nx2-1] = indata[nx2-1];
        for (i = 1; i < nx2-1; i++)
            smoothed[i] = 0.25*(indata[i-1] + 2.0*indata[i] + indata[i+1]);

        /* Detect the peaks */

        findpeaks(nx2,indata,smoothed,detthr,width,spatbin,&npeaks,&results);

        /* If this is the first time you're doing the lower half, then
           set the lastindex in each object structure to point to the
           first point in the array as this is where this next peak
           will connect */

        if (lookatzeroth) {
            lookatzeroth = 0;
            for (i = 0; i < nobj; i++) 
                objects[i]->lastindex = 0;
        }

        /* Search the object list and see if you can find a match */
        searchmatches(objects,nobj,results,npeaks,
                      ypos,
                      2.0*(float)iblock*specbin,
                      &omatches);

        /* Loop for each of the peaks you've found */

        for (i = 0; i < npeaks; i++) {
            omatch = omatches[i];

            /* If it didn't match any of them, then create a new
               object. If it did match then reference the correct one */

            if (omatch == -1 && ib == 0) {
                obj = objopen(nblocks);
                nobj++;
                if (nobj == nobj_alloc) {
                    nobj_alloc += 16;
                    objects = cpl_realloc(objects,
                                          nobj_alloc*sizeof(objstruct *));
                }
                objects[nobj-1] = obj;
                obj->objnum = nobj;
                obj->ihalf = 1;
            } else if (omatch == -1 && ib != 0) {
                continue;
            } else {
                obj = objects[omatch];
            }

            /* Now add a data point for this object */

            obj->xpos[obj->npos] = (double)(results[i].xpos);
            obj->ypos[obj->npos] = ypos;
            obj->peak[obj->npos] = results[i].tmax;
            obj->contrast[obj->npos] = results[i].contrast;
            obj->lastindex = obj->npos;
            obj->npos += 1;
            if (ypos < obj->miny) {
                obj->miny = ypos;
                obj->minx = results[i].xpos;
            }
            if (ypos > obj->maxy) {
                obj->maxy = ypos;
                obj->maxx = results[i].xpos;
            }
        }

        cpl_free(omatches);
        omatches = NULL;
    }

    /* Flag any that don't have many points */

    for (i = 0; i < nobj; i++) {
	if (objects[i]->npos <= nblocks/6) {
	    objects[i]->skipme = 1;
        }
    }

    /* Fit a trace to each object. Start by getting some space for 
       the coefficients */

    for (i = 0; i < nobj; i++) {
	obj = objects[i];
	if (obj->skipme)
	    continue;
	obj->nord = nord;

	ytemp = cpl_malloc(obj->npos*sizeof(double));

	/* Scale the spectral axis to between -1 and 1 */

	for (j = 0; j < obj->npos; j++)
	    ytemp[j] = (obj->ypos[j] - obj->ypos[0])/obj->ypos[0];
	obj->yref = obj->ypos[0];

	/* Do the fit.  Not enough points or singular matrix
         * are converted into non-fatal error messages where all
         * coefficients are zero. */
        prestate = cpl_errorstate_get();

        obj->coefs = qmost_polynm(ytemp, obj->xpos, obj->npos, obj->nord+1, 0);
        if(obj->coefs == NULL) {
            switch(cpl_error_get_code()) {
            case CPL_ERROR_DATA_NOT_FOUND:
                cpl_errorstate_set(prestate);
                
                cpl_msg_warning(cpl_func,
                                "not enough valid points for degree %d "
                                "polynomial fit to fibre %d: %d",
                                obj->nord, i+1, obj->npos);
                
                obj->coefs = cpl_polynomial_new(1);
                
                break;
            case CPL_ERROR_SINGULAR_MATRIX:
                cpl_errorstate_set(prestate);
                
                cpl_msg_warning(cpl_func,
                                "singular matrix in degree %d polynomial "
                                "fit to fibre %d with %d data points",
                                obj->nord, i+1, obj->npos);
                
                obj->coefs = cpl_polynomial_new(1);
                
                break;
            default:
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to fit polynomial to "
                                             "trace for fibre %d", i+1);
            }
        }

        /* Work out RMS of fit */
        
        dsum = 0.0;
        dsumsq = 0.0;
        for (j = 0; j < obj->npos; j++) {
            fitval = cpl_polynomial_eval_1d(obj->coefs, ytemp[j], NULL);
            ydiff = obj->xpos[j] - fitval;
            dsum += ydiff;
            dsumsq += ydiff*ydiff;
        }
        dsum /= (double)obj->npos;
        obj->rms = sqrt(dsumsq/(double)obj->npos - dsum*dsum);

        cpl_free(ytemp);
        ytemp = NULL;
    }

    /* Get an index to sort the objects by their middle X position */

    sortbyx(objects,nobj,&iord);

    /* If reference was given, open that and use to assign indices. */

    if(ref_trace_tbl != NULL) {
        /* Read reference table */
        if(qmost_tropen(ref_trace_tbl, ref_trace_hdr,
                        &ntrref, &trreflist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read reference "
                                         "trace table");
        }

        /* Centre of start (reference) block, calculated as above */
        ib1 = blockoffset + startblock*iblock;
        ib2 = ib1 + iblock - 1;
        ypos = 0.5 * (ib1 + ib2);
        ypos = (ypos - 0.5) * specbin + 0.5;

        /* Match detected objects to reference to assign fibre numbers */
        searchref(objects, nobj, iord,
                  trreflist, ntrref,
                  ypos, 0.5*width*spatbin, 10.0*width*spatbin,
                  &ref_index,
                  &missing,
                  &nmissing);

        /* Clean up */
        qmost_trclose(ntrref, &trreflist);
        trreflist = NULL;
    }

    /* Create trace table */

    if(qmost_trcreate(out_trace,nord,nblocks) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't create output trace table");
    }

    /* Now create a row in the table for each object that we've found.
       Don't put in the identifier column yet as we have to substitute
       the missing ones first.  The output needs to be in unbinned
       pixels, so we correct for binning if there was any. */

    ntab = 0;
    minmin = 1;
    maxmax = naxis[1];
    for (i = 0; i < nobj; i++) {
	obj = objects[iord[i]];
	if (obj->skipme) {
	    continue;
        }

        if(ref_index != NULL) {
            /* Did we match to reference? */
            if(ref_index[iord[i]] < 0) {
                /* If not, assume false */
                continue;
            }

            /* Assign fibre number based on reference */
            tr.specnum = ref_index[iord[i]] + 1;
        }
        else {
            tr.specnum = i + 1;
        }

        tr.live = 1;
        findstfn(&tr,obj,tmp_sm_buf,naxis[0],naxis[1],specbin);

        if(ntab == 0 || tr.yst < minmin)
            minmin = tr.yst;
        if(ntab == 0 || tr.yfn > maxmax)
            maxmax = tr.yfn;

        tr.yref = obj->yref;
        tr.trrms = obj->rms;
        tr.nord = nord;
        tr.coefs = obj->coefs;
        tr.npos = obj->npos;
        tr.xpos = obj->xpos;
        tr.ypos = obj->ypos;
        tr.fwhm = obj->fwhm;
        tr.peak = obj->peak;
        tr.contrast = obj->contrast;

        qmost_get_fwhm_spa(&tr,in_image_buf,in_bpm_buf,
                           naxis[0],naxis[1],spatbin,specbin);

        ntab++;
        if(qmost_trwrite1(*out_trace,tr,ntab) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write trace table "
                                         "entry %d", ntab);
        }
    }
    cpl_propertylist_update_int(out_trace_hdr,
                                "ESO DRS MINYST",
                                minmin);
    cpl_propertylist_set_comment(out_trace_hdr,
                                 "ESO DRS MINYST",
                                 "[pix] Minimum yst");
    cpl_propertylist_update_int(out_trace_hdr,
                                "ESO DRS MAXYFN", maxmax);
    cpl_propertylist_set_comment(out_trace_hdr,
                                 "ESO DRS MAXYFN",
                                 "[pix] Maximum yfn");

    /* Now fill in the information for the missing fibres */

    if (replmissing) {
        for (i = 0; i < nmissing; i++) {
            if(qmost_trinsert_dead(*out_trace,missing[i]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to insert trace table "
                                             "entry for missing "
                                             "fibre %d", missing[i]);
            }
        }
    }

    /* Renumber the trace, just in case, add a few header items and 
       close it up */

    if(qmost_trrenumber(*out_trace) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't renumber trace table");
    }

    cpl_propertylist_update_int(out_trace_hdr,
                                "ESO DRS NSPECDET",
                                ntab);
    cpl_propertylist_set_comment(out_trace_hdr,
                                 "ESO DRS NSPECDET",
                                 "Number of detected spectra");
    cpl_propertylist_update_int(out_trace_hdr,
                                "ESO DRS NMISSING",
                                nmissing);
    cpl_propertylist_set_comment(out_trace_hdr,
                                 "ESO DRS NMISSING",
                                 "Number of spectra missing");

    /* Now create the mask file */
    naxis_mask[0] = naxis[0] * spatbin;
    naxis_mask[1] = naxis[1] * specbin;

    *out_mask = cpl_mask_new(naxis_mask[0], naxis_mask[1]);

    mdata = cpl_mask_get_data(*out_mask);

    /* Loop for each spectrum and blank out the bits around the centres */

    wov2 = (float)width*spatbin/2.0;
    for (i = 0; i < nobj; i++) {
	obj = objects[iord[i]];
	if (obj->skipme)
	    continue;

        for (j = 1; j <= naxis_mask[1]; j++) {
	    ypos = ((double)j - obj->yref)/obj->yref;
	    fitval = cpl_polynomial_eval_1d(obj->coefs, ypos, NULL);
	    ix1 = qmost_max(1,(int)(fitval - wov2 + 0.5));
	    ix2 = qmost_min(naxis_mask[0],(int)(fitval + wov2 + 0.5));
	    for (k = ix1; k <= ix2; k++) 
		mdata[(j-1)*naxis_mask[0] + k - 1] = 1;
        }
    }	   

    /* Read in slit mask and remove all masked out edge regions */
    if(in_sm != NULL) {
        in_sm_buf = cpl_mask_get_data(in_sm);
        if(in_sm_buf == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get pointer to "
                                         "input slitmask");
        }
        
        npts = naxis_mask[0]*naxis_mask[1];
        for (i = 0; i < npts; i++) {
            if (in_sm_buf[i] == 1)
                mdata[i] = 1;
        }
    }

    TIDY;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Update traceinfo y range using slitmask.

 * Determine range of object's y pixels included in slit mask and
 * update traceinfo.
 *
 * @param   tr             (Modified) The traceinfo structure.
 * @param   obj            (Given)    The object's objstruct.
 * @param   sm_buf         (Given)    The slitmask.
 * @param   nx             (Given)    Slitmask x dimension.
 * @param   ny             (Given)    Slitmask y dimension.
 * @param   ybin           (Given)    The spectral binning of the
 *                                    fibre flat being analysed.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void findstfn(qmost_traceinfo *tr, objstruct *obj,
                     cpl_binary *sm_buf,
                     int nx, int ny, int ybin) {

    int x, y;

    /* First find yst. Read the column from the slitmask corresponding to
       the x location of the lowest point in the trace for the object */

    x = (int)(obj->minx + 0.5) - 1;

    tr->yst = 0;
    for (y = 0; y < ny; y++) {
        if (sm_buf[y*nx+x] == 0) {
            tr->yst = y * ybin + 1;
            break;
        }
    }

    /* Now do the same for yfn */

    x = (int)(obj->minx + 0.5) - 1;

    tr->yfn = ny * ybin + 1;
    for (y = ny - 1; y > 0; y--) {
        if (sm_buf[y*nx+x] == 0) {
            tr->yfn = (y + 1) * ybin;
            break;
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Create an array of entries for missing/unused fibres.
 *
 * @param   in_fibinfo     (Given)    The FIBINFO table.
 * @param   nbad           (Returned) The number of missing fibres.
 * @param   bad            (Returned) The array of missing fibres.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the FIB_ST column is missing
 *                                    in the FIBINFO table.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code getmissing (
    cpl_table *in_fibinfo,
    int *nbad,
    int **bad)
{
    cpl_size row, nrows;
    int *fib_st;

    nrows = cpl_table_get_nrow(in_fibinfo);
    if(nrows < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine length of FIBINFO");
    }

    fib_st = cpl_table_get_data_int(in_fibinfo, "FIB_ST");
    if(!fib_st) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get FIB_ST column data "
                                     "pointer as int");
    }

    *bad = cpl_malloc(nrows*sizeof(int));

    *nbad = 0;
    for(row = 0; row < nrows; row++) {
        if(fib_st[row] == 0) {
            (*bad)[(*nbad)] = row+1;
            (*nbad)++;
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Identify peaks in a fibre flat.
 *
 * Searches the input fibre flat sections for peaks, defining a peak
 * as a local maximum in the smoothed data where the signal:to:noise
 * in the integrated peak flux exceeds the threshold value.
 *
 * @param   nx             (Given)    The number of pixels in the
 *                                    input spectrum.
 * @param   indata         (Given)    The data array with the input
 *                                    spectrum.
 * @param   smoothed       (Given)    The smoothed input spectrum.
 * @param   thr            (Given)    The detection threshold for the
 *                                    emission peaks in units of
 *                                    signal:to:noise.
 * @param   width          (Given)    The nominal width of each
 *                                    profile used to trap edge
 *                                    effects.
 * @param   xbin           (Given)    The spatial binning.
 * @param   npeaks         (Returned) The number of features
 *                                    detected.
 * @param   results        (Returned) The data structure for all the
 *                                    features found.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void findpeaks(int nx, float *indata, float *smoothed, float thr, 
                      int width, int xbin, int *npeaks, resstruct **results) {
    float sum,xbar,pkflux,sumw,sumt,cont,s2n,wt,sumf;
    int i,j,jl,jh,win2;
    float trough;
    int ntrough;

    /* Now search for the peaks */

    win2 = qmost_max(4,width/2);
    for (i = win2; i < nx-win2; i++) {
        /* Define a peak as a local maximum in the smoothed data */
	if (smoothed[i] >= smoothed[i-1] && smoothed[i] >= smoothed[i+1] &&
          smoothed[i-1] > smoothed[i-2] && smoothed[i+1] > smoothed[i+2]) {
	  jl = i-2;
	  while (smoothed[jl] < smoothed[jl+1] && jl > 0) {
	    jl = jl-1;
	  }	    
	  jl = jl+1;                                    /* back off */
          jh = i+2;
	  while (smoothed[jh] < smoothed[jh-1] && jh < nx-1) {
	    jh = jh+1;
	  }
          jh = jh-1;
          jl = qmost_max(jl,i-win2);                    /* trap edge effects */
          jh = qmost_min(jh,i+win2);
          if (jl > 0 && jh < nx-1) {         
            sum = 0.0;
            sumw = 0.0;
	    sumt = 0.0;
	    sumf = 0.0;
	    pkflux = 0.0;
            xbar = (float)(i+1);
	    cont = 0.5*(smoothed[jl]+smoothed[jh]);
            for (j = jl; j <= jh; j++) {
	      wt = qmost_max(0.0,smoothed[j]-cont);
	      sum += qmost_max(0.0,indata[j]-cont)*((float)(j+1)-xbar)*wt;
              sumw += qmost_max(0.0,indata[j]-cont)*wt;
	      sumt += qmost_max(0.0,indata[j]);
	      sumf += qmost_max(0.0,indata[j]-cont);
	      pkflux = qmost_max(pkflux,indata[j]-cont);
            }
            if (sumw <= 0.0 || sumt <= 0.0)
	        continue;
	    s2n = sumf/sqrt(sumt);                          /* Poisson-like */
            if (s2n < thr)                
	        continue;
            xbar += sum/sumw;
            if (xbar > (float)nx || xbar < 1.0)
                continue;
            *results = cpl_realloc(*results,
                                   ((*npeaks)+1) * sizeof(resstruct));
	    (*results)[*npeaks].xpos = (xbar - 0.5) * xbin + 0.5;
	    (*results)[*npeaks].tmax = pkflux + cont;

            /* Decide trough level for contrast calculation by looking
             * for a local minimum.  Requires that the flux goes up
             * again after the minimum (we're already using for
             * continuum), such that it's a trough between fibres
             * rather than the end of a slitlet.  This uses two pixels
             * as for the peak detection to reduce susceptibility to
             * noise.  If neither side is a local minimum, it's an
             * isolated fibre and we use the continuum value. */
            trough = 0;
            ntrough = 0;

            if(jl-2 >= 0 &&
               smoothed[jl-1] > smoothed[jl] &&
               smoothed[jl-2] > smoothed[jl-1]) {
                trough += smoothed[jl];
                ntrough++;
            }
            if(jh+2 < nx &&
               smoothed[jh+1] > smoothed[jh] &&
               smoothed[jh+2] > smoothed[jh+1]) {
                trough += smoothed[jh];
                ntrough++;
            }
            if(ntrough > 0) {
                trough /= ntrough;
            }
            else {
                trough = cont;
            }

	    (*results)[*npeaks].contrast = ((pkflux + cont - trough) /
                                            (pkflux + cont));

	    (*results)[*npeaks].iobj = *npeaks + 1;
	    (*npeaks)++;

            /* If we got here due to the condition
             * smoothed[i-1] == smoothed[i] == smoothed[i+1]
             * this ensures we advance past it so we can't trigger
             * again on the same peak. */
            i++;
	  }
	}
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Sort the detected objects by X position.
 *
 * The detected peaks in the object structure are sorted in ascending
 * X (spatial) position. This is done without modifying the structure,
 * as an integer array is written out with the order of each peak.
 *
 * @param   objects        (Given)    The list of detected objects.
 * @param   nobj           (Given)    The number of objects in the
 *                                    structure.
 * @param   iorder         (Returned) The array of order positions for
 *                                    each object. This array will
 *                                    need to be deallocated when no
 *                                    longer required.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void sortbyx(objstruct **objects, int nobj, int **iorder) {
    int ir,m,indxt,i,j;
    double objx;

    /* Initialise the index array */

    *iorder = cpl_malloc(nobj*sizeof(int));
    for (i = 0; i < nobj; i++) 
	(*iorder)[i] = i;

    /* Return now if there's nothing to do */
    
    if(nobj < 2) {
        return;
    }
    
    /* Heap sort, but without rearranging the input array */

    m = nobj/2 + 1;
    ir = nobj;
    while (1) {
	if (m > 1) {
	    m--;
	    indxt = (*iorder)[m-1];
	    objx = objects[indxt]->xpos[0];
	} else {
	    indxt = (*iorder)[ir-1];
	    objx = objects[indxt]->xpos[0];
	    (*iorder)[ir-1] = (*iorder)[0];
	    ir--;
	    if (ir <= 1) { /* used to be ==. changed to <= to avoid compiler 
                              warning */
		(*iorder)[0] = indxt;
		return;
	    }
	}
	i = m;
	j = 2*m;
	while (j <= ir) {
	    if (j < ir) {
		if (objects[(*iorder)[j-1]]->xpos[0] < objects[(*iorder)[j]]->xpos[0])
		    j++;
	    }
	    if (objx < objects[(*iorder)[j-1]]->xpos[0]) {
		(*iorder)[i-1] = (*iorder)[j-1];
		i = j;
		j = 2*j;
	    } else {
		j = ir + 1;
	    }
	}
	(*iorder)[i-1] = indxt;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Open an object structure.
 *
 * Open a single structure to hold the information for a single
 * detected object. No information is inserted about the object. 
 *
 * @param   nblocks        (Given)    The maximum number of detections
 *                                    allowed for the object.
 *
 * @return  objstruct                 The object structure.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static objstruct *objopen(int nblocks) {
    objstruct *o;

    o = cpl_malloc(sizeof(objstruct));

    /* We use calloc here in the CPL version because we write out the
       entire array to the file even if we didn't use the entire
       array. Zeroing it ensures any missing elements get written out
       as zero rather than writing nonsense. */
    o->xpos = cpl_calloc(nblocks,sizeof(double));
    o->ypos = cpl_calloc(nblocks,sizeof(double));
    o->fwhm = cpl_calloc(nblocks,sizeof(float));
    o->peak = cpl_calloc(nblocks,sizeof(float));
    o->contrast = cpl_calloc(nblocks,sizeof(float));
    o->objnum = -1;
    o->ihalf = 0;
    o->npos = 0;
    o->lastindex = -1;
    o->nord = -1;
    o->coefs = NULL;
    o->skipme = 0;
    o->miny = 1.0e6;
    o->maxy = -1.0e6;
    return(o);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Close an object data structure.
 *
 * Close a single object data structure by freeing up all the memory
 * associated with it.
 *
 * @param   o              (Modified) The object data structure to be
 *                                    freed.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void objclose(objstruct *o) {
    if(o != NULL) {
        if(o->xpos != NULL) {
            cpl_free(o->xpos);
            o->xpos = NULL;
        }
        if(o->ypos != NULL) {
            cpl_free(o->ypos);
            o->ypos = NULL;
        }
        if(o->fwhm != NULL) {
            cpl_free(o->fwhm);
            o->fwhm = NULL;
        }
        if(o->peak != NULL) {
            cpl_free(o->peak);
            o->peak = NULL;
        }
        if(o->contrast != NULL) {
            cpl_free(o->contrast);
            o->contrast = NULL;
        }
        if(o->coefs != NULL) {
            cpl_polynomial_delete(o->coefs);
            o->coefs = NULL;
        }
        cpl_free(o);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Search for matches in the object structure list for new
 *          positions.
 *
 * Given a list of newly detected objects, see if they are part of one
 * of the objects that we already have in the object list.
 *
 * @param   objects        (Given)    The current list of all of the
 *                                    detected objects.
 * @param   nobj           (Given)    The number of objects in the
 *                                    current list.
 * @param   results        (Given)    The list of features found in
 *                                    the current block.
 * @param   npeaks         (Given)    The number of features found in
 *                                    the current block.
 * @param   ypos           (Given)    The spectral position of the
 *                                    block.
 * @param   maxyoff        (Given)    A maximum distance along the
 *                                    spectral direction that a new
 *                                    peak is allowed to be from an
 *                                    existing object.
 * @param   matches        (Returned) For each peak, the index of the
 *                                    object in the list that matched,
 *                                    or -1 if there was no match.
 *
 * @return  void
 *
 * @note    For a match the current object can have a spatial position
 *          no more than 1 pixel away from the most recent spatial
 *          coordinate an object in the list.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void searchmatches (
    objstruct **objects,
    int nobj,
    resstruct *results,
    int npeaks,
    float ypos,
    float maxyoff,
    int **matches)
{
    double *xpred = NULL;
    int *objindex = NULL;

    double *bestforobj = NULL;
    int *claimedby = NULL;

    int irem,i,ipk,lastindex;
    double xresid,best,yresid,x1,x2,y1,y2,slope,inter;

    int isp, ifp, index, ioth;
    double errlim, xpos;

    /* Predict */
    xpred = cpl_malloc(nobj*sizeof(double));
    objindex = cpl_malloc(nobj*sizeof(int));

    for (i = 0; i < nobj; i++) {
	lastindex = objects[i]->lastindex;
        if (lastindex > 0) {
            x1 = objects[i]->xpos[lastindex-1];
            x2 = objects[i]->xpos[lastindex];
            y1 = objects[i]->ypos[lastindex-1];
            y2 = objects[i]->ypos[lastindex];
            slope = (x2 - x1)/(y2 - y1);
            inter = x2 - y2*slope;
            xpred[i] = slope*ypos + inter;
        } else {
            xpred[i] = objects[i]->xpos[lastindex];
        }

        objindex[i] = i;
    }

    /* Sort */
    qmost_sort_di(xpred, objindex, nobj);

    /* Create status arrays to detect and resolve conflicts */
    bestforobj = cpl_malloc(nobj*sizeof(double));
    claimedby = cpl_malloc(nobj*sizeof(int));

    for(i = 0; i < nobj; i++) {
        bestforobj[i] = 1000.0;
        claimedby[i] = -1;
    }

    /* Find best match for each using binary chop in x */
    *matches = cpl_malloc(npeaks*sizeof(int));

    errlim = 1.0;

    for(ipk = 0; ipk < npeaks; ipk++) {
        xpos = results[ipk].xpos;

        isp = 0;
        ifp = nobj-1;
        index = (isp + ifp)/2;

        while(ifp-isp >= 2) {
            if(xpred[index] < xpos - errlim) {
                isp = index;
                index = (index + ifp) / 2;
            }
            else if(xpred[index] > xpos - errlim) {
                ifp = index;
                index = (index + isp) / 2;
            }
            else {
                isp = index;
                break;
            }
        }

        irem = -1;
        best = 1.0e10;

        for(index = isp; index < nobj; index++) {
            if(xpred[index] > xpos + errlim)
                break;

            i = objindex[index];

            xresid = fabs(xpos - xpred[index]);

            lastindex = objects[i]->lastindex;
            yresid = fabs(ypos - objects[i]->ypos[lastindex]);

            if (xresid < best && xresid < 1.0 && yresid < maxyoff) { /***/
                irem = i;
                best = xresid;
            }
        }

        if(irem >= 0) {
            /* Check if another peak already matched to this object */
            ioth = claimedby[irem];

            if(ioth >= 0) {  /* yes */
                /* Is the new one better? */
                if(best < bestforobj[irem]) {
                    /* Yes, it is better, unflag the other one */
                    (*matches)[ioth] = -1;
                }
                else {
                    /* No, it's not better, unflag the present one */
                    irem = -1;
                }
            }
        }
        
        /* Record in result */
        (*matches)[ipk] = irem;

        /* Record information for conflict detection if we matched */
        if(irem >= 0) {
            bestforobj[irem] = best;
            claimedby[irem] = ipk;
        }
    }

    cpl_free(xpred);
    cpl_free(objindex);
    cpl_free(bestforobj);
    cpl_free(claimedby);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Match detected fibre images to reference traces.
 *
 * Given a list of detected objects, match them to a set of reference
 * traces.
 *
 * @param   objects        (Given)    The list of detected objects.
 * @param   nobj           (Given)    The number of objects in the
 *                                    structure.
 * @param   iorder         (Given)    The array of order positions for
 *                                    each object from sortbyx.
 * @param   trreflist      (Given)    The reference traces.
 * @param   ntrref         (Given)    The number of reference traces.
 * @param   y              (Given)    Detector y-coordinate where we
 *                                    should do the matching.
 * @param   errlim         (Given)    The maximum displacement in the
 *                                    spatial direction for a match
 *                                    (pixels).
 * @param   searchgrid     (Given)    Search limit for initial grid
 *                                    search (pixels).
 * @param   matches        (Returned) For each peak, the index of the
 *                                    reference trace that matched,
 *                                    or -1 if there was no match.
 * @param   missing        (Modified) The array of missing fibres.
 * @param   nmissing       (Modified) The number of missing fibres.
 *
 * @return  void
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void searchref (
    objstruct **objects,
    int nobj,
    int *iorder,
    qmost_traceinfo *trreflist,
    int ntrref,
    double y,
    double errlim,
    double searchgrid,
    int **matches,
    int **missing,
    int *nmissing)
{
    double *xpred = NULL;
    int *objindex = NULL;

    double *bestforobj = NULL;
    int *claimedby = NULL;

    objstruct *obj;
    qmost_traceinfo *trref;

    int irem, i, iref, nuse, imissing, imove;
    double xresid, best;

    int isp, ifp, index, ioth;
    double xpos;

    int l, lgrid, lmax, nmax, nmatch;
    double sdmax, sumdiff, offset;

    /* Evaluate positions for detected object list */
    xpred = cpl_malloc(nobj*sizeof(double));
    objindex = cpl_malloc(nobj*sizeof(int));

    nuse = 0;
    for(index = 0; index < nobj; index++) {
        i = iorder[index];
        obj = objects[i];

        if(obj->skipme) {
            continue;
        }

        /* Measured position in reference trace block used here for
         * stability against polynomial fit issues. */
        xpred[nuse] = obj->xpos[0];
        objindex[nuse] = i;
        nuse++;
    }

    /* Sort.  This should be a no-op since our sort order should match
     * the order the list was already sorted on in the parent, but
     * it's here as a precaution in case somebody changes what's used
     * in xpred above. */
    qmost_sort_di(xpred, objindex, nuse);

    /* Grid search limit */
    lgrid = qmost_nint(2 * searchgrid / errlim);
    if(lgrid < 0) {
        lgrid = 0;
    }

    /* Coarse grid search to find best offset */
    nmax = 0;
    sdmax = -1;
    lmax = 0;
    for (l = -lgrid; l <= lgrid; l++) {
        offset = 0.5*l*errlim;
        nmatch = 0;
        sumdiff = 0;

        imissing = 0;
        for(iref = 0; iref < ntrref; iref++) {
            trref = trreflist + iref;

            /* If fibre is not live in reference, skip */
            if(!trref->live) {
                continue;
            }

            /* Ensure fibre is not listed in missing */
            while(imissing < (*nmissing) &&
                  iref+1 > (*missing)[imissing]) {
                imissing++;
            }

            if(imissing < (*nmissing) &&
               (*missing)[imissing] == iref+1) {
                /* Fibre listed in missing, skip */
                continue;
            }

            /* x position based on reference trace table, with offset */
            xpos = cpl_polynomial_eval_1d(trref->coefs,
                                          (y - trref->yref) / trref->yref,
                                          NULL) + offset;

            /* Search for best match in detected objects */
            isp = 0;
            ifp = nuse-1;
            index = (isp + ifp)/2;

            while(ifp-isp >= 2) {
                if(xpred[index] < xpos - errlim) {
                    isp = index;
                    index = (index + ifp) / 2;
                }
                else if(xpred[index] > xpos - errlim) {
                    ifp = index;
                    index = (index + isp) / 2;
                }
                else {
                    isp = index;
                    break;
                }
            }

            irem = -1;
            best = 1.0e10;

            for(index = isp; index < nuse; index++) {
                if(xpred[index] > xpos + errlim)
                    break;

                i = objindex[index];

                xresid = fabs(xpos - xpred[index]);

                if (xresid < best && xresid < errlim) {
                    irem = i;
                    best = xresid;
                }
            }

            if(irem >= 0) {
                nmatch++;
                sumdiff += best;
            }
        }

        if (nmatch > nmax || (nmatch == nmax && sumdiff < sdmax)) {
            nmax = nmatch;
            sdmax = sumdiff;
            lmax = l;
        }
    }

    offset = 0.5*lmax*errlim;

    if(lmax != 0) {
        cpl_msg_info(cpl_func,
                     "Best trace offset relative to reference = %.1f pixels",
                     offset);
    }

    /* Find best match for each using binary chop in x */
    *matches = cpl_malloc(nobj*sizeof(int));
    bestforobj = cpl_malloc(nobj*sizeof(double));

    for(index = 0; index < nobj; index++) {
        (*matches)[index] = -1;
        bestforobj[index] = 1000.0;
    }

    claimedby = cpl_malloc(ntrref*sizeof(int));

    imissing = 0;
    for(iref = 0; iref < ntrref; iref++) {
        trref = trreflist + iref;

        /* If fibre is not live in reference, skip.  A side effect of
         * doing this it can never reappear.  The reference is
         * supposed to be made including all fibres, with predicted
         * positions for any that weren't present in the actual trace
         * used to make the reference, so they should all be flagged
         * good and it shouldn't be possible to hit this condition.
         * It's here to protect against misuse by the end user, in
         * case they attempt to use a normal trace file as the
         * reference. */
        if(!trref->live) {
            claimedby[iref] = -1;
            continue;
        }

        /* Ensure fibre is not listed in missing */
        while(imissing < (*nmissing) &&
              iref+1 > (*missing)[imissing]) {
            imissing++;
        }

        if(imissing < (*nmissing) &&
           (*missing)[imissing] == iref+1) {
            /* Fibre listed in missing, skip */
            claimedby[iref] = -1;
            continue;
        }

        /* Calculate x position based on reference trace table */
        xpos = cpl_polynomial_eval_1d(trref->coefs,
                                      (y - trref->yref) / trref->yref,
                                      NULL) + offset;

        /* Search for best match in detected objects */
        isp = 0;
        ifp = nuse-1;
        index = (isp + ifp)/2;

        while(ifp-isp >= 2) {
            if(xpred[index] < xpos - errlim) {
                isp = index;
                index = (index + ifp) / 2;
            }
            else if(xpred[index] > xpos - errlim) {
                ifp = index;
                index = (index + isp) / 2;
            }
            else {
                isp = index;
                break;
            }
        }

        irem = -1;
        best = 1.0e10;

        for(index = isp; index < nuse; index++) {
            if(xpred[index] > xpos + errlim)
                break;

            i = objindex[index];

            xresid = fabs(xpos - xpred[index]);

            if (xresid < best && xresid < errlim) {
                irem = i;
                best = xresid;
            }
        }

        /* Did we find one? */
        if(irem >= 0) {
            /* Check if another ref already matched to this object */
            ioth = (*matches)[irem];

            if(ioth >= 0) {  /* yes */
                /* Is the new one better? */
                if(best < bestforobj[irem]) {
                    /* Yes, it is better, unflag the other one */
                    claimedby[ioth] = -1;
                }
                else {
                    /* No, it's not better, unflag the present one */
                    irem = -1;
                }
            }
        }
        
        /* Record in result */
        claimedby[iref] = irem;

        /* Did we match? */
        if(irem >= 0) {
            (*matches)[irem] = iref;
            bestforobj[irem] = best;
        }
    }

    /* Check any fibres that didn't match */
    imissing = 0;
    for(iref = 0; iref < ntrref; iref++) {
        if(claimedby[iref] < 0) {
            /* Ensure fibre is listed in missing */
            while(imissing < (*nmissing) &&
                  iref+1 > (*missing)[imissing]) {
                imissing++;
            }

            if(!(imissing < (*nmissing) &&
                 (*missing)[imissing] == iref+1)) {
                /* Insert into missing, maintaining sort order */
                *missing = cpl_realloc(*missing,
                                       ((*nmissing)+1) * sizeof(int));

                for(imove = (*nmissing)-1; imove >= imissing; imove--) {
                    (*missing)[imove+1] = (*missing)[imove];
                }

                (*missing)[imissing] = iref+1;
                (*nmissing)++;

                cpl_msg_info(cpl_func,
                             "Fibre %d was not recovered in trace "
                             "or flagged as missing, check FIBINFO",
                             iref+1);
            }
        }
    }

    cpl_free(xpred);
    cpl_free(objindex);
    cpl_free(bestforobj);
    cpl_free(claimedby);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Linearly interpolate out bad pixels.
 *
 * Do a linear interpolation of pixels that are flagged as bad in the
 * input profile array by the input bad pixel mask.
 *
 * @param   indata         (Modified) The array with the input
 *                                    profiles.
 * @param   inbpmdata      (Given)    The array with the bad pixel
 *                                    mask for the profiles.
 * @param   nx             (Given)    The number of pixels in the
 *                                    profile and BPM arrays.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

static void badinterp(float *indata, unsigned char *inbpmdata, int nx) {
    int i,keep1,keep2,good1,good2,j;
    float val1,val2,slope,inter;

    /* Locate the first good pixel. If it isn't the first pixel in the 
       array, then make the leading bad pixels equal to the first good one */

    keep1 = 0;
    keep2 = nx-1;
    for (i = 0; i < nx; i++) {
	keep1 = i;
	if (! inbpmdata[i]) 
	    break;
    }
    for (i = 0; i < keep1; i++)
	indata[i] = indata[keep1];

    /* Locate the last  good pixel. If it isn't the last pixel in the 
       array, then make the trailing bad pixels equal to the last good one */

    for (i = nx-1; i >= 0; i--) {
	keep2 = i;
	if (! inbpmdata[i]) 
	    break;
    }
    for (i = nx-1; i > keep2; i--)
	indata[i] = indata[keep2];
    
    /* Ok, now look for places in the array where we need to interpolate */

    for (i = keep1; i <= keep2; i++) {
	if (inbpmdata[i]) {
	    val1 = indata[i-1];
	    good1 = i - 1;
            val2 = val1;
            good2 = good1 + 1;
	    for (j = good1+1; j <= keep2; j++) {
		if (! inbpmdata[j]) {
		    good2 = j;
		    val2 = indata[j];
		    break;
		}
	    }
	    i = j;
	    slope = (val2 - val1)/(float)(good2 - good1);
	    inter = val2 - slope*(float)good2;
	    for (j = good1+1; j <= good2-1; j++) 
		indata[j] = slope*(float)j + inter;
	}
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Get spatial FWHM for fibre at each trace position.
 *
 * @param   tr             (Modified) The traceinfo structure for the
 *                                    fibre to be populated with the
 *                                    measured FWHM.
 * @param   img            (Given)    The fibre flat image used to
 *                                    measure the trace.
 * @param   bpm            (Given)    The corresponding bad pixel
 *                                    mask.
 * @param   nx             (Given)    The x dimension of the image.
 * @param   ny             (Given)    The y dimension of the image.
 * @param   xbin           (Given)    The x (spatial) binning of the
 *                                    image.
 * @param   ybin           (Given)    The y (spectral) binning of the
 *                                    image.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void qmost_get_fwhm_spa (
    qmost_traceinfo *tr,
    float *img,
    cpl_binary *bpm,
    int nx,
    int ny,
    int xbin,
    int ybin)
{
    int i,pkloc,j,ixpeak,iypeak,iwin,iwin2;
    int fpix[2],lpix[2];
    float pk,dist,loc1,loc2;
    int found1,found2;
    double ypos,xpos;
    float tmpdata[FWHMWIN];
    unsigned char tmpbpm[FWHMWIN];
    float data[FWHMWIN];
    float val, bk, fwhm;
    int ipos, p;
    cpl_errorstate prestate;

    iwin = FWHMWIN;
    iwin2 = iwin/2;
    
    for(ipos = 0; ipos < tr->npos; ipos++) {
        ypos = tr->ypos[ipos];
        xpos = qmost_tracexpos(*tr,ypos);
        
        /* Subset of the data at this position */
        ixpeak = qmost_nint((xpos - 0.5) / xbin + 0.5);
        iypeak = qmost_nint((ypos - 0.5) / ybin + 0.5);

        fpix[0] = ixpeak - iwin2;
        fpix[1] = iypeak - iwin2;
        lpix[0] = fpix[0] + iwin - 1;
        lpix[1] = fpix[1] + iwin - 1;

        /* Check it's entirely on the detector, skip if not */
        if(fpix[0] < 1 || fpix[0] > nx ||
           lpix[0] < 1 || lpix[0] > nx ||
           fpix[1] < 1 || fpix[1] > ny ||
           lpix[1] < 1 || lpix[1] > ny) {
            tr->fwhm[ipos] = NAN;
            continue;
        }
        
        /* Block median along the spectral direction */
        for (i = 0; i < iwin; i++) {
            for (j = 0; j < iwin; j++) {
                p = (fpix[1] + j - 1) * nx + (fpix[0] + i - 1);
                tmpdata[j] = img[p];
                tmpbpm[j] = bpm[p];
            }

            prestate = cpl_errorstate_get();

            if(qmost_med(tmpdata,tmpbpm,iwin,&val) != CPL_ERROR_NONE) {
                cpl_errorstate_set(prestate);
                val = 0;
            }

            data[i] = val;
        }
        
        /* Find the location of the peak */
        
        pk = data[iwin2];
        pkloc = iwin2;
        for (j = iwin2-2; j <= iwin2+2; j++) {
            if (data[j] > pk) {
                pk = data[j];
                pkloc = j;
            }
        }
        
        /* Find a reference background level */
        bk = data[0];
        for (j = 1; j < iwin; j++) {
            if (data[j] < bk) {
                bk = data[j];
            }
        }
        pk += bk; /* weird but saves subracting bk below */

        /* Look for the first half light point */

        found1 = 0;
        for (j = pkloc - 1; j > 0; j--) {
            if (data[j] < 0.5*pk) {
                found1 = 1;
                break;
            }
        }

        if(found1 && data[j+1] != data[j]) {
            dist = (0.5*pk - data[j])/(data[j+1] - data[j]);
        }
        else {
            dist = 0.5;
        }

        loc1 = (float)j + dist;

        /* Look for the second half light point */

        found2 = 0;
        for (j = pkloc + 1; j < iwin; j++) {
            if (data[j] < 0.5*pk) {
                found2 = 1;
                break;
            }
        }

        if(found2 && data[j-1] != data[j]) {
            dist = (0.5*pk - data[j])/(data[j-1] - data[j]);
        }
        else {
            dist = 0.5;
        }

        loc2 = (float)j - dist;

        /* Full distance now with traps */

        if(found1 && found2) {
            fwhm = qmost_max(1.0,loc2 - loc1);
            fwhm = qmost_min((float)iwin2,fwhm);

            tr->fwhm[ipos] = fwhm * xbin;
        }
        else {
            /* Flag as couldn't determine */
            tr->fwhm[ipos] = -1;
        }
    }
}

/**@}*/

/*

$Log$
Revision 1.22  20230216  jmi
Overhauled the calculation of the block positions to make it safer and
more robust against nonsense input values of startline, and require
each object matches to a maximum of one peak, resolving any conflicts
by choosing the closer peak.  The first block was previously larger
than the others and got partially repeated in the next block, so the
samples weren't independent, and it was possible to run out of bounds
in the results array in two ways: by running nblocks+1 times, or by
matching an object to more than one peak.  These issues should now be
fixed.  Also cleaned up some old commented out code.

Revision 1.21  20221120  mji
added fix to trap edge effects when searching for trough boundaries

Revision 1.20  20221030  mji
replaced mask half-width estimate with float for better precision
and minor update to correct base level estimate in findpeaks

Revision 1.19  20220802  mji
minor mod to improve edge of slitmask performance

Revision 1.18  20220510  mji
slight upgrade to peak definition in findpeaks

Revision 1.17  20210419  mji
replaced peak finding algortihm removed background code and tided up some

Revision 1.16  2019/03/31 18:34:40  jrl
modified _get_fwhm_spa to calculate sigma

Revision 1.15  2019/03/26 13:54:01  jrl
Added a line to _get_fwhm_spa so that fwhm has a default return value

Revision 1.14  2019/03/21 14:04:50  jrl
Fixed memory leaks in _get_fwhm_spa

Revision 1.13  2019/03/21 11:35:55  jrl
fixed bugs in _get_fwhm_spa

Revision 1.12  2019/03/18 13:26:24  jrl
Added routine _get_fwhm_spa

Revision 1.11  2019/02/25 10:32:19  jrl
New memory allocation scheme

Revision 1.10  2018/08/29 21:17:05  jrl
changed minimum displacement between adjacent peaks (in x)

Revision 1.9  2018/06/27 09:41:29  jim
change size of declaration for results structure

Revision 1.8  2017/08/02 08:58:03  jim
Modified to use a slit mask now

Revision 1.7  2017/03/14 10:59:32  jim
Now detects the ends of each trace

Revision 1.6  2017/01/17 08:55:41  jim
Lots of changes including QC data being generated

Revision 1.5  2016/10/23 15:55:38  jim
Added docs

Revision 1.4  2016/08/24 11:52:29  jim
*** empty log message ***

Revision 1.3  2016/07/06 11:02:00  jim
Modified mainly to clean out commented code

Revision 1.2  2016/05/20 09:58:57  jim
Uses new version of trace file

Revision 1.1  2016/05/16 09:00:23  jim
Initial entry

*/
