/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_DFS_H
#define QMOST_DFS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Define here the RAW   tag keywords */
#define QMOST_RAW_BIAS                  "BIAS"
#define QMOST_RAW_DARK                  "DARK"
#define QMOST_RAW_DETECTOR_FLAT         "DETECTOR_FLAT"
#define QMOST_RAW_DETECTOR_FLAT_LIN     "DETECTOR_FLAT_LIN"
#define QMOST_RAW_DETECTOR_FLAT_MON     "DETECTOR_FLAT_MON"
#define QMOST_RAW_FIBRE_FLAT_DAY        "FIBRE_FLAT_DAY"
#define QMOST_RAW_FIBRE_FLAT_NIGHT      "FIBRE_FLAT_NIGHT"
#define QMOST_RAW_FIBRE_FLAT_SKY        "FIBRE_FLAT_SKY"
#define QMOST_RAW_FIBRE_WAVE_DAY        "FIBRE_WAVE_DAY"
#define QMOST_RAW_FIBRE_WAVE_NIGHT      "FIBRE_WAVE_NIGHT"
#define QMOST_RAW_FIBRE_WAVE_SIMUARC    "FIBRE_WAVE_SIMUARC"
#define QMOST_RAW_FIBRE_WAVE_SIMUFPE    "FIBRE_WAVE_SIMUFPE"
#define QMOST_RAW_SCIENCE               "OBJECT"

/* Define here the CALIB tag keywords */
#define QMOST_CALIB_MASTER_BPM              "MASTER_BPM"
#define QMOST_CALIB_REFERENCE_BIAS          "REFERENCE_BIAS"
#define QMOST_CALIB_REFERENCE_DARK          "REFERENCE_DARK"
#define QMOST_CALIB_REFERENCE_DETECTOR_FLAT "REFERENCE_DETECTOR_FLAT"
#define QMOST_CALIB_REFERENCE_FIBRE_TRACE   "REFERENCE_FIBRE_TRACE"
#define QMOST_CALIB_REFERENCE_WAVE          "REFERENCE_WAVE"
#define QMOST_CALIB_REFERENCE_FIBRE_FLAT    "REFERENCE_FIBRE_FLAT"
#define QMOST_CALIB_SLIT_MASK               "SLIT_MASK"
#define QMOST_CALIB_MASTER_WAVE_MAP         "WAVE_MAP"
#define QMOST_CALIB_ARC_LINELIST            "ARC_LINELIST"
#define QMOST_CALIB_SENSITIVITY             "SENSITIVITY"

/* Define here the OUT   tag keywords */
#define QMOST_PRO_MASTER_BIAS                   "MASTER_BIAS"
#define QMOST_PRO_DIFFIMG_BIAS                  "DIFFIMG_BIAS"
#define QMOST_PRO_MASTER_DARK                   "MASTER_DARK"
#define QMOST_PRO_DIFFIMG_DARK                  "DIFFIMG_DARK"
#define QMOST_PRO_MASTER_DETECTOR_FLAT          "MASTER_DETECTOR_FLAT"
#define QMOST_PRO_DIFFIMG_DETECTOR_FLAT         "DIFFIMG_DETECTOR_FLAT"
#define QMOST_PRO_UNFILTERED_DETECTOR_FLAT      "UNFILTERED_DETECTOR_FLAT"
#define QMOST_PRO_LINEARITY                     "LINEARITY"
#define QMOST_PRO_BPM                           "BPM"
#define QMOST_PRO_CRMASK                        "COSMIC_RAY_MASK"
#define QMOST_PRO_FIBRE_TRACE                   "FIBRE_TRACE"
#define QMOST_PRO_FIBRE_MASK                    "FIBRE_MASK"
#define QMOST_PROC_FIBRE_FLAT                   "PROC_FIBRE_FLAT"
#define QMOST_PRO_FPE_LINELIST                  "FPE_LINELIST"
#define QMOST_PRO_FPE_SPECTRUM                  "FPE_SPECTRUM"
#define QMOST_PROC_FIBRE_WAVE                   "PROC_FIBRE_WAVE"
#define QMOST_PRO_MASTER_ARC                    "MASTER_ARC"
#define QMOST_PRO_MASTER_WAVE                   "MASTER_WAVE"
#define QMOST_PRO_MASTER_WARC                   "MASTER_WARC"
#define QMOST_PRO_OB_ARC                        "OB_ARC"
#define QMOST_PRO_OB_WAVE                       "OB_WAVE"
#define QMOST_PRO_OB_WARC                       "OB_WARC"
#define QMOST_PRO_SIMUARC_ARC                   "SIMUARC_ARC"
#define QMOST_PRO_SIMUARC_WAVE                  "SIMUARC_WAVE"
#define QMOST_PRO_SIMUARC_WARC                  "SIMUARC_WARC"
#define QMOST_PRO_MASTER_PSF                    "MASTER_PSF"
#define QMOST_PRO_MASTER_FIBRE_FLAT             "MASTER_FIBRE_FLAT"
#define QMOST_PRO_EXTRACTED_FIBRE_FLAT          "EXTRACTED_FIBRE_FLAT"
#define QMOST_PRO_OB_FIBRE_FLAT                 "OB_FIBRE_FLAT"
#define QMOST_PRO_SKY_FIBRE_FLAT                "SKY_FIBRE_FLAT"
#define QMOST_PRO_READGAIN                      "READGAIN"
#define QMOST_PROC_SCIENCE                      "PROC_SCIENCE"
#define QMOST_PRO_EIGEN                         "EIGEN"
#define QMOST_PRO_SKY                           "SKY"
#define QMOST_PRO_UNCALIBRATED_SCIENCE          "UNCALIBRATED_SCIENCE"
#define QMOST_PRO_SCIENCE                       "SCIENCE"

/**
 * @ingroup qmost_dfs
 *
 * @brief   Regular expression used to copy non-structural FITS
 *          keywords from an input to output HDU.
 *
 * This giant regular expression is used to filter out keywords we
 * should not copy from the input extension when creating an output
 * extension.  The list is based on CFITSIO's ffgkcl function, and the
 * comments note which of CFITSIO's TYP_*_KEY constants these
 * correspond to.  If we used ffgkcl we'd also filter the COMMENT
 * keywords defining the FITS format as part of TYP_STRUC_KEY.  I
 * haven't tried to do that here.
 */

#define QMOST_REGEXP_HDUCOPY "^("                                       \
  "SIMPLE|BITPIX|NAXIS|NAXIS[0-9][0-9]*|EXTEND|BLOCKED|"  /* STRUC */   \
  "GROUPS|PCOUNT|GCOUNT|END|"                                           \
  "XTENSION|TFIELDS|TTYPE[0-9][0-9]*|TBCOL[0-9][0-9]*|TFORM[0-9][0-9]*|THEAP|" \
  "Z("  /* CMPRS, these all start with Z */                             \
  "IMAGE|CMPTYPE|NAME[0-9][0-9]*|VAL[0-9][0-9]*|TILE[0-9][0-9]*|"       \
  "BITPIX|NAXIS|NAXIS[0-9][0-9]*|SCALE|ERO|BLANK|"                      \
  "SIMPLE|TENSION|EXTEND|BLOCKED|PCOUNT|GCOUNT|"                        \
  "QUANTIZ|DITHER0"                                                     \
  ")|"                                                                  \
  "BSCALE|BZERO|TSCAL[0-9][0-9]*|TZERO[0-9][0-9]*|"  /* SCAL */         \
  "BLANK|TNULL[0-9][0-9]*|"  /* NULL */                                 \
  "TDIM[0-9][0-9]*|"  /* DIM */                                         \
  "DATAMIN|DATAMAX|"  /* RANG */                                        \
  "TLMIN[0-9][0-9]*|TLMAX[0-9][0-9]*|TDMIN[0-9][0-9]*|TDMAX[0-9][0-9]*|" \
  "BUNIT|TUNIT[0-9][0-9]*|"  /* UNIT */                                 \
  "TDISP[0-9][0-9]*|"  /* DISP */                                       \
  "EXTNAME|EXTVER|EXTLEVEL|HDUNAME|HDUVER|HDULEVEL|"  /* HDUID */       \
  "CHECKSUM|DATASUM|"  /* CKSUM */                                      \
  "CTYPE[0-9][0-9]*|CUNIT[0-9][0-9]*|CRVAL[0-9][0-9]*|"                 \
  "CRPIX[0-9][0-9]*|CROTA[0-9][0-9]*|CRDER[0-9][0-9]*|"                 \
  "CSYER[0-9][0-9]*|CDELT[0-9][0-9]*|CD[0-9][0-9]*_[0-9][0-9]*|"        \
  "LONPOLE[0-9][0-9]*|LATPOLE[0-9][0-9]*|"                              \
  "LONP[0-9][0-9]*|LATP[0-9][0-9]*|"                                    \
  "PC[0-9][0-9]*_[0-9][0-9]*|PV[0-9][0-9]*_[0-9][0-9]*|"                \
  "PS[0-9][0-9]*_[0-9][0-9]*"  /* main WCS, no alt. or table */         \
")$"

/*----------------------------------------------------------------------------*/
/*
 *                              Functions prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_dfs_set_groups(cpl_frameset *);

char *qmost_dfs_product_filename (
    const int ispec,
    const char *procatg);

const cpl_frame *qmost_dfs_setup_product (
    cpl_frameset *frameset,
    const cpl_parameterlist *parlist,
    const char *recipe,
    const char *filename,
    const char *procatg,
    const cpl_frame_type protype,
    const cpl_propertylist *applist);

const cpl_frame *qmost_dfs_setup_product_default (
    cpl_frameset *frameset,
    const cpl_parameterlist *parlist,
    const char *recipe,
    const int ispec,
    const char *procatg,
    const cpl_frame_type protype,
    const cpl_propertylist *applist);

cpl_propertylist *qmost_dfs_setup_product_extension_header (
    cpl_propertylist *inhdr,
    const char *extname,
    int dummy,
    cpl_propertylist *qclist);

cpl_error_code qmost_dfs_save_image_extension (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_image *image,
    cpl_type type);

cpl_error_code qmost_dfs_save_image_and_var (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_image *image,
    cpl_image *var,
    cpl_type type);

cpl_error_code qmost_dfs_save_image_and_ivar (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_image *image,
    cpl_image *var,
    cpl_type type);

cpl_error_code qmost_dfs_save_imagelist_extension (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_imagelist *imagelist,
    cpl_type type);

cpl_error_code qmost_dfs_save_mask_extension (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_mask *mask);

cpl_error_code qmost_dfs_save_table_extension (
    const cpl_frame *frame,
    cpl_propertylist *inhdr,
    const char *extname,
    cpl_propertylist *qclist,
    cpl_table *table);

CPL_END_DECLS

#endif
