/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_blk.h"
#include "qmost_dfs.h"
#include "qmost_filt1d.h"
#include "qmost_scattered.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_scattered  qmost_scattered
 *
 * Scattered light estimation and removal.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_scattered.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Minimum ADU value to histogram for background stats */

#define MINHISTVAL -1000

/* Maximum ADU value to histogram for background stats */

#define MAXHISTVAL 65535

/* Size of histogram array */

#define MAXHIST (MAXHISTVAL-MINHISTVAL+1)

/* Flag value for bad background bin */

#define BLANKVAL -1000.0

/*----------------------------------------------------------------------------*/
/**
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static void qmost_backmap (
    float *data,
    unsigned char *bpm,
    long *naxis, 
    int nbsize,
    float nullval,
    float **out,
    float *avsky,
    float *medresid,
    float *sigresid);

static void backmap_bfilt (
    float **xbuf,
    int nx,
    int ny);

static void hanning (
    float *xbuf,
    int npt);

/*----------------------------------------------------------------------------*/
/**
 * @brief   Estimate and remove scattered light from image.
 *
 * The given image is divided into cells of nbsize x nbsize pixels.
 * Within each cell, pixels that are set false in the in_mask argument
 * are histogrammed and a robust 3 sigma upper envelope clipped median
 * background level determined in each cell.  The cells are smoothed
 * using sequential 1D median filters (kernel width 3 cells) in each
 * axis (x then y), followed by another pair of sequential 1D linear
 * 1-2-1 Hanning weight filters, to obtain a smoothed, low resolution
 * background map.  This map is then interpolated back onto the
 * original pixel grid using bilinear interpolation to yield the final
 * scattered light map.  The scattered light map is finally subtracted
 * from the original image (in place) to correct the scattered light
 * background.
 *
 * @param   in_image   (Modified) Input image to process.  The data
 *                                type must be CPL_TYPE_FLOAT.
 * @param   in_hdr     (Modified) FITS header to populate with QC.
 * @param   in_mask    (Given)    Fibre mask flagging pixels that are
 *                                part of a fibre image as true, and
 *                                pixels that are not part of a fibre
 *                                image as false.  The pixels flagged
 *                                as false will be used to determine
 *                                the background (scattered light).
 * @param   nbsize     (Given)    The size of the background
 *                                estimation cells in pixels.
 * @param   pedestal   (Given)    If true, the average background
 *                                level is subtracted off as well as
 *                                the variations, leaving the average
 *                                background in the output close to
 *                                zero.  If false, only the background
 *                                variations are subtracted, leaving
 *                                the average background level (DC
 *                                offset) in place.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If sizes of input images and
 *                                        masks don't match.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input image data type was
 *                                    not float, or if one of the
 *                                    input FITS header keyword values
 *                                    had an incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS SPATBIN</b>
 *
 * @par Output DRS Headers:
 *   - <b>AVBCKREM</b> (ADU): The average background level of the
 *     scattered light map.
 *
 * @par Output QC Parameters:
 *   - <b>SCATTL RESID MED</b> (ADU): The median residual scattered
 *     light background remaining after subtraction.
 *   - <b>SCATTL RESID RMS</b> (ADU): The RMS residual of the
 *     scattered light background remaining after subtraction.
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_scattered (
    cpl_image *in_image,
    cpl_propertylist *in_hdr,
    cpl_mask *in_mask,
    int nbsize,
    int pedestal)
{
    cpl_image *tmp_image = NULL;
    cpl_mask *mask;
    cpl_mask *binned_mask = NULL;
    cpl_binary *mask_buf;

    cpl_image *scat_image;

    int i,specbin,spatbin,isbinned,blkfac[2];
    long naxis[2],naxis_mask[2],npts;
    float avsky,*indata=NULL,*scatdata=NULL,*odata=NULL;
    unsigned char *mdata=NULL;

    float medresid, sigresid;

#undef TIDY
#define TIDY                                    \
    if(tmp_image != NULL) {                     \
        cpl_image_delete(tmp_image);            \
        tmp_image = NULL;                       \
    }                                           \
    if(binned_mask != NULL) {                   \
        cpl_mask_delete(binned_mask);           \
        binned_mask = NULL;                     \
    }                                           \
    if(mdata != NULL) {                         \
        cpl_free(mdata);                        \
        mdata = NULL;                           \
    }                                           \
    if(odata != NULL) {                         \
        cpl_free(odata);                        \
        odata = NULL;                           \
    }

    /* Get size of image */
    naxis[0] = cpl_image_get_size_x(in_image);
    naxis[1] = cpl_image_get_size_y(in_image);

    npts = naxis[0]*naxis[1];

    /* Check if binned */
    qmost_isbinned(in_hdr,&specbin,&spatbin,&isbinned);

    if(in_mask) {
        /* Check size */
        naxis_mask[0] = cpl_mask_get_size_x(in_mask);
        naxis_mask[1] = cpl_mask_get_size_y(in_mask);
        
        if (naxis_mask[0] != naxis[0]*spatbin ||
            naxis_mask[1] != naxis[1]*specbin) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "image (%ld,%ld), mask (%ld,%ld) "
                                         "sizes don't match",
                                         naxis[0]*spatbin,naxis[1]*specbin,
                                         naxis_mask[0],naxis_mask[1]);
        }
        
        /* Bin down to match data */
        blkfac[0] = spatbin;
        blkfac[1] = specbin;
        
        binned_mask = qmost_maskblk(in_mask, blkfac);
        if(binned_mask == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "binning (%d,%d) of mask "
                                         "failed",
                                         blkfac[0], blkfac[1]);
        }

        /* Duplicate input image */
        tmp_image = cpl_image_duplicate(in_image);
        if(tmp_image == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to duplicate input image");
        }

        if(cpl_mask_or(cpl_image_get_bpm(tmp_image),
                       binned_mask) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't combine masks");
        }

        cpl_mask_delete(binned_mask);
        binned_mask = NULL;

        scat_image = tmp_image;
    }
    else {
        scat_image = in_image;
    }

    /* Get pointer to image */
    indata = cpl_image_get_data_float(in_image);
    if(indata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "input image");
    }

    scatdata = cpl_image_get_data_float(scat_image);
    if(scatdata == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get float pointer to "
                                     "scattered light image");
    }

    /* Get mask */
    mask = cpl_image_get_bpm(scat_image);
    if(mask == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get scattered light mask");
    }

    /* Copy the mask to internal data structure */
    mask_buf = cpl_mask_get_data(mask);
    if(mask_buf == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get pointer to "
                                     "scattered light mask");
    }

    mdata = (unsigned char *) cpl_malloc(npts * sizeof(unsigned char));
    memcpy(mdata, mask_buf, npts * sizeof(unsigned char));

    /* Filter the data and get an output scattered light map */

    qmost_backmap(scatdata,mdata,naxis,nbsize,BLANKVAL,&odata,&avsky,
                  &medresid,&sigresid);

    /* Done with the mask and temporary image */
    cpl_free(mdata);
    mdata = NULL;

    scatdata = NULL;
    scat_image = NULL;
    if(tmp_image) {
        cpl_image_delete(tmp_image);
        tmp_image = NULL;
    }

    cpl_msg_info(cpl_func, "Average background = %f", avsky);

    /* Correct the data. Do one loop or the other depending on whether you're
       removing the pedestal level too */

    if (pedestal) 
	for (i = 0; i < npts; i++) 
	    indata[i] -= (odata[i] + avsky);
    else
	for (i = 0; i < npts; i++)
	    indata[i] -= odata[i];

    cpl_free(odata);
    odata = NULL;

    /* Update the input file */

    cpl_propertylist_update_float(in_hdr, "ESO DRS AVBCKREM", avsky);
    cpl_propertylist_set_comment(in_hdr, "ESO DRS AVBCKREM",
                                 "[ADU] Average background level removed");

    cpl_propertylist_update_float(in_hdr,
                                  "ESO QC SCATTL RESID MED", medresid);
    cpl_propertylist_set_comment(in_hdr,
                                 "ESO QC SCATTL RESID MED",
                                 "[ADU] Residual scattered light level");

    cpl_propertylist_update_float(in_hdr,
                                  "ESO QC SCATTL RESID RMS", sigresid);
    cpl_propertylist_set_comment(in_hdr,
                                 "ESO QC SCATTL RESID RMS",
                                 "[ADU] Residual scattered light RMS");

    /* Right, get out of here */

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Create a background map from in input data file and its
 *          mask.
 *
 * Filters an input map (taking into account pixel masking) and
 * outputs a separate array with the background map. The output array
 * will have been normalised to zero median.
 *
 * @param   data       (Given)    The input data array.
 * @param   bpm        (Given)    The input bad pixel mask.
 * @param   naxis      (Given)    The axis size array.
 * @param   nbsize     (Given)    The size of the smoothing cells in
 *                                pixels.
 * @param   nullval    (Given)    Value used in data for NULL pixels.
 * @param   out        (Returned) The output background map.
 * @param   avsky      (Returned) The median background level.
 * @param   medresid   (Returned) The median residual after background
 *                                subtraction for QC.
 * @param   sigresid   (Returned) The sigma residual after background
 *                                subtraction for QC.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void qmost_backmap (
    float *data,
    unsigned char *bpm,
    long *naxis, 
    int nbsize,
    float nullval,
    float **out,
    float *avsky,
    float *medresid,
    float *sigresid)
{
    float fracx,fracy,skymedc,sigmac,fnbsize,dely,delx,backval,resid;
    float t1,t2,**bvals,*work;
    int ifracx,ifracy,nbsizx,nbsizy,nbx,nby,npixstripe,l,i,ll,*shist,nx,ny;
    int isquare,ilev,j,nbsizo2,kk,k,iby,ibyp1,ibx,ibxp1;
    int **hist,*nnp;
    cpl_errorstate prestate;

    /* Set up some variables */

    nx = (int)naxis[0];
    ny = (int)naxis[1];
    *out = NULL;
    *avsky = 0.0;

    /* Check to see if nbsize is close to exact divisor */

    fracx = ((float)nx)/((float)nbsize);
    fracy = ((float)ny)/((float)nbsize);
    ifracx = (int)(fracx + 0.1);
    ifracy = (int)(fracy + 0.1);
    nbsizx = nx/ifracx;
    nbsizy = ny/ifracy;
    nbsize = qmost_max(qmost_nint(0.9*nbsize),qmost_min(nbsize,qmost_min(nbsizx,nbsizy)));
    nbsize = qmost_min(nx,qmost_min(ny,nbsize)); 

    /* Divide the map into partitions */

    nbx = nx/nbsize;
    nby = ny/nbsize;
    npixstripe = nbsize*nx;

    /* Get histogram workspace if you can */

    hist = cpl_calloc(nbx,sizeof(int *));
    for (l = 0; l < nbx; l++)
        hist[l] = cpl_malloc(MAXHIST*sizeof(int));

    /* Same for background values array */

    bvals = cpl_malloc(nby*sizeof(float *));
    for (l = 0; l < nby; l++)
        bvals[l] = cpl_malloc(nbx*sizeof(float));

    /* Finally a counter array */

    nnp = cpl_malloc(nbx*sizeof(int));

    /* Loop for each row of background squares. Start by initialising
       the accumulators and histograms */

    for (l = 0; l < nby; l++) {
        memset(nnp,0,nbx*sizeof(int));
        for (i = 0; i < nbx; i++) 
            memset(hist[i],0,MAXHIST*sizeof(int));

        /* Skim through the data in this stripe. Find out which square each
           belongs to and add it to the relevant histogram */

        ll = l*npixstripe;
        for (i = 0; i < npixstripe; i++) {
            if (data[i+ll] != nullval && bpm[i+ll] == 0) {
                isquare = (int)((float)(i % nx)/(float)nbsize);
                isquare = qmost_min(nbx-1,qmost_max(0,isquare));
                ilev = qmost_min(MAXHISTVAL,qmost_max(MINHISTVAL,qmost_nint(data[i+ll])));
                hist[isquare][ilev-MINHISTVAL] += 1;
                nnp[isquare] += 1;
            }
        }
  
        /* But only do background estimation if enough pixels ----------- */

        for (j = 0; j < nbx; j++) {
            if (nnp[j] > 4.0*nbsize) {
                shist = hist[j];
                qmost_skylevel(shist, 0, MAXHIST-1, nnp[j],
                               -FLT_MAX, 3, 3,
                               &skymedc, &sigmac, NULL);
                skymedc += MINHISTVAL;
#ifdef SCATTERED_LOWER_QUARTILE
                bvals[l][j] = skymedc - sigmac/1.48;  /* lower quartile */
#else
                bvals[l][j] = skymedc;
#endif  /* SCATTERED_LOWER_QUARTILE */
            } else {
                bvals[l][j] = BLANKVAL;
            }
        }
    }

    /* filter raw background values */

    backmap_bfilt(bvals,nbx,nby);

    /* Get the output map */

    work = cpl_malloc(nbx*nby*sizeof(float));
    *out = cpl_malloc(nx*ny*sizeof(float));

    /* compute average sky level */

    k = 0;
    for (l = 0; l < nby; l++)
        for (j = 0; j < nbx; j++) 
            work[k++] = bvals[l][j];

    prestate = cpl_errorstate_get();

    if(qmost_med(work,NULL,nbx*nby,avsky) != CPL_ERROR_NONE) {
        cpl_errorstate_set(prestate);
        avsky = 0;
    }

    cpl_free(work);
    work = NULL;

    /* OK now model the full background map (normalised to zero mean) */

    nnp[0] = 0;
    memset(hist[0],0,MAXHIST*sizeof(int));

    nbsizo2 = nbsize/2;
    fnbsize = 1.0/((float)nbsize);
    for (k = 0; k < ny; k++) {
        kk = k*nx;

        /* Nearest background pixel vertically */

        iby = (k + 1 + nbsizo2)/nbsize;
        ibyp1 = iby + 1;
        iby = qmost_min(nby,qmost_max(1,iby));
        ibyp1 = qmost_min(nby,ibyp1);
        dely = (k + 1 - nbsize*iby + nbsizo2)*fnbsize;

        for (j = 0; j < nx; j++) {
            if (data[kk+j] == nullval) {
		(*out)[kk+j] = 0.0;
                continue;
	    }

            /* nearest background pixel across */

            ibx = (j + 1 + nbsizo2)/nbsize;
            ibxp1 = ibx + 1;
            ibx = qmost_min(nbx,qmost_max(1,ibx));
            ibxp1 = qmost_min(nbx,ibxp1);
            delx = (j + 1 - nbsize*ibx + nbsizo2)*fnbsize;

            /* bilinear interpolation to find background */
            
            t1 = (1.0 - dely)*bvals[iby-1][ibx-1] + dely*bvals[ibyp1-1][ibx-1];
            t2 = (1.0 - dely)*bvals[iby-1][ibxp1-1] + dely*bvals[ibyp1-1][ibxp1-
1];
            backval = (1.0 - delx)*t1 + delx*t2;

            (*out)[kk+j] = backval - *avsky;

            /* Accumulate statistics of residual */
            if(bpm[kk+j] == 0) {
                resid = data[kk+j] - backval;
                ilev = qmost_min(MAXHISTVAL,qmost_max(MINHISTVAL,qmost_nint(resid)));
                hist[0][ilev-MINHISTVAL] += 1;
                nnp[0] += 1;
            }
        }
    }

    /* Residual level and RMS */
    qmost_skylevel(hist[0], 0, MAXHIST-1, nnp[0],
                   -FLT_MAX, 3, 3,
                   medresid, sigresid, NULL);
    *medresid += MINHISTVAL;

    /* Free up some workspace */

    for (i = 0; i < nby; i++)
	cpl_free(bvals[i]);
    cpl_free(bvals);

    for (i = 0; i < nbx; i++) {
	cpl_free(hist[i]);
        hist[i] = NULL;
    }
    cpl_free(hist);
    hist = NULL;
    cpl_free(nnp);
    nnp = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Smooth background map using median and mean filters.
 *
 * @param   xbuf       (Modified) The background map.
 * @param   nx         (Given)    Number of pixels in x (most rapidly
 *                                varying direction).
 * @param   ny         (Given)    Number of pixels in y.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void backmap_bfilt(
    float **xbuf,
    int nx,
    int ny)
{
    float *ybuf,*save;
    int mfilt=3,j,k;
 
    /* Allocate temporary storage */

    ybuf = cpl_malloc(qmost_max(nx,ny)*sizeof(float));
    save = cpl_malloc((nx+1)*ny*sizeof(float));

    /* Median filter across */

    for (k = 0; k < ny; k++) {
        for (j = 0; j < nx; j++) {
            save[(nx+1)*k+j] = xbuf[k][j];
            ybuf[j] = xbuf[k][j];
	}
	qmost_filt1d(ybuf,nx,mfilt,0,BLANKVAL);
        for (j = 0; j < nx; j++) {
            if (ybuf[j] == BLANKVAL) {
	        ybuf[j] = 0.0;
            }
            xbuf[k][j] = ybuf[j];
        }
    }
 
    /* And now vertically */

    for (k = 0; k < nx; k++) {
        for (j = 0; j < ny; j++) 
	    ybuf[j] = xbuf[j][k];
        qmost_filt1d(ybuf,ny,mfilt,0,BLANKVAL);
        for (j = 0; j < ny; j++) {
            xbuf[j][k] = ybuf[j];
        }
    }

    /* Now repeat with linear filters across */

    for (k = 0; k < ny; k++) {
        for (j = 0; j < nx; j++) 
	    ybuf[j] = xbuf[k][j];
	hanning(ybuf,nx);
        for (j = 0; j < nx; j++) 
	    xbuf[k][j] = ybuf[j];
    }

    /* And now vertically */

    for (k = 0; k < nx; k++) {
        for (j = 0; j < ny; j++) 
	    ybuf[j] = xbuf[j][k];
	hanning(ybuf,ny);
        for (j = 0; j < ny; j++) 
	    xbuf[j][k] = ybuf[j];
    }

    /* Free temporary storage */

    cpl_free(ybuf);
    cpl_free(save);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Filter 1D array using 1-2-1 Hanning filter.
 *
 * @param   xbuf       (Modified) Array to filter (modified in place).
 * @param   npt        (Given)    Number of data points in array.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void hanning(
    float *xbuf,
    int npt)
{
    float *ybuf;
    float sum = 0.0,xmns,xmnf;
    int nfilt=3,i,il,ilow,nelem;

    if (npt <= nfilt)
        return;

    /* Set first and last edges equal */

    il   = nfilt/2;
    ilow = qmost_max(3,nfilt/4);
    ilow = (ilow/2)*2 + 1;
    for (i = 0; i < ilow; i++)
	sum += xbuf[i];
    xmns = sum/((float)ilow);
    sum=0.0;
    for (i = 0; i < ilow; i++)
	sum += xbuf[npt-1-i];
    xmnf = sum/((float)ilow);

    /* Allocate ybuf array to the maximum number of elements required */

    nelem = npt + nfilt;
    ybuf = cpl_malloc(nelem*sizeof(float));

    /* Reflect edges before filtering */

    for (i = 0; i < il; i++) {
        ybuf[i] = 2.0*xmns - xbuf[il+ilow-1-i];
        ybuf[npt+i+il] = 2.0*xmnf - xbuf[npt-i-ilow-1];
    } 
    for (i = 0; i < npt; i++)
        ybuf[i+il] = xbuf[i];

    /* Do linear 1-2-1 Hanning weighted filtering on rest */

    for (i = 0; i < npt; i++)
       xbuf[i] = 0.25*(ybuf[i] + 2.0*ybuf[i+1] + ybuf[i+2]); 

    /* Tidy up */

    cpl_free(ybuf);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Populate scattered light diagnostics in QC.
 *
 * @param   in_image   (Modified) Input image to process.  The data
 *                                type must be CPL_TYPE_FLOAT.
 * @param   in_hdr     (Modified) The corresponding image FITS
 *                                header.
 * @param   in_mask    (Given)    Master mask defining background
 *                                pixels.
 * @param   qclist     (Modified) The output FITS header to receive
 *                                the QC parameters.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If sizes of input images and
 *                                        masks don't match.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input image data type was
 *                                    not float, or if one of the
 *                                    input FITS header keyword values
 *                                    had an incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS SPECBIN</b>
 *   - <b>ESO DRS SPATBIN</b>
 *
 * @par Output QC Parameters:
 *   - <b>SCATTL MED</b> (ADU): The median scattered light level on
 *     the detector outside the region illuminated by the fibres.
 *   - <b>SCATTL RMS</b> (ADU): The robustly-estimated RMS of the
 *     scattered light level on the detector outside the region
 *     illuminated by the fibres.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_scattered_qc (
    cpl_image *in_image,
    cpl_propertylist *in_hdr,
    cpl_mask *in_mask,
    cpl_propertylist *qclist)
{
    cpl_image *tmp_image = NULL;
    cpl_mask *binned_mask = NULL;

    cpl_image *scat_image;
    float scatmed, scatsig;

    int specbin,spatbin,isbinned,blkfac[2];
    long naxis[2],naxis_mask[2];

#undef TIDY
#define TIDY                                    \
    if(tmp_image != NULL) {                     \
        cpl_image_delete(tmp_image);            \
        tmp_image = NULL;                       \
    }                                           \
    if(binned_mask != NULL) {                   \
        cpl_mask_delete(binned_mask);           \
        binned_mask = NULL;                     \
    }

    /* Get size of image */
    naxis[0] = cpl_image_get_size_x(in_image);
    naxis[1] = cpl_image_get_size_y(in_image);

    /* Check if binned */
    qmost_isbinned(in_hdr,&specbin,&spatbin,&isbinned);

    if(in_mask) {
        /* Check size */
        naxis_mask[0] = cpl_mask_get_size_x(in_mask);
        naxis_mask[1] = cpl_mask_get_size_y(in_mask);
        
        if (naxis_mask[0] != naxis[0]*spatbin ||
            naxis_mask[1] != naxis[1]*specbin) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "image (%ld,%ld), mask (%ld,%ld) "
                                         "sizes don't match",
                                         naxis[0]*spatbin,naxis[1]*specbin,
                                         naxis_mask[0],naxis_mask[1]);
        }
        
        /* Bin down to match data */
        blkfac[0] = spatbin;
        blkfac[1] = specbin;
        
        binned_mask = qmost_maskblk(in_mask, blkfac);
        if(binned_mask == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "binning (%d,%d) of mask "
                                         "failed",
                                         blkfac[0], blkfac[1]);
        }

        /* Duplicate input image */
        tmp_image = cpl_image_duplicate(in_image);
        if(tmp_image == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to duplicate input image");
        }

        if(cpl_mask_or(cpl_image_get_bpm(tmp_image),
                       binned_mask) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't combine masks");
        }

        cpl_mask_delete(binned_mask);
        binned_mask = NULL;

        scat_image = tmp_image;
    }
    else {
        scat_image = in_image;
    }

    /* Compute QC statistics and write to header. */
    if(qmost_skylevel_image(scat_image,
                            MINHISTVAL, MAXHISTVAL,
                            -FLT_MAX, 3, 3,
                            &scatmed, &scatsig) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't compute scattered "
                                     "light level stats on image");
    }

    cpl_propertylist_update_float(qclist,
                                  "ESO QC SCATTL MED",
                                  scatmed);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC SCATTL MED",
                                 "[ADU] Median scattered light");
    
    cpl_propertylist_update_float(qclist,
                                  "ESO QC SCATTL RMS",
                                  scatsig);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC SCATTL RMS",
                                 "[ADU] Robust RMS scattered light");

    /* Done with the mask and temporary image */
    scat_image = NULL;
    if(tmp_image) {
        cpl_image_delete(tmp_image);
        tmp_image = NULL;
    }

    return CPL_ERROR_NONE;
}

/**@}*/

/*

$Log$
Revision 1.2  2019/02/25 10:46:18  jrl
New memory allocation scheme,

Revision 1.1  2018/06/27 09:58:26  jim
new entry

*/
