/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_dfs.h"
#include "qmost_stats.h"
#include "qmost_utils.h"
#include "qmost_waveinfo.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_waveinfo  qmost_waveinfo
 *
 * Wavelength solution table utilities
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Output table definition */

#define SPECNUM_COL   "specnum"
#define FIBLIVE_COL   "fiblive"
#define NORD_COL      "nord"
#define COEFS_COL     "coefs"
#define XREF_COL      "xref"
#define NGOOD_COL     "ngood"
#define MRESID_COL    "medresid"
#define FITRMS_COL    "fit_rms"
#define WAVE1_COL     "wave1"
#define WAVEN_COL     "waven"
#define DWAVE_COL     "dwave"
#define NLINES_COL    "nlines"
#define XPOS_COL      "xpos"
#define FWHM_COL      "fwhm"
#define WCALC_COL     "wave_calc"
#define WTRUE_COL     "wave_true"
#define FFLAG_COL     "fit_flag"
#define WCOR_COL      "wave_cor"
#define PEAK_COL      "peak"
#define CONTRAST_COL  "contrast"

static int wv_ncol = 20;

/* This gives the number of mandatory columns (all except peak and
 * contrast, which were later additions) for wvchk */
static int wv_mincol =  18;

static char *wv_ttype[] = {
    SPECNUM_COL, FIBLIVE_COL, NORD_COL,
    COEFS_COL, XREF_COL, NGOOD_COL,
    MRESID_COL, FITRMS_COL, WAVE1_COL,
    WAVEN_COL, DWAVE_COL, NLINES_COL,
    XPOS_COL, FWHM_COL, WCALC_COL,
    WTRUE_COL, FFLAG_COL, WCOR_COL,
    PEAK_COL, CONTRAST_COL };

static char *wv_tunit[] = {
    "","","",
    "","pix","",
    "angstrom","angstrom","angstrom",
    "angstrom","angstrom/pix","",
    "pix","angstrom","angstrom",
    "angstrom","","angstrom",
    "ADU","" };

static cpl_type wv_cpltype[] = {
    CPL_TYPE_INT, CPL_TYPE_INT, CPL_TYPE_INT,
    CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE, CPL_TYPE_INT,
    CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE,
    CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE, CPL_TYPE_INT,
    CPL_TYPE_DOUBLE, CPL_TYPE_FLOAT, CPL_TYPE_DOUBLE,
    CPL_TYPE_DOUBLE, CPL_TYPE_INT, CPL_TYPE_DOUBLE,
    CPL_TYPE_FLOAT, CPL_TYPE_FLOAT };

/*----------------------------------------------------------------------------*/
/**
 * @brief   Create a wavelength solution table.
 *
 * @param   tptr       (Returned) The created cpl_table.
 * @param   nord       (Given)    The order of the polynomial to be
 *                                used for the wavelength solution,
 *                                for sizing the table column
 *                                containing the coefficients.
 * @param   nlinesmax  (Given)    Maximum number of lines detected in
 *                                any fibre, for sizing the table
 *                                columns containing the individual
 *                                line measurements.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   nord or nlinesmax are negative.
 * @retval  CPL_ERROR_NULL_INPUT      The required output pointer tptr
 *                                    was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_wvcreate (
    cpl_table **tptr,
    int nord,
    int nlinesmax)
{
    int icol;

    int wv_array[] = { 0, 0, 0,
                       1, 0, 0,
                       0, 0, 0,
                       0, 0, 0,
                       1, 1, 1,
                       1, 1, 1,
                       1, 1 };

    cpl_size wv_depth[] = { 1, 1, 1,
                            nord+1, 1, 1,
                            1, 1, 1,
                            nlinesmax, nlinesmax, nlinesmax,
                            nlinesmax, nlinesmax, nlinesmax,
                            nlinesmax, nlinesmax, nlinesmax,
                            nlinesmax, nlinesmax };

    /* Check inputs and output pointer */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nord >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(nlinesmax >= 0, CPL_ERROR_ILLEGAL_INPUT);

    /* Create the table */
    *tptr = cpl_table_new(0);
    if(!(*tptr)) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create table");
    }

    for(icol = 0; icol < wv_ncol; icol++) {
        if(wv_array[icol]) {
            if(cpl_table_new_column_array(*tptr,
                                          wv_ttype[icol],
                                          wv_cpltype[icol],
                                          wv_depth[icol]) != CPL_ERROR_NONE) {
                cpl_table_delete(*tptr);
                *tptr = NULL;

                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create array column "
                                             "%d = %s with length %lld",
                                             icol+1,
                                             wv_ttype[icol],
                                             wv_depth[icol]);
            }
        }
        else {
            if(cpl_table_new_column(*tptr,
                                    wv_ttype[icol],
                                    wv_cpltype[icol]) != CPL_ERROR_NONE) {
                cpl_table_delete(*tptr);
                *tptr = NULL;

                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create column "
                                             "%d = %s",
                                             icol+1, wv_ttype[icol]);
            }
        }

        if(cpl_table_set_column_unit(*tptr,
                                     wv_ttype[icol],
                                     wv_tunit[icol]) != CPL_ERROR_NONE) {
            cpl_table_delete(*tptr);
            *tptr = NULL;

            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set units for column "
                                         "%d = %s to %s",
                                         icol+1, wv_ttype[icol],
                                         wv_tunit[icol]);
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Open and read a QMOST wave info file.
 *
 * @param   tptr       (Given)    cpl_table to read from.
 * @param   nwv        (Returned) The number of wavelength solutions
 *                                read from the file.
 * @param   wv         (Returned) The array of waveinfo structures
 *                                with the wavelength solutions.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE   If the length of an array
 *                                          specified in the table is
 *                                          larger than the actual
 *                                          length of the array
 *                                          column.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the table is not a true
 *                                    waveinfo table.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      The required output pointer tptr
 *                                    was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If a column data type of an
 *                                    integer column isn't correct.
 *
 * @note    The returned waveinfo array should be deallocated with
 *          qmost_wvclose.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_wvopen(
    cpl_table *tptr,
    int *nwv,
    qmost_waveinfo **wv)
{
    cpl_size nrows;
    int i;

    /* Check required inputs and outputs */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nwv, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(wv, CPL_ERROR_NULL_INPUT);

    /* Initialise a few things */

    *nwv = 0;
    *wv = NULL;

#undef TIDY
#define TIDY                                    \
    if(*wv) {                                   \
        qmost_wvclose(*nwv, wv);                \
        *nwv = 0;                               \
        *wv = NULL;                             \
    }

    /* Check to make sure this conforms to the wave file design */

    if (qmost_wvchk(tptr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "waveinfo table conformity check failed");
    }

    /* OK, we're happy now. Get the space for the output and clear it
       to ensure the internal pointers are set to NULL. */

    nrows = cpl_table_get_nrow(tptr);
    if(nrows < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of rows");
    }

    *nwv = nrows;
    *wv = cpl_calloc(nrows, sizeof(qmost_waveinfo));

    /* Loop through the table and read all the info. */

    for (i = 0; i < *nwv; i++) {
        if (qmost_wvread1(tptr,i+1,(*wv+i)) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read row %d", i+1);
        }   
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read a single row out of the waveinfo table.
 * 
 * @param   tptr       (Given)    cpl_table to read from.
 * @param   row        (Given)    The row of the table to be read,
 *                                numbering from 1 (FITS convention).
 * @param   wv         (Returned) The output waveinfo structure for
 *                                the row that was read.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE   The row number was outside
 *                                          the table, or if the
 *                                          length of an array
 *                                          specified in the table is
 *                                          larger than the actual
 *                                          length of the array
 *                                          column.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If a table column was
 *                                    missing.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      A required input or output
 *                                    pointer was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If a column data type of an
 *                                    integer column isn't correct.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_wvread1(
    cpl_table *tptr,
    int row,
    qmost_waveinfo *wv)
{
    cpl_errorstate prestate;
    int iline;

    /* Check required inputs and outputs */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(row > 0, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(wv, CPL_ERROR_NULL_INPUT);

    memset(wv, 0, sizeof(qmost_waveinfo));

#undef TIDY
#define TIDY                                    \
    if(wv->coefs) {                             \
        cpl_polynomial_delete(wv->coefs);       \
        wv->coefs = NULL;                       \
    }                                           \
    if(wv->xpos) {                              \
        cpl_free(wv->xpos);                     \
        wv->xpos = NULL;                        \
    }                                           \
    if(wv->fwhm) {                              \
        cpl_free(wv->fwhm);                     \
        wv->fwhm = NULL;                        \
    }                                           \
    if(wv->wave_calc) {                         \
        cpl_free(wv->wave_calc);                \
        wv->wave_calc = NULL;                   \
    }                                           \
    if(wv->wave_true) {                         \
        cpl_free(wv->wave_true);                \
        wv->wave_true = NULL;                   \
    }                                           \
    if(wv->fit_flag) {                          \
        cpl_free(wv->fit_flag);                 \
        wv->fit_flag = NULL;                    \
    }                                           \
    if(wv->wave_cor) {                          \
        cpl_free(wv->wave_cor);                 \
        wv->wave_cor = NULL;                    \
    }                                           \
    if(wv->peak) {                              \
        cpl_free(wv->peak);                     \
        wv->peak = NULL;                        \
    }                                           \
    if(wv->contrast) {                          \
        cpl_free(wv->contrast);                 \
        wv->contrast = NULL;                    \
    }

    /* Read the spectrum number and see if this is a live fibre */
    prestate = cpl_errorstate_get();

    wv->specnum = cpl_table_get_int(tptr, SPECNUM_COL, row-1, NULL);
    wv->live = cpl_table_get_int(tptr, FIBLIVE_COL, row-1, NULL);
    wv->ngood = cpl_table_get_int(tptr, NGOOD_COL, row-1, NULL);
    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read %s/%s/%s columns "
                                     "for row %d",
                                     SPECNUM_COL,
                                     FIBLIVE_COL,
                                     NGOOD_COL,
                                     row);
    }

    /* If this fibre is dead then zero off pretty much everything */

    if (! wv->live) {
        wv->nord = 0;
        wv->coefs = NULL;
        wv->xref = 0.0;
        wv->ngood = 0;
        wv->medresid = 0.0;
        wv->fit_rms = 0.0;
        wv->wave1 = 0.0;
        wv->waven = 0.0;
        wv->dwave = 0.0;
        wv->nlines = 0;
        wv->xpos = NULL;
        wv->fwhm = NULL;
        wv->wave_calc = NULL;
        wv->wave_true = NULL;
        wv->fit_flag = NULL;
        wv->wave_cor = NULL;
        wv->peak = NULL;
        wv->contrast = NULL;
        return CPL_ERROR_NONE;
    }

    /* Ok, this is a live fibre. So let's get the rest of the info */

    wv->nord = cpl_table_get_int(tptr, NORD_COL, row-1, NULL);
    wv->nlines = cpl_table_get_int(tptr, NLINES_COL, row-1, NULL);
    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read %s/%s columns "
                                     "for row %d",
                                     NORD_COL,
                                     NLINES_COL,
                                     row);
    }

    /* Get coefficients */
    wv->coefs = qmost_cpl_table_get_polynomial(tptr, COEFS_COL, row-1,
                                               wv->nord);
    if(wv->coefs == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read %s column "
                                     "for row %d",
                                     COEFS_COL,
                                     row);
    }

    /* Read next block of scalars */
    prestate = cpl_errorstate_get();

    wv->xref = cpl_table_get(tptr, XREF_COL, row-1, NULL);
    wv->ngood = cpl_table_get_int(tptr, NGOOD_COL, row-1, NULL);
    wv->medresid = cpl_table_get(tptr, MRESID_COL, row-1, NULL);
    wv->fit_rms = cpl_table_get(tptr, FITRMS_COL, row-1, NULL);
    wv->wave1 = cpl_table_get(tptr, WAVE1_COL, row-1, NULL);
    wv->waven = cpl_table_get(tptr, WAVEN_COL, row-1, NULL);
    wv->dwave = cpl_table_get(tptr, DWAVE_COL, row-1, NULL);
    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read waveinfo columns "
                                     "for row %d",
                                     row);
    }

    /* Read remaining vectors */
    if(wv->nlines > 0) {
        wv->xpos = cpl_malloc(wv->nlines*sizeof(double));
        wv->fwhm = cpl_malloc(wv->nlines*sizeof(float));
        wv->wave_calc = cpl_malloc(wv->nlines*sizeof(double));
        wv->wave_true = cpl_malloc(wv->nlines*sizeof(double));
        wv->fit_flag = cpl_calloc(wv->nlines, sizeof(unsigned char));
        wv->wave_cor = cpl_malloc(wv->nlines*sizeof(double));
        wv->peak = cpl_malloc(wv->nlines*sizeof(float));
        wv->contrast = cpl_malloc(wv->nlines*sizeof(float));

        if(qmost_cpl_table_get_double_array(tptr, XPOS_COL, row-1,
                                            wv->xpos,
                                            wv->nlines) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         XPOS_COL,
                                         row);
        }

        if(qmost_cpl_table_get_float_array(tptr, FWHM_COL, row-1,
                                           wv->fwhm,
                                           wv->nlines) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         FWHM_COL,
                                         row);
        }

        if(qmost_cpl_table_get_double_array(tptr, WCALC_COL, row-1,
                                            wv->wave_calc,
                                            wv->nlines) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         WCALC_COL,
                                         row);
        }

        if(qmost_cpl_table_get_double_array(tptr, WTRUE_COL, row-1,
                                            wv->wave_true,
                                            wv->nlines) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         WTRUE_COL,
                                         row);
        }

        if(qmost_cpl_table_get_byte_array(tptr, FFLAG_COL, row-1,
                                          wv->fit_flag,
                                          wv->nlines) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         FFLAG_COL,
                                         row);
        }

        if(qmost_cpl_table_get_double_array(tptr, WCOR_COL, row-1,
                                            wv->wave_cor,
                                            wv->nlines) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         WCOR_COL,
                                         row);
        }

        /* The older wave format didn't have peak, so we need to
         * check if it exists and substitute dummy values if not */
        if(cpl_table_has_column(tptr, PEAK_COL)) {
            if(qmost_cpl_table_get_float_array(tptr,
                                               PEAK_COL,
                                               row-1,
                                               wv->peak,
                                               wv->nlines) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             PEAK_COL,
                                             row);
            }
        }
        else {
            for(iline = 0; iline < wv->nlines; iline++) {
                wv->peak[iline] = NAN;
            }
        }

        /* The older wave format didn't have contrast, so we need to
         * check if it exists and substitute dummy values if not */
        if(cpl_table_has_column(tptr, CONTRAST_COL)) {
            if(qmost_cpl_table_get_float_array(tptr,
                                               CONTRAST_COL,
                                               row-1,
                                               wv->contrast,
                                               wv->nlines) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             CONTRAST_COL,
                                             row);
            }
        }
        else {
            for(iline = 0; iline < wv->nlines; iline++) {
                wv->contrast[iline] = NAN;
            }
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Write a single row into an existing waveinfo table.
 * 
 * @param   tptr       (Given)    cpl_table to write to.
 * @param   row        (Given)    The row of the table to be written,
 *                                numbering from 1 (FITS convention).
 * @param   wv         (Given)    The waveinfo structure containing
 *                                the information to write.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row number was less
 *                                         than 1 or if the length of
 *                                         an array exceeds the length
 *                                         of the array column in the
 *                                         table.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If a table column was
 *                                    missing.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      A required input pointer was
 *                                    NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If a column data type of an
 *                                    integer column isn't correct.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_wvwrite1(
    cpl_table *tptr,
    int row,
    qmost_waveinfo wv)
{
    cpl_size nrows;

    /* Check required inputs */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(row > 0, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* If the row is off the end of the table, extend the table */
    
    nrows = cpl_table_get_nrow(tptr);
    if(nrows < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of rows");
    }

    if(row > nrows) {
        if(qmost_cpl_table_extend_blank(tptr, row) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not extend table to %d rows",
                                         row);
        }
    }

    /* Do the spectrum number and live */

    if(cpl_table_set_int(tptr,
                         SPECNUM_COL,
                         row-1,
                         wv.specnum) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to write %s for row %d",
                                     SPECNUM_COL, row);
    }

    if(cpl_table_set_int(tptr,
                         FIBLIVE_COL,
                         row-1,
                         wv.live) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to write %s for row %d",
                                     FIBLIVE_COL, row);
    }

    /* If this is a dead fibre then fill the rest of the columns with
       the appropriate null values */

    if (! wv.live || wv.nord == -1) {
        if(cpl_table_set_int(tptr,
                             NORD_COL,
                             row-1,
                             -1) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NORD_COL, row);
        }

        if(cpl_table_set(tptr,
                         XREF_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         XREF_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             NGOOD_COL,
                             row-1,
                             -1) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NGOOD_COL, row);
        }

        if(cpl_table_set(tptr,
                         MRESID_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         MRESID_COL, row);
        }

        if(cpl_table_set(tptr,
                         FITRMS_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         FITRMS_COL, row);
        }

        if(cpl_table_set(tptr,
                         WAVE1_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         WAVE1_COL, row);
        }

        if(cpl_table_set(tptr,
                         WAVEN_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         WAVEN_COL, row);
        }

        if(cpl_table_set(tptr,
                         DWAVE_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         DWAVE_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             NLINES_COL,
                             row-1,
                             -1) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NLINES_COL, row);
        }
    } else {
        /* If it's a live fibre, then write out the information */

        if(cpl_table_set_int(tptr,
                             NORD_COL,
                             row-1,
                             wv.nord) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NORD_COL, row);
        }

        if(qmost_cpl_table_set_polynomial(tptr,
                                          COEFS_COL,
                                          row-1,
                                          wv.coefs,
                                          wv.nord) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         COEFS_COL, row);
        }

        if(cpl_table_set(tptr,
                         XREF_COL,
                         row-1,
                         wv.xref) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         XREF_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             NGOOD_COL,
                             row-1,
                             wv.ngood) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NGOOD_COL, row);
        }

        if(cpl_table_set(tptr,
                         MRESID_COL,
                         row-1,
                         wv.medresid) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         MRESID_COL, row);
        }

        if(cpl_table_set(tptr,
                         FITRMS_COL,
                         row-1,
                         wv.fit_rms) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         FITRMS_COL, row);
        }

        if(cpl_table_set(tptr,
                         WAVE1_COL,
                         row-1,
                         wv.wave1) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         WAVE1_COL, row);
        }

        if(cpl_table_set(tptr,
                         WAVEN_COL,
                         row-1,
                         wv.waven) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         WAVEN_COL, row);
        }

        if(cpl_table_set(tptr,
                         DWAVE_COL,
                         row-1,
                         wv.dwave) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         DWAVE_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             NLINES_COL,
                             row-1,
                             wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NLINES_COL, row);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            XPOS_COL,
                                            row-1,
                                            wv.xpos,
                                            wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         XPOS_COL, row);
        }

        if(qmost_cpl_table_set_float_array(tptr,
                                           FWHM_COL,
                                           row-1,
                                           wv.fwhm,
                                           wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         FWHM_COL, row);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            WCALC_COL,
                                            row-1,
                                            wv.wave_calc,
                                            wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         WCALC_COL, row);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            WTRUE_COL,
                                            row-1,
                                            wv.wave_true,
                                            wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         WTRUE_COL, row);
        }

        if(qmost_cpl_table_set_byte_array(tptr,
                                          FFLAG_COL,
                                          row-1,
                                          wv.fit_flag,
                                          wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         FFLAG_COL, row);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            WCOR_COL,
                                            row-1,
                                            wv.wave_cor,
                                            wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         WCOR_COL, row);
        }

        if(qmost_cpl_table_set_float_array(tptr,
                                           PEAK_COL,
                                           row-1,
                                           wv.peak,
                                           wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         PEAK_COL, row);
        }

        if(qmost_cpl_table_set_float_array(tptr,
                                           CONTRAST_COL,
                                           row-1,
                                           wv.contrast,
                                           wv.nlines) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         CONTRAST_COL, row);
        }
    }  

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Check an open waveinfo table to see if it conforms to the
 *          standard.
 * 
 * @param   tptr       (Given)    cpl_table to check.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the table doesn't conform to
 *                                    the standard for waveinfo
 *                                    tables.
 * @retval  CPL_ERROR_NULL_INPUT      A required input pointer was
 *                                    NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_wvchk(
    cpl_table *tptr)
{
    int i, ncols;
    cpl_array *colnames;
    const char *colname;

    /* Check required input */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);

    /* Does it have the right number of columns and the right column names */

    ncols = cpl_table_get_ncol(tptr);
    if(ncols < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of columns");
    }
    else if(ncols < wv_mincol) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "input file has %d columns. "
                                     "Should have at least %d",
                                     ncols, wv_mincol);
    }

    colnames = cpl_table_get_column_names(tptr);
    if(!colnames) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get column names");
    }

    for (i = 0; i < wv_mincol; i++) {
        colname = cpl_array_get_string(colnames, i);
        if(colname == NULL) {
            cpl_array_delete(colnames);
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get column %d name",
                                         i+1);
        }

        if (strcmp(colname,wv_ttype[i]) != 0) {
            cpl_array_delete(colnames);
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ILLEGAL_INPUT,
                                         "input file column %d name "
                                         "not standard",
                                         i+1);
        }
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate the mean dispersion of the entries in a waveinfo
 *          table.
 * 
 * @param   nwv        (Given)    The number of waveinfo structures in
 *                                the list.
 * @param   wv         (Given)    The list of waveinfo structures.
 *
 * @return  double                The calculated mean value of the
 *                                dispersion.
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

double qmost_wvmeandisp(
    int nwv,
    qmost_waveinfo *wv)
{
    int i,ngood;
    double sum;

    /* Work out the mean dispersion of the good fibres */

    sum = 0.0;
    ngood = 0;
    for (i = 0; i < nwv; i++) {
        if (wv[i].live) {
            sum += wv[i].dwave;
            ngood++;
        }
    }
    if (ngood > 0) {
        sum /= (double)ngood;
        return(sum);
    } else {
        return(0.0);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Calculate the min/max wavelength endpoints for a set of
 *          wavelength solutions.
 * 
 * @param   nwv        (Given)    The number of waveinfo structures in
 *                                the list.
 * @param   wv         (Given)    The list of waveinfo structures.
 * @param   minwave1   (Returned) The ensemble minimum value of the
 *                                lower endpoint.
 * @param   maxwave1   (Returned) The ensemble maximum value of the
 *                                lower endpoint.
 * @param   minwaven   (Returned) The ensemble minimum value of the
 *                                upper endpoint.
 * @param   maxwaven   (Returned) The ensemble maximum value of the
 *                                upper endpoint.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_wvendpoints(
    int nwv,
    qmost_waveinfo *wv,
    double *minwave1,
    double *maxwave1,
    double *minwaven, 
    double *maxwaven)
{
    int i;

    /* Loop through the fibres and find the endpoints */

    *minwave1 = 1.0e10;
    *maxwave1 = -1.0e10;
    *minwaven = 1.0e10;
    *maxwaven = -1.0e10;
    for (i = 0; i < nwv; i++) {
        if (wv[i].live) {
            *minwave1 = qmost_min(*minwave1,wv[i].wave1);
            *maxwave1 = qmost_max(*maxwave1,wv[i].wave1);
            *minwaven = qmost_min(*minwaven,wv[i].waven);
            *maxwaven = qmost_max(*maxwaven,wv[i].waven);
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Populate QC headers for wavelength solution.
 *
 * @param   wave_tbl     (Given)    The waveinfo table.
 * @param   ref_wave_tbl (Given)    An optional reference waveinfo
 *                                  table to compare with, or NULL if
 *                                  none.
 * @param   obsol        (Given)    If true, the waveinfo table is an
 *                                  OB-level wavelength correction.
 *                                  If false, the waveinfo table is a
 *                                  master wavelength solution.
 * @param   qclist       (Modified) The output FITS header to receive
 *                                  the QC parameters.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required inputs or
 *                                    outputs is NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the input spectrum data type
 *                                    was not float, or if a column
 *                                    data type of an integer column
 *                                    isn't correct.
 *
 * @par Output QC Parameters:
 *   - <b>WAVE FWHM MED</b> (A): Median FWHM of the arc lines.
 *   - <b>WAVE FWHM RMS</b> (A): Robustly-estimated RMS of the FWHM of
 *     the the arc lines.
 *   - <b>WAVE SPAN MAX</b>: The maximum over all the fibres of the
 *     fraction of the spectral axis spanned by the lines used in the
 *     wavelength solution.
 *   - <b>WAVE SPAN MAXSPC</b>: The fibre with the maximum fraction of the
 *     spectral axis spanned.
 *   - <b>WAVE SPAN MED</b>: The median over all the fibres of the
 *     fraction of the spectral axis spanned by the lines used in the
 *     wavelength solution.
 *   - <b>WAVE SPAN MIN</b>: The minimum over all the fibres of the
 *     fraction of the spectral axis spanned by the lines used in the
 *     wavelength solution.
 *   - <b>WAVE SPAN MINSPC</b>: The fibre with the minimum fraction of
 *     the spectral axis spanned.
 *   - <b>WAVE SPAN RMS</b>: The robustly-estimated RMS over all the
 *     fibres of the fraction of the spectral axis spanned by the
 *     lines used in the wavelength solution.
 *   - <b>WAVE LINES MAX</b>: The maximum number of arc lines measured
 *     in a fibre.
 *   - <b>WAVE LINES MAXSPC</b>: The fibre with the maximum number of
 *     arc lines.
 *   - <b>WAVE LINES MED</b>: The median number of arc lines measured
 *     in a fibre.
 *   - <b>WAVE LINES MIN</b>: The minimum number of arc lines measured
 *     in a fibre.
 *   - <b>WAVE LINES MINSPC</b>: The fibre with the minimum number of
 *     arc lines.
 *   - <b>WAVE LINES TOT</b>: The total number of arc lines measured
 *     across all fibres.
 *   - <b>WAVE OFFSET MAX</b> (A): The maximum wavelength offset of
 *     the new wavelength solution relative to the reference over all
 *     of the fibres.
 *   - <b>WAVE OFFSET MAXSPC</b>: The fibre with the maximum
 *     wavelength offset.
 *   - <b>WAVE OFFSET MED</b> (A): The median wavelength offset of the
 *     new wavelength solution relative to the reference over all of
 *     the fibres.
 *   - <b>WAVE OFFSET MIN</b> (A): The minimum wavelength offset of the
 *     new wavelength solution relative to the reference over all of
 *     the fibres.
 *   - <b>WAVE OFFSET MINSPC</b>: The fibre with the minimum
 *     wavelength offset.
 *   - <b>WAVE OFFSET RMS</b> (A): The RMS of the wavelength offset of
 *     the new wavelength solution relative to the reference over the
 *     fibres, as a measure of how consistent the wavelength offset is
 *     from fibre to fibre.
 *   - <b>WAVE RMS MAX</b> (A): The maximum RMS error in the arc fit
 *     in a fibre.
 *   - <b>WAVE RMS MAXSPC</b>: The fibre with the maximum RMS error.
 *   - <b>WAVE RMS MED</b> (A): The median RMS error in the arc fit in
 *     a fibre.
 *   - <b>WAVE RMS MIN</b> (A): The minimum RMS error in the arc fit
 *     in a fibre.
 *   - <b>WAVE RMS MINSPC</b>: The fibre with the minimum RMS error.
 *
 * @author  Jim Lewis, CASU
 * @author  Mike Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_wave_qc (
    cpl_table *wave_tbl,
    cpl_table *ref_wave_tbl,
    int obsol,
    cpl_propertylist *qclist)
{
    int *live = NULL;
    int *nlines = NULL;
    float *fbuf = NULL;
    float *lbuf = NULL;
    unsigned char *bpm = NULL;
    unsigned char *bpmrms = NULL;
    float *wspan = NULL;
    double *fitrms = NULL;
    int isnull;

    int totlines,goodlines,nfwhm,i,minlines,spec_minlines,ngood;
    int maxlines,spec_maxlines,spec_minrms,spec_maxrms,mednlines;
    int n,nbad,nbadrms;
    cpl_size nrows;
    float fwhm_med,fwhm_sigmad,val;
    double minrms,maxrms,medrms;

    float minwspan, maxwspan, medwspan, sigwspan;
    int iminwspan, imaxwspan, nwspan;
    float wave1, waven, wavel, waveh;
    int j, anygood;

    qmost_waveinfo *wv = NULL;
    qmost_waveinfo *wvref = NULL;
    int nwv = 0, nwvref = 0, nwvcomp;

    cpl_array *dllist = NULL;
    cpl_array *fiblist = NULL;
    int iwv, ndl = 0;
    double lam, lamref, dl;
    double med_dl, mad_dl;
    cpl_size minspc, maxspc;

    cpl_ensure_code(wave_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(live) {                                  \
        cpl_free(live);                         \
        live = NULL;                            \
    }                                           \
    if(nlines) {                                \
        cpl_free(nlines);                       \
        nlines = NULL;                          \
    }                                           \
    if(fbuf) {                                  \
        cpl_free(fbuf);                         \
        fbuf = NULL;                            \
    }                                           \
    if(lbuf) {                                  \
        cpl_free(lbuf);                         \
        lbuf = NULL;                            \
    }                                           \
    if(bpm) {                                   \
        cpl_free(bpm);                          \
        bpm = NULL;                             \
    }                                           \
    if(bpmrms) {                                \
        cpl_free(bpmrms);                       \
        bpmrms = NULL;                          \
    }                                           \
    if(wspan) {                                 \
        cpl_free(wspan);                        \
        wspan = NULL;                           \
    }                                           \
    if(fitrms) {                                \
        cpl_free(fitrms);                       \
        fitrms = NULL;                          \
    }                                           \
    if(wv != NULL) {                            \
        qmost_wvclose(nwv, &wv);                \
        wv = NULL;                              \
    }                                           \
    if(wvref != NULL) {                         \
        qmost_wvclose(nwvref, &wvref);          \
        wvref = NULL;                           \
    }                                           \
    if(dllist != NULL) {                        \
        cpl_array_delete(dllist);               \
        dllist = NULL;                          \
    }                                           \
    if(fiblist != NULL) {                       \
        cpl_array_delete(fiblist);              \
        fiblist = NULL;                         \
    }

    /* Start by working out the total number of lines that we detected */

    nrows = cpl_table_get_nrow(wave_tbl);

    live = cpl_malloc(nrows*sizeof(int));
    nlines = cpl_malloc(nrows*sizeof(int));

    totlines = 0;

    for(i = 0; i < nrows; i++) {
        live[i] = cpl_table_get_int(wave_tbl, FIBLIVE_COL, i, &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         FIBLIVE_COL,
                                         i+1);
        }
        else if(isnull > 0) {
            live[i] = 0;
        }

        nlines[i] = cpl_table_get_int(wave_tbl, NLINES_COL, i, &isnull);
        if(isnull < 0) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for row %d",
                                         NLINES_COL,
                                         i+1);
        }
        else if(isnull > 0) {
            nlines[i] = 0;
        }

        if(live[i]) {
            totlines += nlines[i];
        }
    }

    /* Get the workspace for the bpm and the fwhm arrays */

    if (totlines > 0) {
        fbuf = cpl_malloc(totlines*sizeof(float));
        lbuf = cpl_malloc(totlines*sizeof(float));
        bpm = cpl_calloc(totlines, sizeof(unsigned char));

        wspan = cpl_malloc(nrows*sizeof(float));
        nwspan = 0;

        iminwspan = -1;
        minwspan = -1.0;
        imaxwspan = -1;
        maxwspan = -1.0;

        /* Read both of these now and count the number of good lines */

        n = 0;
        for (i = 0; i < nrows; i++) {
            if (! live[i] || nlines[i] == 0)
                continue;

            if(qmost_cpl_table_get_float_array(wave_tbl,
                                               FWHM_COL,
                                               i,
                                               fbuf+n,
                                               nlines[i]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             FWHM_COL,
                                             i+1);
            }

            if(qmost_cpl_table_get_float_array(wave_tbl,
                                               WCALC_COL,
                                               i,
                                               lbuf+n,
                                               nlines[i]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             FWHM_COL,
                                             i+1);
            }

            if(qmost_cpl_table_get_byte_array(wave_tbl,
                                              FFLAG_COL,
                                              i,
                                              bpm+n,
                                              nlines[i]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             FFLAG_COL,
                                             i+1);
            }

            /* Wavelength range spanned by good lines relative to total */
            wave1 = cpl_table_get(wave_tbl, WAVE1_COL, i, &isnull);
            if(isnull < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             WAVE1_COL,
                                             i+1);
            }
            else if(isnull > 0) {
                wave1 = 0;
            }

            waven = cpl_table_get(wave_tbl, WAVEN_COL, i, &isnull);
            if(isnull < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             WAVEN_COL,
                                             i+1);
            }
            else if(isnull > 0) {
                waven = 0;
            }

            anygood = 0;
            wavel = 0.0;
            waveh = 0.0;

            for(j = 0; j < nlines[i]; j++) {
                if(!bpm[n+j]) {
                    if(!anygood || lbuf[n+j] < wavel) {
                        wavel = lbuf[n+j];
                        anygood = 1;
                    }

                    if(!anygood || lbuf[n+j] > waveh) {
                        waveh = lbuf[n+j];
                        anygood = 1;
                    }
                }
            }

            if(anygood && (waven - wave1) != 0.0) {
                wspan[nwspan] = (waveh - wavel) / (waven - wave1);

                if(iminwspan < 0 || wspan[nwspan] < minwspan) {
                    minwspan = wspan[nwspan];
                    iminwspan = i;
                }
                if(imaxwspan < 0 || wspan[nwspan] > maxwspan) {
                    maxwspan = wspan[nwspan];
                    imaxwspan = i;
                }

                nwspan++;
            }

            n += nlines[i];
        }
        goodlines = 0;
        for (i = 0; i < totlines; i++)
            if (bpm[i] == 0)
                goodlines++;

        /* Flag any lines where we couldn't measure FWHM as bad so
         * they don't get included in the statistics.  The bpm array
         * isn't used for anything else so we can do it there without
         * causing a problem. */
        nfwhm = 0;
        for (i = 0; i < totlines; i++) {
            if(bpm[i] == 0) {
                if(isfinite(fbuf[i]) && fbuf[i] > 0) {
                    nfwhm++;
                }
                else {
                    bpm[i] = 1;
                }
            }
        }

        /* Right get the median and sigmad of the FWHM array (masking off all
           lines that weren't included in the wavelength solution. If there
           were no good lines, then create some dummy values */

        if (nfwhm > 0) {
            if(qmost_medmad(fbuf,bpm,totlines,
                            &fwhm_med,&fwhm_sigmad) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to compute medmad "
                                             "of FWHM");
            }
            fwhm_sigmad *= 1.48;
        } else {
            fwhm_med = -1.0;
            fwhm_sigmad = -1.0;
        }
        
        cpl_propertylist_update_int(qclist,
                                    "ESO QC WAVE LINES TOT",
                                    goodlines);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE LINES TOT",
                                     "Total num lines in all wavelegth solutions");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC WAVE FWHM MED",
                                      fwhm_med);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE FWHM MED",
                                     "[Angstrom] Median FWHM of all good arclines");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC WAVE FWHM RMS",
                                      fwhm_sigmad);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE FWHM RMS",
                                     "[Angstrom] Sigma FWHM of all good arclines");

        /* Median and 1.48*MAD wavelength span */
        if(nwspan > 0) {
            qmost_medmad(wspan, NULL, nwspan, &medwspan, &sigwspan);
            sigwspan *= CPL_MATH_STD_MAD;
        }
        else {
            medwspan = -1;
            sigwspan = -1;
        }

        cpl_free(wspan);
        wspan = NULL;

        cpl_propertylist_update_float(qclist,
                                      "ESO QC WAVE SPAN MED",
                                      medwspan);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE SPAN MED",
                                     "Median wave solution span");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC WAVE SPAN RMS",
                                      sigwspan);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE SPAN RMS",
                                     "Sigma wave solution span");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC WAVE SPAN MIN",
                                      minwspan);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE SPAN MIN",
                                     "Minimum wave solution span");

        cpl_propertylist_update_int(qclist,
                                    "ESO QC WAVE SPAN MINSPC",
                                    iminwspan+1);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE SPAN MINSPC",
                                     "Fibre with minimum wave solution span");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC WAVE SPAN MAX",
                                      maxwspan);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE SPAN MAX",
                                     "Maximum wave solution span");

        cpl_propertylist_update_int(qclist,
                                    "ESO QC WAVE SPAN MAXSPC",
                                    imaxwspan+1);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE SPAN MAXSPC",
                                     "Fibre with maximum wave solution span");

        cpl_free(fbuf);
        fbuf = NULL;

        cpl_free(bpm);
        bpm = NULL;

        /* Do some QC on the number of lines and on the fit RMS */

        fitrms = cpl_malloc(nrows*sizeof(double));
        fbuf = cpl_malloc(nrows*sizeof(float));
        bpm = cpl_calloc(nrows,sizeof(unsigned char));
        bpmrms = cpl_calloc(nrows,sizeof(unsigned char));

        minlines = 10000000;
        spec_minlines = -1;
        maxlines = -1000000;
        spec_maxlines = -1;
        minrms = 1.0e20;
        spec_minrms = -1;
        maxrms = -1.0e10;
        spec_maxrms = -1;
        nbad = 0;
        nbadrms = 0;
        for (i = 0; i < nrows; i++) {
            /* Read from table */
            ngood = cpl_table_get_int(wave_tbl, NGOOD_COL, i, &isnull);
            if(isnull < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             NGOOD_COL,
                                             i+1);
            }
            else if(isnull > 0) {
                ngood = -1;
            }

            fbuf[i] = (float)ngood;

            /* Accumulate statistics if live and not NULL */
            if(live[i] && ngood >= 0) {
                if (ngood < minlines) {
                    minlines = ngood;
                    spec_minlines = i + 1;
                }
                if (ngood > maxlines) {
                    maxlines = ngood;
                    spec_maxlines = i + 1;
                }
            }
            else {
                bpm[i] = 1;
                nbad++;
            }

            fitrms[i] = cpl_table_get(wave_tbl, FITRMS_COL, i, &isnull);
            if(isnull < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             FITRMS_COL,
                                             i+1);
            }
            else if(isnull > 0) {
                fitrms[i] = -1;
            }

            /* Accumulate statistics if live, has good lines, and not NULL */
            if(live[i] && ngood > 0 && fitrms[i] >= 0) {
                if (fitrms[i] < minrms) {
                    minrms = fitrms[i];
                    spec_minrms = i + 1;
                }
                if (fitrms[i] > maxrms) {
                    maxrms = fitrms[i];
                    spec_maxrms = i + 1;
                }
            }
            else {
                bpmrms[i] = 1;
                nbadrms++;
            }
        }

        if (nbad < nrows) {
            if(qmost_med(fbuf,bpm,nrows,&val) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to compute median "
                                             "number of lines");
            }
            mednlines = (int)val;
        }
        else {
            mednlines = -1;
        }

        if(nbadrms < nrows) {
            if(qmost_dmed(fitrms,bpmrms,nrows,&medrms) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to compute median "
                                             "fit RMS");
            }
        }
        else {
            medrms = -1.0;
        }

        /* Write the results to the header */

        if (spec_minlines == -1)
            minlines = -1;
        if (spec_maxlines == -1)
            maxlines = -1;

        cpl_free(fitrms);
        fitrms = NULL;

        cpl_free(fbuf);
        fbuf = NULL;

        cpl_free(lbuf);
        lbuf = NULL;

        cpl_free(bpm);
        bpm = NULL;

        cpl_free(bpmrms);
        bpmrms = NULL;
    } else {
        minlines = 0;
        spec_minlines = 0;
        maxlines = 0;
        spec_maxlines = 0;
        mednlines = 0;
        minrms = 0.0;
        spec_minrms = 0;
        maxrms = 0.0;
        spec_maxrms = 0;
        medrms = 0.0;
        goodlines = 0;
        fwhm_med = 0.0;
        fwhm_sigmad = 0.0;

        cpl_propertylist_update_int(qclist,
                                    "ESO QC WAVE LINES TOT",
                                    goodlines);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE LINES TOT",
                                     "Total num lines in all wavelegth solutions");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC WAVE FWHM MED",
                                      fwhm_med);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE FWHM MED",
                                     "[Angstrom] Median FWHM of all good arclines");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC WAVE FWHM RMS",
                                      fwhm_sigmad);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC WAVE FWHM RMS",
                                     "[Angstrom] Sigma FWHM of all good arclines");
    }

    cpl_propertylist_update_int(qclist,
                                "ESO QC WAVE LINES MIN",
                                minlines);
    cpl_propertylist_set_comment(qclist,
                                "ESO QC WAVE LINES MIN",
                                 "Minimum number of lines in a fit");

    cpl_propertylist_update_int(qclist,
                                "ESO QC WAVE LINES MINSPC",
                                spec_minlines);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE LINES MINSPC",
                                 "Spectrum with min lines in fit");

    cpl_propertylist_update_int(qclist,
                                "ESO QC WAVE LINES MAX",
                                maxlines);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE LINES MAX",
                                 "Maximum number of lines in a fit");

    cpl_propertylist_update_int(qclist,
                                "ESO QC WAVE LINES MAXSPC",
                                spec_maxlines);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE LINES MAXSPC",
                                 "Spectrum with max lines in fit");

    cpl_propertylist_update_int(qclist,
                                "ESO QC WAVE LINES MED",
                                mednlines);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE LINES MED",
                                 "Median number of lines of all fits");

    cpl_propertylist_update_double(qclist,
                                   "ESO QC WAVE RMS MIN",
                                   minrms);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE RMS MIN",
                                 "[Angstrom] Minimum rms of a fit");

    cpl_propertylist_update_int(qclist,
                                "ESO QC WAVE RMS MINSPC",
                                spec_minrms);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE RMS MINSPC",
                                 "Spectrum with min rms in fit");

    cpl_propertylist_update_double(qclist,
                                   "ESO QC WAVE RMS MAX",
                                   maxrms);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE RMS MAX",
                                 "[Angstrom] Maximum rms of a fit");

    cpl_propertylist_update_int(qclist,
                                "ESO QC WAVE RMS MAXSPC",
                                spec_maxrms);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE RMS MAXSPC",
                                 "Spectrum with max rms in fit");

    cpl_propertylist_update_double(qclist,
                                   "ESO QC WAVE RMS MED",
                                   medrms);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC WAVE RMS MED",
                                 "[Angstrom] Median rms of all fits");

    /* Read table for the next part */
    if(qmost_wvopen(wave_tbl, &nwv, &wv) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "problem reading wavelength "
                                     "solution table");
    }

    if(ref_wave_tbl != NULL) {
        /* If reference was given, do the comparison */
        if(qmost_wvopen(ref_wave_tbl, &nwvref, &wvref) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "problem reading reference "
                                         "wavelength solution table");
        }

        nwvcomp = qmost_min(nwv, nwvref);

        dllist = cpl_array_new(nwvcomp, CPL_TYPE_DOUBLE);
        fiblist = cpl_array_new(nwvcomp, CPL_TYPE_INT);
        if(dllist == NULL || fiblist == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to create dllist length %d",
                                         nwvcomp);

        }
        
        ndl = 0;
        
        for(iwv = 0; iwv < nwvcomp; iwv++) {
            /* Check if fibre is live */
            if(wv[iwv].live && wvref[iwv].live) {
                /* Compare wavelengths at reference spectral
                   coordinate of the reference solution. */
                lam = cpl_polynomial_eval_1d(wv[iwv].coefs,
                                             wvref[iwv].xref - wv[iwv].xref,
                                             NULL);
                lamref = cpl_polynomial_eval_1d(wvref[iwv].coefs, 0, NULL);

                dl = lam - lamref;

                if(cpl_array_set(dllist,
                                 ndl,
                                 dl) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "failed to set dllist[%d]",
                                                 ndl);
                }

                if(cpl_array_set(fiblist,
                                 ndl,
                                 iwv+1) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "failed to set fiblist[%d]",
                                                 ndl);
                }

                ndl++;
            }
        }

        qmost_wvclose(nwvref, &wvref);
        wvref = NULL;
    }
    else if(obsol) {
        /* For an OB solution, the waveinfo table gives the offset
         * directly, so use that in the QC headers. */
        dllist = cpl_array_new(nwv, CPL_TYPE_DOUBLE);
        fiblist = cpl_array_new(nwv, CPL_TYPE_INT);
        if(dllist == NULL || fiblist == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to create dllist length %d",
                                         nwv);

        }
        
        ndl = 0;
        
        for(iwv = 0; iwv < nwv; iwv++) {
            /* Check if fibre is live */
            if(wv[iwv].live) {
                /* Compare wavelengths at reference spectral
                   coordinate of the reference solution. */
                dl = cpl_polynomial_eval_1d(wv[iwv].coefs,
                                            0,
                                            NULL);

                if(cpl_array_set(dllist,
                                 ndl,
                                 dl) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "failed to set dllist[%d]",
                                                 ndl);
                }

                if(cpl_array_set(fiblist,
                                 ndl,
                                 iwv+1) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "failed to set dllist[%d]",
                                                 ndl);
                }

                ndl++;
            }
        }
    }

    qmost_wvclose(nwv, &wv);
    wv = NULL;

    if(dllist != NULL) {
        if(ndl > 0) {
            if(cpl_array_set_size(dllist, ndl) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set dllist size to %d",
                                             ndl);
            }

            if(cpl_array_set_size(fiblist, ndl) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set fiblist size to %d",
                                             ndl);
            }

            /* QC parameters */
            if(qmost_cpl_array_get_med_mad(dllist,
                                           &med_dl,
                                           &mad_dl) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to compute medmad of "
                                             "dllist");
            }

            cpl_propertylist_update_double(qclist,
                                           "ESO QC WAVE OFFSET MED",
                                           med_dl);
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC WAVE OFFSET MED",
                                         "[Angstrom] Median wavelength offset from reference");
            
            cpl_propertylist_update_double(qclist,
                                           "ESO QC WAVE OFFSET RMS",
                                           CPL_MATH_STD_MAD * mad_dl);
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC WAVE OFFSET RMS",
                                         "[Angstrom] RMS of wavelength offset from reference");
            
            cpl_propertylist_update_double(qclist,
                                           "ESO QC WAVE OFFSET MIN",
                                           cpl_array_get_min(dllist));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC WAVE OFFSET MIN",
                                         "[Angstrom] Minimum wavelength offset from reference");
            
            if(cpl_array_get_minpos(dllist, &minspc) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to find minimum of "
                                             "dllist");
            }
            
            cpl_propertylist_update_int(qclist,
                                        "ESO QC WAVE OFFSET MINSPC",
                                        cpl_array_get_int(fiblist, minspc, NULL));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC WAVE OFFSET MINSPC",
                                         "Fibre with minimum wavelength offset from reference");

            cpl_propertylist_update_double(qclist,
                                           "ESO QC WAVE OFFSET MAX",
                                           cpl_array_get_max(dllist));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC WAVE OFFSET MAX",
                                         "[Angstrom] Maximum wavelength offset from reference");

            if(cpl_array_get_maxpos(dllist, &maxspc) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to find maximum of "
                                             "dllist");
            }
            
            cpl_propertylist_update_int(qclist,
                                        "ESO QC WAVE OFFSET MAXSPC",
                                        cpl_array_get_int(fiblist, maxspc, NULL));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC WAVE OFFSET MAXSPC",
                                         "Fibre with maximum wavelength offset from reference");
        }

        /* Clean up */
        cpl_array_delete(dllist);
        dllist = NULL;

        cpl_array_delete(fiblist);
        fiblist = NULL;
    }

    /* Get out of here */

    cpl_free(live);
    live = NULL;

    cpl_free(nlines);
    nlines = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Close a list of QMOST waveinfo structures and free
 *          associated memory.
 * 
 * @param   nwv        (Given)    The number of waveinfo structures in
 *                                the list.
 * @param   wv         (Given)    The list of waveinfo structures.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_wvclose(
    int nwv,
    qmost_waveinfo **wv)
{
    int i;

    if(wv == NULL)
        return;

    for (i = 0; i < nwv; i++) {
        if((*wv)[i].coefs != NULL) {
            cpl_polynomial_delete((*wv)[i].coefs);
        }
        if((*wv)[i].xpos != NULL) {
            cpl_free((*wv)[i].xpos);
        }
        if((*wv)[i].fwhm != NULL) {
            cpl_free((*wv)[i].fwhm);
        }
        if((*wv)[i].wave_calc != NULL) {
            cpl_free((*wv)[i].wave_calc);
        }
        if((*wv)[i].wave_true != NULL) {
            cpl_free((*wv)[i].wave_true);
        }
        if((*wv)[i].fit_flag != NULL) {
            cpl_free((*wv)[i].fit_flag);
        }
        if((*wv)[i].wave_cor != NULL) {
            cpl_free((*wv)[i].wave_cor);
        }
        if((*wv)[i].peak != NULL) {
            cpl_free((*wv)[i].peak);
        }
        if((*wv)[i].contrast != NULL) {
            cpl_free((*wv)[i].contrast);
        }
    }
    cpl_free(*wv);
    *wv = NULL;
}

/**@}*/

/*

$Log$
Revision 1.13  2019/02/25 10:50:43  jrl
New memory allocation scheme

Revision 1.12  2018/11/07 13:50:33  jrl
moved table definitions from here to .h file

Revision 1.11  2018/10/25 06:12:57  jrl
modified write1 to deal with the situation where no good arclines were found

Revision 1.10  2018/09/19 11:29:16  jrl
Modified to replace TNULL values and to add wavelength correction column

Revision 1.9  2017/06/01 20:26:17  jim
Stats in QC now take into account whether the fibre is live or not

Revision 1.8  2017/05/22 11:22:44  jim
Fixed missing initialisation of the FWHM element

Revision 1.7  2017/03/14 11:34:41  jim
superficial changes

Revision 1.6  2017/01/17 09:03:09  jim
Added qc routine. Also added some qc to fibre table

Revision 1.5  2016/10/26 11:20:40  jim
Added docs

Revision 1.4  2016/10/03 14:54:08  jim
Fixed memory leak

Revision 1.3  2016/08/24 11:52:29  jim
*** empty log message ***

Revision 1.2  2016/07/11 15:02:45  jim
Removed diagnostic prints

Revision 1.1  2016/07/06 11:06:14  jim
New file


*/
