/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_constants_test  Unit test of qmost_constants
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_get_rebin_params
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_get_rebin_params(void)
{
    int spec, arm;
    double minwave, maxwave, dlam;
    cpl_error_code code;

    /* NULL pointers */
    code = qmost_get_rebin_params(QMOST_SPEC_HRS, QMOST_ARM_RED,
                                  NULL, &maxwave, &dlam);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_get_rebin_params(QMOST_SPEC_HRS, QMOST_ARM_RED,
                                  &minwave, NULL, &dlam);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_get_rebin_params(QMOST_SPEC_HRS, QMOST_ARM_RED,
                                  &minwave, &maxwave, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid inputs */
    code = qmost_get_rebin_params(0, QMOST_ARM_RED,
                                  &minwave, &maxwave, &dlam);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    code = qmost_get_rebin_params(QMOST_NUM_SPEC + 1, QMOST_ARM_RED,
                                  &minwave, &maxwave, &dlam);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    code = qmost_get_rebin_params(QMOST_SPEC_HRS, 0,
                                  &minwave, &maxwave, &dlam);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    code = qmost_get_rebin_params(QMOST_SPEC_HRS, QMOST_NUM_ARMS + 1,
                                  &minwave, &maxwave, &dlam);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* These should all work.  The test that something was written is
       a bit rudimentary to avoid having the test depend on the
       specific values, instead we initialize with NaN and check the
       value isn't NaN after the call. */
    for(spec = 1; spec <= QMOST_NUM_SPEC; spec++) {
        for(arm = 1; arm <= QMOST_NUM_ARMS; arm++) {
            minwave = NAN;
            maxwave = NAN;
            dlam = NAN;

            code = qmost_get_rebin_params(spec, arm,
                                          &minwave, &maxwave, &dlam);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            cpl_test(!isnan(minwave));
            cpl_test(!isnan(maxwave));
            cpl_test(!isnan(dlam));
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_get_skysubwave
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_get_skysubwave(void)
{
    int spec, arm;
    double minwave, maxwave;
    cpl_error_code code;

    /* NULL pointers */
    code = qmost_get_skysubwave(QMOST_SPEC_HRS, QMOST_ARM_RED,
                                NULL, &maxwave);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_get_skysubwave(QMOST_SPEC_HRS, QMOST_ARM_RED,
                                &minwave, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid inputs */
    code = qmost_get_skysubwave(0, QMOST_ARM_RED,
                                &minwave, &maxwave);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    code = qmost_get_skysubwave(QMOST_NUM_SPEC + 1, QMOST_ARM_RED,
                                &minwave, &maxwave);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    code = qmost_get_skysubwave(QMOST_SPEC_HRS, 0,
                                &minwave, &maxwave);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    code = qmost_get_skysubwave(QMOST_SPEC_HRS, QMOST_NUM_ARMS + 1,
                                &minwave, &maxwave);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* These should all work.  The test that something was written is
       a bit rudimentary to avoid having the test depend on the
       specific values, instead we initialize with NaN and check the
       value isn't NaN after the call. */
    for(spec = 1; spec <= QMOST_NUM_SPEC; spec++) {
        for(arm = 1; arm <= QMOST_NUM_ARMS; arm++) {
            minwave = NAN;
            maxwave = NAN;

            code = qmost_get_skysubwave(spec, arm,
                                        &minwave, &maxwave);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            cpl_test(!isnan(minwave));
            cpl_test(!isnan(maxwave));
        }
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_constants module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_get_rebin_params();
    test_qmost_get_skysubwave();

    return cpl_test_end(0);
}

/**@}*/
