/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_model_psf.h"
#include "qmost_traceinfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_model_psf_test  Unit test of qmost_model_psf
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_model_psf_full
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_model_psf_full(void)
{
    int nx = 256;
    int ny = 512;
    int yref = 256;
    int xfwhm = 3;
    int subsample = 5;
    float profsamp = 1.0 / subsample;

    int nord = 2;
    int nobj = 2;

    int xcen[2] = { 100, 156 };
    double pkht[2] = { 9900, 10100 };
    double coef[3] = { 0, 5.0, -2.5 };

    cpl_polynomial *thetrace = NULL;
    int i;

    double yrel, xpoly;

    cpl_image *img = NULL;
    cpl_propertylist *hdr = NULL;

    float *img_data;
    int x, y, iobj;
    double dx, xsigma, xw;

    cpl_imagelist *ideal_prof_img = NULL;
    cpl_imagelist *ideal_prof_var = NULL;

    cpl_image *this_prof_img = NULL;
    cpl_image *this_prof_var = NULL;

    float *prof_img_data;
    float *prof_var_data;
    int hh, npsf;
    float profsum;

    cpl_table *trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;

    qmost_traceinfo tr;
    double xpdummy[1] = { 0 };
    double ypdummy[1] = { 0 };
    float fwhmdummy[1] = { 3 };
    float peakdummy[1] = { 10000 };
    float contrastdummy[1] = { 1.0 };
    cpl_size icoef;

    int xl, xh;

    cpl_imagelist *prof_img = NULL;
    cpl_imagelist *prof_var = NULL;
    cpl_propertylist *prof_hdr = NULL;

    cpl_error_code code;

    /* Set up polynomial */
    thetrace = cpl_polynomial_new(1);

    for(i = nord; i >= 0; i--) {
        icoef = i;
        cpl_polynomial_set_coeff(thetrace, &icoef, coef[i]);
    }

    /* Generate a test image with specified number of traces */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    img_data = cpl_image_get_data_float(img);
    cpl_test_nonnull(img_data);

    xsigma = xfwhm / CPL_MATH_FWHM_SIG;
    xw = 0.5 / (xsigma*xsigma);

    ideal_prof_img = cpl_imagelist_new();
    ideal_prof_var = cpl_imagelist_new();

    hh = 27;
    npsf = 2*hh+1;

    if(img_data) {
        for(y = 0; y < ny; y++) {
            /* Generate fibre flat spectrum */
            yrel = (y+1 - yref) / ((float) yref);
            xpoly = cpl_polynomial_eval_1d(thetrace, yrel, NULL);

            for(iobj = 0; iobj < nobj; iobj++) {
                xl = xcen[iobj]+xpoly - 3*xfwhm;
                if(xl < 0)
                    xl = 0;
                if(xl >= nx)
                    xl = nx-1;
                
                xh = xcen[iobj]+xpoly + 3*xfwhm;
                if(xh < 0)
                    xh = 0;
                if(xh >= nx)
                    xh = nx-1;
                
                for(x = xl; x <= xh; x++) {
                    dx = (x+1) - (xcen[iobj]+xpoly);
                    img_data[y*nx+x] += pkht[iobj] * exp(-xw * dx*dx);
                }
            }

            /* Generate ideal oversampled PSF */
            this_prof_img = cpl_image_new(npsf, nobj, CPL_TYPE_FLOAT);
            this_prof_var = cpl_image_new(npsf, nobj, CPL_TYPE_FLOAT);

            prof_img_data = cpl_image_get_data_float(this_prof_img);
            prof_var_data = cpl_image_get_data_float(this_prof_var);

            cpl_test_nonnull(prof_img_data);
            cpl_test_nonnull(prof_var_data);

            for(iobj = 0; iobj < nobj; iobj++) {
                profsum = 0;

                for(x = 0; x < npsf; x++) {
                    dx = (x - hh) * profsamp;

                    prof_img_data[iobj*npsf+x] = exp(-xw * dx*dx);
                    prof_var_data[iobj*npsf+x] = 0;

                    profsum += prof_img_data[iobj*npsf+x];
                }

                /* The normalization is a bit subtle, it's before
                   resampling so account for that here. */
                profsum *= profsamp;

                /* Normalize */
                for(x = 0; x < npsf; x++) {
                    prof_img_data[iobj*npsf+x] /= profsum;
                    prof_var_data[iobj*npsf+x] /= profsum;
                }
            }

            cpl_imagelist_set(ideal_prof_img, this_prof_img, y);
            cpl_imagelist_set(ideal_prof_var, this_prof_var, y);
        }
    }
    
    /* A bad pixel */
    cpl_image_reject(img, xcen[0], 42);

    /* The header (blank) */
    hdr = cpl_propertylist_new();

    /* Set up trace table */
    code = qmost_trcreate(&trace_tbl, nord, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(trace_tbl);

    if(trace_tbl) {
        for(iobj = 0; iobj < nobj; iobj++) {
            memset(&tr, 0, sizeof(tr));
            tr.specnum = iobj+1;
            tr.live = 1;
            tr.yst = 1;
            tr.yfn = ny;
            tr.yref = yref;
            tr.trrms = 0;
            tr.nord = nord;

            icoef = 0;
            cpl_polynomial_set_coeff(thetrace, &icoef, xcen[iobj]+coef[0]);

            tr.coefs = thetrace;
            tr.npos = 1;
            tr.xpos = xpdummy;
            tr.ypos = ypdummy;
            tr.fwhm = fwhmdummy;
            tr.peak = peakdummy;
            tr.contrast = contrastdummy;

            qmost_trwrite1(trace_tbl, tr, iobj+1);
        }
    }

    trace_hdr = cpl_propertylist_new();
    cpl_test_nonnull(trace_hdr);
    cpl_propertylist_update_int(trace_hdr, "MINYST", 1);
    cpl_propertylist_update_int(trace_hdr, "MAXYFN", ny);

    prof_hdr = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_model_psf_full(NULL, hdr, trace_tbl, trace_hdr,
                                5, subsample, 100,
                                &prof_img, &prof_var, prof_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_model_psf_full(NULL, NULL, trace_tbl, trace_hdr,
                                5, subsample, 100,
                                &prof_img, &prof_var, prof_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_model_psf_full(img, hdr, NULL, trace_hdr,
                                5, subsample, 100,
                                &prof_img, &prof_var, prof_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_model_psf_full(img, hdr, trace_tbl, NULL,
                                5, subsample, 100,
                                &prof_img, &prof_var, prof_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_model_psf_full(img, hdr, trace_tbl, trace_hdr,
                                5, subsample, 100,
                                NULL, &prof_var, prof_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_model_psf_full(img, hdr, trace_tbl, trace_hdr,
                                5, subsample, 100,
                                &prof_img, NULL, prof_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_model_psf_full(img, hdr, trace_tbl, trace_hdr,
                                5, subsample, 100,
                                &prof_img, &prof_var, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* This should work */
    code = qmost_model_psf_full(img, hdr, trace_tbl, trace_hdr,
                                5, subsample, 100,
                                &prof_img, &prof_var, prof_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Check profile */
    cpl_test_imagelist_abs(prof_img, ideal_prof_img, 0.05);

    cpl_imagelist_delete(prof_img);
    prof_img = NULL;

    cpl_imagelist_delete(prof_var);
    prof_var = NULL;
    
    cpl_propertylist_delete(prof_hdr);
    prof_hdr = NULL;

    cpl_image_delete(img);
    img = NULL;

    cpl_propertylist_delete(hdr);
    hdr = NULL;

    cpl_table_delete(trace_tbl);
    trace_tbl = NULL;

    cpl_propertylist_delete(trace_hdr);
    trace_hdr = NULL;

    cpl_imagelist_delete(ideal_prof_img);
    ideal_prof_img = NULL;

    cpl_imagelist_delete(ideal_prof_var);
    ideal_prof_var = NULL;

    cpl_polynomial_delete(thetrace);
    thetrace = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_psf_stats
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_psf_stats(void)
{
    int ny = 256;
    int xfwhm = 3;
    int subsample = 5;
    float profsamp = 1.0 / subsample;

    int nobj = 2;

    int x, y, iobj;
    double dx, xsigma, xw;

    cpl_imagelist *prof_img = NULL;
    cpl_imagelist *test_img = NULL;
    cpl_propertylist *prof_hdr = NULL;
    cpl_image *this_prof_img = NULL;
    float *prof_img_data;

    cpl_table *fibinfo_tbl = NULL;

    int hh, npsf;
    float profsum;

    float tstfwhm;

    cpl_error_code code;

    /* Generate some profiles */
    xsigma = xfwhm / CPL_MATH_FWHM_SIG;
    xw = 0.5 / (xsigma*xsigma);

    prof_img = cpl_imagelist_new();

    hh = 27;
    npsf = 2*hh+1;

    for(y = 0; y < ny; y++) {
        /* Generate ideal oversampled PSF */
        this_prof_img = cpl_image_new(npsf, nobj, CPL_TYPE_FLOAT);
        
        prof_img_data = cpl_image_get_data_float(this_prof_img);
        cpl_test_nonnull(prof_img_data);
        
        for(iobj = 0; iobj < nobj; iobj++) {
            profsum = 0;
            
            for(x = 0; x < npsf; x++) {
                dx = (x - hh) * profsamp;
                
                prof_img_data[iobj*npsf+x] = exp(-xw * dx*dx);
                
                profsum += prof_img_data[iobj*npsf+x];
            }
            
            /* The normalization is a bit subtle, it's before
               resampling so account for that here. */
            profsum *= profsamp;
            
            /* Normalize */
            for(x = 0; x < npsf; x++) {
                prof_img_data[iobj*npsf+x] /= profsum;
            }
        }
        
        cpl_imagelist_set(prof_img, this_prof_img, y);
    }
    
    prof_hdr = cpl_propertylist_new();

    fibinfo_tbl = cpl_table_new(nobj);

    /* NULL input tests */
    code = qmost_psf_stats(NULL, prof_hdr,
                           QMOST_ARM_RED, -1,
                           fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_psf_stats(prof_img, NULL,
                           QMOST_ARM_RED, -1,
                           fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid start point */
    code = qmost_psf_stats(prof_img, prof_hdr,
                           QMOST_ARM_RED, 2*ny,
                           fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Not float */
    test_img = cpl_imagelist_new();
    cpl_imagelist_set(test_img, cpl_image_new(128, 128, CPL_TYPE_INT), 0);

    code = qmost_psf_stats(test_img, prof_hdr,
                           QMOST_ARM_RED, -1,
                           fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_imagelist_delete(test_img);
    test_img = NULL;

    /* Missing CD1_1 in FITS header */
    code = qmost_psf_stats(prof_img, prof_hdr,
                           QMOST_ARM_RED, -1,
                           fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add it */
    cpl_propertylist_update_float(prof_hdr, "CD1_1", profsamp);

    /* Should now be able to run without a FIBINFO */
    code = qmost_psf_stats(prof_img, prof_hdr,
                           QMOST_ARM_RED, -1,
                           NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    tstfwhm = cpl_propertylist_get_float(prof_hdr,
                                         "ESO QC PSF FWHM MED");
    cpl_test_abs(tstfwhm, xfwhm, 0.1);

    /* Missing FIB_ST in FIBINFO */
    code = qmost_psf_stats(prof_img, prof_hdr,
                           QMOST_ARM_RED, -1,
                           fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add it, still null but it should now run and do nothing */
    cpl_table_new_column(fibinfo_tbl, "FIB_ST", CPL_TYPE_INT);

    code = qmost_psf_stats(prof_img, prof_hdr,
                           QMOST_ARM_RED, -1,
                           fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Now set fibres active and check we get what we expect */
    cpl_table_set_int(fibinfo_tbl, "FIB_ST", 0, 2);
    cpl_table_set_int(fibinfo_tbl, "FIB_ST", 1, 2);

    code = qmost_psf_stats(prof_img, prof_hdr,
                           QMOST_ARM_RED, -1,
                           fibinfo_tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iobj = 0; iobj < nobj; iobj++) {
        tstfwhm = cpl_table_get(fibinfo_tbl, "FWHM_R", iobj, NULL);
        cpl_test_abs(tstfwhm, xfwhm, 0.1);
    }

    cpl_imagelist_delete(prof_img);
    prof_img = NULL;

    cpl_propertylist_delete(prof_hdr);
    prof_hdr = NULL;

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_model_psf module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_model_psf_full();
    test_qmost_psf_stats();

    return cpl_test_end(0);
}

/**@}*/
