/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_rebin_spectra.h"
#include "qmost_waveinfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_rebin_spectra_test  Unit test of qmost_rebin_spectra
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Utility routine to check alignment of features across
 *          fibres and fail test if they aren't aligned.
 *
 * @param   out_spec_img  (Given)     A 2D extracted and rebinned set
 *                                    of spectra.
 * @param   out_spec_var  (Given)     The corresponding variance.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void check_alignment(cpl_image *out_spec_img, cpl_image *out_spec_var)
{
    cpl_image *ref_img = NULL;
    cpl_image *ref_var = NULL;
    cpl_image *this_img = NULL;
    cpl_image *this_var = NULL;
    cpl_image *diff_img = NULL;
    cpl_image *diff_err = NULL;
    cpl_image *diff_snr = NULL;
    int ispec, nspec, nxout;

    /* They should now be roughly aligned with the first one, which
       was not shifted.  Check that here. */
    nxout = cpl_image_get_size_x(out_spec_img);
    nspec = cpl_image_get_size_y(out_spec_img);

    ref_img = cpl_image_extract(out_spec_img,
                                1,
                                1,
                                nxout,
                                1);
    ref_var = cpl_image_extract(out_spec_var,
                                1,
                                1,
                                nxout,
                                1);

    for(ispec = 1; ispec < nspec; ispec++) {
        this_img = cpl_image_extract(out_spec_img,
                                     1,
                                     ispec+1,
                                     nxout,
                                     ispec+1);
        this_var = cpl_image_extract(out_spec_var,
                                     1,
                                     ispec+1,
                                     nxout,
                                     ispec+1);

        diff_img = cpl_image_subtract_create(this_img, ref_img);

        diff_err = cpl_image_add_create(this_var, ref_var);
        cpl_image_threshold(diff_err, 1.0, FLT_MAX, 1.0, FLT_MAX);
        cpl_image_power(diff_err, 0.5);

        diff_snr = cpl_image_divide_create(diff_img, diff_err);

        qmost_test_image_float_abs(diff_snr, 0, 0, 10);

        cpl_image_delete(this_img);
        this_img = NULL;

        cpl_image_delete(this_var);
        this_var = NULL;

        cpl_image_delete(diff_img);
        diff_img = NULL;

        cpl_image_delete(diff_err);
        diff_err = NULL;

        cpl_image_delete(diff_snr);
        diff_snr = NULL;
    }

    cpl_image_delete(ref_img);
    ref_img = NULL;
    
    cpl_image_delete(ref_var);
    ref_var = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_rebin_spectra_off
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_rebin_spectra_off(void)
{
    cpl_image *in_spec_img = NULL;
    cpl_image *in_spec_var = NULL;
    cpl_propertylist *in_spec_hdr = NULL;
    cpl_image *binned_spec_img = NULL;
    cpl_image *binned_spec_var = NULL;
    cpl_propertylist *binned_spec_hdr = NULL;
    cpl_table *wave_tbl = NULL;
    cpl_propertylist *wave_hdr = NULL;
    cpl_table *ob_wave_tbl1 = NULL;
    cpl_propertylist *ob_wave_hdr1 = NULL;
    cpl_table *ob_wave_tbl2 = NULL;
    cpl_propertylist *ob_wave_hdr2 = NULL;
    cpl_image *out_spec_img = NULL;
    cpl_image *out_spec_var = NULL;
    cpl_propertylist *out_spec_hdr = NULL;
    cpl_error_code code;

    int nwave = 2048;
    int nspec = 5;
    int nlines = 32;

    double lam_min = 5000;
    double lam_max = 6000;

    double angppix = (lam_max - lam_min) / (nwave - 1);

    double xfwhm = 2.0;
    double pkht = 1000;

    float *buf = NULL;

    int iline;
    int ix, ispec, xl, xh;
    double lam, xcen, xpoly, dx, xw;

    qmost_waveinfo wv;
    cpl_size deg;

    double *waveoff = NULL;
    double *veloff = NULL;

    /* Simulate an arc spectrum */
    in_spec_img = cpl_image_new(nwave, nspec, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(in_spec_img, -8, 8);

    buf = cpl_image_get_data_float(in_spec_img);

    xw = 0.5*CPL_MATH_FWHM_SIG*CPL_MATH_FWHM_SIG / (xfwhm*xfwhm);

    for(iline = 0; iline < nlines; iline++) {
        /* Wavelength of injected line */
        lam = lam_min + (iline+0.5) * ((lam_max - lam_min) / nlines);
        /* add a bit of variation so spacing isn't completely uniform */
        lam += (iline % 2 ? 5.0 : -5.0) * ((double) iline) / nlines;

        /* Spectral pixel of line centre */
        xcen = 1 + (lam - lam_min) / angppix;

        /* Inject line */
        for(ispec = 0; ispec < nspec; ispec++) {
            /* Misalign the fibres.  This number needs to be even so
             * we can bin by 2 later and not introduce jaggies between
             * the fibres. */
            xpoly = ispec * 4;

            xl = xcen + xpoly - 3*xfwhm;
            if(xl < 0)
                xl = 0;
            if(xl >= nwave)
                xl = nwave-1;
            
            xh = xcen + xpoly + 3*xfwhm;
            if(xh < 0)
                xh = 0;
            if(xh >= nwave)
                xh = nwave-1;
            
            for(ix = xl; ix <= xh; ix++) {
                dx = (ix+1) - (xcen + xpoly);
                buf[ispec*nwave+ix] += pkht * exp(-xw * dx*dx);
            }
        }
    }

    in_spec_var = cpl_image_duplicate(in_spec_img);
    cpl_image_threshold(in_spec_var, 0, FLT_MAX, 0, FLT_MAX);
    cpl_image_add_scalar(in_spec_var, 4*4);

    /* Make some bad pixels to test those branches */
    for(ix = 42; ix < 50; ix++) {
        cpl_image_set(in_spec_img, ix, 1, 0);
        cpl_image_reject(in_spec_img, ix, 1);
        cpl_image_set(in_spec_var, ix, 1, 0);
        cpl_image_reject(in_spec_var, ix, 1);
    }

    in_spec_hdr = cpl_propertylist_new();

    /* Wavelength solution table */
    code = qmost_wvcreate(&wave_tbl, 1, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        xpoly = ispec * 4;  /* misalign the fibres */

        wv.specnum = ispec+1;
        wv.live = 1;
        wv.nord = 1;
        wv.coefs = cpl_polynomial_new(1);

        deg = 0;
        cpl_polynomial_set_coeff(wv.coefs, &deg,
                                 0.5*(lam_min+lam_max) - xpoly * angppix);
        deg = 1;
        cpl_polynomial_set_coeff(wv.coefs, &deg,
                                 angppix);

        wv.xref = 0.5*nwave;
        wv.ngood = 2;
        wv.medresid = 0;
        wv.fit_rms = 0;
        wv.wave1 = lam_min;
        wv.waven = lam_max;
        wv.dwave = angppix;
        wv.nlines = 2;
        wv.xpos = cpl_calloc(2, sizeof(double));
        wv.fwhm = cpl_calloc(2, sizeof(float));
        wv.wave_calc = cpl_calloc(2, sizeof(double));
        wv.wave_true = cpl_calloc(2, sizeof(double));
        wv.fit_flag = cpl_calloc(2, sizeof(unsigned char));
        wv.wave_cor = cpl_calloc(2, sizeof(double));
        wv.peak = cpl_calloc(2, sizeof(float));
        wv.contrast = cpl_calloc(2, sizeof(float));

        qmost_wvwrite1(wave_tbl, ispec+1, wv);

        cpl_polynomial_delete(wv.coefs);
        cpl_free(wv.xpos);
        cpl_free(wv.fwhm);
        cpl_free(wv.wave_calc);
        cpl_free(wv.wave_true);
        cpl_free(wv.fit_flag);
        cpl_free(wv.wave_cor);
        cpl_free(wv.peak);
        cpl_free(wv.contrast);
    }

    wave_hdr = cpl_propertylist_new();

    /* Offsets */
    waveoff = cpl_calloc(nspec, sizeof(double));
    veloff = cpl_calloc(nspec, sizeof(double));

    /* This should work, we let it decide the sampling */
    out_spec_hdr = cpl_propertylist_new();

    code = qmost_rebin_spectra_off(in_spec_img,
                                   in_spec_var,
                                   in_spec_hdr,
                                   wave_tbl,
                                   wave_hdr,
                                   NULL,
                                   NULL,
                                   NULL,
                                   NULL,
                                   waveoff,
                                   veloff,
                                   nspec,
                                   lam_min,
                                   lam_max,
                                   0,
                                   &out_spec_img,
                                   &out_spec_var,
                                   out_spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(out_spec_img);
    cpl_test_nonnull(out_spec_var);

    check_alignment(out_spec_img, out_spec_var);

    cpl_image_delete(out_spec_img);
    out_spec_img = NULL;

    cpl_image_delete(out_spec_var);
    out_spec_var = NULL;

    cpl_propertylist_delete(out_spec_hdr);
    out_spec_hdr = NULL;

    /* Test with binning */
    out_spec_hdr = cpl_propertylist_new();

    binned_spec_img = cpl_image_rebin(in_spec_img, 1, 1, 2, 1);
    binned_spec_var = cpl_image_rebin(in_spec_var, 1, 1, 2, 1);
    binned_spec_hdr = cpl_propertylist_duplicate(in_spec_hdr);
    cpl_propertylist_update_int(binned_spec_hdr, "ESO DRS SPECBIN", 2);
    cpl_propertylist_update_int(binned_spec_hdr, "ESO DRS SPATBIN", 1);
    cpl_propertylist_update_float(binned_spec_hdr, "ESO DRS WVCRV", 1.5);

    code = qmost_rebin_spectra_off(binned_spec_img,
                                   binned_spec_var,
                                   binned_spec_hdr,
                                   wave_tbl,
                                   wave_hdr,
                                   NULL,
                                   NULL,
                                   NULL,
                                   NULL,
                                   waveoff,
                                   veloff,
                                   nspec,
                                   lam_min,
                                   lam_max,
                                   0,
                                   &out_spec_img,
                                   &out_spec_var,
                                   out_spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(out_spec_img);
    cpl_test_nonnull(out_spec_var);

    check_alignment(out_spec_img, out_spec_var);

    cpl_image_delete(out_spec_img);
    out_spec_img = NULL;

    cpl_image_delete(out_spec_var);
    out_spec_var = NULL;

    cpl_propertylist_delete(out_spec_hdr);
    out_spec_hdr = NULL;

    cpl_image_delete(binned_spec_img);
    binned_spec_img = NULL;

    cpl_image_delete(binned_spec_var);
    binned_spec_var = NULL;

    cpl_propertylist_delete(binned_spec_hdr);
    binned_spec_hdr = NULL;

    /* Test user specified sampling and go off the ends a bit to test
     * the padding. */
    out_spec_hdr = cpl_propertylist_new();

    code = qmost_rebin_spectra_off(in_spec_img,
                                   in_spec_var,
                                   in_spec_hdr,
                                   wave_tbl,
                                   wave_hdr,
                                   NULL,
                                   NULL,
                                   NULL,
                                   NULL,
                                   waveoff,
                                   veloff,
                                   nspec,
                                   lam_min-50,
                                   lam_max+50,
                                   1.0,
                                   &out_spec_img,
                                   &out_spec_var,
                                   out_spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(out_spec_img);
    cpl_test_nonnull(out_spec_var);

    check_alignment(out_spec_img, out_spec_var);

    cpl_test_abs(cpl_propertylist_get_double(out_spec_hdr, "CD1_1"), 1.0, 0.1);

    cpl_image_delete(out_spec_img);
    out_spec_img = NULL;

    cpl_image_delete(out_spec_var);
    out_spec_var = NULL;

    cpl_propertylist_delete(out_spec_hdr);
    out_spec_hdr = NULL;

    /* Run again with OB wavelength solution correction tables that
     * don't apply a correction, so we should get the same answer. */
    code = qmost_wvcreate(&ob_wave_tbl1, 1, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = qmost_wvcreate(&ob_wave_tbl2, 1, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(ispec = 0; ispec < nspec; ispec++) {
        wv.specnum = ispec+1;
        wv.live = 1;
        wv.nord = 1;
        wv.coefs = cpl_polynomial_new(1);
        wv.xref = 0.5*nwave;
        wv.ngood = 2;
        wv.medresid = 0;
        wv.fit_rms = 0;
        wv.wave1 = lam_min;
        wv.waven = lam_max;
        wv.dwave = angppix;
        wv.nlines = 2;
        wv.xpos = cpl_calloc(2, sizeof(double));
        wv.fwhm = cpl_calloc(2, sizeof(float));
        wv.wave_calc = cpl_calloc(2, sizeof(double));
        wv.wave_true = cpl_calloc(2, sizeof(double));
        wv.fit_flag = cpl_calloc(2, sizeof(unsigned char));
        wv.wave_cor = cpl_calloc(2, sizeof(double));
        wv.peak = cpl_calloc(2, sizeof(float));
        wv.contrast = cpl_calloc(2, sizeof(float));

        qmost_wvwrite1(ob_wave_tbl1, ispec+1, wv);
        qmost_wvwrite1(ob_wave_tbl2, ispec+1, wv);

        cpl_polynomial_delete(wv.coefs);
        cpl_free(wv.xpos);
        cpl_free(wv.fwhm);
        cpl_free(wv.wave_calc);
        cpl_free(wv.wave_true);
        cpl_free(wv.fit_flag);
        cpl_free(wv.wave_cor);
        cpl_free(wv.peak);
        cpl_free(wv.contrast);
    }

    ob_wave_hdr1 = cpl_propertylist_new();
    ob_wave_hdr2 = cpl_propertylist_new();

    out_spec_hdr = cpl_propertylist_new();

    code = qmost_rebin_spectra_off(in_spec_img,
                                   in_spec_var,
                                   in_spec_hdr,
                                   wave_tbl,
                                   wave_hdr,
                                   ob_wave_tbl1,
                                   ob_wave_hdr1,
                                   ob_wave_tbl2,
                                   ob_wave_hdr2,
                                   waveoff,
                                   veloff,
                                   nspec,
                                   lam_min,
                                   lam_max,
                                   0,
                                   &out_spec_img,
                                   &out_spec_var,
                                   out_spec_hdr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(out_spec_img);
    cpl_test_nonnull(out_spec_var);

    check_alignment(out_spec_img, out_spec_var);

    cpl_image_delete(out_spec_img);
    out_spec_img = NULL;

    cpl_image_delete(out_spec_var);
    out_spec_var = NULL;

    cpl_propertylist_delete(out_spec_hdr);
    out_spec_hdr = NULL;

    /* Clean up */
    cpl_image_delete(in_spec_img);
    cpl_image_delete(in_spec_var);
    cpl_propertylist_delete(in_spec_hdr);
    cpl_table_delete(wave_tbl);
    cpl_propertylist_delete(wave_hdr);
    cpl_table_delete(ob_wave_tbl1);
    cpl_propertylist_delete(ob_wave_hdr1);
    cpl_table_delete(ob_wave_tbl2);
    cpl_propertylist_delete(ob_wave_hdr2);
    cpl_free(waveoff);
    cpl_free(veloff);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_rebin_spectra module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_rebin_spectra_off();

    return cpl_test_end(0);
}

/**@}*/
