/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_stats.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_stats_test  Unit test of qmost_stats
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_med
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_med(void)
{
    float data[] = { 1, 2, 3, 4, 5 };
    unsigned char bpm[] = { 0, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;
    float med;

    cpl_error_code code;

    n = sizeof(data) / sizeof(data[0]);

    /* NULL input */
    code = qmost_med(NULL, NULL, 0, &med);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Zero length input */
    code = qmost_med(data, NULL, 0, &med);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* All invalid */
    code = qmost_med(data, allbad, n, &med);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Only one element */
    code = qmost_med(data, NULL, 1, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 1, FLT_EPSILON);

    /* Check answers for valid inputs */
    code = qmost_med(data, NULL, n, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 3, FLT_EPSILON);

    code = qmost_med(data, bpm, n, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 4, FLT_EPSILON);

    code = qmost_med(data, NULL, n-1, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, FLT_EPSILON);

    code = qmost_med(data, bpm, n-1, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dmed
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_dmed(void)
{
    double data[] = { 1, 2, 3, 4, 5 };
    unsigned char bpm[] = { 0, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;
    double med;

    cpl_error_code code;

    n = sizeof(data) / sizeof(data[0]);

    /* NULL input */
    code = qmost_dmed(NULL, NULL, 0, &med);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Zero length input */
    code = qmost_dmed(data, NULL, 0, &med);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* All invalid */
    code = qmost_dmed(data, allbad, n, &med);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Only one element */
    code = qmost_dmed(data, NULL, 1, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 1, FLT_EPSILON);

    /* Check answers for valid inputs */
    code = qmost_dmed(data, NULL, n, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 3, FLT_EPSILON);

    code = qmost_dmed(data, bpm, n, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 4, FLT_EPSILON);

    code = qmost_dmed(data, NULL, n-1, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, FLT_EPSILON);

    code = qmost_dmed(data, bpm, n-1, &med);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_medmad
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_medmad(void)
{
    float data[] = { 1, 2, 3, 4, 5 };
    unsigned char bpm[] = { 0, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;
    float med, mad;

    cpl_error_code code;

    n = sizeof(data) / sizeof(data[0]);

    /* NULL input */
    code = qmost_medmad(NULL, NULL, 0, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Zero length input */
    code = qmost_medmad(data, NULL, 0, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* All invalid */
    code = qmost_medmad(data, allbad, n, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Only one element */
    code = qmost_medmad(data, NULL, 1, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 1, FLT_EPSILON);
    cpl_test_abs(mad, 0, FLT_EPSILON);

    /* Check answers for valid inputs */
    code = qmost_medmad(data, NULL, n, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 3, FLT_EPSILON);
    cpl_test_abs(mad, 1, FLT_EPSILON);

    code = qmost_medmad(data, bpm, n, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 4, FLT_EPSILON);
    cpl_test_abs(mad, 1, FLT_EPSILON);

    code = qmost_medmad(data, NULL, n-1, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, FLT_EPSILON);
    cpl_test_abs(mad, 1.0, FLT_EPSILON);

    code = qmost_medmad(data, bpm, n-1, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, FLT_EPSILON);
    cpl_test_abs(mad, 1.5, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_dmedmad
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_dmedmad(void)
{
    double data[] = { 1, 2, 3, 4, 5 };
    unsigned char bpm[] = { 0, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;
    double med, mad;

    cpl_error_code code;

    n = sizeof(data) / sizeof(data[0]);

    /* NULL input */
    code = qmost_dmedmad(NULL, NULL, 0, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Zero length input */
    code = qmost_dmedmad(data, NULL, 0, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* All invalid */
    code = qmost_dmedmad(data, allbad, n, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Only one element */
    code = qmost_dmedmad(data, NULL, 1, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 1, DBL_EPSILON);
    cpl_test_abs(mad, 0, DBL_EPSILON);

    /* Check answers for valid inputs */
    code = qmost_dmedmad(data, NULL, n, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 3, DBL_EPSILON);
    cpl_test_abs(mad, 1, DBL_EPSILON);

    code = qmost_dmedmad(data, bpm, n, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 4, DBL_EPSILON);
    cpl_test_abs(mad, 1, DBL_EPSILON);

    code = qmost_dmedmad(data, NULL, n-1, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, DBL_EPSILON);
    cpl_test_abs(mad, 1.0, DBL_EPSILON);

    code = qmost_dmedmad(data, bpm, n-1, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, DBL_EPSILON);
    cpl_test_abs(mad, 1.5, DBL_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_medmadcut
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_medmadcut(void)
{
    float data[] = { 1, 2, 3, 4, 5 };
    unsigned char bpm[] = { 0, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;
    float med, mad;

    cpl_error_code code;

    n = sizeof(data) / sizeof(data[0]);

    /* NULL input */
    code = qmost_medmadcut(NULL, NULL, 0, -10, 10, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Zero length input */
    code = qmost_medmadcut(data, NULL, 0, -10, 10, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* All invalid */
    code = qmost_medmadcut(data, allbad, n, -10, 10, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Only one element */
    code = qmost_medmadcut(data, NULL, 1, -10, 10, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 1, FLT_EPSILON);
    cpl_test_abs(mad, 0, FLT_EPSILON);

    /* Check answers for valid inputs */
    code = qmost_medmadcut(data, NULL, n, -10, 10, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 3, FLT_EPSILON);
    cpl_test_abs(mad, 1, FLT_EPSILON);

    code = qmost_medmadcut(data, bpm, n, -10, 10, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 4, FLT_EPSILON);
    cpl_test_abs(mad, 1, FLT_EPSILON);

    code = qmost_medmadcut(data, NULL, n, -10, 4.5, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, FLT_EPSILON);
    cpl_test_abs(mad, 1.0, FLT_EPSILON);

    code = qmost_medmadcut(data, bpm, n, -10, 4.5, &med, &mad);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(med, 2.5, FLT_EPSILON);
    cpl_test_abs(mad, 1.5, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_meansig
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_meansig(void)
{
    float data[] = { 1, 2, 3, 4, 5 };
    unsigned char bpm[] = { 0, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;
    float mean, sig;

    cpl_error_code code;

    n = sizeof(data) / sizeof(data[0]);

    /* NULL input */
    code = qmost_meansig(NULL, NULL, 0, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* All invalid */
    code = qmost_meansig(data, allbad, n, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_abs(mean, FLT_MAX, FLT_EPSILON);
    cpl_test_abs(sig, FLT_MAX, FLT_EPSILON);

    /* Only one element */
    code = qmost_meansig(data, NULL, 1, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 1, FLT_EPSILON);
    cpl_test_abs(sig, 0, FLT_EPSILON);

    /* Check answers for valid inputs */
    code = qmost_meansig(data, NULL, n, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 3, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(2), FLT_EPSILON);

    code = qmost_meansig(data, bpm, n, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 10.0/3.0, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(26.0/9.0), FLT_EPSILON);

    code = qmost_meansig(data, NULL, n-1, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 2.5, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(1.25), FLT_EPSILON);

    code = qmost_meansig(data, bpm, n-1, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 2.5, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(2.25), FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_meansigcut
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_meansigcut(void)
{
    float data[] = { 1, 2, 3, 4, 5 };
    unsigned char bpm[] = { 0, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;
    float mean, sig;

    cpl_error_code code;

    n = sizeof(data) / sizeof(data[0]);

    /* NULL input */
    code = qmost_meansigcut(NULL, NULL, 0, -1, 7, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* All invalid */
    code = qmost_meansigcut(data, allbad, n, -1, 7, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);
    cpl_test_abs(mean, FLT_MAX, FLT_EPSILON);
    cpl_test_abs(sig, FLT_MAX, FLT_EPSILON);

    /* Only one element */
    code = qmost_meansigcut(data, NULL, 1, -1, 7, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 1, FLT_EPSILON);
    cpl_test_abs(sig, 0, FLT_EPSILON);

    /* Check answers for valid inputs */
    code = qmost_meansigcut(data, NULL, n, -1, 7, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 3, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(2), FLT_EPSILON);

    code = qmost_meansigcut(data, bpm, n, -1, 7, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 10.0/3.0, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(26.0/9.0), FLT_EPSILON);

    code = qmost_meansigcut(data, NULL, n-1, -1, 7, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 2.5, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(1.25), FLT_EPSILON);

    code = qmost_meansigcut(data, bpm, n-1, -1, 7, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 2.5, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(2.25), FLT_EPSILON);

    /* Now using the cut function */
    code = qmost_meansigcut(data, NULL, n, 1.5, 4.5, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 3, FLT_EPSILON);
    cpl_test_abs(sig, sqrt(2.0/3.0), FLT_EPSILON);

    code = qmost_meansigcut(data, bpm, n, 1.5, 5.5, &mean, &sig);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_abs(mean, 4.5, FLT_EPSILON);
    cpl_test_abs(sig, 0.5, FLT_EPSILON);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_sumbpm
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_sumbpm(void)
{
    unsigned char bpm[] = { 0, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;
    int result;

    n = sizeof(bpm) / sizeof(bpm[0]);

    result = 42;
    qmost_sumbpm(bpm, 0, &result);
    cpl_test_eq(result, 0);

    result = 42;
    qmost_sumbpm(bpm, n, &result);
    cpl_test_eq(result, 2);

    result = 42;
    qmost_sumbpm(allbad, n, &result);
    cpl_test_eq(result, 5);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_firstgood
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_firstgood(void)
{
    unsigned char bpma[] = { 0, 1, 1, 0, 0 };
    unsigned char bpmb[] = { 1, 1, 1, 0, 0 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;

    n = sizeof(bpma) / sizeof(bpma[0]);

    cpl_test_eq(qmost_firstgood(bpma, 0), 0);

    cpl_test_eq(qmost_firstgood(bpma, n), 0);

    cpl_test_eq(qmost_firstgood(bpmb, n), 3);

    cpl_test_eq(qmost_firstgood(allbad, n), n);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_lastgood
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_lastgood(void)
{
    unsigned char bpma[] = { 0, 1, 1, 0, 0 };
    unsigned char bpmb[] = { 0, 1, 1, 0, 1 };
    unsigned char allbad[] = { 1, 1, 1, 1, 1 };

    int n;

    n = sizeof(bpma) / sizeof(bpma[0]);

    cpl_test_eq(qmost_lastgood(bpma, 0), -1);

    cpl_test_eq(qmost_lastgood(bpma, n), 4);

    cpl_test_eq(qmost_lastgood(bpmb, n), 3);

    cpl_test_eq(qmost_lastgood(allbad, n), -1);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_skylevel
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_skylevel(void)
{
    int hist1[] = { 1, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 6, 5, 4, 3, 2, 1, 0, 0, 0, 0, 0, 1 };
    int hist2[] = { 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 49, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1};
    int ihmin = 0;
    int ihmax;
    int mpix;
    int i;

    float skylev, sigma, toplev;

    /* Normal(ish) histogram */
    ihmax = (sizeof(hist1) / sizeof(hist1[0])) - 1;

    mpix = 0;
    for(i = ihmin; i <= ihmax; i++)
        mpix += hist1[i];

    qmost_skylevel(hist1,
                   ihmin, ihmax, mpix,
                   -FLT_MAX, FLT_MAX, 0,
                   &skylev, &sigma, &toplev);

    cpl_test_abs(skylev, 12, 0.5);
    cpl_test_abs(sigma, 1.48*2.0, 0.5);

    qmost_skylevel(hist1,
                   ihmin, ihmax, mpix,
                   -3, 3, 3,
                   &skylev, &sigma, &toplev);

    cpl_test_abs(skylev, 12, 0.5);
    cpl_test_abs(sigma, 1.48*2.0, 0.5);

    /* All non-outliers in one bin, to check the 0.5 limit */
    ihmax = (sizeof(hist2) / sizeof(hist2[0])) - 1;

    mpix = 0;
    for(i = ihmin; i <= ihmax; i++)
        mpix += hist2[i];

    qmost_skylevel(hist2,
                   ihmin, ihmax, mpix,
                   -3, 3, 3,
                   &skylev, &sigma, NULL);

    cpl_test_abs(skylev, 12, 0.5);
    cpl_test_abs(sigma, 0.5, 0.1);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_skylevel_image
 */
/*----------------------------------------------------------------------------*/
static void test_qmost_skylevel_image(void)
{
    cpl_image *image = NULL;
    cpl_image *tmp_image = NULL;

    int nx = 256;
    int ny = 256;

    float *image_data;
    double ua;

    int ihot, ipix, npix;

    cpl_error_code code;

    int minlev = 0;
    int maxlev = 65535;

    float real_skylev = 500.0;
    float real_skynoise = 2.0;
    float skylev, sigma;

    image = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    qmost_test_fill_noise_rounded_gauss(image, real_skylev, real_skynoise);

    npix = nx*ny;

    /* Add a smattering of hot pixels */
    image_data = cpl_image_get_data_float(image);

    for(ihot = 0; ihot < 100; ihot++) {
        ua = (((double) rand()) / RAND_MAX);
        ipix = rint(ua * npix);
        if(ipix >= 0 && ipix < npix) {
            image_data[ipix] += 10000 * (((double) rand()) / RAND_MAX);
        }
    }

    /* NULL inputs */
    code = qmost_skylevel_image(NULL, minlev, maxlev,
                                -3, 3, 3,
                                &skylev, &sigma);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_skylevel_image(image, minlev, maxlev,
                                -3, 3, 3,
                                NULL, &sigma);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_skylevel_image(image, minlev, maxlev,
                                -3, 3, 3,
                                &skylev, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid min, max */
    code = qmost_skylevel_image(image, maxlev, minlev,
                                -3, 3, 3,
                                &skylev, &sigma);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Something we can't cast to float */
    tmp_image = cpl_image_new(nx, ny, CPL_TYPE_FLOAT_COMPLEX);

    code = qmost_skylevel_image(tmp_image, maxlev, minlev,
                                -3, 3, 3,
                                &skylev, &sigma);
    cpl_test_eq_error(code, CPL_ERROR_TYPE_MISMATCH);

    cpl_image_delete(tmp_image);
    tmp_image = NULL;

    /* Proper inputs */
    code = qmost_skylevel_image(image, minlev, maxlev,
                                -FLT_MAX, 3, 3,
                                &skylev, &sigma);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(skylev, real_skylev, 0.1);
    cpl_test_abs(sigma, real_skynoise, 0.1);

    tmp_image = cpl_image_cast(image, CPL_TYPE_INT);

    code = qmost_skylevel_image(tmp_image, minlev, maxlev,
                                -FLT_MAX, 3, 3,
                                &skylev, &sigma);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(skylev, real_skylev, 0.1);
    cpl_test_abs(sigma, real_skynoise, 0.1);

    cpl_image_delete(tmp_image);
    tmp_image = NULL;

    cpl_image_delete(image);
    image = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_stats module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_med();
    test_qmost_dmed();
    test_qmost_medmad();
    test_qmost_dmedmad();
    test_qmost_medmadcut();
    test_qmost_meansig();
    test_qmost_meansigcut();
    test_qmost_sumbpm();
    test_qmost_firstgood();
    test_qmost_lastgood();
    test_qmost_skylevel();
    test_qmost_skylevel_image();

    return cpl_test_end(0);
}

/**@}*/
