/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_doarcs.h"
#include "qmost_extract_tram.h"
#include "qmost_fibtab.h"
#include "qmost_dfs.h"
#include "qmost_pfits.h"
#include "qmost_rebin_spectra.h"
#include "qmost_scattered.h"
#include "qmost_utils.h"
#include "qmost_waveinfo.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_arc_analyse"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_arc_analyse_description[] =
    "Combine individual wavecal frames into a stacked wavecal, remove\n"
    "scattered light, and do a boxcar extraction.  Identify emission\n"
    "features in the extracted spectrum and match to a line list.\n"
    "\n"
    "This recipe supports processing three different types of wavecals:\n"
    "the daytime or nighttime Fabry-Perot wavecals used in normal\n"
    "operations, or the special simuarc ThAr wavecals used for absolute\n"
    "wavelength calibration.  The frames should be tagged\n"
    QMOST_RAW_FIBRE_WAVE_DAY ", " QMOST_RAW_FIBRE_WAVE_NIGHT ", or "
    QMOST_RAW_FIBRE_WAVE_SIMUARC "\n"
    "accordingly.  Only one type of input should be given.  The\n"
    "processing to be done and the names of the product files are\n"
    "decided according to what type of input was provided.  The\n"
    "wavelength solution in each of these three cases will be tagged\n"
    QMOST_PRO_MASTER_WAVE ", " QMOST_PRO_OB_WAVE ", or "
    QMOST_PRO_SIMUARC_WAVE " respectively.  Reference wavelength\n"
    "solutions can also be made in place of master wavelength solutions\n"
    "by passing the recipe parameter reference, in which case the output\n"
    "is tagged " QMOST_CALIB_REFERENCE_WAVE " instead.\n"
    "\n"
    "The following files can be specified in the SOF:\n"
    "\n"
    "Description                   Req/Opt? Tag\n"
    "----------------------------- -------- --------------------\n"
    "Daytime FP wavecal frames     Required " QMOST_RAW_FIBRE_WAVE_DAY "\n"
    "or\n"
    "Nighttime FP wavecal frames   Required " QMOST_RAW_FIBRE_WAVE_NIGHT "\n"
    "or\n"
    "simuarc frames                Required " QMOST_RAW_FIBRE_WAVE_SIMUARC "\n"
    "\n"
    "FPE line list                 Required " QMOST_PRO_FPE_LINELIST "\n"
    "or\n"
    "Arc line list                 Required " QMOST_CALIB_ARC_LINELIST "\n"
    "\n"
    "Master bias frame             Optional " QMOST_PRO_MASTER_BIAS "\n"
    "Master dark frame             Optional " QMOST_PRO_MASTER_DARK "\n"
    "Master detector flat          Optional " QMOST_PRO_MASTER_DETECTOR_FLAT "\n"
    "Master bad pixel mask         Optional " QMOST_CALIB_MASTER_BPM "\n"
    "Master linearity table        Optional " QMOST_PRO_LINEARITY "\n"
    "Master wavelength map         Req/opt  " QMOST_CALIB_MASTER_WAVE_MAP "\n"
    "Trace table                   Required " QMOST_PRO_FIBRE_TRACE "\n"
    "Fibre mask                    Required " QMOST_PRO_FIBRE_MASK "\n"
    "Reference wavelength solution Optional " QMOST_CALIB_REFERENCE_WAVE "\n"
    "Master wavelength solution    Req/Opt  " QMOST_PRO_MASTER_WAVE "\n"
    "\n"
    "Note: the wavelength map input is required for master or simuarc\n"
    "wavelength solutions.  The master wavelength solution input is\n"
    "required for OB wavelength solutions.\n"
    "\n"
    "Outputs:\n\n"
    "Description                   Tag\n"
    "----------------------------- -----------------\n"
    "Master wavelength solution    " QMOST_PRO_MASTER_WAVE "\n"
    "or\n"
    "Reference wavelength solution " QMOST_CALIB_REFERENCE_WAVE "\n"
    "or\n"
    "OB wave solution correction   " QMOST_PRO_OB_WAVE "\n"
    "or\n"
    "simuarc wavelength solution   " QMOST_PRO_SIMUARC_WAVE "\n"
    "\n"
    "Optional processed outputs if keep=true:\n\n"
    "Description                   Tag\n"
    "----------------------------- -----------------\n"
    "Stacked arc frame             " QMOST_PROC_FIBRE_WAVE "\n"
    "\n"
    "Master extracted arc          " QMOST_PRO_MASTER_ARC "\n"
    "Master wave calibrated arc    " QMOST_PRO_MASTER_WARC "\n"
    "or\n"
    "OB extracted arc              " QMOST_PRO_OB_ARC "\n"
    "OB wave calibrated arc        " QMOST_PRO_OB_WARC "\n"
    "or\n"
    "Extracted simuarc             " QMOST_PRO_SIMUARC_ARC "\n"
    "Wave calibrated simuarc       " QMOST_PRO_SIMUARC_WARC "\n"
    "\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the master wavelength\n"
    "solution for LRS-A would be QMOST_" QMOST_PRO_MASTER_WAVE "_LRS-A.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_arc_analyse,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Analyse arc frames to create wavelength solution.",
                  qmost_arc_analyse_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_arc_analyse    qmost_arc_analyse
 *
 * @brief    Analyse arc frames to produce a master wavelength
 *           solution.
 *
 * @par Name:
 *   qmost_arc_analyse
 * @par Purpose:
 *   Extract the spectra from an arc exposure using a basic tramline
 *   algorithm. Identify the pixel position of emission lines in each
 *   spectrum and match them up to known arc line wavelengths. Do a
 *   robust fit to the wavelengths versus pixel position.
 * @par Type:
 *   Spectroscopic calibration
 * @par Parameters:
 *   - @b level (int, default 1):
 *     Determines the level of the analysis:
 *       - (0): QC0 analysis.
 *       - (1): QC1 analysis.
 *   - @b keep (bool, default false):
 *     If this flag is set, optional products generated during
 *     execution of the recipe are also saved. In the case of
 *     qmost_arc_analyse, this saves the processed arc image and the
 *     extracted, wavelength calibrated arc spectra.
 *   - @b ccdproc.swapx (string, default 100010101):
 *     String of 0 or 1 specifying for each arm of each spectrograph
 *     if we should flip the x axis to correct the wavelength order of
 *     the spectral axis.
 *   - @b imcombine.combtype (int, default 1):
 *     Determines the type of combination that is done to form the
 *     stacked arc frame.  Can take the following values:
 *       - (1): The output pixels are kappa sigma clipped means of
 *              the input pixels.
 *       - (2): The output pixels are medians of the input pixels.
 *   - @b imcombine.scaletype (int, default 0):
 *     Scaling to use when combining multiple images to form the
 *     stacked arc frame.  Can take the following values:
 *       - (0): No biasing or scaling is done.
 *       - (1): An additive offset is applied to remove any
 *              difference in background levels.
 *       - (2): A multiplicative scaling is applied to remove any
 *              difference in background levels.
 *       - (3): Images are scaled by their relative exposure times
 *              and an additive offset is applied to remove any
 *              remaining difference in background levels.
 *   - @b imcombine.xrej (bool, default false):
 *     If set, then an extra rejection cycle will be run.
 *   - @b imcombine.thresh (float, default 5.0):
 *     The rejection threshold for clipping during the combination in
 *     units of the background sigma.
 *   - @b scattered.enable (bool, default true):
 *     Enable scattered light removal.
 *   - @b scattered.nbsize (int, default 256):
 *     Size of the background smoothing cells for determining
 *     scattered light in pixels.
 *   - @b extract.width (int, default 5):
 *     Boxcar width in pixels over which the summation will be done
 *     at each spectral pixel during extraction.
 *   - @b arc.detthr (float):
 *     Detection threshold used to find emission features in the
 *     extracted arc spectrum in units of the background sigma.  A
 *     suitable default is selected automatically if the recipe
 *     parameter is not specified or the value given is less than
 *     zero.  The default value is 5.0.
 *   - @b arc.rejthr (float):
 *     Rejection threshold for clipping in the wavelength solution
 *     fit in terms of the rms.  A suitable default is selected
 *     automatically if the recipe parameter is not specified or the
 *     value given is less than zero.  The defaults are 5.0 for
 *     FIBRE_WAVE_DAY or FIBRE_WAVE_NIGHT, 3.0 for
 *     FIBRE_WAVE_SIMUARC.
 *   - @b arc.nord (int):
 *     The order of the polynomial used to fit the wavelength
 *     solution.  A suitable default is selected automatically if the
 *     recipe parameter is not specified or the value given is less
 *     than zero.  The defaults are 5 for FIBRE_WAVE_DAY or
 *     FIBRE_WAVE_SIMUARC, 1 for FIBRE_WAVE_NIGHT.
 *   - @b arc.matchwindow (float, default 1.5):
 *     Maximum deviation between the measured positions of an arc line
 *     between different fibres for it to be considered the same line,
 *     in units of the spectral resolution element (evaluated for a
 *     nominal resolving power of 6000 at 6000A for LRS, and 18000 at
 *     6000A for HRS, scaling with wavelength such that it remains
 *     constant in terms of resolving power).
 *   - @b arc.matchgrid (float):
 *     Search limit for initial grid search for simuarc solutions,
 *     in wavelength units (Angstroms).  If non-zero, an initial grid
 *     search out to +/- matchgrid sampled at intervals of half of
 *     matchwindow will be run prior to the final match against the
 *     line list to account for any large shifts of the wavelengths
 *     relative to the reference wavelength solution or wavelength
 *     map.  A value of zero disables the grid search.  Not used
 *     for master or OB-level FPE wavelength solutions due to the
 *     periodicity of the FPE lines and their narrow spacing in the
 *     blue.  For FIBRE_WAVE_SIMUARC, the default is 15.0.
 *   - @b reference (bool, default false):
 *     Emit a reference frame rather than a master frame.  Caution:
 *     the resulting frame must be inspected and approved before being
 *     allowed to enter the calibration database or used as a
 *     reference.  Can only be used when producing a master frame
 *     from daytime FPE wavecals.  Attempting to use for simuarc
 *     or night-time FPE frames will result in an error.
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - Exactly one of the following raw input types (required).
 *      Multiple frames of the same type may be given and will be
 *      combined prior to analysis, but different types cannot be
 *      mixed.
 *      - @b FIBRE_WAVE_DAY: Raw daytime Fabry-Perot Etalon (FPE)
 *           wavecal frame.
 *      - @b FIBRE_WAVE_NIGHT: Raw nighttime attached Fabry-Perot
 *           Etalon wavecal frame.
 *      - @b FIBRE_WAVE_SIMUARC: Raw daytime simuarc frame
 *           (simultaneous calibration fibres illuminated by ThAr arc
 *           lamp).
 *    - Exactly one of the following (required):
 *      - @b FPE_LINELIST: Fabry-Perot Etalon line list.
 *      - @b ARC_LINELIST: ThAr arc lamp line list.
 *    - @b MASTER_BIAS (optional): Master bias frame.
 *    - @b MASTER_DARK (optional): Master dark frame.
 *    - @b MASTER_DETECTOR_FLAT (optional): Master detector flat.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 *    - @b LINEARITY (optional): Master linearity table.
 *    - @b WAVE_MAP: Master wavelength map.  Required when processing
 *         FIBRE_WAVE_DAY or FIBRE_WAVE_SIMUARC.
 *    - @b FIBRE_TRACE (required): Trace table.
 *    - @b FIBRE_MASK (required): Fibre mask.
 *    - @b MASTER_WAVE: Master wavelength solution.  Required when
 *         processing FIBRE_WAVE_NIGHT for OB-level wavelength
 *         solutions.  Otherwise should not be specified.
 *    - @b REFERENCE_WAVE (optional): A library reference wavelength
 *         solution, used to initialise the new wavelength solution.
 *         If given, then the new wavelength solution will also 
 *         be compared to this reference and additional QC headers
 *         giving the results of the comparison are written to the
 *         output files.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b MASTER_WAVE: The master wavelength solution table.  This
 *          product is emitted when processing FIBRE_WAVE_DAY input
 *          files, if the recipe parameter reference is false.
 *     - @b REFERENCE_WAVE: The new reference wavelength solution
 *          table, emitted in place of MASTER_WAVE if the recipe
 *          parameter reference is true.
 *     - @b OB_WAVE: The OB-level wavelength solution correction
 *          table.  This product is emitted when processing
 *          FIBRE_WAVE_NIGHT input files.
 *     - @b SIMUARC_WAVE: The simuarc wavelength solution.  This
 *          product is emitted when processing FIBRE_WAVE_SIMUARC input
 *          files.
 *
 *   - The following intermediate product files are optional and are
 *     only emitted if keep=true is specified in the recipe parameters.
 *     - @b PROC_FIBRE_WAVE: Processed 2D arc image (prior to spectral
 *          extraction).  Can be used to assess the quality of the
 *          initial CCD processing, such as bias, dark, detector flat
 *          field correction, scattered light correction.
 *     - @b MASTER_ARC: Extracted FPE spectrum prior to wavelength
 *          calibration, when processing FIBRE_WAVE_DAY input files.
 *     - @b MASTER_WARC: Wavelength calibrated, extracted FPE spectrum,
 *          when processing FIBRE_WAVE_DAY input files.
 *     - @b OB_ARC: Extracted FPE spectrum prior to wavelength
 *          calibration, when processing FIBRE_WAVE_NIGHT input files.
 *     - @b OB_WARC: Wavelength calibrated, extracted FPE spectrum,
 *          when processing FIBRE_WAVE_NIGHT input files.
 *     - @b SIMUARC_ARC: Extracted ThAr arc lamp spectrum prior to
 *          wavelength calibration, when processing FIBRE_WAVE_SIMUARC
 *          input files. 
 *     - @b SIMUARC_WARC: Wavelength calibrated, extracted ThAr arc lamp
 *          spectrum, when processing FIBRE_WAVE_SIMUARC input files.
 *
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     master wavelength solution for LRS-A would be @c
 *     QMOST_MASTER_WAVE_LRS-A.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean residual background level
 *     in the illuminated region of the overscan subtracted frames
 *     that were combined.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined, as a measure of how
 *     consistent the background levels were.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>EXT FLUX MED</b> (ADU): The median of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX RMS</b> (ADU): The robustly-estimated RMS of
 *     average extracted flux in each fibre.
 *   - <b>EXT FLUX MIN</b> (ADU): The minimum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MINSPC</b>: The fibre with the minimum extracted
 *     flux.
 *   - <b>EXT FLUX MAX</b> (ADU): The maximum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MAXSPC</b>: The fibre with the maximum extracted
 *     flux.
 *   - <b>EXT SN MED</b>: The median of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN RMS</b>: The robustly-estimated RMS of the average
 *     signal to noise ratio in each fibre.
 *   - <b>EXT SN MIN</b>: The minimum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MINSPC</b>: The fibre with the minimum signal to
 *     noise ratio.
 *   - <b>EXT SN MAX</b>: The maximum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MAXSPC</b>: The fibre with the maximum signal to
 *     noise ratio.
 *   - <b>SCATTL MED</b> (ADU): The median scattered light level on
 *     the detector outside the region illuminated by the fibres.
 *   - <b>SCATTL RMS</b> (ADU): The robustly-estimated RMS of the
 *     scattered light level on the detector outside the region
 *     illuminated by the fibres.
 *   - <b>SCATTL RESID MED</b> (ADU): The median residual scattered
 *     light background remaining after subtraction.
 *   - <b>SCATTL RESID RMS</b> (ADU): The RMS residual of the
 *     scattered light background remaining after subtraction.
 *   - <b>WAVE LINES TOT</b>: The total number of arc lines measured
 *     across all fibres.
 *   - <b>WAVE FWHM MED</b> (A): Median FWHM of the arc lines.
 *   - <b>WAVE FWHM RMS</b> (A): Robustly-estimated RMS of the FWHM of
 *     the the arc lines.
 *   - <b>WAVE SPAN MAX</b>: The maximum over all the fibres of the
 *     fraction of the spectral axis spanned by the lines used in the
 *     wavelength solution.
 *   - <b>WAVE SPAN MAXSPC</b>: The fibre with the maximum fraction of the
 *     spectral axis spanned.
 *   - <b>WAVE SPAN MED</b>: The median over all the fibres of the
 *     fraction of the spectral axis spanned by the lines used in the
 *     wavelength solution.
 *   - <b>WAVE SPAN MIN</b>: The minimum over all the fibres of the
 *     fraction of the spectral axis spanned by the lines used in the
 *     wavelength solution.
 *   - <b>WAVE SPAN MINSPC</b>: The fibre with the minimum fraction of
 *     the spectral axis spanned.
 *   - <b>WAVE SPAN RMS</b>: The robustly-estimated RMS over all the
 *     fibres of the fraction of the spectral axis spanned by the
 *     lines used in the wavelength solution.
 *   - <b>WAVE LINES MIN</b>: The minimum number of arc lines measured
 *     in a fibre.
 *   - <b>WAVE LINES MINSPC</b>: The fibre with the minimum number of
 *     arc lines.
 *   - <b>WAVE LINES MAX</b>: The maximum number of arc lines measured
 *     in a fibre.
 *   - <b>WAVE LINES MAXSPC</b>: The fibre with the maximum number of
 *     arc lines.
 *   - <b>WAVE LINES MED</b>: The median number of arc lines measured
 *     in a fibre.
 *   - <b>WAVE RMS MIN</b> (A): The minimum RMS error in the arc fit
 *     in a fibre.
 *   - <b>WAVE RMS MINSPC</b>: The fibre with the minimum RMS error.
 *   - <b>WAVE RMS MAX</b> (A): The maximum RMS error in the arc fit
 *     in a fibre.
 *   - <b>WAVE RMS MAXSPC</b>: The fibre with the maximum RMS error.
 *   - <b>WAVE RMS MED</b> (A): The median RMS error in the arc fit in
 *     a fibre.
 *   - <b>WAVE OFFSET MED</b> (A): The median wavelength offset of the
 *     new wavelength solution relative to the reference over all of
 *     the fibres.
 *   - <b>WAVE OFFSET RMS</b> (A): The RMS of the wavelength offset of
 *     the new wavelength solution relative to the reference over the
 *     fibres, as a measure of how consistent the wavelength offset is
 *     from fibre to fibre.
 *   - <b>WAVE OFFSET MIN</b> (A): The minimum wavelength offset of the
 *     new wavelength solution relative to the reference over all of
 *     the fibres.
 *   - <b>WAVE OFFSET MINSPC</b>: The fibre with the minimum
 *     wavelength offset.
 *   - <b>WAVE OFFSET MAX</b> (A): The maximum wavelength offset of
 *     the new wavelength solution relative to the reference over all
 *     of the fibres.
 *   - <b>WAVE OFFSET MAXSPC</b>: The fibre with the maximum
 *     wavelength offset.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - No raw arc/FP frames in the input frameset.
 *   - No master arc line list available.
 *   - No master wavelength map available.
 *   - No master fibre trace or master fibre mask available.
 *   - When processing FIBRE_WAVE_NIGHT to make an OB-level solution,
 *     if the required master wavelength solution isn't available.
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 *   - If the trace table for an active detector is empty.
 * @par Non-Fatal Error Conditions:
 *   - No arc lines are detected.
 *   - No arc lines or not enough arc lines match to the reference
 *     line list, or all arc lines are rejected.
 *   - The wavelength solution (polynomial fit) fails.
 *   - Missing reference wavelength solution (no comparison is done).
 *   - Missing bad pixel mask (all pixels assumed to be good).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 * @par Functional Diagram for FIBRE_WAVE_DAY to MASTER_WAVE:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 * 
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc_and_combine [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> qmost_extract_tram;
 *   FIBRE_TRACE -> qmost_doarcs_ref;
 *   FPE_LINELIST -> qmost_doarcs_ref;
 *   WAVE_MAP -> qmost_doarcs_ref;
 *   REFERENCE_WAVE -> qmost_doarcs_ref [style="dashed"];
 *   REFERENCE_WAVE -> qmost_wave_qc [style="dashed"];
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     FIBRE_WAVE_DAY -> qmost_ccdproc_and_combine;
 *
 *     qmost_ccdproc_and_combine -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> qmost_extract_tram;
 *     qmost_extract_tram -> qmost_extract_qc;
 *     qmost_extract_qc -> qmost_doarcs_ref;
 *     qmost_doarcs_ref -> qmost_wave_qc;
 *     qmost_wave_qc -> qmost_rebin_spectra_off;
 *
 *     qmost_wave_qc -> MASTER_WAVE;
 *   }
 *
 *   qmost_scattered -> PROC_FIBRE_WAVE;
 *   qmost_extract_qc -> MASTER_ARC;
 *   qmost_rebin_spectra_off -> MASTER_WARC;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   qmost_extract_qc -> QC1;
 *   qmost_wave_qc -> QC1;
 * 
 *   FIBRE_WAVE_DAY [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   FPE_LINELIST [shape="box" fillcolor="#fff5ce"]
 *   WAVE_MAP [shape="box" fillcolor="#fff5ce"]
 *   REFERENCE_WAVE [shape="box" fillcolor="#fff5ce"]
 *   MASTER_WAVE [shape="box" fillcolor="#fff5ce"]
 *   PROC_FIBRE_WAVE [shape="box" fillcolor="#ffffff"]
 *   MASTER_ARC [shape="box" fillcolor="#ffffff"]
 *   MASTER_WARC [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 * @par Functional Diagram for FIBRE_WAVE_NIGHT to OB_WAVE:
 * @dot
 * digraph {
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 * 
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc_and_combine [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> qmost_extract_tram;
 *   FIBRE_TRACE -> qmost_doarcs_ob;
 *   FPE_LINELIST -> qmost_doarcs_ob;
 *   MASTER_WAVE -> qmost_doarcs_ob;
 *   REFERENCE_WAVE -> qmost_doarcs_ob [style="dashed"];
 *   REFERENCE_WAVE -> qmost_wave_qc [style="dashed"];
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     FIBRE_WAVE_NIGHT -> qmost_ccdproc_and_combine;
 *
 *     qmost_ccdproc_and_combine -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> qmost_extract_tram;
 *     qmost_extract_tram -> qmost_extract_qc;
 *     qmost_extract_qc -> qmost_doarcs_ob;
 *     qmost_doarcs_ob -> qmost_wave_qc;
 *     qmost_wave_qc -> qmost_rebin_spectra_off;
 *
 *     qmost_wave_qc -> OB_WAVE;
 *   }
 *
 *   qmost_scattered -> PROC_FIBRE_WAVE;
 *   qmost_extract_qc -> OB_ARC;
 *   qmost_rebin_spectra_off -> OB_WARC;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   qmost_extract_qc -> QC1;
 *   qmost_wave_qc -> QC1;
 * 
 *   FIBRE_WAVE_NIGHT [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   FPE_LINELIST [shape="box" fillcolor="#fff5ce"]
 *   MASTER_WAVE [shape="box" fillcolor="#fff5ce"]
 *   REFERENCE_WAVE [shape="box" fillcolor="#fff5ce"]
 *   OB_WAVE [shape="box" fillcolor="#fff5ce"]
 *   PROC_FIBRE_WAVE [shape="box" fillcolor="#ffffff"]
 *   OB_ARC [shape="box" fillcolor="#ffffff"]
 *   OB_WARC [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 * @par Functional Diagram for FIBRE_WAVE_SIMUARC to SIMUARC_WAVE:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 * 
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc_and_combine [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> qmost_extract_tram;
 *   FIBRE_TRACE -> qmost_doarcs_ref;
 *   ARC_LINELIST -> qmost_doarcs_ref;
 *   WAVE_MAP -> qmost_doarcs_ref;
 *   REFERENCE_WAVE -> qmost_doarcs_ref [style="dashed"];
 *   REFERENCE_WAVE -> qmost_wave_qc [style="dashed"];
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     FIBRE_WAVE_SIMUARC -> qmost_ccdproc_and_combine;
 *
 *     qmost_ccdproc_and_combine -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> qmost_extract_tram;
 *     qmost_extract_tram -> qmost_extract_qc;
 *     qmost_extract_qc -> qmost_doarcs_ref;
 *     qmost_doarcs_ref -> qmost_wave_qc;
 *     qmost_wave_qc -> qmost_rebin_spectra_off;
 *
 *     qmost_wave_qc -> SIMUARC_WAVE;
 *   }
 *
 *   qmost_scattered -> PROC_FIBRE_WAVE;
 *   qmost_extract_qc -> SIMUARC_ARC;
 *   qmost_rebin_spectra_off -> SIMUARC_WARC;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   qmost_extract_qc -> QC1;
 *   qmost_wave_qc -> QC1;
 * 
 *   FIBRE_WAVE_SIMUARC [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   ARC_LINELIST [shape="box" fillcolor="#fff5ce"]
 *   WAVE_MAP [shape="box" fillcolor="#fff5ce"]
 *   REFERENCE_WAVE [shape="box" fillcolor="#fff5ce"]
 *   SIMUARC_WAVE [shape="box" fillcolor="#fff5ce"]
 *   PROC_FIBRE_WAVE [shape="box" fillcolor="#ffffff"]
 *   SIMUARC_ARC [shape="box" fillcolor="#ffffff"]
 *   SIMUARC_WARC [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_arc_analyse(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    const cpl_frame *master_bias_frame = NULL;
    const cpl_frame *master_dark_frame = NULL;
    const cpl_frame *master_detflat_frame = NULL;
    const cpl_frame *master_bpm_frame = NULL;
    const cpl_frame *master_lintab_frame = NULL;
    const cpl_frame *master_wave_map_frame = NULL;
    const cpl_frame *linelist_frame = NULL;
    const cpl_frame *fibre_trace_frame = NULL;
    const cpl_frame *fibre_mask_frame = NULL;
    const cpl_frame *master_wave_frame = NULL;
    const cpl_frame *ref_wave_frame = NULL;

    const cpl_frame *proc_arc_frame = NULL;
    const char *proc_arc_filename = NULL;

    const cpl_frame *extracted_arc_frame = NULL;
    const char *extracted_arc_filename = NULL;
    char *extracted_arc_tag = NULL;

    const cpl_frame *wave_frame = NULL;
    const char *wave_filename = NULL;
    char *wave_tag = NULL;

    const cpl_frame *warc_frame = NULL;
    const char *warc_filename = NULL;
    char *warc_tag = NULL;

    cpl_frameset *arcs_raw = NULL;
    int nin, iin, nff;
    int nday, nnight, nsimuarc;

    const cpl_frame *this_frame;
    const char *this_tag;

    cpl_frame *arc_ref;
    const char *arc_ref_filename;

    const cpl_parameter *par;
    int level = 1;
    int keep = 0;
    const char *swapx_table = QMOST_DEFAULT_SWAPX_TABLE;
    int combtype = QMOST_MEANCALC;
    int scaletype = 0;
    int xrej = 0;
    double combthresh = 5.0;
    int scattered_enable = 1;
    int scattered_nbsize = 256;
    int extract_width = 5;
    float detthr;  /* set later once we know input frame type */
    float rejthr;  /* set later once we know input frame type */
    int nord;  /* set later once we know input frame type */
    float matchwindow = 1.5;
    float matchgrid;  /* set later once we know input frame type */
    float fpecoef = 0.34;

    double dtmp;
    int itmp;

    int fpe;
    float fpecorr = 0.0;

    cpl_size raw_extension;
    int spec, arm, detlive;
    const char *arm_extname;
    char *extname = NULL;

    cpl_mask *master_bpm = NULL;
    cpl_image *master_bias_img = NULL;
    cpl_image *master_bias_var = NULL;
    cpl_image *master_dark_img = NULL;
    cpl_image *master_dark_var = NULL;
    cpl_image *master_detflat_img = NULL;
    cpl_image *master_detflat_var = NULL;
    cpl_table *master_lintab = NULL;

    cpl_image *arc_img = NULL;
    cpl_image *arc_var = NULL;
    cpl_propertylist *arc_hdr = NULL;
    cpl_propertylist *linelist_hdr = NULL;

    cpl_propertylist *qclist = NULL;

    cpl_table *fibinfo_tbl = NULL;
    cpl_propertylist *fibinfo_hdr = NULL;

    cpl_table *trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;
    cpl_mask *fibre_mask = NULL;

    int irow, nrows, nrowsf, nrowst, fibuse, isnull;

    cpl_image *master_wave_map = NULL;
    cpl_table *linelist_tbl = NULL;
    cpl_table *master_wave_tbl = NULL;
    cpl_propertylist *master_wave_hdr = NULL;
    cpl_table *ref_wave_tbl = NULL;
    cpl_propertylist *ref_wave_hdr = NULL;

    cpl_table *wave_tbl = NULL;

    cpl_image *extracted_arc_img = NULL;
    cpl_image *extracted_arc_var = NULL;

    int nfib;
    double *waveoff = NULL;
    double *veloff = NULL;
    double minwave, maxwave, dlam;

    cpl_image *warc_img = NULL;
    cpl_image *warc_var = NULL;
    cpl_propertylist *warc_hdr = NULL;

#undef TIDY
#define TIDY                                            \
    if(arcs_raw) {                                      \
        cpl_frameset_delete(arcs_raw);                  \
        arcs_raw = NULL;                                \
    }                                                   \
    if(fibinfo_tbl) {                                   \
        cpl_table_delete(fibinfo_tbl);                  \
        fibinfo_tbl = NULL;                             \
    }                                                   \
    if(fibinfo_hdr) {                                   \
        cpl_propertylist_delete(fibinfo_hdr);           \
        fibinfo_hdr = NULL;                             \
    }                                                   \
    if(master_bpm) {                                    \
        cpl_mask_delete(master_bpm);                    \
        master_bpm = NULL;                              \
    }                                                   \
    if(master_bias_img) {                               \
        cpl_image_delete(master_bias_img);              \
        master_bias_img = NULL;                         \
    }                                                   \
    if(master_bias_var) {                               \
        cpl_image_delete(master_bias_var);              \
        master_bias_var = NULL;                         \
    }                                                   \
    if(master_dark_img) {                               \
        cpl_image_delete(master_dark_img);              \
        master_dark_img = NULL;                         \
    }                                                   \
    if(master_dark_var) {                               \
        cpl_image_delete(master_dark_var);              \
        master_dark_var = NULL;                         \
    }                                                   \
    if(master_detflat_img) {                            \
        cpl_image_delete(master_detflat_img);           \
        master_detflat_img = NULL;                      \
    }                                                   \
    if(master_detflat_var) {                            \
        cpl_image_delete(master_detflat_var);           \
        master_detflat_var = NULL;                      \
    }                                                   \
    if(master_lintab) {                                 \
        cpl_table_delete(master_lintab);                \
        master_lintab = NULL;                           \
    }                                                   \
    if(arc_img) {                                       \
        cpl_image_delete(arc_img);                      \
        arc_img = NULL;                                 \
    }                                                   \
    if(arc_var) {                                       \
        cpl_image_delete(arc_var);                      \
        arc_var = NULL;                                 \
    }                                                   \
    if(arc_hdr) {                                       \
        cpl_propertylist_delete(arc_hdr);               \
        arc_hdr = NULL;                                 \
    }                                                   \
    if(linelist_hdr) {                                  \
        cpl_propertylist_delete(linelist_hdr);          \
        linelist_hdr = NULL;                            \
    }                                                   \
    if(qclist) {                                        \
        cpl_propertylist_delete(qclist);                \
        qclist = NULL;                                  \
    }                                                   \
    if(trace_tbl) {                                     \
        cpl_table_delete(trace_tbl);                    \
        trace_tbl = NULL;                               \
    }                                                   \
    if(trace_hdr) {                                     \
        cpl_propertylist_delete(trace_hdr);             \
        trace_hdr = NULL;                               \
    }                                                   \
    if(fibre_mask) {                                    \
        cpl_mask_delete(fibre_mask);                    \
        fibre_mask = NULL;                              \
    }                                                   \
    if(master_wave_map) {                               \
        cpl_image_delete(master_wave_map);              \
        master_wave_map = NULL;                         \
    }                                                   \
    if(linelist_tbl) {                                  \
        cpl_table_delete(linelist_tbl);                 \
        linelist_tbl = NULL;                            \
    }                                                   \
    if(master_wave_tbl) {                               \
        cpl_table_delete(master_wave_tbl);              \
        master_wave_tbl = NULL;                         \
    }                                                   \
    if(master_wave_hdr) {                               \
        cpl_propertylist_delete(master_wave_hdr);       \
        master_wave_hdr = NULL;                         \
    }                                                   \
    if(ref_wave_tbl) {                                  \
        cpl_table_delete(ref_wave_tbl);                 \
        ref_wave_tbl = NULL;                            \
    }                                                   \
    if(ref_wave_hdr) {                                  \
        cpl_propertylist_delete(ref_wave_hdr);          \
        ref_wave_hdr = NULL;                            \
    }                                                   \
    if(wave_tbl) {                                      \
        cpl_table_delete(wave_tbl);                     \
        wave_tbl = NULL;                                \
    }                                                   \
    if(extracted_arc_img) {                             \
        cpl_image_delete(extracted_arc_img);            \
        extracted_arc_img = NULL;                       \
    }                                                   \
    if(extracted_arc_var) {                             \
        cpl_image_delete(extracted_arc_var);            \
        extracted_arc_var = NULL;                       \
    }                                                   \
    if(waveoff) {                                       \
        cpl_free(waveoff);                              \
        waveoff = NULL;                                 \
    }                                                   \
    if(veloff) {                                        \
        cpl_free(veloff);                               \
        veloff = NULL;                                  \
    }                                                   \
    if(extname) {                                       \
        cpl_free(extname);                              \
        extname = NULL;                                 \
    }                                                   \
    if(warc_img) {                                      \
        cpl_image_delete(warc_img);                     \
        warc_img = NULL;                                \
    }                                                   \
    if(warc_var) {                                      \
        cpl_image_delete(warc_var);                     \
        warc_var = NULL;                                \
    }                                                   \
    if(warc_hdr) {                                      \
        cpl_propertylist_delete(warc_hdr);              \
        warc_hdr = NULL;                                \
    }

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Master detector-level calibrations, all optional */
    master_bias_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_BIAS);
    master_dark_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_DARK);
    master_detflat_frame = cpl_frameset_find_const(frameset,
                                                   QMOST_PRO_MASTER_DETECTOR_FLAT);
    master_bpm_frame = cpl_frameset_find_const(frameset,
                                               QMOST_CALIB_MASTER_BPM);
    master_lintab_frame = cpl_frameset_find_const(frameset,
                                                  QMOST_PRO_LINEARITY);

    /* Master wavelength map, optional */
    master_wave_map_frame = cpl_frameset_find_const(frameset,
                                                    QMOST_CALIB_MASTER_WAVE_MAP);

    /* Linelist, required */
    linelist_frame = cpl_frameset_find_const(frameset,
                                             QMOST_CALIB_ARC_LINELIST);
    fpe = 0;
    if(linelist_frame == NULL) {
        linelist_frame = cpl_frameset_find_const(frameset,
                                             QMOST_PRO_FPE_LINELIST);
        fpe = 1;
    }
    if(linelist_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged "
                                     "%s or %s",
                                     QMOST_CALIB_ARC_LINELIST,
                                     QMOST_PRO_FPE_LINELIST);
    }    

    /* Trace table and fibre mask, required */
    fibre_trace_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_FIBRE_TRACE);
    if(fibre_trace_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_TRACE);
    }

    fibre_mask_frame = cpl_frameset_find_const(frameset,
                                               QMOST_PRO_FIBRE_MASK);
    if(fibre_mask_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_MASK);
    }

    /* Master wavelength solution, optional */
    master_wave_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_WAVE);

    /* Reference wavelength solution, optional */
    ref_wave_frame = cpl_frameset_find_const(frameset,
                                             QMOST_CALIB_REFERENCE_WAVE);
    

    /* Create frameset of input raw arcs */
    arcs_raw = cpl_frameset_new();  /* can't fail? */

    nin = cpl_frameset_get_size(frameset);

    nday = 0;
    nnight = 0;
    nsimuarc = 0;

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_FIBRE_WAVE_DAY)) {
            cpl_frameset_insert(arcs_raw, cpl_frame_duplicate(this_frame));
            nday++;
        }
        else if(!strcmp(this_tag, QMOST_RAW_FIBRE_WAVE_NIGHT)) {
            cpl_frameset_insert(arcs_raw, cpl_frame_duplicate(this_frame));
            nnight++;
        }
        else if(!strcmp(this_tag, QMOST_RAW_FIBRE_WAVE_SIMUARC)) {
            cpl_frameset_insert(arcs_raw, cpl_frame_duplicate(this_frame));
            nsimuarc++;
        }
    }

    /* Check we have some */
    nff = cpl_frameset_get_size(arcs_raw);
    if(nff < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged "
                                     "%s, %s or %s",
                                     QMOST_RAW_FIBRE_WAVE_DAY,
                                     QMOST_RAW_FIBRE_WAVE_NIGHT,
                                     QMOST_RAW_FIBRE_WAVE_SIMUARC);
    }

    /* Decide what to do */
    if(nday > 0 && (nnight == 0 && nsimuarc == 0)) {
        /* Making master, wave map required */
        if(master_wave_map_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "SOF does not have an image tagged "
                                         "%s",
                                         QMOST_CALIB_MASTER_WAVE_MAP);
        }

        extracted_arc_tag = QMOST_PRO_MASTER_ARC;
        wave_tag = QMOST_PRO_MASTER_WAVE;
        warc_tag = QMOST_PRO_MASTER_WARC;

        /* Set suitable defaults for parameters for master wave */
        detthr = 5.0;
        rejthr = 5.0;
        nord = 5;
        matchgrid = 0;
    }
    else if(nnight > 0 && (nday == 0 && nsimuarc == 0)) {
        /* Making OB, master solution required */
        if(master_wave_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "SOF does not have an image tagged "
                                         "%s",
                                         QMOST_PRO_MASTER_WAVE);
        }

        extracted_arc_tag = QMOST_PRO_OB_ARC;
        wave_tag = QMOST_PRO_OB_WAVE;
        warc_tag = QMOST_PRO_OB_WARC;

        /* Set suitable defaults for parameters for OB wave */
        detthr = 5.0;
        rejthr = 5.0;
        nord = 1;
        matchgrid = 0;
    }
    else if(nsimuarc > 0 && (nday == 0 && nnight == 0)) {
        /* simuarc, wave map required */
        if(master_wave_map_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "SOF does not have an image tagged "
                                         "%s",
                                         QMOST_CALIB_MASTER_WAVE_MAP);
        }

        extracted_arc_tag = QMOST_PRO_SIMUARC_ARC;
        wave_tag = QMOST_PRO_SIMUARC_WAVE;
        warc_tag = QMOST_PRO_SIMUARC_WARC;

        /* Set suitable defaults for parameters for simuarc */
        detthr = 5.0;
        rejthr = 3.0;
        nord = 5;
        matchgrid = 15.0;
    }
    else {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "SOF must contain only one of "
                                     "%s, %s or %s",
                                     QMOST_RAW_FIBRE_WAVE_DAY,
                                     QMOST_RAW_FIBRE_WAVE_NIGHT,
                                     QMOST_RAW_FIBRE_WAVE_SIMUARC);
    }

    /* Retrieve QC level parameter */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".level");
    if(par != NULL)
        level = cpl_parameter_get_int(par);

    if(level < 0 || level > 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "unknown QC level: %d", level);
    }

    /* Switch to save processed products */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".keep");
    if(par != NULL)
        keep = cpl_parameter_get_bool(par);

    /* Retrieve ccdproc parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".ccdproc.swapx");
    if(par != NULL)
        swapx_table = cpl_parameter_get_string(par);

    /* Retrieve imcombine parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.combtype");
    if(par != NULL)
        combtype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.scaletype");
    if(par != NULL)
        scaletype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.xrej");
    if(par != NULL)
        xrej = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.thresh");
    if(par != NULL)
        combthresh = cpl_parameter_get_double(par);

    /* Retrieve scattered light removal parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.enable");
    if(par != NULL)
        scattered_enable = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.nbsize");
    if(par != NULL)
        scattered_nbsize = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.width");
    if(par != NULL)
        extract_width = cpl_parameter_get_int(par);

    /* Retrieve doarcs parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".arc.detthr");
    if(par != NULL) {
        dtmp = cpl_parameter_get_double(par);
        if(dtmp >= 0) {
            detthr = dtmp;
        }
    }

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".arc.rejthr");
    if(par != NULL) {
        dtmp = cpl_parameter_get_double(par);
        if(dtmp >= 0) {
            rejthr = dtmp;
        }
    }

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".arc.nord");
    if(par != NULL) {
        itmp = cpl_parameter_get_int(par);
        if(itmp >= 0) {
            nord = itmp;
        }
    }

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".arc.matchwindow");
    if(par != NULL) {
        matchwindow = cpl_parameter_get_double(par);
    }

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".arc.matchgrid");
    if(par != NULL) {
        dtmp = cpl_parameter_get_double(par);
        if(dtmp >= 0) {
            matchgrid = dtmp;
        }
    }

    /* FPE pressure correction coefficient */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".fpe.prescoef");
    if(par != NULL) {
        fpecoef = cpl_parameter_get_double(par);
    }
    
    /* Reference parameter */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".reference");
    if(par != NULL) {
        if(cpl_parameter_get_bool(par)) {
            /* Must be making master */
            if(nday > 0 && (nnight == 0 && nsimuarc == 0)) {
                /* Change tag to reference */
                wave_tag = QMOST_CALIB_REFERENCE_WAVE;
            }
            else {
                /* Raise error */
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             CPL_ERROR_INCOMPATIBLE_INPUT,
                                             "reference wavelength file "
                                             "can only be made from "
                                             "master, not OB or simuarc");
            }
        }
    }

    /* Get first arc to use as basis for output */
    arc_ref = cpl_frameset_get_position(arcs_raw, 0);
    if(arc_ref == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get first arc "
                                     "from SOF");
    }

    /* Get filename of first arc */
    arc_ref_filename = cpl_frame_get_filename(arc_ref);
    if(arc_ref_filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename for "
                                     "first arc frame");
    }

    /* Extract the primary FITS header of the reference */
    arc_hdr = cpl_propertylist_load(arc_ref_filename,
                                    0);
    if(arc_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FITS primary header "
                                     "from first input file");
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(arc_hdr,
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    if(fpe) {
        /* Extract the primary FITS header of the line list */
        linelist_hdr = cpl_propertylist_load(
            cpl_frame_get_filename(linelist_frame),
            0);
        if(linelist_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS primary header "
                                         "from linelist file");
        }

        /* Compute FPE pressure correction */
        if(qmost_fpcorr(arc_hdr, linelist_hdr, fpecoef,
                        &fpecorr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine FPE pressure "
                                         "correction");
        }

        cpl_propertylist_delete(linelist_hdr);
        linelist_hdr = NULL;
    }

    cpl_propertylist_delete(arc_hdr);
    arc_hdr = NULL;

    /* Get fibinfo */
    if(qmost_fibtabload(arc_ref,
                        spec,
                        &fibinfo_tbl,
                        &fibinfo_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FIBINFO table "
                                     "from first input file");
    }

    /* Create outputs */
    wave_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        wave_tag,
        CPL_FRAME_TYPE_TABLE,
        NULL);
    if(wave_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     wave_tag);
    }

    wave_filename = cpl_frame_get_filename(wave_frame);

    if(keep) {
        proc_arc_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PROC_FIBRE_WAVE,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(proc_arc_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PROC_FIBRE_WAVE);
        }

        proc_arc_filename = cpl_frame_get_filename(proc_arc_frame);

        extracted_arc_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            extracted_arc_tag,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(extracted_arc_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         extracted_arc_tag);
        }
        
        extracted_arc_filename = cpl_frame_get_filename(extracted_arc_frame);

        warc_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            warc_tag,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(warc_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         warc_tag);
        }

        warc_filename = cpl_frame_get_filename(warc_frame);
    }

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Extract the FITS header of the reference */
        arc_hdr = cpl_propertylist_load(arc_ref_filename,
                                        raw_extension);
        if(arc_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS extension "
                                         "%lld header from first input file",
                                         raw_extension);
        }

        /* Check the reference to find out what arm we're processing */
        if(qmost_pfits_get_arm(arc_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine EXTNAME "
                                         "for arm %d", arm);
        }

        /* Is detector live? */
        if(qmost_pfits_get_detlive(arc_hdr,
                                   &detlive) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read live flag for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(!detlive) {
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %lld = %s",
                         raw_extension, arm_extname);

            if(proc_arc_frame != NULL) {
                if(qmost_dfs_save_image_and_var(proc_arc_frame,
                                                arc_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "stacked arc frame %s[%s]",
                                                 proc_arc_filename,
                                                 arm_extname);
                }
            }

            if(extracted_arc_frame != NULL) {
                if(qmost_dfs_save_image_and_var(extracted_arc_frame,
                                                arc_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "extracted arc %s[%s]",
                                                 extracted_arc_filename,
                                                 arm_extname);
                }
            }

            extname = cpl_sprintf("wave_%s", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format wave "
                                             "EXTNAME string");
            }

            if(qmost_dfs_save_table_extension(wave_frame,
                                              arc_hdr,
                                              extname,
                                              NULL,
                                              NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy "
                                             "wave table %s[%s]",
                                             wave_filename,
                                             arm_extname);
            }

            cpl_free(extname);
            extname = NULL;
        
            if(warc_frame != NULL) {
                /* Save wavelength calibrated arc */
                if(qmost_dfs_save_image_and_var(warc_frame,
                                                arc_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "wavelength calibrated "
                                                 "arc %s[%s]",
                                                 warc_filename,
                                                 arm_extname);
                }
            }

            cpl_propertylist_delete(arc_hdr);
            arc_hdr = NULL;

            continue;
        }

        /* Load all masters using the correct EXTNAME for the arm
           we're processing in case the mapping of extension number to
           arm could have changed. */
        if(master_bias_frame != NULL) {
            if(qmost_load_master_image_and_var(master_bias_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_bias_img,
                                               &master_bias_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master bias "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_bias_img = NULL;
            master_bias_var = NULL;
        }
    
        if(master_dark_frame != NULL) {
            if(qmost_load_master_image_and_var(master_dark_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_dark_img,
                                               &master_dark_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master dark "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_dark_img = NULL;
            master_dark_var = NULL;
        }

        if(master_detflat_frame != NULL) {
            if(qmost_load_master_image_and_var(master_detflat_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_detflat_img,
                                               &master_detflat_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master detector "
                                             "flat frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_detflat_img = NULL;
            master_detflat_var = NULL;
        }

        if(master_bpm_frame != NULL) {
            if(qmost_load_master_mask(master_bpm_frame,
                                      arm_extname,
                                      &master_bpm,
                                      NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM");
            }
        }

        if(master_lintab_frame != NULL) {
            if(qmost_load_master_table(master_lintab_frame,
                                       arm_extname,
                                       &master_lintab,
                                       NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master "
                                             "linearity table");
            }
        }

        /* Create QC parameter list */
        qclist = cpl_propertylist_new();

        /* CCD processing and stacking */
        if(qmost_ccdproc_and_combine(arcs_raw,
                                     raw_extension,
                                     master_bpm,
                                     master_bias_img,
                                     master_bias_var,
                                     master_dark_img,
                                     master_dark_var,
                                     master_detflat_img,
                                     master_detflat_var,
                                     1,
                                     swapx_table,
                                     1,
                                     master_lintab,
                                     QMOST_LIN_AFTER_BIAS,
                                     combtype,
                                     scaletype,
                                     1,  /* fibre spectra */
                                     xrej,
                                     combthresh,
                                     &arc_img,
                                     &arc_var,
                                     qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "basic CCD processing failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(master_bias_img != NULL) {
            cpl_image_delete(master_bias_img);
            master_bias_img = NULL;
        }

        if(master_bias_var != NULL) {
            cpl_image_delete(master_bias_var);
            master_bias_var = NULL;
        }

        if(master_dark_img != NULL) {
            cpl_image_delete(master_dark_img);
            master_dark_img = NULL;
        }

        if(master_dark_var != NULL) {
            cpl_image_delete(master_dark_var);
            master_dark_var = NULL;
        }

        if(master_detflat_img != NULL) {
            cpl_image_delete(master_detflat_img);
            master_detflat_img = NULL;
        }

        if(master_detflat_var != NULL) {
            cpl_image_delete(master_detflat_var);
            master_detflat_var = NULL;
        }

        if(master_bpm != NULL) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }

        if(master_lintab != NULL) {
            cpl_table_delete(master_lintab);
            master_lintab = NULL;
        }

        /* Get trace table and fibre mask */
        extname = cpl_sprintf("trace_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format trace "
                                         "EXTNAME string");
        }
        
        if(qmost_load_master_table(fibre_trace_frame,
                                   extname,
                                   &trace_tbl,
                                   &trace_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load trace");
        }
        
        cpl_free(extname);
        extname = NULL;

        extname = cpl_sprintf("mask_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format fibre mask "
                                         "EXTNAME string");
        }

        if(qmost_load_master_mask(fibre_mask_frame,
                                  extname,
                                  &fibre_mask,
                                  NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load fibre mask");
        }
        
        cpl_free(extname);
        extname = NULL;

        /* Populate scattered light QC before we remove it */
        if(qmost_scattered_qc(arc_img,
                              qclist,
                              fibre_mask,
                              qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "scattered light QC failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(scattered_enable) {
            /* Remove scattered light */
            if(qmost_scattered(arc_img,
                               qclist,
                               fibre_mask,
                               scattered_nbsize,
                               1) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "removal of scattered light "
                                             "with box size %d failed "
                                             "for extension %lld",
                                             scattered_nbsize,
                                             raw_extension);
            }
        }
        
        /* Save stacked, scattered light corrected arc if requested */
        if(proc_arc_frame != NULL) {
            if(qmost_dfs_save_image_and_var(proc_arc_frame,
                                            arc_hdr,
                                            arm_extname,
                                            qclist,
                                            arc_img,
                                            arc_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save stacked arc "
                                             "frame %s[%s]",
                                             proc_arc_filename,
                                             arm_extname);
            }
        }

        /* If this is a simuarc, we should only analyse the simucal
         * fibres.  Using the FIBINFO table, unset the live flag in
         * the trace table for anything else.  If there's no FIBINFO
         * table assume all fibres are illuminated by ThAr such as in
         * lab data. */
        if(nsimuarc > 0 && fibinfo_tbl != NULL) {
            nrowsf = cpl_table_get_nrow(fibinfo_tbl);
            nrowst = cpl_table_get_nrow(trace_tbl);

            nrows = qmost_min(nrowsf, nrowst);

            for(irow = 0; irow < nrows; irow++) {
                fibuse = cpl_table_get_int(fibinfo_tbl,
                                           "FIB_USE",
                                           irow,
                                           &isnull);
                if(isnull < 0) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "could not read FIB_USE "
                                                 "for FIBINFO row %d",
                                                 irow+1);
                }
                else if(isnull == 0 && fibuse > 0) {
                    if(cpl_table_set_int(trace_tbl,
                                         "fiblive",
                                         irow,
                                         0) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "could not set fiblive "
                                                     "in trace table row %d",
                                                     irow+1);
                    }
                }
            }
        }

        /* Extract arc */
        if(qmost_extract_tram(arc_img,
                              arc_var,
                              qclist,
                              trace_tbl,
                              trace_hdr,
                              extract_width,
                              &extracted_arc_img,
                              &extracted_arc_var,
                              qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "spectral extraction failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        /* Populate QC */
        if(qmost_extract_qc(extracted_arc_img,
                            extracted_arc_var,
                            qclist,
                            NULL,
                            arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "extracted spectrum QC failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        nfib = cpl_image_get_size_y(extracted_arc_img);

        if(extracted_arc_frame != NULL) {
            /* Save extracted arc */
            if(qmost_dfs_save_image_and_var(extracted_arc_frame,
                                            arc_hdr,
                                            arm_extname,
                                            qclist,
                                            extracted_arc_img,
                                            extracted_arc_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save extracted "
                                             "arc %s[%s]",
                                             extracted_arc_filename,
                                             arm_extname);
            }
        }

        /* Load what we need for wavelength solution */
        extname = cpl_sprintf("linelist_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format linelist "
                                         "EXTNAME string");
        }

        if(qmost_load_master_table(linelist_frame,
                                   extname,
                                   &linelist_tbl,
                                   NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load linelist");
        }

        cpl_free(extname);
        extname = NULL;

        if(fpe) {
            /* Apply pressure correction */
            if(cpl_table_multiply_scalar(
                   linelist_tbl,
                   "Wavelength",
                   1.0 + fpecorr / QMOST_SPEEDOFLIGHT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't apply FPE pressure "
                                             "correction to linelist");
            }

            /* Record what we did */
            cpl_propertylist_update_float(qclist,
                                          "ESO DRS FPE CORR",
                                          fpecorr);
            cpl_propertylist_set_comment(qclist,
                                         "ESO DRS FPE CORR",
                                         "[km/s] Correction applied for "
                                         "FPE pressure variations");
        }

        if(master_wave_frame != NULL) {
            /* Analyse extracted arc to produce OB wavelength correction */
            extname = cpl_sprintf("wave_%s", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format wave "
                                             "EXTNAME string");
            }
            
            if(qmost_load_master_table(master_wave_frame,
                                       extname,
                                       &master_wave_tbl,
                                       &master_wave_hdr) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master "
                                             "wavelength solution");
            }
            
            cpl_free(extname);
            extname = NULL;

            if(qmost_doarcs_ob(extracted_arc_img,
                               qclist,
                               master_wave_tbl,
                               master_wave_hdr,
                               linelist_tbl,
                               trace_tbl,
                               trace_hdr,
                               detthr,
                               rejthr,
                               nord,
                               spec == QMOST_SPEC_HRS ?
                               matchwindow / 3.0 :
                               matchwindow,
                               0,
                               &wave_tbl) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "OB arc line analysis failed "
                                             "for extension %lld",
                                             raw_extension);
            }
        }
        else {
            /* Analyse extracted arc to produce master wavelength solution */
            if(qmost_load_master_image(master_wave_map_frame,
                                       arm_extname,
                                       CPL_TYPE_FLOAT,
                                       &master_wave_map,
                                       NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master wave map");
            }
            
            if(ref_wave_frame != NULL) {
                extname = cpl_sprintf("wave_%s", arm_extname);
                if(extname == NULL) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "could not format wave "
                                                 "EXTNAME string");
                }
                
                if(qmost_load_master_table(ref_wave_frame,
                                           extname,
                                           &ref_wave_tbl,
                                           &ref_wave_hdr) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't load reference "
                                                 "wavelength solution");
                }
                
                cpl_free(extname);
                extname = NULL;
            }
            
            if(qmost_doarcs_ref(extracted_arc_img,
                                qclist,
                                master_wave_map,
                                ref_wave_tbl,
                                ref_wave_hdr,
                                linelist_tbl,
                                trace_tbl,
                                trace_hdr,
                                detthr,
                                rejthr,
                                nord,
                                spec == QMOST_SPEC_HRS ?
                                matchwindow / 3.0 :
                                matchwindow,
                                matchgrid,
                                &wave_tbl) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "master arc line analysis "
                                             "failed for extension %lld",
                                             raw_extension);
            }

            cpl_image_delete(master_wave_map);
            master_wave_map = NULL;
        }

        cpl_table_delete(linelist_tbl);
        linelist_tbl = NULL;

        cpl_table_delete(trace_tbl);
        trace_tbl = NULL;

        cpl_propertylist_delete(trace_hdr);
        trace_hdr = NULL;

        cpl_mask_delete(fibre_mask);
        fibre_mask = NULL;

        /* Do QC on wavelength solution */
        if(qmost_wave_qc(wave_tbl,
                         ref_wave_tbl,
                         master_wave_tbl ? 1 : 0,
                         qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "wavelength solution QC failed for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(ref_wave_tbl != NULL) {
            cpl_table_delete(ref_wave_tbl);
            ref_wave_tbl = NULL;
        }

        if(ref_wave_hdr != NULL) {
            cpl_propertylist_delete(ref_wave_hdr);
            ref_wave_hdr = NULL;
        }

        /* and populate individual fibre QC in FIBINFO */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtab_arcqc(fibinfo_tbl,
                                  arm,
                                  master_wave_tbl ?
                                  master_wave_tbl : wave_tbl,
                                  master_wave_tbl ?
                                  wave_tbl : NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             cpl_error_get_code(),
                                             "failed to populate QC "
                                             "in FIBINFO for "
                                             "extension %lld",
                                             raw_extension);
            }
        }

        /* Save wave table */
        extname = cpl_sprintf("wave_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format wave "
                                         "EXTNAME string");
        }

        if(qmost_dfs_save_table_extension(wave_frame,
                                          arc_hdr,
                                          extname,
                                          qclist,
                                          wave_tbl) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save wave table "
                                         "%s[%s]",
                                         wave_filename,
                                         arm_extname);
        }

        cpl_free(extname);
        extname = NULL;

        /* Wavelength calibrate arc */
        if(qmost_get_rebin_params(spec, arm,
                                  &minwave,
                                  &maxwave,
                                  &dlam) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get wavelength "
                                         "calibration rebinning "
                                         "parameters");
        }

        waveoff = cpl_calloc(nfib, sizeof(double));
        veloff = cpl_calloc(nfib, sizeof(double));
        
        warc_hdr = cpl_propertylist_new();

        if(qmost_rebin_spectra_off(extracted_arc_img,
                                   extracted_arc_var,
                                   qclist,
                                   master_wave_tbl ?
                                   master_wave_tbl : wave_tbl,
                                   master_wave_tbl ?
                                   master_wave_hdr : qclist,
                                   master_wave_tbl ?
                                   wave_tbl : NULL,
                                   master_wave_tbl ?
                                   qclist : NULL,
                                   NULL,
                                   NULL,
                                   waveoff,
                                   veloff,
                                   nfib,
                                   minwave,
                                   maxwave,
                                   dlam,
                                   &warc_img,
                                   &warc_var,
                                   warc_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "wavelength calibration of "
                                         "extracted arc failed for "
                                         "extension %lld",
                                         raw_extension);
        }

        cpl_free(waveoff);
        waveoff = NULL;

        cpl_free(veloff);
        veloff = NULL;
        
        /* Add QC to arc header */
        cpl_propertylist_copy_property_regexp(warc_hdr, qclist, ".*", 0);

        if(warc_frame != NULL) {
            /* Save wavelength calibrated arc */
            if(qmost_dfs_save_image_and_var(warc_frame,
                                            arc_hdr,
                                            arm_extname,
                                            warc_hdr,
                                            warc_img,
                                            warc_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save wavelength "
                                             "calibrated arc %s[%s]",
                                             warc_filename,
                                             arm_extname);
            }
        }

        /* Clean up */
        cpl_image_delete(arc_img);
        arc_img = NULL;

        cpl_image_delete(arc_var);
        arc_var = NULL;

        cpl_propertylist_delete(arc_hdr);
        arc_hdr = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;

        if(master_wave_tbl != NULL) {
            cpl_table_delete(master_wave_tbl);
            master_wave_tbl = NULL;
            cpl_propertylist_delete(master_wave_hdr);
            master_wave_hdr = NULL;
        }

        cpl_table_delete(wave_tbl);
        wave_tbl = NULL;

        cpl_image_delete(extracted_arc_img);
        extracted_arc_img = NULL;

        cpl_image_delete(extracted_arc_var);
        extracted_arc_var = NULL;

        cpl_image_delete(warc_img);
        warc_img = NULL;

        cpl_image_delete(warc_var);
        warc_var = NULL;

        cpl_propertylist_delete(warc_hdr);
        warc_hdr = NULL;
    }

    if(proc_arc_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                proc_arc_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to processed "
                                             "arc frame %s",
                                             proc_arc_filename);
            }
        }
    }

    if(extracted_arc_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                extracted_arc_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to extracted "
                                             "arc frame %s",
                                             extracted_arc_filename);
            }
        }
    }

    if(fibinfo_tbl != NULL) {
        if(qmost_fibtabsave(fibinfo_tbl,
                            fibinfo_hdr,
                            wave_frame) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save FIBINFO table "
                                         "to wave frame %s",
                                         wave_filename);
        }
    }

    /* Copy fibinfo table from the first frame if there was one */
    if(warc_frame != NULL) {
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                warc_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to wavelength calibrated "
                                             "arc frame %s",
                                             warc_filename);
            }
        }
    }

    /* Clean up */
    cpl_frameset_delete(arcs_raw);
    arcs_raw = NULL;

    if(fibinfo_tbl != NULL) {
        cpl_table_delete(fibinfo_tbl);
        fibinfo_tbl = NULL;
    }

    if(fibinfo_hdr != NULL) {
        cpl_propertylist_delete(fibinfo_hdr);
        fibinfo_hdr = NULL;
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_arc_analyse_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_parameter *par;

    /* QC level parameter */
    par = cpl_parameter_new_value(RECIPE_NAME".level",
                                  CPL_TYPE_INT,
                                  "QC level (0: QC0; 1: QC1)",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "level");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Switch to save processed products */
    par = cpl_parameter_new_value(RECIPE_NAME".keep",
                                  CPL_TYPE_BOOL,
                                  "Save optional processed products",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "keep");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* ccdproc parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".ccdproc.swapx",
                                  CPL_TYPE_STRING,
                                  "String of 0 or 1 specifying for each "
                                  "arm of each spectrograph if we should "
                                  "flip the x axis to correct the "
                                  "wavelength order of the spectral axis.",
                                  RECIPE_NAME,
                                  QMOST_DEFAULT_SWAPX_TABLE);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "ccdproc.swapx");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* imcombine parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.combtype",
                                  CPL_TYPE_INT,
                                  "Stacking method when combining multiple "
                                  "images.  1: mean, 2: median.",
                                  RECIPE_NAME,
                                  QMOST_MEANCALC);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.combtype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.scaletype",
                                  CPL_TYPE_INT,
                                  "Scaling to use when combining multiple "
                                  "images.  0: no biasing or scaling, "
                                  "1: bias by relative offset in background "
                                  "levels, 2: scale by ratio of background "
                                  "levels, 3: scale by exposure time and "
                                  "bias by relative offset in background.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.scaletype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.xrej",
                                  CPL_TYPE_BOOL,
                                  "Do extra rejection cycle when combining "
                                  "images.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.xrej");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold in terms of "
                                  "background noise for rejection when "
                                  "combining images.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Scattered light removal parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".scattered.enable",
                                  CPL_TYPE_BOOL,
                                  "Enable scattered light removal",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.enable");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".scattered.nbsize",
                                  CPL_TYPE_INT,
                                  "Size of the background smoothing cells "
                                  "for determining scattered light in pixels.",
                                  RECIPE_NAME,
                                  256);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.nbsize");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Extraction parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".extract.width",
                                  CPL_TYPE_INT,
                                  "Boxcar width in pixels over which the "
                                  "summation will be done at each spectral "
                                  "pixel during extraction.",
                                  RECIPE_NAME,
                                  5);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.width");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);
    
    /* Arc analyse parameters */  
    par = cpl_parameter_new_value(RECIPE_NAME".arc.detthr",
                                  CPL_TYPE_DOUBLE,
                                  "Detection threshold used to find emission "
                                  "features in the extracted arc spectrum in "
                                  "units of the background sigma, or -1 to "
                                  "automatically choose a suitable default.",
                                  RECIPE_NAME,
                                  -1.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "arc.detthr");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".arc.rejthr",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold for clipping in "
                                  "the wavelength solution fit, or -1 to "
                                  "automatically choose a suitable default.",
                                  RECIPE_NAME,
                                  -1.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "arc.rejthr");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".arc.nord",
                                  CPL_TYPE_INT,
                                  "The order of the polynomial used to fit "
                                  "the wavelength solution, or -1 to "
                                  "automatically choose a suitable default.",
                                  RECIPE_NAME,
                                  -1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "arc.nord");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".arc.matchwindow",
                                  CPL_TYPE_DOUBLE,
                                  "Maximum deviation between the predicted "
                                  "position of an arc line and the position "
                                  "of a detected emission feature in units "
                                  "of the spectral resolution element.",
                                  RECIPE_NAME,
                                  1.5);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "arc.matchwindow");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".arc.matchgrid",
                                  CPL_TYPE_DOUBLE,
                                  "Search limit for initial grid search "
                                  "in wavelength units (Angstroms), or "
                                  "-1 to choose a suitable default.",
                                  RECIPE_NAME,
                                  -1.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "arc.matchgrid");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".fpe.prescoef",
                                  CPL_TYPE_DOUBLE,
                                  "Coefficient for correction of FPE "
                                  "pressure variation in km/s/mbar.",
                                  RECIPE_NAME,
                                  0.34);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "fpe.prescoef");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* reference parameter */
    par = cpl_parameter_new_value(RECIPE_NAME".reference",
                                  CPL_TYPE_BOOL,
                                  "Emit reference rather than master frame.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "reference");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    return CPL_ERROR_NONE;
}

/**@}*/
