/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "qmost_blk.h"
#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_pfits.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_utils     qmost_utils
 *
 * Miscellaneous Utilities.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_utils.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_load_master_header (
    const char *filename,
    int extension,
    cpl_propertylist **hdr);

/*----------------------------------------------------------------------------*/
/**
 * @brief    Get the pipeline copyright and license
 *
 * @return   The copyright and license string
 *
 * The function returns a pointer to the statically allocated license string.
 * This string should not be modified using the returned pointer.
 *
 */
/*----------------------------------------------------------------------------*/
const char * qmost_get_license(void)
{
  const char *qmost_license = cpl_get_license("QMOST", "2002,2022");

  return qmost_license ;
}


/*----------------------------------------------------------------------------*/
/**
 * @brief   Check the entries in the recipe and classify the frameset with
 *          the tags. Checks for the possible cases where classification fails.
 *
 * @param   frameset        input set of frames
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the frameset couldn't be
 *                                    classified.
 * @retval  CPL_ERROR_FILE_IO         If a file in the frameset
 *                                    couldn't be opened as FITS.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the tag for a frame
 *                                        wasn't one of the recognised
 *                                        tags for this pipeline.
 * @retval  CPL_ERROR_NULL_INPUT      If the frameset was NULL.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code qmost_check_and_set_groups(
        cpl_frameset *frameset)
{
  /* Check entries */
  cpl_ensure_code(frameset, CPL_ERROR_NULL_INPUT);

  /* Check size of frameset for to know if the sof file is not empty */
  cpl_size nframes = cpl_frameset_get_size(frameset);
  for (cpl_size i = 0; i < nframes; i++) {

    cpl_frame  *frame    = cpl_frameset_get_position(frameset, i);
    const char *filename = cpl_frame_get_filename(frame);

    /* ToDo:
     * Should a minimum number of extension be enforced? */

    /* Check if the FITS file exist and have correct data,
     * return 0 if the fits file is valid without extensions */
    if (cpl_fits_count_extensions(filename) < 0){

      return cpl_error_set_message(cpl_func, cpl_error_get_code(),
              "Problem with the file '%s' (%s --> Code %d)",
              filename, cpl_error_get_message(), cpl_error_get_code());
    }
  }

  /* Identify the RAW, CONF and CALIB frames in the input frameset */
  if (qmost_dfs_set_groups(frameset)) { // Note: if return is 0, all is good

    /* Error classify frames */
    return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
				   "Cannot classify RAW and/or CALIB frames");

  } else { // Return is 0, then all should have worked fine

    /* Check classification */
    for (cpl_size i = 0; i < nframes; i++) {

      cpl_frame       *frame = cpl_frameset_get_position(frameset, i);
      const char      *tag   = cpl_frame_get_tag(frame);
      cpl_frame_group group  = cpl_frame_get_group(frame);

      /* The tag is invalid */
      if (group == CPL_FRAME_GROUP_NONE) {
        return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
             "Frame:%lld with tag:%s is invalid", i, tag);
      }

    }
  }

  /* If all went good with the classification, return the none error */
  return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Return median and median absolute deviation of cpl_array.
 *
 * @param   arr        (Given)    The array.
 * @param   med        (Returned) The median.
 * @param   mad        (Returned) The median absolute deviation.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the arguments was NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_array_get_med_mad (
    cpl_array *arr,
    double *med,
    double *mad)
{
    cpl_array *tmp = NULL;

    cpl_ensure_code(arr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(med, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mad, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(tmp != NULL) {                           \
        cpl_array_delete(tmp);                  \
        tmp = NULL;                             \
    }

    *med = cpl_array_get_median(arr);

    tmp = cpl_array_duplicate(arr);
    if(tmp == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not duplicate input array");
    }

    if(cpl_array_subtract_scalar(tmp, *med) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not subtract median");
    }
    if(cpl_array_abs(tmp) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not abs array");
    }

    *mad = cpl_array_get_median(tmp);

    cpl_array_delete(tmp);
    tmp = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read a floating point value from a cpl_propertylist.
 *
 * This routine increases robustness when reading floating point
 * values from a propertylist compared to the cpl_propertylist API:
 *
 * When reading a FITS header, anything without a decimal point in the
 * value gets read as int, and subsequently attempting to read these
 * as doubles throws an error.  This operation should be valid.  This
 * wrapper function makes it work the way it should by doing the type
 * conversion automatically.
 *
 * The error checking is also made sane for the caller, avoiding the
 * faffing around with saving the error state and then comparing it
 * to find out if the header read didn't work.
 *
 * @param   plist      (Given)    The property list to read from.
 * @param   name       (Given)    Property name to look up.
 * @param   value      (Modified) Pointer to a variable to receive the
 *                                value.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the property wasn't found in
 *                                    the property list.
 * @retval  CPL_ERROR_NULL_INPUT      If any of the arguments are
 *                                    NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the property isn't of a type
 *                                    we can convert to double.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_propertylist_get_double (
    const cpl_propertylist *plist,
    const char *name,
    double *value)
{
    const cpl_property *prop;
    cpl_errorstate prestate;
    cpl_type type;
    const char *p;
    char *ep;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(name != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(value != NULL, CPL_ERROR_NULL_INPUT);

    /* Get pointer to property */
    prop = cpl_propertylist_get_property_const(plist, name);
    if(prop == NULL) {
        if(cpl_error_get_code() != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't retrieve property %s",
                                         name);
        }
        else {
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "property %s not found in "
                                         "property list",
                                         name);
        }
    }

    /* Check type and decide how to proceed */
    prestate = cpl_errorstate_get();

    type = cpl_property_get_type(prop);

    switch(type) {
    case CPL_TYPE_FLOAT:
    case CPL_TYPE_DOUBLE:
        *value = cpl_property_get_double(prop);
        break;
    case CPL_TYPE_BOOL:
        *value = cpl_property_get_bool(prop);
        break;
    case CPL_TYPE_INT:
        *value = cpl_property_get_int(prop);
        break;
    case CPL_TYPE_LONG:
        *value = cpl_property_get_long(prop);
        break;
    case CPL_TYPE_LONG_LONG:
        *value = cpl_property_get_long_long(prop);
        break;        
    case CPL_TYPE_STRING:
        /* There could be a double value inside a string if they put
           single quotes around one.  I've seen it... */
        p = cpl_property_get_string(prop);
        if(p == NULL) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get property %s "
                                         "string value",
                                         name);
        }

        /* Attempt conversion */
        *value = strtod(p, &ep);

        /* Throw error if blank */
        if(p == ep) {
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                         "couldn't convert blank string "
                                         "property %s to double",
                                         name);
        }

        /* Skip any trailing whitespace */
        while(*ep != '\0' && isspace(*ep)) {
            ep++;
        }

        /* Throw error if there were trailing non-whitespace characters */
        if(*ep != '\0') {
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                         "couldn't convert string property "
                                         "%s = '%s' to double",
                                         name, p);
        }

        break;

        break;
    case CPL_TYPE_INVALID:
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get type of property %s",
                                     name);
    default:
        return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                     "don't know how to convert %s "
                                     "property %s to double",
                                     cpl_type_get_name(type),
                                     name);
    }

    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get property %s value",
                                     name);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read a floating point value from a cpl_propertylist.
 *
 * This routine increases robustness when reading floating point
 * values from a propertylist compared to the cpl_propertylist API:
 *
 * When reading a FITS header, anything without a decimal point in the
 * value gets read as int, and subsequently attempting to read these
 * as floats throws an error.  This operation should be valid.  This
 * wrapper function makes it work the way it should by doing the type
 * conversion automatically.
 *
 * The error checking is also made sane for the caller, avoiding the
 * faffing around with saving the error state and then comparing it
 * to find out if the header read didn't work.
 *
 * @param   plist      (Given)    The property list to read from.
 * @param   name       (Given)    Property name to look up.
 * @param   value      (Modified) Pointer to a variable to receive the
 *                                value.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the property wasn't found in
 *                                    the property list.
 * @retval  CPL_ERROR_NULL_INPUT      If any of the arguments are
 *                                    NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the property isn't of a type
 *                                    we can convert to float.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_propertylist_get_float (
    const cpl_propertylist *plist,
    const char *name,
    float *value)
{
    const cpl_property *prop;
    cpl_errorstate prestate;
    cpl_type type;
    const char *p;
    char *ep;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(name != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(value != NULL, CPL_ERROR_NULL_INPUT);

    /* Get pointer to property */
    prop = cpl_propertylist_get_property_const(plist, name);
    if(prop == NULL) {
        if(cpl_error_get_code() != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't retrieve property %s",
                                         name);
        }
        else {
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "property %s not found in "
                                         "property list",
                                         name);
        }
    }

    /* Check type and decide how to proceed */
    prestate = cpl_errorstate_get();

    type = cpl_property_get_type(prop);

    switch(type) {
    case CPL_TYPE_FLOAT:
    case CPL_TYPE_DOUBLE:
        *value = cpl_property_get_float(prop);
        break;
    case CPL_TYPE_BOOL:
        *value = cpl_property_get_bool(prop);
        break;
    case CPL_TYPE_INT:
        *value = cpl_property_get_int(prop);
        break;
    case CPL_TYPE_LONG:
        *value = cpl_property_get_long(prop);
        break;
    case CPL_TYPE_LONG_LONG:
        *value = cpl_property_get_long_long(prop);
        break;        
    case CPL_TYPE_STRING:
        /* There could be a float value inside a string if they put
           single quotes around one.  I've seen it... */
        p = cpl_property_get_string(prop);
        if(p == NULL) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get property %s "
                                         "string value",
                                         name);
        }

        /* Attempt conversion */
        *value = strtod(p, &ep);

        /* Throw error if blank */
        if(p == ep) {
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                         "couldn't convert blank string "
                                         "property %s to float",
                                         name);
        }

        /* Skip any trailing whitespace */
        while(*ep != '\0' && isspace(*ep)) {
            ep++;
        }

        /* Throw error if there were trailing non-whitespace characters */
        if(*ep != '\0') {
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                         "couldn't convert string property "
                                         "%s = '%s' to float",
                                         name, p);
        }

        break;
    case CPL_TYPE_INVALID:
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get type of property %s",
                                     name);
    default:
        return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                     "don't know how to convert %s "
                                     "property %s to float",
                                     cpl_type_get_name(type),
                                     name);
    }

    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get property %s value",
                                     name);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read an integer or boolean value from a cpl_propertylist.
 *
 * This routine increases robustness when reading integer or boolean
 * values from a propertylist compared to the cpl_propertylist API.
 * Integer and boolean are made interchangeable, so 1 or 0 and t or f
 * in the value will both work.  Floating point types are converted
 * by truncation following what CFITSIO does, and strings by reading
 * them as base-10 integers.  This isn't quite as comprehensive as
 * CFITSIO's efforts to convert strings, but it should be sufficient.
 *
 * The error checking is made sane for the caller, avoiding the
 * faffing around with saving the error state and then comparing it
 * to find out if the header read didn't work.
 *
 * @param   plist      (Given)    The property list to read from.
 * @param   name       (Given)    Property name to look up.
 * @param   value      (Modified) Pointer to a variable to receive the
 *                                value.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the property wasn't found in
 *                                    the property list.
 * @retval  CPL_ERROR_NULL_INPUT      If any of the arguments are
 *                                    NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the property isn't of a type
 *                                    we can convert to int.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_propertylist_get_int (
    const cpl_propertylist *plist,
    const char *name,
    int *value)
{
    const cpl_property *prop;
    cpl_errorstate prestate;
    cpl_type type;
    const char *p;
    char *ep;

    cpl_ensure_code(plist != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(name != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(value != NULL, CPL_ERROR_NULL_INPUT);

    /* Get pointer to property */
    prop = cpl_propertylist_get_property_const(plist, name);
    if(prop == NULL) {
        if(cpl_error_get_code() != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't retrieve property %s",
                                         name);
        }
        else {
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "property %s not found in "
                                         "property list",
                                         name);
        }
    }

    /* Check type and decide how to proceed */
    prestate = cpl_errorstate_get();

    type = cpl_property_get_type(prop);

    switch(type) {
    case CPL_TYPE_FLOAT:
    case CPL_TYPE_DOUBLE:
        *value = cpl_property_get_double(prop);
        break;
    case CPL_TYPE_BOOL:
        *value = cpl_property_get_bool(prop);
        break;
    case CPL_TYPE_INT:
        *value = cpl_property_get_int(prop);
        break;
    case CPL_TYPE_LONG:
        *value = cpl_property_get_long(prop);
        break;
    case CPL_TYPE_LONG_LONG:
        *value = cpl_property_get_long_long(prop);
        break;        
    case CPL_TYPE_STRING:
        /* There could be an integer value inside a string if they put
           single quotes around one. */
        p = cpl_property_get_string(prop);
        if(p == NULL) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get property %s "
                                         "string value",
                                         name);
        }

        /* Attempt conversion */
        *value = strtol(p, &ep, 10);

        /* Throw error if blank */
        if(p == ep) {
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                         "couldn't convert blank string "
                                         "property %s to int",
                                         name);
        }

        /* Skip any trailing whitespace */
        while(*ep != '\0' && isspace(*ep)) {
            ep++;
        }

        /* Throw error if there were trailing non-whitespace characters */
        if(*ep != '\0') {
            return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                         "couldn't convert string property "
                                         "%s = '%s' to int",
                                         name, p);
        }

        break;
    case CPL_TYPE_INVALID:
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get type of property %s",
                                     name);
    default:
        return cpl_error_set_message(cpl_func, CPL_ERROR_TYPE_MISMATCH,
                                     "don't know how to convert %s "
                                     "property %s to int",
                                     cpl_type_get_name(type),
                                     name);
    }

    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get property %s value",
                                     name);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Blank a window of columns in a cpl_table.
 *
 * This explicitly sets any integer columns to zero to ensure they are
 * initialized in the output FITS file.  This avoids the problem of
 * ending up with random garbage in them when they get written out.
 * The CPL way of doing this by calling cpl_table_fill_invalid_* has
 * the undesired side effect of setting TNULL which we don't want
 * here.
 *
 * @param   tab        (Given)    Pointer to table.
 * @param   start      (Given)    Start row (numbering from 0).
 * @param   count      (Given)    Length of segment to blank.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_blank_window (
    cpl_table *tab,
    cpl_size start,
    cpl_size count)
{
    cpl_array *colnames = NULL;
    cpl_size icol, ncols;
    const char *colname;
    cpl_type dtype;

#undef TIDY
#define TIDY                                    \
    if(colnames) {                              \
        cpl_array_delete(colnames);             \
        colnames = NULL;                        \
    }

    /* Get column names */
    colnames = cpl_table_get_column_names(tab);
    if(colnames == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get table column names");
    }

    ncols = cpl_array_get_size(colnames);

    for(icol = 0; icol < ncols; icol++) {
        /* Check type of this column */
        colname = cpl_array_get_string(colnames, icol);
        if(colname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get column %lld name",
                                         icol+1);
        }

        dtype = cpl_table_get_column_type(tab, colname);
        switch(dtype) {
        case CPL_TYPE_INT:
            /* Fill with zeros */
            if(cpl_table_fill_column_window_int(tab,
                                                colname,
                                                start,
                                                count,
                                                0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not zero fill int "
                                             "column %lld",
                                             icol+1);
            }

            break;
        case CPL_TYPE_LONG_LONG:
            /* Fill with zeros */
            if(cpl_table_fill_column_window_long_long(tab,
                                                      colname,
                                                      start,
                                                      count,
                                                      0) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not zero fill longlong "
                                             "column %lld",
                                             icol+1);
            }

            break;
        case CPL_TYPE_INVALID:
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get column %lld data type",
                                         icol+1);
        default:
            break;
        }
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Insert window of blank columns into a cpl_table.
 *
 * This does the same thing as cpl_table_insert_window but ensures any
 * integer columns are initialized to zero in the output.  See
 * qmost_cpl_table_blank_window.
 *
 * @param   tab        (Given)    Pointer to table.
 * @param   start      (Given)    Row to insert at.
 * @param   count      (Given)    Length of segment.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_insert_blank_window (
    cpl_table *tab,
    cpl_size start,
    cpl_size count)
{
    /* Insert segment containing only invalid elements */
    if(cpl_table_insert_window(tab,
                               start,
                               count) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not insert table rows "
                                     "start %lld count %lld",
                                     start, count);
    }

    return qmost_cpl_table_blank_window(tab, start, count);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Extend cpl_table to a given size and blank any new rows.
 *
 * This does the same thing as cpl_table_set_size but also ensures any
 * new elements in integer columns are initialized to zero in the
 * output.  See qmost_cpl_table_blank_window.
 *
 * @param   tab        (Given)    Pointer to table.
 * @param   new_length (Given)    New length of table.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_extend_blank (
    cpl_table *tab,
    cpl_size new_length)
{
    cpl_size old_length;

    old_length = cpl_table_get_nrow(tab);
    if(old_length < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of rows in table");
    }

    if(cpl_table_set_size(tab, new_length) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not set table length to %lld",
                                     new_length);
    }

    if(new_length > old_length) {
        if(qmost_cpl_table_blank_window(tab,
                                        old_length,
                                        new_length) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not blank new rows");
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Copy a cell from one table to another.
 *
 * The specified cell in the output table is made equal to the
 * specified cell from the input table.  The column definitions (type
 * and depth) are presumed to be the same and an error will be raised
 * if they are not.  The row numbers don't have to be the same.
 *
 * @param   outtbl     (Given)    The output table to write to.
 * @param   outcol     (Given)    The output column to write.
 * @param   outrow     (Given)    The output row to write.
 * @param   intbl      (Given)    The input table to read from.
 * @param   incol      (Given)    The input column to read.
 * @param   inrow      (Given)    The input row to read.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the input or output row
 *                                         are outside the table
 *                                         boundaries.
 * @retval  CPL_ERROR_DATA_NOT_FOUND       If the input or output
 *                                         column doesn't exist.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT   If the length of an array
 *                                         column doesn't match
 *                                         between the input and
 *                                         output tables.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH        If the data type of the
 *                                         input and output columns
 *                                         doesn't match.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_copy_cell (
    cpl_table *outtbl,
    const char *outcol,
    cpl_size outrow,
    cpl_table *intbl,
    const char *incol,
    cpl_size inrow)
{
    int valid;
    cpl_size depth;
    cpl_error_code code;

    const cpl_array *atmp;

    cpl_type coltype;
    int isnull;

    double dtmp;
    float ftmp;
    int itmp;
    long long lltmp;
    const char *stmp;

    /* Check for NULL inputs */
    cpl_ensure_code(outtbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(outcol != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(intbl != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(incol != NULL, CPL_ERROR_NULL_INPUT);

    /* Invalid -> invalid and can skip rest */
    valid = cpl_table_is_valid(intbl, incol, inrow);
    if(valid < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get valid for column %s",
                                     incol);
    }
    else if(valid == 0) {
        if(cpl_table_set_invalid(outtbl, outcol, outrow) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not invalidate column %s",
                                         outcol);
        }

        return CPL_ERROR_NONE;
    }

    /* Otherwise need to branch for each supported data type */
    code = CPL_ERROR_NONE;

    depth = cpl_table_get_column_depth(intbl, incol);
    if(depth > 0) {
        /* Copying an array */
        atmp = cpl_table_get_array(intbl, incol, inrow);
        if(atmp == NULL) {
            code = cpl_error_get_code();
        }
        else {
            code = cpl_table_set_array(outtbl, outcol, outrow, atmp);
        }
    }
    else {
        /* Copying a scalar, need to check type */
        coltype = cpl_table_get_column_type(intbl, incol);

        switch(coltype) {
        case CPL_TYPE_DOUBLE:
            dtmp = cpl_table_get_double(intbl, incol, inrow, &isnull);
            if(isnull < 0) {
                code = cpl_error_get_code();
            }
            else {
                code = cpl_table_set_double(outtbl, outcol, outrow, dtmp);
            }
            break;
        case CPL_TYPE_FLOAT:
            ftmp = cpl_table_get_float(intbl, incol, inrow, &isnull);
            if(isnull < 0) {
                code = cpl_error_get_code();
            }
            else {
                code = cpl_table_set_float(outtbl, outcol, outrow, ftmp);
            }
            break;
        case CPL_TYPE_INT:
            itmp = cpl_table_get_int(intbl, incol, inrow, &isnull);
            if(isnull < 0) {
                code = cpl_error_get_code();
            }
            else {
                code = cpl_table_set_int(outtbl, outcol, outrow, itmp);
            }
            break;
        case CPL_TYPE_LONG_LONG:
            lltmp = cpl_table_get_long_long(intbl, incol, inrow, &isnull);
            if(isnull < 0) {
                code = cpl_error_get_code();
            }
            else {
                code = cpl_table_set_long_long(outtbl, outcol, outrow, lltmp);
            }
            break;
        case CPL_TYPE_STRING:
            stmp = cpl_table_get_string(intbl, incol, inrow);
            if(stmp == NULL) {
                code = cpl_error_get_code();
            }
            else {
                code = cpl_table_set_string(outtbl, outcol, outrow, stmp);
            }
            break;
        case CPL_TYPE_INVALID:
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get column %s data type",
                                         incol);
        default:
            break;
        }
    }

    if(code != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not copy column %s to %s",
                                     incol, outcol);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read cell of vector (array) column into byte array.
 *
 * Reads one cell from a vector (array) column in a cpl_table into a
 * user supplied array.  Invalid or unset elements are left unchanged
 * so the user can populate the array with a suitable default to be
 * used in place of these null elements prior to calling.
 *
 * @param   tptr       (Given)    The table to read from.
 * @param   column     (Given)    The column to read.
 * @param   row        (Given)    The row to read (numbering from 0).
 * @param   result     (Modified) Array to read the cell into.
 * @param   n          (Given)    The number of elements to read.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row doesn't exist in
 *                                         the table, or if there were
 *                                         less elements in the table
 *                                         cell than requested.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH        If the column data type
 *                                         can't be read as integer.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_get_byte_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    unsigned char *result,
    int n)
{
    cpl_size depth;
    cpl_errorstate prestate;
    const cpl_array *arr = NULL;
    cpl_size narr;
    int i, isnull;
    unsigned char value;

    cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);

    /* Length 1 arrays get turned into scalars, and CPL will not allow
     * these to be retrieved as arrays, so the scalar case has to be
     * handled separately.  This is rather tedious. */
    depth = cpl_table_get_column_depth(tptr, column);
    if(depth < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get column %s",
                                     column);
    }
    else if(depth > 0) {
        /* Array */
        prestate = cpl_errorstate_get();

        arr = cpl_table_get_array(tptr, column, row);
        if(arr == NULL) {
            /* We can get a NULL pointer if the array was never set, or if
               something went wrong.  Check if an error was set with CPL
               to distinguish these.  If the cell is blank, leave the
               output array unchanged. */
            if(!cpl_errorstate_is_equal(prestate) &&
               cpl_error_get_code() != CPL_ERROR_NONE) {
                /* Error */
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get column %s "
                                             "for row %lld",
                                             column, row+1);
            }
            else {
                /* NULL */
                return CPL_ERROR_NONE;
            }
        }

        narr = cpl_array_get_size(arr);
        if(narr < n) {
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "array for column %s row %lld is "
                                         "shorter than requested: %lld < %d",
                                         column, row+1, narr, n);
        }

        for(i = 0; i < n; i++) {
            value = cpl_array_get_int(arr, i, &isnull);
            if(isnull < 0) {
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get element %d "
                                             "of column %s for row %lld",
                                             i, column, row+1);
            }
            else if(isnull == 0) {
                result[i] = value;
            }
            /* Otherwise, it was NULL, leave array unchanged */
        }
    }
    else if(n > 0) {
        /* Scalar */
        if(n > 1) {
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "array for column %s row %lld is "
                                         "shorter than requested: 1 < %d",
                                         column, row+1, n);
        }

        value = cpl_table_get_int(tptr, column, row, &isnull);
        if(isnull < 0) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read column %s "
                                         "for row %lld",
                                         column, row+1);
        }
        else if(isnull == 0) {
            result[0] = value;
        }
        /* Otherwise, it was NULL, leave array unchanged */
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read cell of vector (array) column into double array.
 *
 * Reads one cell from a vector (array) column in a cpl_table into a
 * user supplied array.  Invalid or unset elements are flagged with
 * NaNs.
 *
 * @param   tptr       (Given)    The table to read from.
 * @param   column     (Given)    The column to read.
 * @param   row        (Given)    The row to read (numbering from 0).
 * @param   result     (Modified) Array to read the cell into.
 * @param   n          (Given)    The number of elements to read.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row doesn't exist in
 *                                         the table, or if there were
 *                                         less elements in the table
 *                                         cell than requested.
 * @retval  CPL_ERROR_INVALID_TYPE         If the column data type
 *                                         can't be read as double.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_get_double_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    double *result,
    int n)
{
    cpl_size depth;
    cpl_errorstate prestate;
    const cpl_array *arr = NULL;
    cpl_size narr;
    int i, isnull;
    double value;

    cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);

    /* Length 1 arrays get turned into scalars, and CPL will not allow
     * these to be retrieved as arrays, so the scalar case has to be
     * handled separately.  This is rather tedious. */
    depth = cpl_table_get_column_depth(tptr, column);
    if(depth < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get column %s",
                                     column);
    }
    else if(depth > 0) {
        /* Array */
        prestate = cpl_errorstate_get();

        arr = cpl_table_get_array(tptr, column, row);
        if(arr == NULL) {
            /* We can get a NULL pointer if the array was never set, or if
               something went wrong.  Check if an error was set with CPL
               to distinguish these.  If the cell is blank, return NaNs in
               the output. */
            if(!cpl_errorstate_is_equal(prestate) &&
               cpl_error_get_code() != CPL_ERROR_NONE) {
                /* Error */
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get column %s "
                                             "for row %lld",
                                             column, row+1);
            }
            else {
                /* NULL */
                for(i = 0; i < n; i++) {
                    result[i] = NAN;
                }

                return CPL_ERROR_NONE;
            }
        }

        narr = cpl_array_get_size(arr);
        if(narr < n) {
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "array for column %s row %lld is "
                                         "shorter than requested: %lld < %d",
                                         column, row+1, narr, n);
        }

        for(i = 0; i < n; i++) {
            value = cpl_array_get(arr, i, &isnull);
            if(isnull < 0) {
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get element %d "
                                             "of column %s for row %lld",
                                             i, column, row+1);
            }
            else if(isnull > 0) {
                result[i] = NAN;
            }
            else {
                result[i] = value;
            }
        }
    }
    else if(n > 0) {
        /* Scalar */
        if(n > 1) {
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "array for column %s row %lld is "
                                         "shorter than requested: 1 < %d",
                                         column, row+1, n);
        }

        value = cpl_table_get(tptr, column, row, &isnull);
        if(isnull < 0) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get column %s "
                                         "for row %lld",
                                         column, row+1);
        }
        else if(isnull > 0) {
            result[0] = NAN;
        }
        else {
            result[0] = value;
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read cell of vector (array) column into float array.
 *
 * Reads one cell from a vector (array) column in a cpl_table into a
 * user supplied array.  Invalid or unset elements are flagged with
 * NaNs.
 *
 * @param   tptr       (Given)    The table to read from.
 * @param   column     (Given)    The column to read.
 * @param   row        (Given)    The row to read (numbering from 0).
 * @param   result     (Modified) Array to read the cell into.
 * @param   n          (Given)    The number of elements to read.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row doesn't exist in
 *                                         the table, or if there were
 *                                         less elements in the table
 *                                         cell than requested.
 * @retval  CPL_ERROR_INVALID_TYPE         If the column data type
 *                                         can't be read as float.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_get_float_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    float *result,
    int n)
{
    cpl_size depth;
    cpl_errorstate prestate;
    const cpl_array *arr = NULL;
    cpl_size narr;
    int i, isnull;
    float value;

    cpl_ensure_code(result, CPL_ERROR_NULL_INPUT);

    /* Length 1 arrays get turned into scalars, and CPL will not allow
     * these to be retrieved as arrays, so the scalar case has to be
     * handled separately.  This is rather tedious. */
    depth = cpl_table_get_column_depth(tptr, column);
    if(depth < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get column %s",
                                     column);
    }
    else if(depth > 0) {
        /* Array */
        prestate = cpl_errorstate_get();

        arr = cpl_table_get_array(tptr, column, row);
        if(arr == NULL) {
            /* We can get a NULL pointer if the array was never set, or if
               something went wrong.  Check if an error was set with CPL
               to distinguish these.  If the cell is blank, return NaNs in
               the output. */
            if(!cpl_errorstate_is_equal(prestate) &&
               cpl_error_get_code() != CPL_ERROR_NONE) {
                /* Error */
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get column %s "
                                             "for row %lld",
                                             column, row+1);
            }
            else {
                /* NULL */
                for(i = 0; i < n; i++) {
                    result[i] = NAN;
                }

                return CPL_ERROR_NONE;
            }
        }

        narr = cpl_array_get_size(arr);
        if(narr < n) {
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "array for column %s row %lld is "
                                         "shorter than requested: %lld < %d",
                                         column, row+1, narr, n);
        }

        for(i = 0; i < n; i++) {
            value = cpl_array_get(arr, i, &isnull);
            if(isnull < 0) {
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not get element %d "
                                             "of column %s for row %lld",
                                             i, column, row+1);
            }
            else if(isnull > 0) {
                result[i] = NAN;
            }
            else {
                result[i] = value;
            }
        }
    }
    else if(n > 0) {
        /* Scalar */
        if(n > 1) {
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                         "array for column %s row %lld is "
                                         "shorter than requested: 1 < %d",
                                         column, row+1, n);
        }

        value = cpl_table_get(tptr, column, row, &isnull);
        if(isnull < 0) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get column %s "
                                         "for row %lld",
                                         column, row+1);
        }
        else if(isnull > 0) {
            result[0] = NAN;
        }
        else {
            result[0] = value;
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read 1D polynomial coefficients from a table cell.
 *
 * Reads a cell from a vector (array) table column containing
 * polynomial coefficients into a 1-D cpl_polynomial object.
 *
 * @param   tptr       (Given)    The table to read from.
 * @param   column     (Given)    The column to read.
 * @param   row        (Given)    The row to read (numbering from 0).
 * @param   degree     (Given)    The polynomial degree to read.  If
 *                                there are less than degree + 1
 *                                coefficients given in the table, it
 *                                is an error, but if there are more
 *                                they will be truncated.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row doesn't exist in
 *                                         the table, or if there were
 *                                         less elements in the table
 *                                         cell than requested.
 * @retval  CPL_ERROR_INVALID_TYPE         If the column data type
 *                                         can't be read as double.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_polynomial *qmost_cpl_table_get_polynomial (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    int degree)
{
    cpl_size depth;
    cpl_errorstate prestate;
    const cpl_array *arr = NULL;
    cpl_polynomial *poly = NULL;
    int i, isnull, ncoefs;
    double value;
    cpl_size power;

#undef TIDY
#define TIDY                                    \
    if(poly != NULL) {                          \
        cpl_polynomial_delete(poly);            \
        poly = NULL;                            \
    }

    /* Length 1 arrays get turned into scalars, and CPL will not allow
     * these to be retrieved as arrays, so the scalar case has to be
     * handled separately.  This is rather tedious. */
    depth = cpl_table_get_column_depth(tptr, column);
    if(depth < 0) {
        TIDY;
        cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                  "could not get column %s",
                                  column);
        return NULL;
    }
    else if(depth > 0) {
        /* Array */
        prestate = cpl_errorstate_get();

        arr = cpl_table_get_array(tptr, column, row);
        if(arr == NULL) {
            /* We can get a NULL pointer if the array was never set, or if
               something went wrong.  Check if an error was set with CPL
               to distinguish these.  If the cell is blank, return an
               empty polynomial. */
            if(!cpl_errorstate_is_equal(prestate) &&
               cpl_error_get_code() != CPL_ERROR_NONE) {
                /* Error */
                TIDY;
                cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                      "could not get column %s for row %lld",
                                      column, row+1);
                return NULL;
            }
            else {
                /* NULL, return an empty polynomial */
                poly = cpl_polynomial_new(1);
                return poly;
            }
        }

        poly = cpl_polynomial_new(1);

        ncoefs = cpl_array_get_size(arr);
        if(ncoefs < degree + 1) {
            TIDY;
            cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                  "array for column %s row %lld is "
                                  "shorter than expected: %d < %d",
                                  column, row + 1,
                                  ncoefs, degree + 1);
            return NULL;
        }

        for(i = degree; i >= 0; i--) {
            value = cpl_array_get(arr, i, &isnull);
            if(isnull < 0) {
                TIDY;
                cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                      "could not get element %d "
                                      "of column %s for row %lld",
                                      i, column, row+1);
                return NULL;
            }
            else if(isnull == 0) {  /* not NULL */
                power = i;
                if(cpl_polynomial_set_coeff(poly,
                                            &power,
                                            value) != CPL_ERROR_NONE) {
                    TIDY;
                    cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                          "could not set degree %d "
                                          "coefficient for row %lld",
                                          i, row+1);
                    return NULL;
                }
            }
        }
    }
    else {
        /* Scalar */
        if(1 < degree + 1) {
            TIDY;
            cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                  "array for column %s row %lld is "
                                  "shorter than expected: %d < %d",
                                  column, row + 1,
                                  1, degree + 1);
            return NULL;
        }

        poly = cpl_polynomial_new(1);

        value = cpl_table_get(tptr, column, row, &isnull);
        if(isnull < 0) {
            TIDY;
            cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                  "could not get column %s for row %lld",
                                  column, row+1);
            return NULL;
        }
        else if(isnull == 0) {  /* not NULL */
            power = 0;
            if(cpl_polynomial_set_coeff(poly,
                                        &power,
                                        value) != CPL_ERROR_NONE) {
                TIDY;
                cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                      "could not set degree %lld "
                                      "coefficient for row %lld",
                                      power, row+1);
                return NULL;
            }
        }
    }

    return poly;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Write byte array into vector (array) table column.
 *
 * Sets a cell of a vector (array) column in a cpl_table using a
 * user supplied array.  If the supplied array is shorter than the
 * depth of the cell, the remaining elements are padded out with
 * zeros.
 *
 * @param   tptr       (Given)    The table to write to.
 * @param   column     (Given)    The column to write.
 * @param   row        (Given)    The row to write (numbering from 0).
 * @param   values     (Given)    Values to write into the cell.  Can
 *                                be NULL if n == 0.
 * @param   n          (Given)    The number of elements to write.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row doesn't exist in
 *                                         the table, or if there were
 *                                         less than n elements in the
 *                                         table cell.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH        If the column data type
 *                                         isn't an array of integer.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_set_byte_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    unsigned char *values,
    int n)
{
    cpl_array *arr = NULL;
    cpl_size narr;
    int i;

    cpl_ensure_code(n == 0 || values != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(arr) {                                   \
        cpl_array_delete(arr);                  \
        arr = NULL;                             \
    }

    narr = cpl_table_get_column_depth(tptr, column);
    if(narr < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get depth for column %s",
                                     column);
    }

    if(narr < n) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                     "depth for column %s row %lld is "
                                     "less than expected: %lld < %d",
                                     column, row+1, narr, n);
    }

    /* This has to have length equal to the column depth otherwise
       cpl_table_set_array fails. */
    arr = cpl_array_new(narr, CPL_TYPE_INT);
    if(arr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create byte array "
                                     "of length %lld",
                                     narr);
    }

    /* Set non-NULL elements */
    for(i = 0; i < n; i++) {
        if(cpl_array_set_int(arr, i, values[i]) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set element %d "
                                         "of column %s for row %lld",
                                         i, column, row+1);
        }
    }

    /* Fill out the rest of the array with zeros */
    for(i = n; i < narr; i++) {
        if(cpl_array_set_int(arr, i, 0) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set element %d "
                                         "of column %s for row %lld",
                                         i, column, row+1);
        }
    }

    if(cpl_table_set_array(tptr, column, row, arr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not set column %s for row %lld",
                                     column, row);
    }

    cpl_array_delete(arr);
    arr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Write double array into vector (array) table column.
 *
 * Sets a cell of a vector (array) column in a cpl_table using a
 * user supplied array.  Any values for which isfinite returns false
 * such as NaNs are flagged invalid.
 *
 * @param   tptr       (Given)    The table to write to.
 * @param   column     (Given)    The column to write.
 * @param   row        (Given)    The row to write (numbering from 0).
 * @param   values     (Given)    Values to write into the cell.  Can
 *                                be NULL if n == 0.
 * @param   n          (Given)    The number of elements to write.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row doesn't exist in
 *                                         the table, or if there were
 *                                         less than n elements in the
 *                                         table cell.
 * @retval  CPL_ERROR_INVALID_TYPE         If the column data type
 *                                         isn't double or float.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_set_double_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    double *values,
    int n)
{
    cpl_array *arr = NULL;
    cpl_size narr;
    cpl_type type;
    int i;

    cpl_ensure_code(n == 0 || values != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(arr) {                                   \
        cpl_array_delete(arr);                  \
        arr = NULL;                             \
    }

    /* Get depth */
    narr = cpl_table_get_column_depth(tptr, column);
    if(narr < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get depth for column %s",
                                     column);
    }

    if(narr < n) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                     "depth for column %s row %lld is "
                                     "less than expected: %lld < %d",
                                     column, row+1, narr, n);
    }

    /* Get base type of the array */
    type = cpl_table_get_column_type(tptr, column);
    if(type == CPL_TYPE_INVALID) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get data type for column %s",
                                     column);
    }

    type &= ~(CPL_TYPE_FLAG_ARRAY | CPL_TYPE_POINTER);

    /* This has to have length equal to the column depth otherwise
       cpl_table_set_array fails. */
    arr = cpl_array_new(narr, type);
    if(arr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create array "
                                     "of length %lld",
                                     narr);
    }

    for(i = 0; i < n; i++) {
        if(isfinite(values[i])) {
            if(cpl_array_set(arr, i, values[i]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not set element %d "
                                             "of column %s for row %lld",
                                             i, column, row+1);
            }
        }
        else {
            if(cpl_array_set_invalid(arr, i) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not invalidate element %d "
                                             "of column %s for row %lld",
                                             i, column, row+1);
            }
        }
    }

    if(cpl_table_set_array(tptr, column, row, arr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not set column %s for row %lld",
                                     column, row);
    }

    cpl_array_delete(arr);
    arr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Write float array into vector (array) table column.
 *
 * Sets a cell of a vector (array) column in a cpl_table using a
 * user supplied array.  Any values for which isfinite returns false
 * such as NaNs are flagged invalid.
 *
 * @param   tptr       (Given)    The table to write to.
 * @param   column     (Given)    The column to write.
 * @param   row        (Given)    The row to write (numbering from 0).
 * @param   values     (Given)    Values to write into the cell.  Can
 *                                be NULL if n == 0.
 * @param   n          (Given)    The number of elements to write.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row doesn't exist in
 *                                         the table, or if there were
 *                                         less than n elements in the
 *                                         table cell.
 * @retval  CPL_ERROR_INVALID_TYPE         If the column data type
 *                                         isn't double or float.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_set_float_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    float *values,
    int n)
{
    cpl_array *arr = NULL;
    cpl_size narr;
    cpl_type type;
    int i;

    cpl_ensure_code(n == 0 || values != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(arr) {                                   \
        cpl_array_delete(arr);                  \
        arr = NULL;                             \
    }

    /* Get depth */
    narr = cpl_table_get_column_depth(tptr, column);
    if(narr < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get depth for column %s",
                                     column);
    }

    if(narr < n) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                     "depth for column %s row %lld is "
                                     "less than expected: %lld < %d",
                                     column, row+1, narr, n);
    }

    /* Get base type of the array */
    type = cpl_table_get_column_type(tptr, column);
    if(type == CPL_TYPE_INVALID) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get data type for column %s",
                                     column);
    }

    type &= ~(CPL_TYPE_FLAG_ARRAY | CPL_TYPE_POINTER);

    /* This has to have length equal to the column depth otherwise
       cpl_table_set_array fails. */
    arr = cpl_array_new(narr, type);
    if(arr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create array "
                                     "of length %lld",
                                     narr);
    }

    for(i = 0; i < n; i++) {
        if(isfinite(values[i])) {
            if(cpl_array_set(arr, i, values[i]) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not set element %d "
                                             "of column %s for row %lld",
                                             i, column, row+1);
            }
        }
        else {
            if(cpl_array_set_invalid(arr, i) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not invalidate element %d "
                                             "of column %s for row %lld",
                                             i, column, row+1);
            }
        }
    }

    if(cpl_table_set_array(tptr, column, row, arr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not set column %s for row %lld",
                                     column, row);
    }

    cpl_array_delete(arr);
    arr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Write 1D polynomial coefficients out to a table cell.
 *
 * Writes polynomial coefficients from a 1-D cpl_polynomial object
 * into a cell of a cpl_table.
 *
 * @param   tptr       (Given)    The table to write to.
 * @param   column     (Given)    The column to write.
 * @param   row        (Given)    The row to write (numbering from 0).
 * @param   poly       (Given)    The polynomial.
 * @param   degree     (Given)    The polynomial degree to write.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                 If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the row doesn't exist in
 *                                         the table, or if there were
 *                                         less than degree + 1
 *                                         elements in the table
 *                                         cell. 
 * @retval  CPL_ERROR_INVALID_TYPE         If the column data type
 *                                         isn't double or float.
 * @retval  CPL_ERROR_NULL_INPUT           If one of the required
 *                                         inputs or outputs was
 *                                         NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_cpl_table_set_polynomial (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    cpl_polynomial *poly,
    int degree)
{
    cpl_array *arr = NULL;
    cpl_size narr, ncoefs, i;
    cpl_type type;
    cpl_errorstate prestate;
    double value;

    cpl_ensure_code(poly, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(arr) {                                   \
        cpl_array_delete(arr);                  \
        arr = NULL;                             \
    }

    /* Get depth */
    narr = cpl_table_get_column_depth(tptr, column);
    if(narr < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get depth for column %s",
                                     column);
    }

    ncoefs = degree + 1;

    if(narr < ncoefs) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_ACCESS_OUT_OF_RANGE,
                                     "depth for column %s row %lld is "
                                     "less than expected: %lld < %lld",
                                     column, row+1, narr, ncoefs);
    }

    /* Get base type of the array */
    type = cpl_table_get_column_type(tptr, column);
    if(type == CPL_TYPE_INVALID) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get data type for column %s",
                                     column);
    }

    type &= ~(CPL_TYPE_FLAG_ARRAY | CPL_TYPE_POINTER);

    /* This has to have length equal to the column depth otherwise
       cpl_table_set_array fails. */
    arr = cpl_array_new(narr, type);
    if(arr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create array "
                                     "of length %lld",
                                     narr);
    }

    prestate = cpl_errorstate_get();

    for(i = 0; i <= degree; i++) {
        value = cpl_polynomial_get_coeff(poly, &i);
        if(!cpl_errorstate_is_equal(prestate) &&
           cpl_error_get_code() != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get degree %lld "
                                         "coefficient for row %lld",
                                         i, row+1);
        }
        
        if(cpl_array_set(arr, i, value) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not write degree %lld "
                                         "coefficient for row %lld",
                                         i, row+1);
        }
    }

    if(cpl_table_set_array(tptr, column, row, arr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not set column %s for row %lld",
                                     column, row+1);
    }

    cpl_array_delete(arr);
    arr = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Load the FITS header from a named extension and check for
 *          a dummy HDU.
 *
 * This utility function performs the operation common to all of the
 * "load master" functions where the input FITS header is read and
 * checked for a dummy HDU.  If the HDU is a dummy, this means we're
 * trying to process a raw frame where the detector was live with a
 * master where it was not, so an error is raised.
 *
 * @param   filename   (Given)    The input file to load.
 * @param   extension  (Given)    Extension number to load.
 * @param   hdr        (Returned) The FITS header loaded from the
 *                                named extension.  Argument can be
 *                                NULL if the header isn't needed.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the named extension was not
 *                                    found in the file.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required arguments
 *                                    was NULL.
 * @retval  QMOST_ERROR_DUMMY         If the master is a dummy.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS DUMMY</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static cpl_error_code qmost_load_master_header (
    const char *filename,
    int extension,
    cpl_propertylist **hdr)
{
    cpl_propertylist *tmp_hdr = NULL;
    int dummy;

    cpl_ensure_code(filename != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(tmp_hdr != NULL) {                       \
        cpl_propertylist_delete(tmp_hdr);       \
        tmp_hdr = NULL;                         \
    }

    tmp_hdr = cpl_propertylist_load(filename, extension);
    if(tmp_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load FITS header %s[%d]",
                                     filename,
                                     extension);
    }

    if(qmost_pfits_get_dummy(tmp_hdr,
                             &dummy) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read dummy flag "
                                     "for %s[%d]",
                                     filename,
                                     extension);
    }

    if(dummy) {
        /* If we get to here, the detector was live in the input raw
         * but the master is a dummy.  This means we can't process it,
         * so raise an error. */ 
        TIDY;
        return cpl_error_set_message(cpl_func, QMOST_ERROR_DUMMY,
                                     "master extension %s[%d] is a dummy, "
                                     "but raw extension is live, can't "
                                     "process",
                                     filename,
                                     extension);
    }

    if(hdr != NULL) {
        *hdr = tmp_hdr;
    }
    else {
        cpl_propertylist_delete(tmp_hdr);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Load a master mask from a named extension.
 *
 * @param   frame      (Given)    The input frame to load.
 * @param   extname    (Given)    Extension name.
 * @param   mask       (Returned) The mask loaded from the named
 *                                extension.
 * @param   hdr        (Returned) The FITS header loaded from the
 *                                named extension.  Argument can be
 *                                NULL if the header isn't needed.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_BAD_FILE_FORMAT If the extension doesn't contain
 *                                    an image compatible with being
 *                                    loaded as a mask.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there was no filename set in
 *                                    the frame, or the named
 *                                    extension was not found in the
 *                                    file.
 * @retval  CPL_ERROR_FILE_IO         If the file couldn't be opened
 *                                    as FITS.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required arguments
 *                                    was NULL.
 * @retval  QMOST_ERROR_DUMMY         If the master is a dummy.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS DUMMY</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_load_master_mask (
    const cpl_frame *frame,
    const char *extname,
    cpl_mask **mask,
    cpl_propertylist **hdr)
{
    const char *filename;
    int extension;

    cpl_ensure_code(frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(extname != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mask != NULL, CPL_ERROR_NULL_INPUT);

    filename = cpl_frame_get_filename(frame);
    if(filename == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    extension = cpl_fits_find_extension(filename, extname);
    if(extension < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't find extension "
                                     "%s[%s]",
                                     filename,
                                     extname);
    }
    else if(extension == 0) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "file %s doesn't seem to have a "
                                     "%s extension",
                                     filename,
                                     extname);
    }

    if(qmost_load_master_header(filename,
                                extension,
                                hdr) != CPL_ERROR_NONE) {
        return cpl_error_get_code();
    }

    *mask = cpl_mask_load(filename,
                          0,
                          extension);
    if(*mask == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load mask %s[%s]",
                                     extname,
                                     filename);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Load a master image from a named extension.
 *
 * @param   frame      (Given)    The input frame to load.
 * @param   extname    (Given)    Extension name.
 * @param   type       (Given)    The data type to load.
 * @param   image      (Returned) The image loaded from the named
 *                                extension.
 * @param   hdr        (Returned) The FITS header loaded from the
 *                                named extension.  Argument can be
 *                                NULL if the header isn't needed.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_BAD_FILE_FORMAT If the extension doesn't contain
 *                                    an image that can be converted
 *                                    to float.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there was no filename set in
 *                                    the frame, or the named
 *                                    extension was not found in the
 *                                    file.
 * @retval  CPL_ERROR_FILE_IO         If the file couldn't be opened
 *                                    as FITS.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required arguments
 *                                    was NULL.
 * @retval  QMOST_ERROR_DUMMY         If the master is a dummy.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS DUMMY</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_load_master_image (
    const cpl_frame *frame,
    const char *extname,
    const cpl_type type,
    cpl_image **image,
    cpl_propertylist **hdr)
{
    const char *filename;
    int extension;

    cpl_ensure_code(frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(extname != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);

    filename = cpl_frame_get_filename(frame);
    if(filename == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    extension = cpl_fits_find_extension(filename, extname);
    if(extension < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't find extension "
                                     "%s[%s]",
                                     filename,
                                     extname);
    }
    else if(extension == 0) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "file %s doesn't seem to have a "
                                     "%s extension",
                                     filename,
                                     extname);
    }

    if(qmost_load_master_header(filename,
                                extension,
                                hdr) != CPL_ERROR_NONE) {
        return cpl_error_get_code();
    }

    *image = cpl_image_load(filename,
                            type,
                            0,
                            extension);
    if(*image == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load image %s[%s]",
                                     extname,
                                     filename);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Load a master image and variance from named extensions.
 *
 * A master image is loaded from the named extension given in
 * extname.  The corresponding variance array is loaded from an
 * extension named extname_var.  This follows the extension naming
 * scheme used in qmost_dfs.c:qmost_dfs_save_image_and_var for QC
 * and the scheme used for the variance array in L1, meaning the
 * routine should be compatible with master images written by either.
 *
 * As a temporary measure for backward compatibility with earlier
 * versions of QC, an error array named extname_ERROR will also be
 * tried if the variance array can't be found.
 *
 * @param   frame      (Given)    The input frame to load.
 * @param   extname    (Given)    Extension name.
 * @param   type       (Given)    The data type to load.
 * @param   image      (Returned) The image loaded from the named
 *                                extension.
 * @param   var        (Returned) The variance array loaded from the
 *                                extension named per notes above.
 * @param   hdr        (Returned) The FITS header loaded from the
 *                                named extension.  Argument can be
 *                                NULL if the header isn't needed.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_BAD_FILE_FORMAT If the image and variance
 *                                    extensions didn't contain FITS
 *                                    images that can be converted
 *                                    to float.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there was no filename set in
 *                                    the frame, or the named image or
 *                                    variance extensions weren't
 *                                    found in the file.
 * @retval  CPL_ERROR_FILE_IO         If the file couldn't be opened
 *                                    as FITS.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required arguments
 *                                    was NULL.
 * @retval  QMOST_ERROR_DUMMY         If the master is a dummy.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS DUMMY</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_load_master_image_and_var (
    const cpl_frame *frame,
    const char *extname,
    const cpl_type type,
    cpl_image **image,
    cpl_image **var,
    cpl_propertylist **hdr)
{
    cpl_errorstate prestate;
    cpl_error_code code;
    char *var_extname = NULL;

    cpl_ensure_code(frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(extname != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(image != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(var != NULL, CPL_ERROR_NULL_INPUT);

    *image = NULL;
    *var = NULL;

    if(hdr != NULL) {
        *hdr = NULL;
    }

#undef TIDY
#define TIDY                                    \
    if(*image != NULL) {                        \
        cpl_image_delete(*image);               \
        *image = NULL;                          \
    }                                           \
    if(*var != NULL) {                          \
        cpl_image_delete(*var);                 \
        *var = NULL;                            \
    }                                           \
    if(hdr != NULL && *hdr != NULL) {           \
        cpl_propertylist_delete(*hdr);          \
        *hdr = NULL;                            \
    }                                           \
    if(var_extname != NULL) {                   \
        cpl_free(var_extname);                  \
        var_extname = NULL;                     \
    }

    /* Read image */
    code = qmost_load_master_image(frame, extname, type, image, hdr);
    if(code != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load image array");
    }

    /* EXTNAME for the variance array */
    var_extname = cpl_sprintf("%s_var", extname);
    if(var_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not format EXTNAME string "
                                     "for variance extension");
    }

    /* Attempt to load error */
    prestate = cpl_errorstate_get();

    code = qmost_load_master_image(frame, var_extname, type, var, NULL);
    if(code == CPL_ERROR_DATA_NOT_FOUND) {
        /* Not found, it might be error.  Reset state and try that. */
        cpl_errorstate_set(prestate);
        
        cpl_free(var_extname);
        var_extname = NULL;

        /* Try error in case it's an old QC master file */
        var_extname = cpl_sprintf("%s_ERROR", extname);
        if(var_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format EXTNAME string "
                                         "for error extension");
        }

        code = qmost_load_master_image(frame, var_extname, type, var, NULL);
        if(code == CPL_ERROR_DATA_NOT_FOUND) {
            /* Not found, create variance array filled with zeros to
             * allow use of old master detector calibrations without
             * variance arrays. */
            cpl_errorstate_set(prestate);

            cpl_msg_warning(cpl_func,
                            "No variance array for %s[%s]",
                            cpl_frame_get_filename(frame),
                            extname);

            *var = cpl_image_new(cpl_image_get_size_x(*image),
                                 cpl_image_get_size_y(*image),
                                 type);
        }
        else if(code != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not load error array");
        }
        else {
            /* Convert from error to variance */
            cpl_image_power(*var, 2);
        }
    }
    else if(code != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load variance array");
    }

    cpl_free(var_extname);
    var_extname = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Load a master imagelist from a named extension.
 *
 * This loads a 3D (planar) FITS image into an imagelist.  In qmost
 * these are used for PSF files.
 *
 * @param   frame      (Given)    The input frame to load.
 * @param   extname    (Given)    Extension name.
 * @param   type       (Given)    The data type to load.
 * @param   imagelist  (Returned) The imagelist loaded from the named
 *                                extension.
 * @param   hdr        (Returned) The FITS header loaded from the
 *                                named extension.  Argument can be
 *                                NULL if the header isn't needed.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_BAD_FILE_FORMAT If the extension doesn't contain
 *                                    an image that can be converted
 *                                    to float.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there was no filename set in
 *                                    the frame, or the named
 *                                    extension was not found in the
 *                                    file.
 * @retval  CPL_ERROR_FILE_IO         If the file couldn't be opened
 *                                    as FITS.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required arguments
 *                                    was NULL.
 * @retval  QMOST_ERROR_DUMMY         If the master is a dummy.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS DUMMY</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_load_master_imagelist (
    const cpl_frame *frame,
    const char *extname,
    const cpl_type type,
    cpl_imagelist **imagelist,
    cpl_propertylist **hdr)
{
    const char *filename;
    int extension;

    cpl_ensure_code(frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(extname != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(imagelist != NULL, CPL_ERROR_NULL_INPUT);

    filename = cpl_frame_get_filename(frame);
    if(filename == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    extension = cpl_fits_find_extension(filename, extname);
    if(extension < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't find extension "
                                     "%s[%s]",
                                     filename,
                                     extname);
    }
    else if(extension == 0) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "file %s doesn't seem to have a "
                                     "%s extension",
                                     filename,
                                     extname);
    }

    if(qmost_load_master_header(filename,
                                extension,
                                hdr) != CPL_ERROR_NONE) {
        return cpl_error_get_code();
    }

    *imagelist = cpl_imagelist_load(filename,
                                    type,
                                    extension);
    if(*imagelist == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load imagelist %s[%s]",
                                     extname,
                                     filename);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Load a master table from a named extension.
 *
 * @param   frame      (Given)    The input frame to load.
 * @param   extname    (Given)    Extension name.
 * @param   tbl        (Returned) The table loaded from the named
 *                                extension.
 * @param   hdr        (Returned) The FITS header loaded from the
 *                                named extension.  Argument can be
 *                                NULL if the header isn't needed.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there was no filename set in
 *                                    the frame, or the named
 *                                    extension was not found in the
 *                                    file.
 * @retval  CPL_ERROR_FILE_IO         If the file couldn't be opened
 *                                    as FITS.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the extension doesn't contain
 *                                    a table.
 * @retval  CPL_ERROR_NULL_INPUT      If one of the required arguments
 *                                    was NULL.
 * @retval  QMOST_ERROR_DUMMY         If the master is a dummy.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS DUMMY</b>
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_load_master_table (
    const cpl_frame *frame,
    const char *extname,
    cpl_table **tbl,
    cpl_propertylist **hdr)
{
    const char *filename;
    int extension;

    cpl_ensure_code(frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(extname != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tbl != NULL, CPL_ERROR_NULL_INPUT);

    filename = cpl_frame_get_filename(frame);
    if(filename == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename from frame");
    }

    extension = cpl_fits_find_extension(filename, extname);
    if(extension < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't find extension "
                                     "%s[%s]",
                                     filename,
                                     extname);
    }
    else if(extension == 0) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "file %s doesn't seem to have a "
                                     "%s extension",
                                     filename,
                                     extname);
    }

    if(qmost_load_master_header(filename,
                                extension,
                                hdr) != CPL_ERROR_NONE) {
        return cpl_error_get_code();
    }

    *tbl = cpl_table_load(filename,
                          extension,
                          1);
    if(*tbl == NULL) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not load table %s[%s]",
                                     extname,
                                     filename);
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Copy and scale sensitivity function to output file.
 *
 * The sensitivity function is read from the input static sensitivity
 * file, scaled to the appropriate units for the target spectrum, and
 * appended to the output file in an extension named ARM_SENSFUNC.
 *
 * @param   science_frame      (Given)    The output science frame to
 *                                        append to.
 * @param   sensfunc_frame     (Given)    The input sensitity function
 *                                        frame to copy from.
 * @param   arm                (Given)    One of the QMOST_ARM_*
 *                                        constants saying which arm
 *                                        we're processing.
 * @param   exptime            (Given)    The exposure time (seconds).
 * @param   raw_hdr            (Given)    The FITS header of the raw
 *                                        input file to retrieve the
 *                                        GAIN from.
 * @param   ext_hdr            (Given)    The extension FITS header of
 *                                        the output spectrum file,
 *                                        with spectral WCS.  Can be
 *                                        NULL to emit a dummy
 *                                        sensitivity function HDU.
 *
 * @par Input FITS Header Information:
 *   - <b>CD1_1</b>
 *   - <b>ESO DET OUT1 GAIN</b>
 *   - <b>EXTNAME</b>
 *
 * @par Output FITS Headers:
 *   - <b>BUNIT</b>
 *   - <b>EXTNAME</b>
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND      If one of the required input
 *                                        FITS headers was not found
 *                                        in the property lists.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If the parameter arm is
 *                                        invalid.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs was NULL.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_copy_sensfunc (
    const cpl_frame *science_frame,
    const cpl_frame *sensfunc_frame,
    int arm,
    double exptime,
    cpl_propertylist *raw_hdr,
    cpl_propertylist *ext_hdr)
{
    const char *arm_extname;

    char *in_extname = NULL;
    char *out_extname = NULL;

    char *p;
    int n;

    cpl_image *sensfunc_img = NULL;
    cpl_propertylist *sensfunc_hdr = NULL;
    cpl_propertylist *add_hdr = NULL;

    double delwave, gain, scfac;
    int specbin, spatbin, isbinned;

    cpl_image *tmp_img = NULL;
    int xin, nxin, xout, nxout, xsub, y, ny;
    float *in_buf, *tmp_buf;
    float sum;
    
    cpl_ensure_code(science_frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sensfunc_frame != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(raw_hdr != NULL, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(in_extname != NULL) {                    \
        cpl_free(in_extname);                   \
        in_extname = NULL;                      \
    }                                           \
    if(out_extname != NULL) {                   \
        cpl_free(out_extname);                  \
        out_extname = NULL;                     \
    }                                           \
    if(sensfunc_img != NULL) {                  \
        cpl_image_delete(sensfunc_img);         \
        sensfunc_img = NULL;                    \
    }                                           \
    if(sensfunc_hdr != NULL) {                  \
        cpl_propertylist_delete(sensfunc_hdr);  \
        sensfunc_hdr = NULL;                    \
    }                                           \
    if(add_hdr != NULL) {                       \
        cpl_propertylist_delete(add_hdr);       \
        add_hdr = NULL;                         \
    }                                           \
    if(tmp_img != NULL) {                       \
        cpl_image_delete(tmp_img);              \
        tmp_img = NULL;                         \
    }

    /* Get EXTNAME for arm */
    arm_extname = qmost_pfits_get_extname(arm);
    if(arm_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine EXTNAME "
                                     "for arm %d", arm);
    }

    /* Get binning, wavelength sampling and gain */
    if(ext_hdr != NULL) {
        qmost_isbinned(ext_hdr, &specbin, &spatbin, &isbinned);
        
        if(qmost_cpl_propertylist_get_double(ext_hdr,
                                             "CD1_1",
                                             &delwave) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read CD1_1");
        }
    }
    else {
        /* We're saving a dummy, use 1 and set units below accordingly */
        delwave = 1.0;
        specbin = 1;
        spatbin = 1;
        isbinned = 0;
    }

    if(qmost_cpl_propertylist_get_double(raw_hdr,
                                         "ESO DET OUT1 GAIN",
                                         &gain) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read GAIN");
    }

    /* Get sensitivity function */
    in_extname = cpl_sprintf("sens_%s", arm_extname);
    if(in_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't format EXTNAME for "
                                     "sensitivity function");
    }

    if(qmost_load_master_image(sensfunc_frame,
                               in_extname,
                               CPL_TYPE_FLOAT,
                               &sensfunc_img,
                               &sensfunc_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load master "
                                     "sensitivity function");
    }

    /* Set up output header additional keywords */
    if(ext_hdr != NULL) {
        add_hdr = cpl_propertylist_duplicate(ext_hdr);
    }
    else {
        add_hdr = cpl_propertylist_new();
    }

    cpl_propertylist_update_string(
        add_hdr,
        "BUNIT",
        "erg.cm**(-2).s**(-1).angstrom**(-1).adu**(-1)");

    /* Scale to these units, noting "gain" is the non-standard ESO
     * definition (ADU/e-). */
    scfac = 1.0 / (gain * exptime * delwave);

    if(specbin != 1) {
        /* Bin down sensfunc to match spectrum */
        nxin = cpl_image_get_size_x(sensfunc_img);
        ny = cpl_image_get_size_y(sensfunc_img);

        in_buf = cpl_image_get_data_float(sensfunc_img);

        nxout = (nxin - 1) / specbin + 1;

        tmp_img = cpl_image_new(nxout, ny, CPL_TYPE_FLOAT);
        tmp_buf = cpl_image_get_data_float(tmp_img);
        
        for(y = 0; y < ny; y++) {
            for(xout = 0, xin = 0; xout < nxout; xout++) {
                sum = 0;
                n = 0;

                for(xsub = 0; xsub < specbin && xin < nxin; xsub++, xin++) {
                    sum += in_buf[y*nxin+xin];
                    n++;
                }

                tmp_buf[y*nxout+xout] = scfac * sum / n;
            }
        }

        /* Swap in */
        cpl_image_delete(sensfunc_img);
        sensfunc_img = tmp_img;
        tmp_img = NULL;
    }
    else {
        if(cpl_image_multiply_scalar(sensfunc_img,
                                     scfac) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't rescale master "
                                         "sensitivity function");
        }
    }
    
    /* Write out */
    p = strchr(arm_extname, '_');
    if(p != NULL) {
        n = p - arm_extname;
    }
    else {
        n = strlen(arm_extname);
    }

    out_extname = cpl_sprintf("%.*s_SENSFUNC", n, arm_extname);
    if(out_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't format EXTNAME for "
                                     "sensitivity function");
    }

    if(qmost_dfs_save_image_extension(science_frame,
                                      sensfunc_hdr,
                                      out_extname,
                                      add_hdr,
                                      sensfunc_img,
                                      CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't save sensitivity "
                                     "function");
    }

    cpl_free(in_extname);
    in_extname = NULL;

    cpl_free(out_extname);
    out_extname = NULL;

    cpl_image_delete(sensfunc_img);
    sensfunc_img = NULL;

    cpl_propertylist_delete(sensfunc_hdr);
    sensfunc_hdr = NULL;

    cpl_propertylist_delete(add_hdr);
    add_hdr = NULL;

    return cpl_error_get_code();
}

/**@}*/
