/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_filt1d.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_filt1d_test  Unit test of qmost_filt1d
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_filt1d using noise only
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_filt1d_noise(void)
{
    int npt = 1024;

    cpl_image *img = NULL, *testimg = NULL;
    float *testimg_data;

    double median_rms = 0;
    double mean_rms = 0;
    double both_rms = 0;

    /* Generate a test array */
    img = cpl_image_new(npt, 1, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -32, 32);
    cpl_image_add_scalar(img, 500);

    /* Should do nothing */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 0, 0, -1000);
        cpl_test_image_abs(testimg, img, 1.0e-6);
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Median filter */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 5, 0, -1000);
        cpl_test_abs(cpl_image_get_mean(testimg), 500, 1);
        median_rms = cpl_image_get_stdev(testimg);
        cpl_test_lt(median_rms,
                    0.9*cpl_image_get_stdev(img));
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Mean filter */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 0, 5, -1000);
        cpl_test_abs(cpl_image_get_mean(testimg), 500, 1);
        mean_rms = cpl_image_get_stdev(testimg);
        cpl_test_lt(mean_rms,
                    0.9*cpl_image_get_stdev(img));
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Both */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 5, 3, -1000);
        cpl_test_abs(cpl_image_get_mean(testimg), 500, 1);
        both_rms = cpl_image_get_stdev(testimg);
        cpl_test_lt(both_rms, median_rms);
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Larger kernel */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 15, 9, -1000);
        cpl_test_abs(cpl_image_get_mean(testimg), 500, 1);
        cpl_test_lt(cpl_image_get_stdev(testimg), 0.9*both_rms);
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Clean up */
    cpl_image_delete(img);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_filt1d with added bright lines
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_filt1d_brightlines(void)
{
    int npt = 1024;

    cpl_image *img = NULL, *testimg = NULL;
    float *img_data, *testimg_data;
    int ipt;

    /* Generate a test array */
    img = cpl_image_new(npt, 1, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    img_data = cpl_image_get_data_float(img);
    cpl_test_nonnull(img_data);

    if(img_data) {
        /* Set every 16th line 100 counts higher */
        for(ipt = 0; ipt < npt; ipt++) {
            img_data[ipt] = 500 + 100 * (ipt % 16 == 0);
        }
    }

    /* Should do nothing */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 0, 0, -1000);
        cpl_test_image_abs(testimg, img, 1.0e-6);
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Median filter should remove the bright lines entirely */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 5, 0, -1000);
        cpl_test_abs(cpl_image_get_mean(testimg), 500, 1);
        cpl_test_abs(cpl_image_get_stdev(testimg), 0, 1);
        cpl_test_abs(cpl_image_get_min(testimg), 500, 1);
        cpl_test_abs(cpl_image_get_max(testimg), 500, 1);
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Mean filter reduces intensity of bright lines */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 0, 5, -1000);

        /* Shouldn't affect the mean */
        cpl_test_abs(cpl_image_get_mean(testimg), cpl_image_get_mean(img), 1);

        /* or minimum */
        cpl_test_abs(cpl_image_get_min(testimg), 500, 1);

        /* but rms should be strongly suppressed */
        cpl_test_lt(cpl_image_get_stdev(testimg),
                    0.5*cpl_image_get_stdev(img));
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Flagging the bright lines shouldn't make any difference for the
       median filter */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 5, 0, 600);
        cpl_test_abs(cpl_image_get_mean(testimg), 500, 1);
        cpl_test_abs(cpl_image_get_stdev(testimg), 0, 1);
        cpl_test_abs(cpl_image_get_min(testimg), 500, 1);
        cpl_test_abs(cpl_image_get_max(testimg), 500, 1);
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* If we flag the bright lines, they shouldn't make it through
       the mean filter. */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 0, 5, 600);
        cpl_test_abs(cpl_image_get_mean(testimg), 500, 1);
        cpl_test_abs(cpl_image_get_stdev(testimg), 0, 1);
        cpl_test_abs(cpl_image_get_min(testimg), 500, 1);
        cpl_test_abs(cpl_image_get_max(testimg), 500, 1);
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Flagging the bright lines shouldn't make any difference for the
       combined filter */
    testimg = cpl_image_duplicate(img);

    testimg_data = cpl_image_get_data_float(testimg);
    cpl_test_nonnull(testimg_data);

    if(testimg_data) {
        qmost_filt1d(testimg_data, npt, 5, 5, 600);
        cpl_test_abs(cpl_image_get_mean(testimg), 500, 1);
        cpl_test_abs(cpl_image_get_stdev(testimg), 0, 1);
        cpl_test_abs(cpl_image_get_min(testimg), 500, 1);
        cpl_test_abs(cpl_image_get_max(testimg), 500, 1);
    }

    testimg_data = NULL;
    cpl_image_delete(testimg);

    /* Clean up */
    cpl_image_delete(img);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_skyfilt
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_skyfilt(void)
{
    int npt = 1024;

    cpl_image *img = NULL;
    cpl_image *testcont = NULL, *testline = NULL, *testsum;
    float *img_data;
    float *skycont = NULL;
    float *skyline = NULL;
    float *skymask = NULL;
    float sigma;
    int ipt;

    /* Generate a test array */
    img = cpl_image_new(npt, 1, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    img_data = cpl_image_get_data_float(img);
    cpl_test_nonnull(img_data);

    if(img_data) {
        /* Set every 16th pixel 100 counts higher */
        for(ipt = 0; ipt < npt; ipt++) {
            img_data[ipt] = 500 + 100 * (ipt % 16 == 0);
        }

        /* Output arrays */
        skycont = cpl_calloc(npt, sizeof(float));
        skyline = cpl_calloc(npt, sizeof(float));
        skymask = cpl_calloc(npt, sizeof(float));

        /* nx = 0 */
        qmost_skyfilt(0, img_data,
                      skycont, skyline, skymask,
                      &sigma, 25, 15, 3, 5.0, 5.0);

        /* The real value, this should work */
        qmost_skyfilt(npt, img_data,
                      skycont, skyline, skymask,
                      &sigma, 25, 15, 3, 5.0, 5.0);

        /* Continuum should have the correct value and low noise */
        testcont = cpl_image_wrap_float(npt, 1, skycont);
        testline = cpl_image_wrap_float(npt, 1, skyline);

        cpl_test_abs(cpl_image_get_mean(testcont), 500, 1);
        cpl_test_abs(cpl_image_get_stdev(testcont), 0, 1);
        cpl_test_abs(cpl_image_get_min(testcont), 500, 1);
        cpl_test_abs(cpl_image_get_max(testcont), 500, 1);

        /* Check that cont + line ~= original image */
        testsum = cpl_image_add_create(testcont, testline);

        cpl_test_image_abs(testsum, img, 0.1);

        cpl_image_delete(testsum);
        testsum = NULL;

        cpl_image_unwrap(testcont);
        testcont = NULL;

        cpl_image_unwrap(testline);
        testline = NULL;

        cpl_free(skycont);
        skycont = NULL;

        cpl_free(skyline);
        skyline = NULL;

        cpl_free(skymask);
        skymask = NULL;
    }

    /* Clean up */
    cpl_image_delete(img);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_filt1d module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_filt1d_noise();
    test_qmost_filt1d_brightlines();
    test_qmost_skyfilt();

    return cpl_test_end(0);
}

/**@}*/
