/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_imcombine_lite.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_imcombine_lite_test  Unit test of qmost_imcombine_lite
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

#define TOL 1.0e-15
#define FTOL 1.0e-5

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_imcombine_lite
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_imcombine_lite(void)
{
    cpl_imagelist *in_imglist = NULL;
    cpl_imagelist *in_varlist = NULL;
    double *in_explist = NULL;

    cpl_image *out_img = NULL;
    cpl_image *out_var = NULL;

    int nx = 128;
    int ny = 256;

    int ifile, nfiles = 9;
    float flatcts = 10000, flatsig = 100;
    cpl_image *img = NULL;
    cpl_image *var = NULL;

    double median, sigma;

    int xrej, icalc, scaletype;

    cpl_error_code code;

    cpl_propertylist *qclist = NULL;

    /* Create empty inputs */
    in_imglist = cpl_imagelist_new();
    in_varlist = cpl_imagelist_new();

    /* NULL input checks */
    code = qmost_imcombine_lite(NULL, NULL, NULL,
                                QMOST_MEANCALC, 0, 0, 0, 5.0,
                                &out_img, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_imcombine_lite(in_imglist, NULL, NULL,
                                QMOST_MEANCALC, 0, 0, 0, 5.0,
                                NULL, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty imagelist */
    code = qmost_imcombine_lite(in_imglist, NULL, NULL,
                                QMOST_MEANCALC, 0, 0, 0, 5.0,
                                &out_img, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Prepare a proper test flat series */
    in_explist = cpl_calloc(nfiles, sizeof(double));

    for(ifile = 0; ifile < nfiles; ifile++) {
        /* Simulate flat */
        img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
        qmost_test_fill_noise_rounded_gauss(img, flatcts, flatsig);

        /* Add some outliers in different places on each image so
         * they should get rejected */
        cpl_image_set(img, ifile+1, ifile+1, 65535);

        /* Clamp to range of 16-bit unsigned */
        cpl_image_threshold(img, 0, 65535, 0, 65535);

        /* Add to list */
        cpl_imagelist_set(in_imglist, img, ifile);
        /* now owned by imagelist */

        var = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
        cpl_image_fill_window(var, 1, 1, nx, ny, flatcts);

        cpl_imagelist_set(in_varlist, var, ifile);
        /* now owned by imagelist */

        in_explist[ifile] = 5.0;
    }

    /* Invalid calc type */
    code = qmost_imcombine_lite(in_imglist, NULL, NULL,
                                42, 0, 0, 0, 5.0,
                                &out_img, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* This should work, basic mean calculation without variance */
    code = qmost_imcombine_lite(in_imglist, NULL, NULL,
                                QMOST_MEANCALC, 0, 0, 0, 5.0,
                                &out_img, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    median = cpl_image_get_mad(out_img, &sigma);
    sigma *= CPL_MATH_STD_MAD;

    cpl_test_abs(median, flatcts, 10.0);
    cpl_test_abs(sigma, flatsig/sqrt(nfiles), 5.0);

    cpl_image_delete(out_img);
    out_img = NULL;

    /* Fibre spectra mode, changes sigma so needs modified test */
    code = qmost_imcombine_lite(in_imglist, NULL, NULL,
                                QMOST_MEANCALC, 2, 1, 0, 5.0,
                                &out_img, NULL, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    median = cpl_image_get_mad(out_img, &sigma);
    sigma *= CPL_MATH_STD_MAD;

    cpl_test_abs(median, flatcts, 25.0);
    cpl_test_abs(sigma, flatsig/sqrt(nfiles), 25.0);

    cpl_image_delete(out_img);
    out_img = NULL;

    /* This should also work but not populate variance due to lacking
     * input. */
    out_var = NULL;

    code = qmost_imcombine_lite(in_imglist, NULL, NULL,
                                QMOST_MEDIANCALC, 0, 0, 0, 5.0,
                                &out_img, &out_var, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_null(out_var);

    median = cpl_image_get_mad(out_img, &sigma);
    sigma *= CPL_MATH_STD_MAD;

    cpl_test_abs(median, flatcts, 10.0);
    cpl_test_abs(sigma, flatsig/sqrt(nfiles), 5.0);

    cpl_image_delete(out_img);
    out_img = NULL;

    /* Now test each of the possible calculation types */
    for(xrej = 0; xrej <= 1; xrej++) {
        for(icalc = 1; icalc <= 2; icalc++) {
            for(scaletype = 0; scaletype <= 3; scaletype++) {
                qclist = cpl_propertylist_new();

                code = qmost_imcombine_lite(in_imglist, in_varlist, in_explist,
                                            icalc > 1 ?
                                            QMOST_MEDIANCALC :
                                            QMOST_MEANCALC,
                                            scaletype, 0, xrej, 5.0,
                                            &out_img, &out_var, qclist);
                cpl_test_eq_error(code, CPL_ERROR_NONE);
                
                median = cpl_image_get_mad(out_img, &sigma);
                sigma *= CPL_MATH_STD_MAD;
                
                cpl_test_abs(median, flatcts, 10.0);
                cpl_test_abs(sigma, flatsig/sqrt(nfiles), 5.0);
                
                cpl_image_delete(out_img);
                out_img = NULL;

                cpl_image_delete(out_var);
                out_var = NULL;

                cpl_propertylist_delete(qclist);
                qclist = NULL;
            }
        }
    }

    /* Clean up inputs */
    cpl_imagelist_delete(in_imglist);
    in_imglist = NULL;

    cpl_imagelist_delete(in_varlist);
    in_varlist = NULL;

    cpl_free(in_explist);
    in_explist = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_imcombine_lite module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_imcombine_lite();

    return cpl_test_end(0);
}

/**@}*/
