/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_linear.h"
#include "qmost_lininfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_linear_test  Unit test of qmost_linear
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

#define TOL 1.0e-15
#define FTOL 1.0e-5

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_flatstats_compute
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_flatstats_compute(void)
{
    int nx = 128;
    int ny = 256;
    double ctspersec = 690;
    int iampx, iampy, nampsx = 2, nampsy = 2;
    int iamp, namps = 4;

    double exptime, mjd, flatcts, flatsig;

    qmost_flatstats self;
    cpl_image *img = NULL;
    cpl_propertylist *hdr = NULL;
    cpl_propertylist *testhdr = NULL;
    cpl_propertylist *qclist = NULL;
    char *key = NULL;
    char *value = NULL;

    cpl_error_code code;

    /* Simulate a flat */
    exptime = 20.0;
    mjd = 59000;
    flatcts = ctspersec * exptime;
    flatsig = sqrt(flatcts);

    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    qmost_test_fill_noise_rounded_gauss(img, flatcts, flatsig);
    cpl_image_threshold(img, 0, 65535, 0, 65535);
    
    hdr = cpl_propertylist_new();
    qclist = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_flatstats_compute(NULL, img, hdr, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_flatstats_compute(&self, NULL, hdr, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_flatstats_compute(&self, img, NULL, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_flatstats_compute(&self, img, hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* PHDU headers missing */
    code = qmost_flatstats_compute(&self, img, hdr, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_update_double(hdr, "EXPTIME", exptime);

    code = qmost_flatstats_compute(&self, img, hdr, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_update_double(hdr, "MJD-OBS", mjd);

    /* PHDU complete but qclist headers still missing */
    code = qmost_flatstats_compute(&self, img, hdr, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* add them */
    cpl_propertylist_update_int(qclist, "ESO DRS NAMPS", namps);

    code = qmost_flatstats_compute(&self, img, hdr, qclist);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add some nonsense for the amp sect */
    testhdr = cpl_propertylist_duplicate(qclist);

    cpl_propertylist_update_string(testhdr,
                                   "ESO DRS AMP1 SECT",
                                   "Basil");

    code = qmost_flatstats_compute(&self, img, hdr, testhdr);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_propertylist_delete(testhdr);
    testhdr = NULL;

    /* and out of bounds */
    testhdr = cpl_propertylist_duplicate(qclist);

    cpl_propertylist_update_string(testhdr,
                                   "ESO DRS AMP1 SECT",
                                   "[-1:9999,42:100]");

    code = qmost_flatstats_compute(&self, img, hdr, testhdr);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_propertylist_delete(testhdr);
    testhdr = NULL;

    /* Proper inputs */
    for(iampy = 0; iampy < nampsy; iampy++) {
        for(iampx = 0; iampx < nampsx; iampx++) {
            key = cpl_sprintf("ESO DRS AMP%d SECT", iampy*nampsx+iampx+1);
            value = cpl_sprintf("[%d:%d,%d:%d]",
                                1 + iampx * nx / nampsx,
                                (iampx + 1) * nx / nampsx,
                                1 + iampy * ny / nampsy,
                                (iampy + 1) * ny / nampsy);

            cpl_propertylist_update_string(qclist, key, value);

            cpl_free(key);
            cpl_free(value);
        }
    }

    code = qmost_flatstats_compute(&self, img, hdr, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iamp = 0; iamp < namps; iamp++) {
        cpl_test_abs(self.median[iamp], flatcts, 100);
        cpl_test_abs(self.sigma[iamp], flatsig, 10);
    }

    qmost_flatstats_free(&self);

    cpl_image_delete(img);
    img = NULL;

    cpl_propertylist_delete(hdr);
    hdr = NULL;

    cpl_propertylist_delete(qclist);
    qclist = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_linfit
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_linfit(void)
{
    qmost_flatstats *lin_stats = NULL;
    qmost_flatstats *tmp_stats = NULL;
    cpl_table *table = NULL;
    cpl_propertylist *qclist = NULL;

    int nx = 128;
    int ny = 256;
    double ctspersec = 690;

    int ifile, ntosim = 10, nfiles = 0, imon, nmon = 9;
    int iampx, nampsx = 2;
    int iampy, nampsy = 2;
    int iamp, namps = 4;
    int nord = 2;
    int niter = 5;
    double clipthr = 3.0;
    double underexp = 1000;
    double overexp = 60000;

    double exptime, mjd, flatcts, flatsig;
    cpl_image *img;

    char *key, *value;
    const cpl_array *coefs;

    qmost_flatstats *mon_stats = NULL;
    cpl_propertylist *hdr;

    cpl_error_code code;

    /* Create empty inputs */
    lin_stats = cpl_calloc(ntosim, sizeof(qmost_flatstats));
    qclist = cpl_propertylist_new();

    /* Populate qclist */
    cpl_propertylist_update_int(qclist, "ESO DRS NAMPS", namps);

    for(iampy = 0; iampy < nampsy; iampy++) {
        for(iampx = 0; iampx < nampsx; iampx++) {
            key = cpl_sprintf("ESO DRS AMP%d SECT", iampy*nampsx+iampx+1);
            value = cpl_sprintf("[%d:%d,%d:%d]",
                                1 + iampx * nx / nampsx,
                                (iampx + 1) * nx / nampsx,
                                1 + iampy * ny / nampsy,
                                (iampy + 1) * ny / nampsy);

            cpl_propertylist_update_string(qclist, key, value);

            cpl_free(key);
            cpl_free(value);
        }
    }

    /* NULL input checks */
    code = qmost_linfit(NULL, ntosim, NULL, 0, 1,
                        &table, qclist,
                        nord, niter, clipthr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_linfit(lin_stats, ntosim, NULL, 0, 1,
                        NULL, qclist,
                        nord, niter, clipthr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_linfit(lin_stats, ntosim, NULL, 0, 1,
                        &table, NULL,
                        nord, niter, clipthr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Zero input files */
    code = qmost_linfit(lin_stats, 0, NULL, 0, 1,
                        &table, qclist,
                        nord, niter, clipthr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Prepare a proper test flat series */
    nfiles = 0;

    for(ifile = 0; ifile < ntosim; ifile++) {
        /* Simulate flat with non-linearity 1% per 10k  */
        exptime = 10.0 * (ifile+1);
        mjd = 59000 + 0.01 * ifile;
        flatcts = ctspersec * exptime;
        flatcts += 1.0e-6 * flatcts*flatcts;  /* nonlin 1% per 10k */
        flatsig = sqrt(flatcts);

        img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
        qmost_test_fill_noise_rounded_gauss(img, flatcts, flatsig);
        cpl_image_threshold(img, 0, 65535, 0, 65535);

        /* Header */
        hdr = cpl_propertylist_new();

        cpl_propertylist_update_double(hdr, "EXPTIME", exptime);
        cpl_propertylist_update_double(hdr, "MJD-OBS", mjd);

        /* Do analysis */
        code = qmost_flatstats_compute(lin_stats + nfiles,
                                       img,
                                       hdr,
                                       qclist);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        cpl_image_delete(img);
        img = NULL;

        cpl_propertylist_delete(hdr);
        hdr = NULL;

        if(qmost_flatstats_check(lin_stats + nfiles, underexp, overexp)) {
            nfiles++;
        }
        else {
            qmost_flatstats_free(lin_stats + nfiles);
        }
    }

    /* This should work */
    code = qmost_linfit(lin_stats, nfiles, NULL, 0, 1,
                        &table, qclist,
                        nord, niter, clipthr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iamp = 0; iamp < namps; iamp++) {
        coefs = cpl_table_get_array(table, "coefs", iamp);
        cpl_test_abs(cpl_array_get(coefs, 0, NULL), 0, DBL_EPSILON);
        cpl_test_abs(cpl_array_get(coefs, 1, NULL), 1, DBL_EPSILON);
        cpl_test_abs(cpl_array_get(coefs, 2, NULL), -1.0e-6, 5.0e-7);
    }

    cpl_test_lt(cpl_propertylist_get_double(qclist, "ESO QC LIN RMS MEAN"),
                10);
    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC LIN NL10K"),
                 1, 0.5);

    cpl_table_delete(table);
    table = NULL;

    /* Check singular matrix by giving the same input every time, we
     * should get back a null set of coefficients. */
    tmp_stats = cpl_calloc(nfiles, sizeof(qmost_flatstats));

    for(ifile = 0; ifile < nfiles; ifile++) {
        tmp_stats[ifile] = lin_stats[0];
    }

    code = qmost_linfit(tmp_stats, nfiles, NULL, 0, 1,
                        &table, qclist,
                        nord, niter, clipthr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iamp = 0; iamp < namps; iamp++) {
        coefs = cpl_table_get_array(table, "coefs", iamp);
        cpl_test_abs(cpl_array_get(coefs, 0, NULL), 0, DBL_EPSILON);
        cpl_test_abs(cpl_array_get(coefs, 1, NULL), 1, DBL_EPSILON);
        cpl_test_abs(cpl_array_get(coefs, 2, NULL), 0, DBL_EPSILON);
    }

    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC LIN NL10K"),
                 0, 0.5);

    cpl_table_delete(table);
    table = NULL;

    cpl_free(tmp_stats);
    tmp_stats = NULL;

    /* Make monitor series and do analysis */
    mon_stats = cpl_calloc(nmon, sizeof(qmost_flatstats));

    for(imon = 0; imon < nmon; imon++) {
        /* Simulate monitor flat */
        exptime = 20.0;
        mjd = 59000 + 0.01 * (imon + 0.5);
        flatcts = ctspersec * exptime;
        flatsig = sqrt(flatcts);

        img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
        qmost_test_fill_noise_rounded_gauss(img, flatcts, flatsig);
        cpl_image_threshold(img, 0, 65535, 0, 65535);

        hdr = cpl_propertylist_new();
        cpl_propertylist_update_double(hdr, "EXPTIME", exptime);
        cpl_propertylist_update_double(hdr, "MJD-OBS", mjd);

        /* Do analysis */
        code = qmost_flatstats_compute(mon_stats + imon, img, hdr, qclist);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        cpl_image_delete(img);
        img = NULL;

        cpl_propertylist_delete(hdr);
        hdr = NULL;
    }

    /* Monitor only */
    code = qmost_linfit(lin_stats, nfiles, mon_stats, nmon, 0,
                        &table, qclist,
                        nord, niter, clipthr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iamp = 0; iamp < namps; iamp++) {
        coefs = cpl_table_get_array(table, "coefs", iamp);
        cpl_test_abs(cpl_array_get(coefs, 0, NULL), 0, DBL_EPSILON);
        cpl_test_abs(cpl_array_get(coefs, 1, NULL), 1, DBL_EPSILON);
        cpl_test_abs(cpl_array_get(coefs, 2, NULL), -1.0e-6, 5.0e-7);
    }

    cpl_test_lt(cpl_propertylist_get_double(qclist, "ESO QC LIN RMS MEAN"),
                50);
    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC LIN NL10K"),
                 1, 0.5);

    cpl_table_delete(table);
    table = NULL;

    /* With correction */
    code = qmost_linfit(lin_stats, nfiles, mon_stats, nmon, 1,
                        &table, qclist,
                        nord, niter, clipthr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    for(iamp = 0; iamp < namps; iamp++) {
        coefs = cpl_table_get_array(table, "coefs", iamp);
        cpl_test_abs(cpl_array_get(coefs, 0, NULL), 0, DBL_EPSILON);
        cpl_test_abs(cpl_array_get(coefs, 1, NULL), 1, DBL_EPSILON);
        cpl_test_abs(cpl_array_get(coefs, 2, NULL), -1.0e-6, 5.0e-7);
    }

    cpl_test_lt(cpl_propertylist_get_double(qclist, "ESO QC LIN RMS MEAN"),
                50);
    cpl_test_abs(cpl_propertylist_get_double(qclist, "ESO QC LIN NL10K"),
                 1, 0.5);

    cpl_table_delete(table);
    table = NULL;

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    for(ifile = 0; ifile < nfiles; ifile++) {
        qmost_flatstats_free(lin_stats + ifile);
    }

    cpl_free(lin_stats);
    lin_stats = NULL;

    for(imon = 0; imon < nmon; imon++) {
        qmost_flatstats_free(mon_stats + imon);
    }

    cpl_free(mon_stats);
    mon_stats = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_lincor
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_lincor(void)
{
    cpl_image *inimg = NULL;
    cpl_image *img = NULL;
    cpl_image *tmpimg = NULL;
    cpl_image *tmpadj = NULL;
    cpl_image *ansimg = NULL;
    cpl_propertylist *hdr = NULL;
    cpl_propertylist *testhdr = NULL;
    cpl_table *tbl = NULL;

    float *inimgbuf = NULL;
    int nx = 256;
    int ny = 512;
    int ipix, npix;

    cpl_mask *maska = NULL;
    cpl_mask *maskb = NULL;

    qmost_lininfo lins[2];
    cpl_size degree;

    cpl_error_code code;

    /* Make a test image */
    inimg = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(inimg, -0.5, 0.5);
    inimgbuf = cpl_image_get_data_float(inimg);
    cpl_test_nonnull(inimgbuf);

    if(inimgbuf != NULL) {
        npix = nx * ny;

        /* Test ramp -1024 to 65536 + 1024 dithered by noise from above */
        for(ipix = 0; ipix < npix; ipix++) {
            inimgbuf[ipix] += (ipix % 67584) - 1024;
        }
    }

    img = cpl_image_duplicate(inimg);

    /* Empty FITS header */
    hdr = cpl_propertylist_new();

    /* Empty table */
    tbl = cpl_table_new(0);

    /* NULL input tests */
    code = qmost_lincor(NULL, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_lincor(img, NULL, tbl);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_lincor(img, hdr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No NAMPS */
    code = qmost_lincor(img, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add NAMPS, should now make it to linearity table check and fail
     * that because the table is empty. */
    cpl_propertylist_update_int(hdr, "ESO DRS NAMPS", 2);

    code = qmost_lincor(img, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_table_delete(tbl);
    tbl = NULL;

    /* Create the linearity table.  Should now pass the check but
     * fail the read because it's still empty. */
    qmost_lincrtab(&tbl, 10, 3);

    code = qmost_lincor(img, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add one set, intentionally not enough so we fail the check on
     * number of amps.  The linearity coefficients are set up for some
     * mild linearity correction. */
    qmost_lincrinfo(10, 3, &(lins[0]));
    qmost_lincrinfo(10, 3, &(lins[1]));

    degree = 1;
    cpl_polynomial_set_coeff(lins[0].coefs, &degree, 1.0);

    degree = 2;
    cpl_polynomial_set_coeff(lins[0].coefs, &degree, -1.0e-6);

    degree = 1;
    cpl_polynomial_set_coeff(lins[1].coefs, &degree, 1.0);

    degree = 2;
    cpl_polynomial_set_coeff(lins[1].coefs, &degree, -2.0e-6);

    qmost_linwrite(tbl, hdr, lins, 1);

    code = qmost_lincor(img, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    /* Now write the proper table with the right size */
    qmost_linwrite(tbl, hdr, lins, 2);

    qmost_lindelinfo(&(lins[0]));
    qmost_lindelinfo(&(lins[1]));

    /* Make an image containing the answer, start with left half */
    ansimg = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);

    tmpimg = cpl_image_extract(inimg, 1, 1, 128, 512);
    tmpadj = cpl_image_multiply_scalar_create(tmpimg, -1.0e-6);
    cpl_image_add_scalar(tmpadj, 1.0);
    cpl_image_multiply(tmpimg, tmpadj);
    cpl_image_delete(tmpadj);

    cpl_image_copy(ansimg, tmpimg, 1, 1);
    cpl_image_delete(tmpimg);

    /* Now the right half */
    tmpimg = cpl_image_extract(inimg, 129, 1, 256, 512);
    tmpadj = cpl_image_multiply_scalar_create(tmpimg, -2.0e-6);
    cpl_image_add_scalar(tmpadj, 1.0);
    cpl_image_multiply(tmpimg, tmpadj);
    cpl_image_delete(tmpadj);

    cpl_image_copy(ansimg, tmpimg, 129, 1);
    cpl_image_delete(tmpimg);

    /* Mark as bad any pixels that fall outside the LUT range so the
     * comparison works. */
    maska = cpl_mask_threshold_image_create(inimg, -1000, 65535);
    cpl_mask_not(maska);

    maskb = cpl_mask_threshold_image_create(ansimg, -1000, 65535);
    cpl_mask_not(maskb);

    cpl_mask_or(maska, maskb);
    cpl_mask_delete(maskb);

    cpl_image_reject_from_mask(ansimg, maska);
    cpl_mask_delete(maska);

    /* This should now make it into the amp loop and fail due to
     * missing AMP SECT header */
    code = qmost_lincor(img, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add some nonsense */
    testhdr = cpl_propertylist_duplicate(hdr);

    cpl_propertylist_update_string(testhdr,
                                   "ESO DRS AMP1 SECT",
                                   "Basil");

    code = qmost_lincor(img, testhdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    cpl_propertylist_delete(testhdr);
    testhdr = NULL;

    /* and out of bounds to test that */
    testhdr = cpl_propertylist_duplicate(hdr);

    cpl_propertylist_update_string(testhdr,
                                   "ESO DRS AMP1 SECT",
                                   "[-1:9999,42:100]");

    code = qmost_lincor(img, testhdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_propertylist_delete(testhdr);
    testhdr = NULL;

    /* Add the missing headers */
    cpl_propertylist_update_string(hdr,
                                   "ESO DRS AMP1 SECT",
                                   "[1:128,1:512]");
    cpl_propertylist_update_string(hdr,
                                   "ESO DRS AMP2 SECT",
                                   "[129:256,1:512]");

    /* This should work */
    code = qmost_lincor(img, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_image_abs(img, ansimg, 1.0);

    cpl_image_delete(img);
    img = NULL;

    /* This should also work */
    img = cpl_image_cast(inimg, CPL_TYPE_DOUBLE);

    code = qmost_lincor(img, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_image_abs(img, ansimg, 1.0);

    cpl_image_delete(img);
    img = NULL;
    
    /* This should not */
    img = cpl_image_cast(inimg, CPL_TYPE_INT);

    code = qmost_lincor(img, hdr, tbl);
    cpl_test_eq_error(code, CPL_ERROR_UNSUPPORTED_MODE);

    cpl_image_delete(img);
    img = NULL;

    cpl_image_delete(inimg);
    inimg = NULL;

    cpl_image_delete(ansimg);
    ansimg = NULL;

    cpl_propertylist_delete(hdr);
    hdr = NULL;

    cpl_table_delete(tbl);
    tbl = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_gen_bpm
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_gen_bpm(void)
{
    cpl_imagelist *flats = NULL;
    cpl_propertylist *qclist = NULL;
    cpl_mask *bpm = NULL;
    cpl_mask *tst_bpm = NULL;

    int nx = 128;
    int ny = 256;
    double ctspersec = 690;

    int ifile, nfiles = 8;  /* ensure we don't saturate */

    double exptime, flatcts, flatsig;
    cpl_image *img;

    cpl_error_code code;

    /* Create empty inputs */
    flats = cpl_imagelist_new();
    qclist = cpl_propertylist_new();

    /* NULL input checks */
    code = qmost_gen_bpm(NULL, qclist, 8, 8, 0.25, &bpm);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_gen_bpm(flats, NULL, 8, 8, 0.25, &bpm);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_gen_bpm(flats, qclist, 8, 8, 0.25, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty imagelist */
    code = qmost_gen_bpm(flats, qclist, 8, 8, 0.25, &bpm);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Prepare a proper test flat series */
    for(ifile = 0; ifile < nfiles; ifile++) {
        /* Simulate flat with non-linearity 1% per 10k  */
        exptime = 10.0 * (ifile+1);
        flatcts = ctspersec * exptime;
        flatcts += 1.0e-6 * flatcts*flatcts;  /* nonlin 1% per 10k */
        flatsig = sqrt(flatcts);

        img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
        qmost_test_fill_noise_rounded_gauss(img, flatcts, flatsig);
        cpl_image_threshold(img, 0, 65535, 0, 65535);

        /* Add to list */
        cpl_imagelist_set(flats, img, ifile);
        /* now owned by imagelist */
    }

    /* This should work, no bad pixels */
    code = qmost_gen_bpm(flats, qclist, 8, 8, 0.25, &bpm);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test(cpl_mask_is_empty(bpm));

    cpl_mask_delete(bpm);
    bpm = NULL;

    /* Now put in a dead pixel and a saturated pixel */
    for(ifile = 0; ifile < nfiles; ifile++) {
        img = cpl_imagelist_get(flats, ifile);
        cpl_image_set(img, 42, 43, 0);
        cpl_image_set(img, 43, 42, 65535);
    }

    code = qmost_gen_bpm(flats, qclist, 8, 8, 0.25, &bpm);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* ...which should be recovered */
    tst_bpm = cpl_mask_new(nx, ny);
    cpl_mask_set(tst_bpm, 42, 43, 1);
    cpl_mask_set(tst_bpm, 43, 42, 1);
    cpl_mask_xor(tst_bpm, bpm);  /* 1 wherever not equal */

    cpl_test(cpl_mask_is_empty(tst_bpm));

    cpl_mask_delete(tst_bpm);
    tst_bpm = NULL;

    cpl_mask_delete(bpm);
    bpm = NULL;

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    cpl_imagelist_delete(flats);
    flats = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_linear module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_flatstats_compute();
    test_qmost_linfit();
    test_qmost_lincor();
    test_qmost_gen_bpm();

    return cpl_test_end(0);
}

/**@}*/
