/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_scattered.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_scattered_test  Unit test of qmost_scattered
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_scattered
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_scattered(void)
{
    int nx = 1024;
    int ny = 2048;
    int nbsize = 128;

    cpl_image *img = NULL, *refimg = NULL, *testimg = NULL, *testsubimg = NULL;
    cpl_mask *mask = NULL;
    float *img_data;
    cpl_binary *mask_data;
    int x, y;

    cpl_propertylist *hdr = NULL;
    cpl_error_code code;

    double skylevel, skylevel1, skylevel2;

    /* Generate a test image with known background level */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    img_data = cpl_image_get_data_float(img);
    cpl_test_nonnull(img_data);

    if(img_data) {
        /* Ramp from 500 at top to 600 at bottom */
        for(y = 0; y < ny; y++) {
            for(x = 0; x < nx; x++) {
                img_data[y*nx+x] += 500 + (100.0*y) / ny;

                /* and occasional outliers */
                if(y*nx+x % 99 == 0)
                    img_data[y*nx+x] += 10000;
            }
        }
    }

    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_scattered(NULL, NULL, NULL, nbsize, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_scattered(testimg, NULL, NULL, nbsize, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_scattered(NULL, hdr, NULL, nbsize, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid nbsize */
    code = qmost_scattered(refimg, hdr, NULL, 0, 0);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No mask, leave background on */
    refimg = cpl_image_duplicate(img);

    code = qmost_scattered(refimg, hdr, NULL, nbsize, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    skylevel = cpl_image_get_median(refimg);
    cpl_test_abs(skylevel, 550, 1);

    testsubimg = cpl_image_extract(refimg, 1, 1, nx, ny/4);
    skylevel1 = cpl_image_get_median(testsubimg);
    cpl_image_delete(testsubimg);

    testsubimg = cpl_image_extract(refimg, 1, 3*ny/4, nx, ny);
    skylevel2 = cpl_image_get_median(testsubimg);
    cpl_image_delete(testsubimg);

    cpl_test_abs(skylevel1, skylevel, 1);
    cpl_test_abs(skylevel2, skylevel, 1);
    cpl_test_abs(skylevel1, skylevel2, 1);

    /* No mask, take background off */
    testimg = cpl_image_duplicate(img);

    code = qmost_scattered(testimg, hdr, NULL, nbsize, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* The background estimation methods differ quite a bit so we need
       to be fairly lax here. */
    skylevel = cpl_image_get_median(testimg);
    cpl_test_abs(skylevel, 0, 10);

    cpl_image_delete(testimg);

    /* With mask, wrong size */
    mask = cpl_mask_new(nx-1, ny+1);

    code = qmost_scattered(img, hdr, mask, nbsize, 0);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_mask_delete(mask);

    /* With null mask, shouldn't affect result */
    mask = cpl_mask_new(nx, ny);

    testimg = cpl_image_duplicate(img);

    code = qmost_scattered(testimg, hdr, mask, nbsize, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_image_abs(testimg, refimg, 1);

    cpl_image_delete(testimg);

    /* Set some pixels bad, shouldn't affect the answer */
    mask_data = cpl_mask_get_data(mask);
    cpl_test_nonnull(mask_data);

    if(mask_data) {
        for(y = 0; y < ny; y++) {
            for(x = 0; x < nx; x++) {
                mask_data[y*nx+x] = (x+y) % 4 == 0 ? 1 : 0;
            }
        }
    }

    testimg = cpl_image_duplicate(img);

    code = qmost_scattered(testimg, hdr, mask, nbsize, 0);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_image_abs(testimg, refimg, 1);

    cpl_image_delete(testimg);

    cpl_mask_delete(mask);

    /* Clean up */
    cpl_image_delete(refimg);
    cpl_propertylist_delete(hdr);
    cpl_image_delete(img);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_scattered_qc
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_scattered_qc(void)
{
    int nx = 1024;
    int ny = 2048;

    cpl_image *img = NULL;
    cpl_mask *mask = NULL;
    float *img_data;
    int x, y;

    cpl_propertylist *hdr = NULL;
    cpl_propertylist *qclist = NULL;
    cpl_error_code code;

    double value;

    /* Generate a test image with known background level */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    img_data = cpl_image_get_data_float(img);
    cpl_test_nonnull(img_data);

    if(img_data) {
        /* Ramp from 500 at top to 600 at bottom */
        for(y = 0; y < ny; y++) {
            for(x = 0; x < nx; x++) {
                img_data[y*nx+x] += 500 + (100.0*y) / ny;
            }
        }
    }

    hdr = cpl_propertylist_new();
    qclist = cpl_propertylist_new();

    /* With mask, wrong size */
    mask = cpl_mask_new(nx-1, ny+1);

    code = qmost_scattered_qc(img, hdr, mask, qclist);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    cpl_mask_delete(mask);

    /* With null mask */
    mask = cpl_mask_new(nx, ny);

    code = qmost_scattered_qc(img, hdr, mask, qclist);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    value = cpl_propertylist_get_double(qclist, "ESO QC SCATTL MED");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 550, 1);

    value = cpl_propertylist_get_double(qclist, "ESO QC SCATTL RMS");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, CPL_MATH_STD_MAD * 25, 1);  /* MAD should be 25 */

    cpl_mask_delete(mask);

    /* Clean up */
    cpl_image_delete(img);
    cpl_propertylist_delete(hdr);
    cpl_propertylist_delete(qclist);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_scattered module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_scattered();
    test_qmost_scattered_qc();

    return cpl_test_end(0);
}

/**@}*/
