/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_constants.h"
#include "qmost_traceinfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_traceinfo_test  Unit test of qmost_traceinfo
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

#define TOL 1.0e-15

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_trcreate
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_trcreate(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    /* NULL input */
    code = qmost_trcreate(NULL, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid dimensions */
    code = qmost_trcreate(&table, -1, 10);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    code = qmost_trcreate(&table, 7, -1);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Create a simple table */
    code = qmost_trcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* Test it has the right format */
    code = qmost_trchk(table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Delete table */
    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_tropen
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_tropen(void)
{
    cpl_table *table = NULL;
    cpl_propertylist *hdr = NULL;
    cpl_error_code code;

    qmost_traceinfo *tr = NULL;
    int ntr = 0;

    /* Empty table */
    table = cpl_table_new(0);
    cpl_test_nonnull(table);
    hdr = cpl_propertylist_new();
    cpl_test_nonnull(hdr);

    /* NULL inputs */
    code = qmost_tropen(NULL, hdr, &ntr, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_tropen(table, NULL, &ntr, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_tropen(table, hdr, NULL, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);
    code = qmost_tropen(table, hdr, &ntr, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Attempt to open empty table */
    code = qmost_tropen(table, hdr, &ntr, &tr);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_table_delete(table);
    table = NULL;

    /* Create a simple table with the correct format */
    code = qmost_trcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* And add a dummy entry */
    code = qmost_trinsert_dead(table, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Correct table format, missing header keywords */
    code = qmost_tropen(table, hdr, &ntr, &tr);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Add the missing headers */
    cpl_propertylist_update_int(hdr, "MINYST", 1);
    cpl_propertylist_update_int(hdr, "MAXYFN", 2);

    code = qmost_tropen(table, hdr, &ntr, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(ntr, 1);
    cpl_test_nonnull(tr);

    if(tr) {
        cpl_test_eq(tr[0].live, 0);
        cpl_test_eq(tr[0].minyst, 1);
        cpl_test_eq(tr[0].maxyfn, 2);
    }

    /* Clean up */
    qmost_trclose(ntr, &tr);

    cpl_table_delete(table);
    table = NULL;

    cpl_propertylist_delete(hdr);
    hdr = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_trchk
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_trchk(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    int i;
    char *colname = NULL;

    /* NULL input */
    code = qmost_trchk(NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Create an empty table */
    table = cpl_table_new(0);
    cpl_test_nonnull(table);
    
    /* Empty table */
    code = qmost_trchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Incorrect number of columns */
    code = cpl_table_new_column(table, "BADNAME", CPL_TYPE_INT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    code = qmost_trchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Correct number of columns but incorrect names */
    for(i = 0; i < 10; i++) {
        colname = cpl_sprintf("BAD%d", i);
        code = cpl_table_new_column(table, colname, CPL_TYPE_INT);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        cpl_free(colname);
    }
    
    code = qmost_trchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_table_delete(table);
    table = NULL;

    /* Proper format */
    code = qmost_trcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* Test it has the right format */
    code = qmost_trchk(table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Delete table */
    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_trread1
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_trread1(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    qmost_traceinfo trlive, tr;
    cpl_size zero;
    double value;

    int icol, ncols;
    const char *colname;
    cpl_array *colnames = NULL;
    cpl_table *test_tbl = NULL;

    /* Fill struct with nonsense */
    memset(&tr, 0xa5, sizeof(tr));

    /* Make empty table */
    code = qmost_trcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* NULL inputs */
    code = qmost_trread1(NULL, 1, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_trread1(table, 1, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row number */
    code = qmost_trread1(table, 0, &tr);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Valid row number beyond end of table */
    code = qmost_trread1(table, 1, &tr);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    
    /* Now populate table and run again, should work this time */
    code = qmost_trinsert_dead(table, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = qmost_trread1(table, 1, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(tr.live, 0);

    /* Add a live fibre */
    memset(&trlive, 0, sizeof(trlive));
    trlive.specnum = 2;
    trlive.live = 1;
    trlive.yst = 3;
    trlive.yfn = 4;
    trlive.yref = 5;
    trlive.trrms = 6;
    trlive.nord = 0;
    trlive.coefs = cpl_polynomial_new(1);
    zero = 0;
    cpl_polynomial_set_coeff(trlive.coefs, &zero, 7);
    trlive.npos = 1;
    trlive.xpos = cpl_malloc(sizeof(double));
    trlive.xpos[0] = 8;
    trlive.ypos = cpl_malloc(sizeof(double));
    trlive.ypos[0] = 9;
    trlive.fwhm = cpl_malloc(sizeof(float));
    trlive.fwhm[0] = 10;
    trlive.peak = cpl_malloc(sizeof(float));
    trlive.peak[0] = 11;
    trlive.contrast = cpl_malloc(sizeof(float));
    trlive.contrast[0] = 12;

    code = qmost_trwrite1(table, trlive, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* and read it back */
    code = qmost_trread1(table, 2, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(tr.specnum, trlive.specnum);
    cpl_test_eq(tr.live, trlive.live);
    cpl_test_eq(tr.yst, trlive.yst);
    cpl_test_eq(tr.yfn, trlive.yfn);
    cpl_test_abs(tr.yref, trlive.yref, TOL);
    cpl_test_abs(tr.trrms, trlive.trrms, TOL);
    cpl_test_eq(tr.nord, trlive.nord);
    cpl_test_nonnull(tr.coefs);
    zero = 0;
    value = cpl_polynomial_get_coeff(tr.coefs, &zero);
    cpl_test_abs(value, 7, TOL);
    cpl_test_eq(tr.npos, 1);
    cpl_test_nonnull(tr.xpos);
    if(tr.xpos) {
        cpl_test_abs(tr.xpos[0], trlive.xpos[0], TOL);
    }
    cpl_test_nonnull(tr.ypos);
    if(tr.ypos) {
        cpl_test_abs(tr.ypos[0], trlive.ypos[0], TOL);
    }
    cpl_test_nonnull(tr.fwhm);
    if(tr.fwhm) {
        cpl_test_abs(tr.fwhm[0], trlive.fwhm[0], TOL);
    }
    cpl_test_nonnull(tr.peak);
    if(tr.peak) {
        cpl_test_abs(tr.peak[0], trlive.peak[0], TOL);
    }
    cpl_test_nonnull(tr.contrast);
    if(tr.contrast) {
        cpl_test_abs(tr.contrast[0], trlive.contrast[0], TOL);
    }

    cpl_polynomial_delete(tr.coefs);
    cpl_free(tr.xpos);
    cpl_free(tr.ypos);
    cpl_free(tr.fwhm);
    cpl_free(tr.peak);
    cpl_free(tr.contrast);

    /* Clean up */
    cpl_polynomial_delete(trlive.coefs);
    cpl_free(trlive.xpos);
    cpl_free(trlive.ypos);
    cpl_free(trlive.fwhm);
    cpl_free(trlive.peak);
    cpl_free(trlive.contrast);

    /* Do a series of tests where we feed in the table with each
     * column missing to set off the individual error handlers for
     * the column reads.  Need to use the live fibre so we do all of
     * the tests. */
    colnames = cpl_table_get_column_names(table);
    ncols = cpl_table_get_ncol(table);

    for(icol = 0; icol < ncols; icol++) {
        colname = cpl_array_get_string(colnames, icol);

        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_trread1(test_tbl, 2, &tr);
        if(icol >= ncols - 3) {
            /* FWHM, peak and contrast are optional */
            cpl_test_eq_error(code, CPL_ERROR_NONE);
            cpl_polynomial_delete(tr.coefs);
            cpl_free(tr.xpos);
            cpl_free(tr.ypos);
            cpl_free(tr.fwhm);
            cpl_free(tr.peak);
            cpl_free(tr.contrast);
        }
        else {
            cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);
        }

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_trinsert_dead
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_trinsert_dead(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    qmost_traceinfo tr;

    int icol, ncols;
    const char *colnames[] = { "fiblive", "yst", "yfn",
                               "tryref", "trrms", "trorder",
                               "npos" };
    const char *colname;
    cpl_table *test_tbl = NULL;

    /* Make empty table */
    code = qmost_trcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* NULL inputs */
    code = qmost_trinsert_dead(NULL, 1);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row number */
    code = qmost_trinsert_dead(table, 0);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Do a series of tests where we feed in the table with each
     * column this routine uses missing to set off the individual
     * error handlers for the column writes. */
    ncols = sizeof(colnames) / sizeof(colnames[0]);

    for(icol = 0; icol < ncols; icol++) {
        colname = colnames[icol];
        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_trinsert_dead(test_tbl, 1);
        cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    /* Insert a dead fibre */
    code = qmost_trinsert_dead(table, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Read it back */
    code = qmost_trread1(table, 1, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(tr.specnum, 1);
    cpl_test_eq(tr.live, 0);

    /* and another one in front of the first, check they get numbered
       correctly. */
    code = qmost_trinsert_dead(table, 1);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = qmost_trread1(table, 1, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(tr.specnum, 1);
    cpl_test_eq(tr.live, 0);

    code = qmost_trread1(table, 2, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(tr.specnum, 2);
    cpl_test_eq(tr.live, 0);

    /* Clean up */
    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_trrenumber
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_trrenumber(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    int *specnum = NULL;
    int specnum_ans[9] = { 1, 2, 3, 4, 5, 6, 7, 8, 9 };

    /* Make empty table */
    code = qmost_trcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* NULL input */
    code = qmost_trrenumber(NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty table */
    code = qmost_trrenumber(table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Give it something to do */
    code = cpl_table_set_size(table, 9);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    code = qmost_trrenumber(table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    specnum = cpl_table_get_data_int(table, "specnum");
    if(specnum) {
        qmost_test_arrays_int_abs(specnum, specnum_ans, 9, 0);
    }

    /* Clean up */
    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_trwrite1
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_trwrite1(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    qmost_traceinfo trlive, tr;
    cpl_size zero;
    double value;

    int icol, ncols;
    const char *colname;
    cpl_array *colnames = NULL;
    cpl_table *test_tbl = NULL;

    const char *dead_colnames[] = { "fiblive", "yst", "yfn",
                                    "tryref", "trrms", "trorder",
                                    "npos" };

    /* Fill struct with nonsense */
    memset(&tr, 0xa5, sizeof(tr));

    /* Make empty table */
    code = qmost_trcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* NULL inputs */
    code = qmost_trwrite1(NULL, tr, 1);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid row number */
    code = qmost_trwrite1(table, tr, 0);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* A live fibre */
    memset(&trlive, 0, sizeof(trlive));
    trlive.specnum = 2;
    trlive.live = 1;
    trlive.yst = 3;
    trlive.yfn = 4;
    trlive.yref = 5;
    trlive.trrms = 6;
    trlive.nord = 0;
    trlive.coefs = cpl_polynomial_new(1);
    zero = 0;
    cpl_polynomial_set_coeff(trlive.coefs, &zero, 7);
    trlive.npos = 1;
    trlive.xpos = cpl_malloc(sizeof(double));
    trlive.xpos[0] = 8;
    trlive.ypos = cpl_malloc(sizeof(double));
    trlive.ypos[0] = 9;
    trlive.fwhm = cpl_malloc(sizeof(float));
    trlive.fwhm[0] = 10;
    trlive.peak = cpl_malloc(sizeof(float));
    trlive.peak[0] = 11;
    trlive.contrast = cpl_malloc(sizeof(float));
    trlive.contrast[0] = 12;

    /* Do a series of tests where we feed in the table with each
     * column missing to set off the individual error handlers for
     * the column writes. */
    colnames = cpl_table_get_column_names(table);
    ncols = cpl_table_get_ncol(table);

    for(icol = 0; icol < ncols; icol++) {
        colname = cpl_array_get_string(colnames, icol);
        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_trwrite1(test_tbl, trlive, 2);
        cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    /* Finally write it properly */
    code = qmost_trwrite1(table, trlive, 2);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* and read it back */
    code = qmost_trread1(table, 2, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(tr.specnum, trlive.specnum);
    cpl_test_eq(tr.live, trlive.live);
    cpl_test_eq(tr.yst, trlive.yst);
    cpl_test_eq(tr.yfn, trlive.yfn);
    cpl_test_abs(tr.yref, trlive.yref, TOL);
    cpl_test_abs(tr.trrms, trlive.trrms, TOL);
    cpl_test_eq(tr.nord, trlive.nord);
    cpl_test_nonnull(tr.coefs);
    zero = 0;
    value = cpl_polynomial_get_coeff(tr.coefs, &zero);
    cpl_test_abs(value, 7, TOL);
    cpl_test_eq(tr.npos, 1);
    cpl_test_nonnull(tr.xpos);
    if(tr.xpos) {
        cpl_test_abs(tr.xpos[0], trlive.xpos[0], TOL);
    }
    cpl_test_nonnull(tr.ypos);
    if(tr.ypos) {
        cpl_test_abs(tr.ypos[0], trlive.ypos[0], TOL);
    }
    cpl_test_nonnull(tr.fwhm);
    if(tr.fwhm) {
        cpl_test_abs(tr.fwhm[0], trlive.fwhm[0], TOL);
    }
    cpl_test_nonnull(tr.peak);
    if(tr.peak) {
        cpl_test_abs(tr.peak[0], trlive.peak[0], TOL);
    }
    cpl_test_nonnull(tr.contrast);
    if(tr.contrast) {
        cpl_test_abs(tr.contrast[0], trlive.contrast[0], TOL);
    }

    cpl_polynomial_delete(tr.coefs);
    cpl_free(tr.xpos);
    cpl_free(tr.ypos);
    cpl_free(tr.fwhm);
    cpl_free(tr.peak);
    cpl_free(tr.contrast);

    /* Add a dead fibre */
    trlive.specnum = 3;
    trlive.live = 0;

    /* Do a series of tests where we feed in the table with each
     * column missing to set off the individual error handlers for
     * the column writes. */
    ncols = sizeof(dead_colnames) / sizeof(dead_colnames[0]);

    for(icol = 0; icol < ncols; icol++) {
        colname = dead_colnames[icol];
        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_trwrite1(test_tbl, trlive, 3);
        cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    /* Finally do it properly */
    code = qmost_trwrite1(table, trlive, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* and read it back */
    code = qmost_trread1(table, 3, &tr);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_eq(tr.specnum, trlive.specnum);
    cpl_test_eq(tr.live, trlive.live);
    cpl_test_eq(tr.yst, 0);
    cpl_test_eq(tr.yfn, 0);
    cpl_test_abs(tr.yref, 0, TOL);
    cpl_test_abs(tr.trrms, 0, TOL);
    cpl_test_eq(tr.nord, 0);
    cpl_test_null(tr.coefs);
    cpl_test_eq(tr.npos, 0);
    cpl_test_null(tr.xpos);
    cpl_test_null(tr.ypos);
    cpl_test_null(tr.fwhm);
    cpl_test_null(tr.peak);
    cpl_test_null(tr.contrast);

    /* Clean up */
    cpl_polynomial_delete(trlive.coefs);
    cpl_free(trlive.xpos);
    cpl_free(trlive.ypos);
    cpl_free(trlive.fwhm);
    cpl_free(trlive.peak);
    cpl_free(trlive.contrast);

    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_tracexpos
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_tracexpos(void)
{
    qmost_traceinfo tr;
    double value;
    cpl_size degree;

    memset(&tr, 0, sizeof(tr));

    /* Not live */
    value = qmost_tracexpos(tr, 512.0);
    cpl_test_abs(value, 0.0, TOL);

    /* Live but polynomial null */
    tr.live = 1;

    value = qmost_tracexpos(tr, 512.0);
    cpl_test_error(CPL_ERROR_NULL_INPUT);

    /* Now set up a simple polynomial */
    tr.yref = 1024;
    tr.coefs = cpl_polynomial_new(1);
    degree = 0;
    cpl_polynomial_set_coeff(tr.coefs, &degree, 128);
    degree = 1;
    cpl_polynomial_set_coeff(tr.coefs, &degree, 64);

    value = qmost_tracexpos(tr, 512.0);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 96.0, TOL);

    /* Clean up */
    cpl_polynomial_delete(tr.coefs);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_trclose
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_trclose(void)
{
    qmost_traceinfo *tr = NULL;
    int itr, ntr = 2;

    /* NULL argument */
    qmost_trclose(0, &tr);

    /* Empty argument, should free "tr" and null it */
    tr = cpl_calloc(ntr, sizeof(qmost_traceinfo));
    qmost_trclose(ntr, &tr);
    cpl_test_null(tr);

    /* Populated argument, should free internal references as well */
    tr = cpl_calloc(ntr, sizeof(qmost_traceinfo));
    for(itr = 0; itr < ntr; itr++) {
        tr[itr].coefs = cpl_polynomial_new(1);
        tr[itr].xpos = cpl_malloc(2 * sizeof(double));
        tr[itr].ypos = cpl_malloc(2 * sizeof(double));
        tr[itr].fwhm = cpl_malloc(2 * sizeof(float));
        tr[itr].peak = cpl_malloc(2 * sizeof(float));
        tr[itr].contrast = cpl_malloc(2 * sizeof(float));
    }

    qmost_trclose(2, &tr);
    cpl_test_null(tr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_trace_qc
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_trace_qc(void)
{
    cpl_table *table = NULL;
    cpl_propertylist *hdr = NULL;
    cpl_propertylist *qclist = NULL;
    cpl_table *fibinfo_tbl = NULL;
    cpl_error_code code;

    qmost_traceinfo tr;
    cpl_size zero;
    double value;

    int ny = 1024;
    int xcen = 256;
    double xfwhm = 3;
    double xpeak = 1000;
    int isnull;

    int ifib, nfib;

    /* Make empty table */
    code = qmost_trcreate(&table, 7, 10);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    hdr = cpl_propertylist_new();

    /* NULL inputs */
    code = qmost_trace_qc(NULL, hdr, NULL, NULL, qclist,
                          NULL, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_trace_qc(table, NULL, NULL, NULL, qclist,
                          NULL, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_trace_qc(table, hdr, NULL, NULL, NULL,
                          NULL, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty table and header */
    qclist = cpl_propertylist_new();

    code = qmost_trace_qc(table, hdr, NULL, NULL, qclist,
                          NULL, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* Add fibres, first one is bad but others populated */
    memset(&tr, 0, sizeof(tr));
    tr.live = 1;
    tr.yst = 1;
    tr.yfn = ny;
    tr.yref = ny/2;
    tr.trrms = 6;
    tr.nord = 0;
    tr.coefs = cpl_polynomial_new(1);
    zero = 0;
    cpl_polynomial_set_coeff(tr.coefs, &zero, xcen+1);  /* deliberately off */
    tr.npos = 1;
    tr.xpos = cpl_malloc(sizeof(double));
    tr.xpos[0] = xcen;
    tr.ypos = cpl_malloc(sizeof(double));
    tr.ypos[0] = xcen;
    tr.fwhm = cpl_malloc(sizeof(float));
    tr.fwhm[0] = xfwhm;
    tr.peak = cpl_malloc(sizeof(float));
    tr.peak[0] = xpeak;
    tr.contrast = cpl_malloc(sizeof(float));
    tr.contrast[0] = 1.0;

    nfib = 10 + QMOST_NFIB_PER_SLITLET;

    for(ifib = 1; ifib < nfib; ifib++) {
        tr.specnum = ifib+1;

        code = qmost_trwrite1(table, tr, tr.specnum);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
    }

    cpl_polynomial_delete(tr.coefs);
    cpl_free(tr.xpos);
    cpl_free(tr.ypos);
    cpl_free(tr.fwhm);
    cpl_free(tr.peak);
    cpl_free(tr.contrast);

    cpl_propertylist_update_int(hdr, "MINYST", 1);
    cpl_propertylist_update_int(hdr, "MAXYFN", ny);

    /* Populated, should generate some QC */
    qclist = cpl_propertylist_new();

    code = qmost_trace_qc(table, hdr, NULL, NULL, qclist,
                          NULL, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    value = cpl_propertylist_get_double(qclist, "ESO QC TRACE RMS MED");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 6, DBL_EPSILON);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* With reference, should generate offsets */
    qclist = cpl_propertylist_new();

    code = qmost_trace_qc(table, hdr, table, hdr, qclist,
                          NULL, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    value = cpl_propertylist_get_double(qclist, "ESO QC TRACE OFFSET MED");
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test_abs(value, 0, 1.0e-3);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    /* Also with FIBINFO, first wrong size */
    qclist = cpl_propertylist_new();
    fibinfo_tbl = cpl_table_new(0);
    cpl_table_new_column(fibinfo_tbl, "FIB_USE", CPL_TYPE_INT);

    code = qmost_trace_qc(table, hdr, table, hdr, qclist,
                          fibinfo_tbl, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;
    
    /* Now right size */
    fibinfo_tbl = cpl_table_new(nfib);
    cpl_table_new_column(fibinfo_tbl, "FIB_USE", CPL_TYPE_INT);

    code = qmost_trace_qc(table, hdr, table, hdr, qclist,
                          fibinfo_tbl, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_table_get(fibinfo_tbl, "TRACE_FWHM_MED_R", 0, &isnull);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test(isnull);

    value = cpl_table_get(fibinfo_tbl, "TRACE_FWHM_MED_R", 1, &isnull);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test(!isnull);
    cpl_test_abs(value, xfwhm, 0.5);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;

    /* All simucals */
    qclist = cpl_propertylist_new();
    fibinfo_tbl = cpl_table_new(nfib);
    cpl_table_new_column(fibinfo_tbl, "FIB_USE", CPL_TYPE_INT);
    for(ifib = 0; ifib < nfib; ifib++) {
        cpl_table_set_int(fibinfo_tbl, "FIB_USE", ifib, 0);
    }

    code = qmost_trace_qc(table, hdr, table, hdr, qclist,
                          fibinfo_tbl, QMOST_ARM_RED);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_table_get(fibinfo_tbl, "TRACE_FWHM_MED_R", 0, &isnull);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test(isnull);

    value = cpl_table_get(fibinfo_tbl, "TRACE_FWHM_MED_R", 1, &isnull);
    cpl_test_error(CPL_ERROR_NONE);
    cpl_test(!isnull);
    cpl_test_abs(value, xfwhm, 0.5);

    cpl_propertylist_delete(qclist);
    qclist = NULL;

    cpl_table_delete(fibinfo_tbl);
    fibinfo_tbl = NULL;

    /* Clean up */
    cpl_table_delete(table);
    cpl_propertylist_delete(hdr);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_traceinfo module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_trcreate();
    test_qmost_tropen();
    test_qmost_trchk();
    test_qmost_trread1();
    test_qmost_trinsert_dead();
    test_qmost_trrenumber();
    test_qmost_trwrite1();
    test_qmost_tracexpos();
    test_qmost_trclose();
    test_qmost_trace_qc();

    return cpl_test_end(0);
}

/**@}*/
