/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_dfs.h"
#include "qmost_stats.h"
#include "qmost_utils.h"
#include "qmost_lininfo.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_lininfo  qmost_lininfo
 *
 * Linearity correction table input/output.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_lininfo.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Output table definition */

#define AMP_COL     "amplifier"
#define XMIN_COL    "xmin"
#define XMAX_COL    "xmax"
#define YMIN_COL    "ymin"
#define YMAX_COL    "ymax"
#define NORD_COL    "nord"
#define COEFS_COL   "coefs"
#define RMS_COL     "rms"
#define LIN1K_COL   "lin1k"
#define LIN5K_COL   "lin5k"
#define LIN10K_COL  "lin10k"
#define LIN20K_COL  "lin20k"
#define LIN30K_COL  "lin30k"
#define LIN40K_COL  "lin40k"
#define LIN50K_COL  "lin50k"
#define NFILES_COL  "nfiles"
#define NGOOD_COL   "ngood"
#define MED_COL     "adu_med"
#define EXPTIME_COL "exptime"
#define MONCORR_COL "mon_corr"
#define LINMED_COL  "lin_med"
#define FFLAG_COL   "fit_flag"

static int lin_ncol = 22;

static char *lin_ttype[] = {
    AMP_COL, XMIN_COL, XMAX_COL,
    YMIN_COL, YMAX_COL, NORD_COL,
    COEFS_COL, RMS_COL, LIN1K_COL,
    LIN5K_COL, LIN10K_COL, LIN20K_COL,
    LIN30K_COL, LIN40K_COL, LIN50K_COL,
    NFILES_COL, NGOOD_COL, MED_COL,
    EXPTIME_COL, MONCORR_COL, LINMED_COL,
    FFLAG_COL
};

static char *lin_tunit[] = {
    "", "pix", "pix",
    "pix", "pix", "",
    "ADU", "ADU", "percent",
    "percent", "percent", "percent",
    "percent", "percent", "percent",
    "", "", "ADU",
    "s", "", "ADU",
    ""
};

static cpl_type lin_cpltype[] = {
    CPL_TYPE_INT, CPL_TYPE_INT, CPL_TYPE_INT,
    CPL_TYPE_INT, CPL_TYPE_INT, CPL_TYPE_INT,
    CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE,
    CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE,
    CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE,
    CPL_TYPE_INT, CPL_TYPE_INT, CPL_TYPE_DOUBLE,
    CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE, CPL_TYPE_DOUBLE,
    CPL_TYPE_INT
};

/*----------------------------------------------------------------------------*/
/**
 * @brief   Create a non-linearity correction table.
 *
 * @param   tptr       (Returned) The created cpl_table.
 * @param   nord       (Given)    The order of the polynomial to be
 *                                used for the linearity fit, for
 *                                sizing the table column containing
 *                                the coefficients.
 * @param   nfiles     (Given)    The number of flats included in the
 *                                linearity solution, to be used for
 *                                sizing the table columns containing
 *                                the individual flat measurements.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      The required output pointer tptr
 *                                    was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   nord or nlinesmax are negative.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_lincrtab (
    cpl_table **tptr,
    int nfiles,
    int nord)
{
    int icol;

    int lin_array[] = { 0, 0, 0,
                        0, 0, 0,
                        1, 0, 0,
                        0, 0, 0,
                        0, 0, 0,
                        0, 0, 1,
                        1, 1, 1,
                        1 };

    cpl_size lin_depth[] = { 1, 1, 1,
                             1, 1, 1,
                             nord+1, 1, 1,
                             1, 1, 1,
                             1, 1, 1,
                             0, 0, nfiles,
                             nfiles, nfiles, nfiles,
                             nfiles };

    /* Check inputs and output pointer */
    cpl_ensure_code(tptr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nord >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(nfiles >= 0, CPL_ERROR_ILLEGAL_INPUT);

    /* Create the table */
    *tptr = cpl_table_new(0);
    if(!(*tptr)) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create table");
    }

    for(icol = 0; icol < lin_ncol; icol++) {
        if(lin_array[icol]) {
            if(cpl_table_new_column_array(*tptr,
                                          lin_ttype[icol],
                                          lin_cpltype[icol],
                                          lin_depth[icol]) != CPL_ERROR_NONE) {
                cpl_table_delete(*tptr);
                *tptr = NULL;

                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create array column "
                                             "%d = %s with length %lld",
                                             icol+1,
                                             lin_ttype[icol],
                                             lin_depth[icol]);
            }
        }
        else {
            if(cpl_table_new_column(*tptr,
                                    lin_ttype[icol],
                                    lin_cpltype[icol]) != CPL_ERROR_NONE) {
                cpl_table_delete(*tptr);
                *tptr = NULL;

                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create column "
                                             "%d = %s",
                                             icol+1, lin_ttype[icol]);
            }
        }

        if(cpl_table_set_column_unit(*tptr,
                                     lin_ttype[icol],
                                     lin_tunit[icol]) != CPL_ERROR_NONE) {
            cpl_table_delete(*tptr);
            *tptr = NULL;

            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set units for column "
                                         "%d = %s to %s",
                                         icol+1, lin_ttype[icol],
                                         lin_tunit[icol]);
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Initialise a non-linearity information structure.
 *
 * @param   nfiles     (Given)    The number of flat files in the
 *                                non-linearity fit.
 * @param   nord       (Given)    The degree of the non-linearity fit
 *                                polynomial.
 * @param   lin        (Modified) The lininfo structure to initialise.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      The required output pointer lin
 *                                    was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   nord or nlinesmax are negative.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_lincrinfo (
    int nfiles,
    int nord,
    qmost_lininfo *lin)
{
    /* Check inputs and output pointer */
    cpl_ensure_code(nord >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(nfiles >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(lin != NULL, CPL_ERROR_NULL_INPUT);

    /* Initialise */
    memset(lin, 0, sizeof(qmost_lininfo));

    lin->nord = nord;
    lin->coefs = cpl_polynomial_new(1);
    lin->nfiles = nfiles;
    lin->med = cpl_calloc(nfiles,sizeof(double));
    lin->exptime = cpl_calloc(nfiles,sizeof(double));
    lin->moncorr = cpl_calloc(nfiles,sizeof(double));
    lin->linmed = cpl_calloc(nfiles,sizeof(double));
    lin->fit_flag = cpl_calloc(nfiles,sizeof(unsigned char));

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Free a non-linearity information structure.
 *
 * @param   lin        (Modified) The lininfo structure to free.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_lindelinfo (
    qmost_lininfo *lin)
{
    /* Trap null pointer */
    if(lin == NULL)
        return;

    /* Free workspace */
    if(lin->coefs != NULL) {
        cpl_polynomial_delete(lin->coefs);
        lin->coefs = NULL;
    }

    if(lin->med != NULL) {
        cpl_free(lin->med);
        lin->med = NULL;
    }

    if(lin->exptime != NULL) {
        cpl_free(lin->exptime);
        lin->exptime = NULL;
    }

    if(lin->moncorr != NULL) {
        cpl_free(lin->moncorr);
        lin->moncorr = NULL;
    }

    if(lin->linmed != NULL) {
        cpl_free(lin->linmed);
        lin->linmed = NULL;
    }

    if(lin->fit_flag != NULL) {
        cpl_free(lin->fit_flag);
        lin->fit_flag = NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Check an open lininfo table to see if it conforms to the
 *          standard.
 * 
 * @param   tptr       (Given)    cpl_table to check.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the table doesn't conform to
 *                                    the standard for lininfo
 *                                    tables.
 * @retval  CPL_ERROR_NULL_INPUT      A required input pointer was
 *                                    NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_linchk (
    cpl_table *tptr)
{
    int i, ncols;
    cpl_array *colnames;
    const char *colname;

    /* Check required input */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);

    /* Does it have the right number of columns and the right column names */

    ncols = cpl_table_get_ncol(tptr);
    if(ncols < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of columns");
    }
    else if(ncols != lin_ncol) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "input file has %d columns. "
                                     "Should have %d",
                                     ncols, lin_ncol);
    }

    colnames = cpl_table_get_column_names(tptr);
    if(!colnames) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get column names");
    }

    for (i = 0; i < lin_ncol; i++) {
        colname = cpl_array_get_string(colnames, i);
        if(colname == NULL) {
            cpl_array_delete(colnames);
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get column %d name",
                                         i+1);
        }

        if (strcmp(colname,lin_ttype[i]) != 0) {
            cpl_array_delete(colnames);
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ILLEGAL_INPUT,
                                         "input file column %d name "
                                         "not standard",
                                         i+1);
        }
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Write out a non-linearity solution.
 *
 * Writes out the non-linearity solution from the given lininfo
 * structures to a cpl_table, and populates QC FITS headers.  All
 * amplifiers of a single chip should be passed in at the same time.
 *
 * @param   tptr       (Modified) The cpl_table to update.
 * @param   hdr        (Modified) The FITS header to update.
 * @param   lins       (Given)    An array of lininfo structures
 *                                containing the solution for each
 *                                amplifier to be written out.
 * @param   namps      (Given)    The number of amplifiers in the
 *                                array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the number of polynomial
 *                                         coefficients exceeds the
 *                                         length of the array column.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If there were no amps or one of
 *                                    the table columns was not found.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the number of files
 *                                        doesn't match between the
 *                                        amplifiers.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      One of the required input or
 *                                    output pointers was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If a column data type of an
 *                                    integer column isn't correct.
 *
 * @par Output QC Parameters:
 *   - <b>DETFLAT MAX</b> (ADU): The maximum level of the flat fields
 *     that were combined to form the linearity table.
 *   - <b>DETFLAT MAXFILE</b>: The index (numbering from 1) of the
 *     flat with the highest count level in the linearity sequence.
 *   - <b>DETFLAT MIN</b> (ADU): The minimum level of the flat fields
 *     that were combined to form the linearity table.
 *   - <b>DETFLAT MINFILE</b>: The index (numbering from 1) of the
 *     flat with the lowest count level in the linearity sequence.
 *   - <b>LIN NL1K</b> (percent): The estimated non-linearity at 1000
 *     ADU averaged over all of the amplifiers.
 *   - <b>LIN NL5K</b> (percent): The estimated non-linearity at 5000
 *     ADU averaged over all of the amplifiers.
 *   - <b>LIN NL10K</b> (percent): The estimated non-linearity at
 *     10000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL20K</b> (percent): The estimated non-linearity at
 *     20000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL30K</b> (percent): The estimated non-linearity at
 *     30000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL40K</b> (percent): The estimated non-linearity at
 *     40000 ADU averaged over all of the amplifiers.
 *   - <b>LIN NL50K</b> (percent): The estimated non-linearity at
 *     50000 ADU averaged over all of the amplifiers.
 *   - <b>LIN RMS MAX</b> (ADU): The RMS of the linearity fit for the
 *     amplifier with the worst linearity fit.
 *   - <b>LIN RMS MAXAMP</b>: The amplifier with the worst linearity
 *     fit.
 *   - <b>LIN RMS MEAN</b> (ADU): The RMS of the linearity fit
 *     averaged over all of the amplifiers.
 *   - <b>LIN RMS MIN</b> (ADU): The RMS of the linearity fit for the
 *     amplifier with the best linearity fit.
 *   - <b>LIN RMS MINAMP</b>: The amplifier with the best linearity
 *     fit.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_linwrite (
    cpl_table *tptr,
    cpl_propertylist *hdr,
    qmost_lininfo *lins,
    int namps)
{
    cpl_size nrows;
    int iamp, ifile, nfiles;
    qmost_lininfo *lin;

    struct {
        char *column;
        char *qc;
        double level;
        double sum;
    } lin_levels[] = {
        { LIN1K_COL,  "ESO QC LIN NL1K",  1000.0,  0.0 },
        { LIN5K_COL,  "ESO QC LIN NL5K",  5000.0,  0.0 },
        { LIN10K_COL, "ESO QC LIN NL10K", 10000.0, 0.0 },
        { LIN20K_COL, "ESO QC LIN NL20K", 20000.0, 0.0 },
        { LIN30K_COL, "ESO QC LIN NL30K", 30000.0, 0.0 },
        { LIN40K_COL, "ESO QC LIN NL40K", 40000.0, 0.0 },
        { LIN50K_COL, "ESO QC LIN NL50K", 50000.0, 0.0 },
    };
    int ilin_level, nlin_levels;
    double polyval, value;

    double mean_rms = 0;
    double min_rms = 0;
    int imin_rms = -1;
    double max_rms = 0;
    int imax_rms = -1;

    double *mean_byfile = NULL;
    double min_cts = 0;
    int imin_cts = -1;
    double max_cts = 0;
    int imax_cts = -1;

    /* Check pointers */
    cpl_ensure_code(tptr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(hdr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(lins != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(namps > 0, CPL_ERROR_DATA_NOT_FOUND);

#undef TIDY
#define TIDY                                    \
    if(mean_byfile != NULL) {                   \
        cpl_free(mean_byfile);                  \
        mean_byfile = NULL;                     \
    }

    /* Ensure table has correct length */
    nrows = cpl_table_get_nrow(tptr);
    if(nrows < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of rows");
    }

    if(namps > nrows) {
        if(qmost_cpl_table_extend_blank(tptr, namps) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not extend table to %d rows",
                                         namps);
        }
    }

    /* Write table and accumulate statistics for QC */
    nlin_levels = sizeof(lin_levels) / sizeof(lin_levels[0]);
    nfiles = lins[0].nfiles;

    mean_byfile = cpl_calloc(nfiles, sizeof(double));

    for(iamp = 0; iamp < namps; iamp++) {
        lin = lins + iamp;

        if(lin->nfiles != nfiles) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_INCOMPATIBLE_INPUT,
                                         "number of files doesn't match "
                                         "for amp %d: %d != %d",
                                         iamp+1, lin->nfiles, nfiles);
        }

        /* Table output */
        if(cpl_table_set_int(tptr,
                             AMP_COL,
                             iamp,
                             lin->amp) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         AMP_COL, iamp+1);
        }

        if(cpl_table_set_int(tptr,
                             XMIN_COL,
                             iamp,
                             lin->xmin) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         XMIN_COL, iamp+1);
        }

        if(cpl_table_set_int(tptr,
                             XMAX_COL,
                             iamp,
                             lin->xmax) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         XMAX_COL, iamp+1);
        }

        if(cpl_table_set_int(tptr,
                             YMIN_COL,
                             iamp,
                             lin->ymin) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         YMIN_COL, iamp+1);
        }

        if(cpl_table_set_int(tptr,
                             YMAX_COL,
                             iamp,
                             lin->ymax) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         YMAX_COL, iamp+1);
        }

        if(cpl_table_set_int(tptr,
                             NORD_COL,
                             iamp,
                             lin->nord) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NORD_COL, iamp+1);
        }

        if(qmost_cpl_table_set_polynomial(tptr,
                                          COEFS_COL,
                                          iamp,
                                          lin->coefs,
                                          lin->nord) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         COEFS_COL, iamp+1);
        }

        if(cpl_table_set(tptr,
                         RMS_COL,
                         iamp,
                         lin->rms) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         RMS_COL, iamp+1);
        }

        /* Calculate the enumerated list of percentage non-linearity at
         * different count levels */
        for(ilin_level = 0; ilin_level < nlin_levels; ilin_level++) {
            polyval = cpl_polynomial_eval_1d(lin->coefs,
                                             lin_levels[ilin_level].level,
                                             NULL);

            value = 100.0 * fabs(polyval / lin_levels[ilin_level].level - 1.0);

            if(cpl_table_set(tptr,
                             lin_levels[ilin_level].column,
                             iamp,
                             value) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to write %s for row %d",
                                             lin_levels[ilin_level].column,
                                             iamp+1);
            }

            lin_levels[ilin_level].sum += value;
        }

        if(cpl_table_set_int(tptr,
                             NFILES_COL,
                             iamp,
                             lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NFILES_COL, iamp+1);
        }

        if(cpl_table_set_int(tptr,
                             NGOOD_COL,
                             iamp,
                             lin->ngood) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NGOOD_COL, iamp+1);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            MED_COL,
                                            iamp,
                                            lin->med,
                                            lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         MED_COL, iamp+1);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            EXPTIME_COL,
                                            iamp,
                                            lin->exptime,
                                            lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         EXPTIME_COL, iamp+1);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            MONCORR_COL,
                                            iamp,
                                            lin->moncorr,
                                            lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         MONCORR_COL, iamp+1);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            LINMED_COL,
                                            iamp,
                                            lin->linmed,
                                            lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         LINMED_COL, iamp+1);
        }

        if(qmost_cpl_table_set_byte_array(tptr,
                                          FFLAG_COL,
                                          iamp,
                                          lin->fit_flag,
                                          lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         FFLAG_COL, iamp+1);
        }

        /* Other stats for QC */
        mean_rms += lin->rms;

        if(imin_rms < 0 || lin->rms < min_rms) {
            min_rms = lin->rms;
            imin_rms = iamp;
        }
        if(imax_rms < 0 || lin->rms > max_rms) {
            max_rms = lin->rms;
            imax_rms = iamp;
        }

        for(ifile = 0; ifile < nfiles; ifile++) {
            mean_byfile[ifile] += lin->med[ifile];
        }
    }

    mean_rms /= namps;

    for(ifile = 0; ifile < nfiles; ifile++) {
        mean_byfile[ifile] /= namps;

        if(imin_cts < 0 || mean_byfile[ifile] < min_cts) {
            min_cts = mean_byfile[ifile];
            imin_cts = ifile;
        }
        if(imax_cts < 0 || mean_byfile[ifile] > max_cts) {
            max_cts = mean_byfile[ifile];
            imax_cts = ifile;
        }
    }

    /* Write QC headers */
    cpl_propertylist_update_double(hdr, "ESO QC DETFLAT MIN", min_cts);
    cpl_propertylist_set_comment(hdr, "ESO QC DETFLAT MIN",
                                 "[ADU] Minimum flat counts in linearity fit");

    cpl_propertylist_update_int(hdr, "ESO QC DETFLAT MINFILE", imin_cts+1);
    cpl_propertylist_set_comment(hdr, "ESO QC DETFLAT MINFILE",
                                 "File with minimum flat counts");

    cpl_propertylist_update_double(hdr, "ESO QC DETFLAT MAX", max_cts);
    cpl_propertylist_set_comment(hdr, "ESO QC DETFLAT MAX",
                                 "[ADU] Maximum flat counts in linearity fit");

    cpl_propertylist_update_int(hdr, "ESO QC DETFLAT MAXFILE", imax_cts+1);
    cpl_propertylist_set_comment(hdr, "ESO QC DETFLAT MAXFILE",
                                 "File with maximum flat counts");

    for(ilin_level = 0; ilin_level < nlin_levels; ilin_level++) {
        value = lin_levels[ilin_level].sum / namps;

        cpl_propertylist_update_double(hdr,
                                       lin_levels[ilin_level].qc,
                                       value);
        cpl_propertylist_set_comment(hdr,
                                     lin_levels[ilin_level].qc,
                                     "[percent] Mean non-linearity at "
                                     "graduated ADU levels");
    }

    cpl_propertylist_update_double(hdr, "ESO QC LIN RMS MEAN", mean_rms);
    cpl_propertylist_set_comment(hdr, "ESO QC LIN RMS MEAN",
                                 "[ADU] Mean RMS of linearity fit");

    cpl_propertylist_update_double(hdr, "ESO QC LIN RMS MIN", min_rms);
    cpl_propertylist_set_comment(hdr, "ESO QC LIN RMS MIN",
                                 "[ADU] Minimum RMS of linearity fit");

    cpl_propertylist_update_int(hdr, "ESO QC LIN RMS MINAMP", imin_rms+1);
    cpl_propertylist_set_comment(hdr, "ESO QC LIN RMS MINAMP",
                                 "Amplifier with minimum RMS");

    cpl_propertylist_update_double(hdr, "ESO QC LIN RMS MAX", max_rms);
    cpl_propertylist_set_comment(hdr, "ESO QC LIN RMS MAX",
                                 "[ADU] Maximum RMS of linearity fit");

    cpl_propertylist_update_int(hdr, "ESO QC LIN RMS MAXAMP", imax_rms+1);
    cpl_propertylist_set_comment(hdr, "ESO QC LIN RMS MAXAMP",
                                 "Amplifier with maximum RMS");

    cpl_free(mean_byfile);
    mean_byfile = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read a non-linearity solution.
 *
 * Reads the non-linearity solution from a cpl_table into an array of
 * lininfo structures for each amplifier.
 *
 * @param   tptr       (Given)    The cpl_table to read.
 * @param   lins       (Returned) An array of lininfo structures
 *                                containing the solution for each
 *                                amplifier.
 * @param   namps      (Returned) The number of amplifiers in the
 *                                array.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the polynomial degree
 *                                         indicated in the table
 *                                         is longer than the length
 *                                         of the polynomial 
 *                                         coefficient array.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the requested table row
 *                                    doesn't exist in the table, one
 *                                    of the table columns is missing,
 *                                    or if there were no amps.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      One of the required input or
 *                                    output pointers was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If a column data type of an
 *                                    integer column isn't correct.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_linread (
    cpl_table *tptr,
    qmost_lininfo **lins,
    int *namps)
{
    int iamp, iampt;

    cpl_errorstate prestate;
    int nfiles, nord;

    qmost_lininfo *lin = NULL;

    /* Check pointers */
    cpl_ensure_code(tptr != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(lins != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(namps != NULL, CPL_ERROR_NULL_INPUT);

    /* Initialize these */
    *lins = NULL;
    *namps = 0;

#undef TIDY
#define TIDY                                            \
    if(*lins != NULL) {                                 \
        for(iampt = 0; iampt < *namps; iampt++) {       \
            qmost_lindelinfo((*lins) + iampt);          \
        }                                               \
        cpl_free(*lins);                                \
        *lins = NULL;                                   \
    }

    /* Get number of rows in table = number of amps */
    *namps = cpl_table_get_nrow(tptr);
    if(*namps < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "linearity table is empty");
    }

    /* Allocate output */
    *lins = cpl_calloc(*namps, sizeof(qmost_lininfo));

    /* Read table */
    for(iamp = 0; iamp < *namps; iamp++) {
        /* Read what we need to allocate the lininfo structure first */
        prestate = cpl_errorstate_get();

        nfiles = cpl_table_get_int(tptr, NFILES_COL, iamp, NULL);
        nord = cpl_table_get_int(tptr, NORD_COL, iamp, NULL);
        if(!cpl_errorstate_is_equal(prestate) &&
           cpl_error_get_code() != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s/%s columns "
                                         "for amp %d",
                                         NFILES_COL,
                                         NORD_COL,
                                         iamp+1);
        }

        /* Allocate structure */
        lin = (*lins) + iamp;
        if(qmost_lincrinfo(nfiles, nord, lin) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not create lininfo for "
                                         "amp %d",
                                         iamp+1);
        }

        /* Now read the rest */
        prestate = cpl_errorstate_get();

        lin->amp = cpl_table_get_int(tptr, AMP_COL, iamp, NULL);
        lin->xmin = cpl_table_get_int(tptr, XMIN_COL, iamp, NULL);
        lin->xmax = cpl_table_get_int(tptr, XMAX_COL, iamp, NULL);
        lin->ymin = cpl_table_get_int(tptr, YMIN_COL, iamp, NULL);
        lin->ymax = cpl_table_get_int(tptr, YMAX_COL, iamp, NULL);
        if(!cpl_errorstate_is_equal(prestate) &&
           cpl_error_get_code() != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read lininfo columns "
                                         "for amp %d",
                                         iamp+1);
        }

        if(lin->coefs != NULL) {
            cpl_polynomial_delete(lin->coefs);
            lin->coefs = NULL;
        }

        lin->coefs = qmost_cpl_table_get_polynomial(tptr, COEFS_COL, iamp,
                                                    lin->nord);
        if(lin->coefs == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for amp %d",
                                         COEFS_COL,
                                         iamp+1);
        }

        prestate = cpl_errorstate_get();

        lin->rms = cpl_table_get(tptr, RMS_COL, iamp, NULL);
        lin->ngood = cpl_table_get_int(tptr, NGOOD_COL, iamp, NULL);
        if(!cpl_errorstate_is_equal(prestate) &&
           cpl_error_get_code() != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s/%s columns "
                                         "for amp %d",
                                         RMS_COL,
                                         NGOOD_COL,
                                         iamp+1);
        }

        if(qmost_cpl_table_get_double_array(tptr,
                                            MED_COL,
                                            iamp,
                                            lin->med,
                                            lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for amp %d",
                                         MED_COL,
                                         iamp+1);
        }

        if(qmost_cpl_table_get_double_array(tptr,
                                            EXPTIME_COL,
                                            iamp,
                                            lin->exptime,
                                            lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for amp %d",
                                         EXPTIME_COL,
                                         iamp+1);
        }

        if(qmost_cpl_table_get_double_array(tptr,
                                            MONCORR_COL,
                                            iamp,
                                            lin->moncorr,
                                            lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for amp %d",
                                         MONCORR_COL,
                                         iamp+1);
        }

        if(qmost_cpl_table_get_double_array(tptr,
                                            LINMED_COL,
                                            iamp,
                                            lin->linmed,
                                            lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for amp %d",
                                         LINMED_COL,
                                         iamp+1);
        }

        if(qmost_cpl_table_get_byte_array(tptr,
                                          FFLAG_COL,
                                          iamp,
                                          lin->fit_flag,
                                          lin->nfiles) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column "
                                         "for amp %d",
                                         FFLAG_COL,
                                         iamp+1);
        }
    }

    return CPL_ERROR_NONE;
}

/**@}*/
