/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_pfits.h"
#include "qmost_stats.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_traceinfo  qmost_traceinfo
 *
 * Trace table utilities
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/* Output table definition */

#define SPECNUM_COL      "specnum"
#define FIBLIVE_COL      "fiblive"
#define YST_COL          "yst"
#define YFN_COL          "yfn"
#define TRYREF_COL       "tryref"
#define TRRMS_COL        "trrms"
#define TRORDER_COL      "trorder"
#define TRCOEFS_COL      "trcoefs"
#define NPOS_COL         "npos"
#define XPOS_COL         "xpos"
#define YPOS_COL         "ypos"
#define FWHM_COL         "fwhm"
#define PEAK_COL         "peak"
#define CONTRAST_COL     "contrast"

#define SPECNUM_TYPE    CPL_TYPE_INT
#define FIBLIVE_TYPE    CPL_TYPE_INT
#define YST_TYPE        CPL_TYPE_INT
#define YFN_TYPE        CPL_TYPE_INT
#define TRYREF_TYPE     CPL_TYPE_DOUBLE
#define TRRMS_TYPE      CPL_TYPE_DOUBLE
#define TRORDER_TYPE    CPL_TYPE_INT
#define TRCOEFS_TYPE    CPL_TYPE_DOUBLE
#define NPOS_TYPE       CPL_TYPE_INT
#define XPOS_TYPE       CPL_TYPE_DOUBLE
#define YPOS_TYPE       CPL_TYPE_DOUBLE
#define FWHM_TYPE       CPL_TYPE_FLOAT
#define PEAK_TYPE       CPL_TYPE_FLOAT
#define CONTRAST_TYPE   CPL_TYPE_FLOAT

static int tr_ncol =  14;

/* This gives the number of mandatory columns (all except FWHM and
 * peak, which were later additions) for trchk */
static int tr_mincol =  11;

static char *tr_ttype[] = { SPECNUM_COL, FIBLIVE_COL,
                            YST_COL, YFN_COL,
                            TRYREF_COL, TRRMS_COL,
                            TRORDER_COL, TRCOEFS_COL,
                            NPOS_COL, XPOS_COL, YPOS_COL,
                            FWHM_COL, PEAK_COL, CONTRAST_COL };

static char *tr_tunit[] = {"","",
                           "pix","pix",
                           "pix","pix",
                           "","",
                           "","pix","pix",
                           "pix","ADU",""};

static cpl_type tr_cpltype[] = { SPECNUM_TYPE, FIBLIVE_TYPE,
                                 YST_TYPE, YFN_TYPE,
                                 TRYREF_TYPE, TRRMS_TYPE,
                                 TRORDER_TYPE, TRCOEFS_TYPE,
                                 NPOS_TYPE, XPOS_TYPE, YPOS_TYPE,
                                 FWHM_TYPE, PEAK_TYPE, CONTRAST_TYPE };

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Create the trace information table.
 *
 * @param   tptr       (Returned) The created cpl_table.
 * @param   nord       (Given)    The order of the polynomial to be
 *                                fit to the trace, for sizing the
 *                                table column containing the
 *                                coefficients.
 * @param   nblocks    (Given)    Maximum number of trace samples, for
 *                                sizing the table columns containing
 *                                the x,y samples.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT      The required output pointer tptr
 *                                    was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   nord or nblocks are negative.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_trcreate (
    cpl_table **tptr,
    int nord,
    int nblocks)
{
    int icol;

    int tr_array[] = { 0, 0,
                       0, 0,
                       0, 0,
                       0, 1,
                       0, 1, 1,
                       1, 1, 1 };

    cpl_size tr_depth[] = { 1, 1,
                            1, 1,
                            1, 1,
                            1, nord+1,
                            1, nblocks, nblocks,
                            nblocks, nblocks, nblocks };

    /* Check inputs and output pointer */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(nord >= 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(nblocks >= 0, CPL_ERROR_ILLEGAL_INPUT);

    /* Create the table */
    *tptr = cpl_table_new(0);
    if(!(*tptr)) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not create table");
    }

    for(icol = 0; icol < tr_ncol; icol++) {
        if(tr_array[icol]) {
            if(cpl_table_new_column_array(*tptr,
                                          tr_ttype[icol],
                                          tr_cpltype[icol],
                                          tr_depth[icol]) != CPL_ERROR_NONE) {
                cpl_table_delete(*tptr);
                *tptr = NULL;

                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create array column "
                                             "%d = %s with length %lld",
                                             icol+1,
                                             tr_ttype[icol],
                                             tr_depth[icol]);
            }
        }
        else {
            if(cpl_table_new_column(*tptr,
                                    tr_ttype[icol],
                                    tr_cpltype[icol]) != CPL_ERROR_NONE) {
                cpl_table_delete(*tptr);
                *tptr = NULL;

                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create column "
                                             "%d = %s",
                                             icol+1, tr_ttype[icol]);
            }
        }

        if(cpl_table_set_column_unit(*tptr,
                                     tr_ttype[icol],
                                     tr_tunit[icol]) != CPL_ERROR_NONE) {
            cpl_table_delete(*tptr);
            *tptr = NULL;

            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not set units for column "
                                         "%d = %s to %s",
                                         icol+1, tr_ttype[icol],
                                         tr_tunit[icol]);
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Open and read a single trace information extension
 *
 * @param   tptr       (Given)    cpl_table to read from
 * @param   thdr       (Given)    FITS header of input table.
 * @param   ntr        (Returned) The number of traces in the current
 *                                extension.
 * @param   tr         (Returned) The array of trace information
 *                                structures.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE   If the length of an array
 *                                          specified in the table is
 *                                          larger than the actual
 *                                          length of the array
 *                                          column.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If one of the required input
 *                                    FITS header keywords was not
 *                                    found.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the table is not a true
 *                                    trace table.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      A required input or output
 *                                    pointer was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If one of the required input
 *                                    FITS header keyword values had
 *                                    an incorrect data type, or if a
 *                                    column data type of an integer
 *                                    column isn't correct.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MAXYFN</b>
 *   - <b>ESO DRS MINYST</b>
 *   - <b>MAXYFN</b>
 *   - <b>MINYST</b>
 *
 * @note    The trace information array must be deallocated with
 *          qmost_trclose.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_tropen (
    cpl_table *tptr,
    cpl_propertylist *thdr,
    int *ntr,
    qmost_traceinfo **tr)
{
    int i,minyst,maxyfn;
    int nrows;
    
    /* Check required inputs and outputs */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(thdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ntr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(tr, CPL_ERROR_NULL_INPUT);

    /* Initialise some things */

    *ntr = 0;
    *tr = NULL;

#undef TIDY
#define TIDY                                    \
    if(*tr) {                                   \
        qmost_trclose(*ntr, tr);                \
        *ntr = 0;                               \
        *tr = NULL;                             \
    }

    /* Check to make sure this conforms to the trace file design */
    
    if (qmost_trchk(tptr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "trace table conformity check failed");
    }

    /* OK, we're happy now. Get the space for the output and clear it
       to ensure the internal pointers are set to NULL. */
 
    nrows = cpl_table_get_nrow(tptr);
    if(nrows < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of rows");
    }

    *ntr = nrows;
    *tr = cpl_calloc(nrows, sizeof(qmost_traceinfo));

    /* Read header keywords */
    minyst = -1;

    if(qmost_pfits_get_minyst(thdr, &minyst) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read minimum yst");
    }

    if(minyst < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "ESO DRS MINYST or MINYST missing "
                                     "or invalid");
    }

    maxyfn = -1;

    if(qmost_pfits_get_maxyfn(thdr, &maxyfn) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read maximum yfn");
    }

    if(maxyfn < 0) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "ESO DRS MAXYFN or MAXYFN missing "
                                     "or invalid");
    }

    /* Loop through the table and read all the info. If this is not a live
       fibre, then fill most of the stuff with NULL values */

    for (i = 0; i < *ntr; i++) {
        if (qmost_trread1(tptr,i+1,(*tr+i)) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read row %d", i+1);
        }
        (*tr+i)->minyst = minyst;
        (*tr+i)->maxyfn = maxyfn;
    }
    
    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Check the current extension to see if it is a true trace
 *          information table.
 *
 * Check a table to see if it conforms to the specification of a
 * proper trace information file.
 *
 * @param   tptr       (Given)    cpl_table to check.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ILLEGAL_INPUT   If the table is not a true trace
 *                                    table.
 * @retval  CPL_ERROR_NULL_INPUT      The required input pointer tptr
 *                                    was NULL.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_trchk(
    cpl_table *tptr)
{
    int i, ncols;
    cpl_array *colnames;
    const char *colname;

    /* Check required input */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);

    /* Does it have the right number of columns and the right column names */

    ncols = cpl_table_get_ncol(tptr);
    if(ncols < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of columns");
    }
    else if(ncols < tr_mincol) {
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "input file has %d columns. "
                                     "Should have at least %d",
                                     ncols, tr_mincol);
    }

    colnames = cpl_table_get_column_names(tptr);
    if(!colnames) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get column names");
    }

    for (i = 0; i < tr_mincol; i++) {
        colname = cpl_array_get_string(colnames, i);
        if(colname == NULL) {
            cpl_array_delete(colnames);
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not get column %d name",
                                         i+1);
        }

        if (strcmp(colname,tr_ttype[i]) != 0) {
            cpl_array_delete(colnames);
            return cpl_error_set_message(cpl_func,
                                         CPL_ERROR_ILLEGAL_INPUT,
                                         "input file column %d name "
                                         "not standard",
                                         i+1);
        }
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Read the trace for a single fibre.
 * 
 * Read the trace for a single fibre and store it away in a traceinfo
 * structure.
 *
 * @param   tptr       (Given)    cpl_table to read from.
 * @param   row        (Given)    The row of the table to be read,
 *                                numbering from 1 (FITS convention).
 * @param   tr         (Returned) The output traceinfo structure for
 *                                the trace.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE   The row number was outside
 *                                          the table, or if the
 *                                          length of an array
 *                                          specified in the table is
 *                                          larger than the actual
 *                                          length of the array
 *                                          column.
 * @retval  CPL_ERROR_DATA_NOT_FOUND        If a table column was
 *                                          missing.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      A required input or output
 *                                    pointer was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If a column data type of an
 *                                    integer column isn't correct.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_trread1(
    cpl_table *tptr,
    int row,
    qmost_traceinfo *tr)
{
    cpl_errorstate prestate;
    int ipos;

    /* Check required inputs and outputs */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(row > 0, CPL_ERROR_ACCESS_OUT_OF_RANGE);
    cpl_ensure_code(tr, CPL_ERROR_NULL_INPUT);

    memset(tr, 0, sizeof(qmost_traceinfo));

#undef TIDY
#define TIDY                                    \
    if(tr->coefs != NULL) {                     \
        cpl_polynomial_delete(tr->coefs);       \
        tr->coefs = NULL;                       \
    }                                           \
    if(tr->xpos != NULL) {                      \
        cpl_free(tr->xpos);                     \
        tr->xpos = NULL;                        \
    }                                           \
    if(tr->ypos != NULL) {                      \
        cpl_free(tr->ypos);                     \
        tr->ypos = NULL;                        \
    }                                           \
    if(tr->fwhm != NULL) {                      \
        cpl_free(tr->fwhm);                     \
        tr->fwhm = NULL;                        \
    }                                           \
    if(tr->peak != NULL) {                      \
        cpl_free(tr->peak);                     \
        tr->peak = NULL;                        \
    }                                           \
    if(tr->contrast != NULL) {                  \
        cpl_free(tr->contrast);                 \
        tr->contrast = NULL;                    \
    }

    /* Read the spectrum number and see if this is a live fibre */
    prestate = cpl_errorstate_get();

    tr->specnum = cpl_table_get_int(tptr, SPECNUM_COL, row-1, NULL);
    tr->live = cpl_table_get_int(tptr, FIBLIVE_COL, row-1, NULL);
    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read %s/%s columns for row %d",
                                     SPECNUM_COL, FIBLIVE_COL, row);
    }

    /* If this fibre is dead, then zero off pretty much everything */
    
    if (! tr->live) {
        tr->yst = 0;
        tr->yfn = 0;
        tr->yref = 0.0;
        tr->trrms = 0.0;
        tr->nord = 0;
        tr->coefs = NULL;
        tr->npos = 0;
        tr->xpos = NULL;
        tr->ypos = NULL;
        tr->fwhm = NULL;
        tr->peak = NULL;
        tr->contrast = NULL;
        return CPL_ERROR_NONE;
    }

    /* OK, this is a live fibre. So let's get the rest of the info */
    tr->yst = cpl_table_get_int(tptr, YST_COL, row-1, NULL);
    tr->yfn = cpl_table_get_int(tptr, YFN_COL, row-1, NULL);
    tr->yref = cpl_table_get(tptr, TRYREF_COL, row-1, NULL);
    tr->trrms = cpl_table_get(tptr, TRRMS_COL, row-1, NULL);
    tr->nord = cpl_table_get_int(tptr, TRORDER_COL, row-1, NULL);
    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read trace info columns "
                                     "for row %d",
                                     row);
    }

    tr->coefs = qmost_cpl_table_get_polynomial(tptr,
                                               TRCOEFS_COL,
                                               row-1,
                                               tr->nord);
    if(tr->coefs == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read %s column for row %d",
                                     TRCOEFS_COL, row);
    }

    tr->npos = cpl_table_get_int(tptr, NPOS_COL, row-1, NULL);
    if(!cpl_errorstate_is_equal(prestate) &&
       cpl_error_get_code() != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to read %s column for row %d",
                                     NPOS_COL, row);
    }

    if(tr->npos > 0) {
        tr->xpos = cpl_malloc((tr->npos)*sizeof(double));
        tr->ypos = cpl_malloc((tr->npos)*sizeof(double));
        tr->fwhm = cpl_malloc((tr->npos)*sizeof(float));
        tr->peak = cpl_malloc((tr->npos)*sizeof(float));
        tr->contrast = cpl_malloc((tr->npos)*sizeof(float));
        
        if(qmost_cpl_table_get_double_array(tptr,
                                            XPOS_COL,
                                            row-1,
                                            tr->xpos,
                                            tr->npos) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column for row %d",
                                         XPOS_COL, row);
        }
        
        if(qmost_cpl_table_get_double_array(tptr,
                                            YPOS_COL,
                                            row-1,
                                            tr->ypos,
                                            tr->npos) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to read %s column for row %d",
                                         YPOS_COL, row);
        }

        /* The older trace format didn't have FWHM, so we need to
         * check if it exists and substitute dummy values if not */
        if(cpl_table_has_column(tptr, FWHM_COL)) {
            if(qmost_cpl_table_get_float_array(tptr,
                                               FWHM_COL,
                                               row-1,
                                               tr->fwhm,
                                               tr->npos) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             FWHM_COL, row);
            }
        }
        else {
            for(ipos = 0; ipos < tr->npos; ipos++) {
                tr->fwhm[ipos] = NAN;
            }
        }

        /* The older trace format didn't have peak, so we need to
         * check if it exists and substitute dummy values if not */
        if(cpl_table_has_column(tptr, PEAK_COL)) {
            if(qmost_cpl_table_get_float_array(tptr,
                                               PEAK_COL,
                                               row-1,
                                               tr->peak,
                                               tr->npos) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             PEAK_COL, row);
            }
        }
        else {
            for(ipos = 0; ipos < tr->npos; ipos++) {
                tr->peak[ipos] = NAN;
            }
        }

        /* The older trace format didn't have contrast, so we need to
         * check if it exists and substitute dummy values if not */
        if(cpl_table_has_column(tptr, CONTRAST_COL)) {
            if(qmost_cpl_table_get_float_array(tptr,
                                               CONTRAST_COL,
                                               row-1,
                                               tr->contrast,
                                               tr->npos) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d",
                                             CONTRAST_COL, row);
            }
        }
        else {
            for(ipos = 0; ipos < tr->npos; ipos++) {
                tr->contrast[ipos] = NAN;
            }
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Create a dead fibre entry in a trace info table in a given
 *          row.
 *
 * @param   tptr       (Modified) cpl_table to update.
 * @param   row        (Given)    The row of the table to update,
 *                                numbering from 1 (FITS convention).
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE   The row number was less
 *                                          than 1.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If a table column was missing.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      The required input pointer tptr
 *                                    was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If a column data type of an
 *                                    integer column isn't correct.
 *
 * @note    A row is inserted into the table and the LIVE column is
 *          flagged.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_trinsert_dead(
    cpl_table *tptr,
    int row)
{
    cpl_size nrows;

    /* Check required inputs */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(row > 0, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* Check table size.  If we've been asked to insert a row after
       the end of the existing table, we need to extend it.
       Otherwise, insert the row. */
    nrows = cpl_table_get_nrow(tptr);
    if(nrows < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of rows");
    }

    if(row > nrows) {
        /* Extend the table */
        if(qmost_cpl_table_extend_blank(tptr, row) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not extend table to %d rows",
                                         row);
        }
    }
    else {
        /* Insert row */
        if(qmost_cpl_table_insert_blank_window(tptr,
                                               row-1,
                                               1) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not insert row %d", row);
        }
    }

    /* Set FIB_LIVE */
    if(cpl_table_set_int(tptr,
                         FIBLIVE_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to set %s for row %d",
                                     FIBLIVE_COL, row);
    }

    /* Zero other columns */
    if(cpl_table_set_int(tptr,
                         YST_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to set %s for row %d",
                                     YST_COL, row);
    }
    
    if(cpl_table_set_int(tptr,
                         YFN_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to set %s for row %d",
                                     YFN_COL, row);
    }
    
    if(cpl_table_set(tptr,
                     TRYREF_COL,
                     row-1,
                     0) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to set %s for row %d",
                                     TRYREF_COL, row);
    }
    
    if(cpl_table_set(tptr,
                     TRRMS_COL,
                     row-1,
                     0) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to set %s for row %d",
                                     TRRMS_COL, row);
    }
    
    if(cpl_table_set_int(tptr,
                         TRORDER_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to set %s for row %d",
                                     TRORDER_COL, row);
    }
    
    if(cpl_table_set_int(tptr,
                         NPOS_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to set %s for row %d",
                                     NPOS_COL, row);
    }

    /* Now renumber them */
    return qmost_trrenumber(tptr);
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Renumber the entries in a trace table.
 * 
 * Renumber the entries in a trace table. This may be necessary if new
 * rows are added.
 *
 * @param   tptr       (Modified) cpl_table to update.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If the specnum column doesn't
 *                                    exist.
 * @retval  CPL_ERROR_NULL_INPUT      The required input pointer tptr
 *                                    was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If the specnum column data type
 *                                    isn't correct.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_trrenumber(
    cpl_table *tptr)
{
    cpl_size i, nrows;

    /* Check required input */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);

    /* Renumber the trace in the current file */

    nrows = cpl_table_get_nrow(tptr);
    if(nrows < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of rows");
    }

    for(i = 0; i < nrows; i++) {
        if(cpl_table_set_int(tptr,
                             SPECNUM_COL,
                             i,
                             i+1) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to set %s for row %lld",
                                         SPECNUM_COL, i+1);
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Write a single row of a trace information table.
 *
 * @param   tptr       (Modified) cpl_table to update.
 * @param   tr         (Given)    The traceinfo structure to be
 *                                written to the file.
 * @param   row        (Given)    The row of the table to write,
 *                                numbering from 1 (FITS convention).
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE   The row number was less
 *                                          than 1 or the length of an
 *                                          array exceeds the length
 *                                          of the array column in the
 *                                          table.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If a table column was missing.
 * @retval  CPL_ERROR_INVALID_TYPE    If a column data type of a float
 *                                    or double column isn't correct.
 * @retval  CPL_ERROR_NULL_INPUT      The required input pointer tptr
 *                                    was NULL.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If a column data type of an
 *                                    integer column isn't correct.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_trwrite1(
    cpl_table *tptr,
    qmost_traceinfo tr,
    int row)
{
    cpl_size nrows;

    /* Check required inputs */
    cpl_ensure_code(tptr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(row > 0, CPL_ERROR_ACCESS_OUT_OF_RANGE);

    /* If the row is off the end of the table, extend the table */
    
    nrows = cpl_table_get_nrow(tptr);
    if(nrows < 0) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not get number of rows");
    }

    if(row > nrows) {
        if(qmost_cpl_table_extend_blank(tptr, row) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not extend table to %d rows",
                                         row);
        }
    }

    /* Do the spectrum number and live */

    if(cpl_table_set_int(tptr,
                         SPECNUM_COL,
                         row-1,
                         tr.specnum) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to write %s for row %d",
                                     SPECNUM_COL, row);
    }

    if(cpl_table_set_int(tptr,
                         FIBLIVE_COL,
                         row-1,
                         tr.live) != CPL_ERROR_NONE) {
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to write %s for row %d",
                                     FIBLIVE_COL, row);
    }

    /* If this is a dead fibre, then write lots of NULL stuff to the
       rest of the columns */

    if (! tr.live) {
        if(cpl_table_set_int(tptr,
                             YST_COL,
                             row-1,
                             0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         YST_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             YFN_COL,
                             row-1,
                             0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         YFN_COL, row);
        }

        if(cpl_table_set(tptr,
                         TRYREF_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         TRYREF_COL, row);
        }

        if(cpl_table_set(tptr,
                         TRRMS_COL,
                         row-1,
                         0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         TRRMS_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             TRORDER_COL,
                             row-1,
                             0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         TRORDER_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             NPOS_COL,
                             row-1,
                             0) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NPOS_COL, row);
        }
     } else {
        if(cpl_table_set_int(tptr,
                             YST_COL,
                             row-1,
                             tr.yst) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         YST_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             YFN_COL,
                             row-1,
                             tr.yfn) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         YFN_COL, row);
        }

        if(cpl_table_set(tptr,
                         TRYREF_COL,
                         row-1,
                         tr.yref) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         TRYREF_COL, row);
        }

        if(cpl_table_set(tptr,
                         TRRMS_COL,
                         row-1,
                         tr.trrms) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         TRRMS_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             TRORDER_COL,
                             row-1,
                             tr.nord) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         TRORDER_COL, row);
        }

        if(qmost_cpl_table_set_polynomial(tptr,
                                          TRCOEFS_COL,
                                          row-1,
                                          tr.coefs,
                                          tr.nord) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         TRCOEFS_COL, row);
        }

        if(cpl_table_set_int(tptr,
                             NPOS_COL,
                             row-1,
                             tr.npos) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         NPOS_COL, row);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            XPOS_COL,
                                            row-1,
                                            tr.xpos,
                                            tr.npos) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         XPOS_COL, row);
        }

        if(qmost_cpl_table_set_double_array(tptr,
                                            YPOS_COL,
                                            row-1,
                                            tr.ypos,
                                            tr.npos) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         YPOS_COL, row);
        }

        if(qmost_cpl_table_set_float_array(tptr,
                                           FWHM_COL,
                                           row-1,
                                           tr.fwhm,
                                           tr.npos) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         FWHM_COL, row);
        }

        if(qmost_cpl_table_set_float_array(tptr,
                                           PEAK_COL,
                                           row-1,
                                           tr.peak,
                                           tr.npos) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         PEAK_COL, row);
        }

        if(qmost_cpl_table_set_float_array(tptr,
                                           CONTRAST_COL,
                                           row-1,
                                           tr.contrast,
                                           tr.npos) != CPL_ERROR_NONE) {
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to write %s for row %d",
                                         CONTRAST_COL, row);
        }
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Work out the spatial pixel position for a trace at a
 *          spectral pixel position.
 *
 * @param   tr         (Given)    The traceinfo structure.
 * @param   ypos       (Given)    The position along the spectral axis
 *                                of the trace.
 *
 * @return  double                The position along the spatial axis.
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

double qmost_tracexpos(
    qmost_traceinfo tr,
    double ypos)
{
    double y,x;

    if (! tr.live)
        return(0.0);
    y = (ypos - tr.yref)/tr.yref;
    x = cpl_polynomial_eval_1d(tr.coefs, y, NULL);
    return(x);
}  

/*----------------------------------------------------------------------------*/
/**
 * @brief   Close a traceinfo structure array and free the associated
 *          memory.
 *
 * @param   ntr        (Given)    The number of traceinfo structures
 *                                in the input array.
 * @param   tr         (Modified) The traceinfo structure array.
 *
 * @return  void
 *
 * @author  Jim Lewis, CASU
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

void qmost_trclose(
    int ntr,
    qmost_traceinfo **tr)
{
    int i;

    if(tr == NULL)
        return;

    for (i = 0; i < ntr; i++) {
        if((*tr)[i].coefs != NULL) {
            cpl_polynomial_delete((*tr)[i].coefs);
        }
        if((*tr)[i].xpos != NULL) {
            cpl_free((*tr)[i].xpos);
        }
        if((*tr)[i].ypos != NULL) {
            cpl_free((*tr)[i].ypos);
        }
        if((*tr)[i].fwhm != NULL) {
            cpl_free((*tr)[i].fwhm);
        }
        if((*tr)[i].peak != NULL) {
            cpl_free((*tr)[i].peak);
        }
        if((*tr)[i].contrast != NULL) {
            cpl_free((*tr)[i].contrast);
        }
    }
    cpl_free(*tr);
    *tr = NULL;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Populate QC headers using trace table.
 *
 * @param   trace_tbl      (Given)    The trace table.
 * @param   trace_hdr      (Given)    The corresponding FITS header.
 * @param   ref_trace_tbl  (Given)    An optional reference master
 *                                    trace table to compare the trace
 *                                    table with.  NULL if none.
 * @param   ref_trace_hdr  (Given)    The corresponding FITS header
 *                                    for the reference trace table.
 * @param   qclist         (Modified) The output FITS header to
 *                                    receive the QC parameters.
 * @param   fibinfo_tbl    (Modified) Output FIBINFO table to receive
 *                                    per-fibre QC, or NULL for none.
 * @param   arm            (Given)    One of the QMOST_ARM_* constants
 *                                    specifying which arm we're
 *                                    processing.  Only used if
 *                                    fibinfo_tbl != NULL and can be
 *                                    given as 0 otherwise.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE            If everything is OK.
 * @retval  CPL_ERROR_ACCESS_OUT_OF_RANGE  If the FIBINFO table was
 *                                         shorter than the trace
 *                                         table.
 * @retval  CPL_ERROR_DATA_NOT_FOUND  If one of the required FITS
 *                                    headers or trace table columns
 *                                    was not found.
 * @retval  CPL_ERROR_TYPE_MISMATCH   If an input FITS header keyword
 *                                    had an incorrect data type.
 *
 * @par Input FITS Header Information:
 *   - <b>ESO DRS MAXYFN</b>
 *   - <b>ESO DRS MINYST</b>
 *   - <b>MAXYFN</b>
 *   - <b>MINYST</b>
 *
 * @par Output QC Parameters:
 *   - <b>TRACE CONTRAST MAX</b>: The maximum of the fibre-averaged
 *     peak to trough contrast ratio over the traces.
 *   - <b>TRACE CONTRAST MAXSPC</b>: The fibre with the maximum trace
 *     contrast (numbering from 1).
 *   - <b>TRACE CONTRAST MEAN</b>: The mean of the fibre-averaged peak
 *     to trough contrast ratio over the traces.
 *   - <b>TRACE CONTRAST MIN</b>: The minimum of the fibre-averaged peak
 *     to trough contrast ratio over the traces.
 *   - <b>TRACE CONTRAST MINSPC</b>: The fibre with the minimum trace
 *     contrast (numbering from 1).
 *   - <b>TRACE CONTRAST RMS</b>: The RMS of the fibre-averaged peak
 *     to trough contrast ratio over the traces.
 *   - <b>TRACE FWHM MED</b> (pix): Average median FWHM of the spatial
 *     profile of the fibres.
 *   - <b>TRACE FWHM RMS</b> (pix): Average robustly-estimated RMS
 *     FWHM of the spatial profile of the fibres.
 *   - <b>TRACE GAP MEAN</b> (pix): The mean spacing between adjacent
 *     fibres in a slitlet, at the reference spectral coordinate.
 *   - <b>TRACE GAP RMS</b> (pix): The RMS of the spacing between
 *     adjacent fibres in a slitlet, at the reference spectral
 *     coordinate.
 *   - <b>TRACE GAP MIN</b> (pix): The minimum spacing between
 *     adjacent fibres in a slitlet, at the reference spectral
 *     coordinate.
 *   - <b>TRACE GAP MAX</b> (pix): The maximum spacing between
 *     adjacent fibres in a slitlet, at the reference spectral
 *     coordinate.
 *   - <b>SLITLET GAP MEAN</b> (pix): The mean spacing between
 *     slitlets, at the reference spectral coordinate.
 *   - <b>SLITLET GAP RMS</b> (pix): The RMS of the spacing between
 *     slitlets, at the reference spectral coordinate.
 *   - <b>SLITLET GAP MIN</b> (pix): The minimum spacing between
 *     slitlets, at the reference spectral coordinate.
 *   - <b>SLITLET GAP MAX</b> (pix): The maximum spacing between
 *     slitlets, at the reference spectral coordinate.
 *   - <b>TRACE NUM FIBRES</b>: The number of fibres detected and
 *     traced.
 *   - <b>TRACE OFFSET MAX</b> (pix): The maximum shift of the traces
 *     in the spatial direction relative to the provided master trace,
 *     measured at the reference spectral coordinate.
 *   - <b>TRACE OFFSET MAXSPC</b>: The fibre with the maximum trace
 *     offset.
 *   - <b>TRACE OFFSET MED</b> (pix): The median shift of the traces
 *     in the spatial direction relative to the provided reference
 *     master trace, measured at the reference spectral coordinate.
 *   - <b>TRACE OFFSET MIN</b> (pix): The minimum shift of the traces
 *     in the spatial direction relative to the provided master trace,
 *     measured at the reference spectral coordinate.
 *   - <b>TRACE OFFSET MINSPC</b>: The fibre with the minimum trace
 *     offset.
 *   - <b>TRACE OFFSET RMS</b> (pix): The robustly-estimated RMS of
 *     the shift of the traces in the spatial direction relative to
 *     the provided master trace, measured at the reference spectral
 *     coordinate.
 *   - <b>TRACE RMS MAX</b> (pix): The maximum RMS error in the trace
 *     fit over all fibres.
 *   - <b>TRACE RMS MAXSPC</b>: The fibre with the maximum trace RMS.
 *   - <b>TRACE RMS MED</b> (pix): The median of the RMS error in the
 *     trace fit computed over all fibres.
 *   - <b>TRACE RMS MIN</b> (pix): The minimum RMS error in the trace
 *     fit over all fibres.
 *   - <b>TRACE RMS MINSPC</b>: The fibre with the minimum trace RMS.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_trace_qc (
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    cpl_table *ref_trace_tbl,
    cpl_propertylist *ref_trace_hdr,
    cpl_propertylist *qclist,
    cpl_table *fibinfo_tbl,
    int arm)
{
    qmost_traceinfo *trlist = NULL, *tr;
    int itr, ntr = 0;

    qmost_traceinfo *trreflist = NULL, *trref;
    int ntrref = 0, ntrcomp;

    const char *arm_extname;
    char arm_ltr = 'U';
    char *colname_fwhm_med = NULL;
    char *colname_fwhm_rms = NULL;
    char *colname_offset = NULL;

    int ifib, fib_use, anynul, nsimua, nsimub, nsci, nslitlet;

    cpl_array *rmslist = NULL;
    cpl_array *dxfib = NULL;
    cpl_array *dxslit = NULL;
    cpl_array *dxlist = NULL;
    cpl_array *fwhmmedlist = NULL;
    cpl_array *fwhmsiglist = NULL;
    cpl_array *contrastmedlist = NULL;
    int nrms, ndxfib, ndxslit, ndx, nfwhm, ncontrast;

    float *res = NULL;
    int ipos, nres;
    float fwhm, contrast, sig;

    int last_itr;
    int slitlet, last_slitlet;
    double x, xref, last_xref;

    cpl_size zero;

    cpl_errorstate prestate;
    float avfwhm, avsig;
    double dx, med_dx, mad_dx;

    cpl_size minspc, maxspc;

    cpl_ensure_code(trace_tbl, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(trace_hdr, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist, CPL_ERROR_NULL_INPUT);

#undef TIDY
#define TIDY                                    \
    if(trlist != NULL) {                        \
        qmost_trclose(ntr, &trlist);            \
        trlist = NULL;                          \
    }                                           \
    if(trreflist != NULL) {                     \
        qmost_trclose(ntrref, &trreflist);      \
        trreflist = NULL;                       \
    }                                           \
    if(colname_fwhm_med != NULL) {              \
        cpl_free(colname_fwhm_med);             \
        colname_fwhm_med = NULL;                \
    }                                           \
    if(colname_fwhm_rms != NULL) {              \
        cpl_free(colname_fwhm_rms);             \
        colname_fwhm_rms = NULL;                \
    }                                           \
    if(colname_offset) {                        \
        cpl_free(colname_offset);               \
        colname_offset = NULL;                  \
    }                                           \
    if(rmslist) {                               \
        cpl_array_delete(rmslist);              \
        rmslist = NULL;                         \
    }                                           \
    if(dxfib) {                                 \
        cpl_array_delete(dxfib);                \
        dxfib = NULL;                           \
    }                                           \
    if(dxslit) {                                \
        cpl_array_delete(dxslit);               \
        dxslit = NULL;                          \
    }                                           \
    if(dxlist) {                                \
        cpl_array_delete(dxlist);               \
        dxlist = NULL;                          \
    }                                           \
    if(fwhmmedlist) {                           \
        cpl_array_delete(fwhmmedlist);          \
        fwhmmedlist = NULL;                     \
    }                                           \
    if(fwhmsiglist) {                           \
        cpl_array_delete(fwhmsiglist);          \
        fwhmsiglist = NULL;                     \
    }                                           \
    if(contrastmedlist) {                       \
        cpl_array_delete(contrastmedlist);      \
        contrastmedlist = NULL;                 \
    }                                           \
    if(res) {                                   \
        cpl_free(res);                          \
        res = NULL;                             \
    }

    /* Read trace table */
    if(qmost_tropen(trace_tbl, trace_hdr, &ntr, &trlist) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not read trace table");
    }

    /* Set up FIBINFO columns if given */
    if(fibinfo_tbl != NULL) {
        /* Get the appropriate letter for the arm */
        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not determine EXTNAME "
                                         "for arm %d", arm);
        }
    
        arm_ltr = arm_extname[0];

        colname_fwhm_med = cpl_sprintf("TRACE_FWHM_MED_%c", arm_ltr);
        if(colname_fwhm_med == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format column "
                                         "names for arm %d", arm);
        }

        if(!cpl_table_has_column(fibinfo_tbl, colname_fwhm_med)) {
            cpl_table_new_column(fibinfo_tbl,
                                 colname_fwhm_med,
                                 CPL_TYPE_FLOAT);
            cpl_table_set_column_unit(fibinfo_tbl,
                                      colname_fwhm_med,
                                      "pix");
        }

        colname_fwhm_rms = cpl_sprintf("TRACE_FWHM_RMS_%c", arm_ltr);
        if(colname_fwhm_rms == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format column "
                                         "names for arm %d", arm);
        }

        if(!cpl_table_has_column(fibinfo_tbl, colname_fwhm_rms)) {
            cpl_table_new_column(fibinfo_tbl,
                                 colname_fwhm_rms,
                                 CPL_TYPE_FLOAT);
            cpl_table_set_column_unit(fibinfo_tbl,
                                      colname_fwhm_rms,
                                      "pix");
        }
    }

    /* Number of science fibres, not less than zero.  If there's no
     * FIBINFO table, we assume they're all science fibres. */
    nsimua = 0;
    nsimub = 0;

    if(fibinfo_tbl != NULL) {
        /* Count simucals (FIB_USE = 0) at start */
        for(nsimua = 0; nsimua < ntr; nsimua++) {
            /* Get FIB_USE */
            fib_use = cpl_table_get_int(fibinfo_tbl,
                                        "FIB_USE",
                                        nsimua,
                                        &anynul);
            if(anynul < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d from FIBINFO",
                                             "FIB_USE",
                                             nsimua+1);
            }
            else if(anynul > 0) {  /* NULL */
                break;
            }
            else if(fib_use != 0) {
                break;
            }
        }

        if(nsimua > QMOST_NFIB_PER_SLITLET) {
            nsimua = QMOST_NFIB_PER_SLITLET;
        }

        /* Find index of last non-simucal fibre */
        nsimub = 0;

        for(ifib = ntr-1; ifib >= nsimua; ifib--) {
            /* Get FIB_USE */
            fib_use = cpl_table_get_int(fibinfo_tbl,
                                        "FIB_USE",
                                        ifib,
                                        &anynul);
            if(anynul < 0) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to read %s column "
                                             "for row %d from FIBINFO",
                                             "FIB_USE",
                                             ifib+1);
            }
            else if(anynul > 0) {  /* NULL */
                break;
            }
            else if(fib_use != 0) {
                break;
            }
            else {
                nsimub++;
            }
        }

        if(nsimub > QMOST_NFIB_PER_SLITLET) {
            nsimub = QMOST_NFIB_PER_SLITLET;
        }
    }

    nsci = qmost_max(0, ntr - (nsimua + nsimub));

    /* Number of slitlets */
    nslitlet = nsci / QMOST_NFIB_PER_SLITLET;
    if(nsci % QMOST_NFIB_PER_SLITLET) {
        cpl_msg_warning(cpl_func,
                        "unexpected number of fibres: %d", ntr);
        nslitlet++;
    }

    if(nsimua > 0) {
        nslitlet++;
    }
    if(nsimub > 0) {
        nslitlet++;
    }

    /* Allocate workspace for statistics */
    rmslist = cpl_array_new(ntr, CPL_TYPE_DOUBLE);
    if(rmslist == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to create rmslist length %d",
                                     ntr);
    }

    dxfib = cpl_array_new(ntr, CPL_TYPE_DOUBLE);
    if(dxfib == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to create dxfib length %d",
                                     ntr);
    }

    dxslit = cpl_array_new(nslitlet, CPL_TYPE_DOUBLE);
    if(dxslit == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to create dxfib length %d",
                                     ntr);
    }

    fwhmmedlist = cpl_array_new(ntr, CPL_TYPE_FLOAT);
    if(fwhmmedlist == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to create fwhmmedlist length %d",
                                     ntr);
    }

    fwhmsiglist = cpl_array_new(ntr, CPL_TYPE_FLOAT);
    if(fwhmsiglist == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to create fwhmsiglist length %d",
                                     ntr);
    }

    contrastmedlist = cpl_array_new(ntr, CPL_TYPE_FLOAT);
    if(contrastmedlist == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "failed to create contrast "
                                     "medlist length %d",
                                     ntr);
    }

    /* Accumulate statistics */
    last_itr = -1;
    last_slitlet = -1;
    last_xref = 0;

    nrms = 0;
    ndxfib = 0;
    ndxslit = 0;
    nfwhm = 0;
    ncontrast = 0;

    for(itr = 0; itr < ntr; itr++) {
        tr = trlist + itr;

        /* Slitlet number (numbering from 0) */
        if(nsimua > 0) {
            slitlet = (itr + QMOST_NFIB_PER_SLITLET -
                       nsimua) / QMOST_NFIB_PER_SLITLET;
        }
        else {
            slitlet = itr / QMOST_NFIB_PER_SLITLET;
        }

        /* Check if fibre is live */
        if(tr->live) {
            /* Add RMS to array for stats */
            if(cpl_array_set(rmslist,
                             nrms,
                             tr->trrms) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set rmslist[%d]",
                                             nrms);
            }

            nrms++;

            /* Get reference x coordinate = zeroth degree coefficient
               of trace polynomial. */
            zero = 0;
            xref = cpl_polynomial_get_coeff(tr->coefs, &zero);

            /* Only use adjacent traces */
            if(last_itr >= 0 && last_itr == itr-1) {
                if(last_slitlet == slitlet) {
                    if(cpl_array_set(dxfib,
                                     ndxfib,
                                     xref-last_xref) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "failed to set "
                                                     "dxfib[%d]",
                                                     ndxfib);
                    }

                    ndxfib++;
                }
                else {
                    if(cpl_array_set(dxslit,
                                     ndxslit,
                                     xref-last_xref) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "failed to set "
                                                     "dxslit[%d]",
                                                     ndxslit);
                    }

                    ndxslit++;
                }
            }

            /* Get the median of the non-NULL trace FWHMs */
            res = cpl_malloc(tr->npos * sizeof(float));

            nres = 0;

            for(ipos = 0; ipos < tr->npos; ipos++) {
                if(isfinite(tr->fwhm[ipos]) &&
                   tr->fwhm[ipos] > 0) {
                    res[nres] = tr->fwhm[ipos];
                    nres++;
                }
            }

            /* Get a median */
            if(nres > 0) {
                prestate = cpl_errorstate_get();

                if(qmost_medmad(res,NULL,nres,
                                &fwhm,&sig) != CPL_ERROR_NONE) {
                    cpl_errorstate_set(prestate);
                    fwhm = 0;
                    sig = 0;
                }

                sig *= CPL_MATH_STD_MAD;
                
                if(fibinfo_tbl != NULL) {
                    if(cpl_table_set(fibinfo_tbl,
                                     colname_fwhm_med,
                                     itr,
                                     fwhm) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "couldn't set FWHM_MED "
                                                     "in FIBINFO for "
                                                     "trace %d",
                                                     itr+1);
                    }
                    
                    if(cpl_table_set(fibinfo_tbl,
                                     colname_fwhm_rms,
                                     itr,
                                     sig) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "couldn't set FWHM_RMS "
                                                     "in FIBINFO for "
                                                     "trace %d",
                                                     itr+1);
                    }
                }
                
                /* Accumulate for mean */
                if(cpl_array_set(fwhmmedlist,
                                 nfwhm,
                                 fwhm) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "failed to set "
                                                 "fwhmmedlist[%d]",
                                                 nfwhm);
                }
                
                if(cpl_array_set(fwhmsiglist,
                                 nfwhm,
                                 sig) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "failed to set "
                                                 "fwhmsiglist[%d]",
                                                 nfwhm);
                }
                
                nfwhm++;
            }

            /* Get the median of the non-NULL trace contrast */
            nres = 0;

            for(ipos = 0; ipos < tr->npos; ipos++) {
                if(isfinite(tr->contrast[ipos])) {
                    res[nres] = tr->contrast[ipos];
                    nres++;
                }
            }

            /* Get a median */
            if(nres > 0) {
                prestate = cpl_errorstate_get();

                if(qmost_med(res,NULL,nres,
                             &contrast) != CPL_ERROR_NONE) {
                    cpl_errorstate_set(prestate);
                    contrast = 0;
                    sig = 0;
                }

                /* Accumulate for mean */
                if(cpl_array_set(contrastmedlist,
                                 ncontrast,
                                 contrast) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "failed to set "
                                                 "contrastmedlist[%d]",
                                                 ncontrast);
                }
                
                ncontrast++;
            }

            cpl_free(res);
            res = NULL;

            last_itr = itr;
            last_slitlet = slitlet;
            last_xref = xref;
        }
    }

    if(nrms > 0) {
        if(cpl_array_set_size(rmslist, nrms) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to set rmslist size to %d",
                                         nrms);
        }

        /* QC parameters */
        cpl_propertylist_update_double(qclist,
                                       "ESO QC TRACE RMS MED",
                                       cpl_array_get_median(rmslist));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE RMS MED",
                                     "[pix] Median trace fit RMS");

        cpl_propertylist_update_double(qclist,
                                       "ESO QC TRACE RMS MIN",
                                       cpl_array_get_min(rmslist));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE RMS MIN",
                                     "[pix] Minimum trace fit RMS");

        if(cpl_array_get_minpos(rmslist, &minspc) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to find minimum of rmslist");
        }

        cpl_propertylist_update_int(qclist,
                                    "ESO QC TRACE RMS MINSPC",
                                    (int) (minspc + 1));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE RMS MINSPC",
                                     "Fibre with minimum trace RMS");
        
        cpl_propertylist_update_double(qclist,
                                       "ESO QC TRACE RMS MAX",
                                       cpl_array_get_max(rmslist));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE RMS MAX",
                                     "[pix] Maximum trace fit RMS");

        if(cpl_array_get_maxpos(rmslist, &maxspc) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to find maximum of rmslist");
        }

        cpl_propertylist_update_int(qclist,
                                    "ESO QC TRACE RMS MAXSPC",
                                    (int) (maxspc + 1));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE RMS MAXSPC",
                                     "Fibre with maximum trace RMS");
    }

    if(ndxfib > 0) {
        if(cpl_array_set_size(dxfib, ndxfib) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to set dxfib size to %d",
                                         ndxfib);
        }

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE GAP MEAN",
                                      cpl_array_get_mean(dxfib));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE GAP MEAN",
                                     "[pix] Mean fibre spacing within slitlet");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE GAP RMS",
                                      cpl_array_get_stdev(dxfib));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE GAP RMS",
                                     "[pix] RMS fibre spacing within slitlet");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE GAP MIN",
                                      cpl_array_get_min(dxfib));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE GAP MIN",
                                     "[pix] Minimum fibre spacing within "
                                     "slitlet");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE GAP MAX",
                                      cpl_array_get_max(dxfib));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE GAP MAX",
                                     "[pix] Maximum fibre spacing within "
                                     "slitlet");
    }

    if(ndxslit > 0) {
        if(cpl_array_set_size(dxslit, ndxslit) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to set dxslit size to %d",
                                         ndxslit);
        }

        cpl_propertylist_update_float(qclist,
                                      "ESO QC SLITLET GAP MEAN",
                                      cpl_array_get_mean(dxslit));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC SLITLET GAP MEAN",
                                     "[pix] Mean slitlet spacing");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC SLITLET GAP RMS",
                                      cpl_array_get_stdev(dxslit));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC SLITLET GAP RMS",
                                     "[pix] RMS slitlet spacing");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC SLITLET GAP MIN",
                                      cpl_array_get_min(dxslit));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC SLITLET GAP MIN",
                                     "[pix] Minimum slitlet spacing");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC SLITLET GAP MAX",
                                      cpl_array_get_max(dxslit));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC SLITLET GAP MAX",
                                     "[pix] Maximum slitlet spacing");
    }

    cpl_propertylist_update_int(qclist,
                                "ESO QC TRACE NUM FIBRES",
                                nrms);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC TRACE NUM FIBRES",
                                 "Number of fibres detected and traced");

    cpl_array_delete(rmslist);
    rmslist = NULL;

    cpl_array_delete(dxfib);
    dxfib = NULL;

    cpl_array_delete(dxslit);
    dxslit = NULL;

    if(nfwhm > 0) {
        if(cpl_array_set_size(fwhmmedlist, nfwhm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to set fwhmmedlist "
                                         "size to %d",
                                         nfwhm);
        }

        if(cpl_array_set_size(fwhmsiglist, nfwhm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to set fwhmsiglist "
                                         "size to %d",
                                         nfwhm);
        }

        avfwhm = cpl_array_get_mean(fwhmmedlist);
        avsig = cpl_array_get_mean(fwhmsiglist);

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE FWHM MED",
                                      avfwhm);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE FWHM MED",
                                     "[pix] Median FWHM of spatial profiles");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE FWHM RMS",
                                      avsig);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE FWHM RMS",
                                     "[pix] RMS FWHM of spatial profiles");
    }

    cpl_array_delete(fwhmmedlist);
    cpl_array_delete(fwhmsiglist);

    if(ncontrast > 0) {
        if(cpl_array_set_size(contrastmedlist, ncontrast) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to set contrastmedlist "
                                         "size to %d",
                                         ncontrast);
        }

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE CONTRAST MEAN",
                                      cpl_array_get_mean(contrastmedlist));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE CONTRAST MEAN",
                                     "Average trace peak to trough contrast");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE CONTRAST RMS",
                                      cpl_array_get_stdev(contrastmedlist));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE CONTRAST RMS",
                                     "RMS trace peak to trough contrast");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE CONTRAST MIN",
                                      cpl_array_get_min(contrastmedlist));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE CONTRAST MIN",
                                     "Minimum trace peak to trough contrast");

        if(cpl_array_get_minpos(contrastmedlist, &minspc) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to find minimum of "
                                         "contrastmedlist");
        }

        cpl_propertylist_update_int(qclist,
                                    "ESO QC TRACE CONTRAST MINSPC",
                                    (int) (minspc + 1));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE CONTRAST MINSPC",
                                     "Fibre with minimum trace contrast");

        cpl_propertylist_update_float(qclist,
                                      "ESO QC TRACE CONTRAST MAX",
                                      cpl_array_get_max(contrastmedlist));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE CONTRAST MAX",
                                     "Maximum trace peak to trough contrast");

        if(cpl_array_get_maxpos(contrastmedlist, &maxspc) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to find maximum of "
                                         "contrastmedlist");
        }

        cpl_propertylist_update_int(qclist,
                                    "ESO QC TRACE CONTRAST MAXSPC",
                                    (int) (maxspc + 1));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC TRACE CONTRAST MAXSPC",
                                     "Fibre with maximum trace contrast");
    }

    cpl_array_delete(contrastmedlist);

    if(colname_fwhm_med != NULL) {
        cpl_free(colname_fwhm_med);
        colname_fwhm_med = NULL;
    }

    if(colname_fwhm_rms != NULL) {
        cpl_free(colname_fwhm_rms);
        colname_fwhm_rms = NULL;
    }

    /* If reference was given, compare the two by computing statistics
       of trace offset. */
    if(ref_trace_tbl != NULL) {
        /* Read reference table */
        if(qmost_tropen(ref_trace_tbl, ref_trace_hdr,
                        &ntrref, &trreflist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read reference "
                                         "trace table");
        }

        ntrcomp = qmost_min(ntr, ntrref);

        /* Set up FIBINFO column if given */
        if(fibinfo_tbl != NULL) {
            colname_offset = cpl_sprintf("TRACE_OFFSET_%c", arm_ltr);
            if(colname_offset == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format column "
                                             "names for arm %d", arm);
            }
            
            if(!cpl_table_has_column(fibinfo_tbl, colname_offset)) {
                cpl_table_new_column(fibinfo_tbl,
                                     colname_offset,
                                     CPL_TYPE_DOUBLE);
                cpl_table_set_column_unit(fibinfo_tbl,
                                          colname_offset,
                                          "pix");
            }
        }

        dxlist = cpl_array_new(ntrcomp, CPL_TYPE_DOUBLE);
        if(dxlist == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "failed to create dxlist length %d",
                                         ntrcomp);

        }
        
        ndx = 0;
        
        for(itr = 0; itr < ntrcomp; itr++) {
            tr = trlist + itr;
            trref = trreflist + itr;
            
            /* Check if fibre is live */
            if(tr->live && trref->live) {
                /* Get reference x coordinate, making sure we use the
                   same reference y coordinate for both. */

                x = qmost_tracexpos(*tr, trref->yref);
                xref = qmost_tracexpos(*trref, trref->yref);
                
                dx = x-xref;

                if(fibinfo_tbl != NULL) {
                    if(cpl_table_set(fibinfo_tbl,
                                     colname_offset,
                                     itr,
                                     dx) != CPL_ERROR_NONE) {
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "couldn't set "
                                                     "TRACE_OFFSET in "
                                                     "FIBINFO for trace %d",
                                                     itr+1);
                    }
                }

                if(cpl_array_set(dxlist,
                                 ndx,
                                 dx) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "failed to set dxlist[%d]",
                                                 ndx);
                }

                ndx++;
            }
        }

        if(ndx > 0) {
            if(cpl_array_set_size(dxlist, ndx) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to set dxlist size to %d",
                                             ndx);
            }

            /* QC parameters */
            if(qmost_cpl_array_get_med_mad(dxlist,
                                           &med_dx,
                                           &mad_dx) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to compute medmad of "
                                             "dxlist");
            }

            cpl_propertylist_update_double(qclist,
                                           "ESO QC TRACE OFFSET MED",
                                           med_dx);
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC TRACE OFFSET MED",
                                         "[pix] Median trace offset");
            
            cpl_propertylist_update_double(qclist,
                                           "ESO QC TRACE OFFSET RMS",
                                           CPL_MATH_STD_MAD * mad_dx);
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC TRACE OFFSET RMS",
                                         "[pix] RMS of trace offset");
            
            cpl_propertylist_update_double(qclist,
                                           "ESO QC TRACE OFFSET MIN",
                                           cpl_array_get_min(dxlist));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC TRACE OFFSET MIN",
                                         "[pix] Minimum trace offset");
            
            if(cpl_array_get_minpos(dxlist, &minspc) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to find minimum of "
                                             "dxlist");
            }
            
            cpl_propertylist_update_int(qclist,
                                        "ESO QC TRACE OFFSET MINSPC",
                                        (int) (minspc + 1));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC TRACE OFFSET MINSPC",
                                         "Fibre with minimum trace offset");

            cpl_propertylist_update_double(qclist,
                                           "ESO QC TRACE OFFSET MAX",
                                           cpl_array_get_max(dxlist));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC TRACE OFFSET MAX",
                                         "[pix] Maximum trace offset");

            if(cpl_array_get_maxpos(dxlist, &maxspc) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "failed to find maximum of "
                                             "dxlist");
            }
            
            cpl_propertylist_update_int(qclist,
                                        "ESO QC TRACE OFFSET MAXSPC",
                                        (int) (maxspc + 1));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC TRACE OFFSET MAXSPC",
                                         "Fibre with maximum trace offset");

            if(colname_offset) {
                cpl_free(colname_offset);
                colname_offset = NULL;
            }
        }

        /* Clean up */
        qmost_trclose(ntrref, &trreflist);
        trreflist = NULL;

        cpl_array_delete(dxlist);
        dxlist = NULL;
    }

    qmost_trclose(ntr, &trlist);
    trlist = NULL;

    return CPL_ERROR_NONE;
}

/**@}*/

/*

$Log$
Revision 1.6  2019/02/25 10:49:52  jrl
New memory allocation scheme

Revision 1.5  2017/03/14 11:32:56  jim
added yst and yfn to trace object

Revision 1.4  2016/10/26 11:20:40  jim
Added docs

Revision 1.3  2016/07/21 08:35:23  jim
Fixed problem in _trrenumber

Revision 1.2  2016/07/06 11:05:21  jim
Modified to use new trace file format


*/
