/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_UTILS_H
#define QMOST_UTILS_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/**
 * @ingroup qmost_utils
 *
 * @brief   Round floating-point argument to nearest integer.
 */

#undef qmost_nint
#define qmost_nint(v) ((int) rint((v)))

/**
 * @ingroup qmost_utils
 *
 * @brief   Macro to swap two variables x and y via temporary variable t.
 */

#undef qmost_swap
#define qmost_swap(x, y, t) (t) = (x); (x) = (y); (y) = (t)

/**
 * @ingroup qmost_utils
 *
 * @brief   Macro to return the smaller of A and B.
 */

#undef qmost_min
#define qmost_min(A,B) ((A) < (B) ? (A) : (B))

/**
 * @ingroup qmost_utils
 *
 * @brief   Macro to return the larger of A and B.
 */

#undef qmost_max
#define qmost_max(A,B) ((A) > (B) ? (A) : (B))

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

const char * qmost_get_license(void) ;

cpl_error_code qmost_check_and_set_groups(
        cpl_frameset *frameset);

cpl_error_code qmost_cpl_array_get_med_mad (
    cpl_array *arr,
    double *med,
    double *mad);

cpl_error_code qmost_cpl_propertylist_get_double (
    const cpl_propertylist *plist,
    const char *name,
    double *value);

cpl_error_code qmost_cpl_propertylist_get_float (
    const cpl_propertylist *plist,
    const char *name,
    float *value);

cpl_error_code qmost_cpl_propertylist_get_int (
    const cpl_propertylist *plist,
    const char *name,
    int *value);

cpl_error_code qmost_cpl_table_blank_window (
    cpl_table *tab,
    cpl_size start,
    cpl_size count);

cpl_error_code qmost_cpl_table_insert_blank_window (
    cpl_table *tab,
    cpl_size start,
    cpl_size count);

cpl_error_code qmost_cpl_table_extend_blank (
    cpl_table *tab,
    cpl_size new_length);

cpl_error_code qmost_cpl_table_copy_cell (
    cpl_table *outtbl,
    const char *outcol,
    cpl_size outrow,
    cpl_table *intbl,
    const char *incol,
    cpl_size inrow);

cpl_error_code qmost_cpl_table_get_byte_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    unsigned char *result,
    int n);

cpl_error_code qmost_cpl_table_get_double_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    double *result,
    int n);

cpl_error_code qmost_cpl_table_get_float_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    float *result,
    int n);

cpl_polynomial *qmost_cpl_table_get_polynomial (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    int degree);

cpl_error_code qmost_cpl_table_set_byte_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    unsigned char *values,
    int n);

cpl_error_code qmost_cpl_table_set_double_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    double *values,
    int n);

cpl_error_code qmost_cpl_table_set_polynomial (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    cpl_polynomial *poly,
    int degree);

cpl_error_code qmost_cpl_table_set_float_array (
    cpl_table *tptr,
    const char *column,
    cpl_size row,
    float *values,
    int n);

cpl_error_code qmost_load_master_mask (
    const cpl_frame *frame,
    const char *extname,
    cpl_mask **mask,
    cpl_propertylist **hdr);

cpl_error_code qmost_load_master_image (
    const cpl_frame *frame,
    const char *extname,
    const cpl_type type,
    cpl_image **image,
    cpl_propertylist **hdr);

cpl_error_code qmost_load_master_image_and_var (
    const cpl_frame *frame,
    const char *extname,
    const cpl_type type,
    cpl_image **image,
    cpl_image **var,
    cpl_propertylist **hdr);

cpl_error_code qmost_load_master_imagelist (
    const cpl_frame *frame,
    const char *extname,
    const cpl_type type,
    cpl_imagelist **imagelist,
    cpl_propertylist **hdr);

cpl_error_code qmost_load_master_table (
    const cpl_frame *frame,
    const char *extname,
    cpl_table **tbl,
    cpl_propertylist **hdr);

cpl_error_code qmost_copy_sensfunc (
    const cpl_frame *science_frame,
    const cpl_frame *sensfunc_frame,
    int arm,
    double exptime,
    cpl_propertylist *raw_hdr,
    cpl_propertylist *ext_hdr);

CPL_END_DECLS

#endif
