/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_detect_and_trace_fib.h"
#include "qmost_traceinfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_detect_and_trace_fib_test  Unit test of qmost_detect_and_trace_fib
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief  Helper for unit test of qmost_detect_and_trace_fib_lim.
 *
 * @param  xcen         (Given)  Array of reference x positions for
 *                               simulated fibre traces.
 * @param  nobj         (Given)  Number of fibre images to inject
 *                               (length of xcen array).
 * @param  coef         (Given)  Polynomial coefficients for traces.
 *                               The first coefficient (zeroth degree)
 *                               must be zero.
 * @param  nord         (Given)  Degree of polynomial (array coef
 *                               should have nord+1 elements).
 * @param  replmissing  (Given)  Add blank entries to trace table for
 *                               missing fibre images and attempt to
 *                               recover them if this flag is set.
 *
 * @return void
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_detect_and_trace_fib_lim_internal (
    int *xcen,
    int nobj,
    double *coef,
    int nord,
    int replmissing,
    int doref,
    int xbin,
    int ybin)
{
    int nx = 1024;
    int ny = 2048;
    int yref = 1025;
    int xfwhm = 3;
    double pkht = 64;

    cpl_polynomial *thetrace = NULL;
    int i;

    double yrel, xpoly;

    cpl_image *img = NULL;
    cpl_image *tmp_img = NULL;
    cpl_propertylist *hdr = NULL;
    float *img_data;
    int x, y, iobj;
    double dx, xw;

    cpl_table *fibinfo = NULL;

    cpl_table *ref_trace_tbl = NULL;
    cpl_propertylist *ref_trace_hdr = NULL;
    qmost_traceinfo trref;

    cpl_mask *slitmask = NULL;

    cpl_table *out_trace = NULL;
    cpl_propertylist *out_trace_hdr = NULL;
    cpl_mask *out_mask = NULL;

    qmost_traceinfo *tr = NULL;
    cpl_size icoef;
    double value;

    double *xexpect = NULL;
    int ipos;

    int xl, xh;

    cpl_error_code code;

    /* Set up polynomial */
    thetrace = cpl_polynomial_new(1);

    for(i = nord; i >= 0; i--) {
        icoef = i;
        cpl_polynomial_set_coeff(thetrace, &icoef, coef[i]);
    }

    /* Generate a test image with specified number of traces */
    img = cpl_image_new(nx, ny, CPL_TYPE_FLOAT);
    cpl_image_fill_noise_uniform(img, -8, 8);

    img_data = cpl_image_get_data_float(img);
    cpl_test_nonnull(img_data);

    if(img_data) {
        xw = 0.5*CPL_MATH_FWHM_SIG*CPL_MATH_FWHM_SIG / (xfwhm*xfwhm);

        for(y = 0; y < ny; y++) {
            yrel = (y+1 - yref) / ((float) yref);
            xpoly = cpl_polynomial_eval_1d(thetrace, yrel, NULL);

            for(iobj = 0; iobj < nobj; iobj++) {
                xl = xcen[iobj]+xpoly - 3*xfwhm;
                if(xl < 0)
                    xl = 0;
                if(xl >= nx)
                    xl = nx-1;
                
                xh = xcen[iobj]+xpoly + 3*xfwhm;
                if(xh < 0)
                    xh = 0;
                if(xh >= nx)
                    xh = nx-1;
                
                for(x = xl; x <= xh; x++) {
                    dx = (x+1) - (xcen[iobj]+xpoly);
                    img_data[y*nx+x] += pkht * exp(-xw * dx*dx);
                }
            }
        }
    }

    /* Bin, if requested */
    if(xbin != 1 || ybin != 1) {
        tmp_img = img;

        img = cpl_image_rebin(tmp_img, 1, 1, xbin, ybin);

        cpl_image_delete(tmp_img);
        tmp_img = NULL;
    }

    /* Minimal header */
    hdr = cpl_propertylist_new();
    cpl_propertylist_update_int(hdr, "ESO DRS SPATBIN", xbin);
    cpl_propertylist_update_int(hdr, "ESO DRS SPECBIN", ybin);

    /* A minimal fibinfo table for it (only columns we actually use)
       with a missing fibre in position 0 and the fibre we're
       extracting in position 1. */
    fibinfo = cpl_table_new(nobj+1);

    cpl_table_new_column(fibinfo, "FIB_ST", CPL_TYPE_INT);

    cpl_table_set_int(fibinfo, "FIB_ST", 0, 0);

    for(iobj = 0; iobj < nobj; iobj++) {
        cpl_table_set_int(fibinfo, "FIB_ST", 1+iobj, 1);
    }
    
    /* Ref trace */
    if(doref) {
        code = qmost_trcreate(&ref_trace_tbl, nord, 1);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        cpl_test_nonnull(ref_trace_tbl);

        ref_trace_hdr = cpl_propertylist_new();
        cpl_propertylist_update_int(ref_trace_hdr,
                                    "ESO DRS MINYST",
                                    1);
        cpl_propertylist_update_int(ref_trace_hdr,
                                    "ESO DRS MAXYFN",
                                    ny);

        for(iobj = 0; iobj < nobj; iobj++) {
            memset(&tr, 0, sizeof(tr));
            trref.specnum = iobj+1;
            trref.live = 1;
            trref.yst = 1;
            trref.yfn = ny;
            trref.yref = yref;
            trref.trrms = 0;
            trref.nord = nord;
            trref.coefs = cpl_polynomial_new(1);
            for(i = nord; i > 0; i--) {
                icoef = i;
                cpl_polynomial_set_coeff(trref.coefs, &icoef, coef[i]);
            }
            icoef = 0;
            cpl_polynomial_set_coeff(trref.coefs, &icoef, coef[0]+xcen[iobj]);
            trref.npos = 1;
            trref.xpos = cpl_malloc(sizeof(double));
            trref.xpos[0] = xcen[iobj];
            trref.ypos = cpl_malloc(sizeof(double));
            trref.ypos[0] = yref;
            trref.fwhm = cpl_malloc(sizeof(float));
            trref.fwhm[0] = xfwhm;
            trref.peak = cpl_malloc(sizeof(float));
            trref.peak[0] = pkht;
            trref.contrast = cpl_malloc(sizeof(float));
            trref.contrast[0] = 1.0;

            code = qmost_trwrite1(ref_trace_tbl, trref, trref.specnum);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            cpl_polynomial_delete(trref.coefs);
            cpl_free(trref.xpos);
            cpl_free(trref.ypos);
            cpl_free(trref.fwhm);
            cpl_free(trref.peak);
            cpl_free(trref.contrast);
        }
    }

    /* Make sure nothing went wrong while we were doing that */
    cpl_test_error(CPL_ERROR_NONE);

    /* NULL inputs */
    code = qmost_detect_and_trace_fib_lim(NULL,
                                          NULL,
                                          NULL,
                                          NULL,
                                          NULL,
                                          NULL,
                                          1,
                                          1,
                                          1,
                                          1,
                                          NULL,
                                          NULL,
                                          NULL,
                                          yref,
                                          replmissing);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Valid inputs */
    slitmask = cpl_mask_new(nx, ny);

    out_trace_hdr = cpl_propertylist_new();

    code = qmost_detect_and_trace_fib_lim(img,
                                          hdr,
                                          fibinfo,
                                          slitmask,
                                          ref_trace_tbl,
                                          ref_trace_hdr,
                                          16,
                                          5.0,
                                          nord,
                                          xfwhm,
                                          &out_trace,
                                          out_trace_hdr,
                                          &out_mask,
                                          yref,
                                          replmissing);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(out_trace);
    cpl_test_nonnull(out_mask);

    if(out_trace) {
        /* Make sure it's a valid trace table */
        code = qmost_trchk(out_trace);
        cpl_test_eq_error(code, CPL_ERROR_NONE);

        /* With length nobj+replmissing */
        cpl_test_eq(cpl_table_get_nrow(out_trace), nobj+replmissing);

        if(replmissing) {
            /* Read the entry for the bad fibre */
            tr = cpl_malloc(sizeof(qmost_traceinfo));
            
            code = qmost_trread1(out_trace, 1, tr);
            cpl_test_eq_error(code, CPL_ERROR_NONE);
            
            /* Check we got what we expect */
            cpl_test_eq(tr->live, 0);
            
            /* Clean up */
            qmost_trclose(1, &tr);
        }

        for(iobj = 0; iobj < nobj; iobj++) {
            /* Read the entry */
            tr = cpl_malloc(sizeof(qmost_traceinfo));

            code = qmost_trread1(out_trace, 1+iobj+replmissing, tr);
            cpl_test_eq_error(code, CPL_ERROR_NONE);

            /* Check we got what we expect */
            cpl_test_eq(tr->live, 1);
            cpl_test_abs(tr->yref, yref, 1);

            xexpect = cpl_malloc(tr->npos * sizeof(double));

            for(ipos = 0; ipos < tr->npos; ipos++) {
                yrel = (tr->ypos[ipos] - yref) / yref;
                xexpect[ipos] = xcen[iobj] + cpl_polynomial_eval_1d(thetrace,
                                                                    yrel,
                                                                    NULL);
            }

            qmost_test_arrays_double_abs(tr->xpos, xexpect, tr->npos, 1);

            cpl_free(xexpect);
            xexpect = NULL;

            /* Zero order */
            icoef = 0;
            value = cpl_polynomial_get_coeff(tr->coefs, &icoef);
            cpl_test_abs(value, xcen[iobj]+coef[0], 0.1);

            /* Other coefficients */
            for(i = 1; i <= nord; i++) {
                icoef = i;
                value = cpl_polynomial_get_coeff(tr->coefs, &icoef);
                cpl_test_abs(value, coef[i], 1.0);
            }

            /* Clean up */
            qmost_trclose(1, &tr);
        }

        cpl_table_delete(out_trace);
        out_trace = NULL;
    }

    if(out_mask) {
        /* We only do this test the first time where it's simple and
           should produce an unambiguous result. */
        if(nobj == 1 && nord == 0 && coef[0] == 0) {
            /* The fibre itself should be set */
            qmost_test_mask_subset_eq(out_mask, xcen[0], 1, xcen[0], ny, 1);
            
            /* The rest should not */
            xl = xcen[0]-xfwhm;
            qmost_test_mask_subset_eq(out_mask, 1, 1, xl, ny, 0);
            
            xh = xcen[0]+xfwhm;
            qmost_test_mask_subset_eq(out_mask, xh, 1, nx, ny, 0);
        }

        cpl_mask_delete(out_mask);
        out_mask = NULL;
    }

    cpl_mask_delete(slitmask);
    slitmask = NULL;

    cpl_propertylist_delete(out_trace_hdr);
    out_trace_hdr = NULL;

    /* Clean up */
    cpl_image_delete(img);
    img = NULL;

    cpl_propertylist_delete(hdr);
    hdr = NULL;

    cpl_table_delete(fibinfo);
    fibinfo = NULL;

    cpl_polynomial_delete(thetrace);
    thetrace = NULL;

    if(ref_trace_tbl != NULL) {
        cpl_table_delete(ref_trace_tbl);
        ref_trace_tbl = NULL;
    }

    if(ref_trace_hdr != NULL) {
        cpl_propertylist_delete(ref_trace_hdr);
        ref_trace_hdr = NULL;
    }
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_detect_and_trace_fib_lim
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_detect_and_trace_fib_lim(void)
{
    int xcen1[1] = { 512 };
    double coef1[1] = { 0 };

    int xcen2[1] = { 512 };
    double coef2[3] = { 0, 10.0, -10.0 };

    int xcen3[2] = { 512, 524 };
    double coef3[1] = { 0 };

    int xcen4[2] = { 512, 524 };
    double coef4[3] = { 0, 10.0, -10.0 };

    int xcen5[3] = { 500, 512, 524 };
    double coef5[3] = { 0, 10.0, -10.0 };

    /* Inject nothing, should detect nothing */
    test_qmost_detect_and_trace_fib_lim_internal(xcen1, 0, coef1, 0, 0, 0, 1, 1);

    /* replmissing = 0 */
    test_qmost_detect_and_trace_fib_lim_internal(xcen1, 1, coef1, 0, 0, 0, 1, 1);
    test_qmost_detect_and_trace_fib_lim_internal(xcen2, 1, coef2, 2, 0, 0, 1, 1);
    test_qmost_detect_and_trace_fib_lim_internal(xcen3, 2, coef3, 0, 0, 0, 1, 1);
    test_qmost_detect_and_trace_fib_lim_internal(xcen4, 2, coef4, 2, 0, 0, 1, 1);

    /* replmissing = 1 */
    test_qmost_detect_and_trace_fib_lim_internal(xcen1, 1, coef1, 0, 1, 0, 1, 1);
    test_qmost_detect_and_trace_fib_lim_internal(xcen2, 1, coef2, 2, 1, 0, 1, 1);
    test_qmost_detect_and_trace_fib_lim_internal(xcen3, 2, coef3, 0, 1, 0, 1, 1);
    test_qmost_detect_and_trace_fib_lim_internal(xcen4, 2, coef4, 2, 1, 0, 1, 1);

    /* Tests with binning */
    test_qmost_detect_and_trace_fib_lim_internal(xcen1, 1, coef1, 0, 0, 0, 2, 1);
    test_qmost_detect_and_trace_fib_lim_internal(xcen1, 1, coef1, 0, 0, 0, 1, 2);
    test_qmost_detect_and_trace_fib_lim_internal(xcen1, 1, coef1, 0, 0, 0, 2, 4);

    /* Tests with ref trace */
    test_qmost_detect_and_trace_fib_lim_internal(xcen5, 3, coef5, 2, 0, 1, 1, 1);
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_detect_and_trace_fib module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_detect_and_trace_fib_lim();

    return cpl_test_end(0);
}

/**@}*/
