/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
                                Includes
 -----------------------------------------------------------------------------*/

#include "qmost_lininfo.h"
#include "qmost_testutil.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_lininfo_test  Unit test of qmost_lininfo
 *
 */
/*----------------------------------------------------------------------------*/

/**@{*/

#define TOL 1.0e-15
#define FTOL 1.0e-5

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_lincrtab
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_lincrtab(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;

    /* NULL input */
    code = qmost_lincrtab(NULL, 7, 3);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid dimensions */
    code = qmost_lincrtab(&table, -1, 3);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    code = qmost_lincrtab(&table, 7, -1);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Create a simple table */
    code = qmost_lincrtab(&table, 7, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* Test it has the right format */
    code = qmost_linchk(table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Delete table */
    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_lincrinfo
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_lincrinfo(void)
{
    qmost_lininfo lin;
    cpl_error_code code;

    /* NULL input */
    code = qmost_lincrinfo(7, 3, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Invalid dimensions */
    code = qmost_lincrinfo(-1, 3, &lin);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    code = qmost_lincrinfo(7, -1, &lin);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);

    /* Valid call, should allocate workspace */
    code = qmost_lincrinfo(7, 3, &lin);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Free */
    qmost_lindelinfo(&lin);

    /* We shouldn't have leaked anything */
    cpl_test_memory_is_empty();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_lindelinfo
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_lindelinfo(void)
{
    qmost_lininfo *lin = NULL;

    /* NULL argument */
    qmost_lindelinfo(lin);

    /* Empty argument, should do nothing */
    lin = cpl_calloc(1, sizeof(qmost_lininfo));
    qmost_lindelinfo(lin);

    /* Populated argument, should free internal references as well */
    lin->coefs = cpl_polynomial_new(1);
    lin->med = cpl_malloc(2 * sizeof(double));
    lin->exptime = cpl_malloc(2 * sizeof(float));
    lin->linmed = cpl_malloc(2 * sizeof(double));
    lin->fit_flag = cpl_malloc(2 * sizeof(unsigned char));

    qmost_lindelinfo(lin);

    cpl_free(lin);

    /* We shouldn't have leaked anything */
    cpl_test_memory_is_empty();
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_linchk
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_linchk(void)
{
    cpl_table *table = NULL;
    cpl_error_code code;
    int i;
    char *colname = NULL;

    /* NULL input */
    code = qmost_linchk(NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Create an empty table */
    table = cpl_table_new(0);
    cpl_test_nonnull(table);
    
    /* Empty table */
    code = qmost_linchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Incorrect number of columns */
    code = cpl_table_new_column(table, "BADNAME", CPL_TYPE_INT);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    
    code = qmost_linchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    /* Correct number of columns but incorrect names */
    for(i = 0; i < 21; i++) {
        colname = cpl_sprintf("BAD%d", i);
        code = cpl_table_new_column(table, colname, CPL_TYPE_INT);
        cpl_test_eq_error(code, CPL_ERROR_NONE);
        cpl_free(colname);
    }
    
    code = qmost_linchk(table);
    cpl_test_eq_error(code, CPL_ERROR_ILLEGAL_INPUT);
    
    cpl_table_delete(table);
    table = NULL;

    /* Proper format */
    code = qmost_lincrtab(&table, 7, 3);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* Test it has the right format */
    code = qmost_linchk(table);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    /* Delete table */
    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_linwrite
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_linwrite(void)
{
    cpl_table *table = NULL;
    cpl_propertylist *hdr = NULL;
    qmost_lininfo *lins = NULL;
    int namps = 4, nfiles = 10, nord = 3;
    int iamp;
    cpl_error_code code;
    cpl_size degree;

    int icol, ncols;
    const char *colname;
    cpl_array *colnames = NULL;
    cpl_table *test_tbl = NULL;

    /* Empty arguments */
    table = cpl_table_new(0);
    lins = cpl_calloc(namps, sizeof(qmost_lininfo));
    hdr = cpl_propertylist_new();

    /* NULL input tests */
    code = qmost_linwrite(NULL, hdr, lins, namps);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_linwrite(table, NULL, lins, namps);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_linwrite(table, hdr, NULL, namps);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* No amps */
    code = qmost_linwrite(table, hdr, lins, 0);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);
    
    /* Attempt to write into an empty, uniniitialized table.  Fails
     * when column can't be found. */
    code = qmost_linwrite(table, hdr, lins, namps);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Create a proper table now */
    cpl_table_delete(table);
    table = NULL;

    code = qmost_lincrtab(&table, nfiles, nord);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* Now try again, fails at trying to set polynomial, which is NULL */
    code = qmost_linwrite(table, hdr, lins, namps);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Create the lininfos */
    for(iamp = 0; iamp < namps; iamp++) {
        qmost_lincrinfo(nfiles, nord, lins + iamp); 
    }

    /* Fudge the number of files so we can trigger the mismatch */
    lins[0].nfiles = 0;

    code = qmost_linwrite(table, hdr, lins, namps);
    cpl_test_eq_error(code, CPL_ERROR_INCOMPATIBLE_INPUT);

    /* This should work but write zeros */
    lins[0].nfiles = nfiles;

    code = qmost_linwrite(table, hdr, lins, namps);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_propertylist_get_double(hdr, "ESO QC LIN RMS MEAN"),
                 0.0, TOL);
    cpl_test_abs(cpl_propertylist_get_double(hdr, "ESO QC LIN NL10K"),
                 100.0, TOL);

    /* Do a series of tests where we feed in the table with each
     * column missing to set off the individual error handlers for
     * the column writes. */
    colnames = cpl_table_get_column_names(table);
    ncols = cpl_table_get_ncol(table);

    for(icol = 0; icol < ncols; icol++) {
        colname = cpl_array_get_string(colnames, icol);
        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_linwrite(test_tbl, hdr, lins, namps);
        cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    cpl_propertylist_delete(hdr);
    hdr = NULL;

    /* More realistic inputs are tested in linear but here we do a
     * basic check of the level calculation by setting the polynomial
     * to a linear detector (linear term = 1) to make sure the result
     * is zero. */
    for(iamp = 0; iamp < namps; iamp++) {
        degree = 1;
        cpl_polynomial_set_coeff(lins[iamp].coefs, &degree, 1.0);
    }

    hdr = cpl_propertylist_new();

    code = qmost_linwrite(table, hdr, lins, namps);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_test_abs(cpl_propertylist_get_double(hdr, "ESO QC LIN NL10K"),
                 0.0, TOL);

    cpl_table_delete(table);
    table = NULL;

    cpl_propertylist_delete(hdr);
    hdr = NULL;

    for(iamp = 0; iamp < namps; iamp++) {
        qmost_lindelinfo(lins + iamp); 
    }

    cpl_free(lins);
    lins = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit test of qmost_linread
 */
/*----------------------------------------------------------------------------*/

static void test_qmost_linread(void)
{
    cpl_table *table = NULL;
    cpl_propertylist *hdr = NULL;
    qmost_lininfo *lins = NULL;
    qmost_lininfo *lins_ans = NULL;
    int namps, namps_ans = 4, ifile, nfiles = 10, nord = 3;
    int iamp;
    cpl_error_code code;
    cpl_size degree;

    int icol, ncols;
    const char *colname;
    cpl_array *colnames = NULL;
    cpl_table *test_tbl = NULL;

    /* Empty arguments */
    table = cpl_table_new(0);

    /* NULL input tests */
    code = qmost_linread(NULL, &lins, &namps);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_linread(table, NULL, &namps);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    code = qmost_linread(table, &lins, NULL);
    cpl_test_eq_error(code, CPL_ERROR_NULL_INPUT);

    /* Empty table */
    code = qmost_linread(table, &lins, &namps);
    cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

    /* Create a proper table now */
    cpl_table_delete(table);
    table = NULL;

    code = qmost_lincrtab(&table, nfiles, nord);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(table);

    /* Create the lininfos and write out */
    lins_ans = cpl_calloc(namps_ans, sizeof(qmost_lininfo));

    for(iamp = 0; iamp < namps_ans; iamp++) {
        qmost_lincrinfo(nfiles, nord, lins_ans + iamp); 

        lins_ans[iamp].amp = iamp+1;
        lins_ans[iamp].xmin = 1 + 64*iamp;
        lins_ans[iamp].xmax = 64*(iamp+1);
        lins_ans[iamp].ymin = 1 + 128*iamp;
        lins_ans[iamp].ymax = 128*(iamp+1);

        degree = 1;
        cpl_polynomial_set_coeff(lins_ans[iamp].coefs,
                                 &degree,
                                 1.0);

        lins_ans[iamp].rms = 1.0;
        lins_ans[iamp].ngood = 4;

        for(ifile = 0; ifile < nfiles; ifile++) {
            lins_ans[iamp].med[ifile] = 1000*ifile;
            lins_ans[iamp].exptime[ifile] = ifile;
            lins_ans[iamp].moncorr[ifile] = 1;
            lins_ans[iamp].linmed[ifile] = 0;
            lins_ans[iamp].fit_flag[ifile] = 0;
        }
    }

    hdr = cpl_propertylist_new();

    code = qmost_linwrite(table, hdr, lins_ans, namps_ans);
    cpl_test_eq_error(code, CPL_ERROR_NONE);

    cpl_propertylist_delete(hdr);
    hdr = NULL;

    /* Should be able to read it back successfully now */
    code = qmost_linread(table, &lins, &namps);
    cpl_test_eq_error(code, CPL_ERROR_NONE);
    cpl_test_nonnull(lins);
    cpl_test_eq(namps, namps_ans);

    for(iamp = 0; iamp < namps; iamp++) {
        cpl_test_eq(lins[iamp].amp, lins_ans[iamp].amp);
        cpl_test_eq(lins[iamp].xmin, lins_ans[iamp].xmin);
        cpl_test_eq(lins[iamp].xmax, lins_ans[iamp].xmax);
        cpl_test_eq(lins[iamp].ymin, lins_ans[iamp].ymin);
        cpl_test_eq(lins[iamp].ymax, lins_ans[iamp].ymax);
        cpl_test_eq(lins[iamp].nord, lins_ans[iamp].nord);

        for(degree = 0; degree < lins[iamp].nord; degree++) {
            cpl_test_abs(cpl_polynomial_get_coeff(lins[iamp].coefs,
                                                  &degree),
                         cpl_polynomial_get_coeff(lins_ans[iamp].coefs,
                                                  &degree),
                         DBL_EPSILON);
        }

        cpl_test_eq(lins[iamp].nfiles, lins_ans[iamp].nfiles);
        cpl_test_eq(lins[iamp].ngood, lins_ans[iamp].ngood);

        for(ifile = 0; ifile < nfiles; ifile++) {
            cpl_test_abs(lins[iamp].med[ifile],
                         lins_ans[iamp].med[ifile],
                         1.0);
            cpl_test_abs(lins[iamp].exptime[ifile],
                         lins_ans[iamp].exptime[ifile],
                         10*DBL_EPSILON);
            cpl_test_abs(lins[iamp].moncorr[ifile],
                         lins_ans[iamp].moncorr[ifile],
                         DBL_EPSILON);
            cpl_test_abs(lins[iamp].linmed[ifile],
                         lins_ans[iamp].linmed[ifile],
                         DBL_EPSILON);
            cpl_test_eq(lins[iamp].fit_flag[ifile],
                        lins_ans[iamp].fit_flag[ifile]);
        }

        qmost_lindelinfo(lins + iamp);
    }
    cpl_free(lins);
    lins = NULL;

    /* Do a series of tests where we feed in the table with each
     * column missing to set off the individual error handlers for
     * the column reads. */
    colnames = cpl_table_get_column_names(table);
    ncols = cpl_table_get_ncol(table);

    for(icol = 0; icol < ncols; icol++) {
        colname = cpl_array_get_string(colnames, icol);

        /* lins aren't read so need to skip */
        if(!strncmp(colname, "lin", 3))
            continue;

        test_tbl = cpl_table_duplicate(table);
        cpl_table_erase_column(test_tbl, colname);

        code = qmost_linread(test_tbl, &lins, &namps);
        cpl_test_eq_error(code, CPL_ERROR_DATA_NOT_FOUND);

        cpl_table_delete(test_tbl);
        test_tbl = NULL;
    }

    cpl_array_delete(colnames);
    colnames = NULL;

    for(iamp = 0; iamp < namps_ans; iamp++) {
        qmost_lindelinfo(lins_ans + iamp);
    }
    cpl_free(lins_ans);
    lins_ans = NULL;

    cpl_table_delete(table);
    table = NULL;
}

/*----------------------------------------------------------------------------*/
/**
  @brief    Unit tests of qmost_lininfo module
 */
/*----------------------------------------------------------------------------*/

int main(void)
{
    cpl_test_init(PACKAGE_BUGREPORT, CPL_MSG_WARNING);

    test_qmost_lincrtab();
    test_qmost_lincrinfo();
    test_qmost_lindelinfo();
    test_qmost_linchk();
    test_qmost_linwrite();
    test_qmost_linread();

    return cpl_test_end(0);
}

/**@}*/
