/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_dfs.h"
#include "qmost_diffimg.h"
#include "qmost_filt2d.h"
#include "qmost_gaincor.h"
#include "qmost_imcombine_lite.h"
#include "qmost_pfits.h"
#include "qmost_stats.h"
#include "qmost_utils.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_detector_flat_analyse"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_detector_flat_analyse_description[] =
    "Process a set of detector flats to form a master detector flat.\n"
    "The input exposures are overscan corrected, trimmed, bias, and\n"
    "dark corrected.  The resulting processed frames are combined to\n"
    "form a master detector flat.  Ratios of the count levels in the\n"
    "overlap regions along the amp boundaries are used to determine\n"
    "the relative gains of the amps and apply a gain correction to\n"
    "remove discontinuities at the amp boundaries.  The resulting\n"
    "now contiguous image is then filtered using 2D median and\n"
    "boxcar filters to remove the strong illumination pattern cast\n"
    "by the LEDs to obtain a flat suitable for correcting pixel\n"
    "to pixel sensitivity variations.  The gain correction is then\n"
    "applied in reverse to the resulting flat to obtain a final\n"
    "detector flat that applies both the gain correction and the\n"
    "pixel to pixel flat field correction.\n\n"
    "Optionally, the detector flat can be compared to a reference\n"
    "to look for statistical or spatial variations.  This is done\n"
    "prior to filtering using the unfiltered detector flat output\n"
    "(see tables below) so the comparison is sensitive to any\n"
    "variations in the illumination pattern.\n\n"
    "The following files can be specified in the SOF:\n\n"
    "Description                  Req/Opt? Tag\n"
    "---------------------------- -------- --------------------\n"
    "Detector flat                Required " QMOST_RAW_DETECTOR_FLAT "\n"
    "Master bias frame            Optional " QMOST_PRO_MASTER_BIAS "\n"
    "Master dark frame            Optional " QMOST_PRO_MASTER_DARK "\n"
    "Master bad pixel mask        Optional " QMOST_CALIB_MASTER_BPM "\n"
    "Reference detector flat      Optional " QMOST_CALIB_REFERENCE_DETECTOR_FLAT "\n"
    "\n"
    "Outputs:\n\n"
    "Description                  Tag\n"
    "---------------------------- -----------------\n"
    "Master detector flat         " QMOST_PRO_MASTER_DETECTOR_FLAT "\n"
    "Unfiltered detector flat     " QMOST_PRO_UNFILTERED_DETECTOR_FLAT "\n"
    "or\n"
    "Reference detector flat      " QMOST_CALIB_REFERENCE_DETECTOR_FLAT "\n"
    "\n"
    "The type of output depends on the recipe parameter reference.\n"
    "\n"
    "Additional outputs if reference frame is given:\n\n"
    "Description                  Tag\n"
    "---------------------------- -----------------\n"
    "Detector flat comparison     " QMOST_PRO_DIFFIMG_DETECTOR_FLAT "\n"
    "\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the master detector flat\n"
    "for LRS-B would be QMOST_" QMOST_PRO_MASTER_DETECTOR_FLAT "_LRS-B.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_detector_flat_analyse,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Process detector flats to form a master detector flat",
                  qmost_detector_flat_analyse_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_detector_flat_analyse    qmost_detector_flat_analyse
 *
 * @brief  Process detector flats to form a master detector flat.
 *
 * @par Name:
 *   qmost_detector_flat_analyse
 * @par Purpose:
 *   Process a set of detector flats to form a master detector flat.
 *   The input exposures are overscan corrected, trimmed, bias, and
 *   dark corrected.  The resulting processed frames are combined to
 *   form a master detector flat.  Ratios of the count levels in the
 *   overlap regions along the amp boundaries are used to determine
 *   the relative gains of the amps and apply a gain correction to
 *   remove discontinuities at the amp boundaries.  The resulting
 *   now contiguous image is then filtered using 2D median and
 *   boxcar filters to remove the strong illumination pattern cast
 *   by the LEDs to obtain a flat suitable for correcting pixel
 *   to pixel sensitivity variations.  The gain correction is then
 *   applied in reverse to the resulting flat to obtain a final
 *   detector flat that applies both the gain correction and the
 *   pixel to pixel flat field correction.
 *   Optionally, the detector flat can be compared to a reference to
 *   look for statistical or spatial variations.  This is done prior
 *   to filtering using the unfiltered detector flat output (see
 *   tables below) so the comparison is sensitive to any variations in
 *   the illumination pattern.
 * @par Type:
 *   Detector calibration
 * @par Parameters:
 *   - @b imcombine.combtype (int, default 1):
 *     Determines the type of combination that is done to form the
 *     stacked detector flat frame.  Can take the following values:
 *       - (1): The output pixels are kappa sigma clipped means of
 *              the input pixels.
 *       - (2): The output pixels are medians of the input pixels.
 *   - @b imcombine.scaletype (int, default 2):
 *     Scaling to use when combining multiple images to form the
 *     stacked detector flat frame.  Can take the following values:
 *       - (0): No biasing or scaling is done.
 *       - (1): An additive offset is applied to remove any
 *              difference in background levels.
 *       - (2): A multiplicative scaling is applied to remove any
 *              difference in background levels.
 *       - (3): Images are scaled by their relative exposure times
 *              and an additive offset is applied to remove any
 *              remaining difference in background levels.
 *   - @b imcombine.xrej (bool, default false):
 *     If set, then an extra rejection cycle will be run.
 *   - @b imcombine.thresh (float, default 5.0):
 *     The rejection threshold for clipping during the combination in
 *     units of the background sigma.
 *   - @b gaincor.window (int, default 16)
 *     Measurement window size of the strip at the adjacent edges of
 *     the amplifiers to use for gain correction.
 *   - @b detflat.medfil (int, default 25)
 *     Window size of 2D median filter used to remove the LED
 *     illumination pattern.
 *   - @b detflat.linfil (int, default 5)
 *     Window size of 2D boxcar filter used to remove the LED
 *     illumination pattern.
 *   - @b diffimg.ncells (int, default 256):
 *     If a difference image is being formed between the current
 *     combined bias frame and a library reference bias product, then
 *     this is the number of cells in which to divide the image to
 *     compute statistics.  The value should be a power of 2 up to and
 *     including 256.
 *   - @b reference (bool, default false):
 *     Emit a reference frame.  Caution: the resulting frame must be
 *     inspected and approved before being allowed to enter the
 *     calibration database or used as a reference.
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - @b DETECTOR_FLAT: Raw detector flat frames.
 *    - @b MASTER_BIAS (optional): Master bias frame.
 *    - @b MASTER_DARK (optional): Master dark frame.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 *    - @b REFERENCE_DETECTOR_FLAT (optional): A library reference
 *         UNFILTERED_DETECTOR_FLAT frame.  If given, then the new
 *         detector flat produced by the recipe will be compared to
 *         this reference and additional QC headers giving the results
 *         of the comparison are written to the output files.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b MASTER_DETECTOR_FLAT: The master detector flat frame.
 *     - @b UNFILTERED_DETECTOR_FLAT: If recipe parametere reference
 *          is false, the CCD processed, unfiltered original detector
 *          flat prior to removal of the illumination pattern.  This
 *          can be used to assess the quality of the bias and dark
 *          correction as well as the flat field illumination to check
 *          for any patterns or variations that are removed by the
 *          filtering applied to the master file, and is used for
 *          comparing to a reference.
 *     - @b REFERENCE_DETECTOR_FLAT: If recipe parameter reference is
 *          true, the new reference detector flat frame.
 *     - @b DIFFIMG_DETECTOR_FLAT): If a REFERENCE_DETECTOR_FLAT is
 *          included in the inputs, this output is the ratio image
 *          formed by dividing the new detector flat by the
 *          reference.  This is done on the unfiltered detector flat
 *          to make it sensitive to changes in the illumination
 *          pattern indicative of a problem with the lamps.
 *
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     master detector flat frame for LRS-B would be @c
 *     QMOST_MASTER_DETECTOR_FLAT_LRS-B.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean residual background level
 *     in the illuminated region of the overscan subtracted frames
 *     that were combined.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined, as a measure of how
 *     consistent the background levels were.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>COMPARE MED</b>: The median of the pixels in the ratio
 *     image.
 *   - <b>COMPARE RMS</b>: The robustly-estimated RMS of the pixels in
 *     the ratio image.
 *   - <b>COMPARE BIN MEAN</b>: The mean over all of the cells
 *     of the median of the pixels in each cell.
 *   - <b>COMPARE BIN RMS</b>: The RMS over all of the cells of
 *     the median of the pixels in each cell.
 *   - <b>COMPARE BIN MIN</b>: The minimum over all of the cells
 *     of the median of the pixels in each cell.
 *   - <b>COMPARE BIN MAX</b>: The maximum over all of the cells
 *     of the median of the pixels in each cell.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - No raw detector flat frames in the input frameset.
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 * @par Non-Fatal Error Conditions:
 *   - Missing reference detector flat frame (no comparison is done).
 *   - Missing bad pixel mask (all pixels assumed to be good).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 *   - Dummy extension in reference file.
 * @par Functional Diagram:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 *
 *   "MASTER_BPM\nMASTER_BIAS\nMASTER_DARK" -> qmost_ccdproc_and_combine [style="dashed"];
 *   REFERENCE_DETECTOR_FLAT -> qmost_diffimg [style="dashed"];
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     DETECTOR_FLAT -> qmost_ccdproc_and_combine;
 *     qmost_ccdproc_and_combine -> qmost_gaincor;
 *     qmost_gaincor -> qmost_bfilt2;
 *     qmost_gaincor -> cpl_image_divide;
 *     qmost_bfilt2 -> cpl_image_divide;
 *
 *     cpl_image_divide -> MASTER_DETECTOR_FLAT;
 *   }
 *
 *   qmost_gaincor -> qmost_diffimg;
 *
 *   qmost_gaincor -> UNFILTERED_DETECTOR_FLAT;
 *   qmost_diffimg -> DIFFIMG_DETECTOR_FLAT;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_gaincor -> QC1;
 *   qmost_diffimg -> QC1;
 *
 *   DETECTOR_FLAT [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BPM\nMASTER_BIAS\nMASTER_DARK" [shape="box" fillcolor="#fff5ce"]
 *   REFERENCE_DETECTOR_FLAT [shape="box" fillcolor="#fff5ce"]
 *   MASTER_DETECTOR_FLAT [shape="box" fillcolor="#fff5ce"]
 *   UNFILTERED_DETECTOR_FLAT [shape="box" fillcolor="#ffffff"]
 *   DIFFIMG_DETECTOR_FLAT [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_detector_flat_analyse(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    const cpl_frame *master_bias_frame = NULL;
    const cpl_frame *master_dark_frame = NULL;
    const cpl_frame *master_bpm_frame = NULL;
    const cpl_frame *master_lintab_frame = NULL;
    const cpl_frame *reference_detflat_frame = NULL;

    const cpl_frame *master_detflat_frame = NULL;
    const char *master_detflat_filename = NULL;
    const cpl_frame *unfiltered_detflat_frame = NULL;
    const char *unfiltered_detflat_filename = NULL;
    const cpl_frame *diffimg_detflat_frame = NULL;
    const char *diffimg_detflat_filename = NULL;

    cpl_frameset *detflats_raw = NULL;
    int iin, nin, nfiles;

    const cpl_frame *this_frame;
    const char *this_tag;

    const cpl_frame *first_detflat_frame;
    const char *first_detflat_filename = NULL;
    int spec;

    const cpl_parameter *par;
    int combtype = QMOST_MEANCALC;
    int scaletype = 2;
    int xrej = 0;
    double combthresh = 5.0;
    int stripsize = 16;
    int medfil = 25;
    int linfil = 5;
    int ncells = 256;
    int make_reference = 0;

    cpl_size raw_extension;
    int arm, detlive;
    const char *arm_extname;

    cpl_mask *master_bpm = NULL;
    cpl_image *master_bias_img = NULL;
    cpl_image *master_bias_var = NULL;
    cpl_image *master_dark_img = NULL;
    cpl_image *master_dark_var = NULL;
    cpl_table *master_lintab = NULL;

    cpl_propertylist *detflat_hdr = NULL;

    cpl_propertylist *qclist = NULL;

    cpl_image *detflat_img = NULL;
    cpl_image *detflat_var = NULL;
    cpl_image *filtered_img = NULL;

    cpl_image *reference_detflat_img = NULL;
    cpl_image *diff_img = NULL;
    cpl_errorstate prestate;

#undef TIDY
#define TIDY                                    \
    if(detflats_raw != NULL) {                  \
        cpl_frameset_delete(detflats_raw);      \
        detflats_raw = NULL;                    \
    }                                           \
    if(master_bpm != NULL) {                    \
        cpl_mask_delete(master_bpm);            \
        master_bpm = NULL;                      \
    }                                           \
    if(master_bias_img != NULL) {               \
        cpl_image_delete(master_bias_img);      \
        master_bias_img = NULL;                 \
    }                                           \
    if(master_bias_var != NULL) {               \
        cpl_image_delete(master_bias_var);      \
        master_bias_var = NULL;                 \
    }                                           \
    if(master_dark_img != NULL) {               \
        cpl_image_delete(master_dark_img);      \
        master_dark_img = NULL;                 \
    }                                           \
    if(master_dark_var != NULL) {               \
        cpl_image_delete(master_dark_var);      \
        master_dark_var = NULL;                 \
    }                                           \
    if(master_lintab) {                         \
        cpl_table_delete(master_lintab);        \
        master_lintab = NULL;                   \
    }                                           \
    if(detflat_hdr != NULL) {                   \
        cpl_propertylist_delete(detflat_hdr);   \
        detflat_hdr = NULL;                     \
    }                                           \
    if(qclist != NULL) {                        \
        cpl_propertylist_delete(qclist);        \
        qclist = NULL;                          \
    }                                           \
    if(detflat_img != NULL) {                   \
        cpl_image_delete(detflat_img);          \
        detflat_img = NULL;                     \
    }                                           \
    if(detflat_var != NULL) {                   \
        cpl_image_delete(detflat_var);          \
        detflat_var = NULL;                     \
    }                                           \
    if(filtered_img != NULL) {                  \
        cpl_image_delete(filtered_img);         \
        filtered_img = NULL;                    \
    }                                           \
    if(reference_detflat_img != NULL) {            \
        cpl_image_delete(reference_detflat_img);   \
        reference_detflat_img = NULL;              \
    }                                           \
    if(diff_img != NULL) {                      \
        cpl_image_delete(diff_img);             \
        diff_img = NULL;                        \
    }                                           \

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Master detector-level calibrations, all optional */
    master_bias_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_BIAS);
    master_dark_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_DARK);
    master_bpm_frame = cpl_frameset_find_const(frameset,
                                               QMOST_CALIB_MASTER_BPM);
    master_lintab_frame = cpl_frameset_find_const(frameset,
                                                  QMOST_PRO_LINEARITY);
    reference_detflat_frame = cpl_frameset_find_const(frameset,
                                                   QMOST_CALIB_REFERENCE_DETECTOR_FLAT);

    /* Create frameset of inputs */
    detflats_raw = cpl_frameset_new();

    nin = cpl_frameset_get_size(frameset);

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_DETECTOR_FLAT)) {
            cpl_frameset_insert(detflats_raw,
                                cpl_frame_duplicate(this_frame));
        }
    }

    /* Check we have some */
    nfiles = cpl_frameset_get_size(detflats_raw);
    if(nfiles < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_RAW_DETECTOR_FLAT);
    }

    /* Retrieve imcombine parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.combtype");
    if(par != NULL)
        combtype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.scaletype");
    if(par != NULL)
        scaletype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.xrej");
    if(par != NULL)
        xrej = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.thresh");
    if(par != NULL)
        combthresh = cpl_parameter_get_double(par);

    /* Retrieve gaincor parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".gaincor.window");
    if(par != NULL)
        stripsize = cpl_parameter_get_int(par);

    /* Retrieve detflat parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".detflat.medfil");
    if(par != NULL)
        medfil = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".detflat.linfil");
    if(par != NULL)
        linfil = cpl_parameter_get_int(par);

    /* Retrieve diffimg parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".diffimg.ncells");
    if(par != NULL)
        ncells = cpl_parameter_get_int(par);

    /* Reference parameter */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".reference");
    if(par != NULL)
        make_reference = cpl_parameter_get_bool(par);

    /* Get first file to use as basis for output */
    first_detflat_frame = cpl_frameset_get_position(detflats_raw, 0);
    if(first_detflat_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get first detector flat "
                                     "from SOF");
    }

    /* Get filename of first detector flat */
    first_detflat_filename = cpl_frame_get_filename(first_detflat_frame);
    if(first_detflat_filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename for "
                                     "first detector flat frame");
    }

    /* Extract the primary FITS header of the reference */
    detflat_hdr = cpl_propertylist_load(first_detflat_filename, 0);
    if(detflat_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FITS primary header "
                                     "from first input file");
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(detflat_hdr,
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    cpl_propertylist_delete(detflat_hdr);
    detflat_hdr = NULL;

    /* Create output */
    master_detflat_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        QMOST_PRO_MASTER_DETECTOR_FLAT,
        CPL_FRAME_TYPE_TABLE,
        NULL);
    if(master_detflat_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     QMOST_PRO_MASTER_DETECTOR_FLAT);
    }

    master_detflat_filename = cpl_frame_get_filename(master_detflat_frame);

    unfiltered_detflat_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        make_reference ?
        QMOST_CALIB_REFERENCE_DETECTOR_FLAT :
        QMOST_PRO_UNFILTERED_DETECTOR_FLAT,
        CPL_FRAME_TYPE_TABLE,
        NULL);
    if(unfiltered_detflat_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     make_reference ?
                                     QMOST_CALIB_REFERENCE_DETECTOR_FLAT :
                                     QMOST_PRO_UNFILTERED_DETECTOR_FLAT);
    }
    
    unfiltered_detflat_filename = cpl_frame_get_filename(unfiltered_detflat_frame);

    if(reference_detflat_frame != NULL) {
        diffimg_detflat_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PRO_DIFFIMG_DETECTOR_FLAT,
            CPL_FRAME_TYPE_TABLE,
            NULL);
        if(diffimg_detflat_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PRO_DIFFIMG_DETECTOR_FLAT);
        }
        
        diffimg_detflat_filename = cpl_frame_get_filename(diffimg_detflat_frame);
    }

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Extract the FITS header of the reference */
        detflat_hdr = cpl_propertylist_load(
            first_detflat_filename,
            raw_extension);
        if(detflat_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS extension "
                                         "%lld header from first input file",
                                         raw_extension);
        }

        /* Check the reference to find out what arm we're processing */
        if(qmost_pfits_get_arm(detflat_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine EXTNAME "
                                         "for arm %d", arm);
        }

        /* Is detector live? */
        if(qmost_pfits_get_detlive(detflat_hdr, &detlive) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read live flag for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(!detlive) {
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %lld = %s",
                         raw_extension, arm_extname);

            /* Detector is not live, write out dummy HDUs to products
             * and move on. */
            if(reference_detflat_frame != NULL) {
                if(qmost_dfs_save_image_extension(diffimg_detflat_frame,
                                                  detflat_hdr,
                                                  arm_extname,
                                                  NULL,
                                                  NULL,
                                                  CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "detector flat "
                                                 "difference image %s[%s]",
                                                 diffimg_detflat_filename,
                                                 arm_extname);
                }
            }

            if(qmost_dfs_save_image_and_var(unfiltered_detflat_frame,
                                            detflat_hdr,
                                            arm_extname,
                                            NULL,
                                            NULL,
                                            NULL,
                                            CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy unfiltered "
                                             "detector flat frame %s[%s]",
                                             unfiltered_detflat_filename,
                                             arm_extname);
            }

            if(qmost_dfs_save_image_and_var(master_detflat_frame,
                                            detflat_hdr,
                                            arm_extname,
                                            NULL,
                                            NULL,
                                            NULL,
                                            CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy master "
                                             "detector flat frame %s[%s]",
                                             master_detflat_filename,
                                             arm_extname);
            }

            cpl_propertylist_delete(detflat_hdr);
            detflat_hdr = NULL;

            continue;
        }

        cpl_msg_info(cpl_func,
                     "Processing extension %lld = %s",
                     raw_extension, arm_extname);
        cpl_msg_indent_more();

        /* Load all masters using the correct EXTNAME for the arm
           we're processing in case the mapping of extension number to
           arm could have changed. */
        if(master_bias_frame != NULL) {
            if(qmost_load_master_image_and_var(master_bias_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_bias_img,
                                               &master_bias_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master bias "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_bias_img = NULL;
            master_bias_var = NULL;
        }
    
        if(master_dark_frame != NULL) {
            if(qmost_load_master_image_and_var(master_dark_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_dark_img,
                                               &master_dark_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master dark "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_dark_img = NULL;
            master_dark_var = NULL;
        }

        if(master_bpm_frame != NULL) {
            if(qmost_load_master_mask(master_bpm_frame,
                                      arm_extname,
                                      &master_bpm,
                                      NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM");
            }
        }

        if(master_lintab_frame != NULL) {
            if(qmost_load_master_table(master_lintab_frame,
                                       arm_extname,
                                       &master_lintab,
                                       NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master "
                                             "linearity table");
            }
        }

        /* Create QC parameter list */
        qclist = cpl_propertylist_new();

        /* CCD processing and stacking */
        if(qmost_ccdproc_and_combine(detflats_raw,
                                     raw_extension,
                                     master_bpm,
                                     master_bias_img,
                                     master_bias_var,
                                     master_dark_img,
                                     master_dark_var,
                                     NULL,
                                     NULL,
                                     0,
                                     NULL,
                                     1,
                                     master_lintab,
                                     QMOST_LIN_AFTER_BIAS,
                                     combtype,
                                     scaletype,
                                     0,
                                     xrej,
                                     combthresh,
                                     &detflat_img,
                                     &detflat_var,
                                     qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "basic CCD processing failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(master_bias_img != NULL) {
            cpl_image_delete(master_bias_img);
            master_bias_img = NULL;
        }

        if(master_bias_var != NULL) {
            cpl_image_delete(master_bias_var);
            master_bias_var = NULL;
        }

        if(master_dark_img != NULL) {
            cpl_image_delete(master_dark_img);
            master_dark_img = NULL;
        }

        if(master_dark_var != NULL) {
            cpl_image_delete(master_dark_var);
            master_dark_var = NULL;
        }

        if(master_bpm) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }

        if(master_lintab != NULL) {
            cpl_table_delete(master_lintab);
            master_lintab = NULL;
        }

        /* Gain correction */
        filtered_img = cpl_image_duplicate(detflat_img);

        if(qmost_gaincor(filtered_img,
                         qclist,
                         stripsize) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't gain correct "
                                         "detector flat frame");
        }

        /* Difference, if reference was given.  This needs to be done
         * before we save the masters so the FITS header includes the
         * QC from the comparison. */
        if(reference_detflat_frame != NULL) {
            /* Load reference.  Failure to do so is non-fatal and just
             * produces a warning and a dummy HDU in the difference
             * image file.  This includes the case where the
             * reference was a dummy. */
            prestate = cpl_errorstate_get();

            if(qmost_load_master_image(reference_detflat_frame,
                                       arm_extname,
                                       CPL_TYPE_FLOAT,
                                       &reference_detflat_img,
                                       NULL) != CPL_ERROR_NONE) {
                cpl_msg_warning(cpl_func,
                                "couldn't load reference "
                                "detector flat frame "
                                "%s[%s]: %s",
                                cpl_frame_get_filename(reference_detflat_frame),
                                arm_extname,
                                cpl_error_get_message());

                cpl_errorstate_set(prestate);
                reference_detflat_img = NULL;
            }

            if(reference_detflat_img != NULL) {
                if(qmost_diffimg(filtered_img,
                                 reference_detflat_img,
                                 ncells,
                                 1,
                                 &diff_img,
                                 qclist) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func,
                                                 cpl_error_get_code(),
                                                 "couldn't difference "
                                                 "detector flat frame");
                }
            }
            else {
                /* Produce a dummy HDU */
                diff_img = NULL;
            } 

            if(qmost_dfs_save_image_extension(diffimg_detflat_frame,
                                              detflat_hdr,
                                              arm_extname,
                                              qclist,
                                              diff_img,
                                              CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save detector flat "
                                             "difference image %s[%s]",
                                             diffimg_detflat_filename,
                                             arm_extname);
            }

            cpl_image_delete(diff_img);
            diff_img = NULL;

            cpl_image_delete(reference_detflat_img);
            reference_detflat_img = NULL;
        }

        /* Save gain corrected, unfiltered flat */
        if(qmost_dfs_save_image_and_var(unfiltered_detflat_frame,
                                        detflat_hdr,
                                        arm_extname,
                                        qclist,
                                        filtered_img,
                                        detflat_var,
                                        CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save unfiltered "
                                         "detector flat frame %s[%s]",
                                         unfiltered_detflat_filename,
                                         arm_extname);
        }

        /* Filter gain corrected flat to determine illumination
         * pattern.  This is done column first for the median filter
         * to reduce ringing around bad columns. */
        if(qmost_bfilt2(filtered_img,
                        medfil,
                        QMOST_MEDIANCALC,
                        2) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't median filter "
                                         "detector flat frame "
                                         "by kernel %d pixels",
                                         medfil);
        }

        if(qmost_bfilt2(filtered_img,
                        linfil,
                        QMOST_MEANCALC,
                        1) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't median filter "
                                         "detector flat frame "
                                         "by kernel %d pixels",
                                         linfil);
        }

        /* Divide out to remove illumination pattern.  This also
         * normalises the flat to unity. */
        cpl_image_divide(detflat_img, filtered_img);
        cpl_image_divide(detflat_var, filtered_img);
        cpl_image_divide(detflat_var, filtered_img);

        cpl_image_delete(filtered_img);
        filtered_img = NULL;

        /* Save result */
        if(qmost_dfs_save_image_and_var(master_detflat_frame,
                                        detflat_hdr,
                                        arm_extname,
                                        qclist,
                                        detflat_img,
                                        detflat_var,
                                        CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save master "
                                         "detector flat frame %s[%s]",
                                         master_detflat_filename,
                                         arm_extname);
        }

        /* Clean up */
        cpl_propertylist_delete(detflat_hdr);
        detflat_hdr = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;

        cpl_image_delete(detflat_img);
        detflat_img = NULL;

        cpl_image_delete(detflat_var);
        detflat_var = NULL;

        cpl_msg_indent_less();
    }

    /* Clean up */
    cpl_frameset_delete(detflats_raw);
    detflats_raw = NULL;

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_detector_flat_analyse_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_parameter *par;

    /* imcombine parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.combtype",
                                  CPL_TYPE_INT,
                                  "Stacking method when combining multiple "
                                  "images.  1: mean, 2: median.",
                                  RECIPE_NAME,
                                  QMOST_MEANCALC);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.combtype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.scaletype",
                                  CPL_TYPE_INT,
                                  "Scaling to use when combining multiple "
                                  "images.  0: no biasing or scaling, "
                                  "1: bias by relative offset in background "
                                  "levels, 2: scale by ratio of background "
                                  "levels, 3: scale by exposure time and "
                                  "bias by relative offset in background.",
                                  RECIPE_NAME,
                                  2);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.scaletype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.xrej",
                                  CPL_TYPE_BOOL,
                                  "Do extra rejection cycle when combining "
                                  "images.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.xrej");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold in terms of "
                                  "background noise for rejection when "
                                  "combining images.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Gain correction parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".gaincor.window",
                                  CPL_TYPE_INT,
                                  "Measurement window size of the "
                                  "strip at the edge of the amplifiers "
                                  "to use for gain correction.",
                                  RECIPE_NAME,
                                  16);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "gaincor.window");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Detector flat parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".detflat.medfil",
                                  CPL_TYPE_INT,
                                  "2D median filter window size used "
                                  "to remove the LED illumination "
                                  "pattern.",
                                  RECIPE_NAME,
                                  25);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "detflat.medfil");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".detflat.linfil",
                                  CPL_TYPE_INT,
                                  "2D boxcar filter window size used "
                                  "to remove the LED illumination "
                                  "pattern.",
                                  RECIPE_NAME,
                                  5);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "detflat.linfil");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* diffimg parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".diffimg.ncells",
                                  CPL_TYPE_INT,
                                  "Number of smoothing cells to use "
                                  "for difference image statistics.",
                                  RECIPE_NAME,
                                  256);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "diffimg.ncells");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* reference parameter */
    par = cpl_parameter_new_value(RECIPE_NAME".reference",
                                  CPL_TYPE_BOOL,
                                  "Emit reference rather than master frame.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "reference");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    return CPL_ERROR_NONE;
}

/**@}*/
