/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_extract_psf.h"
#include "qmost_extract_tram.h"
#include "qmost_ffnorm_fib.h"
#include "qmost_fibtab.h"
#include "qmost_dfs.h"
#include "qmost_pfits.h"
#include "qmost_rebin_spectra.h"
#include "qmost_scattered.h"
#include "qmost_spec_combine.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_fibre_flat_analyse"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                 Typedefs: Structs and enum types
 */
/*----------------------------------------------------------------------------*/

struct qmost_fibre_flat_analyse_calibs {
    const cpl_frame *master_bias_frame;
    const cpl_frame *master_dark_frame;
    const cpl_frame *master_detflat_frame;
    const cpl_frame *master_bpm_frame;
    const cpl_frame *master_lintab_frame;
    const cpl_frame *fibre_trace_frame;
    const cpl_frame *fibre_mask_frame;
    const cpl_frame *fibre_psf_frame;
    const cpl_frame *wave_frame;
    const cpl_frame *master_fibre_flat_frame;
};

struct qmost_fibre_flat_analyse_params {
    int level;
    int keep;
    const char *swapx_table;
    int combtype;
    int scaletype;
    int xrej;
    double combthresh;
    int scattered_enable;
    int scattered_nbsize;
    int extract_crosstalk;
    float extract_crrej_thr;
    int extract_niter;
    int extract_width;
    int skyflat_underexp;
    int skyflat_overexp;
    int skyflat_combthresh;
    int fftrack;
    int ffsmooth;
    int ffrescale;
};

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fibre_flat_analyse_extract_one(
    cpl_image *fibre_flat_img,
    cpl_image *fibre_flat_var,
    cpl_propertylist *fibre_flat_hdr,
    cpl_propertylist *qclist,
    int arm,
    struct qmost_fibre_flat_analyse_params params,
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    cpl_mask *fibre_mask,
    cpl_imagelist *fibre_psf_img,
    cpl_imagelist *fibre_psf_var,
    cpl_propertylist *fibre_psf_hdr,
    const cpl_frame *proc_fibre_flat_frame,
    const cpl_frame *crmask_frame,
    cpl_image **extracted_fibre_flat_img,
    cpl_image **extracted_fibre_flat_var);

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_fibre_flat_analyse_description[] =
    "Combine individual fibre flat images into master, OB or sky\n"
    "fibre flat.  Extract the fibre flat spectra and normalise them\n"
    "so the ensemble average of each spectrum is unity.\n\n"
    "This recipe can be used for processing master fibre flats, \n"
    "OB-level fibre flats, or twilight fibre flats.  The type of raw\n"
    "input determines the type of output produced and tagged per the\n"
    "tables below.\n"
    "\n"
    "The following files can be specified in the SOF:\n\n"
    "Description                 Req/Opt? Tag\n"
    "--------------------------- -------- --------------------\n"
    "Daytime fibre flat frames   Required " QMOST_RAW_FIBRE_FLAT_DAY "\n"
    "or\n"
    "Nighttime fibre flat frames Required " QMOST_RAW_FIBRE_FLAT_NIGHT "\n"
    "or\n"
    "Twilight fibre flat frames  Required " QMOST_RAW_FIBRE_FLAT_SKY "\n\n"
    "Master bias frame           Optional " QMOST_PRO_MASTER_BIAS "\n"
    "Master dark frame           Optional " QMOST_PRO_MASTER_DARK "\n"
    "Master detector flat        Optional " QMOST_PRO_MASTER_DETECTOR_FLAT "\n"
    "Master bad pixel mask       Optional " QMOST_CALIB_MASTER_BPM "\n"
    "Master linearity table      Optional " QMOST_PRO_LINEARITY "\n"
    "Master fibre trace table    Required " QMOST_PRO_FIBRE_TRACE "\n"
    "Master fibre mask           Required " QMOST_PRO_FIBRE_MASK "\n"
    "Master PSF                  Req/Opt  " QMOST_PRO_MASTER_PSF "\n"
    "Master wavelength solution  Required " QMOST_PRO_MASTER_WAVE "\n"
    "Master fibre flat           Req/Opt  " QMOST_PRO_MASTER_FIBRE_FLAT "\n"
    "\n"
    "Note: Master PSF is only required for QC1 (--level=1).  Master fibre\n"
    "flat is required for processing OB and twilight flats.\n"
    "\n"
    "Outputs:\n\n"
    "Description                 Tag\n"
    "--------------------------- -----------------\n"
    "Master fibre flat           " QMOST_PRO_MASTER_FIBRE_FLAT "\n"
    "or\n"
    "OB-level fibre flat         " QMOST_PRO_OB_FIBRE_FLAT "\n"
    "or\n"
    "Twilight fibre flat         " QMOST_PRO_SKY_FIBRE_FLAT "\n"
    "\n"
    "Optional processed outputs if keep=true:\n\n"
    "Description                 Tag\n"
    "--------------------------- -----------------\n"
    "Processed fibre flat image  " QMOST_PROC_FIBRE_FLAT "\n"
    "Cosmic ray mask             " QMOST_PRO_CRMASK "\n"
    "Extracted fibre flat        " QMOST_PRO_EXTRACTED_FIBRE_FLAT "\n"
    "\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the OB-level fibre flat\n"
    "for LRS-B would be QMOST_" QMOST_PRO_OB_FIBRE_FLAT "_LRS-B.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_fibre_flat_analyse,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Extract and normalise master or OB fibre flat.",
                  qmost_fibre_flat_analyse_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_fibre_flat_analyse    qmost_fibre_flat_analyse
 *
 * @brief Combine individual fibre flat images into master, OB or sky
 *        fibre flat.
 *
 * @par Name:
 *   qmost_fibre_flat_analyse
 * @par Purpose:
 *   Individual fibre flat images are processed, combined, spectrally
 *   extracted, and wavelength calibrated to produce 1D fibre flat
 *   spectra.  The fibre flat spectra are then processed to divide out
 *   the spectral response of the flat lamp, smooth the resulting
 *   flat, and normalise the fibres to correct their relative
 *   throughputs to a common system.
 *   Three types of fibre flats can be processed: internal daytime
 *   fibre flats are used to produce a master fibre flat, night-time
 *   OB-level fibre flats are used to correct for changes in the
 *   relative fibre throughputs caused by tilt of the spines used to
 *   position the fibres, and twilight ("sky") fibre flats are used to
 *   define the "correct" relative throughputs of the fibres.  The
 *   processing done is slightly different for internal fibre flats
 *   compared to twilight flats.  Internal fibre flats should all have
 *   the same exposure level, and are stacked at the 2D image level
 *   prior to spectral extraction.  Twilight fibre flats have variable
 *   exposure levels due to change in twilight sky brightness and are
 *   extracted first, checked for under or over exposure, and then the
 *   good fibre flats are stacked in 1D.  This means some recipe
 *   parameters and processing steps apply to only one type of flat,
 *   as indicated below.
 * @par Type:
 *   Spectroscopic calibration
 * @par Parameters:
 *   - @b level (int, default 1):
 *     Determines the level of the analysis:
 *       - (0): QC0 analysis.
 *       - (1): QC1 analysis.
 *   - @b keep (bool, default false):
 *     If this flag is set, optional products generated during
 *     execution of the recipe are also saved.  In the case of
 *     qmost_fibre_flat_analyse, this saves the processed fibre flat
 *     image, cosmic ray mask, and the extracted, unfiltered fibre
 *     flat spectra.
 *   - @b ccdproc.swapx (string, default 100010101):
 *     String of 0 or 1 specifying for each arm of each spectrograph
 *     if we should flip the x axis to correct the wavelength order of
 *     the spectral axis.
 *   - @b imcombine.combtype (int, default 1):
 *     Determines the type of combination that is done to form the
 *     stacked fibre flat frame when stacking internal facility fibre
 *     flats (daytime or night-time).  Can take the following values:
 *       - (1): The output pixels are kappa sigma clipped means of
 *              the input pixels.
 *       - (2): The output pixels are medians of the input pixels.
 *   - @b imcombine.scaletype (int, default 2):
 *     Scaling to use when combining multiple images to form the
 *     stacked fibre flat frame when stacking internal facility fibre
 *     flats (daytime or night-time).  Can take the following values:
 *       - (0): No biasing or scaling is done.
 *       - (1): An additive offset is applied to remove any
 *              difference in background levels.
 *       - (2): A multiplicative scaling is applied to remove any
 *              difference in background levels.
 *       - (3): Images are scaled by their relative exposure times
 *              and an additive offset is applied to remove any
 *              remaining difference in background levels.
 *   - @b imcombine.xrej (bool, default false):
 *     If set, then an extra rejection cycle will be run (internal
 *     fibre flats only).
 *   - @b imcombine.thresh (float, default 5.0):
 *     The rejection threshold for clipping during the combination in
 *     units of the background sigma (internalfibre flats only).
 *   - @b scattered.enable (bool, default false):
 *     Enable scattered light removal.
 *   - @b scattered.nbsize (int, default 256):
 *     Size of the background smoothing cells for determining
 *     scattered light in pixels.
 *   - @b extract.crosstalk (bool, default true):
 *     Correct for crosstalk between fibres using the overlap of the
 *     fibre PSFs if true.  Can be set to false to disable crosstalk
 *     correction.
 *   - @b extract.cr_thresh (float, default 10.0):
 *     Rejection threshold as number of sigma used to define cosmic
 *     ray hits during PSF extraction.
 *   - @b extract.niter (int, default 1):
 *     Number of rejection iterations for PSF spectral extraction.
 *   - @b extract.width (int, default 6):
 *     Boxcar width in pixels over which the summation will be done
 *     at each spectral pixel during extraction (QC0 only).
 *   - @b skyflat.underexp (float, default 100):
 *     If the average extracted counts (in ADU) in the twilight flat
 *     spectrum are below this threshold, then the image is rejected
 *     as underexposed.
 *   - @b skyflat.overexp (float, default 120000):
 *     If the average extracted counts (in ADU) in the twilight flat
 *     spectrum are above this threshold, then the image is rejected
 *     as overexposed.
 *   - @b skyflat.rej_thresh (float, default 5.0):
 *     Threshold for outlier rejection when combining extracted
 *     twilight flat spectra.
 *   - @b ffnorm.track (bool, default true):
 *     If set, then a median response will be calculated from all of
 *     the good fibre flat spectra and removed.
 *   - @b ffnorm.smooth (int, default 51):
 *     Size of fibre flat smoothing box in pixels in the spectral
 *     direction, or 0 for no smoothing.
 *   - @b ffnorm.rescale (bool, default true)
 *     If set, then the fibres will be rescaled by their relative
 *     throughputs. If not, then all fibres will be individually
 *     scaled to a median of unity.
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - Exactly one of the following raw input types (required).
 *      Multiple frames of the same type may be given and will be
 *      combined prior to analysis, but different types cannot be
 *      mixed.
 *      - @b FIBRE_FLAT_DAY: Raw daytime internal facility fibre
 *           flat.
 *      - @b FIBRE_FLAT_NIGHT: Raw night-time attached internal
 *           facility fibre flat.
 *      - @b FIBRE_FLAT_SKY: Raw twilight fibre flat.
 *    - @b MASTER_BIAS (optional): Master bias frame.
 *    - @b MASTER_DARK (optional): Master dark frame.
 *    - @b MASTER_DETECTOR_FLAT (optional): Master detector flat.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 *    - @b LINEARITY (optional): Master linearity table.
 *    - @b FIBRE_TRACE (required): Trace table.
 *    - @b FIBRE_MASK (required): Fibre mask.
 *    - @b MASTER_WAVE (required): Master wavelength solution.
 *    - @b MASTER_FIBRE_FLAT: Master fibre flat.  Required when
 *         processing FIBRE_FLAT_NIGHT or FIBRE_FLAT_SKY.  Otherwise
 *         should not be specified.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b MASTER_FIBRE_FLAT: The master fibre flat.  This
 *          product is emitted when processing FIBRE_FLAT_DAY input
 *          files.
 *     - @b OB_FIBRE_FLAT: The OB-level wavelength solution correction
 *          table.  This product is emitted when processing
 *          FIBRE_FLAT_NIGHT input files.
 *     - @b SKY_FIBRE_FLAT: The simuarc wavelength solution.  This
 *          product is emitted when processing FIBRE_FLAT_SKY input
 *          files.
 *
 *   - The following intermediate product files are optional and are
 *     only emitted if keep=true is specified in the recipe parameters.
 *     - @b PROC_FIBRE_FLAT: Processed 2D fibre flat image (prior to
 *          spectral extraction).  Can be used to assess the quality
 *          of the initial CCD processing, such as bias, dark,
 *          detector flat field correction, scattered light
 *          correction.
 *     - @b COSMIC_RAY_MASK: An optional mask showing which pixels of
 *          the processed 2D fibre flat image in the previous output
 *          were flagged as cosmic rays during PSF extraction (QC1
 *          only).  This is an integer image using the enumerated
 *          constants QMOST_WMASK_*, the values of which can be found
 *          in the documentation for qmost_constants.
 *     - @b EXTRACTED_FIBRE_FLAT: The extracted fibre flat spectra
 *          prior to filtering and normalisation.
 *
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     OB-level fibre flat for LRS-B would be @c
 *     QMOST_OB_FIBRE_FLAT_LRS-B.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean residual background level
 *     in the illuminated region of the overscan subtracted frames
 *     that were combined.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined, as a measure of how
 *     consistent the background levels were.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>EXT FLUX MED</b> (ADU): The median of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX RMS</b> (ADU): The robustly-estimated RMS of
 *     average extracted flux in each fibre.
 *   - <b>EXT FLUX MIN</b> (ADU): The minimum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MINSPC</b>: The fibre with the minimum extracted
 *     flux.
 *   - <b>EXT FLUX MAX</b> (ADU): The maximum of the average extracted
 *     flux in each fibre.
 *   - <b>EXT FLUX MAXSPC</b>: The fibre with the maximum extracted
 *     flux.
 *   - <b>EXT SN MED</b>: The median of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN RMS</b>: The robustly-estimated RMS of the average
 *     signal to noise ratio in each fibre.
 *   - <b>EXT SN MIN</b>: The minimum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MINSPC</b>: The fibre with the minimum signal to
 *     noise ratio.
 *   - <b>EXT SN MAX</b>: The maximum of the average signal to noise
 *     ratio in each fibre.
 *   - <b>EXT SN MAXSPC</b>: The fibre with the maximum signal to
 *     noise ratio.
 *   - <b>EXT GOF MED</b>: The median goodness of fit of the profile
 *     in PSF extraction.
 *   - <b>EXT NUM REJECTED</b>: The number of rejected pixels during
 *     PSF extraction.
 *   - <b>SCATTL MED</b> (ADU): The median scattered light level on
 *     the detector outside the region illuminated by the fibres.
 *   - <b>SCATTL RMS</b> (ADU): The robustly-estimated RMS of the
 *     scattered light level on the detector outside the region
 *     illuminated by the fibres.
 *   - <b>SCATTL RESID MED</b> (ADU): The median residual scattered
 *     light background remaining after subtraction.
 *   - <b>SCATTL RESID RMS</b> (ADU): The RMS residual of the
 *     scattered light background remaining after subtraction.
 *   - <b>FIBFLAT ENS MED</b> (ADU): The median of the average fluxes
 *     of all of the individual fibres' flats before
 *     normalisation. Used to assess the relative throughput of 
 *     the fibres.
 *   - <b>FIBFLAT ENS RMS</b> (ADU): The robustly-estimated RMS of the
 *     average fluxes of all of the individual fibres' flats before
 *     normalisation. Used to assess consistency of the throughput of
 *     the fibres.
 *   - <b>FIBFLAT ENS MIN</b> (ADU): The minimum of the average fluxes
 *     of all of the individual fibres' flats before normalisation.
 *   - <b>FIBFLAT ENS MINFIB</b>: The spectrum with the minimum
 *     average flux.
 *   - <b>FIBFLAT ENS MINSPC</b>: The fibre ID of the fibre with the
 *     minimum average flux.
 *   - <b>FIBFLAT ENS MAX</b> (ADU): The maximum of the average fluxes
 *     of all of the individual fibres' flats before normalisation.
 *   - <b>FIBFLAT ENS MAXFIB</b>: The spectrum with the maximum
 *     average flux.
 *   - <b>FIBFLAT ENS MAXSPC</b>: The fibre ID of the fibre with the
 *     maximum average flux.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - No raw fibre flat frames in the input frameset.
 *   - Mandatory calibration images/tables not specified in SOF or
 *     unreadable.
 *   - When processing FIBRE_FLAT_NIGHT or FIBRE_FLAT_SKY to make an
 *     OB-level or twilight fibre flat, if the required master fibre
 *     flat isn't available.
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - When processing twilight flats, if no flats were within the
 *     under and over exposure limits.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 *   - If all fibres are flagged bad (FIB_ST=0).
 *   - If the trace table for an active detector is empty.
 * @par Non-Fatal Error Conditions:
 *   - No bad pixel mask (all pixels assumed to be good).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 * @par Functional Diagram for FIBRE_FLAT_DAY to MASTER_FIBRE_FLAT:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 * 
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc_and_combine [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> "qmost_extract_psf\nqmost_extract_tram";
 *   MASTER_PSF -> "qmost_extract_psf\nqmost_extract_tram" [style="dashed"];
 *   MASTER_WAVE -> qmost_rebin_spectra_off;
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     FIBRE_FLAT_DAY -> qmost_ccdproc_and_combine;
 *
 *     qmost_ccdproc_and_combine -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> "qmost_extract_psf\nqmost_extract_tram";
 *     "qmost_extract_psf\nqmost_extract_tram" -> qmost_extract_qc;
 *     qmost_extract_qc -> qmost_rebin_spectra_off;
 *     qmost_rebin_spectra_off -> qmost_ffnorm_fib;
 *
 *     qmost_ffnorm_fib -> MASTER_FIBRE_FLAT;
 *   }
 *
 *   qmost_scattered -> PROC_FIBRE_FLAT;
 *   qmost_extract_qc -> EXTRACTED_FIBRE_FLAT;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   qmost_extract_qc -> QC1;
 *   qmost_ffnorm_fib -> QC1;
 * 
 *   FIBRE_FLAT_DAY [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   MASTER_PSF [shape="box" fillcolor="#fff5ce"]
 *   MASTER_WAVE [shape="box" fillcolor="#fff5ce"]
 *   MASTER_FIBRE_FLAT [shape="box" fillcolor="#fff5ce"]
 *   PROC_FIBRE_FLAT [shape="box" fillcolor="#ffffff"]
 *   EXTRACTED_FIBRE_FLAT [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 * @par Functional Diagram for FIBRE_FLAT_NIGHT to OB_FIBRE_FLAT:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 * 
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc_and_combine [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> "qmost_extract_psf\nqmost_extract_tram";
 *   MASTER_PSF -> "qmost_extract_psf\nqmost_extract_tram" [style="dashed"];
 *   MASTER_WAVE -> qmost_rebin_spectra_off;
 *   MASTER_FIBRE_FLAT -> qmost_ffdiv_fib;
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     FIBRE_FLAT_NIGHT -> qmost_ccdproc_and_combine;
 *
 *     qmost_ccdproc_and_combine -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> "qmost_extract_psf\nqmost_extract_tram";
 *     "qmost_extract_psf\nqmost_extract_tram" -> qmost_extract_qc;
 *     qmost_extract_qc -> qmost_rebin_spectra_off;
 *     qmost_rebin_spectra_off -> qmost_ffdiv_fib;
 *     qmost_ffdiv_fib -> qmost_ffnorm_fib;
 *
 *     qmost_ffnorm_fib -> OB_FIBRE_FLAT;
 *   }
 *
 *   qmost_scattered -> PROC_FIBRE_FLAT;
 *   qmost_extract_qc -> EXTRACTED_FIBRE_FLAT;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   qmost_extract_qc -> QC1;
 *   qmost_ffnorm_fib -> QC1;
 * 
 *   FIBRE_FLAT_NIGHT [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   MASTER_PSF [shape="box" fillcolor="#fff5ce"]
 *   MASTER_WAVE [shape="box" fillcolor="#fff5ce"]
 *   MASTER_FIBRE_FLAT [shape="box" fillcolor="#fff5ce"]
 *   OB_FIBRE_FLAT [shape="box" fillcolor="#fff5ce"]
 *   PROC_FIBRE_FLAT [shape="box" fillcolor="#ffffff"]
 *   EXTRACTED_FIBRE_FLAT [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 * @par Functional Diagram for FIBRE_FLAT_SKY to SKY_FIBRE_FLAT:
 * @dot
 * digraph {
 *   edge [fontname="monospace" fontsize=8]
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 * 
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> "qmost_extract_psf\nqmost_extract_tram";
 *   MASTER_PSF -> "qmost_extract_psf\nqmost_extract_tram" [style="dashed"];
 *   MASTER_WAVE -> qmost_rebin_spectra_off;
 *   MASTER_FIBRE_FLAT -> qmost_ffdiv_fib;
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     FIBRE_FLAT_SKY -> qmost_ccdproc;
 *
 *     qmost_ccdproc -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> "qmost_extract_psf\nqmost_extract_tram";
 *     "qmost_extract_psf\nqmost_extract_tram" -> qmost_extract_qc;
 *     qmost_extract_qc -> qmost_spec_combine [label=" x nflats"];
 *     qmost_spec_combine -> qmost_rebin_spectra_off;
 *     qmost_rebin_spectra_off -> qmost_ffdiv_fib;
 *     qmost_ffdiv_fib -> qmost_ffnorm_fib;
 *
 *     qmost_ffnorm_fib -> SKY_FIBRE_FLAT;
 *   }
 *
 *   qmost_scattered -> PROC_FIBRE_FLAT;
 *   qmost_extract_qc -> EXTRACTED_FIBRE_FLAT;
 *
 *   qmost_ccdproc -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   qmost_extract_qc -> QC1;
 *   qmost_ffnorm_fib -> QC1;
 * 
 *   FIBRE_FLAT_SKY [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   MASTER_PSF [shape="box" fillcolor="#fff5ce"]
 *   MASTER_WAVE [shape="box" fillcolor="#fff5ce"]
 *   MASTER_FIBRE_FLAT [shape="box" fillcolor="#fff5ce"]
 *   SKY_FIBRE_FLAT [shape="box" fillcolor="#fff5ce"]
 *   PROC_FIBRE_FLAT [shape="box" fillcolor="#ffffff"]
 *   EXTRACTED_FIBRE_FLAT [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_fibre_flat_analyse(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    struct qmost_fibre_flat_analyse_calibs calibs = {
        .master_bias_frame = NULL,
        .master_dark_frame = NULL,
        .master_detflat_frame = NULL,
        .master_bpm_frame = NULL,
        .master_lintab_frame = NULL,
        .fibre_trace_frame = NULL,
        .fibre_mask_frame = NULL,
        .fibre_psf_frame = NULL,
        .wave_frame = NULL,
        .master_fibre_flat_frame = NULL
    };

    const cpl_frame *proc_fibre_flat_frame = NULL;
    const char *proc_fibre_flat_filename = NULL;
    const cpl_frame *crmask_frame = NULL;
    const char *crmask_filename = NULL;
    const cpl_frame *extracted_fibre_flat_frame = NULL;
    const char *extracted_fibre_flat_filename = NULL;
    const cpl_frame *ffnorm_frame = NULL;
    const char *ffnorm_filename = NULL;
    char *ffnorm_tag = NULL;

    cpl_frameset *fibflats_raw = NULL;
    int nin, iin, nff;
    int nday, nnight, nsky;

    const cpl_frame *this_frame;
    const char *this_tag;

    cpl_frame *fibflat_ref;
    const char *fibflat_ref_filename;

    const cpl_parameter *par;
    struct qmost_fibre_flat_analyse_params params = {
        .level = 1,
        .keep = 0,
        .swapx_table = QMOST_DEFAULT_SWAPX_TABLE,
        .combtype = QMOST_MEANCALC,
        .scaletype = 2,
        .xrej = 0,
        .combthresh = 5.0,
        .scattered_enable = 0,
        .scattered_nbsize = 256,
        .extract_crosstalk = 1,
        .extract_crrej_thr = 10.0,
        .extract_niter = 1,
        .extract_width = 6,
        .skyflat_underexp = 100.0,
        .skyflat_overexp = 120000.0,
        .skyflat_combthresh = 5.0,
        .fftrack = 1,
        .ffsmooth = 51,
        .ffrescale = 1
    };

    cpl_size raw_extension;
    int spec, arm, detlive;
    const char *arm_extname;
    char *extname = NULL;

    cpl_mask *master_bpm = NULL;
    cpl_image *master_bias_img = NULL;
    cpl_image *master_bias_var = NULL;
    cpl_image *master_dark_img = NULL;
    cpl_image *master_dark_var = NULL;
    cpl_image *master_detflat_img = NULL;
    cpl_image *master_detflat_var = NULL;
    cpl_table *master_lintab = NULL;

    cpl_image *fibre_flat_img = NULL;
    cpl_image *fibre_flat_var = NULL;
    cpl_propertylist *fibre_flat_hdr = NULL;

    cpl_propertylist *qclist = NULL;
    cpl_propertylist *qctmp = NULL;
    cpl_propertylist *qcuse;

    cpl_table *fibinfo_tbl = NULL;
    cpl_propertylist *fibinfo_hdr = NULL;

    cpl_table *trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;
    cpl_mask *fibre_mask = NULL;
    cpl_imagelist *fibre_psf_img = NULL;
    cpl_imagelist *fibre_psf_var = NULL;
    cpl_propertylist *fibre_psf_hdr = NULL;
    cpl_table *wave_tbl = NULL;
    cpl_propertylist *wave_hdr = NULL;

    cpl_image *extracted_fibre_flat_img = NULL;
    cpl_image *extracted_fibre_flat_var = NULL;

    cpl_imagelist *ff_imglist = NULL;
    cpl_imagelist *ff_varlist = NULL;

    int ifile, nfiles, ngood;
    double medcts;

    int nfib;
    double *waveoff = NULL;
    double *veloff = NULL;
    double minwave, maxwave, dlam;

    cpl_image *wff_img = NULL;
    cpl_image *wff_var = NULL;
    cpl_propertylist *wff_hdr = NULL;

    cpl_image *master_ffnorm_img = NULL;
    cpl_image *master_ffnorm_var = NULL;

#undef TIDY
#define TIDY                                            \
    if(fibflats_raw) {                                  \
        cpl_frameset_delete(fibflats_raw);              \
        fibflats_raw = NULL;                            \
    }                                                   \
    if(fibinfo_tbl) {                                   \
        cpl_table_delete(fibinfo_tbl);                  \
        fibinfo_tbl = NULL;                             \
    }                                                   \
    if(fibinfo_hdr) {                                   \
        cpl_propertylist_delete(fibinfo_hdr);           \
        fibinfo_hdr = NULL;                             \
    }                                                   \
    if(master_bpm) {                                    \
        cpl_mask_delete(master_bpm);                    \
        master_bpm = NULL;                              \
    }                                                   \
    if(master_bias_img) {                               \
        cpl_image_delete(master_bias_img);              \
        master_bias_img = NULL;                         \
    }                                                   \
    if(master_bias_var) {                               \
        cpl_image_delete(master_bias_var);              \
        master_bias_var = NULL;                         \
    }                                                   \
    if(master_dark_img) {                               \
        cpl_image_delete(master_dark_img);              \
        master_dark_img = NULL;                         \
    }                                                   \
    if(master_dark_var) {                               \
        cpl_image_delete(master_dark_var);              \
        master_dark_var = NULL;                         \
    }                                                   \
    if(master_detflat_img) {                            \
        cpl_image_delete(master_detflat_img);           \
        master_detflat_img = NULL;                      \
    }                                                   \
    if(master_detflat_var) {                            \
        cpl_image_delete(master_detflat_var);           \
        master_detflat_var = NULL;                      \
    }                                                   \
    if(master_lintab) {                                 \
        cpl_table_delete(master_lintab);                \
        master_lintab = NULL;                           \
    }                                                   \
    if(fibre_flat_img) {                                \
        cpl_image_delete(fibre_flat_img);               \
        fibre_flat_img = NULL;                          \
    }                                                   \
    if(fibre_flat_var) {                                \
        cpl_image_delete(fibre_flat_var);               \
        fibre_flat_var = NULL;                          \
    }                                                   \
    if(fibre_flat_hdr) {                                \
        cpl_propertylist_delete(fibre_flat_hdr);        \
        fibre_flat_hdr = NULL;                          \
    }                                                   \
    if(qclist) {                                        \
        cpl_propertylist_delete(qclist);                \
        qclist = NULL;                                  \
    }                                                   \
    if(qctmp) {                                         \
        cpl_propertylist_delete(qctmp);                 \
        qctmp = NULL;                                   \
    }                                                   \
    if(trace_tbl) {                                     \
        cpl_table_delete(trace_tbl);                    \
        trace_tbl = NULL;                               \
    }                                                   \
    if(trace_hdr) {                                     \
        cpl_propertylist_delete(trace_hdr);             \
        trace_hdr = NULL;                               \
    }                                                   \
    if(fibre_mask) {                                    \
        cpl_mask_delete(fibre_mask);                    \
        fibre_mask = NULL;                              \
    }                                                   \
    if(fibre_psf_img) {                                 \
        cpl_imagelist_delete(fibre_psf_img);            \
        fibre_psf_img = NULL;                           \
    }                                                   \
    if(fibre_psf_var) {                                 \
        cpl_imagelist_delete(fibre_psf_var);            \
        fibre_psf_var = NULL;                           \
    }                                                   \
    if(fibre_psf_hdr) {                                 \
        cpl_propertylist_delete(fibre_psf_hdr);         \
        fibre_psf_hdr = NULL;                           \
    }                                                   \
    if(wave_tbl) {                                      \
        cpl_table_delete(wave_tbl);                     \
        wave_tbl = NULL;                                \
    }                                                   \
    if(wave_hdr) {                                      \
        cpl_propertylist_delete(wave_hdr);              \
        wave_hdr = NULL;                                \
    }                                                   \
    if(extracted_fibre_flat_img) {                      \
        cpl_image_delete(extracted_fibre_flat_img);     \
        extracted_fibre_flat_img = NULL;                \
    }                                                   \
    if(extracted_fibre_flat_var) {                      \
        cpl_image_delete(extracted_fibre_flat_var);     \
        extracted_fibre_flat_var = NULL;                \
    }                                                   \
    if(ff_imglist) {                                    \
        cpl_imagelist_delete(ff_imglist);               \
        ff_imglist = NULL;                              \
    }                                                   \
    if(ff_varlist) {                                    \
        cpl_imagelist_delete(ff_varlist);               \
        ff_varlist = NULL;                              \
    }                                                   \
    if(extname) {                                       \
        cpl_free(extname);                              \
        extname = NULL;                                 \
    }                                                   \
    if(waveoff) {                                       \
        cpl_free(waveoff);                              \
        waveoff = NULL;                                 \
    }                                                   \
    if(veloff) {                                        \
        cpl_free(veloff);                               \
        veloff = NULL;                                  \
    }                                                   \
    if(extname) {                                       \
        cpl_free(extname);                              \
        extname = NULL;                                 \
    }                                                   \
    if(wff_img) {                                       \
        cpl_image_delete(wff_img);                      \
        wff_img = NULL;                                 \
    }                                                   \
    if(wff_var) {                                       \
        cpl_image_delete(wff_var);                      \
        wff_var = NULL;                                 \
    }                                                   \
    if(wff_hdr) {                                       \
        cpl_propertylist_delete(wff_hdr);               \
        wff_hdr = NULL;                                 \
    }                                                   \
    if(master_ffnorm_img) {                             \
        cpl_image_delete(master_ffnorm_img);            \
        master_ffnorm_img = NULL;                       \
    }                                                   \
    if(master_ffnorm_var) {                             \
        cpl_image_delete(master_ffnorm_var);            \
        master_ffnorm_var = NULL;                       \
    }

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Master detector-level calibrations, all optional */
    calibs.master_bias_frame = cpl_frameset_find_const(frameset,
                                                       QMOST_PRO_MASTER_BIAS);
    calibs.master_dark_frame = cpl_frameset_find_const(frameset,
                                                       QMOST_PRO_MASTER_DARK);
    calibs.master_detflat_frame = cpl_frameset_find_const(
        frameset,
        QMOST_PRO_MASTER_DETECTOR_FLAT);
    calibs.master_bpm_frame = cpl_frameset_find_const(frameset,
                                                      QMOST_CALIB_MASTER_BPM);
    calibs.master_lintab_frame = cpl_frameset_find_const(frameset,
                                                         QMOST_PRO_LINEARITY);

    /* Trace table and fibre mask, required */
    calibs.fibre_trace_frame = cpl_frameset_find_const(frameset,
                                                       QMOST_PRO_FIBRE_TRACE);
    if(calibs.fibre_trace_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_TRACE);
    }

    calibs.fibre_mask_frame = cpl_frameset_find_const(frameset,
                                                      QMOST_PRO_FIBRE_MASK);
    if(calibs.fibre_mask_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_MASK);
    }

    /* Master PSF, required for PSF extraction but not otherwise so
     * check later once we know if we're using PSF extraction. */
    calibs.fibre_psf_frame = cpl_frameset_find_const(frameset,
                                                     QMOST_PRO_MASTER_PSF);

    /* Wavelength solution, required */
    calibs.wave_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_WAVE);
    if(calibs.wave_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_MASTER_WAVE);
    }

    /* Master fibre flat frame, optional, will be given if processing
     * OB-level or twilight fibre flats. */
    calibs.master_fibre_flat_frame =
        cpl_frameset_find_const(frameset,
                                QMOST_PRO_MASTER_FIBRE_FLAT);

    /* Create frameset of input raw fibre flats */
    fibflats_raw = cpl_frameset_new();  /* can't fail? */

    nin = cpl_frameset_get_size(frameset);

    nday = 0;
    nnight = 0;
    nsky = 0;

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_FIBRE_FLAT_DAY)) {
            cpl_frameset_insert(fibflats_raw, cpl_frame_duplicate(this_frame));
            nday++;
        }
        else if(!strcmp(this_tag, QMOST_RAW_FIBRE_FLAT_NIGHT)) {
            cpl_frameset_insert(fibflats_raw, cpl_frame_duplicate(this_frame));
            nnight++;
        }
        else if(!strcmp(this_tag, QMOST_RAW_FIBRE_FLAT_SKY)) {
            cpl_frameset_insert(fibflats_raw, cpl_frame_duplicate(this_frame));
            nsky++;
        }
    }

    /* Check we have some */
    nff = cpl_frameset_get_size(fibflats_raw);
    if(nff < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged "
                                     "%s or %s",
                                     QMOST_RAW_FIBRE_FLAT_DAY,
                                     QMOST_RAW_FIBRE_FLAT_NIGHT);
    }

    /* Decide what to do */
    if(nday > 0 && (nnight == 0 && nsky == 0)) {
        /* Master */
        ffnorm_tag = QMOST_PRO_MASTER_FIBRE_FLAT;
    }
    else if(nnight > 0 && (nday == 0 && nsky == 0)) {
        /* OB */
        if(calibs.master_fibre_flat_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "SOF does not have an image tagged "
                                         "%s",
                                         QMOST_PRO_MASTER_FIBRE_FLAT);
        }

        ffnorm_tag = QMOST_PRO_OB_FIBRE_FLAT;
    }
    else if(nsky > 0 && (nday == 0 && nnight == 0)) {
        /* Twilight */
        if(calibs.master_fibre_flat_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "SOF does not have an image tagged "
                                         "%s",
                                         QMOST_PRO_MASTER_FIBRE_FLAT);
        }

        ffnorm_tag = QMOST_PRO_SKY_FIBRE_FLAT;
    }
    else {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_INCOMPATIBLE_INPUT,
                                     "SOF must contain only one of "
                                     "%s, %s or %s",
                                     QMOST_RAW_FIBRE_FLAT_DAY,
                                     QMOST_RAW_FIBRE_FLAT_NIGHT,
                                     QMOST_RAW_FIBRE_FLAT_SKY);
    }

    /* Retrieve QC level parameter */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".level");
    if(par != NULL)
        params.level = cpl_parameter_get_int(par);

    if(params.level < 0 || params.level > 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_ILLEGAL_INPUT,
                                     "unknown QC level: %d", params.level);
    }

    /* Check PSF if we're using PSF extraction */
    if(params.level != 0 && calibs.fibre_psf_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_MASTER_PSF);
    }

    /* Switch to save processed products */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".keep");
    if(par != NULL)
        params.keep = cpl_parameter_get_bool(par);

    /* Retrieve ccdproc parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".ccdproc.swapx");
    if(par != NULL)
        params.swapx_table = cpl_parameter_get_string(par);

    /* Retrieve imcombine parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.combtype");
    if(par != NULL)
        params.combtype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.scaletype");
    if(par != NULL)
        params.scaletype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.xrej");
    if(par != NULL)
        params.xrej = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.thresh");
    if(par != NULL)
        params.combthresh = cpl_parameter_get_double(par);

    /* Retrieve scattered light removal parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.enable");
    if(par != NULL)
        params.scattered_enable = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.nbsize");
    if(par != NULL)
        params.scattered_nbsize = cpl_parameter_get_int(par);

    /* Retrieve extraction parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.crosstalk");
    if(par != NULL)
        params.extract_crosstalk = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.cr_thresh");
    if(par != NULL)
        params.extract_crrej_thr = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.niter");
    if(par != NULL)
        params.extract_niter = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".extract.width");
    if(par != NULL)
        params.extract_width = cpl_parameter_get_int(par);

    /* Retrieve twilight flat parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".skyflat.underexp");
    if(par != NULL)
        params.skyflat_underexp = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".skyflat.overexp");
    if(par != NULL)
        params.skyflat_overexp = cpl_parameter_get_double(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".skyflat.combthresh");
    if(par != NULL)
        params.skyflat_combthresh = cpl_parameter_get_double(par);

    /* Retrieve ffnorm parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".ffnorm.track");
    if(par != NULL)
        params.fftrack = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".ffnorm.smooth");
    if(par != NULL)
        params.ffsmooth = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".ffnorm.rescale");
    if(par != NULL)
        params.ffrescale = cpl_parameter_get_bool(par);

    /* Get first fibre flat to use as reference */
    fibflat_ref = cpl_frameset_get_position(fibflats_raw, 0);
    if(fibflat_ref == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get first fibre flat "
                                     "from SOF");
    }

    /* Get filename of first fibre flat */
    fibflat_ref_filename = cpl_frame_get_filename(fibflat_ref);
    if(fibflat_ref_filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename for "
                                     "first fibre flat frame");
    }

    /* Extract the primary FITS header of the reference */
    fibre_flat_hdr = cpl_propertylist_load(fibflat_ref_filename,
                                           0);
    if(fibre_flat_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FITS primary header "
                                     "from first input file");
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(fibre_flat_hdr,
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    cpl_propertylist_delete(fibre_flat_hdr);
    fibre_flat_hdr = NULL;

    /* Get fibinfo */
    if(qmost_fibtabload(fibflat_ref,
                        spec,
                        &fibinfo_tbl,
                        &fibinfo_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FIBINFO table "
                                     "from first input file");
    }

    /* Create outputs */
    ffnorm_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        ffnorm_tag,
        CPL_FRAME_TYPE_IMAGE,
        NULL);
    if(ffnorm_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     ffnorm_tag);
    }

    ffnorm_filename = cpl_frame_get_filename(ffnorm_frame);

    if(params.keep) {
        proc_fibre_flat_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PROC_FIBRE_FLAT,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(proc_fibre_flat_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PROC_FIBRE_FLAT);
        }

        proc_fibre_flat_filename = 
            cpl_frame_get_filename(proc_fibre_flat_frame);

        if(params.level) {
            crmask_frame = qmost_dfs_setup_product_default(
                frameset,
                parlist,
                RECIPE_NAME,
                spec,
                QMOST_PRO_CRMASK,
                CPL_FRAME_TYPE_IMAGE,
                NULL);
            if(crmask_frame == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't set up output product "
                                             "file for %s",
                                             QMOST_PRO_CRMASK);
            }

            crmask_filename = cpl_frame_get_filename(crmask_frame);
        }

        extracted_fibre_flat_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PRO_EXTRACTED_FIBRE_FLAT,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(extracted_fibre_flat_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PRO_EXTRACTED_FIBRE_FLAT);
        }
        
        extracted_fibre_flat_filename = 
            cpl_frame_get_filename(extracted_fibre_flat_frame);
    }

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Extract the FITS header of the reference */
        fibre_flat_hdr = cpl_propertylist_load(
            fibflat_ref_filename,
            raw_extension);
        if(fibre_flat_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS extension "
                                         "%lld header from first input file",
                                         raw_extension);
        }

        /* Check the reference to find out what arm we're processing */
        if(qmost_pfits_get_arm(fibre_flat_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine EXTNAME "
                                         "for arm %d", arm);
        }

        /* Is detector live? */
        if(qmost_pfits_get_detlive(fibre_flat_hdr,
                                   &detlive) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read live flag for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(!detlive) {
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %lld = %s",
                         raw_extension, arm_extname);

            if(proc_fibre_flat_frame != NULL) {
                if(qmost_dfs_save_image_and_var(proc_fibre_flat_frame,
                                                fibre_flat_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy fibre "
                                                 "flat frame %s[%s]",
                                                 proc_fibre_flat_filename,
                                                 arm_extname);
                }
            }

            if(crmask_frame != NULL) {
                if(qmost_dfs_save_image_extension(crmask_frame,
                                                  fibre_flat_hdr,
                                                  arm_extname,
                                                  NULL,
                                                  NULL,
                                                  CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "cosmic ray mask %s[%s]",
                                                 crmask_filename,
                                                 arm_extname);
                }
            }

            if(extracted_fibre_flat_frame != NULL) {
                /* Save extracted fibre flat */
                if(qmost_dfs_save_image_and_var(extracted_fibre_flat_frame,
                                                fibre_flat_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy "
                                                 "extracted fibre flat "
                                                 "%s[%s]",
                                                 extracted_fibre_flat_filename,
                                                 arm_extname);
                }
            }

            if(qmost_dfs_save_image_and_var(ffnorm_frame,
                                            fibre_flat_hdr,
                                            arm_extname,
                                            NULL,
                                            NULL,
                                            NULL,
                                            CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy "
                                             "fibre flat %s[%s]",
                                             ffnorm_filename,
                                             arm_extname);
            }

            cpl_propertylist_delete(fibre_flat_hdr);
            fibre_flat_hdr = NULL;

            continue;
        }

        cpl_msg_info(cpl_func,
                     "Processing extension %lld = %s",
                     raw_extension, arm_extname);
        cpl_msg_indent_more();

        /* Load all masters using the correct EXTNAME for the arm
           we're processing in case the mapping of extension number to
           arm could have changed. */
        if(calibs.master_bias_frame != NULL) {
            if(qmost_load_master_image_and_var(calibs.master_bias_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_bias_img,
                                               &master_bias_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master bias "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_bias_img = NULL;
            master_bias_var = NULL;
        }
    
        if(calibs.master_dark_frame != NULL) {
            if(qmost_load_master_image_and_var(calibs.master_dark_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_dark_img,
                                               &master_dark_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master dark "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_dark_img = NULL;
            master_dark_var = NULL;
        }

        if(calibs.master_detflat_frame != NULL) {
            if(qmost_load_master_image_and_var(calibs.master_detflat_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_detflat_img,
                                               &master_detflat_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master detector "
                                             "flat frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_detflat_img = NULL;
            master_detflat_var = NULL;
        }

        if(calibs.master_bpm_frame != NULL) {
            if(qmost_load_master_mask(calibs.master_bpm_frame,
                                      arm_extname,
                                      &master_bpm,
                                      NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM");
            }
        }

        if(calibs.master_lintab_frame != NULL) {
            if(qmost_load_master_table(calibs.master_lintab_frame,
                                       arm_extname,
                                       &master_lintab,
                                       NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master "
                                             "linearity table");
            }
        }

        /* Create QC parameter list */
        qclist = cpl_propertylist_new();

        /* Get trace table and fibre mask */
        extname = cpl_sprintf("trace_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format trace "
                                         "EXTNAME string");
        }
        
        if(qmost_load_master_table(calibs.fibre_trace_frame,
                                   extname,
                                   &trace_tbl,
                                   &trace_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load trace");
        }
        
        cpl_free(extname);
        extname = NULL;

        extname = cpl_sprintf("mask_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format fibre mask "
                                         "EXTNAME string");
        }

        if(qmost_load_master_mask(calibs.fibre_mask_frame,
                                  extname,
                                  &fibre_mask,
                                  NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load fibre mask");
        }
        
        cpl_free(extname);
        extname = NULL;

        if(params.level != 0) {
            /* Load PSF */
            extname = cpl_sprintf("prof_%s", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format PSF variance "
                                             "EXTNAME string");
            }
            
            if(qmost_load_master_imagelist(calibs.fibre_psf_frame,
                                           extname,
                                           CPL_TYPE_FLOAT,
                                           &fibre_psf_img,
                                           &fibre_psf_hdr) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master PSF");
            }
            
            cpl_free(extname);
            extname = NULL;
            
            extname = cpl_sprintf("profvar_%s", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format PSF variance "
                                             "EXTNAME string");
            }
            
            if(qmost_load_master_imagelist(calibs.fibre_psf_frame,
                                           extname,
                                           CPL_TYPE_FLOAT,
                                           &fibre_psf_var,
                                           NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master PSF "
                                             "variance");
            }
            
            cpl_free(extname);
            extname = NULL;
        }

        /* Stacking method depends on type of flat.  For twilight
         * flats, these could have very different count levels so we
         * need to extract the spectrum first, check the counts are 
         * okay, and then stack the good spectra.  For internal flats
         * we just stack the raws which should give better cosmic
         * rejection. */
        if(nsky > 0) {
            /* Extract spectra individually */
            nfiles = cpl_frameset_get_size(fibflats_raw);

            ff_imglist = cpl_imagelist_new();
            ff_varlist = cpl_imagelist_new();

            ngood = 0;

            for(ifile = 0; ifile < nfiles; ifile++) {
                cpl_msg_info(cpl_func,
                             "Processing flat %d of %d",
                             ifile+1, nfiles);
                cpl_msg_indent_more();
                
                if(ifile > 0) {
                    qctmp = cpl_propertylist_duplicate(qclist);
                    qcuse = qctmp;
                }
                else {
                    qcuse = qclist;
                }
                
                if(qmost_ccdproc(
                       cpl_frameset_get_position(fibflats_raw, ifile),
                       raw_extension,
                       master_bpm,
                       master_bias_img,
                       master_bias_var,
                       master_dark_img,
                       master_dark_var,
                       master_detflat_img,
                       master_detflat_var,
                       1,
                       params.swapx_table,
                       1,
                       master_lintab,
                       QMOST_LIN_AFTER_BIAS,
                       &fibre_flat_img,
                       &fibre_flat_var,
                       qcuse) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "CCD processing failed "
                                                 "for file %d extension %lld",
                                                 ifile+1,
                                                 raw_extension);
                }

                /* Extract */
                if(qmost_fibre_flat_analyse_extract_one(
                       fibre_flat_img,
                       fibre_flat_var,
                       fibre_flat_hdr,
                       qcuse,
                       arm,
                       params,
                       trace_tbl,
                       trace_hdr,
                       fibre_mask,
                       fibre_psf_img,
                       fibre_psf_var,
                       fibre_psf_hdr,
                       ifile == 0 ? proc_fibre_flat_frame : NULL,
                       ifile == 0 ? crmask_frame : NULL,
                       &extracted_fibre_flat_img,
                       &extracted_fibre_flat_var) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "extraction of fibre flat "
                                                 "%d extension %lld failed",
                                                 ifile+1,
                                                 raw_extension);
                }

                if(qctmp != NULL) {
                    cpl_propertylist_delete(qctmp);
                    qctmp = NULL;
                }

                /* Check counts.  This method is perhaps a little over
                 * simplified, we define a range of acceptable median
                 * extracted counts over all of the fibres.  It would
                 * be better to directly check for saturation in the
                 * individual fibres after cosmic rejection but this
                 * is a lot more trouble. */
                medcts = cpl_image_get_median(extracted_fibre_flat_img);

                if(medcts < params.skyflat_underexp ||
                   medcts >= params.skyflat_overexp) {
                    /* Bad, discard */
                    cpl_msg_info(cpl_func,
                                 "Median extracted counts/pixel = %.0f, BAD",
                                 medcts);

                    cpl_image_delete(extracted_fibre_flat_img);
                    extracted_fibre_flat_img = NULL;

                    cpl_image_delete(extracted_fibre_flat_var);
                    extracted_fibre_flat_var = NULL;
                }
                else {
                    /* Set in imagelist for stacking */
                    cpl_msg_info(cpl_func,
                                 "Median extracted counts/pixel = %.0f, OK",
                                 medcts);

                    cpl_imagelist_set(ff_imglist,
                                      extracted_fibre_flat_img,
                                      ngood);
                    cpl_imagelist_set(ff_varlist,
                                      extracted_fibre_flat_var,
                                      ngood);
                    ngood++;
                    
                    /* These are now owned by the imagelists */
                    extracted_fibre_flat_img = NULL;
                    extracted_fibre_flat_var = NULL;
                }

                cpl_image_delete(fibre_flat_img);
                fibre_flat_img = NULL;
                
                cpl_image_delete(fibre_flat_var);
                fibre_flat_var = NULL;

                cpl_msg_indent_less();
            }

            /* Stack if necessary */
            if(ngood < 1) {
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             CPL_ERROR_DATA_NOT_FOUND,
                                             "there were no good sky "
                                             "fibre flats");
            }
            else if(ngood > 1) {
                cpl_msg_info(cpl_func,
                             "Stacking %d spectra",
                             nfiles);
                cpl_msg_indent_more();
                
                if(qmost_spec_combine_lite(
                       ff_imglist,
                       ff_varlist,
                       params.skyflat_combthresh,
                       &extracted_fibre_flat_img,
                       &extracted_fibre_flat_var) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "stacking failed "
                                                 "for extension %lld",
                                                 raw_extension);
                }

                cpl_msg_indent_less();
            }
            else {
                extracted_fibre_flat_img = cpl_imagelist_unset(ff_imglist, 0);
                extracted_fibre_flat_var = cpl_imagelist_unset(ff_varlist, 0);
            }

            cpl_imagelist_delete(ff_imglist);
            ff_imglist = NULL;

            cpl_imagelist_delete(ff_varlist);
            ff_varlist = NULL;
        }
        else {
            /* CCD processing and stacking */
            if(qmost_ccdproc_and_combine(fibflats_raw,
                                         raw_extension,
                                         master_bpm,
                                         master_bias_img,
                                         master_bias_var,
                                         master_dark_img,
                                         master_dark_var,
                                         master_detflat_img,
                                         master_detflat_var,
                                         1,
                                         params.swapx_table,
                                         1,
                                         master_lintab,
                                         QMOST_LIN_AFTER_BIAS,
                                         params.combtype,
                                         params.scaletype,
                                         1,  /* fibre spectra */
                                         params.xrej,
                                         params.combthresh,
                                         0,
                                         0.0,
                                         0.0,
                                         &fibre_flat_img,
                                         &fibre_flat_var,
                                         qclist) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "basic CCD processing failed "
                                             "for extension %lld",
                                             raw_extension);
            }

            /* Extract */
            if(qmost_fibre_flat_analyse_extract_one(
                       fibre_flat_img,
                       fibre_flat_var,
                       fibre_flat_hdr,
                       qclist,
                       arm,
                       params,
                       trace_tbl,
                       trace_hdr,
                       fibre_mask,
                       fibre_psf_img,
                       fibre_psf_var,
                       fibre_psf_hdr,
                       proc_fibre_flat_frame,
                       crmask_frame,
                       &extracted_fibre_flat_img,
                       &extracted_fibre_flat_var) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "extraction of fibre flat "
                                             "extension %lld failed",
                                             raw_extension);
            }

            cpl_image_delete(fibre_flat_img);
            fibre_flat_img = NULL;
            
            cpl_image_delete(fibre_flat_var);
            fibre_flat_var = NULL;
        }

        if(master_bias_img != NULL) {
            cpl_image_delete(master_bias_img);
            master_bias_img = NULL;
        }

        if(master_bias_var != NULL) {
            cpl_image_delete(master_bias_var);
            master_bias_var = NULL;
        }

        if(master_dark_img != NULL) {
            cpl_image_delete(master_dark_img);
            master_dark_img = NULL;
        }

        if(master_dark_var != NULL) {
            cpl_image_delete(master_dark_var);
            master_dark_var = NULL;
        }

        if(master_detflat_img != NULL) {
            cpl_image_delete(master_detflat_img);
            master_detflat_img = NULL;
        }

        if(master_detflat_var != NULL) {
            cpl_image_delete(master_detflat_var);
            master_detflat_var = NULL;
        }

        if(master_bpm != NULL) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }

        if(master_lintab != NULL) {
            cpl_table_delete(master_lintab);
            master_lintab = NULL;
        }

        if(fibre_psf_img != NULL) {
            cpl_imagelist_delete(fibre_psf_img);
            fibre_psf_img = NULL;
        }
         
        if(fibre_psf_var != NULL) {
            cpl_imagelist_delete(fibre_psf_var);
            fibre_psf_var = NULL;
        }
         
        if(fibre_psf_hdr != NULL) {
            cpl_propertylist_delete(fibre_psf_hdr);
            fibre_psf_hdr = NULL;
        }

        /* Populate QC */
        if(qmost_extract_qc(extracted_fibre_flat_img,
                            extracted_fibre_flat_var,
                            qclist,
                            NULL,
                            arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "extracted spectrum QC failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(extracted_fibre_flat_frame != NULL) {
            /* Save extracted fibre flat */
            if(qmost_dfs_save_image_and_var(extracted_fibre_flat_frame,
                                            fibre_flat_hdr,
                                            arm_extname,
                                            qclist,
                                            extracted_fibre_flat_img,
                                            extracted_fibre_flat_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save extracted "
                                             "fibre flat %s[%s]",
                                             extracted_fibre_flat_filename,
                                             arm_extname);
            }
        }

        /* If we got nothing, abort here.  This usually means there were
         * no traces. */
        if(extracted_fibre_flat_img == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                         "no spectra were extracted in "
                                         "extension %s, check trace file",
                                         arm_extname);
        }

        /* Get wavelength solution */
        extname = cpl_sprintf("wave_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format wave "
                                         "EXTNAME string");
        }
        
        if(qmost_load_master_table(calibs.wave_frame,
                                   extname,
                                   &wave_tbl,
                                   &wave_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load wavelength "
                                         "solution");
        }
        
        cpl_free(extname);
        extname = NULL;

        /* Wavelength calibrate extracted fibre flat */
        nfib = cpl_image_get_size_y(extracted_fibre_flat_img);

        if(qmost_get_rebin_params(spec, arm,
                                  &minwave,
                                  &maxwave,
                                  &dlam) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't get wavelength "
                                         "calibration rebinning "
                                         "parameters");
        }

        waveoff = cpl_calloc(nfib, sizeof(double));
        veloff = cpl_calloc(nfib, sizeof(double));
        
        wff_hdr = cpl_propertylist_new();

        if(qmost_rebin_spectra_off(extracted_fibre_flat_img,
                                   extracted_fibre_flat_var,
                                   qclist,
                                   wave_tbl,
                                   wave_hdr,
                                   NULL,
                                   NULL,
                                   NULL,
                                   NULL,
                                   waveoff,
                                   veloff,
                                   nfib,
                                   minwave,
                                   maxwave,
                                   dlam,
                                   &wff_img,
                                   &wff_var,
                                   wff_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "wavelength calibration of "
                                         "extracted fibre flat failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        cpl_free(waveoff);
        waveoff = NULL;

        cpl_free(veloff);
        veloff = NULL;

        /* Add QC to header */
        cpl_propertylist_copy_property_regexp(wff_hdr, qclist, ".*", 0);

        /* Divide fibre flat if given */
        if(calibs.master_fibre_flat_frame != NULL) {
            if(qmost_load_master_image_and_var(calibs.master_fibre_flat_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_ffnorm_img,
                                               &master_ffnorm_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master "
                                             "fibre flat");
            }

            if(qmost_ffdiv_fib(wff_img,
                               wff_var,
                               wff_hdr,
                               trace_tbl,
                               master_ffnorm_img,
                               master_ffnorm_var) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "fibre flat division "
                                             "failed for extension %lld",
                                             raw_extension);
            }

            cpl_image_delete(master_ffnorm_img);
            master_ffnorm_img = NULL;

            cpl_image_delete(master_ffnorm_var);
            master_ffnorm_var = NULL;
        }

        /* A minimal FIBINFO table is needed to be able to emit fibre
         * flat normalisation information for subsequent processing
         * steps, so it can't be omitted as we usually do elsewhere in
         * the pipeline when there was no input FIBINFO table.  We
         * therefore create a minimal FIBINFO containing only the
         * columns we need to do this here as a workaround to allow
         * processing of lab data (which don't have FIBINFO tables).
         * This is only executed once during processing of the first
         * arm with data. */
        if(fibinfo_tbl == NULL) {
            if(qmost_fibtab_dummy(cpl_image_get_size_y(wff_img),
                                  &fibinfo_tbl,
                                  &fibinfo_hdr) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not create dummy "
                                             "FIBINFO table");
            }
        }

        /* Normalise fibre flat */
        if(qmost_ffnorm_fib(wff_img,
                            wff_var,
                            wff_hdr,
                            arm,
                            fibinfo_tbl,
                            params.fftrack,
                            params.ffsmooth,
                            params.ffrescale) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "normalisation of fibre flat "
                                         "failed for extension %lld",
                                         raw_extension);
        }

        /* Save extracted, wavelength calibrated fibre flat */
        if(qmost_dfs_save_image_and_var(ffnorm_frame,
                                        fibre_flat_hdr,
                                        arm_extname,
                                        wff_hdr,
                                        wff_img,
                                        wff_var,
                                        CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save output "
                                         "fibre flat %s[%s]",
                                         ffnorm_filename,
                                         arm_extname);
        }

        /* Clean up */
        cpl_propertylist_delete(fibre_flat_hdr);
        fibre_flat_hdr = NULL;

        cpl_table_delete(trace_tbl);
        trace_tbl = NULL;

        cpl_propertylist_delete(trace_hdr);
        trace_hdr = NULL;

        cpl_mask_delete(fibre_mask);
        fibre_mask = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;

        cpl_image_delete(extracted_fibre_flat_img);
        extracted_fibre_flat_img = NULL;

        cpl_image_delete(extracted_fibre_flat_var);
        extracted_fibre_flat_var = NULL;

        cpl_table_delete(wave_tbl);
        wave_tbl = NULL;

        cpl_propertylist_delete(wave_hdr);
        wave_hdr = NULL;

        cpl_image_delete(wff_img);
        wff_img = NULL;

        cpl_image_delete(wff_var);
        wff_var = NULL;

        cpl_propertylist_delete(wff_hdr);
        wff_hdr = NULL;

        cpl_msg_indent_less();
    }

    /* Output fibre flat frame, if requested */
    if(proc_fibre_flat_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                proc_fibre_flat_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to master fibre "
                                             "flat frame %s",
                                             proc_fibre_flat_filename);
            }
        }
    }
    
    if(extracted_fibre_flat_frame != NULL) {
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                extracted_fibre_flat_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to output extracted fibre "
                                             "flat frame %s",
                                             extracted_fibre_flat_filename);
            }
        }
    }    

    /* Copy fibinfo table from the first frame if there was one */
    if(fibinfo_tbl != NULL) {
        if(qmost_fibtabsave(fibinfo_tbl,
                            fibinfo_hdr,
                            ffnorm_frame) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save FIBINFO table "
                                         "to output fibre "
                                         "flat frame %s",
                                         ffnorm_filename);
        }
    }

    /* Clean up */
    cpl_frameset_delete(fibflats_raw);
    fibflats_raw = NULL;

    if(fibinfo_tbl != NULL) {
        cpl_table_delete(fibinfo_tbl);
        fibinfo_tbl = NULL;
    }

    if(fibinfo_hdr != NULL) {
        cpl_propertylist_delete(fibinfo_hdr);
        fibinfo_hdr = NULL;
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_fibre_flat_analyse_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_parameter *par;

    /* QC level parameter */
    par = cpl_parameter_new_value(RECIPE_NAME".level",
                                  CPL_TYPE_INT,
                                  "QC level (0: QC0; 1: QC1)",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "level");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Switch to save processed products */
    par = cpl_parameter_new_value(RECIPE_NAME".keep",
                                  CPL_TYPE_BOOL,
                                  "Save optional processed products",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "keep");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* ccdproc parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".ccdproc.swapx",
                                  CPL_TYPE_STRING,
                                  "String of 0 or 1 specifying for each "
                                  "arm of each spectrograph if we should "
                                  "flip the x axis to correct the "
                                  "wavelength order of the spectral axis.",
                                  RECIPE_NAME,
                                  QMOST_DEFAULT_SWAPX_TABLE);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "ccdproc.swapx");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* imcombine parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.combtype",
                                  CPL_TYPE_INT,
                                  "Stacking method when combining multiple "
                                  "images.  1: mean, 2: median.",
                                  RECIPE_NAME,
                                  QMOST_MEANCALC);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.combtype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.scaletype",
                                  CPL_TYPE_INT,
                                  "Scaling to use when combining multiple "
                                  "images.  0: no biasing or scaling, "
                                  "1: bias by relative offset in background "
                                  "levels, 2: scale by ratio of background "
                                  "levels, 3: scale by exposure time and "
                                  "bias by relative offset in background.",
                                  RECIPE_NAME,
                                  2);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.scaletype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.xrej",
                                  CPL_TYPE_BOOL,
                                  "Do extra rejection cycle when combining "
                                  "images.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.xrej");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold in terms of "
                                  "background noise for rejection when "
                                  "combining images.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Scattered light removal parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".scattered.enable",
                                  CPL_TYPE_BOOL,
                                  "Enable scattered light removal",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.enable");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".scattered.nbsize",
                                  CPL_TYPE_INT,
                                  "Size of the background smoothing cells "
                                  "for determining scattered light in pixels.",
                                  RECIPE_NAME,
                                  256);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.nbsize");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);
    
    /* Spectral extraction parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".extract.crosstalk",
                                  CPL_TYPE_BOOL,
                                  "Enable crosstalk correction",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.crosstalk");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".extract.cr_thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold for PSF "
                                  "spectral extraction.",
                                  RECIPE_NAME,
                                  10.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.cr_thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".extract.niter",
                                  CPL_TYPE_INT,
                                  "Number of rejection iterations for PSF "
                                  "spectral extraction.",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.niter");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".extract.width",
                                  CPL_TYPE_INT,
                                  "Boxcar width for spectral extraction.",
                                  RECIPE_NAME,
                                  6);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "extract.width");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Twilight flat parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".skyflat.underexp",
                                  CPL_TYPE_DOUBLE,
                                  "A minimum count level in ADU to be "
                                  "used to reject underexposed sky flats.",
                                  RECIPE_NAME,
                                  100.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "skyflat.underexp");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".skyflat.overexp",
                                  CPL_TYPE_DOUBLE,
                                  "A saturation level in ADU to be "
                                  "used to reject overexposed sky flats.",
                                  RECIPE_NAME,
                                  120000.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "skyflat.overexp");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".skyflat.rej_thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Threshold for outlier rejection "
                                  "when combining sky flats.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "skyflat.rej_thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Flat field normalisation parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".ffnorm.track",
                                  CPL_TYPE_BOOL,
                                  "Remove median response at each "
                                  "wavelength.",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "ffnorm.track");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".ffnorm.smooth",
                                  CPL_TYPE_INT,
                                  "Size of fibre flat smoothing box "
                                  "in spectral direction, or 0 for none.",
                                  RECIPE_NAME,
                                  51);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "ffnorm.smooth");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".ffnorm.rescale",
                                  CPL_TYPE_BOOL,
                                  "Scale fibres by their relative "
                                  "throughputs",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "ffnorm.rescale");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Extract a single spectrum.
 *
 * This routine contains all the steps needed to go from a processed
 * 2D image to an extracted and calibrated spectrum: scattered light
 * removal, spectral extraction, wavelength calibration, fibre flat
 * fielding, and (optionally) sky subtraction.  This is called in a
 * loop by the main body of qmost_science_process to do the parts of
 * processing that should be done individually on each science frame
 * from an OB prior to OB-level stacking.
 *
 * @param   fibre_flat_img       (Given)    The input processed image
 *                                          from qmost_ccdproc.
 * @param   fibre_flat_var       (Given)    The input variance array
 *                                          from qmost_ccdproc.
 * @param   fibre_flat_hdr       (Given)    The raw FITS header for
 *                                          the image extension being
 *                                          processed.
 * @param   qclist               (Modified) A propertylist with QC
 *                                          headers, usually also from
 *                                          qmost_ccdproc.  Will be
 *                                          augmented with appropriate
 *                                          QC from the extraction
 *                                          process.
 * @param   arm                  (Given)    One of the QMOST_ARM_*
 *                                          constants saying which arm
 *                                          we're processing.
 * @param   params               (Given)    A structure containing all
 *                                          of the recipe parameters.
 * @param   trace_tbl            (Given)    The trace table.
 * @param   trace_hdr            (Given)    The trace table extension
 *                                          FITS header.
 * @param   fibre_mask           (Given)    The fibre mask defining
 *                                          which regions of the
 *                                          detector are illuminated
 *                                          by the fibres.
 * @param   fibre_psf_img        (Given)    An image list containing
 *                                          the fibre PSFs at each
 *                                          spectral pixel.
 * @param   fibre_psf_var        (Given)    The corresponding variances
 *                                          of the fibre PSFs.
 * @param   fibre_psf_hdr        (Given)    The FITS header of the
 *                                          profile file with WCS.
 * @param   proc_fibre_flat_frame (Given)   If non-NULL the final
 *                                          scattered light corrected
 *                                          2D image will be written
 *                                          to this frame.
 * @param   crmask_frame          (Given)   If non-NULL the cosmic ray
 *                                          masked produced during PSF
 *                                          spectral extraction will
 *                                          be written to this frame.
 * @param   extracted_fibre_flat_img (Returned) The extracted spectrum.
 * @param   extracted_fibre_flat_var (Returned) The extracted spectrum
 *                                              variance.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_fibre_flat_analyse_extract_one(
    cpl_image *fibre_flat_img,
    cpl_image *fibre_flat_var,
    cpl_propertylist *fibre_flat_hdr,
    cpl_propertylist *qclist,
    int arm,
    struct qmost_fibre_flat_analyse_params params,
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    cpl_mask *fibre_mask,
    cpl_imagelist *fibre_psf_img,
    cpl_imagelist *fibre_psf_var,
    cpl_propertylist *fibre_psf_hdr,
    const cpl_frame *proc_fibre_flat_frame,
    const cpl_frame *crmask_frame,
    cpl_image **extracted_fibre_flat_img,
    cpl_image **extracted_fibre_flat_var)
{
    const char *arm_extname;
    char *extname = NULL;

    const char *proc_fibre_flat_filename = NULL;
    const char *crmask_filename = NULL;

    cpl_image *extract_rej = NULL;

    *extracted_fibre_flat_img = NULL;
    *extracted_fibre_flat_var = NULL;

#undef TIDY
#define TIDY                                            \
    if(extname != NULL) {                               \
        cpl_free(extname);                              \
        extname = NULL;                                 \
    }                                                   \
    if(*extracted_fibre_flat_img != NULL) {             \
        cpl_image_delete(*extracted_fibre_flat_img);    \
        *extracted_fibre_flat_img = NULL;               \
    }                                                   \
    if(*extracted_fibre_flat_var != NULL) {             \
        cpl_image_delete(*extracted_fibre_flat_var);    \
        *extracted_fibre_flat_var = NULL;               \
    }                                                   \
    if(extract_rej != NULL) {                           \
        cpl_image_delete(extract_rej);                  \
        extract_rej = NULL;                             \
    }

    /* Get EXTNAME for arm */
    arm_extname = qmost_pfits_get_extname(arm);
    if(arm_extname == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine EXTNAME "
                                     "for arm %d", arm);
    }

    /* Get filenames, if frames were given */
    if(proc_fibre_flat_frame != NULL) {
        proc_fibre_flat_filename = cpl_frame_get_filename(
            proc_fibre_flat_frame);
    }

    if(crmask_frame != NULL) {
        crmask_filename = cpl_frame_get_filename(crmask_frame);
    }

    /* Populate scattered light QC before we remove it */
    if(qmost_scattered_qc(fibre_flat_img,
                          qclist,
                          fibre_mask,
                          qclist) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "scattered light QC failed");
    }

    if(params.scattered_enable) {
        /* Remove scattered light if requested */
        if(qmost_scattered(fibre_flat_img,
                           qclist,
                           fibre_mask,
                           params.scattered_nbsize,
                           1) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "removal of scattered light "
                                         "with box size %d failed",
                                         params.scattered_nbsize);
        }
    }

    /* Save processed fibre flat if requested */
    if(proc_fibre_flat_frame != NULL) {
        if(qmost_dfs_save_image_and_var(proc_fibre_flat_frame,
                                        fibre_flat_hdr,
                                        arm_extname,
                                        qclist,
                                        fibre_flat_img,
                                        fibre_flat_var,
                                        CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save master fibre "
                                         "flat frame %s[%s]",
                                         proc_fibre_flat_filename,
                                         arm_extname);
        }
    }

    if(params.level == 0) {
        /* Extract fibre flat */
        if(qmost_extract_tram(fibre_flat_img,
                              fibre_flat_var,
                              qclist,
                              trace_tbl,
                              trace_hdr,
                              params.extract_width,
                              extracted_fibre_flat_img,
                              extracted_fibre_flat_var,
                              qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "spectral extraction failed");
        }
    }
    else {
        /* Extract fibre flat */
        if(qmost_extract_psf(fibre_flat_img,
                             fibre_flat_var,
                             qclist,
                             NULL,  /* no cosmic in QC */
                             fibre_psf_img,
                             fibre_psf_var,
                             fibre_psf_hdr,
                             trace_tbl,
                             trace_hdr,
                             params.extract_crosstalk,
                             params.extract_niter,
                             params.extract_crrej_thr,
                             extracted_fibre_flat_img,
                             extracted_fibre_flat_var,
                             qclist,
                             &extract_rej) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "spectral extraction failed");
        }

        /* Save optional cosmic ray mask */
        if(crmask_frame != NULL) {
            if(qmost_dfs_save_image_extension(crmask_frame,
                                              fibre_flat_hdr,
                                              arm_extname,
                                              qclist,
                                              extract_rej,
                                              CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save cosmic ray "
                                             "mask %s[%s]",
                                             crmask_filename,
                                             arm_extname);
            }
        }

        cpl_image_delete(extract_rej);
        extract_rej = NULL;
    }

    return CPL_ERROR_NONE;
}

/**@}*/
