/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_CCDPROC_H
#define QMOST_CCDPROC_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/**
 * @ingroup qmost_ccdproc
 *
 * @brief   The values of the lin_when parameter to ccdproc to control
 *          linearity correction.
 */

enum _lin_when {
    /** Disable linearity correction. */
    QMOST_LIN_NEVER = 0,
    /** Apply linearity correction before subtraction of bias. */
    QMOST_LIN_BEFORE_BIAS,
    /** Apply linearity correction after subtraction of bias. */
    QMOST_LIN_AFTER_BIAS
};

/**
 * @ingroup qmost_ccdproc
 *
 * @brief   Default for the swapx_table parameter to ccdproc to
 *          control axis flipping.
 *
 * This is a character string with an entry for each arm of each
 * spectrograph where the image will be flipped in x direction if the
 * character is '1' and not if it is '0'.
 *
 * The order of the entries is:
 * HRS(R) HRS(G) HRS(B)
 * LRS-A(R) LRS-A(G) LRS-A(B)
 * LRS-B(R) LRS-B(G) LRS-B(B)
 *
 * The default value is set for the test slit lab data set from 2023
 * March 9 to allow for integration tests on these data at ESO.  The
 * previous default suitable for simulated data can be recovered at
 * build time by defining QMOST_SIMULATED_DATA.
 */

#ifdef QMOST_SIMULATED_DATA
#define QMOST_DEFAULT_SWAPX_TABLE "100101010"
#else
#define QMOST_DEFAULT_SWAPX_TABLE "100010101"
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_ccdproc (
    const cpl_frame *raw_frame,
    int extension,
    cpl_mask *master_bpm,
    cpl_image *master_bias_img,
    cpl_image *master_bias_var,
    cpl_image *master_dark_img,
    cpl_image *master_dark_var,
    cpl_image *master_detflat_img,
    cpl_image *master_detflat_var,
    int flip,
    const char *swapx_table,
    int oscor,
    cpl_table *linearity,
    int lin_when,
    cpl_image **processed_img,
    cpl_image **processed_var,
    cpl_propertylist *qclist);

cpl_error_code qmost_ccdproc_and_combine (
    cpl_frameset *raw_frames,
    int extension,
    cpl_mask *master_bpm,
    cpl_image *master_bias_img,
    cpl_image *master_bias_var,
    cpl_image *master_dark_img,
    cpl_image *master_dark_var,
    cpl_image *master_detflat_img,
    cpl_image *master_detflat_var,
    int flip,
    const char *swapx_table,
    int oscor,
    cpl_table *linearity,
    int lin_when,
    int combtype,
    int scaletype,
    int fibre,
    int xrej,
    float thresh,
    int checkexp,
    float underexp,
    float overexp,
    cpl_image **out_img,
    cpl_image **out_var,
    cpl_propertylist *qclist);

cpl_error_code qmost_findgain (
    cpl_image **images,
    cpl_propertylist **pri_hdr,
    cpl_propertylist **ext_hdr,
    int nimgin,
    cpl_mask *master_bpm,
    cpl_propertylist *qclist);

CPL_END_DECLS

#endif  /* QMOST_CCDPROC_H */
