/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>

#include "qmost_constants.h"
#include "qmost_diffimg.h"
#include "qmost_utils.h"

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_diffimg  qmost_diffimg
 *
 * Differencing to compare new calibration to reference.
 *
 * @par Synopsis:
 * @code
 *   #include "qmost_diffimg.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief   Compute difference image and statistics.
 *
 * Computes a difference or ratio image comparing new master
 * calibration frame to a given reference master calibration frame, in
 * the sense comparison - reference or comparison / reference,
 * depending on the value of the parameter divide.  Statistics of the
 * difference image are recorded into QC FITS headers, both on a pixel
 * by pixel basis, and binned down into cells to reduce noise and
 * increase sensitivity to low-level features.
 *
 * @param   com_img          (Given)    The image to compare.
 * @param   ref_img          (Given)    The reference image.
 * @param   ncells           (Given)    The number of cells (bins) to
 *                                      use.  This should be a power
 *                                      of two, in which case each
 *                                      axis is divided evenly into
 *                                      sqrt(ncells) cells (for
 *                                      example, 16x16 if ncells =
 *                                      256).
 * @param   divide           (Given)    If true, divide the images.
 *                                      If false, subtract the images.
 * @param   diff_img         (Returned) The difference image.  The
 *                                      units of the difference image
 *                                      depend on the setting of the
 *                                      parameter divide and are the
 *                                      same as the input images if
 *                                      divide=0, or a dimensionless
 *                                      ratio if divide=1.
 * @param   qclist           (Modified) A caller allocated property
 *                                      list to receive the QC FITS
 *                                      headers.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 * @retval  CPL_ERROR_ILLEGAL_INPUT       If the number of cells is
 *                                        invalid.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the input images aren't
 *                                        the same size.
 *
 * @par Output QC Parameters:
 *   Note: all output QC parameters have the same units as the
 *   difference image, which are ADU if divide is false and a
 *   dimensionless ratio if divide is true.
 *   - <b>COMPARE BIN MAX</b>: The maximum over all of the cells of
 *     the median of the pixels in each cell.
 *   - <b>COMPARE BIN MEAN</b>: The mean over all of the cells of the
 *     median of the pixels in each cell.
 *   - <b>COMPARE BIN MIN</b>: The minimum over all of the cells of
 *     the median of the pixels in each cell.
 *   - <b>COMPARE BIN RMS</b>: The RMS over all of the cells of the
 *     median of the pixels in each cell.
 *   - <b>COMPARE MED</b>: The median of the pixels in the difference
 *     image.
 *   - <b>COMPARE RMS</b>: The robustly-estimated RMS of the pixels in
 *     the difference image.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_diffimg (
    cpl_image *com_img,
    cpl_image *ref_img,
    int ncells,
    int divide,
    cpl_image **diff_img,
    cpl_propertylist *qclist)
{
    double median, sigma;
    int nx, ny, nbx, nby, nbins, nbsizex, nbsizey, nxbord, nybord;
    int bx, by, ox, oy, pb;

    cpl_array *binmeds = NULL;
    cpl_array *binsigs = NULL;

    cpl_errorstate prestate;

    cpl_ensure_code(com_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ncells > 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(diff_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

    *diff_img = NULL;

#undef TIDY
#define TIDY                                    \
    if(*diff_img != NULL) {                     \
        cpl_image_delete(*diff_img);            \
        *diff_img = NULL;                       \
    }                                           \
    if(binmeds != NULL) {                       \
        cpl_array_delete(binmeds);              \
        binmeds = NULL;                         \
    }                                           \
    if(binsigs != NULL) {                       \
        cpl_array_delete(binsigs);              \
        binsigs = NULL;                         \
    }

    /* Compute difference or divide as requested */
    if(divide) {
        *diff_img = cpl_image_divide_create(com_img, ref_img);
    }
    else {
        *diff_img = cpl_image_subtract_create(com_img, ref_img);
    }

    if(*diff_img == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not difference images");
    }

    /* Compute stats of the entire difference image first */
    median = cpl_image_get_mad(*diff_img, &sigma);
    sigma *= CPL_MATH_STD_MAD;

    cpl_propertylist_update_double(qclist,
                                   "ESO QC COMPARE MED",
                                   median);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC COMPARE MED",
                                 "Median compared to reference");

    cpl_propertylist_update_double(qclist,
                                   "ESO QC COMPARE RMS",
                                   sigma);
    cpl_propertylist_set_comment(qclist,
                                 "ESO QC COMPARE RMS",
                                 "RMS of comparison to reference");

    /* Image size */
    nx = cpl_image_get_size_x(*diff_img);
    ny = cpl_image_get_size_y(*diff_img);

    /* Decide number of bins */
    nby = sqrt(ncells);
    if(nby < 1) {
        nby = 1;
    }

    nbx = ncells / nby;
    if(nbx < 1) {
        nbx = 1;
    }

    nbins = nbx * nby;

    if(nbins > 1) {
        /* Actual bin size */
        nbsizex = nx / nbx;
        nbsizey = ny / nby;
        
        /* Crop border size to ensure exact number of pixels */
        nxbord = (nx - nbx * nbsizex) / 2;
        nybord = (ny - nby * nbsizey) / 2;
        
        /* Allocate workspace */
        binmeds = cpl_array_new(nbins, CPL_TYPE_DOUBLE);
        binsigs = cpl_array_new(nbins, CPL_TYPE_DOUBLE);
        
        /* Compute stats in each bin */
        for(by = 0; by < nby; by++) {
            oy = nybord + by * nbsizey;
            
            for(bx = 0; bx < nbx; bx++) {
                ox = nxbord + bx * nbsizex;
                
                pb = by*nbx + bx;
                
                prestate = cpl_errorstate_get();
        
                median = cpl_image_get_mad_window(*diff_img,
                                                  ox + 1,
                                                  oy + 1,
                                                  ox + nbsizex,
                                                  oy + nbsizey,
                                                  &sigma);
                if(!cpl_errorstate_is_equal(prestate)) {
                    switch(cpl_error_get_code()) {
                    case CPL_ERROR_DATA_NOT_FOUND:
                        /* No good pixels in bin */
                        cpl_errorstate_set(prestate);
                        cpl_array_set_invalid(binmeds, pb);
                        cpl_array_set_invalid(binsigs, pb);
                        break;
                    case CPL_ERROR_NONE:
                        /* Not sure how this can happen but what the hey */
                        cpl_errorstate_set(prestate);
                        cpl_array_set(binmeds, pb, median);
                        cpl_array_set(binsigs, pb, sigma * CPL_MATH_STD_MAD);
                        break;
                    default:
                        TIDY;
                        return cpl_error_set_message(cpl_func,
                                                     cpl_error_get_code(),
                                                     "could not compute "
                                                     "difference image "
                                                     "stats in bin "
                                                     "[%d:%d,%d:%d]",
                                                     ox + 1,
                                                     ox + nbsizex,
                                                     oy + 1,
                                                     oy + nbsizey);
                    }
                }
                else {
                    cpl_array_set(binmeds, pb, median);
                    cpl_array_set(binsigs, pb, sigma * CPL_MATH_STD_MAD);
                }
            }
        }
        
        if(cpl_array_has_valid(binmeds)) {
            cpl_propertylist_update_double(qclist,
                                           "ESO QC COMPARE BIN MEAN",
                                           cpl_array_get_mean(binmeds));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC COMPARE BIN MEAN",
                                         "Mean of binned comparison "
                                         "to reference");

            cpl_propertylist_update_double(qclist,
                                           "ESO QC COMPARE BIN RMS",
                                           cpl_array_get_stdev(binmeds));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC COMPARE BIN RMS",
                                         "RMS of binned comparison "
                                         "to reference");

            cpl_propertylist_update_double(qclist,
                                           "ESO QC COMPARE BIN MIN",
                                           cpl_array_get_min(binmeds));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC COMPARE BIN MIN",
                                         "Lowest bin median in "
                                         "comparison to reference");

            cpl_propertylist_update_double(qclist,
                                           "ESO QC COMPARE BIN MAX",
                                           cpl_array_get_max(binmeds));
            cpl_propertylist_set_comment(qclist,
                                         "ESO QC COMPARE BIN MAX",
                                         "Highest bin median in "
                                         "comparison to reference");
        }

        cpl_array_delete(binmeds);
        binmeds = NULL;
        
        cpl_array_delete(binsigs);
        binsigs = NULL;
    }

    return CPL_ERROR_NONE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief   Compute difference image and statistics for spectra.
 *
 * Computes a difference or ratio image of an image where each row
 * is a 1D spectrum.  Statistics of the difference image are computed
 * fibre by fibre and recorded into QC FITS headers.
 *
 * @param   com_img          (Given)    The image to compare.
 * @param   ref_img          (Given)    The reference image.
 * @param   divide           (Given)    If true, divide the images.
 *                                      If false, subtract the images.
 * @param   diff_img         (Returned) The difference image.
 * @param   qclist           (Modified) A caller allocated property
 *                                      list to receive the QC FITS
 *                                      headers.
 *
 * @return  cpl_error_code
 *
 * @retval  CPL_ERROR_NONE                If everything is OK.
 * @retval  CPL_ERROR_INCOMPATIBLE_INPUT  If the input images aren't
 *                                        the same size.
 * @retval  CPL_ERROR_NULL_INPUT          If one of the required
 *                                        inputs or outputs was NULL.
 *
 * @par Output QC Parameters:
 *   - <b>COMPARE MAX</b>: The maximum over the fibres of the median
 *     difference in each fibre.
 *   - <b>COMPARE MAXSPC</b>: The fibre with the largest median
 *     difference.
 *   - <b>COMPARE MEAN</b>: The mean over the fibres of the median
 *     difference in each fibre.
 *   - <b>COMPARE MIN</b>: The minimum over the fibres of the median
 *     difference in each fibre.
 *   - <b>COMPARE RMS</b>: The RMS over the fibres of the median
 *     difference in each fibre.
 *   - <b>COMPARE MINSPC</b>: The fibre with the smallest median
 *     difference.
 *
 * @author  Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_diffimg_fib (
    cpl_image *com_img,
    cpl_image *ref_img,
    int divide,
    cpl_image **diff_img,
    cpl_propertylist *qclist)
{
    double median, sigma;
    int nx, y, ny;

    cpl_array *fibmeds = NULL;
    cpl_array *fibsigs = NULL;

    cpl_size pos;

    cpl_errorstate prestate;

    cpl_ensure_code(com_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(ref_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(diff_img != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(qclist != NULL, CPL_ERROR_NULL_INPUT);

    *diff_img = NULL;

#undef TIDY
#define TIDY                                    \
    if(*diff_img != NULL) {                     \
        cpl_image_delete(*diff_img);            \
        *diff_img = NULL;                       \
    }                                           \
    if(fibmeds != NULL) {                       \
        cpl_array_delete(fibmeds);              \
        fibmeds = NULL;                         \
    }                                           \
    if(fibsigs != NULL) {                       \
        cpl_array_delete(fibsigs);              \
        fibsigs = NULL;                         \
    }

    /* Compute difference or divide as requested */
    if(divide) {
        *diff_img = cpl_image_divide_create(com_img, ref_img);
    }
    else {
        *diff_img = cpl_image_subtract_create(com_img, ref_img);
    }

    if(*diff_img == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "could not difference images");
    }

    /* Image size */
    nx = cpl_image_get_size_x(*diff_img);
    ny = cpl_image_get_size_y(*diff_img);

    /* Allocate workspace */
    fibmeds = cpl_array_new(ny, CPL_TYPE_DOUBLE);
    fibsigs = cpl_array_new(ny, CPL_TYPE_DOUBLE);
        
    /* Compute stats for each fibre (row) */
    for(y = 0; y < ny; y++) {
        prestate = cpl_errorstate_get();
    
        median = cpl_image_get_mad_window(*diff_img,
                                          1,
                                          y + 1,
                                          nx,
                                          y + 1,
                                          &sigma);
        if(!cpl_errorstate_is_equal(prestate)) {
            switch(cpl_error_get_code()) {
            case CPL_ERROR_DATA_NOT_FOUND:
                /* No good pixels */
                cpl_errorstate_set(prestate);
                cpl_array_set_invalid(fibmeds, y);
                cpl_array_set_invalid(fibsigs, y);
                break;
            case CPL_ERROR_NONE:
                /* Not sure how this can happen but what the hey */
                cpl_errorstate_set(prestate);
                cpl_array_set(fibmeds, y, median);
                cpl_array_set(fibsigs, y, sigma * CPL_MATH_STD_MAD);
                break;
            default:
                TIDY;
                return cpl_error_set_message(cpl_func,
                                             cpl_error_get_code(),
                                             "could not compute "
                                             "difference image "
                                             "stats for fibre %d",
                                             y + 1);
            }
        }
        else {
            cpl_array_set(fibmeds, y, median);
            cpl_array_set(fibsigs, y, sigma * CPL_MATH_STD_MAD);
        }
    }

    if(cpl_array_has_valid(fibmeds)) {
        cpl_propertylist_update_double(qclist,
                                       "ESO QC COMPARE MEAN",
                                       cpl_array_get_mean(fibmeds));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC COMPARE MEAN",
                                     divide ?
                                     "Mean of fibre comparison "
                                     "to reference" :
                                     "[ADU] Mean of fibre comparison "
                                     "to reference");
        
        cpl_propertylist_update_double(qclist,
                                       "ESO QC COMPARE RMS",
                                       cpl_array_get_stdev(fibmeds));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC COMPARE RMS",
                                     divide ?
                                     "RMS of fibre comparison "
                                     "to reference" :
                                     "[ADU] RMS of fibre comparison "
                                     "to reference");
        
        cpl_propertylist_update_double(qclist,
                                       "ESO QC COMPARE MIN",
                                       cpl_array_get_min(fibmeds));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC COMPARE MIN",
                                     divide ?
                                     "Lowest fibre median in "
                                     "comparison to reference" :
                                     "[ADU] Lowest fibre median in "
                                     "comparison to reference");

        cpl_array_get_minpos(fibmeds, &pos);

        cpl_propertylist_update_int(qclist,
                                    "ESO QC COMPARE MINSPC",
                                    pos + 1);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC COMPARE MINSPC",
                                     "Fibre with lowest median in "
                                     "comparison to reference");
        
        cpl_propertylist_update_double(qclist,
                                       "ESO QC COMPARE MAX",
                                       cpl_array_get_max(fibmeds));
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC COMPARE MAX",
                                     divide ?
                                     "Highest fibre median in "
                                     "comparison to reference" :
                                     "[ADU] Highest fibre median in "
                                     "comparison to reference");

        cpl_array_get_maxpos(fibmeds, &pos);

        cpl_propertylist_update_int(qclist,
                                    "ESO QC COMPARE MAXSPC",
                                    pos + 1);
        cpl_propertylist_set_comment(qclist,
                                     "ESO QC COMPARE MAXSPC",
                                     "Fibre with highest median in "
                                     "comparison to reference");
    }

    cpl_array_delete(fibmeds);
    fibmeds = NULL;
    
    cpl_array_delete(fibsigs);
    fibsigs = NULL;

    return CPL_ERROR_NONE;
}

/**@}*/
