/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_TRACEINFO_H
#define QMOST_TRACEINFO_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/**
 * @file qmost_traceinfo.h
 */

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @ingroup qmost_traceinfo
 * @struct qmost_traceinfo
 * Trace information structure.
 */
/*----------------------------------------------------------------------------*/

typedef struct {
    /** @brief Spectrum number in table. */
    int    specnum;
    /** @brief Is fibre live? */
    int    live;
    /** @brief Start y coordinate for trace. */
    int    yst;
    /** @brief End y coordinate for trace. */
    int    yfn;
    /** @brief Reference y coordinate for trace. */
    double yref;
    /** @brief RMS residual in x. */
    double trrms;
    /** @brief Degree of polynomial. */
    int    nord;
    /** @brief Polynomial for x(y). */
    cpl_polynomial *coefs;
    /** @brief Number of positions measured. */
    int    npos;
    /** @brief Trace x measurements. */
    double *xpos;
    /** @brief y coordinates trace was measured at. */
    double *ypos;
    /** @brief Spatial FWHM at each trace point */
    float *fwhm;
    /** @brief Peak counts at each trace point (ADU) */
    float *peak;
    /** @brief Measure of contrast of peak compared to continuum */
    float *contrast;
    /** @brief Minimum yst over all traces. */
    int    minyst;
    /** @brief Maximum yfn over all traces. */
    int    maxyfn;
} qmost_traceinfo;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_trcreate(
    cpl_table **fptr,
    int nord,
    int nblocks);

cpl_error_code qmost_tropen (
    cpl_table *tptr,
    cpl_propertylist *thdr,
    int *ntr,
    qmost_traceinfo **tr);

cpl_error_code qmost_trchk(
    cpl_table *tptr);

cpl_error_code qmost_trread1(
    cpl_table *tptr,
    int row,
    qmost_traceinfo *tr);

cpl_error_code qmost_trinsert_dead(
    cpl_table *tptr,
    int row);

cpl_error_code qmost_trrenumber(
    cpl_table *tptr);

cpl_error_code qmost_trwrite1(
    cpl_table *tptr,
    qmost_traceinfo tr,
    int row);

double qmost_tracexpos(
    qmost_traceinfo tr,
    double ypos);

void qmost_trclose(
    int ntr,
    qmost_traceinfo **tr);

cpl_error_code qmost_trace_qc (
    cpl_table *trace_tbl,
    cpl_propertylist *trace_hdr,
    cpl_table *ref_trace_tbl,
    cpl_propertylist *ref_trace_hdr,
    cpl_propertylist *qclist,
    cpl_table *fibinfo_tbl,
    int arm);

CPL_END_DECLS

#endif  /* QMOST_TRACEINFO_H */
