/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_LINEAR_H
#define QMOST_LINEAR_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>

#include "qmost_lininfo.h"

CPL_BEGIN_DECLS

/**
  * @file qmost_linear.h
  */

/*----------------------------------------------------------------------------*/
/*
 *                              New types
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @ingroup qmost_linear
 * @struct qmost_flatstats
 * Non-linearity sequence statistics structure holding statistics for
 * a single flat.
 */
/*----------------------------------------------------------------------------*/

typedef struct {
    /** @brief Flat exposure time */
    double exptime;
    /** @brief Flat MJD */
    double mjd;
    /** @brief x pixel lower limit of flat region analysed in each
     * amplifier */
    int *xmin;
    /** @brief x pixel upper limit of flat region analysed in each
     * amplifier */
    int *xmax;
    /** @brief y pixel lower limit of flat region analysed in each
     * amplifier */
    int *ymin;
    /** @brief y pixel upper limit of flat region analysed in each
     * amplifier */
    int *ymax;
    /** @brief Median counts for each amplifier (ADU) */
    float *median;
    /** @brief Sigma of counts for each amplifier (ADU) */
    float *sigma;
    /** @brief Number of amplifiers */
    int namps;
} qmost_flatstats;

/*----------------------------------------------------------------------------*/
/*
 *                              Function prototypes
 */
/*----------------------------------------------------------------------------*/

cpl_error_code qmost_flatstats_compute (
    qmost_flatstats *self,
    cpl_image *img,
    cpl_propertylist *pri_hdr,
    cpl_propertylist *qclist);

int qmost_flatstats_check (
    qmost_flatstats *self,
    float underexp,
    float overexp);

void qmost_flatstats_free (
    qmost_flatstats *self);

cpl_error_code qmost_linfit (
    qmost_flatstats *lin_stats,
    int nlin,
    qmost_flatstats *mon_stats,
    int nmon,
    int docorr,
    cpl_table **tbl,
    cpl_propertylist *qclist,
    int nord,
    int niter,
    float clipthr);

cpl_error_code qmost_lincor (
    cpl_image *img,
    cpl_propertylist *qclist,
    cpl_table *linearity_tbl);

void qmost_lincor1_double (
    double *data,
    int nx,
    int ixl,
    int ixh,
    int iyl,
    int iyh,
    qmost_lininfo *lin);

void qmost_lincor1_float (
    float *data,
    int nx,
    int ixl,
    int ixh,
    int iyl,
    int iyh,
    qmost_lininfo *lin);

cpl_error_code qmost_gen_bpm (
    cpl_imagelist *flats,
    cpl_propertylist *qclist,
    float lthr,
    float hthr,
    float badfrac,
    cpl_mask **bpm);

CPL_END_DECLS

#endif  /* QMOST_LINEAR_H */
