/*
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef QMOST_TESTUTIL_H
#define QMOST_TESTUTIL_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_testutil  Unit test utility macros.
 *
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include <cpl.h>
#include <string.h>
#include <math.h>

CPL_BEGIN_DECLS

/*----------------------------------------------------------------------------*/
/**
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test that a > b, fail test if false.
 *
 * This is the same as cpl_test_le with the arguments reversed but
 * avoids the confusing inverted logic.
 *
 * @param a         The first expression.
 * @param b         The second expression.
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_gt(a, b) cpl_test_leq(b, a)

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test that a >= b, fail test if false.
 *
 * This is the same as cpl_test_lt with the arguments reversed but
 * avoids the confusing inverted logic.
 *
 * @param a         The first expression.
 * @param b         The second expression.
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_ge(a, b) cpl_test_lt(b, a)

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test if a float array is equal to a scalar with a given
 *        absolute tolerance.
 *
 * @param arr       Pointer (float *) to the array to test.
 * @param n         Number of elements in the array.
 * @param scalar    The scalar to test against.
 * @param tolerance The tolerance (not negative).
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_array_float_abs(arr, n, scalar, tolerance) { \
    cpl_array *testarr = cpl_array_wrap_float((arr), (n));      \
    cpl_array *answer = cpl_array_new((n), CPL_TYPE_FLOAT);     \
    cpl_array_fill_window_float(answer, 0, (n), (scalar));      \
    cpl_test_array_abs(testarr, answer, (tolerance));           \
    cpl_array_unwrap(testarr);                                  \
    cpl_array_delete(answer);                                   \
}

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test if a double array is equal to a scalar with a given
 *        absolute tolerance.
 *
 * @param arr       Pointer (double *) to the array to test.
 * @param n         Number of elements in the array.
 * @param scalar    The scalar to test against.
 * @param tolerance The tolerance (not negative).
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_array_double_abs(arr, n, scalar, tolerance) {        \
    cpl_array *testarr = cpl_array_wrap_double((arr), (n));             \
    cpl_array *answer = cpl_array_new((n), CPL_TYPE_DOUBLE);            \
    cpl_array_fill_window_double(answer, 0, (n), (scalar));             \
    cpl_test_array_abs(testarr, answer, (tolerance));                   \
    cpl_array_unwrap(testarr);                                          \
    cpl_array_delete(answer);                                           \
}

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test if all of the elements of two double arrays are equal to
 *        within a given absolute tolerance.
 *
 * @param arr1      Pointer (double *) to the first array.
 * @param arr2      Pointer (double *) to the second array.
 * @param n         Number of elements in the arrays.
 * @param tolerance The tolerance (not negative).
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_arrays_double_abs(arr1, arr2, n, tolerance) { \
    cpl_array *first = cpl_array_wrap_double((arr1), (n));       \
    cpl_array *second = cpl_array_wrap_double((arr2), (n));      \
    cpl_test_array_abs(first, second, (tolerance));             \
    cpl_array_unwrap(first);                                    \
    cpl_array_unwrap(second);                                   \
}

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test if all of the elements of two float arrays are equal to
 *        within a given absolute tolerance.
 *
 * @param arr1      Pointer (float *) to the first array.
 * @param arr2      Pointer (float *) to the second array.
 * @param n         Number of elements in the arrays.
 * @param tolerance The tolerance (not negative).
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_arrays_float_abs(arr1, arr2, n, tolerance) { \
    cpl_array *first = cpl_array_wrap_float((arr1), (n));       \
    cpl_array *second = cpl_array_wrap_float((arr2), (n));      \
    cpl_test_array_abs(first, second, (tolerance));             \
    cpl_array_unwrap(first);                                    \
    cpl_array_unwrap(second);                                   \
}

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test if all of the elements of two int arrays are equal to
 *        within a given absolute tolerance.
 *
 * @param arr1      Pointer (int *) to the first array.
 * @param arr2      Pointer (int *) to the second array.
 * @param n         Number of elements in the arrays.
 * @param tolerance The tolerance (not negative).
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_arrays_int_abs(arr1, arr2, n, tolerance) {   \
    cpl_array *first = cpl_array_wrap_int((arr1), (n));         \
    cpl_array *second = cpl_array_wrap_int((arr2), (n));        \
    cpl_test_array_abs(first, second, (tolerance));             \
    cpl_array_unwrap(first);                                    \
    cpl_array_unwrap(second);                                   \
}

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test image against float scalar.

 * Test if all of the elements in an image are equal to a float value
 * within a given (absolute) tolerance and/or are flagged bad.
 *
 * @param image      The image.
 * @param scalar     The scalar.
 * @param bad        Should pixels be bad?
 * @param tolerance  Tolerance (not negative).
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_image_float_abs(image, scalar, bad, tolerance) {     \
    cpl_image *answer = cpl_image_new(cpl_image_get_size_x((image)),    \
                                      cpl_image_get_size_y((image)),    \
                                      CPL_TYPE_FLOAT);                  \
                                                                        \
    cpl_image_fill_window(answer,                                       \
                          1,                                            \
                          1,                                            \
                          cpl_image_get_size_x((image)),                \
                          cpl_image_get_size_y((image)),                \
                          (scalar));                                    \
                                                                        \
    if((bad)) {                                                         \
        cpl_mask *mask = cpl_image_get_bpm(answer);                     \
        cpl_mask_not(mask);                                             \
    }                                                                   \
                                                                        \
    cpl_test_image_abs((image), answer, (tolerance));                   \
    cpl_image_delete(answer);                                           \
}

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test if all of the elements in a mask are set/unset.
 *
 * @param mask      Pointer (cpl_mask *) to the mask.
 * @param set       Boolean value (true or false) to test against.
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_mask_eq(mask, set) {                                 \
    cpl_mask *answer = cpl_mask_new(cpl_mask_get_size_x((mask)),        \
                                    cpl_mask_get_size_y((mask)));       \
    if((set)) {                                                         \
        cpl_mask_not(answer);                                           \
    }                                                                   \
    cpl_test_eq_mask((mask), answer);                                   \
    cpl_mask_delete(answer);                                            \
}

/*----------------------------------------------------------------------------*/
/**
 * @hideinitializer
 *
 * @brief Test if a of the elements in a subset of a mask are set/unset.
 *
 * @param mask      Pointer (cpl_mask *) to the mask.
 * @param xl        Lower left x coordinate (number from 1 per FITS).
 * @param yl        Lower left y coordinate.
 * @param xh        Upper right x coordinate.
 * @param yh        Upper right y coordinate.
 * @param set       Boolean value (true or false) to test against.
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/
#define qmost_test_mask_subset_eq(mask, xl, yl, xh, yh, set) {          \
    cpl_mask *subset = cpl_mask_extract((mask), (xl), (yl), (xh), (yh)); \
    qmost_test_mask_eq(subset, (set));                                  \
    cpl_mask_delete(subset);                                            \
}

/*----------------------------------------------------------------------------*/
/**
 *                              Functions
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Fill an image with random Gaussian noise.
 *
 * The image is filled with rounded (to integer) Gaussian deviates
 * drawn from N(mean, sigma).  This is needed for some tests of
 * statistical functions where we can't use uniform deviates.
 *
 * @param image     The image to fill.
 * @param mean      Gaussian mean.
 * @param sigma     Gaussian noise standard deviation.
 *
 * @return nothing
 *
 * @author Jonathan Irwin, CASU
 */
/*----------------------------------------------------------------------------*/

static inline void qmost_test_fill_noise_rounded_gauss (
    cpl_image *image,
    double mean,
    double sigma)
{
    float *image_data = NULL;
    int x, y, nx, ny;
    double ua, ub, rsq, fac;

    image_data = cpl_image_get_data_float(image);
    if(image_data == NULL) {
        return;
    }

    nx = cpl_image_get_size_x(image);
    ny = cpl_image_get_size_y(image);

    for(y = 0; y < ny; y++) {
        for(x = 0; x < nx; x += 2) {
            /* This is a bad RNG with well documented issues, but
             * since we're limited to the C library and it's only a
             * unit test, it'll probably do. */
            do {
                ua = 2.0 * (((double) rand()) / RAND_MAX) - 1.0;
                ub = 2.0 * (((double) rand()) / RAND_MAX) - 1.0;
                
                rsq = ua*ua + ub*ub;
            } while(rsq <= 0.0 || rsq >= 1.0);

            fac = sqrt(-2.0 * log(rsq) / rsq);
                
            image_data[y*nx+x] = rint(mean + sigma * ua * fac);
            if(x+1 < nx) {
                image_data[y*nx+x+1] = rint(mean + sigma * ub * fac);
            }
        }
    }
}

CPL_END_DECLS

#endif  /* QMOST_TESTUTIL_H */
