/* 
 * This file is part of the QMOST Pipeline
 * Copyright (C) 2002-2022 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*----------------------------------------------------------------------------*/
/*
 *                              Includes
 */
/*----------------------------------------------------------------------------*/

#include "qmost_ccdproc.h"
#include "qmost_constants.h"
#include "qmost_fibtab.h"
#include "qmost_dfs.h"
#include "qmost_model_psf.h"
#include "qmost_pfits.h"
#include "qmost_scattered.h"
#include "qmost_traceinfo.h"
#include "qmost_utils.h"

#include <cpl.h>

/*----------------------------------------------------------------------------*/
/*
 *                              Defines
 */
/*----------------------------------------------------------------------------*/

#define RECIPE_NAME      "qmost_psf_analyse"
#define CONTEXT          "qmost."RECIPE_NAME

/*----------------------------------------------------------------------------*/
/*
 *                          Static variables
 */
/*----------------------------------------------------------------------------*/

static const char qmost_psf_analyse_description[] =
    "Combine individual fibre flat images into a master fibre flat.\n"
    "Use the master fibre flat to measure the spatial profile at\n"
    "each spectral coordinate along each fibre, and record the\n"
    "results in a 3-dimensional map.\n\n"
    "The following files can be specified in the SOF:\n\n"
    "Description                 Req/Opt? Tag\n"
    "--------------------------- -------- --------------------\n"
    "Daytime fibre flat frames   Required " QMOST_RAW_FIBRE_FLAT_DAY "\n"
    "Master bias frame           Optional " QMOST_PRO_MASTER_BIAS "\n"
    "Master dark frame           Optional " QMOST_PRO_MASTER_DARK "\n"
    "Master detector flat        Optional " QMOST_PRO_MASTER_DETECTOR_FLAT "\n"
    "Master bad pixel mask       Optional " QMOST_CALIB_MASTER_BPM "\n"
    "Master linearity table      Optional " QMOST_PRO_LINEARITY "\n"
    "Master fibre trace table    Required " QMOST_PRO_FIBRE_TRACE "\n"
    "Master fibre mask           Required " QMOST_PRO_FIBRE_MASK "\n"
    "\n"
    "Outputs:\n\n"
    "Description                 Tag\n"
    "--------------------------- -----------------\n"
    "Master PSF file             " QMOST_PRO_MASTER_PSF "\n"
    "\n"
    "Optional processed outputs if keep=true:\n\n"
    "Description                 Tag\n"
    "--------------------------- -----------------\n"
    "Processed fibre flat image  " QMOST_PROC_FIBRE_FLAT "\n\n"
    "All output filenames are machine generated based on the spectrograph\n"
    "and tag being processed, and are named:\n"
    "  QMOST_tag_spec.fits\n"
    "where tag is replaced by the tag from the tables above, and spec is\n"
    "one of HRS, LRS-A or LRS-B.  So, for example, the master PSF file\n"
    "for HRS would be QMOST_" QMOST_PRO_MASTER_PSF "_HRS.fits\n"
    ;

/* Standard CPL recipe definition */
cpl_recipe_define(qmost_psf_analyse,
                  QMOST_BINARY_VERSION,
                  "Jonathan Irwin",
                  "https://support.eso.org",
                  "2022",
                  "Calculate the fibre PSF from a set of fibre flats.",
                  qmost_psf_analyse_description);

/*----------------------------------------------------------------------------*/
/**
 * @defgroup qmost_psf_analyse    qmost_psf_analyse
 *
 * @brief    Calculate the fibre PSF from a set of fibre flats.
 *
 * @par Name:
 *   qmost_psf_analyse
 * @par Purpose:
 *   Combine individual fibre flat images into a master fibre flat.
 *   Use the master fibre flat to measure the spatial profile at
 *   each spectral coordinate along each fibre, and record the
 *   results in a 3-dimensional map.
 * @par Type:
 *   Spectroscopic calibration
 * @par Parameters:
 *   - @b keep (bool, default false):
 *     If this flag is set, optional products generated during
 *     execution of the recipe are also saved.  In the case of
 *     qmost_psf_analyse, this saves the processed fibre flat image.
 *   - @b ccdproc.swapx (string, default 100010101):
 *     String of 0 or 1 specifying for each arm of each spectrograph
 *     if we should flip the x axis to correct the wavelength order of
 *     the spectral axis.
 *   - @b imcombine.combtype (int, default 1):
 *     Determines the type of combination that is done to form the
 *     stacked fibre flat frame when stacking internal facility fibre
 *     flats (daytime or night-time).  Can take the following values:
 *       - (1): The output pixels are kappa sigma clipped means of
 *              the input pixels.
 *       - (2): The output pixels are medians of the input pixels.
 *   - @b imcombine.scaletype (int, default 2):
 *     Scaling to use when combining multiple images to form the
 *     stacked fibre flat frame when stacking internal facility fibre
 *     flats (daytime or night-time).  Can take the following values:
 *       - (0): No biasing or scaling is done.
 *       - (1): An additive offset is applied to remove any
 *              difference in background levels.
 *       - (2): A multiplicative scaling is applied to remove any
 *              difference in background levels.
 *       - (3): Images are scaled by their relative exposure times
 *              and an additive offset is applied to remove any
 *              remaining difference in background levels.
 *   - @b imcombine.xrej (bool, default false):
 *     If set, then an extra rejection cycle will be run (internal
 *     fibre flats only).
 *   - @b imcombine.thresh (float, default 5.0):
 *     The rejection threshold for clipping during the combination in
 *     units of the background sigma (internalfibre flats only).
 *   - @b scattered.enable (bool, default true):
 *     Enable scattered light removal.
 *   - @b scattered.nbsize (int, default 256):
 *     Size of the background smoothing cells for determining
 *     scattered light in pixels.
 *   - @b psf.hwidth (int, default 5)
 *     The maximum extent of the PSF measurement region relative to
 *     the fibre trace centre in pixels. The resulting PSF will have
 *     (2*psf.hwidth+1)*psf.subsample samples.
 *   - @b psf.subsample (int, default 5)
 *     A sub-sampling factor in the spatial axis, specifying the
 *     number of PSF samples per spatial pixel.
 *   - @b psf.sblock (int, default 100)
 *     The number of spectral pixels to block average when measuring
 *     the PSF.
 * @par Input File Types:
 *   The following files can be specified in the SOF.  The word in
 *   bold is the tag (DO CATG keyword value).
 *    - @b FIBRE_FLAT_DAY: Raw daytime internal facility fibre flats.
 *    - @b MASTER_BIAS (optional): Master bias frame.
 *    - @b MASTER_DARK (optional): Master dark frame.
 *    - @b MASTER_DETECTOR_FLAT (optional): Master detector flat.
 *    - @b MASTER_BPM (optional): Master bad pixel mask.
 *    - @b LINEARITY (optional): Master linearity table.
 *    - @b FIBRE_TRACE (required): Trace table.
 *    - @b FIBRE_MASK (required): Fibre mask.
 * @par Output Products:
 *   - The following product files are generated by this recipe.  The
 *     word in bold is the tag (PRO CATG keyword value).
 *     - @b MASTER_PSF: The master PSF file.
 *
 *   - The following intermediate product files are optional and are
 *     only emitted if keep=true is specified in the recipe parameters.
 *     - @b PROC_FIBRE_FLAT: Processed 2D fibre flat image.  Can be
 *          used to assess the quality of the initial CCD processing,
 *          such as bias, dark, detector flat field correction,
 *          scattered light correction.
 *
 *   - All output filenames are machine generated based on the
 *     spectrograph and tag being processed, and are named:
 *       @c QMOST_tag_spec.fits
 *     where tag is replaced by the tag from the tables above, and
 *     @c spec is one of HRS, LRS-A or LRS-B.  So, for example, the
 *     master PSF file for HRS would be @c
 *     QMOST_MASTER_PSF_HRS.fits
 * @par Output QC Parameters:
 *   This section contains a brief description of each QC parameter.
 *   Please refer to the Quality Control parameter dictionary
 *   ESO-DFS-DIC.QMOST_QC for more detailed descriptions of the QC
 *   parameters.
 *   - <b>NUM SAT</b>: The number of saturated pixels in the raw
 *     image.
 *   - <b>OS MED AMPn</b> (ADU): The median bias level in the overscan
 *     region of amplifier n.
 *   - <b>OS RMS AMPn</b> (ADU): The RMS noise in the overscan region
 *     of amplifier n.
 *   - <b>IMCOMBINE MEAN</b> (ADU): The mean residual background level
 *     in the illuminated region of the overscan subtracted frames
 *     that were combined.
 *   - <b>IMCOMBINE RMS</b> (ADU): The RMS of the background levels
 *     over the frames that were combined, as a measure of how
 *     consistent the background levels were.
 *   - <b>IMCOMBINE MIN</b> (ADU): The minimum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE MAX</b> (ADU): The maximum background level of the
 *     frames that were combined.
 *   - <b>IMCOMBINE NOISE MEAN</b> (ADU): The average RMS of the
 *     background in the frames that were combined.
 *   - <b>IMCOMBINE NUM COMBINED</b>: The number of frames that were
 *     combined, after rejection of any bad frames.
 *   - <b>IMCOMBINE NUM INPUTS</b>: The number of frames that were
 *     passed to the combination routine, before rejection of any bad
 *     frames.
 *   - <b>IMCOMBINE NUM REJECTED</b>: The total number of pixels
 *     rejected during combination.
 *   - <b>SCATTL MED</b> (ADU): The median scattered light level on
 *     the detector outside the region illuminated by the fibres.
 *   - <b>SCATTL RMS</b> (ADU): The robustly-estimated RMS of the
 *     scattered light level on the detector outside the region
 *     illuminated by the fibres.
 *   - <b>SCATTL RESID MED</b> (ADU): The median residual scattered
 *     light background remaining after subtraction.
 *   - <b>SCATTL RESID RMS</b> (ADU): The RMS residual of the
 *     scattered light background remaining after subtraction.
 *   - <b>PSF FWHM MED</b> (pix): The ensemble median FWHM of the
 *     PSFs.
 *   - <b>PSF FWHM RMS</b> (pix): The ensemble robustly-estimated RMS
 *     FWHM of the PSFs.
 *   - <b>PSF FWHM MIN</b> (pix): The ensemble minimum FWHM of the
 *     PSFs.
 *   - <b>PSF FWHM MINSPC</b>: The fibre with the minimum FWHM.
 *   - <b>PSF FWHM MAX</b> (pix): The ensemble maximum FWHM of the
 *     PSFs.
 *   - <b>PSF FWHM MAXSPC</b>: The fibre with the maximum FWHM.
 *   - <b>PSF FWHM POS</b> (pix): The coordinate along the spectral
 *     axis at which the PSF FWHM was measured.
 * @par Fatal Error Conditions:
 *   - NULL input frameset.
 *   - Input frameset headers incorrect meaning that RAW and CALIB
 *     frames cannot be distinguished.
 *   - No raw fibre flat frames in the input frameset.
 *   - Mandatory calibration images/tables not specified in SOF or
 *     unreadable.
 *   - Inability to save output products.
 *   - Input files have missing extensions.
 *   - A dummy extension in any of the calibration files is used to
 *     process an active detector in the raw file.
 *   - If the trace table for an active detector is empty.
 * @par Non-Fatal Error Conditions:
 *   - No bad pixel mask (all pixels assumed to be good).
 * @par Conditions Leading to Dummy Products:
 *   - The detector for the current image extension is disabled.
 * @par Functional Diagram:
 * @dot
 * digraph {
 *   node [fontname="monospace" fontsize=10]
 *   node [fillcolor="#ffdddd" height=0.1 style="filled"]
 * 
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" -> qmost_ccdproc_and_combine [style="dashed"];
 *   FIBRE_MASK -> qmost_scattered;
 *   FIBRE_MASK -> qmost_scattered_qc;
 *   FIBRE_TRACE -> qmost_model_psf_full;
 *
 *   subgraph cluster_primary_data_flow {
 *     style="invis";
 *
 *     FIBRE_FLAT_DAY -> qmost_ccdproc_and_combine;
 *
 *     qmost_ccdproc_and_combine -> qmost_scattered_qc;
 *     qmost_scattered_qc -> qmost_scattered;
 *     qmost_scattered -> qmost_model_psf_full;
 *     qmost_model_psf_full -> qmost_psfstats;
 *
 *     qmost_psfstats -> MASTER_PSF;
 *   }
 *
 *   qmost_scattered -> PROC_FIBRE_FLAT;
 *
 *   qmost_ccdproc_and_combine -> QC1;
 *   qmost_scattered_qc -> QC1;
 *   qmost_psfstats -> QC1;
 * 
 *   FIBRE_FLAT_DAY [shape="box" fillcolor="#eeeeee"]
 *   "MASTER_BIAS\nMASTER_DARK\nMASTER_DETECTOR_FLAT\nMASTER_BPM\nLINEARITY" [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_TRACE [shape="box" fillcolor="#fff5ce"]
 *   FIBRE_MASK [shape="box" fillcolor="#fff5ce"]
 *   MASTER_PSF [shape="box" fillcolor="#fff5ce"]
 *   PROC_FIBRE_FLAT [shape="box" fillcolor="#ffffff"]
 *   QC1 [fillcolor="#ffffff"]
 * }
 * @enddot
 */
/*----------------------------------------------------------------------------*/

/**@{*/

/*----------------------------------------------------------------------------*/
/*
 *                              Functions code
 */
/*----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @brief    Interpret the command line options and execute the data processing
 *
 * @param    frameset   the frames list
 * @param    parlist    the parameters list
 *
 * @return   CPL_ERROR_NONE(0) if everything is ok
 *
 */
/*----------------------------------------------------------------------------*/
static int qmost_psf_analyse(
    cpl_frameset            *frameset,
    const cpl_parameterlist *parlist)
{
    const cpl_frame *master_bias_frame = NULL;
    const cpl_frame *master_dark_frame = NULL;
    const cpl_frame *master_detflat_frame = NULL;
    const cpl_frame *master_bpm_frame = NULL;
    const cpl_frame *master_lintab_frame = NULL;
    const cpl_frame *fibre_trace_frame = NULL;
    const cpl_frame *fibre_mask_frame = NULL;

    const cpl_frame *proc_fibre_flat_frame = NULL;
    const char *proc_fibre_flat_filename = NULL;

    const cpl_frame *fibre_psf_frame = NULL;
    const char *fibre_psf_filename = NULL;

    cpl_frameset *fibflats_raw = NULL;
    int nin, iin, nff;

    const cpl_frame *this_frame;
    const char *this_tag;

    cpl_frame *fibflat_ref;
    const char *fibflat_ref_filename;

    const cpl_parameter *par;
    int keep = 0;
    const char *swapx_table = QMOST_DEFAULT_SWAPX_TABLE;
    int combtype = QMOST_MEANCALC;
    int scaletype = 2;
    int xrej = 0;
    double combthresh = 5.0;
    int scattered_enable = 1;
    int scattered_nbsize = 256;
    int profwidth = 5;
    int subsample = 5;
    int sblock = 100;

    cpl_size raw_extension;
    int spec, arm, detlive;
    const char *arm_extname;
    char *extname = NULL;

    cpl_mask *master_bpm = NULL;
    cpl_image *master_bias_img = NULL;
    cpl_image *master_bias_var = NULL;
    cpl_image *master_dark_img = NULL;
    cpl_image *master_dark_var = NULL;
    cpl_image *master_detflat_img = NULL;
    cpl_image *master_detflat_var = NULL;
    cpl_table *master_lintab = NULL;

    cpl_image *fibre_flat_img = NULL;
    cpl_image *fibre_flat_var = NULL;
    cpl_propertylist *fibre_flat_hdr = NULL;

    cpl_propertylist *qclist = NULL;

    cpl_table *fibinfo_tbl = NULL;
    cpl_propertylist *fibinfo_hdr = NULL;

    cpl_table *trace_tbl = NULL;
    cpl_propertylist *trace_hdr = NULL;
    cpl_mask *fibre_mask = NULL;

    cpl_imagelist *fibre_psf_img = NULL;
    cpl_imagelist *fibre_psf_var = NULL;

#undef TIDY
#define TIDY                                            \
    if(fibflats_raw) {                                  \
        cpl_frameset_delete(fibflats_raw);              \
        fibflats_raw = NULL;                            \
    }                                                   \
    if(fibinfo_tbl) {                                   \
        cpl_table_delete(fibinfo_tbl);                  \
        fibinfo_tbl = NULL;                             \
    }                                                   \
    if(fibinfo_hdr) {                                   \
        cpl_propertylist_delete(fibinfo_hdr);           \
        fibinfo_hdr = NULL;                             \
    }                                                   \
    if(master_bpm) {                                    \
        cpl_mask_delete(master_bpm);                    \
        master_bpm = NULL;                              \
    }                                                   \
    if(master_bias_img) {                               \
        cpl_image_delete(master_bias_img);              \
        master_bias_img = NULL;                         \
    }                                                   \
    if(master_bias_var) {                               \
        cpl_image_delete(master_bias_var);              \
        master_bias_var = NULL;                         \
    }                                                   \
    if(master_dark_img) {                               \
        cpl_image_delete(master_dark_img);              \
        master_dark_img = NULL;                         \
    }                                                   \
    if(master_dark_var) {                               \
        cpl_image_delete(master_dark_var);              \
        master_dark_var = NULL;                         \
    }                                                   \
    if(master_detflat_img) {                            \
        cpl_image_delete(master_detflat_img);           \
        master_detflat_img = NULL;                      \
    }                                                   \
    if(master_detflat_var) {                            \
        cpl_image_delete(master_detflat_var);           \
        master_detflat_var = NULL;                      \
    }                                                   \
    if(master_lintab) {                                 \
        cpl_table_delete(master_lintab);                \
        master_lintab = NULL;                           \
    }                                                   \
    if(fibre_flat_img) {                                \
        cpl_image_delete(fibre_flat_img);               \
        fibre_flat_img = NULL;                          \
    }                                                   \
    if(fibre_flat_var) {                                \
        cpl_image_delete(fibre_flat_var);               \
        fibre_flat_var = NULL;                          \
    }                                                   \
    if(fibre_flat_hdr) {                                \
        cpl_propertylist_delete(fibre_flat_hdr);        \
        fibre_flat_hdr = NULL;                          \
    }                                                   \
    if(qclist) {                                        \
        cpl_propertylist_delete(qclist);                \
        qclist = NULL;                                  \
    }                                                   \
    if(trace_tbl) {                                     \
        cpl_table_delete(trace_tbl);                    \
        trace_tbl = NULL;                               \
    }                                                   \
    if(trace_hdr) {                                     \
        cpl_propertylist_delete(trace_hdr);             \
        trace_hdr = NULL;                               \
    }                                                   \
    if(fibre_mask) {                                    \
        cpl_mask_delete(fibre_mask);                    \
        fibre_mask = NULL;                              \
    }                                                   \
    if(fibre_psf_img) {                                 \
        cpl_imagelist_delete(fibre_psf_img);            \
        fibre_psf_img = NULL;                           \
    }                                                   \
    if(fibre_psf_var) {                                 \
        cpl_imagelist_delete(fibre_psf_var);            \
        fibre_psf_var = NULL;                           \
    }                                                   \
    if(extname) {                                       \
        cpl_free(extname);                              \
        extname = NULL;                                 \
    }

    /* Classify frames */
    if(qmost_check_and_set_groups(frameset) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't classify input frames");
    }

    /* Master detector-level calibrations, all optional */
    master_bias_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_BIAS);
    master_dark_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_MASTER_DARK);
    master_detflat_frame = cpl_frameset_find_const(frameset,
                                                   QMOST_PRO_MASTER_DETECTOR_FLAT);
    master_bpm_frame = cpl_frameset_find_const(frameset,
                                               QMOST_CALIB_MASTER_BPM);
    master_lintab_frame = cpl_frameset_find_const(frameset,
                                                  QMOST_PRO_LINEARITY);

    /* Trace table and fibre mask, required */
    fibre_trace_frame = cpl_frameset_find_const(frameset,
                                                QMOST_PRO_FIBRE_TRACE);
    if(fibre_trace_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_TRACE);
    }

    fibre_mask_frame = cpl_frameset_find_const(frameset,
                                               QMOST_PRO_FIBRE_MASK);
    if(fibre_mask_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged %s",
                                     QMOST_PRO_FIBRE_MASK);
    }

    /* Create frameset of input raw fibre flats */
    fibflats_raw = cpl_frameset_new();  /* can't fail? */

    nin = cpl_frameset_get_size(frameset);

    for(iin = 0; iin < nin; iin++) {
        this_frame = cpl_frameset_get_position_const(frameset, iin);
        this_tag = cpl_frame_get_tag(this_frame);
        if(!strcmp(this_tag, QMOST_RAW_FIBRE_FLAT_DAY)) {
            cpl_frameset_insert(fibflats_raw, cpl_frame_duplicate(this_frame));
        }
    }

    /* Check we have some */
    nff = cpl_frameset_get_size(fibflats_raw);
    if(nff < 1) {
        TIDY;
        return cpl_error_set_message(cpl_func,
                                     CPL_ERROR_DATA_NOT_FOUND,
                                     "SOF does not have an image tagged "
                                     "%s",
                                     QMOST_RAW_FIBRE_FLAT_DAY);
    }

    /* Switch to save processed products */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".keep");
    if(par != NULL)
        keep = cpl_parameter_get_bool(par);

    /* Retrieve ccdproc parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".ccdproc.swapx");
    if(par != NULL)
        swapx_table = cpl_parameter_get_string(par);

    /* Retrieve imcombine parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.combtype");
    if(par != NULL)
        combtype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.scaletype");
    if(par != NULL)
        scaletype = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.xrej");
    if(par != NULL)
        xrej = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".imcombine.thresh");
    if(par != NULL)
        combthresh = cpl_parameter_get_double(par);

    /* Retrieve scattered light removal parameters */
    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.enable");
    if(par != NULL)
        scattered_enable = cpl_parameter_get_bool(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".scattered.nbsize");
    if(par != NULL)
        scattered_nbsize = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".psf.hwidth");
    if(par != NULL)
        profwidth = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".psf.subsample");
    if(par != NULL)
        subsample = cpl_parameter_get_int(par);

    par = cpl_parameterlist_find_const(parlist,
                                       RECIPE_NAME".psf.sblock");
    if(par != NULL)
        sblock = cpl_parameter_get_int(par);

    /* Get first fibre flat to use as reference */
    fibflat_ref = cpl_frameset_get_position(fibflats_raw, 0);
    if(fibflat_ref == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get first fibre flat "
                                     "from SOF");
    }

    /* Get filename of first fibre flat */
    fibflat_ref_filename = cpl_frame_get_filename(fibflat_ref);
    if(fibflat_ref_filename == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't get filename for "
                                     "first fibre flat frame");
    }

    /* Extract the primary FITS header of the reference */
    fibre_flat_hdr = cpl_propertylist_load(fibflat_ref_filename,
                                           0);
    if(fibre_flat_hdr == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FITS primary header "
                                     "from first input file");
    }

    /* Figure out which spectrograph we're processing */
    if(qmost_pfits_get_spectrograph(fibre_flat_hdr,
                                    &spec) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't determine which "
                                     "spectrograph we're processing");
    }

    cpl_propertylist_delete(fibre_flat_hdr);
    fibre_flat_hdr = NULL;

    /* Get fibinfo */
    if(qmost_fibtabload(fibflat_ref,
                        spec,
                        &fibinfo_tbl,
                        &fibinfo_hdr) != CPL_ERROR_NONE) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't load FIBINFO table "
                                     "from first input file");
    }

    /* Create outputs */
    fibre_psf_frame = qmost_dfs_setup_product_default(
        frameset,
        parlist,
        RECIPE_NAME,
        spec,
        QMOST_PRO_MASTER_PSF,
        CPL_FRAME_TYPE_IMAGE,
        NULL);
    if(fibre_psf_frame == NULL) {
        TIDY;
        return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                     "couldn't set up output product "
                                     "file for %s",
                                     QMOST_PRO_MASTER_PSF);
    }

    fibre_psf_filename = cpl_frame_get_filename(fibre_psf_frame);

    if(keep) {
        proc_fibre_flat_frame = qmost_dfs_setup_product_default(
            frameset,
            parlist,
            RECIPE_NAME,
            spec,
            QMOST_PROC_FIBRE_FLAT,
            CPL_FRAME_TYPE_IMAGE,
            NULL);
        if(proc_fibre_flat_frame == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't set up output product "
                                         "file for %s",
                                         QMOST_PROC_FIBRE_FLAT);
        }

        proc_fibre_flat_filename = 
            cpl_frame_get_filename(proc_fibre_flat_frame);
    }

    for(raw_extension = 1; raw_extension <= 3; raw_extension++) {
        /* Extract the FITS header of the reference */
        fibre_flat_hdr = cpl_propertylist_load(
            fibflat_ref_filename,
            raw_extension);
        if(fibre_flat_hdr == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load FITS extension "
                                         "%lld header from first input file",
                                         raw_extension);
        }

        /* Check the reference to find out what arm we're processing */
        if(qmost_pfits_get_arm(fibre_flat_hdr,
                               &arm) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine which "
                                         "arm we're processing");
        }

        arm_extname = qmost_pfits_get_extname(arm);
        if(arm_extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't determine EXTNAME "
                                         "for arm %d", arm);
        }

        /* Is detector live? */
        if(qmost_pfits_get_detlive(fibre_flat_hdr,
                                   &detlive) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not read live flag for "
                                         "extension %lld",
                                         raw_extension);
        }

        if(!detlive) {
            cpl_msg_info(cpl_func,
                         "Writing dummy extension %lld = %s",
                         raw_extension, arm_extname);

            /* Detector is not live, write out dummy HDUs to products
             * and move on. */
            if(proc_fibre_flat_frame != NULL) {
                if(qmost_dfs_save_image_and_var(proc_fibre_flat_frame,
                                                fibre_flat_hdr,
                                                arm_extname,
                                                NULL,
                                                NULL,
                                                NULL,
                                                CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                    TIDY;
                    return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                                 "couldn't save dummy fibre "
                                                 "flat frame %s[%s]",
                                                 proc_fibre_flat_filename,
                                                 arm_extname);
                }
            }

            extname = cpl_sprintf("prof_%s", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format PSF "
                                             "EXTNAME string");
            }
            
            if(qmost_dfs_save_imagelist_extension(fibre_psf_frame,
                                                  fibre_flat_hdr,
                                                  extname,
                                                  NULL,
                                                  NULL,
                                                  CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy "
                                             "PSF %s[%s]",
                                             fibre_psf_filename,
                                             arm_extname);
            }
            
            cpl_free(extname);
            extname = NULL;
            
            extname = cpl_sprintf("profvar_%s", arm_extname);
            if(extname == NULL) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "could not format PSF variance "
                                             "EXTNAME string");
            }
            
            if(qmost_dfs_save_imagelist_extension(fibre_psf_frame,
                                                  fibre_flat_hdr,
                                                  extname,
                                                  NULL,
                                                  NULL,
                                                  CPL_TYPE_UCHAR) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save dummy "
                                             "PSF variance %s[%s]",
                                             fibre_psf_filename,
                                             arm_extname);
            }
            
            cpl_free(extname);
            extname = NULL;

            cpl_propertylist_delete(fibre_flat_hdr);
            fibre_flat_hdr = NULL;

            continue;
        }

        /* Load all masters using the correct EXTNAME for the arm
           we're processing in case the mapping of extension number to
           arm could have changed. */
        if(master_bias_frame != NULL) {
            if(qmost_load_master_image_and_var(master_bias_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_bias_img,
                                               &master_bias_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master bias "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_bias_img = NULL;
            master_bias_var = NULL;
        }
    
        if(master_dark_frame != NULL) {
            if(qmost_load_master_image_and_var(master_dark_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_dark_img,
                                               &master_dark_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master dark "
                                             "frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_dark_img = NULL;
            master_dark_var = NULL;
        }

        if(master_detflat_frame != NULL) {
            if(qmost_load_master_image_and_var(master_detflat_frame,
                                               arm_extname,
                                               CPL_TYPE_FLOAT,
                                               &master_detflat_img,
                                               &master_detflat_var,
                                               NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master detector "
                                             "flat frame extension %s",
                                             arm_extname);
            }
        }
        else {
            master_detflat_img = NULL;
            master_detflat_var = NULL;
        }

        if(master_bpm_frame != NULL) {
            if(qmost_load_master_mask(master_bpm_frame,
                                      arm_extname,
                                      &master_bpm,
                                      NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master BPM");
            }
        }

        if(master_lintab_frame != NULL) {
            if(qmost_load_master_table(master_lintab_frame,
                                       arm_extname,
                                       &master_lintab,
                                       NULL) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't load master "
                                             "linearity table");
            }
        }

        /* Create QC parameter list */
        qclist = cpl_propertylist_new();

        /* CCD processing and stacking */
        if(qmost_ccdproc_and_combine(fibflats_raw,
                                     raw_extension,
                                     master_bpm,
                                     master_bias_img,
                                     master_bias_var,
                                     master_dark_img,
                                     master_dark_var,
                                     master_detflat_img,
                                     master_detflat_var,
                                     1,
                                     swapx_table,
                                     1,
                                     master_lintab,
                                     QMOST_LIN_AFTER_BIAS,
                                     combtype,
                                     scaletype,
                                     1,  /* fibre spectra */
                                     xrej,
                                     combthresh,
                                     0,
                                     0.0,
                                     0.0,
                                     &fibre_flat_img,
                                     &fibre_flat_var,
                                     qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "basic CCD processing failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(master_bias_img != NULL) {
            cpl_image_delete(master_bias_img);
            master_bias_img = NULL;
        }

        if(master_bias_var != NULL) {
            cpl_image_delete(master_bias_var);
            master_bias_var = NULL;
        }

        if(master_dark_img != NULL) {
            cpl_image_delete(master_dark_img);
            master_dark_img = NULL;
        }

        if(master_dark_var != NULL) {
            cpl_image_delete(master_dark_var);
            master_dark_var = NULL;
        }

        if(master_detflat_img != NULL) {
            cpl_image_delete(master_detflat_img);
            master_detflat_img = NULL;
        }

        if(master_detflat_var != NULL) {
            cpl_image_delete(master_detflat_var);
            master_detflat_var = NULL;
        }

        if(master_bpm != NULL) {
            cpl_mask_delete(master_bpm);
            master_bpm = NULL;
        }

        if(master_lintab != NULL) {
            cpl_table_delete(master_lintab);
            master_lintab = NULL;
        }

        /* Get trace table and fibre mask */
        extname = cpl_sprintf("trace_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format trace "
                                         "EXTNAME string");
        }
        
        if(qmost_load_master_table(fibre_trace_frame,
                                   extname,
                                   &trace_tbl,
                                   &trace_hdr) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load trace");
        }
        
        cpl_free(extname);
        extname = NULL;

        extname = cpl_sprintf("mask_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format fibre mask "
                                         "EXTNAME string");
        }

        if(qmost_load_master_mask(fibre_mask_frame,
                                  extname,
                                  &fibre_mask,
                                  NULL) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't load fibre mask");
        }
        
        cpl_free(extname);
        extname = NULL;

        /* Populate scattered light QC before we remove it */
        if(qmost_scattered_qc(fibre_flat_img,
                              qclist,
                              fibre_mask,
                              qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "scattered light QC failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(scattered_enable) {
            /* Remove scattered light if requested */
            if(qmost_scattered(fibre_flat_img,
                               qclist,
                               fibre_mask,
                               scattered_nbsize,
                               1) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "removal of scattered light "
                                             "with box size %d failed "
                                             "for extension %lld",
                                             scattered_nbsize,
                                             raw_extension);
            }
        }

        /* Save processed fibre flat if requested */
        if(proc_fibre_flat_frame != NULL) {
            if(qmost_dfs_save_image_and_var(proc_fibre_flat_frame,
                                            fibre_flat_hdr,
                                            arm_extname,
                                            qclist,
                                            fibre_flat_img,
                                            fibre_flat_var,
                                            CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save master fibre "
                                             "flat frame %s[%s]",
                                             proc_fibre_flat_filename,
                                             arm_extname);
            }
        }

        /* PSF analysis */
        if(qmost_model_psf_full(fibre_flat_img,
                                qclist,
                                trace_tbl,
                                trace_hdr,
                                profwidth,
                                subsample,
                                sblock,
                                &fibre_psf_img,
                                &fibre_psf_var,
                                qclist) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "PSF analysis failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        if(qmost_psf_stats(fibre_psf_img,
                           qclist,
                           arm,
                           -1,
                           fibinfo_tbl) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "PSF QC failed "
                                         "for extension %lld",
                                         raw_extension);
        }

        /* Save PSF */
        extname = cpl_sprintf("prof_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format PSF "
                                         "EXTNAME string");
        }

        if(qmost_dfs_save_imagelist_extension(fibre_psf_frame,
                                              fibre_flat_hdr,
                                              extname,
                                              qclist,
                                              fibre_psf_img,
                                              CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save output "
                                         "PSF %s[%s]",
                                         fibre_psf_filename,
                                         arm_extname);
        }

        cpl_free(extname);
        extname = NULL;

        extname = cpl_sprintf("profvar_%s", arm_extname);
        if(extname == NULL) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "could not format PSF variance "
                                         "EXTNAME string");
        }

        if(qmost_dfs_save_imagelist_extension(fibre_psf_frame,
                                              fibre_flat_hdr,
                                              extname,
                                              qclist,
                                              fibre_psf_var,
                                              CPL_TYPE_FLOAT) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save output "
                                         "PSF variance %s[%s]",
                                         fibre_psf_filename,
                                         arm_extname);
        }

        cpl_free(extname);
        extname = NULL;

        /* Clean up */
        cpl_image_delete(fibre_flat_img);
        fibre_flat_img = NULL;

        cpl_image_delete(fibre_flat_var);
        fibre_flat_var = NULL;

        cpl_propertylist_delete(fibre_flat_hdr);
        fibre_flat_hdr = NULL;

        cpl_table_delete(trace_tbl);
        trace_tbl = NULL;

        cpl_propertylist_delete(trace_hdr);
        trace_hdr = NULL;

        cpl_mask_delete(fibre_mask);
        fibre_mask = NULL;

        cpl_imagelist_delete(fibre_psf_img);
        fibre_psf_img = NULL;

        cpl_imagelist_delete(fibre_psf_var);
        fibre_psf_var = NULL;

        cpl_propertylist_delete(qclist);
        qclist = NULL;
    }

    /* Output fibre flat frame, if requested */
    if(proc_fibre_flat_frame != NULL) {
        /* Copy fibinfo table from the first frame if there was one */
        if(fibinfo_tbl != NULL) {
            if(qmost_fibtabsave(fibinfo_tbl,
                                fibinfo_hdr,
                                proc_fibre_flat_frame) != CPL_ERROR_NONE) {
                TIDY;
                return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                             "couldn't save FIBINFO table "
                                             "to master fibre "
                                             "flat frame %s",
                                             proc_fibre_flat_filename);
            }
        }
    }

    /* Copy fibinfo table from the first frame if there was one */
    if(fibinfo_tbl != NULL) {
        if(qmost_fibtabsave(fibinfo_tbl,
                            fibinfo_hdr,
                            fibre_psf_frame) != CPL_ERROR_NONE) {
            TIDY;
            return cpl_error_set_message(cpl_func, cpl_error_get_code(),
                                         "couldn't save FIBINFO table "
                                         "to output PSF %s ",
                                         fibre_psf_filename);
        }
    }
    
    /* Clean up */
    cpl_frameset_delete(fibflats_raw);
    fibflats_raw = NULL;

    if(fibinfo_tbl != NULL) {
        cpl_table_delete(fibinfo_tbl);
        fibinfo_tbl = NULL;
    }

    if(fibinfo_hdr != NULL) {
        cpl_propertylist_delete(fibinfo_hdr);
        fibinfo_hdr = NULL;
    }

    return cpl_error_get_code();
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Function needed by cpl_recipe_define to fill the input parameters
 *
 * @param  self   parameterlist where you need put parameters
 *
 * @return cpl_error_code
 *
 */
/*----------------------------------------------------------------------------*/
static cpl_error_code qmost_psf_analyse_fill_parameterlist(
    cpl_parameterlist *self)
{
    cpl_parameter *par;

    /* Switch to save processed products */
    par = cpl_parameter_new_value(RECIPE_NAME".keep",
                                  CPL_TYPE_BOOL,
                                  "Save optional processed products",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "keep");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* ccdproc parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".ccdproc.swapx",
                                  CPL_TYPE_STRING,
                                  "String of 0 or 1 specifying for each "
                                  "arm of each spectrograph if we should "
                                  "flip the x axis to correct the "
                                  "wavelength order of the spectral axis.",
                                  RECIPE_NAME,
                                  QMOST_DEFAULT_SWAPX_TABLE);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "ccdproc.swapx");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* imcombine parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.combtype",
                                  CPL_TYPE_INT,
                                  "Stacking method when combining multiple "
                                  "images.  1: mean, 2: median.",
                                  RECIPE_NAME,
                                  QMOST_MEANCALC);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.combtype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.scaletype",
                                  CPL_TYPE_INT,
                                  "Scaling to use when combining multiple "
                                  "images.  0: no biasing or scaling, "
                                  "1: bias by relative offset in background "
                                  "levels, 2: scale by ratio of background "
                                  "levels, 3: scale by exposure time and "
                                  "bias by relative offset in background.",
                                  RECIPE_NAME,
                                  2);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.scaletype");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.xrej",
                                  CPL_TYPE_BOOL,
                                  "Do extra rejection cycle when combining "
                                  "images.",
                                  RECIPE_NAME,
                                  0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.xrej");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".imcombine.thresh",
                                  CPL_TYPE_DOUBLE,
                                  "Rejection threshold in terms of "
                                  "background noise for rejection when "
                                  "combining images.",
                                  RECIPE_NAME,
                                  5.0);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "imcombine.thresh");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* Scattered light removal parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".scattered.enable",
                                  CPL_TYPE_BOOL,
                                  "Enable scattered light removal",
                                  RECIPE_NAME,
                                  1);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.enable");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".scattered.nbsize",
                                  CPL_TYPE_INT,
                                  "Size of the background smoothing cells "
                                  "for determining scattered light in pixels.",
                                  RECIPE_NAME,
                                  256);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "scattered.nbsize");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    /* PSF measurement parameters */
    par = cpl_parameter_new_value(RECIPE_NAME".psf.hwidth",
                                  CPL_TYPE_INT,
                                  "The maximum extent of the PSF "
                                  "measurement region relative to the "
                                  "fibre trace centre in pixels.  The "
                                  "measurement region must not overlap "
                                  "any adjacent fibre image.",
                                  RECIPE_NAME,
                                  5);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "psf.hwidth");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".psf.subsample",
                                  CPL_TYPE_INT,
                                  "Number of PSF samples per spatial "
                                  "pixel.",
                                  RECIPE_NAME,
                                  5);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "psf.subsample");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    par = cpl_parameter_new_value(RECIPE_NAME".psf.sblock",
                                  CPL_TYPE_INT,
                                  "The number of spectral pixels to "
                                  "block average when measuring the PSF.",
                                  RECIPE_NAME,
                                  100);
    cpl_parameter_set_alias(par,
                            CPL_PARAMETER_MODE_CLI,
                            "psf.sblock");
    cpl_parameter_disable(par, CPL_PARAMETER_MODE_ENV);
    cpl_parameterlist_append(self, par);

    return CPL_ERROR_NONE;
}

/**@}*/
