/*
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2016 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "sph_ifs_simple_adi_common.h"
#include "sph_keyword_manager.h"
#include "sph_extract_angles.h"
#include "sph_master_frame.h"
#include "sph_simple_adi.h"
#include "sph_ifs_tags.h"
#include "sph_common_keywords.h"
#include "sph_fits.h"
#include <cpl.h>
#include <strings.h>

static cpl_vector*
sph_ifs_simple_adi_extract_angles__(cpl_frameset* current_raw_frameset);

cpl_error_code sph_ifs_simple_adi_common(cpl_frameset* inframes,
        cpl_frameset* current_raw_frameset, cpl_parameterlist* inparams,
        const char* tag, const char* recipe, const char* outfilename,
        int coll_alg, int transform_method, int filter_method,
        double filter_radius, double fermi_temp, double butter_pass,
        double butter_stop) {
    sph_simple_adi* adi = NULL;
    sph_transform* transform = NULL;
    int nf = 0;
    sph_cube* outcube = NULL;
    int np = 0;
    int pp = 0;
    int ff = 0;
    sph_master_frame* mframe = NULL;
    double angle = 0.0;
    cpl_vector* angles = NULL;
    cpl_propertylist* plist = NULL;

    nf = (int) cpl_frameset_get_size(current_raw_frameset);
    cpl_ensure_code(nf > 0, CPL_ERROR_ILLEGAL_INPUT);

    np = sph_fits_get_nplanes(
            cpl_frame_get_filename(
                    cpl_frameset_get_first(current_raw_frameset)), 0);
    cpl_ensure_code(np > 0, CPL_ERROR_ILLEGAL_INPUT);

    if (transform_method == SPH_TRANSFORM_METHOD_FFT) {
        if (filter_method == SPH_FFT_FILTER_METHOD_NONE) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_NONE, 0.0,
                    0.0);
        } else if (filter_method == SPH_FFT_FILTER_METHOD_TH) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_TH,
                    filter_radius, 0.0);
        } else if (filter_method == SPH_FFT_FILTER_METHOD_FERMI) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_FERMI,
                    filter_radius, fermi_temp);
        } else if (filter_method == SPH_FFT_FILTER_METHOD_BUTTER) {
            transform = sph_transform_new_fft(SPH_FFT_FILTER_METHOD_BUTTER,
                    butter_pass, butter_stop);
        }
    } else if (transform_method == 1) {
        transform = sph_transform_new_cpl_warp();
    }cpl_ensure_code(transform, cpl_error_get_code());

    angles = sph_ifs_simple_adi_extract_angles__(current_raw_frameset);

    SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

    outcube = sph_cube_new_dfs(outfilename, inframes,
            cpl_frameset_get_first(current_raw_frameset), inparams, tag, recipe,
            SPH_PIPELINE_NAME_IFS, plist);
    for (pp = 0; pp < np; ++pp) {
        adi = sph_simple_adi_new(transform);
        for (ff = 0; ff < nf; ++ff) {
            mframe = sph_master_frame_load_(cpl_frameset_get_position_const
                                            (current_raw_frameset, ff), pp);
            if (mframe) {
                angle = cpl_vector_get(angles, ff);
                SPH_ERROR_ENSURE_GOTO_EXIT(
                        sph_simple_adi_append_image( adi, mframe->image, mframe->badpixelmap, mframe->ncombmap, angle) == CPL_ERROR_NONE,
                        cpl_error_get_code());
            }
            sph_master_frame_delete(mframe);
            mframe = NULL;
        }
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        //sph_simple_adi_calculate_speckle_frame(adi);
        mframe = sph_simple_adi_process_imlists(adi, coll_alg);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT
        if (mframe) {
            sph_cube_append_master(outcube, mframe, (float) pp);
        }
        sph_master_frame_delete(mframe);
        mframe = NULL;
        sph_simple_adi_delete(adi);
        adi = NULL;
    }

    EXIT: sph_master_frame_delete(mframe);
    mframe = NULL;
    sph_simple_adi_delete(adi);
    adi = NULL;
    if (outcube)
        sph_cube_finalise_file(outcube->filename);
    sph_cube_delete(outcube);
    outcube = NULL;
    sph_transform_delete(transform);
    transform = NULL;
    cpl_vector_delete(angles);
    angles = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static cpl_vector*
sph_ifs_simple_adi_extract_angles__(cpl_frameset* current_raw_frameset) {
    cpl_vector* result = NULL;
    int nframes = 0;
    int ff = 0;
    cpl_propertylist* prop = NULL;
    cpl_vector* tmpangles = NULL;
    cpl_frame* arawframe = NULL;
    double angle = 0.0;

    nframes = (int) cpl_frameset_get_size(current_raw_frameset);
    result = cpl_vector_new(nframes);
    cpl_vector_fill(result, 0.0);
    for (ff = 0; ff < nframes; ++ff) {

        arawframe = cpl_frameset_get_position(current_raw_frameset, ff);
        prop = sph_keyword_manager_load_properties(
                cpl_frame_get_filename(arawframe), 0);

        if (cpl_propertylist_has(prop, SPH_COMMON_KEYWORD_DROT2_MODE)
                && strcasecmp(
                        cpl_propertylist_get_string(prop,
                                SPH_COMMON_KEYWORD_DROT2_MODE),
                        SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB) == 0) {
            tmpangles = sph_extract_angles_from_cube(arawframe,
                                      SPH_IFS_INSTRUMENT_MODEL_DEROT_OFFSETELEV,
                                                     0.0);
        } else if (cpl_propertylist_has(prop, SPH_COMMON_KEYWORD_DROT2_MODE)
                && strcasecmp(
                        cpl_propertylist_get_string(prop,
                                SPH_COMMON_KEYWORD_DROT2_MODE),
                        SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB) == 0) {
            tmpangles = sph_extract_angles_from_cube(arawframe,
                                       SPH_IFS_INSTRUMENT_MODEL_DEROT_OFFSETSKY,
                                                     0.0);
        }
        if (tmpangles && cpl_vector_get_size(tmpangles) > 0) {
            angle = cpl_vector_get(tmpangles, 0);
            cpl_vector_set(result, ff, angle);
        } else {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "The angles could not be extracted. "
                    "Can not do ADI.");
            cpl_vector_delete(result);
            result = NULL;
            break;
        }
        cpl_vector_delete(tmpangles);
        tmpangles = NULL;
        cpl_propertylist_delete(prop);
        prop = NULL;
    }
    cpl_vector_delete(tmpangles);
    tmpangles = NULL;
    cpl_propertylist_delete(prop);
    prop = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;

}

