/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_common_keywords.h"
#include "sph_ifs_keywords.h"
#include "sph_ifs_tags.h"
#include "sph_ifs_spectra_positions.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_fft.h"
#include "sph_utils.h"
#include "sph_ifs_lenslet_model.h"
#include "sph_point_pattern.h"
#include "sph_ifs_subtract_dark_scaled.h"
#include "sph_distortion_model.h"
#include "sph_framecombination.h"
#include "sph_keyword_manager.h"

#include <gsl/gsl_rng.h>
#include <string.h>
#include <math.h>
#include <assert.h>

#define IFS_MIN_SPECWIDTH 5.0

#define SPH_REDUCE(A, B) if ((B) < A) A = (B)
#define SPH_INCREASE(A, B) if ((B) > A) A = (B)

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IFS_SPECTRA_POSITIONS_GENERAL;
extern sph_error_code SPH_IFS_SPECTRA_POSITIONS_FRAMES_MISSING;

static cpl_error_code
sph_ifs_spectra_positions_determine_offset(const sph_point_pattern*,
                                           const sph_point_pattern*,
                                           int,
                                           double*,
                                           double*,
                                           int) CPL_ATTR_NONNULL;

static sph_point_pattern*
sph_ifs_spectra_positions_extract_pp(const cpl_image* self,
                                     int nobs,
                                     int minpix) {
    sph_point_pattern* result = NULL;

    cpl_vector* bottom   = cpl_vector_new(nobs);
    cpl_vector* left     = cpl_vector_new(nobs);
    cpl_vector* top      = cpl_vector_new(nobs);
    cpl_vector* right    = cpl_vector_new(nobs);
    cpl_vector* npix     = cpl_vector_new(nobs);
    const cpl_size nx    = cpl_image_get_size_x(self);
    const cpl_size ny    = cpl_image_get_size_y(self);
#ifndef NDEBUG
    const cpl_mask* bpm  = cpl_image_get_bpm_const(self);
    const cpl_type ptype = cpl_image_get_type(self);
#endif
    const int*     pimg  = cpl_image_get_data_int_const(self);
    double* pbottom = cpl_vector_get_data(bottom);
    double* pleft   = cpl_vector_get_data(left);
    double* pright  = cpl_vector_get_data(right);
    double* ptop    = cpl_vector_get_data(top);
    double* pnpix   = cpl_vector_get_data(npix);

    assert(bpm == NULL);
    assert(ptype == CPL_TYPE_INT);
    assert(pimg != NULL);

    cpl_vector_fill(left,   (double)INT_MAX);
    cpl_vector_fill(bottom, (double)INT_MAX);
    cpl_vector_fill(right,  -1.0);
    cpl_vector_fill(top,    -1.0);
    cpl_vector_fill(npix,    0.0);

    for (size_t j = 0; j < (size_t)ny; j++) {
        for (size_t i = 0; i < (size_t)nx; i++) {
            const int id = pimg[i + j * (size_t)nx] - 1;

            if (0 <= id && id < nobs) {

                SPH_REDUCE(pleft[id],   (double)i);
                SPH_REDUCE(pbottom[id], (double)j);

                SPH_INCREASE(pright[id], (double)i);
                SPH_INCREASE(ptop[id],   (double)j);

                pnpix[id] += 1.0;
            }
        }
    }

    result = sph_point_pattern_new_(nobs);

    for (size_t k = 0; k < (size_t)nobs; k++) {
        if ((int)pnpix[k] > minpix) {
            const double xpos = 0.5 * (pright[k] + pleft[k] + 1.0);
            const double ypos = 0.5 * (ptop[k] + pbottom[k] + 1.0);
            if (xpos < 0.99 * (double)INT_MAX &&
                ypos < 0.99 * (double)INT_MAX) {
                sph_point_pattern_add_point(result, xpos, ypos);
            }
        }
    }
    cpl_vector_delete(top);
    cpl_vector_delete(bottom);
    cpl_vector_delete(left);
    cpl_vector_delete(right);
    cpl_vector_delete(npix);
    return result;
}

static cpl_error_code
sph_ifs_spectra_positions_determine_offset(const sph_point_pattern* pp1,
                                           const sph_point_pattern* pp2,
                                           int nsamples,
                                           double* offx,
                                           double* offy,
                                           int detsize)
{
    const double centre = detsize / 2.0;
    const int    np1    = sph_point_pattern_get_size(pp1);
    int         cc      = 0;
    cpl_vector* distsx;
    cpl_vector* distsy;

    cpl_ensure_code(nsamples > 0, CPL_ERROR_ILLEGAL_INPUT);

    /* If np1 is less than 1 the vectors are NULL, causing an error return */
    distsx = cpl_vector_new(np1);
    distsy = cpl_vector_new(np1);

    for (int ii = 0; ii < np1; ++ii) {
        const double* p = sph_point_pattern_get(pp1, ii);
        if (fabs(p[0] - centre) < nsamples &&
            fabs(p[1] - centre) < nsamples) {
            const cpl_size jj = sph_point_pattern_find_closest(pp2, p);
            if (jj >= 0) {
                const double*  p2 = sph_point_pattern_get(pp2, jj);
                const double distx = (p[0] - p2[0]);
                const double disty = (p[1] - p2[1]);

                cpl_vector_set(distsx, cc, distx);
                cpl_vector_set(distsy, cc, disty);
                cc++;
            }
        }
    }

    if (cc > 0) {
        cpl_vector* vwrap;

        vwrap = cpl_vector_wrap(cc, cpl_vector_get_data(distsx));
        *offx = -cpl_vector_get_median(vwrap);
        (void)cpl_vector_unwrap(vwrap);

        vwrap = cpl_vector_wrap(cc, cpl_vector_get_data(distsy));
        *offy = -cpl_vector_get_median(vwrap);
        (void)cpl_vector_unwrap(vwrap);
    }

    cpl_vector_delete(distsx);
    cpl_vector_delete(distsy);

    return cc > 0 ? CPL_ERROR_NONE :
        cpl_error_set(cpl_func, CPL_ERROR_DATA_NOT_FOUND);
}
static
double sph_ifs_spectra_positions_get_angle_model(sph_ifs_lenslet_model* model) {
    sph_polygon* poly = NULL;
    double x1 = 0.0;
    double x2 = 0.0;
    double y1 = 0.0;
    double y2 = 0.0;
    double ll = 0.0;
    double angle = 0.0;

    poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(model, 0, 0, 0.0,
            0.0);
    sph_polygon_get_midxy_(poly, &x1, &y1);
    poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(model, -2, 3, 0.0,
            0.0);
    sph_polygon_get_midxy_(poly, &x2, &y2);

    ll = (x2 - x1) * (x2 - x1) + (y2 - y1) * (y2 - y1);

    angle = acos((x2 - x1) / sqrt(ll)) * CPL_MATH_DEG_RAD;
    angle = 90.0 - angle;
    return angle;
}

static
double sph_ifs_spectra_positions_get_scale_model(sph_ifs_lenslet_model* model) {
    sph_polygon* poly = NULL;
    double x1 = 0.0;
    double x2 = 0.0;
    poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(model, 0, 0, 0.0,
            0.0);
    x1 = poly->points[0].x;
    poly = sph_ifs_lenslet_model_get_spec_region_pixel_coords(model, 1, -1, 0.0,
            0.0);
    x2 = poly->points[0].x;
    return fabs(x1 - x2);
}

static
double sph_ifs_spectra_positions_get_scale(cpl_image* specid_image_obs) {
    sph_fft* sphfft = NULL;
    cpl_image* im2 = NULL;
    cpl_image* imcut = NULL;
    cpl_image* im = NULL;
    double offx_sigma = 0.0;
    cpl_size nx, ny;
    int yy = 0;
    double offx = 0.0;

    im2 = cpl_image_cast(specid_image_obs, CPL_TYPE_DOUBLE);
    if (cpl_image_get_size_x(im2) > 500 + 1024) {
        imcut = cpl_image_extract(im2, 500, 500, 500 + 1024, 500 + 1024);
    } else {
        imcut = cpl_image_duplicate(im2);
    }
    cpl_image_delete(im2);
    im2 = cpl_image_collapse_create(imcut, 0);
#ifdef SPH_SAVE_NONDFS
    cpl_image_save(im2, "input_image_row.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
#endif
    if (im2 == NULL)
        return CPL_ERROR_NULL_INPUT;

    nx = cpl_image_get_size_x(imcut);
    ny = cpl_image_get_size_y(imcut);
    im = cpl_image_new(nx, ny, CPL_TYPE_DOUBLE);
    if (im == NULL)
        return CPL_ERROR_NULL_INPUT;
    for (yy = 0; yy < ny; ++yy) {
        cpl_image_copy(im, im2, 1, yy + 1);
        if (cpl_error_get_code()) {
            SPH_RAISE_CPL
            exit(1);
        }
    }
#ifdef SPH_SAVE_NONDFS
    cpl_image_save(im, "input_image2.fits", CPL_TYPE_DOUBLE, NULL,
            CPL_IO_CREATE);
#endif

    sphfft = sph_fft_new(SPH_FFT_GSL_MIXEDRADIX);
    offx = sph_fft_get_scale_vector(sphfft, im, &offx_sigma);
    cpl_image_delete(im2);
    cpl_image_delete(imcut);
    cpl_image_delete(im);
    sph_fft_delete(sphfft);
    return offx;
}

static sph_distortion_model*
sph_ifs_spectra_positions_create_distmap(sph_point_pattern* pp2,
                                         sph_point_pattern* pp,
                                         double length_threshold,
                                         int polyfit_order,
                                         double* chiX, double* chiY) {
    sph_distortion_model* result = NULL;
    sph_distortion_map* dmap = NULL;
    sph_distortion_map* ddmap = NULL;
    int nremoved = 0;
    if (!pp2)
        return NULL;

    //Do NOT centralise since this can cause problem if some points are missing in pp2.
    //sph_point_pattern_centralise(pp2);
    //sph_point_pattern_centralise(pp);
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "The number of points are: %d,%d", sph_point_pattern_get_size(pp), sph_point_pattern_get_size(pp2));
    dmap = sph_distortion_map_new(pp2, pp);
    if (!dmap) {
        sph_point_pattern_delete(pp2);
        return NULL;
    }

    //sph_distortion_map_remove_offset(dmap);

    nremoved = sph_distortion_map_threshold(dmap, length_threshold, CPL_TRUE);
    sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, "Removed %d vectors above distortion %f "
                    "from distortion map from a total of %d "
                    "initally.", nremoved, length_threshold,
            sph_distortion_map_get_size(dmap) + nremoved);
    result = sph_distortion_model_new(NULL, NULL);
    if (result != NULL) {
        ddmap = sph_distortion_model_fit_distortion_map(result, dmap,
                                                                   polyfit_order,
                                                                   chiX, chiY);
        sph_distortion_map_delete(ddmap);
    }
    sph_distortion_map_delete(dmap);

    return result;
}

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ifs_spectra_positions_run Run the Instrument Flat Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * instrument flat recipe for IFS.
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ifs_spectra_positions.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list
 @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
static sph_pixel_description_table*
sph_ifs_spectra_positions_process_prism(sph_ifs_spectra_positions* self,
        cpl_frameset* inframes, sph_master_frame* flat, sph_master_frame* dark,
        sph_ifs_lenslet_model* lenslet_model, cpl_propertylist* pl) {
    int rerr = CPL_ERROR_NONE;
    sph_master_frame* combined = NULL;
    sph_pixel_description_table* predicted_pdt = NULL;
    cpl_image* centres = NULL;
    cpl_image* centres_obs = NULL;
    cpl_propertylist* ipl = NULL;
    cpl_propertylist* poldist_pl = NULL;
    sph_point_pattern* point_pattern = NULL;
    sph_point_pattern* point_pattern_model = NULL;
    cpl_image* specid_image_obs = NULL;
    cpl_mask* mask = NULL;
    cpl_size nobs = 0;
    double median = 0.0;
    double dev = 0.0;
    double scale = 0.0;
    double offx = 0.0;
    double offy = 0.0;
    double chiX = 0.0;
    double chiY = 0.0;
    int pixsin = 0;
    int pixsout = 0;
    int darkin = 0;
    sph_distortion_model* poldist = NULL;
    if (!lenslet_model) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "I need a valid IFS lenslet model to "
                        "produce the spectra positions table. "
                        "Please provide one with the %s tag",
                SPH_IFS_TAG_LENSLET_MODEL_CALIB);
        return NULL;
    }
    cpl_error_reset();

    combined = sph_framecombination_master_frame_from_cpl_frameset(inframes,
            self->coll_alg, self->framecomb_parameterlist);

    // recover the DIT from the inframes to allow for later scaling
    ipl = sph_keyword_manager_load_properties(
            cpl_frame_get_filename(cpl_frameset_get_first(inframes)), 0);

    if (ipl != NULL && cpl_propertylist_has(ipl, SPH_COMMON_KEYWORD_SEQ1DIT)) {
        cpl_propertylist_append_property(
                combined->properties,
                cpl_propertylist_get_property_const(ipl,
                        SPH_COMMON_KEYWORD_SEQ1DIT));
        cpl_propertylist_delete(ipl);
        ipl = NULL;
    } else
        SPH_WARNING(
                "Cannot find DIT in raw data, will subtract unscaled dark subsequently!");

    if (dark) {
        sph_ifs_subtract_dark_scaled(combined, dark);
    }
    if (flat) {
        sph_master_frame_divide_master_frame(combined, flat);
    }

    median = cpl_image_get_median_dev(combined->image, &dev);

    if (self->threshold < 0.0) {
        self->threshold = median + 0.1 * dev;
    }
    sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
            SPH_ERROR_INFO, "Am attemtping to detect regions on image "
                    "with median %f and average difference to "
                    "median of %f using a threshold of %f", median, dev,
            self->threshold);
    cpl_propertylist_append_double(pl, SPH_IFS_KEYWORD_SPECPOS_QC_THRESHOLD,
            self->threshold);
#ifdef SPH_SAVE_NONDFS
    cpl_image_save( combined->image, "combined.fits",
            CPL_TYPE_DOUBLE,
            NULL,
            CPL_IO_DEFAULT);
#endif
    mask = cpl_mask_threshold_image_create(combined->image, self->threshold,
            1000000.0);
    specid_image_obs = cpl_image_labelise_mask_create(mask, &nobs);
    cpl_mask_delete(mask);
    mask = NULL;

    SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO, "Detected %d spectra", (int)nobs);

    cpl_propertylist_append_int(pl, SPH_IFS_KEYWORD_SPECPOS_QC_NREGS, nobs);
    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Creating point pattern...");
#ifdef SPH_SAVE_NONDFS
    cpl_image_save( specid_image_obs, "detected_spectra_regions.fits",
            CPL_TYPE_DOUBLE,
            NULL,
            CPL_IO_DEFAULT);
#endif
    point_pattern = sph_ifs_spectra_positions_extract_pp(specid_image_obs, nobs,
            20);

    if (point_pattern) {
        cpl_size nx, ny;
        /* This label image is of type CPL_TYPE_INT with not bpm */
        const int* pspecid_image_obs
            = cpl_image_get_data_int_const(specid_image_obs);
        const cpl_binary* pmask;

        centres_obs = sph_point_pattern_create_image(point_pattern,
                cpl_image_get_size_x(combined->image),
                cpl_image_get_size_y(combined->image), 1.0);
#ifdef SPH_SAVE_NONDFS
        cpl_image_save( centres_obs, "observed_point_pattern.fits",
                CPL_TYPE_DOUBLE,
                NULL,
                CPL_IO_DEFAULT);
#endif
        scale = sph_ifs_spectra_positions_get_scale(specid_image_obs);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Determined a scale of %f.", scale);
        scale = scale
                / sph_ifs_spectra_positions_get_scale_model(lenslet_model);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

        cpl_propertylist_append_double(pl, SPH_IFS_KEYWORD_SPECPOS_QC_SCALE,
                scale);

        if (scale > 0.0) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO, "Determined a scale of %f.", scale);
            lenslet_model->stretch_x *= scale;
            lenslet_model->stretch_y *= scale;
            lenslet_model->specwidth_pixels *= scale;
            lenslet_model->specwidth_microns *= scale;
            // Below commented out, since IFS request is that spectra alwyays have
            // same length
            //lenslet_model->speclength_pixels *= scale;
            //lenslet_model->speclength_microns *= scale;
        } else {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "Problem determining scale");
            goto EXIT;
        }
        if (lenslet_model->specwidth_pixels < IFS_MIN_SPECWIDTH) {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Determined a avg. spectra width of %f, which is "
                    "below the tolerance of %f.", lenslet_model->specwidth_pixels, IFS_MIN_SPECWIDTH);
            goto EXIT;
        }
        offx = offy = 0.0;
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

        point_pattern_model = sph_ifs_lenslet_model_predict_spec_loc_pp(
                lenslet_model, 0.0, 0.0);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        sph_ifs_spectra_positions_determine_offset(point_pattern_model,
                point_pattern, 500, &offx, &offy,
                lenslet_model->detsize_pixels / 2);
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
        sph_point_pattern_delete(point_pattern_model);
        point_pattern_model = NULL;
        SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

        if (rerr == CPL_ERROR_NONE) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Determined an offset of %f in x and %f in y.", offx, offy);
            lenslet_model->zero_offsetx += offx;
            lenslet_model->zero_offsety += offy;
        } else {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "Problem determining offset");
            goto EXIT;
        }

        cpl_propertylist_append_double(pl, SPH_IFS_KEYWORD_SPECPOS_QC_OFFSETX,
                offx);
        cpl_propertylist_append_double(pl, SPH_IFS_KEYWORD_SPECPOS_QC_OFFSETY,
                offy);
        if (self->dist) {
            SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                    "Determining distortion...");
            point_pattern_model = sph_ifs_lenslet_model_predict_spec_loc_pp(
                    lenslet_model, 0.0, 0.0);
            poldist = sph_ifs_spectra_positions_create_distmap(point_pattern,
                    point_pattern_model, 10.0, 4, &chiX, &chiY);
            if (poldist) {
                poldist_pl = cpl_propertylist_new();
                cpl_propertylist_append_double(poldist_pl,
                        SPH_IFS_KEYWORD_DISTMAP_POLFIT_CHIX, chiX);
                cpl_propertylist_append_double(poldist_pl,
                        SPH_IFS_KEYWORD_DISTMAP_POLFIT_CHIY, chiY);
#ifdef SPH_SAVE_NONDFS
                sph_distortion_model_save(poldist,
                        "spectra_positions_distortion.fits", 2048, 2048,
                        CPL_IO_CREATE, poldist_pl, NULL);
#endif
                sph_ifs_lenslet_model_set_detector_distortion(lenslet_model,
                        poldist);
                cpl_propertylist_delete(poldist_pl);
                poldist_pl = NULL;
            } else
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_GENERAL,
                        "Could not determine distortion. Assuming zero distortion.");
            sph_point_pattern_delete(point_pattern_model);
            point_pattern_model = NULL;
        }SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "Creating new final PDT...");
        predicted_pdt = sph_pixel_description_table_new_from_model(
                lenslet_model, 0.0, 0.0);
        mask = sph_pixel_description_table_get_mask(predicted_pdt);
        cpl_image_threshold(specid_image_obs, 0.0, 0.9, 0.0, 1.0);

        pmask = cpl_mask_get_data_const(mask);
        nx = cpl_image_get_size_x(specid_image_obs);
        ny = cpl_image_get_size_y(specid_image_obs);
        for (size_t i = 0; i < (size_t)(nx *ny); i++) {
            if (pspecid_image_obs[i] > 0) {
                if (pmask[i] > 0) {
                    pixsin++;
                } else {
                    pixsout++;
                }
            } else if (pmask[i] > 0) {
                darkin++;
            }
        }

        cpl_propertylist_append_int(pl, SPH_IFS_KEYWORD_SPECPOS_QC_PIXS_IN,
                pixsin);
        cpl_propertylist_append_int(pl, SPH_IFS_KEYWORD_SPECPOS_QC_PIXS_OUT,
                pixsout);
        cpl_propertylist_append_int(pl, SPH_IFS_KEYWORD_SPECPOS_QC_DARK_PIXS_IN,
                darkin);
        cpl_propertylist_append_double(pl, SPH_IFS_KEYWORD_DISTMAP_POLFIT_CHIX,
                chiX);
        cpl_propertylist_append_double(pl, SPH_IFS_KEYWORD_DISTMAP_POLFIT_CHIY,
                chiY);

        cpl_mask_delete(mask);
        mask = NULL;
        sph_point_pattern_delete(point_pattern);
        point_pattern = NULL;
        SPH_RAISE_CPL_RESET;
    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Could not create the point pattern"
                        " of model.");
    }
    sph_master_frame_delete(combined);
    combined = NULL;

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL
    }
    cpl_image_delete(centres_obs);
    centres_obs = NULL;
    cpl_image_delete(centres);
    centres = NULL;
    cpl_image_delete(specid_image_obs);
    specid_image_obs = NULL;
    cpl_propertylist_delete(ipl);
    ipl = NULL;
    return predicted_pdt;
    EXIT: sph_master_frame_delete(combined);
    combined = NULL;
    sph_point_pattern_delete(point_pattern);
    point_pattern = NULL;
    sph_point_pattern_delete(point_pattern_model);
    point_pattern_model = NULL;
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL
    }
    cpl_image_delete(centres_obs);
    centres_obs = NULL;
    cpl_image_delete(centres);
    centres = NULL;
    cpl_image_delete(specid_image_obs);
    specid_image_obs = NULL;
    cpl_propertylist_delete(ipl);
    ipl = NULL;

    sph_pixel_description_table_delete(predicted_pdt);
    return NULL;
}
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list
 @return   0 if everything is ok
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ifs_spectra_positions_run(sph_ifs_spectra_positions* self) {
    const cpl_frame* background_dark_frame
        = self->master_background_frame != NULL
        ? self->master_background_frame
        : self->master_dark_frame;
    sph_master_frame* dark = NULL;
    sph_master_frame* flat = NULL;
    sph_ifs_lenslet_model* model = NULL;
    sph_pixel_description_table* predicted_pdt = NULL;
    cpl_propertylist* pl  = NULL;
    cpl_propertylist* pli = NULL;

    pli = cpl_propertylist_load( cpl_frame_get_filename(cpl_frameset_get_first(self->rawframes)),
                                    0); // read template property list of first frame to copy singular keys later
    cpl_ensure(pli,cpl_error_get_code(),0);

    if (self->model_frame) {
        model = sph_ifs_lenslet_model_load(
                cpl_frame_get_filename(self->model_frame));
    } else {
        model = sph_ifs_lenslet_model_new();
        if ( self->header ) {
            cpl_ensure(
                    cpl_propertylist_has(pli,SPH_IFS_KEYWORD_PRISM_MODE),
                    CPL_ERROR_ILLEGAL_INPUT, cpl_error_get_code());

            if ( strcmp(
                    cpl_propertylist_get_string(
                            pli,SPH_IFS_KEYWORD_PRISM_MODE),
                    SPH_IFS_KEYWORD_VALUE_PRISM_MODE_JH) == 0 )
            {
                self->hmode = 1;
            }
            else if ( strcmp(
                    cpl_propertylist_get_string(
                            pli,SPH_IFS_KEYWORD_PRISM_MODE),
                    SPH_IFS_KEYWORD_VALUE_PRISM_MODE_J) == 0 )
            {
                self->hmode = 0;
            }
            else {
                SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,"Illegal prism mode in input header.");
                return CPL_ERROR_ILLEGAL_INPUT;
            }
        }
        if (self->hmode) {
            // These two lines were active BEFORE the Sept. 12 IFS tests
            // They may have been correct, but baseline now is that
            // speclength and specwidth are mode independent
            //model->speclength_pixels = 39.4951734171714;
            //model->specwidth_pixels =  4.9293975935862;
            model->maxlambda = SPH_IFS_LENSLET_MODEL_MAX_LAMBDA_JH;
            model->minlambda = SPH_IFS_LENSLET_MODEL_MIN_LAMBDA_JH;
            model->dispersion = (model->maxlambda - model->minlambda)
                    / (model->speclength_pixels - 1);
        }
    }
    if (background_dark_frame) {
        dark = sph_master_frame_load_(background_dark_frame, 0);
    }

    if (self->master_flat_frame) {
        flat = sph_master_frame_load_(self->master_flat_frame, 0);
    }

    if (self->angle < -360.0) {
        model->rotangle = 0.0;
        self->angle = sph_ifs_spectra_positions_get_angle_model(model);
    }
    model->rotangle = self->angle;
    pl = cpl_propertylist_new();

    predicted_pdt = sph_ifs_spectra_positions_process_prism(self,
            self->rawframes, flat, dark, model, pl);
    sph_utils_simple_copy_singular(pli, pl);

    if (predicted_pdt) {
		if (self->correct_nonlin){
			/* Apply non linear correction at the end */
			if ( self->hmode ) {
					sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
							SPH_ERROR_INFO,
							"Applying new non linear model correction for H mode.");
				sph_pixel_description_table_correct_non_linear(predicted_pdt, 0.30870, -0.74312, 0.41505);
			}
			else {
					sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
							SPH_ERROR_INFO,
							"Applying new non linear model correction for Y mode.");
				sph_pixel_description_table_correct_non_linear(predicted_pdt, 0.67754, -1.4464, 0.75754);
			}
		}

		if (sph_pixel_description_table_save_dfs(predicted_pdt,
				self->spectra_positions_filename, model, self->inframes, NULL,
				self->inparams, SPH_IFS_TAG_SPEC_POS_CALIB,
				SPH_RECIPE_NAME_IFS_SPECTRA_POSITIONS, SPH_PIPELINE_NAME_IFS,
				pl) != CPL_ERROR_NONE) {
			SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
					"Could not write main product.");
		}

    }
    sph_master_frame_delete(flat);
    flat = NULL;
    sph_master_frame_delete(dark);
    dark = NULL;
    sph_ifs_lenslet_model_delete(model);
    model = NULL;
    sph_pixel_description_table_delete(predicted_pdt);
    predicted_pdt = NULL;
    cpl_propertylist_delete(pl);
    pl = NULL;
    if(pli){
        cpl_propertylist_delete(pli);
        pli = NULL;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/**@}*/
