/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_ird_common_science.h"
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"
#include "sph_extract_angles.h"
#include "sph_ird_tags.h"
#include "sph_ird_gain.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_filemanager.h"
#include "sph_ird_instrument_model.h"
#include "sph_ird_star_center.h"
#include "sph_keyword_manager.h"
#include "sph_fft.h"
#include "sph_fits.h"
#include "sph_smart_imagelist.h"
#include "sph_common_science.h"
#include "sph_differential_imaging.h"
#include "sph_time.h"
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>
#include "sph_fctable.h"
#include "sph_andromeda_support.h"
#include "sph_cube.h"

#include <cpl.h>

#include <math.h>
#include <strings.h>
#include <string.h>
#include <assert.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/

/* Disable incorrect WCS for now */
#ifdef SPH_USE_WCS
#define SPH_CHECK_RADEC cpl_propertylist_has
#else
#define SPH_CHECK_RADEC(A,B) 0
#endif

/* Pixel scale [mas] */
#ifndef SPH_PIX_SCALE
#define SPH_PIX_SCALE 12.5
#endif

const char* const SPH_IRD_STAR_CENTER_TIME_UT_NAME = "TIME_UT";
const char* const SPH_IRD_STAR_CENTER_CENTLEFTX_NAME = "CENTRE_LEFT_X";
const char* const SPH_IRD_STAR_CENTER_CENTLEFTY_NAME = "CENTRE_LEFT_Y";
const char* const SPH_IRD_STAR_CENTER_CENTRIGHTX_NAME = "CENTRE_RIGHT_X";
const char* const SPH_IRD_STAR_CENTER_CENTRIGHTY_NAME = "CENTRE_RIGHT_Y";
const char* const SPH_IRD_STAR_CENTER_DMS_POS_X_NAME = "DMS_POS_X";
const char* const SPH_IRD_STAR_CENTER_DMS_POS_Y_NAME = "DMS_POS_Y";

static sph_ird_common_science*
sph_ird_common_science_new__(void);
static sph_error_code
sph_ird_common_science_interpolate_centers(cpl_frame* fcframe,
        cpl_vector* timev, int plane, double* clx, double* cly, double* crx,
        double* cry);
static sph_fctable*
sph_ird_common_science_create_table_single_cube(cpl_frame* fcframe,
        cpl_frame* arawframe);
static cpl_frameset*
sph_ird_common_science_create_fctables_internal(cpl_frame* fcframe,
        cpl_frameset* rawframes, int keep_fctable);
static sph_error_code
sph_ird_common_science_processing_iteration__(sph_ird_common_science* sci,
        cpl_frameset* loopframes, cpl_frameset* scframes,
        cpl_frameset** left_out_frames, cpl_frameset** right_out_frames,
        cpl_frameset** total_out_frames, int ff, int xcl, int ycl, int xcr,
        int ycr,
                                     const char* left_procatg,
                                     const char* right_procatg,
                                     const char* total_procatg);

static sph_error_code
sph_ird_common_science_process_single_cube(sph_ird_common_science* sci,
        const cpl_frame* frame_of_raw_cube, cpl_vector* centx_left,
        cpl_vector* centy_left, cpl_vector* centx_right,
        cpl_vector* centy_right, sph_cube* left_cube, sph_cube* right_cube,
        sph_cube* total_cube);

/*----------------------------------------------------------------------------*/
/**
 * @brief Create new sph_common_science structure
 *
 * @return pointer to new sph_common_science structure
 */
/*----------------------------------------------------------------------------*/
sph_ird_common_science*
sph_ird_common_science_new(cpl_frameset* inframes, cpl_parameterlist* inparams,
        cpl_frame* dark, cpl_frame* flat, cpl_frame* bpix,
        cpl_frame* distortion, cpl_frame* filter, cpl_frame* irdmodel,
        const cpl_frameset * fctable_frames,
        const char* recipe, const char* tag) {
    sph_ird_common_science* result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    result = sph_ird_common_science_new__();
    result->inframes = (cpl_frameset*) inframes;
    result->inparams = (cpl_parameterlist*) inparams;
    if (dark) {
        result->dark = sph_master_frame_load_(dark, 0);
    }
    if (flat) {
        result->flat = sph_master_frame_load_(flat, 0);
    }
    if (bpix) {
        result->bpix = cpl_image_load(cpl_frame_get_filename(bpix),
                CPL_TYPE_INT, 0, 0);
    }

    if (distortion) {
        const char * distfile = cpl_frame_get_filename(distortion);
        double opt_cent_l_x, opt_cent_l_y, opt_cent_r_x, opt_cent_r_y;
        int align_right = 0;

        result->distort_left = sph_distortion_model_load_opt(distfile, 0,
                SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFX,
                SPH_IRD_KEYWORD_DISTMAP_LEFT_COEFFY,
                SPH_IRD_KEYWORD_DISTMAP_LEFT_OPTICAL_AXIS_X,
                SPH_IRD_KEYWORD_DISTMAP_LEFT_OPTICAL_AXIS_Y,
                &opt_cent_l_x, &opt_cent_l_y, &align_right);
        result->distort_right = sph_distortion_model_load_opt(distfile, 8,
                SPH_IRD_KEYWORD_DISTMAP_RIGHT_COEFFX,
                SPH_IRD_KEYWORD_DISTMAP_RIGHT_COEFFY,
                SPH_IRD_KEYWORD_DISTMAP_RIGHT_OPTICAL_AXIS_X,
                SPH_IRD_KEYWORD_DISTMAP_RIGHT_OPTICAL_AXIS_Y,
                &opt_cent_r_x, &opt_cent_r_y, NULL);

        if (fctable_frames != NULL && align_right) {
            const cpl_size pows0[2] = {0, 0};
            const double rdistx =
                cpl_polynomial_get_coeff(result->distort_right->polyx, pows0);
            const double rdisty =
                cpl_polynomial_get_coeff(result->distort_right->polyy, pows0);

            const double cshiftx = opt_cent_l_x - opt_cent_r_x;
            const double cshifty = opt_cent_l_y - opt_cent_r_y;

            cpl_msg_warning(cpl_func, "De-aligning right channel off from left "
                            "by (%g,%g)", cshiftx, cshifty);

            cpl_polynomial_set_coeff(result->distort_right->polyx, pows0,
                                     rdistx - cshiftx);
            cpl_polynomial_set_coeff(result->distort_right->polyy, pows0,
                                     rdisty - cshifty);
        }
    }
    if (filter) {
        result->filter_table = sph_filter_table_load_fits(
                cpl_frame_get_filename(filter));
    }
    if (irdmodel)
        result->irdmodel = sph_ird_instrument_model_load(
                cpl_frame_get_filename(irdmodel));
    else {
        if (flat)
            result->irdmodel = sph_ird_instrument_model_load(
                    cpl_frame_get_filename(flat));
        else
            result->irdmodel = sph_ird_instrument_model_new();
    }
    result->transform = sph_transform_new_default();
    result->recipe = recipe;
    result->tag = tag;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}
/*----------------------------------------------------------------------------*/
/**
 * @brief Set the telescope flat field
 * @param self          pointer to self
 * @param tff_frame     frame of the telescope flat or NULL
 *
 * @return error code
 *
 * Set the telescope flat field to use.
 * If this is non NULL, the telescope flat field is read
 * from the frame and used in the processing.
 *
 * If NULL is passed the telescope flat is unset.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_ird_common_science_set_tff(sph_ird_common_science* self,
        cpl_frame* tff) {
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);

    sph_master_frame_delete(self->tff);
    self->tff = NULL;

    if (tff) {
        self->tff = sph_master_frame_load_(tff, 0);
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Set the transform method
 * @param self          the IRDIS common science
 * @param transform     the transform to set for future operations
 *
 * @return error code
 *
 * This sets the new transform method. This new transform will be used
 * for all future calls to ird common science functionality.
 * The transform structure will be managed by the ird common science
 * module. Do not call delete on the transform structure passed after
 * set_transform has been called.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_ird_common_science_set_transform(
        sph_ird_common_science* self, sph_transform* transform) {
    cpl_ensure_code(self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(transform, CPL_ERROR_NULL_INPUT);
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (self->transform) {
        sph_transform_delete(self->transform);
        self->transform = NULL;
    }
    self->transform = transform;

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Put WCS keywords into header of science product
 * @param self  The propertylist to be updated
 * @param x     The X coordinate of reference pixel
 * @param y     The Y coordinate of reference pixel
 * @return error code
 *
 * This puts the WCS keywords in the header. Reference RA and DEC are taken from the
 * propertylist (target coordinates). Projection is always RA---TAN/DEC--TAN.
 * The pixel scale is fixed to SPH_PIX_SCALE (12.5) in mas.
 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_common_science_put_wcs(cpl_propertylist* self,
                                              double x, double y) {

    cpl_ensure_code(self != NULL, CPL_ERROR_NULL_INPUT);

    if (SPH_CHECK_RADEC(self, "RA") &&
        SPH_CHECK_RADEC(self, "DEC")) {
        const double ra  = cpl_propertylist_get_float(self, "RA");
        const double dec = cpl_propertylist_get_float(self, "DEC");
        char comment_string[256];
        char time_string[256];

        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,
                                       ra);

        sph_radec_deg_to_iso8601string(ra, time_string, 1);
        sprintf(comment_string,"%s, RA at ref pixel", time_string);
        cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,
                                     comment_string);

        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,
                                       dec);

        sph_radec_deg_to_iso8601string(dec, time_string, 0);
        sprintf(comment_string,"%s, DEC at ref pixel", time_string);
        cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,
                                     comment_string);

        cpl_propertylist_update_string(self, SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                       SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE_VAL);

        cpl_propertylist_update_string(self, SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                       SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE_VAL);

        /* Unit is degrees, convert from mas, i.e. divide by 1000 * 60 * 60 */
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CD11,
                                       -SPH_PIX_SCALE / 3600e3);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CD12, 0.0);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CD21, 0.0);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CD22,
                                       SPH_PIX_SCALE / 3600e3);

        cpl_propertylist_update_string(self, SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                       SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT_VAL);
        cpl_propertylist_update_string(self, SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                       SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT_VAL);

    } else {
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,
                                       x);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,
                                       y);

        cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXXVAL,
                                     "Reference pixel (X)");
        cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXYVAL,
                                     "Reference pixel (Y)");


        cpl_msg_warning(cpl_func, "Writing WCS without (RA,DEC)");
        cpl_propertylist_update_string(self, SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                       "PIXEL");
        cpl_propertylist_update_string(self, SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                       "PIXEL");

        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CD11,
                                       1.0);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CD12, 0.0);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CD21, 0.0);
        cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CD22,
                                       1.0);

        cpl_propertylist_update_string(self, SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                       "pixel");
        cpl_propertylist_update_string(self, SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                       "pixel");
    }


    cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CRPIXX, x);
    cpl_propertylist_update_double(self, SPH_COMMON_KEYWORD_WCS_CRPIXY, y);

    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXX,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXX_COMMENT);
    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXY,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXY_COMMENT);

    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXXTYPE,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXTYPE_COMMENT);
    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXYTYPE,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXTYPE_COMMENT);
    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CD11,
                                 SPH_COMMON_KEYWORD_WCS_CD_COMMENT);
    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CD12,
                                 SPH_COMMON_KEYWORD_WCS_CD_COMMENT);
    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CD21,
                                 SPH_COMMON_KEYWORD_WCS_CD_COMMENT);
    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CD22,
                                 SPH_COMMON_KEYWORD_WCS_CD_COMMENT);

    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXXUNIT,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXUNIT_COMMENT);
    cpl_propertylist_set_comment(self, SPH_COMMON_KEYWORD_WCS_CRPIXYUNIT,
                                 SPH_COMMON_KEYWORD_WCS_CRPIXUNIT_COMMENT);

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}


cpl_frameset*
sph_ird_common_science_create_fctables(cpl_frameset* rawframes,
        cpl_frame* fcframe, int keep_fctable, int verbose) {
    cpl_frameset* fctable_frames = NULL;
    cpl_frame* aframe = NULL;
    sph_fctable* fctab = NULL;
    char timestamp[256];
    int ii = 0;

    cpl_ensure( rawframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( fcframe, CPL_ERROR_NULL_INPUT, NULL);

    fctable_frames = sph_ird_common_science_create_fctables_internal(fcframe,
            rawframes, keep_fctable);

    cpl_ensure( fctable_frames, cpl_error_get_code(), NULL);

    if (verbose) {
        aframe = cpl_frameset_get_first(fctable_frames);
        while (aframe) {
            fctab = sph_fctable_load_ascii(cpl_frame_get_filename(aframe));

            if (!fctab) {
                SPH_ERROR_RAISE_ERR(CPL_ERROR_FILE_IO,
                        "Could not load field center table.");
                break;
            }

            for (ii = 0; ii < sph_fctable_get_size(fctab); ++ii) {
                sph_time_mjd_to_iso8601string(sph_fctable_get_time(fctab, ii),
                        timestamp);

                SPH_ERROR_RAISE_INFO(
                        CPL_ERROR_CONTINUE,
                        "Extracted angle %7.4f at time %s from "
                        "fctable file %s", sph_fctable_get_angle(fctab,ii), timestamp, cpl_frame_get_filename(aframe));
            }
            aframe = cpl_frameset_get_next(fctable_frames);

            sph_fctable_delete(fctab);
            fctab = NULL;
        }
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return fctable_frames;
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Delete sph_common_science structure
 * @param self   pointer to structure to delete
 */
/*----------------------------------------------------------------------------*/
void sph_ird_common_science_delete(sph_ird_common_science* self) {

    if (self) {
        sph_master_frame_delete(self->dark);
        sph_master_frame_delete(self->flat);
        cpl_image_delete(self->bpix);
        sph_distortion_model_delete(self->distort_left);
        sph_distortion_model_delete(self->distort_right);
        sph_filter_table_delete(self->filter_table);
        sph_ird_instrument_model_delete(self->irdmodel);
        self->irdmodel = NULL;
        sph_transform_delete(self->transform);
        self->transform = NULL;
        if (self->tff)
            sph_master_frame_delete(self->tff);
        self->tff = NULL;
        cpl_free(self);
    }
}

/*----------------------------------------------------------------------------*/
/**
 * @brief Process a set of raw input cubes
 * @param inframes      the frameset containing the IRDIS raw cubes
 * @param scframes      the frameset containing the star center tables (optional)
 * @param dark          the master dark (optional)
 * @param flat          the master flat (optional)
 * @param filer_frame   the frame of the filter table (optional)
 * @param distortion    the distortion frame (optional)
 * @param inparams      the input parameterlist
 * @param minr          the minimum annulus radius
 * @param maxr          the maximim annulus radius
 * @param filter_radius the size of the filter kernel
 * @param rotate        flag to set rotation on or off
 * @param sdi           flag to set SDI (scaling+difference imaging) on or off
 * @param tag           the tag to give to resulting frames
 * @param recipe        the name of the calling recipe
 * @param left_out      output frameset of processed left FOV (optional)
 * @param right_out     output frameset of processed right FOV (optional)
 * @param total_out     output frames of processed sum or difference (optional)
 * @param left_procatg  PRO CATG of left  FOV product, or NULL
 * @param right_procatg PRO CATG of right FOV product, or NULL
 * @param total_procatg PRO CATG of sum/difference FOV product, or NULL
 *
 * @return error code
 *
 * This function processes a set of input cubes. Most inputs are optional.
 * Depending on whether SDI is selected or not, a resulting total frameset is
 * either the sum of left and right FOV divided by 2 or the difference of
 * right FOV and left FOV (when SDI is on).
 * Note that if SDI is off, frames will NOT be rescaled even if the filter in the
 * filter list has different wavelengths for left and right FOV.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_ird_common_science_process_cubes(sph_ird_common_science* sci,
                                                    const cpl_frameset* rawframes,
                                                    cpl_frameset* scframes,
                                                    cpl_frameset** left_out,
                                                    cpl_frameset** right_out,
                                                    cpl_frameset** total_out,
                                                    const char* left_procatg,
                                                    const char* right_procatg,
                                                    const char* total_procatg) {
    const cpl_size nframes = cpl_frameset_get_size(rawframes);
    sph_master_frame* mdark = NULL;
    sph_master_frame* mflat = NULL;
    sph_distortion_model* distort_left = NULL;
    sph_distortion_model* distort_right = NULL;
    double xcl = 0.0;
    double xcr = 0.0;
    double ycl = 0.0;
    double ycr = 0.0;
    int ff = 0;
    cpl_frameset* loopframes = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_error_code rerrtmp = CPL_ERROR_NONE;

    cpl_ensure_code(sci, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sci->rotate_flag == 0, CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(sci->inframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sci->inparams, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(rawframes, CPL_ERROR_NULL_INPUT);

    if (scframes) {
        if (nframes != cpl_frameset_get_size(scframes)) {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "The number of frames in the star center frameset "
                    "is not the same "
                    "as the number of raw files. "
                    "Please ensure that the .sof file "
                    "specifies exactly as many star center frames as raw frames.");
            return cpl_error_get_code();
        }
    } else {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "No star center table provided. "
                "Am assuming dithering is not used.");
    }

    sph_ird_instrument_model_get_centre_left_subframe_coords(sci->irdmodel,
            &xcl, &ycl);
    sph_ird_instrument_model_get_centre_right_subframe_coords(sci->irdmodel,
            &xcr, &ycr);
    if (left_out)
        *left_out = cpl_frameset_new();
    if (right_out)
        *right_out = cpl_frameset_new();
    if (total_out)
        *total_out = cpl_frameset_new();
    SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;

    for (ff = 0; ff < nframes; ++ff) {
        const cpl_frame* aframe = cpl_frameset_get_position_const(rawframes, ff);
        const char* filename = cpl_frame_get_filename(aframe);
        char* basename = sph_filemanager_remove_dir(filename);

        if (left_out) {
            char* fname_left = sph_filemanager_new_filename_from_base(basename,
                                                                      "left");
            cpl_frame* cube_frame_left = cpl_frame_new();

        
            cpl_frame_set_filename(cube_frame_left, fname_left);
            cpl_frame_set_tag(cube_frame_left, sci->tag);
            cpl_frame_set_group(cube_frame_left, CPL_FRAME_GROUP_PRODUCT);
            cpl_frame_set_level(cube_frame_left, CPL_FRAME_LEVEL_FINAL);

            cpl_frameset_insert(*left_out, cube_frame_left);
            cpl_free(fname_left);
        }
        if (right_out) {
            char* fname_right = sph_filemanager_new_filename_from_base(basename,
                                                                       "right");
            cpl_frame* cube_frame_right = cpl_frame_new();

        
            cpl_frame_set_filename(cube_frame_right, fname_right);
            cpl_frame_set_tag(cube_frame_right, sci->tag);
            cpl_frame_set_group(cube_frame_right, CPL_FRAME_GROUP_PRODUCT);
            cpl_frame_set_level(cube_frame_right, CPL_FRAME_LEVEL_FINAL);

            cpl_frameset_insert(*right_out, cube_frame_right);
            cpl_free(fname_right);
        }
        if (total_out) {
            char* fname_total = sph_filemanager_new_filename_from_base(basename,
                                                                       "total");
            cpl_frame* cube_frame_total = cpl_frame_new();

        
            cpl_frame_set_filename(cube_frame_total, fname_total);
            cpl_frame_set_tag(cube_frame_total, sci->tag);
            cpl_frame_set_group(cube_frame_total, CPL_FRAME_GROUP_PRODUCT);
            cpl_frame_set_level(cube_frame_total, CPL_FRAME_LEVEL_FINAL);

            cpl_frameset_insert(*total_out, cube_frame_total);
            cpl_free(fname_total);
        }
        cpl_free(basename);
    }

    // Need to make duplicate since DFS product frames are added to inframes
    // but those should not to be looped over.
    loopframes = cpl_frameset_duplicate(rawframes);
#ifdef _OPENMP
#pragma omp parallel for firstprivate(rerr,rerrtmp), lastprivate(rerr)
#endif
    for (ff = 0; ff < cpl_frameset_get_size(loopframes); ++ff) {
        if ((rerrtmp = sph_ird_common_science_processing_iteration__(sci,
                loopframes, scframes, left_out, right_out, total_out, ff,
                (int) xcl, (int) ycl, (int) xcr, (int) ycr,
                left_procatg, right_procatg, total_procatg))
                != CPL_ERROR_NONE) {
            rerr = rerrtmp;
        }
    }
    if (rerr == CPL_ERROR_DATA_NOT_FOUND) {
        SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,
                "Could not find matching fctable "
                "for some input frames."
                "Please double check the input.");
    }
    assert (cpl_frameset_get_size(loopframes) == nframes);
    SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
    EXIT: cpl_frameset_delete(loopframes);
    loopframes = NULL;
    sph_master_frame_delete(mdark);
    mdark = NULL;
    sph_master_frame_delete(mflat);
    mflat = NULL;
    sph_distortion_model_delete(distort_left);
    distort_left = NULL;
    sph_distortion_model_delete(distort_right);
    distort_right = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*----------------------------------------------------------------------------*/
/**
 */
/*----------------------------------------------------------------------------*/
static sph_ird_common_science*
sph_ird_common_science_new__(void) {
    sph_ird_common_science* result = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    result = cpl_calloc(1, sizeof(sph_ird_common_science));
    result->least_square = 1;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return result;
}

static sph_error_code sph_ird_common_science_process_single_cube(
        sph_ird_common_science* sci, const cpl_frame* frame_of_raw_cube,
        cpl_vector* cx_left, cpl_vector* cy_left, cpl_vector* cx_right,
        cpl_vector* cy_right, sph_cube* left_cube, sph_cube* right_cube,
        sph_cube* total_cube) {
    cpl_vector* posind = NULL;
    cpl_vector* negind = NULL;
    cpl_vector* normalisations = NULL;
    cpl_image* rawimage = NULL;
    sph_master_frame* calibmframe = NULL;
    sph_master_frame* leftmframe = NULL;
    sph_master_frame* rightmframe = NULL;
    sph_master_frame* tmppointer = NULL;
    double scale_left = 1.0;
    double scale_right = 1.0;
    double lambda_left = 1.0;
    double lambda_right = 1.0;
    cpl_imagelist* diffimlist = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    const char* filtername = NULL;
    int pp = 0;
    cpl_vector* centx_left = NULL;
    cpl_vector* centy_left = NULL;
    cpl_vector* centx_right = NULL;
    cpl_vector* centy_right = NULL;
    cpl_propertylist* raw_pl = NULL;

    int nplanes = 0;
    cpl_ensure_code(frame_of_raw_cube, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sci, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cx_left, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cy_left, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cx_right, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(cy_right, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(
            cpl_vector_get_size(cx_left) == cpl_vector_get_size(cy_left),
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(
            cpl_vector_get_size(cx_left) == cpl_vector_get_size(cx_right),
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(
            cpl_vector_get_size(cx_left) == cpl_vector_get_size(cy_right),
            CPL_ERROR_ILLEGAL_INPUT);
    nplanes = sph_fits_get_nplanes(cpl_frame_get_filename(frame_of_raw_cube),
            0);
    if (nplanes != cpl_vector_get_size(cx_left)) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_ILLEGAL_INPUT,
                "The number of entries in the "
                "field centre table is not equal to the number of planes in the "
                "raw frames. Please check the field centre table, if needed, "
                "recreate one using the sph_ird_star_center recipe.");
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    sph_ird_instrument_model_adjust_centre_coords_v(sci->irdmodel, cx_left,
            cy_left, &centx_left, &centy_left);
    sph_ird_instrument_model_adjust_centre_coords_v(sci->irdmodel, cx_right,
            cy_right, &centx_right, &centy_right);

    if (sci->filter_table && sci->flag_sdi) {
        cpl_errorstate prestate = cpl_errorstate_get();
        raw_pl = cpl_propertylist_load(
                cpl_frame_get_filename(frame_of_raw_cube), 0);
        if (raw_pl != NULL
                && cpl_propertylist_get_type(raw_pl,
                        SPH_IRD_KEYWORD_COMMON_FILTER_NAME) == CPL_TYPE_STRING) {
            filtername = cpl_propertylist_get_string(raw_pl,
                    SPH_IRD_KEYWORD_COMMON_FILTER_NAME);
            SPH_ERROR_RAISE_INFO(SPH_ERROR_INFO,
                    "Using filter %s!", filtername);
        } else {
            filtername = sci->filter_table->filterNames[0];
            SPH_ERROR_RAISE_WARNING(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Could not find any filter name in the subframe properties!");
            cpl_errorstate_set(prestate);
        }
    }

    if (filtername != NULL
            && sph_filter_table_get_left_right_lambda(sci->filter_table,
                    filtername, &lambda_left, &lambda_right) == CPL_ERROR_NONE) {
        if (lambda_left < lambda_right) {
            scale_left = 1.0;
            scale_right = lambda_left / lambda_right;
        } else {
            scale_right = 1.0;
            scale_left = lambda_right / lambda_left;
        }SPH_ERROR_RAISE_INFO(CPL_ERROR_NONE,
                "lambda (left,right): %f,%f", lambda_left, lambda_right);
        SPH_ERROR_RAISE_INFO(CPL_ERROR_NONE,
                "Scaling by (left,right): %f,%f", scale_left, scale_right);
    } else {
        scale_left = 1.0;
        scale_right = 1.0;
        if (sci->flag_sdi) {
            SPH_ERROR_RAISE_WARNING(CPL_ERROR_ILLEGAL_INPUT,
                    "Am not rescaling since no filter table "
                    "could be read.");
        }
    }

    if (sci->flag_sdi) {
        posind = cpl_vector_new(1);
        negind = cpl_vector_new(1);

        if (lambda_left < lambda_right) {
            cpl_vector_set(posind, 0, 0);
            cpl_vector_set(negind, 0, 1);
        } else {
            cpl_vector_set(posind, 0, 1);
            cpl_vector_set(negind, 0, 0);
        }
    }

    cpl_propertylist_delete(raw_pl); /* filtername not used any longer */

    for (pp = 0; pp < nplanes; ++pp) {
        cpl_msg_info(cpl_func, "Processing plane %d of %d", 1+pp, nplanes);
        cpl_msg_info(cpl_func,"Trying to load file %s!",cpl_frame_get_filename(frame_of_raw_cube));
        rawimage = cpl_image_load(cpl_frame_get_filename(frame_of_raw_cube),
                CPL_TYPE_DOUBLE, pp, 0);
        if (rawimage) {
            calibmframe = sph_common_science_calibrate_raw(rawimage, sci->dark,
                    sci->flat);

            if (calibmframe) {
                cpl_propertylist_delete(calibmframe->properties);
                calibmframe->properties = NULL;
                cpl_image_delete(rawimage);
                rawimage = NULL;
                if (sci->bpix)
                    sph_master_frame_set_bads_from_image(calibmframe,
                            sci->bpix);

                leftmframe = sph_ird_instrument_model_extract_left_master_frame(
                        sci->irdmodel, calibmframe);
                rightmframe =
                        sph_ird_instrument_model_extract_right_master_frame(
                                sci->irdmodel, calibmframe);

                rerr = sph_transform_apply(sci->transform, leftmframe,
                        sci->distort_left, cpl_vector_get(centx_left, pp),
                        cpl_vector_get(centy_left, pp), 0.0, scale_left,
                        sci->irdmodel);
                SPH_ERROR_ENSURE_GOTO_EXIT(rerr == CPL_ERROR_NONE, rerr);
                rerr = sph_transform_apply(sci->transform, rightmframe,
                        sci->distort_right, cpl_vector_get(centx_right, pp),
                        cpl_vector_get(centy_right, pp), 0.0, scale_right,
                        sci->irdmodel);
                SPH_ERROR_ENSURE_GOTO_EXIT(rerr == CPL_ERROR_NONE, rerr);
                if (sci->tff) {
                    sph_master_frame_divide_master_frame(leftmframe, sci->tff);
                }
                if (sci->tff) {
                    sph_master_frame_divide_master_frame(rightmframe, sci->tff);
                }

                if (left_cube) {
                    SPH_ERROR_ENSURE_GOTO_EXIT(
                            sph_cube_append_master( left_cube, leftmframe, (float)pp) == CPL_ERROR_NONE,
                            CPL_ERROR_FILE_IO);
                }
                if (right_cube) {
                    SPH_ERROR_ENSURE_GOTO_EXIT(
                            sph_cube_append_master( right_cube, rightmframe, (float)pp) == CPL_ERROR_NONE,
                            CPL_ERROR_FILE_IO);
                }
                if (total_cube) {
                    if (sci->flag_sdi) {
                        diffimlist = cpl_imagelist_new();
                        cpl_imagelist_set(diffimlist,
                                sph_master_frame_extract_image(leftmframe, 1),
                                0);
                        cpl_imagelist_set(diffimlist,
                                sph_master_frame_extract_image(rightmframe, 1),
                                1);

                        normalisations =
                                sph_andromeda_support_calc_normalisations(
                                        diffimlist, posind, negind, sci->minr,
                                        sci->maxr, sci->least_square);
                        cpl_imagelist_delete(diffimlist);
                        diffimlist = NULL;
                        if (lambda_left < lambda_right) { // left frame is short wavelength
                            sph_master_frame_multiply_double(rightmframe,
                                    cpl_vector_get(normalisations, 0));
                            SPH_ERROR_RAISE_INFO(
                                    SPH_ERROR_GENERAL,
                                    "Applied (multiplied) a renormalisation of %f to right frame", cpl_vector_get(normalisations,0));
                            sph_master_frame_subtract_master_frame(leftmframe,
                                    rightmframe);
                        } else { // right frame is short wavelength
                            sph_master_frame_multiply_double(leftmframe,
                                    cpl_vector_get(normalisations, 0));
                            SPH_ERROR_RAISE_INFO(
                                    SPH_ERROR_GENERAL,
                                    "Applied (multiplied) a renormalisation of %f to left frame", cpl_vector_get(normalisations,0));
                            sph_master_frame_subtract_master_frame(rightmframe,
                                    leftmframe);
                            tmppointer = leftmframe;
                            leftmframe = rightmframe;
                            rightmframe = tmppointer;
                            tmppointer = NULL;
                        }
                        cpl_vector_delete(normalisations);
                        normalisations = NULL;
                    } else {
                        sph_master_frame_add_master_frame(leftmframe,
                                rightmframe);
                        sph_master_frame_multiply_double(leftmframe, 0.5);
                    }SPH_ERROR_ENSURE_GOTO_EXIT(
                            sph_cube_append_master( total_cube, leftmframe, (float)pp) == CPL_ERROR_NONE,
                            CPL_ERROR_FILE_IO);
                }
                sph_master_frame_delete(calibmframe);
                calibmframe = NULL;
                sph_master_frame_delete(leftmframe);
                leftmframe = NULL;
                sph_master_frame_delete(rightmframe);
                rightmframe = NULL;
                SPH_ERROR_CHECK_STATE_ONERR_GOTO_EXIT;
            }
        }
        else{
            SPH_ERROR_RAISE_ERR(SPH_ERROR_INCONSISTENT_INPUT,"Cannot read or understand the file!")
        }
    }

    EXIT: cpl_vector_delete(posind);
    posind = NULL;
    cpl_vector_delete(negind);
    negind = NULL;
    cpl_vector_delete(centx_left);
    centx_left = NULL;
    cpl_vector_delete(centy_left);
    centy_left = NULL;
    cpl_vector_delete(centx_right);
    centx_right = NULL;
    cpl_vector_delete(centy_right);
    centy_right = NULL;
    sph_master_frame_delete(calibmframe);
    calibmframe = NULL;
    sph_master_frame_delete(leftmframe);
    leftmframe = NULL;
    sph_master_frame_delete(rightmframe);
    rightmframe = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_error_code sph_ird_common_science_processing_iteration__(
        sph_ird_common_science* sci, cpl_frameset* loopframes,
        cpl_frameset* scframes, cpl_frameset** left_out_frames,
        cpl_frameset** right_out_frames, cpl_frameset** total_out_frames,
        int ff, int xcl, int ycl, int xcr, int ycr,
                                     const char* left_procatg,
                                     const char* right_procatg,
                                     const char* total_procatg) {
    const cpl_frame* aframe  = NULL;
    sph_fctable* fctab      = NULL;
    cpl_vector* cent_lx     = NULL;
    cpl_vector* cent_ly     = NULL;
    cpl_vector* cent_rx     = NULL;
    cpl_vector* cent_ry     = NULL;
    cpl_propertylist* plist = NULL;
    sph_cube* left_cube     = NULL;
    sph_cube* right_cube    = NULL;
    sph_cube* total_cube    = NULL;
    cpl_frameset* aframeset = NULL;

    aframe = cpl_frameset_get_position_const(loopframes, ff);
    plist  = cpl_propertylist_load(cpl_frame_get_filename(aframe), 0);

    if (scframes) {
        fctab = sph_fctable_find_fctable(aframe, scframes);
        if (fctab) {
            cent_lx = sph_fctable_get_centre_x_left(fctab);
            cent_ly = sph_fctable_get_centre_y_left(fctab);
            cent_rx = sph_fctable_get_centre_x_right(fctab);
            cent_ry = sph_fctable_get_centre_y_right(fctab);
        } else {
            return CPL_ERROR_DATA_NOT_FOUND;
        }
    } else {
        cent_lx = cpl_vector_new(
                sph_fits_get_nplanes(cpl_frame_get_filename(aframe), 0));
        cent_ly = cpl_vector_duplicate(cent_lx);
        cent_rx = cpl_vector_duplicate(cent_lx);
        cent_ry = cpl_vector_duplicate(cent_lx);

        cpl_vector_fill(cent_lx, xcl);
        cpl_vector_fill(cent_ly, ycl);
        cpl_vector_fill(cent_rx, xcr);
        cpl_vector_fill(cent_ry, ycr);
    }

    aframeset = sph_common_science_frameset_extract_raw_single(sci->inframes,
                                                               aframe);

    if (left_out_frames != NULL) {
        const cpl_frame* oframe =
            cpl_frameset_get_position_const(*left_out_frames, ff);

        left_cube = sph_cube_new_dfs(cpl_frame_get_filename(oframe),
                                     aframeset, aframe, sci->inparams,
                                     sci->tag, sci->recipe,
                                     SPH_PIPELINE_NAME_IRDIS, plist);

        cpl_dfs_setup_product_header(left_cube->proplist, oframe,
                                     aframeset, sci->inparams,
                                     sci->recipe, SPH_PIPELINE_NAME_IRDIS,
                                     "PRO-1.16", NULL);
        if (left_procatg != NULL)
            cpl_propertylist_update_string(left_cube->proplist,
                                           SPH_COMMON_KEYWORD_PRO_CATG,
                                           left_procatg);
        /* FIXME: Set WCS to dummy (pixel) value for now */
        sph_utils_reset_wcs_12d(left_cube->proplist);

        /* Update the header if required */
        sph_utils_update_header(left_cube->proplist);
    }
    if (right_out_frames != NULL) {
        const cpl_frame* oframe =
            cpl_frameset_get_position_const(*right_out_frames, ff);

        right_cube = sph_cube_new_dfs(cpl_frame_get_filename(oframe),
                                      aframeset, aframe, sci->inparams,
                                      sci->tag, sci->recipe,
                                      SPH_PIPELINE_NAME_IRDIS, plist);
        cpl_dfs_setup_product_header(right_cube->proplist, oframe,
                                     aframeset, sci->inparams,
                                     sci->recipe, SPH_PIPELINE_NAME_IRDIS,
                                     "PRO-1.16", NULL);
        if (right_procatg != NULL)
            cpl_propertylist_update_string(right_cube->proplist,
                                           SPH_COMMON_KEYWORD_PRO_CATG,
                                           right_procatg);
        /* FIXME: Set WCS to dummy (pixel) value for now */
        sph_utils_reset_wcs_12d(right_cube->proplist);

        /* Update the header if required */
        sph_utils_update_header(right_cube->proplist);
    }
    if (total_out_frames != NULL) {
        const cpl_frame* oframe =
            cpl_frameset_get_position_const(*total_out_frames, ff);

        total_cube = sph_cube_new_dfs(cpl_frame_get_filename(oframe),
                                      aframeset, aframe, sci->inparams,
                                      sci->tag, sci->recipe,
                                      SPH_PIPELINE_NAME_IRDIS, plist);
        cpl_dfs_setup_product_header(total_cube->proplist, oframe,
                                     aframeset, sci->inparams,
                                     sci->recipe, SPH_PIPELINE_NAME_IRDIS,
                                     "PRO-1.16", NULL);
        if (total_procatg != NULL)
            cpl_propertylist_update_string(total_cube->proplist,
                                           SPH_COMMON_KEYWORD_PRO_CATG,
                                           total_procatg);
        /* FIXME: Set WCS to dummy (pixel) value for now */
        sph_utils_reset_wcs_12d(total_cube->proplist);

        /* Update the header if required */
        sph_utils_update_header(total_cube->proplist);
    }
    sph_ird_common_science_process_single_cube(sci, aframe, cent_lx, cent_ly,
            cent_rx, cent_ry, left_cube, right_cube, total_cube);
    if (left_cube)
        {
        sph_cube_finalise_file(left_cube->filename);
        sph_filemanager_add_tmp_file(left_cube->filename);
        }
    if (right_cube)
        {
        sph_cube_finalise_file(right_cube->filename);
        sph_filemanager_add_tmp_file(right_cube->filename);
        }
    if (total_cube)
        {
        sph_cube_finalise_file(total_cube->filename);
        sph_filemanager_add_tmp_file(total_cube->filename);
        }
    sph_cube_delete(left_cube);
    left_cube = NULL;
    sph_cube_delete(right_cube);
    right_cube = NULL;
    sph_cube_delete(total_cube);
    total_cube = NULL;
    cpl_vector_delete(cent_lx);
    cent_lx = NULL;
    cpl_vector_delete(cent_ly);
    cent_ly = NULL;
    cpl_vector_delete(cent_rx);
    cent_rx = NULL;
    cpl_vector_delete(cent_ry);
    cent_ry = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    cpl_propertylist_delete(plist);
    plist = NULL;
    cpl_frameset_delete(aframeset);
    return CPL_ERROR_NONE;
}

static cpl_frameset*
sph_ird_common_science_create_fctables_internal(cpl_frame* fcframe,
        cpl_frameset* rawframes, int keep_fctable) {
    int ff = 0;
    char* tmpfilename = NULL;
    char* tmpfilename2 = NULL;
    sph_fctable* fctable = NULL;
    cpl_propertylist* plist = NULL;
    cpl_table* result = NULL;
    cpl_frameset* resultset = NULL;
    cpl_frame* aframe = NULL;

    cpl_ensure( rawframes, CPL_ERROR_NULL_INPUT, NULL);

    resultset = cpl_frameset_new();
    plist = cpl_propertylist_new();

    for (ff = 0; ff < cpl_frameset_get_size(rawframes); ++ff) {
        fctable = sph_ird_common_science_create_table_single_cube(fcframe,
                cpl_frameset_get_position(rawframes, ff));

        if (fctable) {
            tmpfilename2 = sph_fctable_construct_filename(
                    cpl_frameset_get_position(rawframes, ff), 1);

            aframe = cpl_frame_new();
            cpl_frame_set_tag(aframe, SPH_IRD_TAG_FCTABLE_CALIB);
            cpl_frame_set_filename(aframe, tmpfilename2);
            if (!keep_fctable)
                sph_filemanager_add_tmp_file(tmpfilename2);
            cpl_frameset_insert(resultset, aframe);
            sph_fctable_save_ascii(fctable, tmpfilename2);
            sph_fctable_delete(fctable);
            fctable = NULL;
            cpl_table_delete(result);
            result = NULL;
            cpl_free(tmpfilename);
            tmpfilename = NULL;
            cpl_free(tmpfilename2);
            tmpfilename2 = NULL;
        } else {
            break;
        }
    }
    cpl_propertylist_delete(plist);
    plist = NULL;
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return resultset;
}

static sph_fctable*
sph_ird_common_science_create_table_single_cube(cpl_frame* fcframe,
        cpl_frame* arawframe) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_fctable* fctab = NULL;
    double crx = 0.0;
    double cry = 0.0;
    double clx = 0.0;
    double cly = 0.0;
    double dmsx = 0.0;
    double dmsy = 0.0;
    int ff = 0;
    int nplanes = 0;
    cpl_vector* timev = NULL;
    cpl_vector* angles = NULL;
    cpl_propertylist* pl = NULL;

    fctab = sph_fctable_create_fctable(arawframe, SPH_FCTAB_DOUBLE);
    if (!fctab) {
        SPH_ERR("Could not create centre positions table.");
        return NULL;
    }
    nplanes = sph_fits_get_nplanes(cpl_frame_get_filename(arawframe), 0);

    SPH_ERROR_ENSURE_GOTO_EXIT( nplanes == sph_fctable_get_size(fctab),
            CPL_ERROR_INCOMPATIBLE_INPUT);

    timev = sph_fctable_get_times(fctab);

    pl = sph_keyword_manager_load_properties(cpl_frame_get_filename(arawframe),
            0);
    SPH_ERROR_ENSURE_GOTO_EXIT(pl != NULL, CPL_ERROR_NULL_INPUT);
    SPH_ERROR_ENSURE_GOTO_EXIT(
            cpl_propertylist_has( pl, SPH_IRD_KEYWORD_DMS_POS_X ),
            CPL_ERROR_DATA_NOT_FOUND);
    SPH_ERROR_ENSURE_GOTO_EXIT(
            cpl_propertylist_has( pl, SPH_IRD_KEYWORD_DMS_POS_Y ),
            CPL_ERROR_DATA_NOT_FOUND);
    dmsx = cpl_propertylist_get_double(pl, SPH_IRD_KEYWORD_DMS_POS_X)
            / SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS;
    dmsy = cpl_propertylist_get_double(pl, SPH_IRD_KEYWORD_DMS_POS_Y)
            / SPH_IRD_INSTRUMENT_MODEL_PIX_SIZE_MICRONS;

    if (cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_DROT2_MODE)
            && strcasecmp(
                    cpl_propertylist_get_string(pl,
                            SPH_COMMON_KEYWORD_DROT2_MODE),
                    SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_PUPIL_STAB) == 0) {
        angles = sph_extract_angles_from_cube(arawframe,
                                      SPH_IRD_INSTRUMENT_MODEL_DEROT_OFFSETELEV,
                                              0.0);
    } else if (cpl_propertylist_has(pl, SPH_COMMON_KEYWORD_DROT2_MODE)
            && strcasecmp(
                    cpl_propertylist_get_string(pl,
                            SPH_COMMON_KEYWORD_DROT2_MODE),
                    SPH_COMMON_KEYWORD_VALUE_DROT2_MODE_FIELD_STAB) == 0) {
        angles = sph_extract_angles_from_cube(arawframe,
                                       SPH_IRD_INSTRUMENT_MODEL_DEROT_OFFSETSKY,
                                              0.0);
    } else {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_INCOMPATIBLE_INPUT,
                "The DEROT mode could not be read.");
    }
    if (angles != NULL) {
#ifndef NDEBUG
        const int mplanes = (int)cpl_vector_get_size(angles);
#endif
        assert(mplanes == nplanes);
        for (ff = 0; ff < nplanes; ++ff) {
            rerr = sph_ird_common_science_interpolate_centers(fcframe, timev,
                    ff, &clx, &cly, &crx, &cry);

            SPH_RAISE_CPL;

            SPH_ERROR_ENSURE_GOTO_EXIT( rerr == CPL_ERROR_NONE, rerr);

            clx += dmsx;
            cly += dmsy;
            crx += dmsx;
            cry += dmsy;

            SPH_ERROR_RAISE_INFO(
                    SPH_ERROR_GENERAL,
                    "Setting centre for %d as %f,%f and %f,%f.", ff, clx, cly, crx, cry);

            rerr = sph_fctable_set_cent_double(fctab, ff, clx, cly, crx, cry);

            sph_fctable_set_angle(fctab, ff, cpl_vector_get(angles, ff), 0.0);
        }
    }
    cpl_propertylist_delete(pl);
    pl = NULL;
    cpl_vector_delete(timev);
    timev = NULL;
    cpl_vector_delete(angles);
    angles = NULL;
    return fctab;
    EXIT: cpl_vector_delete(timev);
    timev = NULL;
    sph_fctable_delete(fctab);
    fctab = NULL;
    return NULL;
}

static sph_error_code sph_ird_common_science_interpolate_centers(
        cpl_frame* fcframe, cpl_vector* timev, int plane, double* clx,
        double* cly, double* crx, double* cry) {
    double time_mjd = 0.0;
    double dtime1 = 10000000.0;
    double dtime2 = 10000000.0;
    double time1 = 0.0;
    double time2 = 0.0;
    double atime = 0.0;
    double dmsx = 0.0;
    double dmsy = 0.0;
    int rr = 0;
    int r1 = -1;
    int r2 = -1;
    int bpix = 0;
    double f = 0.0;
    cpl_table* interntab = NULL;
    cpl_error_code errc = CPL_ERROR_NONE;

    cpl_ensure_code( fcframe, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( timev, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( clx, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( cly, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( crx, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( cry, CPL_ERROR_NULL_INPUT);

    time_mjd = cpl_vector_get(timev, plane);

    // Only one star center frame provided so take coords from that
    interntab = cpl_table_load(cpl_frame_get_filename(fcframe), 1, 0);
    if (interntab == NULL) {
        SPH_ERROR_RAISE_ERR(
                CPL_ERROR_ILLEGAL_OUTPUT,
                "Error loading star center calibration file from %s", cpl_frame_get_filename( fcframe ));
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    if (cpl_table_get_nrow(interntab) == 1) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "Using single row field center table");
        dmsx = cpl_table_get_double(interntab,
                SPH_IRD_STAR_CENTER_DMS_POS_X_NAME, 0, &bpix);
        dmsy = cpl_table_get_double(interntab,
                SPH_IRD_STAR_CENTER_DMS_POS_Y_NAME, 0, &bpix);
        *clx = cpl_table_get_double(interntab,
                SPH_IRD_STAR_CENTER_CENTLEFTX_NAME, 0, &bpix) - dmsx;
        *cly = cpl_table_get_double(interntab,
                SPH_IRD_STAR_CENTER_CENTLEFTY_NAME, 0, &bpix) - dmsy;
        *crx = cpl_table_get_double(interntab,
                SPH_IRD_STAR_CENTER_CENTRIGHTX_NAME, 0, &bpix) - dmsx;
        *cry = cpl_table_get_double(interntab,
                SPH_IRD_STAR_CENTER_CENTRIGHTY_NAME, 0, &bpix) - dmsy;
        cpl_table_delete(interntab);
        interntab = NULL;
        errc = cpl_error_get_code();
        if (errc != CPL_ERROR_NONE){
            SPH_ERROR_RAISE_ERR(
                            errc,
                            "Error reading star center positions from  file %s! Does it have the right format (column names with \"_\")?", cpl_frame_get_filename( fcframe ));
        }
        return errc;
    }

    if (sph_time_mjd_from_string(
            &atime,
            cpl_table_get_string(interntab, SPH_IRD_STAR_CENTER_TIME_UT_NAME,
                    cpl_table_get_nrow(interntab) - 1)) != CPL_ERROR_NONE) {
        cpl_table_delete(interntab);
        interntab = NULL;
        return cpl_error_get_code();
    }

    if (atime < time_mjd) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "The latest timestamp for the star center "
                "calibration data is %7.3f which is before %7.3f "
                "of the science frame. This is not allowed."
                "Rerun the star center calibration with input frames"
                " taken after the science frames or run it with a single"
                " input raw frame or remove the star center calibration "
                "inputs altogether.", atime, time_mjd);
        cpl_table_delete(interntab);
        interntab = NULL;
        return CPL_ERROR_ILLEGAL_INPUT;
    }

    for (rr = 0; rr < cpl_table_get_nrow(interntab); ++rr) {
        cpl_ensure_code(sph_time_mjd_from_string(&atime, cpl_table_get_string(
                                interntab,
                                SPH_IRD_STAR_CENTER_TIME_UT_NAME,
                                rr) ) == CPL_ERROR_NONE, cpl_error_get_code() );
        if (atime <= time_mjd && fabs(atime - time_mjd) < dtime1) {
            dtime1 = fabs(atime - time_mjd);
            r1 = rr;
            time1 = atime;
        }
    }
    for (rr = 0; rr < cpl_table_get_nrow(interntab); ++rr) {
        if (sph_time_mjd_from_string(
                &atime,
                cpl_table_get_string(interntab,
                        SPH_IRD_STAR_CENTER_TIME_UT_NAME, rr))
                != CPL_ERROR_NONE) {
            cpl_table_delete(interntab);
            interntab = NULL;
            return cpl_error_get_code();
        }
        if (atime >= time_mjd && fabs(atime - time_mjd) < dtime2 && rr != r1) {
            dtime2 = fabs(atime - time_mjd);
            r2 = rr;
            time2 = atime;
        }
    }

    if (r1 < 0) {
        SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                "There is a problem with the timestamps for the star center "
                "calibration data. Please chekc the data.");
        cpl_table_delete(interntab);
        interntab = NULL;
        return CPL_ERROR_ILLEGAL_INPUT;
    }
    if (r2 < 0) {
        f = 0.0;
        r2 = 0;
    } else {
        if (time2 < time1) {
            f = time2;
            time2 = time1;
            time1 = f;
        }SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
                "time1: %f, time2: %f,, timemjd:%f", time1, time2, time_mjd);
        if (time1 > time_mjd) {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "There is a problem with the timestamps for the star center "
                    "calibration data. Please chekc the data.");
            cpl_table_delete(interntab);
            interntab = NULL;
            return CPL_ERROR_UNSUPPORTED_MODE;
        }
        if (time2 < time_mjd) {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "There is a problem with the timestamps for the star center "
                    "calibration data. Please chekc the data.");
            cpl_table_delete(interntab);
            interntab = NULL;
            return CPL_ERROR_UNSUPPORTED_MODE;
        }

        f = (time_mjd - time1) / (time2 - time1);
    }
    dmsx = cpl_table_get_double(interntab, SPH_IRD_STAR_CENTER_DMS_POS_X_NAME,
            r1, &bpix)
            + f
                    * (cpl_table_get_double(interntab,
                            SPH_IRD_STAR_CENTER_DMS_POS_X_NAME, r2, &bpix)
                            - cpl_table_get_double(interntab,
                                    SPH_IRD_STAR_CENTER_DMS_POS_X_NAME, r1,
                                    &bpix));
    dmsy = cpl_table_get_double(interntab, SPH_IRD_STAR_CENTER_DMS_POS_Y_NAME,
            r1, &bpix)
            + f
                    * (cpl_table_get_double(interntab,
                            SPH_IRD_STAR_CENTER_DMS_POS_Y_NAME, r2, &bpix)
                            - cpl_table_get_double(interntab,
                                    SPH_IRD_STAR_CENTER_DMS_POS_Y_NAME, r1,
                                    &bpix));
    *clx = cpl_table_get_double(interntab, SPH_IRD_STAR_CENTER_CENTLEFTX_NAME,
            r1, &bpix)
            + f
                    * (cpl_table_get_double(interntab,
                            SPH_IRD_STAR_CENTER_CENTLEFTX_NAME, r2, &bpix)
                            - cpl_table_get_double(interntab,
                                    SPH_IRD_STAR_CENTER_CENTLEFTX_NAME, r1,
                                    &bpix)) - dmsx;
    *cly = cpl_table_get_double(interntab, SPH_IRD_STAR_CENTER_CENTLEFTY_NAME,
            r1, &bpix)
            + f
                    * (cpl_table_get_double(interntab,
                            SPH_IRD_STAR_CENTER_CENTLEFTY_NAME, r2, &bpix)
                            - cpl_table_get_double(interntab,
                                    SPH_IRD_STAR_CENTER_CENTLEFTY_NAME, r1,
                                    &bpix)) - dmsy;
    *crx = cpl_table_get_double(interntab, SPH_IRD_STAR_CENTER_CENTRIGHTX_NAME,
            r1, &bpix)
            + f
                    * (cpl_table_get_double(interntab,
                            SPH_IRD_STAR_CENTER_CENTRIGHTX_NAME, r2, &bpix)
                            - cpl_table_get_double(interntab,
                                    SPH_IRD_STAR_CENTER_CENTRIGHTX_NAME, r1,
                                    &bpix)) - dmsx;
    *cry = cpl_table_get_double(interntab, SPH_IRD_STAR_CENTER_CENTRIGHTY_NAME,
            r1, &bpix)
            + f
                    * (cpl_table_get_double(interntab,
                            SPH_IRD_STAR_CENTER_CENTRIGHTY_NAME, r2, &bpix)
                            - cpl_table_get_double(interntab,
                                    SPH_IRD_STAR_CENTER_CENTRIGHTY_NAME, r1,
                                    &bpix)) - dmsy;

    cpl_table_delete(interntab);
    interntab = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
