/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <cpl.h>
#include <math.h>
#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_gain.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_gain_and_ron.h"
/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

extern sph_error_code SPH_IRD_GAIN_GENERAL;
extern sph_error_code SPH_IRD_GAIN_FRAMES_MISSING;

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_gain_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dakr
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_gain.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_gain recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_gain_run(sph_ird_gain* self) {
    cpl_frameset* preproc_frameset = NULL;
    cpl_frameset* collapsed_frameset = NULL;
    cpl_propertylist* pl = NULL;
    sph_master_frame* gain_mframe = NULL;
    cpl_vector* gain = NULL;
    cpl_vector* ron = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    /*------------------------------------------------------------------
     -  Sorting the frames into framesets of equal DIT setup
     --------------------------------------------------------------------*/
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE;

    if (self->preproc) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                "Preprocessing frames to take out offset shifts in cubes...");
        preproc_frameset = cpl_frameset_new();
        if (sph_gain_and_ron_preprocessing(self->rawframes, preproc_frameset)
                != CPL_ERROR_NONE) {
            SPH_ERROR_RAISE_ERR(cpl_error_get_code(),
                    "Could not perform the preprocessing. "
                    "Rerun the recipe again with the "
                    "option --preproc=FALSE");
            cpl_frameset_delete(preproc_frameset);
            return cpl_error_get_code();
        }
    } else {
        preproc_frameset = cpl_frameset_duplicate(self->rawframes);
    }
    if (self->vacca) {
        collapsed_frameset = sph_framecombination_collapse_cubes(
                preproc_frameset, self->coll_alg, self->framecomb_parameterlist,
                SPH_RECIPE_NAME_IRD_GAIN, SPH_IRD_TAG_GAIN_CALIB);

        if (collapsed_frameset) {
            if (sph_gain_and_ron_vacca(collapsed_frameset, self->order, &gain,
                    &ron) == CPL_ERROR_NONE) {

                gain_mframe = sph_gain_and_ron_vacca_gain_image(
                        collapsed_frameset);

                pl = cpl_propertylist_new();
                cpl_vector_multiply(ron, gain);

                cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_GAIN,
                        cpl_vector_get_mean(gain));
                cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_RON,
                        cpl_vector_get_mean(ron));
                cpl_propertylist_update_double(pl,
                        SPH_COMMON_KEYWORD_QC_GAIN_RMS,
                        cpl_vector_get_stdev(gain));
                cpl_propertylist_update_double(pl,
                        SPH_COMMON_KEYWORD_QC_RON_RMS,
                        cpl_vector_get_stdev(ron));
                sph_master_frame_save_dfs(gain_mframe, self->gain_outfilename,
                        self->rawframes,
                        cpl_frameset_get_first(self->rawframes), self->inparams,
                        SPH_IRD_TAG_GAIN_CALIB, SPH_RECIPE_NAME_IRD_GAIN,
                        SPH_PIPELINE_NAME_IRDIS, pl);
                cpl_propertylist_delete(pl);
                pl = NULL;
                sph_master_frame_delete(gain_mframe);
                gain_mframe = NULL;
            }
        }
        cpl_frameset_delete(collapsed_frameset);
        collapsed_frameset = NULL;
    } else {
        rerr = sph_gain_run(self->rawframes, self->coll_alg,
                self->framecomb_parameterlist, self->gain_outfilename,
                self->nonlin_outfilename, self->nonlin_bpixfilename,
                self->inframes, self->inparams, self->static_badpixel_frame,
                SPH_IRD_TAG_GAIN_CALIB, SPH_IRD_TAG_NONLIN_CALIB,
                SPH_IRD_TAG_NONLINBAD_CALIB, SPH_RECIPE_NAME_IRD_GAIN,
                SPH_PIPELINE_NAME_IRDIS, self->lintolerance, self->order,self->save_addprod);
        cpl_frameset_delete(preproc_frameset);
        return rerr;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/**@}*/
