/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include "sph_common_keywords.h"
#include "sph_ird_keywords.h"

#include "sph_ird_tags.h"
#include "sph_ird_star_center.h"
#include "sph_master_frame.h"
#include "sph_error.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_ird_instrument_model.h"
#include "sph_framecombination.h"
#include "sph_ird_common_science.h"
#include "sph_keyword_manager.h"
#include "sph_time.h"
#include "sph_common_science.h"
#include "sph_cube.h"
#include "sph_filemanager.h"
#include "sph_fitting.h"
#include "sph_fits.h"
#include "sph_unsharp_mask.h"
#include "sph_point_pattern.h"

#include <math.h>
#include <float.h>
#include <stdbool.h>

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

const sph_error_code SPH_IRD_STAR_CENTER_BAD_NUMBER_WAFFIMS =
        SPH_IRD_STAR_CENTER_ERR_START + 5;

const int SPH_IRD_STAR_CENTER_MINPIX = 10;
const int SPH_IRD_STAR_CENTER_MAX_WAFFIMS = 4;
const int SPH_IRD_STAR_CENTER_MAX_SEARCH_APERTURES = 20;
const double SPH_IRD_STAR_CENTER_FIT_FWHM_GUESS = 3.0;

const char* const SPH_IRD_STAR_CENTER_FILE_NAME = "WAFFFRAME";
const char* const SPH_IRD_STAR_CENTER_ANGLE_NAME = "ANGLE";
const double SPH_IRD_PIX_SIZE = 18.0;

typedef struct _sph_ird_star_center_star_image_ {
    double cx;
    double cy;
    int npix;
} sph_ird_star_center_star_image;

static cpl_frameset*
sph_ird_star_center_collapse_waffle_frames(sph_ird_star_center* self,
        cpl_frameset* waffle_frames);

static sph_error_code
sph_ird_star_center_init_inter_time_cent_tab__(sph_ird_star_center* self,
        int nwaffle_images);

static sph_error_code
sph_ird_star_center_fill_waffle_table(sph_ird_star_center* self,
        cpl_frameset* collapsed_waffle_frames);

static sph_error_code
sph_ird_star_center_process_next__(sph_ird_star_center* self,
        cpl_frameset* frameset, sph_master_frame** left,
        sph_master_frame** right);

static sph_error_code
sph_ird_star_center_get_center(sph_master_frame* mf, double* dx, double* dy,
        double sigma, int nsources, sph_ird_instrument_model* model,
        int unsharp_window);
static cpl_image*
sph_ird_star_center_get_extraction_image__(sph_master_frame* mf,
        sph_ird_instrument_model* model, cpl_image** errim);
static sph_error_code
sph_ird_star_center_fit_center__(cpl_apertures* aps, int aa, cpl_image* im,
        cpl_image* errim, sph_ird_star_center_star_image* sim);
static
sph_error_code sph_ird_star_center_get_square_centre__(
        sph_point_pattern* square, double* dx, double *dy);
static
bool sph_ird_star_center_lines_do_intersect__(double A0, double B0,
        double A1, double B1,
        double A2, double B2,
        double A3, double B3);
/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_ird_star_center_run Create Master Dark Recipe
 *
 * This module provides the algorithm inplementation for the creation of the
 * master dakr
 *
 * @par Synopsis:
 * @code
 *   #include "sph_ird_star_center.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/
/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_ird_star_center recipe. The error
 code returned is always a cpl error code (to allow maximal compatibility with
 esorex, gasgano, etc.) even if during recipe execution an error in the SPHERE
 API is the cause. In this case (and if the underlying error is not a cpl error)
 the cpl error code is set to the cpl_error_code that matches the failure
 reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/
cpl_error_code sph_ird_star_center_run(sph_ird_star_center* self) {
    cpl_table* result = NULL;
    cpl_propertylist* plist = NULL;
    int nwaffles = 0;
    char* tmpfilename = NULL;
    cpl_frameset* collapsed_waffles = NULL;
    cpl_mask* stat_badpix_mask = NULL;
    cpl_image* stat_badpix_image = NULL;

    /*------------------------------------------------------------------
     -  selecting the right dark
     --------------------------------------------------------------------*/

    if(self->skybg_fit_frame){
           self->dark = sph_master_frame_load_(self->skybg_fit_frame,0);
           cpl_msg_info(cpl_func,"Using SKY_BG_FIT frame as background!");
    }
    else {
        if(self->skybg_frame){
            self->dark = sph_master_frame_load_(self->skybg_frame,0);
            cpl_msg_info(cpl_func,"Using SKY_BG frame as background!");
               }
        else {
            if(self->insbg_fit_frame){
                self->dark = sph_master_frame_load_(self->insbg_fit_frame,0);
                cpl_msg_info(cpl_func,"Using INS_BG_FIT frame as background!");
            }
               else{
                   if(self->insbg_frame){
                       self->dark = sph_master_frame_load_(self->insbg_frame,0);
                       cpl_msg_info(cpl_func,"Using INS_BG frame as background!");
                   }
                   else {
                       if(self->dark_frame){
                           self->dark = sph_master_frame_load_(self->dark_frame,0);
                           cpl_msg_info(cpl_func,"Using master dark as background!");
                       }
                       else
                           cpl_msg_info(cpl_func,"Running without dark/background subtraction!");
                   }
               }
        }
    }


    /*------------------------------------------------------------------
     -  Now process all sets of wavelength frames separately
     --------------------------------------------------------------------*/
//    if ( self->make_template )
//        return sph_ird_star_center_make_template(self);

    if (self->static_badpixel_frame) {
        stat_badpix_image = cpl_image_load(
                cpl_frame_get_filename(self->static_badpixel_frame),
                CPL_TYPE_INT, 0, 0);
        cpl_ensure_code(stat_badpix_image, cpl_error_get_code());
        self->stat_badpix_mask = cpl_mask_threshold_image_create(
                stat_badpix_image, -0.5, 0.5);
        cpl_ensure_code(self->stat_badpix_mask, cpl_error_get_code());
        cpl_mask_not(self->stat_badpix_mask);
        cpl_image_delete(stat_badpix_image);
        stat_badpix_image = NULL;
    }


    if (self->flat_frame) {
        self->flat = sph_master_frame_load_(self->flat_frame, 0);
        plist = sph_keyword_manager_load_properties(
                cpl_frame_get_filename(self->flat_frame), 0);
        self->model = sph_ird_instrument_model_new_from_propertylist(plist);
    }
    if (!self->model) {
        self->model = sph_ird_instrument_model_new();
        plist = sph_ird_instrument_model_get_as_propertylist(self->model);
    }

    if (self->qc) {
        self->qc_left = sph_master_frame_new(self->model->window_size_y,
                self->model->window_size_y);
        self->qc_right = sph_master_frame_new(self->model->window_size_y,
                self->model->window_size_y);
    }
    self->tot_fctab = sph_fctable_new();

    nwaffles = cpl_frameset_get_size(self->rawframes);
    SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL,
            "Using %d waffle frames to create a field center"
            " information. ", nwaffles);

    collapsed_waffles = sph_ird_star_center_collapse_waffle_frames(self,
            self->rawframes);

    sph_ird_star_center_init_inter_time_cent_tab__(self,
            cpl_frameset_get_size(collapsed_waffles));

    if (cpl_error_get_code() == CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "All waffle images collapsed.");
        if (sph_ird_star_center_fill_waffle_table(self, collapsed_waffles)
                == CPL_ERROR_NONE) {

#ifdef SPHERE_DEBUG
            fp = fopen("waffle_table.txt","w");
            cpl_table_dump(self->interntab,0,cpl_table_get_nrow(self->interntab),fp);
            fclose(fp); fp = NULL;
#endif
            cpl_propertylist_update_string(plist, SPH_COMMON_KEYWORD_PRO_CATG,
                    SPH_IRD_TAG_STAR_CENTER_CALIB);
            result = cpl_table_duplicate(self->interntab);
            SPH_RAISE_CPL;
            cpl_dfs_save_table(self->inframes, NULL, self->inparams,
                    self->inframes, NULL, result, NULL,
                    SPH_RECIPE_NAME_IRD_STAR_CENTER, plist, NULL,
                    SPH_PIPELINE_NAME_IRDIS, self->star_center_outfilename);
            SPH_RAISE_CPL;
            if (self->qc_left) {
                tmpfilename = sph_filemanager_new_filename_from_base(
                        self->star_center_outfilename, "qc_left");
                sph_master_frame_save_dfs(self->qc_left, tmpfilename,
                        self->inframes, cpl_frameset_get_first(self->rawframes), self->inparams,
                        SPH_IRD_TAG_STAR_CENTER_QC,
                        SPH_RECIPE_NAME_IRD_STAR_CENTER,
                        SPH_PIPELINE_NAME_IRDIS, NULL);
                cpl_free(tmpfilename);
                tmpfilename = NULL;
            }
            if (self->qc_right) {
                tmpfilename = sph_filemanager_new_filename_from_base(
                        self->star_center_outfilename, "qc_right");
                sph_master_frame_save_dfs(self->qc_right, tmpfilename,
                        self->inframes, cpl_frameset_get_first(self->rawframes), self->inparams,
                        SPH_IRD_TAG_STAR_CENTER_QC,
                        SPH_RECIPE_NAME_IRD_STAR_CENTER,
                        SPH_PIPELINE_NAME_IRDIS, NULL);
                cpl_free(tmpfilename);
                tmpfilename = NULL;
            }
            cpl_table_delete(result);
            result = NULL;
        } else {
            SPH_ERROR_RAISE_ERR(SPH_ERROR_GENERAL,
                    "Could not process science frame.");
        }
        if (cpl_error_get_code() == CPL_ERROR_NONE) {
            SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL, "All frames processed.");
        }
    }
    sph_master_frame_delete(self->dark);
    self->dark = NULL;
    sph_master_frame_delete(self->flat);
    self->flat = NULL;
    sph_master_frame_delete(self->qc_left);
    self->qc_left = NULL;
    sph_master_frame_delete(self->qc_right);
    self->qc_right = NULL;
    sph_fctable_delete(self->tot_fctab);
    self->tot_fctab = NULL;
    cpl_propertylist_delete(plist);
    plist = NULL;
    sph_ird_instrument_model_delete(self->model);
    self->model = NULL;
    cpl_table_delete(self->interntab);
    self->interntab = NULL;
    cpl_frameset_delete(collapsed_waffles);
    collapsed_waffles = NULL;
    cpl_mask_delete(stat_badpix_mask);
    stat_badpix_mask = NULL;
    cpl_mask_delete(self->stat_badpix_mask);
    self->stat_badpix_mask = NULL;

    return (int) cpl_error_get_code();

}
static cpl_frameset*
sph_ird_star_center_collapse_waffle_frames(sph_ird_star_center* self,
        cpl_frameset* waffle_frames) {
    cpl_frameset* result = NULL;
    cpl_frameset* tmpset = NULL;
    cpl_frame* aframe = NULL;
    cpl_frame* newframe = NULL;
    sph_master_frame* mframe = NULL;
    cpl_propertylist* pl = NULL;

    cpl_ensure( self, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( waffle_frames, CPL_ERROR_NULL_INPUT, NULL);

    result = cpl_frameset_new();

    SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Collapsing all raw waffle cubes "
    "(if any) using median...");

    aframe = cpl_frameset_get_first(waffle_frames);
    while ( aframe ) {
        tmpset = cpl_frameset_new();
        cpl_frameset_insert(tmpset, cpl_frame_duplicate(aframe));
        pl = cpl_propertylist_load(cpl_frame_get_filename(aframe), 0);
        newframe = sph_filemanager_get_unique_duplicate_frame(aframe);
        if (!self->save_interprod)
            sph_filemanager_add_tmp_file(cpl_frame_get_filename(newframe));
        mframe = sph_common_science_combine(tmpset, SPH_COLL_ALG_MEDIAN, 0, -1, -1);
        cpl_frameset_delete(tmpset);
        tmpset = NULL;
        sph_master_frame_save(mframe, cpl_frame_get_filename(newframe), pl);
        cpl_frameset_insert(result, newframe);
        cpl_propertylist_delete(pl);
        pl = NULL;
        sph_master_frame_delete(mframe);
        mframe = NULL;
        aframe = cpl_frameset_get_next(waffle_frames);
    }
    return result;
}

static sph_error_code sph_ird_star_center_init_inter_time_cent_tab__(
        sph_ird_star_center* self, int nwaffle_images) {
    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( nwaffle_images > 0, CPL_ERROR_ILLEGAL_INPUT);

    if (self->interntab) {
        cpl_table_delete(self->interntab);
        self->interntab = NULL;
    }
    self->interntab = cpl_table_new(nwaffle_images);
    cpl_table_new_column(self->interntab, SPH_IRD_STAR_CENTER_TIME_UT_NAME,
            CPL_TYPE_STRING);
    cpl_table_new_column(self->interntab, SPH_IRD_STAR_CENTER_CENTLEFTX_NAME,
            CPL_TYPE_DOUBLE);
    cpl_table_new_column(self->interntab, SPH_IRD_STAR_CENTER_CENTLEFTY_NAME,
            CPL_TYPE_DOUBLE);
    cpl_table_new_column(self->interntab, SPH_IRD_STAR_CENTER_CENTRIGHTX_NAME,
            CPL_TYPE_DOUBLE);
    cpl_table_new_column(self->interntab, SPH_IRD_STAR_CENTER_CENTRIGHTY_NAME,
            CPL_TYPE_DOUBLE);
    cpl_table_new_column(self->interntab, SPH_IRD_STAR_CENTER_DMS_POS_X_NAME,
            CPL_TYPE_DOUBLE);
    cpl_table_new_column(self->interntab, SPH_IRD_STAR_CENTER_DMS_POS_Y_NAME,
            CPL_TYPE_DOUBLE);
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}
static sph_error_code sph_ird_star_center_fill_waffle_table(
        sph_ird_star_center* self, cpl_frameset* collapsed_waffle_frames) {
    sph_error_code rerr = CPL_ERROR_NONE;
    sph_master_frame* left = NULL;
    sph_master_frame* right = NULL;
    double crx = 0.0;
    double cry = 0.0;
    double clx = 0.0;
    double cly = 0.0;
    int ff = 0;
    double dms_x = 0.0;
    double dms_y = 0.0;
    cpl_frameset* tmpframeset = NULL;
    sph_transform* transform = NULL;
    char ut_time[256];
    cpl_propertylist* plist = NULL;
    int nsources = SPH_IRD_STAR_CENTER_MAX_WAFFIMS;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( collapsed_waffle_frames, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( self->interntab, CPL_ERROR_ILLEGAL_INPUT);

    transform = sph_transform_new_default();

    if (self->use_waffle == 0) {
        nsources = 1;
    }
    ff = 0;
    while (sph_ird_star_center_process_next__(self, collapsed_waffle_frames,
            &left, &right) == CPL_ERROR_NONE) {
        rerr = sph_ird_star_center_get_center(left, &clx, &cly, self->sigma,
                nsources, self->model, self->unsharp_window);
        SPH_ERROR_ENSURE_GOTO_EXIT( rerr == CPL_ERROR_NONE, rerr);

        if (self->qc_left) {
            sph_transform_apply(transform, left, NULL, clx, cly, 0.0, 1.0,
                    self->model);
            sph_master_frame_add_master_frame(self->qc_left, left);
        }

        plist = sph_keyword_manager_load_properties(
                cpl_frame_get_filename(self->current_frame), 0);

        rerr = sph_ird_star_center_get_center(right, &crx, &cry, self->sigma,
                nsources, self->model, self->unsharp_window);
        SPH_ERROR_ENSURE_GOTO_EXIT( rerr == CPL_ERROR_NONE, rerr);

        dms_x = dms_y = 0.0;
        if (cpl_propertylist_has(plist, SPH_IRD_KEYWORD_DMS_POS_X)) {
            dms_x = cpl_propertylist_get_double(plist,
                    SPH_IRD_KEYWORD_DMS_POS_X);
            dms_x /= SPH_IRD_PIX_SIZE;
        } else {
            SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,
                    "Could not find the %s keyword. "
                    "Assuming value of 0.", SPH_IRD_KEYWORD_DMS_POS_X);
        }
        if (cpl_propertylist_has(plist, SPH_IRD_KEYWORD_DMS_POS_Y)) {
            dms_y = cpl_propertylist_get_double(plist,
                    SPH_IRD_KEYWORD_DMS_POS_Y);
            dms_y /= SPH_IRD_PIX_SIZE;
        } else {
            SPH_ERROR_RAISE_WARNING(SPH_ERROR_GENERAL,
                    "Could not find the %s keyword. "
                    "Assuming value of 0.", SPH_IRD_KEYWORD_DMS_POS_Y);
        }
        if (sph_time_get_time_ut(self->current_frame, ut_time)
                == CPL_ERROR_NONE) {
            cpl_table_set_string(self->interntab,
                    SPH_IRD_STAR_CENTER_TIME_UT_NAME, ff, ut_time);
            cpl_table_set_double(self->interntab,
                    SPH_IRD_STAR_CENTER_CENTLEFTX_NAME, ff, clx);
            cpl_table_set_double(self->interntab,
                    SPH_IRD_STAR_CENTER_CENTLEFTY_NAME, ff, cly);
            cpl_table_set_double(self->interntab,
                    SPH_IRD_STAR_CENTER_CENTRIGHTX_NAME, ff, crx);
            cpl_table_set_double(self->interntab,
                    SPH_IRD_STAR_CENTER_CENTRIGHTY_NAME, ff, cry);
            cpl_table_set_double(self->interntab,
                    SPH_IRD_STAR_CENTER_DMS_POS_X_NAME, ff, dms_x);
            cpl_table_set_double(self->interntab,
                    SPH_IRD_STAR_CENTER_DMS_POS_Y_NAME, ff, dms_y);
        } else {
            SPH_ERROR_RAISE_ERR(
                    CPL_ERROR_ILLEGAL_INPUT,
                    "Could not extract timing info from frame %s", cpl_frame_get_filename(self->current_frame));
            goto EXIT;
        }SPH_ERROR_ENSURE_GOTO_EXIT( rerr == CPL_ERROR_NONE, rerr);

        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_GENERAL,
                "Adding centre for %d as %f,%f and %f,%f.", ff, clx, cly, crx, cry);

        if (self->qc_right) {
            sph_transform_apply(transform, right, NULL, crx, cry, 0.0, 1.0,
                    self->model);
            sph_master_frame_add_master_frame(self->qc_right, right);
        }

        sph_master_frame_delete(right);
        right = NULL;
        sph_master_frame_delete(left);
        left = NULL;
        cpl_propertylist_delete(plist);
        plist = NULL;
        ff++;
    }
    self->current_frame = NULL;
    sph_master_frame_delete(right);
    right = NULL;
    sph_master_frame_delete(left);
    left = NULL;
    cpl_frameset_delete(tmpframeset);
    tmpframeset = NULL;
    sph_transform_delete(transform);
    transform = NULL;

    EXIT: sph_master_frame_delete(right);
    right = NULL;
    sph_master_frame_delete(left);
    left = NULL;
    cpl_frameset_delete(tmpframeset);
    tmpframeset = NULL;
    sph_transform_delete(transform);
    transform = NULL;
    cpl_propertylist_delete(plist);
    plist = NULL;
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_error_code sph_ird_star_center_process_next__(
        sph_ird_star_center* self, cpl_frameset* frameset,
        sph_master_frame** left, sph_master_frame** right) {
    sph_master_frame* master_raw = NULL;
    cpl_mask* bmask = NULL;

    cpl_ensure_code( self, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( left, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( right, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( frameset, CPL_ERROR_NULL_INPUT);

    master_raw = sph_common_science_get_next_master_frame(frameset);

    if (master_raw) {
        if (master_raw) {
            if (self->dark)
                sph_master_frame_subtract_master_frame(master_raw, self->dark);
            if (self->flat)
                sph_master_frame_divide_master_frame(master_raw, self->flat);

            bmask = sph_master_frame_get_badpixelmask(master_raw);

            sph_master_frame_interpolate_bpix(master_raw);

            //FIXME: in principle, this interpolation is useless as long as the badpixel mask
            //is retained and always taken into account, or?
            //Proposal is to either drop the interpolation, or indeed not restore the badpixel mask
            //further down (MFe)


            if (self->stat_badpix_mask) {
                SPH_ERROR_RAISE_INFO(
                        SPH_ERROR_GENERAL,
                        "Applying mask frame %s to image.", cpl_frame_get_filename(self->static_badpixel_frame));
                sph_master_frame_set_bads_from_mask(master_raw,
                        self->stat_badpix_mask);
                SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                    "Setting all pixels masked from the STATIC_BADPIXEL_MAP to 0.");
                sph_master_frame_set_bads(master_raw, 0.0);
            }


            sph_master_frame_set_bads_from_mask(master_raw, bmask);
            cpl_mask_delete(bmask);
            bmask = NULL;


            *left = sph_ird_instrument_model_extract_left_master_frame(
                    self->model, master_raw);
            *right = sph_ird_instrument_model_extract_right_master_frame(
                    self->model, master_raw);
            self->current_frame = sph_common_science_get_current_raw_frame();
        }
        sph_master_frame_delete(master_raw);
        master_raw = NULL;
    } else {
        return CPL_ERROR_EOL;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static sph_error_code sph_ird_star_center_get_center(sph_master_frame* mf,
        double* dx, double* dy, double sigma, int nsources,
        sph_ird_instrument_model* model, int unsharp_window) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_apertures* aps = NULL;
    cpl_image* im = NULL;
    int aa = 0;
    int nwaffles = 0;
    cpl_image* unsharped = NULL;
    cpl_image* errim = NULL;
    sph_point_pattern* ppattern = NULL;
    sph_point_pattern* square = NULL;
    double xw = 3.0;
    double yw = 3.0;

    int maxaps = SPH_IRD_STAR_CENTER_MAX_SEARCH_APERTURES;
    sph_ird_star_center_star_image sim;

    cpl_ensure_code( mf, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( dx, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( dy, CPL_ERROR_NULL_INPUT);

    sim.cx = sim.cy = 0.0;
    sim.npix = 0;

    im = sph_ird_star_center_get_extraction_image__(mf, model, &errim);

    if (im) {
        unsharped = sph_unsharp_mask(im, unsharp_window);
        aps = cpl_apertures_extract_sigma(unsharped, sigma);

        if (aps == NULL) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "Could not find any sources on the input above a sigma "
                    "of %f. Check that the quality of the input is sufficient, "
                    "and that the flat field and dark have reasonable quality."
                    "", sigma);
            cpl_image_delete(im);
            im = NULL;
            cpl_image_delete(errim);
            errim = NULL;
        cpl_image_delete(unsharped);
        unsharped = NULL;
            return cpl_error_get_code();
        }

        cpl_apertures_sort_by_npix(aps);
        cpl_image_accept_all(unsharped);

        if (cpl_apertures_get_size(aps) < maxaps)
            maxaps = cpl_apertures_get_size(aps);

        ppattern = sph_point_pattern_new_(maxaps);

        for (aa = 0; aa < maxaps; ++aa) {
            if (cpl_apertures_get_npix(aps, aa + 1)
                    > SPH_IRD_STAR_CENTER_MINPIX) {
                //if ( nwaffles < SPH_IRD_STAR_CENTER_MAX_WAFFIMS )
                //{

                sph_ird_star_center_fit_center__(aps, aa, unsharped, errim,
                        &sim);
                sph_point_pattern_add_point(ppattern, sim.cx, sim.cy);
                //}
                nwaffles++;
                if ( nsources == 1 ) {
                    *dx = sim.cx; *dy = sim.cy;
                }
            }
        }

        if (ppattern == NULL || sph_point_pattern_get_size(ppattern) < 1) {
            SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                    "Could not find any sources on the input above a sigma "
                    "of %f. Check that the quality of the input is sufficient, "
                    "and that the flat field and dark have reasonable quality."
                    "", sigma);
            goto ERROR_EXIT;
        }

        if (nsources == 1) {
            cpl_errorstate prestate = cpl_errorstate_get();
            const double xguess = *dx;
            const double yguess = *dy;
            SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                    "Not using waffle. Just doing simple central fit. "
                    "Guess positions: %5.2f,%5.2f", *dx, *dy);
            if (sph_fitting_fit_gauss2D(im, errim, dx, dy, &xw, &yw, 6, 3)
                    != CPL_ERROR_NONE) {
                    cpl_msg_warning(cpl_func, "Could not perform simple central "
                                    "gauss fit at (%g,%g), keeping 1st guess:",
                                    xguess, yguess);
                    *dx = xguess;
                    *dy = yguess;
                    cpl_errorstate_dump(prestate, CPL_FALSE,
                                        cpl_errorstate_dump_one_warning);
                    cpl_errorstate_set(prestate);
            } else {
                    SPH_ERROR_RAISE_INFO(SPH_ERROR_GENERAL,
                                         "Gauss fit positions: %5.2f,%5.2f", *dx, *dy);
                }
        } else {
            square = sph_point_pattern_find_square(ppattern);
            if ( !square ) {
                SPH_ERROR_RAISE_ERR(CPL_ERROR_ILLEGAL_INPUT,
                        "Using waffle, but could not find the expected cross shaped waffle pattern "
                        "in the raw input files. Can not proceed.");
            }
            else {
                SPH_ERROR_RAISE_INFO(
                        SPH_ERROR_GENERAL,
                        "Identified the following points as the square pattern of the waffle:");

                sph_ird_star_center_get_square_centre__(square, dx, dy);
            }
        }
        cpl_image_delete(im);
        im = NULL;
        cpl_image_delete(unsharped);
        unsharped = NULL;
        sph_point_pattern_delete(ppattern);
        ppattern = NULL;
        sph_point_pattern_delete(square);
        square = NULL;

        cpl_apertures_delete(aps); aps = NULL;
    }

    cpl_image_delete(errim);
    errim = NULL;

    return rerr;
    ERROR_EXIT: cpl_image_delete(im);
    im = NULL;
    cpl_image_delete(errim);
    errim = NULL;
    sph_point_pattern_delete(ppattern);
    ppattern = NULL;
    cpl_image_delete(unsharped);
    unsharped = NULL;
    return cpl_error_get_code();
}

static cpl_image*
sph_ird_star_center_get_extraction_image__(sph_master_frame* mf,
        sph_ird_instrument_model* model, cpl_image** errim) {
    cpl_image* tmprms = NULL;
    cpl_mask* bmask = NULL;
    cpl_image* im = NULL;

    cpl_ensure( mf, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( model, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure( errim, CPL_ERROR_NULL_INPUT, NULL);

    im = sph_master_frame_extract_image(mf, 1);
    tmprms = sph_master_frame_get_rms(mf);
    sph_master_frame_set_rms_poisson(mf, 1e10, 0);
    *errim = sph_master_frame_get_rms(mf);
    cpl_image_delete(mf->rmsmap);
    mf->rmsmap = tmprms;
    bmask = sph_ird_instrument_model_get_mask_dbi_fov(model, 10.0);
    cpl_image_reject_from_mask(im, bmask);
    cpl_mask_delete(bmask);
    bmask = NULL;
    cpl_image_fill_rejected(im, 0.0);
    cpl_image_accept_all(*errim);
    return im;
}
static sph_error_code sph_ird_star_center_fit_center__(cpl_apertures* aps,
        int aa, cpl_image* im, cpl_image* errim,
        sph_ird_star_center_star_image* sim) {
    double xfit = 0;
    double yfit = 0;
    double wx = 0;
    double wy = 0;

    cpl_ensure_code( aps, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( im, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( errim, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( sim, CPL_ERROR_NULL_INPUT);
    sim->cx = cpl_apertures_get_centroid_x(aps, aa + 1) - 0.5;
    sim->cy = cpl_apertures_get_centroid_y(aps, aa + 1) - 0.5;
    sim->npix = cpl_apertures_get_npix(aps, aa + 1);

    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Peak found: centroid: %f, %f and npix %d", sim->cx, sim->cy, sim->npix);

    xfit = sim->cx + 0.5;
    yfit = sim->cy + 0.5;
    wx = wy = SPH_IRD_STAR_CENTER_FIT_FWHM_GUESS;
    if (sph_fitting_fit_gauss2D(im, errim, &xfit, &yfit, &wx, &wy, 6, 3)
            == CPL_ERROR_NONE) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_GENERAL, "Centroid: %f, %f. "
        "Gauss fit :%f, %f.", sim->cx, sim->cy, xfit - 0.5, yfit - 0.5);

        sim->cx = xfit - 0.5;
        sim->cy = yfit - 0.5;
    } else {
        cpl_error_reset();
        SPH_ERROR_RAISE_WARNING( SPH_ERROR_GENERAL,
                "Could not do gauss fit. Am using centroid.");
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

/*
 * Helper function to find the centre of a square by finding the
 * intersection of the two lines connecting opposite points.
 *
 * TODO: the code is a bit verbose and could be tightened up...
 */
static
sph_error_code sph_ird_star_center_get_square_centre__(sph_point_pattern* square, double* dx, double *dy)
{
    sph_point p11;
    sph_point p12;
    sph_point p21;
    sph_point p22;
    sph_point centrepoint;
    sph_line lineA;
    sph_line lineB;


    cpl_ensure_code(square != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(sph_point_pattern_get_size(square) != 3,
            CPL_ERROR_ILLEGAL_INPUT);
    cpl_ensure_code(dx != NULL, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(dy != NULL, CPL_ERROR_NULL_INPUT);

    sph_point_pattern_get_centre(square, dx, dy);
    p11.x = sph_point_pattern_get_x(square, 0);
    p11.y = sph_point_pattern_get_y(square, 0);
    p12.x = sph_point_pattern_get_x(square, 1);
    p12.y = sph_point_pattern_get_y(square, 1);
    p21.x = sph_point_pattern_get_x(square, 2);
    p21.y = sph_point_pattern_get_y(square, 2);
    p22.x = sph_point_pattern_get_x(square, 3);
    p22.y = sph_point_pattern_get_y(square, 3);

    if ( sph_ird_star_center_lines_do_intersect__(p11.x,p11.y,
                                                  p12.x,p12.y,
                                                  p21.x,p21.y,
                                                  p22.x,p22.y) )
    {
        lineA.p1 = &p11;
        lineA.p2 = &p12;
        lineB.p1 = &p21;
        lineB.p2 = &p22;
    }
    else {
        if ( sph_ird_star_center_lines_do_intersect__(p11.x,p11.y,
                p22.x,p22.y,
                p21.x,p21.y,
                p12.x,p12.y) )
        {
            lineA.p1 = &p11;
            lineA.p2 = &p22;
            lineB.p1 = &p21;
            lineB.p2 = &p12;
        }
        else {
            lineA.p1 = &p11;
            lineA.p2 = &p21;
            lineB.p1 = &p12;
            lineB.p2 = &p22;
        }
    }
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Line A %.2f, %.2f to %.2f, %.2f", lineA.p1->x, lineA.p1->y, lineA.p2->x, lineA.p2->y  );
    SPH_ERROR_RAISE_INFO(
            SPH_ERROR_GENERAL,
            "Line B %.2f, %.2f to %.2f, %.2f", lineB.p1->x, lineB.p1->y, lineB.p2->x, lineB.p2->y  );
    if ( sph_ird_star_center_lines_do_intersect__(lineA.p1->x,lineA.p1->y,
                                                  lineA.p2->x,lineA.p2->y,
                                                  lineB.p1->x,lineB.p1->y,
                                                  lineB.p2->x,lineB.p2->y) )
    {
        sph_line_intersect_point(&centrepoint, &lineA, &lineB);
        *dx = centrepoint.x;
        *dy = centrepoint.y;
    }
    else {
        SPH_ERROR_RAISE_INFO(CPL_ERROR_ILLEGAL_OUTPUT,
                "Could not find intersection point of lines. "
                "Am using centre of square pattern instead.");
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE;
}

static
bool sph_ird_star_center_lines_do_intersect__(double A0, double B0,
        double A1, double B1,
        double A2, double B2,
        double A3, double B3)
{
    return (
            ((A2-A0)*(B1-B0) - (B2-B0)*(A1-A0)) *
            ((A3-A0)*(B1-B0) - (B3-B0)*(A1-A0)) < 0)
    &&
            (((A0-A2)*(B3-B2) - (B0-B2)*(A3-A2)) *
            ((A1-A2)*(B3-B2) - (B1-B2)*(A3-A2)) < 0);
}

/**@}*/
