/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*
 * sph_zpl_common_preproc.c
 *
 *  Created on: Jul 26, 2011
 *      Author: pavlov
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#include <cpl.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <libgen.h>

#include "sph_zpl_common_preproc.h"
#include "sph_error.h"
#include "sph_filemanager.h"
#include "sph_zpl_utils.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_common_keywords.h"
#include "sph_fits.h"
#include "sph_zpl_exposure.h"
#include "sph_zpl_exposure_imaging.h"
#include "sph_master_frame.h"
#include "sph_cube.h"
#include "sph_utils.h"
#include "sph_framecombination.h"

/*-----------------------------------------------------------------------------
 Static variables
 -----------------------------------------------------------------------------*/
// GENERAL KEYWORDS FOR PREPROCESSING
// (presented here only temporarily)
#define SPH_ZPL_KEYWORD_PHASE                   "ESO DRS ZPL DPR PHASE"
#define SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO        "ZERO"
#define SPH_ZPL_KEYWORD_VALUE_PHASE_PI          "PI"
#define SPH_ZPL_KEYWORD_FRAME_ID                "ESO DRS ZPL DPR FRAME ID"
#define SPH_ZPL_KEYWORD_VALUE_FRAME_ID          -1
#define SPH_ZPL_KEYWORD_SPLIT                   "ESO DRS ZPL DPR SPLIT"
#define SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN        "EVEN"
#define SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD         "ODD"
#define SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN          "ESO DRS ZPL ADU1 OVSC MEAN"
#define SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN          "ESO DRS ZPL ADU2 OVSC MEAN"
#define SPH_ZPL_KEYWORD_ADU1_OVSC_RMS           "ESO DRS ZPL ADU1 OVSC RMS"
#define SPH_ZPL_KEYWORD_ADU2_OVSC_RMS           "ESO DRS ZPL ADU2 OVSC RMS"

//Flip image orientation definitions (correspond to the cpl_image_flip function)
#define IMAGE_FLIP_HORIZONTAL                   0
#define IMAGE_FLIP_VERTICAL                     2

/*-----------------------------------------------------------------------------
 Function declaration
 -----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_extract_frames_newformat(const cpl_frame* inframe);
static cpl_frameset* sph_zpl_common_preproc_extract_frames(const cpl_frame* inframe);
static cpl_frameset* sph_zpl_common_preproc_combine_adus(cpl_frameset* exframes);
static cpl_frameset* sph_zpl_common_preproc_junk_rows(cpl_frameset* adusframes);
static cpl_frameset* sph_zpl_common_preproc_split_image(cpl_frameset* jrframes,
		const cpl_boolean swap_odd_even_rows);
static cpl_frameset* sph_zpl_common_preproc_split_image_imaging(
        cpl_frameset* jrframes); //imaging

static cpl_frameset* sph_zpl_common_preproc_create_zpl_exposure(
        cpl_frameset* splitframes);
static cpl_frameset* sph_zpl_common_preproc_get_frames_ccd(
        cpl_frameset* inframes, int ccdID);
static void sph_zpl_common_preproc_get_frame_pairs(
        cpl_frameset* inframes, cpl_frameset* paired_frames, cpl_frameset* single_frames);
static cpl_frameset* sph_zpl_common_preproc_sort_frames_phase(
        cpl_frameset* inframes);

static cpl_frameset* sph_zpl_common_preproc_cube_zpl_exposures(
        cpl_frameset* zexpframes);
static cpl_frameset* sph_zpl_common_preproc_cube_zpl_exposures_imaging(
        cpl_frameset* zexpframes); //imaging

static sph_error_code sph_zpl_common_preproc_recipe__(cpl_frameset* rawframes,
        const char* outfilename_cam1, const char* outfilename_cam2,
        cpl_frameset* preproc_frames_cam1, cpl_frameset* preproc_frames_cam2,
		const cpl_boolean swap_odd_even_rows);

static sph_error_code sph_zpl_common_preproc_imaging_recipe__(cpl_frameset* rawframes,
        const char* outfilename_cam1, const char* outfilename_cam2,
        cpl_frameset* preproc_frames_cam1, cpl_frameset* preproc_frames_cam2);


sph_error_code sph_zpl_common_preproc_imaging_recipe(cpl_frameset* rawframes,
        const char* outfilename_cam1, const char* outfilename_cam2,
        cpl_frameset* preproc_frames_cam1, cpl_frameset* preproc_frames_cam2){
    cpl_frame*      currawframe             = NULL;
    cpl_frame*      duplframe               = NULL;
    cpl_frameset*   currawframes            = NULL;
    cpl_frameset*   curpreproc_frames_cam1  = NULL;
    cpl_frameset*   curpreproc_frames_cam2  = NULL;
    sph_error_code  rerr                    = CPL_ERROR_NONE;

    cpl_ensure_code( rawframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( outfilename_cam1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( outfilename_cam2, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( preproc_frames_cam1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( preproc_frames_cam2, CPL_ERROR_NULL_INPUT);

    currawframe = cpl_frameset_get_first(rawframes);
    while (currawframe) {
        currawframes = cpl_frameset_new();
        duplframe = cpl_frame_duplicate(currawframe);

        rerr = cpl_frameset_insert(currawframes, duplframe);
        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not insert rawframe into the temporal frameset:\n"
                            "cpl error code is: %d", rerr);
            cpl_frameset_delete(currawframes);
            return rerr;
        }

        curpreproc_frames_cam1 = cpl_frameset_new();
        curpreproc_frames_cam2 = cpl_frameset_new();

        rerr = sph_zpl_common_preproc_imaging_recipe__( currawframes,
                outfilename_cam1, outfilename_cam2,
                curpreproc_frames_cam1, curpreproc_frames_cam2 );

        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not pre-processed data:\n"
                            "cpl error code is: %d", rerr);
            cpl_frameset_delete(curpreproc_frames_cam1);
            cpl_frameset_delete(curpreproc_frames_cam2);
            cpl_frameset_delete(currawframes);
            return rerr;
        }

        cpl_frameset_join(preproc_frames_cam1, curpreproc_frames_cam1);
        cpl_frameset_delete(curpreproc_frames_cam1); curpreproc_frames_cam1 = NULL;

        cpl_frameset_join( preproc_frames_cam2, curpreproc_frames_cam2);
        cpl_frameset_delete( curpreproc_frames_cam2 ); curpreproc_frames_cam2 = NULL;

        sph_filemanager_delete(0);
        cpl_frameset_delete(currawframes); currawframes = NULL;

        currawframe = cpl_frameset_get_next(rawframes);
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

}

sph_error_code sph_zpl_common_preproc_recipe( cpl_frameset* rawframes,
        const char* outfilename_cam1, const char* outfilename_cam2,
        cpl_frameset* preproc_frames_cam1, cpl_frameset* preproc_frames_cam2){
	return sph_zpl_common_preproc_recipe_swap(rawframes, outfilename_cam1, outfilename_cam2,
			preproc_frames_cam1, preproc_frames_cam2, CPL_FALSE);
}

sph_error_code sph_zpl_common_preproc_recipe_swap( cpl_frameset* rawframes,
        const char* outfilename_cam1, const char* outfilename_cam2,
        cpl_frameset* preproc_frames_cam1, cpl_frameset* preproc_frames_cam2,
		const cpl_boolean swap_even_odd_rows){

    cpl_frame*      currawframe             = NULL;
    cpl_frame*      duplframe               = NULL;
    cpl_frameset*   currawframes            = NULL;
    cpl_frameset*   curpreproc_frames_cam1  = NULL;
    cpl_frameset*   curpreproc_frames_cam2  = NULL;
    sph_error_code  rerr                    = CPL_ERROR_NONE;

    cpl_ensure_code( rawframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( outfilename_cam1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( outfilename_cam2, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( preproc_frames_cam1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( preproc_frames_cam2, CPL_ERROR_NULL_INPUT);

    currawframe = cpl_frameset_get_first(rawframes);
    while (currawframe) {
        currawframes = cpl_frameset_new();
        duplframe = cpl_frame_duplicate(currawframe);

        rerr = cpl_frameset_insert(currawframes, duplframe);
        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not insert rawframe into the temporal frameset:\n"
                            "cpl error code is: %d", rerr);
            cpl_frameset_delete(currawframes);
            return rerr;
        }

        curpreproc_frames_cam1 = cpl_frameset_new();
        curpreproc_frames_cam2 = cpl_frameset_new();

        rerr = sph_zpl_common_preproc_recipe__(currawframes,
                outfilename_cam1, outfilename_cam2,
                curpreproc_frames_cam1, curpreproc_frames_cam2,
				swap_even_odd_rows);

        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not pre-processed data:\n"
                            "cpl error code is: %d", rerr);
            cpl_frameset_delete(curpreproc_frames_cam1);
            cpl_frameset_delete(curpreproc_frames_cam2);
            cpl_frameset_delete(currawframes);
            return rerr;
        }

        cpl_frameset_join(preproc_frames_cam1, curpreproc_frames_cam1);
        cpl_frameset_delete( curpreproc_frames_cam1 ); curpreproc_frames_cam1 = NULL;

        cpl_frameset_join( preproc_frames_cam2, curpreproc_frames_cam2);
        cpl_frameset_delete( curpreproc_frames_cam2 ); curpreproc_frames_cam2 = NULL;

        sph_filemanager_delete(0);
        cpl_frameset_delete(currawframes); currawframes = NULL;

        currawframe = cpl_frameset_get_next(rawframes);
    }

    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

}


/*----------------------------------------------------------------------------*/
/**
 @brief Pre-processing of the cpl_frameset of the raw zimpol polarimetric data
 @param rawframes                The input rawframes to be pre-processed
 @param outfilename_cam1         The postfix of the filename to save the pre-procesed
 data for the camera-1
 @param outfilename_cam2         The postfix of the filename to save the pre-procesed
 data for the camera-2
 @param preproc_frames_cam1    The output frameset of the pre-processed data, cam1
 @param preproc_frames_cam2    The output frameset of the pre-processed data, cam2
 @param swap_odd_even_rows		Swap even and odd rows when pre-processing if TRUE
 @return Error code of the operation

 This is a wrapper for the main pre-processing function. It takes cpl_frameset
 of the rawframes and creates the pre-processed data for the camera-1
 and camera-2. The pre-processed data are saved as an intermediate data with the
 filenames created by combining the rawfilename and the provided posfixes.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_zpl_common_preproc_recipe__(cpl_frameset* rawframes,
        const char* outfilename_cam1, const char* outfilename_cam2,
        cpl_frameset* preproc_frames_cam1, cpl_frameset* preproc_frames_cam2,
		const cpl_boolean swap_odd_even_rows) {
    cpl_frame* currawframe = NULL;
    cpl_frame* duplframe = NULL;
    cpl_frame* curframe = NULL;
    cpl_frameset* currawframes = NULL;
    cpl_frameset* preproc_frames = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_propertylist* pl = NULL;
    int camera_id = 0;
    //char                outfilename[256];
    char outfilename_tmp[256];
    char* outfilename = NULL;

    cpl_ensure_code( rawframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( outfilename_cam1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( outfilename_cam2, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( preproc_frames_cam1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( preproc_frames_cam2, CPL_ERROR_NULL_INPUT);

    currawframe = cpl_frameset_get_first(rawframes);
    while (currawframe) {

        currawframes = cpl_frameset_new();
        duplframe = cpl_frame_duplicate(currawframe);

        rerr = cpl_frameset_insert(currawframes, duplframe);
        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not insert rawframe into the temporal frameset:\n"
                            "cpl error code is: %d", rerr);
        }

        if (rerr != CPL_ERROR_NONE) {
            sph_error_raise(
                    SPH_ERROR_WARNING,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_WARNING,
                    "Something wrong with the file %s. Trying to continue with others!",
                    cpl_frame_get_filename(duplframe));
            cpl_frameset_delete(currawframes);
            cpl_error_reset();
            currawframe = cpl_frameset_get_next(rawframes);
            continue;
        }

        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_INFO_MESSAGE,
                "Call sph_zpl_common_preproc for file: %s", cpl_frame_get_filename(cpl_frameset_get_first(currawframes)));

        preproc_frames = sph_zpl_common_preproc(currawframes, swap_odd_even_rows);

        /*
        if (!preproc_frames || (cpl_frameset_get_size(preproc_frames) != 2)) {
            SPH_ERR("zpl_common_preproc is failed.");
            cpl_frameset_delete(currawframes);
            return CPL_ERROR_ILLEGAL_OUTPUT;
        }
        */

        if (!preproc_frames ) {
             SPH_ERR("zpl_common_preproc polarization is failed.");
             cpl_frameset_delete(currawframes);
             return CPL_ERROR_ILLEGAL_OUTPUT;
         }
        if ( cpl_frameset_get_size(preproc_frames) == 1 ) {
            SPH_INFO_MSG("Only one camera are given as an output, so I suppose the input was also only for one camera (new format)");
        }

        SPH_INFO_MSG("Save intermediate pre-processed frames");
        curframe = cpl_frameset_get_first(preproc_frames);

        while (curframe) {

            pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe),
                    0, ".*ESO.*", 0);
            //camera_id = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID);
            camera_id = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_DET_CHIP_INDEX);
            if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
                (void)strcpy(outfilename_tmp, outfilename_cam1);
            } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
                (void)strcpy(outfilename_tmp, outfilename_cam2);
            } else {
                sph_error_raise(
                        SPH_ERROR_ERROR,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Camera ID = <%d>  is wrong in the pre-processed file: %s",
                        camera_id, cpl_frame_get_filename(curframe));
                //sprintf( outfilename_tmp,"_%s", cpl_frame_get_filename( curframe ) );
                cpl_frameset_delete(currawframes);
                cpl_frameset_delete(preproc_frames);
                return CPL_ERROR_ILLEGAL_OUTPUT;
            }

            //write the results into the output file
            // if (!rename(cpl_frame_get_filename( curframe ), (const char*) outfilename) )
            SPH_INFO_MSG("Creating intermediate pre-processed filename...");
            outfilename = sph_filemanager_new_filename_from_base(
                    sph_filemanager_get_basename( cpl_frame_get_filename(currawframe)),
                    outfilename_tmp);
            //(void)strcpy( outfilename, sph_filemanager_get_basename( cpl_frame_get_filename( currawframe ) ) );
            //strcpy( outfilename, filename_out );
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                    "pre-processed filename = %s", outfilename);

            //rename the filename from the current result frame
            //outframe = cpl_frame_new();
            //cpl_frame_set_filename()
            if (!rename(cpl_frame_get_filename(curframe), outfilename)) {
                cpl_frame_set_filename(curframe, outfilename);
                if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
                    rerr = cpl_frameset_insert(preproc_frames_cam1,
                            cpl_frame_duplicate(curframe));
                } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
                    rerr = cpl_frameset_insert(preproc_frames_cam2,
                            cpl_frame_duplicate(curframe));
                }

                if (rerr) {
                    sph_error_raise(
                            SPH_ERROR_GENERAL,
                            __FILE__,
                            __func__,
                            __LINE__,
                            SPH_ERROR_ERROR,
                            "Could not insert ouframe into the preproc_frameset:\n"
                                    "cpl error code is: %d", rerr);
                    cpl_frameset_delete(currawframes);
                    cpl_frameset_delete(preproc_frames);
                    return rerr;
                }
            } else {
                SPH_ERROR_RAISE_ERR(
                        SPH_ERROR_GENERAL,
                        "Couldn't create the resulting outframe file from the zpl utility to the "
                        "pre-pocessed intermediate product filename  = %s", outfilename);
                cpl_frameset_delete(currawframes);
                cpl_frameset_delete(preproc_frames);
                return sph_error_get_last_code();
            }
            cpl_propertylist_delete(pl);
            cpl_free(outfilename); outfilename = NULL;
            curframe = cpl_frameset_get_next(preproc_frames);
        }
        cpl_frameset_delete(preproc_frames);
        cpl_frameset_delete(currawframes);
        currawframe = cpl_frameset_get_next(rawframes);

    }

    if (cpl_frameset_is_empty(preproc_frames_cam1)
            && cpl_frameset_is_empty(preproc_frames_cam2)) {
        SPH_ERR("prepoc_frames_cam1 and prepoc_frames_cam2 lists are empty");
        return CPL_ERROR_ILLEGAL_OUTPUT;
    }


    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

}

/*----------------------------------------------------------------------------*/
/**
 @brief Pre-processing of the cpl_frameset of the raw zimpol imaging data
 @param rawframes                The input rawframes to be pre-processed
 @param outfilename_cam1         The postfix of the filename to save the pre-procesed
 data for the camera-1
 @param outfilename_cam2         The postfix of the filename to save the pre-procesed
 data for the camera-2
 @param preproc_frames_cam1    The output frameset of the pre-processed data, cam1
 @param preproc_frames_cam2    The output frameset of the pre-processed data, cam2
 @return Error code of the operation

 This is a wrapper for the main pre-processing function. It takes cpl_frameset
 of the rawframes and creates the pre-processed data for the camera-1
 and camera-2. The pre-processed data are saved as an intermediate data with the
 filenames created by combining the rawfilename and the provided posfixes.
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_zpl_common_preproc_imaging_recipe__(cpl_frameset* rawframes,
        const char* outfilename_cam1, const char* outfilename_cam2,
        cpl_frameset* preproc_frames_cam1, cpl_frameset* preproc_frames_cam2) {
    cpl_frame* currawframe = NULL;
    cpl_frame* duplframe = NULL;
    cpl_frame* curframe = NULL;
    cpl_frameset* currawframes = NULL;
    cpl_frameset* preproc_frames = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_propertylist* pl = NULL;
    int camera_id = 0;
    //char                outfilename[256];
    char outfilename_tmp[256];
    char* outfilename = NULL;

    cpl_ensure_code( rawframes, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( outfilename_cam1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( outfilename_cam2, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( preproc_frames_cam1, CPL_ERROR_NULL_INPUT);
    cpl_ensure_code( preproc_frames_cam2, CPL_ERROR_NULL_INPUT);

    currawframe = cpl_frameset_get_first(rawframes);
    while (currawframe) {

        currawframes = cpl_frameset_new();
        duplframe = cpl_frame_duplicate(currawframe);

        rerr = cpl_frameset_insert(currawframes, duplframe);
        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Could not insert rawframe into the temporal frameset:\n"
                            "cpl error code is: %d", rerr);
        }

        if (rerr != CPL_ERROR_NONE) {
            sph_error_raise(
                    SPH_ERROR_WARNING,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_WARNING,
                    "Something wrong with the file %s. Trying to continue with others!",
                    cpl_frame_get_filename(duplframe));
            cpl_frameset_delete(currawframes);
            cpl_error_reset();
            currawframe = cpl_frameset_get_next(rawframes);
            continue;
        }

        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_INFO_MESSAGE,
                "Call sph_zpl_common_preproc_imaging for file: %s", cpl_frame_get_filename(cpl_frameset_get_first(currawframes)));

        preproc_frames = sph_zpl_common_preproc_imaging(currawframes);

        /*
        if (!preproc_frames || (cpl_frameset_get_size(preproc_frames) != 2)) {
            SPH_ERR("zpl_common_preproc is failed.");
            cpl_frameset_delete(currawframes);
            return CPL_ERROR_ILLEGAL_OUTPUT;
        }
        */
        if (!preproc_frames ) {
             SPH_ERR("zpl_common_preproc imaging is failed.");
             cpl_frameset_delete(currawframes);
             return CPL_ERROR_ILLEGAL_OUTPUT;
         }
        if ( cpl_frameset_get_size(preproc_frames) == 1 ) {
            SPH_INFO_MSG("Only one camera are given as an output, so I suppose the input was also only for one camera (new format)");
        }

        SPH_INFO_MSG("Save intermediate pre-processed frames");
        curframe = cpl_frameset_get_first(preproc_frames);

        while (curframe) {

            pl = cpl_propertylist_load_regexp(cpl_frame_get_filename(curframe),
                    0, ".*ESO.*", 0);
            //camera_id = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID);
            camera_id = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_DET_CHIP_INDEX);
            if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
                (void)strcpy(outfilename_tmp, outfilename_cam1);
            } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
                (void)strcpy(outfilename_tmp, outfilename_cam2);
            } else {
                sph_error_raise(
                        SPH_ERROR_ERROR,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Camera ID = <%d>  is wrong in the pre-processed file: %s",
                        camera_id, cpl_frame_get_filename(curframe));
                //sprintf( outfilename_tmp,"_%s", cpl_frame_get_filename( curframe ) );
                cpl_frameset_delete(currawframes);
                cpl_frameset_delete(preproc_frames);
                return CPL_ERROR_ILLEGAL_OUTPUT;
            }

            //write the results into the output file
            // if (!rename(cpl_frame_get_filename( curframe ), (const char*) outfilename) )
            SPH_INFO_MSG("Creating intermediate pre-processed filename...");
            outfilename = sph_filemanager_new_filename_from_base(
                    sph_filemanager_get_basename( cpl_frame_get_filename(currawframe)),
                    outfilename_tmp);
            //(void)strcpy( outfilename, sph_filemanager_get_basename( cpl_frame_get_filename( currawframe ) ) );
            //strcat ( outfilename, outfilename_tmp );
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                    "pre-processed filename = %s", outfilename);

            //rename the filename from the current result frame
            //outframe = cpl_frame_new();
            //cpl_frame_set_filename()
            if (!rename(cpl_frame_get_filename(curframe), outfilename)) {
                cpl_frame_set_filename(curframe, outfilename);
                if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {
                    rerr = cpl_frameset_insert(preproc_frames_cam1,
                            cpl_frame_duplicate(curframe));
                } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {
                    rerr = cpl_frameset_insert(preproc_frames_cam2,
                            cpl_frame_duplicate(curframe));
                }

                if (rerr) {
                    sph_error_raise(
                            SPH_ERROR_GENERAL,
                            __FILE__,
                            __func__,
                            __LINE__,
                            SPH_ERROR_ERROR,
                            "Could not insert ouframe into the preproc_frameset:\n"
                                    "cpl error code is: %d", rerr);
                    cpl_frameset_delete(currawframes);
                    cpl_frameset_delete(preproc_frames);
                    return rerr;
                }
            } else {
                SPH_ERROR_RAISE_ERR(
                        SPH_ERROR_GENERAL,
                        "Couldn't create the resulting outframe file from the zpl utility to the "
                        "pre-pocessed intermediate product filename  = %s", outfilename);
                cpl_frameset_delete(currawframes);
                cpl_frameset_delete(preproc_frames);
                return sph_error_get_last_code();
            }
            cpl_propertylist_delete(pl);
            cpl_free(outfilename); outfilename = NULL;
//            sph_filemanager_delete(0);
            curframe = cpl_frameset_get_next(preproc_frames);
        }

        cpl_frameset_delete(preproc_frames);
        cpl_frameset_delete(currawframes);
        currawframe = cpl_frameset_get_next(rawframes);
    }


    if (cpl_frameset_is_empty(preproc_frames_cam1)
            && cpl_frameset_is_empty(preproc_frames_cam2)) {
        SPH_ERR("prepoc_frames_cam1 and prepoc_frames_cam2 lists are empty");
        return CPL_ERROR_ILLEGAL_OUTPUT;
    }
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Pre-processing of the cpl_frameset of the raw zimpol data
 @return   a pointer to the cpl_frameset of the pre-processed data or NULL if
 unsucsessfull.
 @note     A CPL error is raised in case there was a problem.

 This is a main pre-processing function which takes cpl_frameset
 of zimpol rawframes as an argument and invokes consequently
 functions which perform needed pre-processing steps. Input zimpol rawdata
 cpl_frameset are cubes and must follow certain convention (see docs).
 */
/*----------------------------------------------------------------------------*/

cpl_frameset* sph_zpl_common_preproc(cpl_frameset* inframes,
		const cpl_boolean swap_odd_even_rows) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frame* curframe = NULL;
    cpl_frame* czexpframe = NULL;
    cpl_frame* czexpframe_dupl = NULL;
    cpl_frameset* exframes = NULL;
    cpl_frameset* adusframes = NULL;
    cpl_frameset* jrframes = NULL;
    cpl_frameset* splitframes = NULL;
    cpl_frameset* zexpframes = NULL;
    cpl_frameset* zexpcubes = NULL;
    cpl_frameset* results = NULL;
    int           format = -1;

    cpl_error_reset();
    if (!inframes) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "No input frames to perform pre-rpocessing steps.");
        return NULL;
    }

    results = cpl_frameset_new();
    for (int index = 0; index < cpl_frameset_get_size(inframes); index++) {
        curframe = cpl_frameset_get_position(inframes, index);
        format = sph_zpl_utils_check_format_frame( curframe );
        if ( format == 0 ) {
            //oldstyle
            exframes = sph_zpl_common_preproc_extract_frames(curframe);
        } else if ( format == 1 ){
            //newstyle
            exframes = sph_zpl_common_preproc_extract_frames_newformat( curframe );
        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Unsupported raw frame format; goto the next raw frame.");
            continue;
        }
        if (!exframes) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "extract frames is failed; goto the next raw frame.");
            continue;
        }

        adusframes = sph_zpl_common_preproc_combine_adus(exframes);
        if (!adusframes) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "combine adus-frames is failed; goto the next raw frame.");
            //delete already created exframes
            cpl_frameset_delete(exframes);
            exframes = NULL;
            continue;
        }

        jrframes = sph_zpl_common_preproc_junk_rows(adusframes);
        if (!jrframes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "create trimmed images is failed; go to the next raw frame.");
            //delete already created exframes, adusframes
            cpl_frameset_delete(exframes);
            cpl_frameset_delete(adusframes);
            exframes = NULL;
            adusframes = NULL;
            continue;
        }

        splitframes = sph_zpl_common_preproc_split_image(jrframes, swap_odd_even_rows);
        if (!splitframes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "split images into even/odd is failed; goto the next raw frame.");

            //delete already created exframes, adusframes, jrframes
            cpl_frameset_delete(exframes);
            cpl_frameset_delete(adusframes);
            cpl_frameset_delete(jrframes);
            exframes = NULL;
            adusframes = NULL;
            jrframes = NULL;
            continue;
        }

        zexpframes = sph_zpl_common_preproc_create_zpl_exposure(splitframes);
        if (!zexpframes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "create zimpol exposure frames is failed; goto the next raw frame.");
            cpl_frameset_delete(exframes);
            cpl_frameset_delete(adusframes);
            cpl_frameset_delete(jrframes);
            cpl_frameset_delete(splitframes);
            exframes = NULL;
            adusframes = NULL;
            jrframes = NULL;
            splitframes = NULL;
            continue;
        }

        zexpcubes = sph_zpl_common_preproc_cube_zpl_exposures(zexpframes);

        if (!zexpcubes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "create cube of zimpol exposure frames is failed; goto the next raw frame");
            cpl_frameset_delete(exframes);
            cpl_frameset_delete(adusframes);
            cpl_frameset_delete(jrframes);
            cpl_frameset_delete(splitframes);
            cpl_frameset_delete(zexpframes);
            exframes = NULL;
            adusframes = NULL;
            jrframes = NULL;
            splitframes = NULL;
            zexpframes = NULL;
            continue;
        }

        //insert two frames -- zimpol exposure cubes (ccd-1 and ccd-2) into result cpl_frameset
        czexpframe = cpl_frameset_get_first(zexpcubes);
        while (czexpframe) {
            czexpframe_dupl = cpl_frame_duplicate(czexpframe);
            cpl_frameset_insert(results, czexpframe_dupl);
            czexpframe = cpl_frameset_get_next(zexpcubes);
        }

        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Current size of the results cpl_frameset: %d",
                (int) cpl_frameset_get_size(results));

        //delete tmp frames used to  treate the raw frame
        cpl_frameset_delete(exframes);
        cpl_frameset_delete(adusframes);
        cpl_frameset_delete(jrframes);
        cpl_frameset_delete(splitframes);
        cpl_frameset_delete(zexpframes);
        cpl_frameset_delete(zexpcubes);
        exframes = NULL;
        adusframes = NULL;
        jrframes = NULL;
        splitframes = NULL;
        zexpframes = NULL;
        zexpcubes = NULL;

    } // end-for (index)

    rerr = sph_error_get_last_code();
    if (rerr != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "SPH error has occured: %d. Free result frameset.", rerr);
        cpl_frameset_delete(results);
        results = NULL;
    }

    //add check sum which is needed for the cpl_dfs_setup_product_header of eso function
    curframe = cpl_frameset_get_first(results);
    while (curframe) {
        sph_fits_update_checksum(cpl_frame_get_filename(curframe));
        curframe = cpl_frameset_get_next(results);
    }
    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Pre-processing of the cpl_frameset of the imaging raw zimpol data
 @return   a pointer to the cpl_frameset of the pre-processed data or NULL if
 unsucsessfull.
 @note     A CPL error is raised in case there was a problem.

 This is a main pre-processing function which takes cpl_frameset
 of zimpol rawframes as an argument and invokes consequently
 functions which perform needed pre-processing steps. Input imaging zimpol rawdata
 cpl_frameset are cubes and must follow certain convention (see docs).
 */
/*----------------------------------------------------------------------------*/

cpl_frameset* sph_zpl_common_preproc_imaging(cpl_frameset* inframes) {
    sph_error_code rerr = CPL_ERROR_NONE;
    cpl_frame* curframe = NULL;
    cpl_frame* czexpframe = NULL;
    cpl_frame* czexpframe_dupl = NULL;
    cpl_frameset* exframes = NULL;
    cpl_frameset* adusframes = NULL;
    cpl_frameset* splitframes = NULL;
    cpl_frameset* zexpcubes = NULL;
    cpl_frameset* results = NULL;
    int           format = -1;

    cpl_error_reset();
    if (!inframes) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR,
                "No input frames to perform imaging pre-rpocessing steps.");
        return NULL;
    }

    results = cpl_frameset_new();
    for (int index = 0; index < cpl_frameset_get_size(inframes); index++) {
        curframe = cpl_frameset_get_position(inframes, index);

        format = sph_zpl_utils_check_format_frame( curframe );
        if ( format == 0 ) {
            //oldstyle
            exframes = sph_zpl_common_preproc_extract_frames(curframe);
        } else if ( format == 1 ){
            //newstyle
            exframes = sph_zpl_common_preproc_extract_frames_newformat( curframe );
        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Unsupported raw frame format; goto the next raw frame.");
            continue;
        }

        if (!exframes) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "extract frames is failed; goto the next raw frame.");
            continue;
        }

        adusframes = sph_zpl_common_preproc_combine_adus(exframes);
        if (!adusframes) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "combine adus-frames is failed; goto the next raw frame.");

            //delete already created exframes
            cpl_frameset_delete(exframes);
            exframes = NULL;
            continue;
        }

        splitframes = sph_zpl_common_preproc_split_image_imaging(adusframes); //be ware of the two Camera
        if (!splitframes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "split images into even/odd is failed; goto the next raw frame.");

            //delete already created exframes, adusframes
            cpl_frameset_delete(exframes);
            cpl_frameset_delete(adusframes);
            exframes = NULL;
            adusframes = NULL;
            continue;
        }

        zexpcubes = sph_zpl_common_preproc_cube_zpl_exposures_imaging(
                splitframes);
        if (!zexpcubes) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "create cube of zimpol exposure imaging frames is failed; goto the next raw frame");

            //delete already created exframes, adusframes, splitframes
            cpl_frameset_delete(exframes);
            cpl_frameset_delete(adusframes);
            cpl_frameset_delete(splitframes);
            exframes = NULL;
            adusframes = NULL;
            splitframes = NULL;
            continue;
        }

        //insert two frames -- zimpol exposure cubes (ccd-1 and ccd-2) into result cpl_frameset
        czexpframe = cpl_frameset_get_first(zexpcubes);
        while (czexpframe) {
            czexpframe_dupl = cpl_frame_duplicate(czexpframe);
            cpl_frameset_insert(results, czexpframe_dupl);
            czexpframe = cpl_frameset_get_next(zexpcubes);
        }

        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Current size of the results cpl_frameset: %d",
                (int) cpl_frameset_get_size(results));

        //delete tmp frames used to treate the raw frame
        cpl_frameset_delete(exframes);
        cpl_frameset_delete(adusframes);
        cpl_frameset_delete(splitframes);
        cpl_frameset_delete(zexpcubes);
        exframes = NULL;
        adusframes = NULL;
        splitframes = NULL;
        zexpcubes = NULL;

    } // end-for (index)

    rerr = sph_error_get_last_code();
    if (rerr != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "SPH error has occured: %d", rerr);
        cpl_frameset_delete(results);
        results = NULL;
    }

    //add check sum which is needed for the cpl_dfs_setup_product_header of eso function
    curframe = cpl_frameset_get_first(results);
    while (curframe) {
        sph_fits_update_checksum(cpl_frame_get_filename(curframe));
        curframe = cpl_frameset_get_next(results);
    }

    return results;
}


/*----------------------------------------------------------------------------*/
/**
 @brief    Extract frames from the single cubes of zimpol rawdata (new format)
 @return   a pointer to the cpl_frameset with the extracted frames or NULL if
 unsuccessfull.
 @note     A SPH and CPL error is raised in case there was a problem.

 This is a first step in pre-processing of the zimpol rawdata.
 This function extracts frames from the single cube and returns
 cpl_frameset, providing for each frame proper keywords
 for the pre-processing needs:
 - SPH_ZPL_KEYWORD_PHASE: "ZERO" & "PI" for the phase
 according to the convention
 - SPH_ZPL_KEYWORD_FRAME_ID: integer ID number to identify
 pair of frames ("ZERO", "PI").
 */
/*----------------------------------------------------------------------------*/

static
cpl_frameset* sph_zpl_common_preproc_extract_frames_newformat(const cpl_frame* inframe) {
    cpl_frameset*       results  = NULL;
    cpl_image*          rawimage = NULL;
    cpl_frame*          frame    = NULL;
    cpl_propertylist*   plm      = NULL;
    cpl_propertylist*   pl       = NULL;
    int frame_id = SPH_ZPL_KEYWORD_VALUE_FRAME_ID;
    int nplane = 0;
    //char*                det_read_curname;

    cpl_error_reset();
    cpl_ensure( inframe, CPL_ERROR_NULL_INPUT, NULL );

    //read main header into plm (note that for new format there is only main header!)
    //plm =  cpl_propertylist_load_regexp( cpl_frame_get_filename( inframe ), 0, ".*ESO.*", 0 );
    plm = cpl_propertylist_load(cpl_frame_get_filename(inframe), 0);
    //det_read_curname = (char*) cpl_propertylist_get_string( plm, SPH_ZPL_KEYWORD_DET_READ_CURNAME );
    if (!plm) {
        SPH_ERR(
                "Error occured by reading keywords from the main header: plm is NULL.");
        SPH_RAISE_CPL;
        return NULL;
    }


    //append keyword of the sphere internal camera id
    //cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID, SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);

    if ( cpl_propertylist_has(plm, SPH_ZPL_KEYWORD_DET_CHIP_INDEX) ){
        int chip_index = cpl_propertylist_get_int( plm, SPH_ZPL_KEYWORD_DET_CHIP_INDEX );
        if ( chip_index == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID ) {
            cpl_propertylist_append_int(plm, SPH_ZPL_KEYWORD_CAMERA_ID, SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
        } else if ( chip_index == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID  ) {
              cpl_propertylist_append_int(plm, SPH_ZPL_KEYWORD_CAMERA_ID, SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);
        } else {
            SPH_ERROR_RAISE_ERR( SPH_ERROR_GENERAL, "Not expected detector chip index: %d (1 or 2)", chip_index);
        }
    }
    if (cpl_error_get_code() != 0) {
        SPH_ERR( "Problem with updating property list: plm." );
        SPH_RAISE_CPL;
        return NULL;
    }

    //workaround for non-existing end of the observations in the raw data of zimpol
    //SPH_COMMON_KEYWORD_DET_FRAM_UTC
    if (!cpl_propertylist_has(plm, SPH_COMMON_KEYWORD_DET_FRAM_UTC)){
        cpl_propertylist_append_string( plm, SPH_COMMON_KEYWORD_DET_FRAM_UTC,
                cpl_propertylist_get_string( plm, SPH_COMMON_KEYWORD_DATE));
    }
    if (cpl_error_get_code() != 0) {
        SPH_ERR( "Problem with updating property list: plm." );
        SPH_RAISE_CPL;
        return NULL;
    }

    results = cpl_frameset_new();
    while (TRUE) {
        const char* phase;
        if (nplane % 2) {
            phase = SPH_ZPL_KEYWORD_VALUE_PHASE_PI; // odd

        } else {
            phase = SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO; //even (we start by even)
            frame_id = frame_id + 1;
        }

        rawimage = cpl_image_load(cpl_frame_get_filename(inframe),
                CPL_TYPE_INT, nplane, 0);

        if (cpl_error_get_code() == 0) {

            pl = cpl_propertylist_duplicate( plm );
            /* Update the header if required */
            sph_utils_update_header(pl);
            frame = sph_filemanager_create_temp_frame(
                    sph_filemanager_get_basename( cpl_frame_get_filename(inframe)), "NONE",
                    CPL_FRAME_GROUP_NONE);
           cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_PHASE, phase);
           cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FRAME_ID, frame_id);

           //cpl_image_save ( rawimage_cam1, "test1.tmp.fits",CPL_BPP_32_SIGNED, (const  cpl_propertylist* ) pl, CPL_IO_CREATE );
            cpl_image_save((const cpl_image*) rawimage,
                    (const char*) cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, (const cpl_propertylist*) pl,
                    CPL_IO_CREATE);
            cpl_propertylist_delete(pl); pl = NULL;
            cpl_image_delete(rawimage); rawimage = NULL;
            cpl_frameset_insert(results, frame);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "extracted image saved in filename: %s.\n",
                    cpl_frame_get_filename(frame));

        } else {
            if (nplane == 0) {
                sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "No image can be loaded from raw file: %s.\n"
                                "CPL ERROR CODE: %d",
                        cpl_frame_get_filename(inframe), cpl_error_get_code());
                if ( plm ) cpl_propertylist_delete(plm);
                if ( rawimage ) cpl_image_delete( rawimage );
                if ( results ) cpl_frameset_delete(results);
                return NULL;
            }
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "%d frames have been extracted from raw file: %s.\n",
                    nplane, cpl_frame_get_filename(inframe));
            //cpl_propertylist_delete(plm);
            if ( rawimage ) cpl_image_delete( rawimage );
            cpl_error_reset();
            break;
        } //end-if-else-block
        nplane = nplane + 1;
    } //end while-loop

    if ( plm ) cpl_propertylist_delete(plm);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        if ( results) cpl_frameset_delete(results);
        results = NULL;
    }
    return results;

}




/*----------------------------------------------------------------------------*/
/**
 @brief    Extract frames from the two cubes of zimpol rawdata
 @return   a pointer to the cpl_frameset with the extracted frames or NULL if
 unsuccessfull.
 @note     A SPH and CPL error is raised in case there was a problem.

 This is a first step in pre-processing of the zimpol rawdata.
 This function extracts frames from the 2 cubes and returns
 cpl_frameset, providing for each frame proper keywords
 for the pre-processing needs:
 - SPH_ZPL_KEYWORD_CAMERA_ID: 1 for camera-1, 2 -- camera-2
 - SPH_ZPL_KEYWORD_PHASE: "ZERO" & "PI" for the phase
 according to the convention
 - SPH_ZPL_KEYWORD_FRAME_ID: integer ID number to identify
 pair of frames ("ZERO", "PI").
 */
/*----------------------------------------------------------------------------*/
static
cpl_frameset* sph_zpl_common_preproc_extract_frames(const cpl_frame* inframe) {
    int frame_id = SPH_ZPL_KEYWORD_VALUE_FRAME_ID;
    int nplane = 0;
    const int ext_cam1 = 1;
    const int ext_cam2 = 2;
    cpl_frameset* results;
    cpl_image* rawimage_cam1;
    cpl_image* rawimage_cam2;
    cpl_frame* frame;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* plm = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    //char*                det_read_curname;

    cpl_error_reset();
    if (!inframe)
        return NULL;

    results = cpl_frameset_new();
    while (TRUE) {
        const char* phase;

        if (nplane % 2) {
            phase = SPH_ZPL_KEYWORD_VALUE_PHASE_PI; // odd

        } else {
            phase = SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO; //even (we start by even)
            frame_id = frame_id + 1;
        }
        //read main header into plm
        //plm =  cpl_propertylist_load_regexp( cpl_frame_get_filename( inframe ), 0, ".*ESO.*", 0 );
        plm = cpl_propertylist_load(cpl_frame_get_filename(inframe), 0);
        //det_read_curname = (char*) cpl_propertylist_get_string( plm, SPH_ZPL_KEYWORD_DET_READ_CURNAME );
        if (!plm) {
            SPH_ERR(
                    "Error occured by reading keywords from the main header: plm is NULL.");
            SPH_RAISE_CPL;
            cpl_frameset_delete(results);
            return NULL;
        }

        //workaround for non-existing end of the observations in the raw data of zimpol
        //SPH_COMMON_KEYWORD_DET_FRAM_UTC
        if (!cpl_propertylist_has(plm, SPH_COMMON_KEYWORD_DET_FRAM_UTC)){
            cpl_propertylist_append_string( plm, SPH_COMMON_KEYWORD_DET_FRAM_UTC,
                    cpl_propertylist_get_string( plm, SPH_COMMON_KEYWORD_DATE));
        }

        /* camera-1 */
        rawimage_cam1 = cpl_image_load(cpl_frame_get_filename(inframe),
                CPL_TYPE_INT, nplane, ext_cam1);
        if (cpl_error_get_code() == 0) {

            frame = sph_filemanager_create_temp_frame(
                    sph_filemanager_get_basename( cpl_frame_get_filename(inframe)), "NONE",
                    CPL_FRAME_GROUP_NONE);
            //pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( inframe ), ext_cam1, ".*ESO.*", 0 );
            pl = cpl_propertylist_load(cpl_frame_get_filename(inframe),
                    ext_cam1);

            if (!pl) {
                SPH_ERR("Can't read filtered propertylist.");
                SPH_RAISE_CPL;
                cpl_propertylist_delete(pl);
                pl = NULL;
                cpl_propertylist_delete(plm);
                plm = NULL;
                cpl_image_delete(rawimage_cam1);
                rawimage_cam1 = NULL;
                cpl_frameset_delete(results);
                return NULL;
            }

            //append keywords for camera id, phase (0,pi), frame_id (unique frame ID for both phases "0,PI")
            cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_PHASE, phase);
            cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FRAME_ID, frame_id);

            //add main property list (extension 0) to the current pl
            if (pl && plm) {
                for (int i = 0; i < cpl_propertylist_get_size(plm); i++) {
                    if (!cpl_propertylist_has(
                            pl,
                            cpl_property_get_name(
                                    cpl_propertylist_get(plm, i)))) {
                        rerr = cpl_propertylist_append_property(pl,
                                cpl_propertylist_get(plm, i));
                        //sph_error_raise(SPH_ERROR_INFO,  __FILE__, __func__, __LINE__, SPH_ERROR_INFO,
                        //        "The property from plm list to append to pl list: %s\n",
                        //        cpl_property_get_name( cpl_propertylist_get( plm, i) ));
                        //if (i==2) break;
                    } else {
                        sph_error_raise(
                                SPH_ERROR_INFO,
                                __FILE__,
                                __func__,
                                __LINE__,
                                SPH_ERROR_INFO,
                                "The propertylist pl has already the same keyword as in the pm: %s\n"
                                        "The value of the keyword in pl will not be changed.",
                                cpl_property_get_name(
                                        cpl_propertylist_get(plm, i)));
                        //rerr = cpl_propertylist_erase( pl, cpl_property_get_name( cpl_propertylist_get( plm, i) ) );
                        //rerr = cpl_propertylist_append_property( pl, cpl_propertylist_get( plm, i) );
                    }
                }
            } else {
                SPH_ERR("Property lists (pl & plm) are NULL.");
                SPH_RAISE_CPL;
                cpl_propertylist_delete(pl);
                pl = NULL;
                cpl_propertylist_delete(plm);
                plm = NULL;
                cpl_image_delete(rawimage_cam1);
                rawimage_cam1 = NULL;
                cpl_frameset_delete(results);
                return NULL;
            }

            if (rerr != CPL_ERROR_NONE) {
                SPH_ERR(
                        "Error occured by appending keywords from the main propertylist (ext=0)");
                SPH_RAISE_CPL;
                cpl_propertylist_delete(pl);
                pl = NULL;
                cpl_propertylist_delete(plm);
                plm = NULL;
                cpl_image_delete(rawimage_cam1);
                rawimage_cam1 = NULL;
                cpl_frameset_delete(results);
                return NULL;
            }

            //cpl_image_save ( rawimage_cam1, "test1.tmp.fits",CPL_BPP_32_SIGNED, (const  cpl_propertylist* ) pl, CPL_IO_CREATE );
            /* Update the header if required */
            sph_utils_update_header(pl);
            cpl_image_save((const cpl_image*) rawimage_cam1,
                    (const char*) cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, (const cpl_propertylist*) pl,
                    CPL_IO_CREATE);
            cpl_propertylist_delete(pl);
            pl = NULL;
            cpl_image_delete(rawimage_cam1);
            rawimage_cam1 = NULL;
            cpl_frameset_insert(results, frame);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "cam1: extracted image saved in filename: %s.\n",
                    cpl_frame_get_filename(frame));

        } else {
            if (nplane == 0) {
                sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "No image can be loaded from raw file: %s.\n"
                                "CPL ERROR CODE: %d",
                        cpl_frame_get_filename(inframe), cpl_error_get_code());
                cpl_propertylist_delete(plm);
                cpl_frameset_delete(results);
                return NULL;
            }
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "%d frames have been extracted from raw file: %s.\n",
                    nplane, cpl_frame_get_filename(inframe));
            //cpl_propertylist_delete(plm);
            cpl_image_delete(rawimage_cam1);
            cpl_error_reset();
            break;
        } //end-if-else-block camera-1
        /* camera-2 */
        rawimage_cam2 = cpl_image_load(cpl_frame_get_filename(inframe),
                CPL_TYPE_INT, nplane, ext_cam2);
        if (cpl_error_get_code() == 0) {
            frame = sph_filemanager_create_temp_frame(
                    sph_filemanager_get_basename( cpl_frame_get_filename(inframe)), "NONE",
                    CPL_FRAME_GROUP_NONE);
            //pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( inframe ), ext_cam2, ".*ESO.*", 0 );
            pl = cpl_propertylist_load(cpl_frame_get_filename(inframe),
                    ext_cam2);

            //append keywords for camera id, phase (0,pi), frame_id (unique frame ID for both phases "0,PI")
            cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_PHASE, phase);
            cpl_propertylist_append_int(pl, SPH_ZPL_KEYWORD_FRAME_ID, frame_id);

            //add main property list (extension 0) to the current pl
            if (pl && plm) {
                for (int i = 0; i < cpl_propertylist_get_size(plm); i++) {
                    if (!cpl_propertylist_has(
                            pl,
                            cpl_property_get_name(
                                    cpl_propertylist_get(plm, i)))) {
                        rerr = cpl_propertylist_append_property(pl,
                                cpl_propertylist_get(plm, i));
                    } else {
                        sph_error_raise(
                                SPH_ERROR_INFO,
                                __FILE__,
                                __func__,
                                __LINE__,
                                SPH_ERROR_INFO,
                                "The propertylist pl has already the same keyword as in the pm: %s\n"
                                        "The value of the keyword in pl will not be changed.",
                                cpl_property_get_name(
                                        cpl_propertylist_get(plm, i)));
                        //rerr = cpl_propertylist_erase( pl, cpl_property_get_name( cpl_propertylist_get( plm, i) ) );
                        //rerr = cpl_propertylist_append_property( pl, cpl_propertylist_get( plm, i) );
                    }
                }
            } else {
                SPH_ERR("Property lists (pl & plm) are NULL.");
                SPH_RAISE_CPL;
                cpl_propertylist_delete(pl);
                pl = NULL;
                cpl_propertylist_delete(plm);
                plm = NULL;
                cpl_image_delete(rawimage_cam2);
                rawimage_cam2 = NULL;
                cpl_frameset_delete(results);
                return NULL;

            }

            if (rerr != CPL_ERROR_NONE) {
                SPH_ERR(
                        "Error occured by appending keywords from the main propertylist (ext=0)");
                SPH_RAISE_CPL;
                cpl_propertylist_delete(pl);
                cpl_propertylist_delete(plm);
                cpl_image_delete(rawimage_cam2);
                cpl_frameset_delete(results);
                return NULL;
            }

            //cpl_image_save( rawimage_cam2, cpl_frame_get_filename( frame ), CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE);
            /* Update the header if required */
            sph_utils_update_header(pl);
            cpl_image_save((const cpl_image*) rawimage_cam2,
                    (const char*) cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, (const cpl_propertylist*) pl,
                    CPL_IO_CREATE);

            cpl_propertylist_delete(pl);
            pl = NULL;
            cpl_image_delete(rawimage_cam2);
            rawimage_cam2 = NULL;
            cpl_frameset_insert(results, frame);
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "cam2: extracted image saved in filename: %s.\n",
                    cpl_frame_get_filename(frame));

        } else {
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "%d frames have been extracted from raw file: %s.\n",
                    nplane, cpl_frame_get_filename(inframe));
            sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING, "Number of frames is not even: %d.",
                    nplane);
            //cpl_propertylist_delete(plm);
            cpl_image_delete(rawimage_cam2);
            cpl_error_reset();
            break;
        } //end-if-else-block camera-2
        cpl_propertylist_delete(plm);
        nplane = nplane + 1;
    } //end while-loop

    cpl_propertylist_delete(plm);

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        SPH_RAISE_CPL;
        cpl_frameset_delete(results);
        results = NULL;
    }
    return results;

}

/*----------------------------------------------------------------------------*/
/**
 @brief    Combine images of the 2 detector segment into a single image
 @return   a pointer to the cpl_frameset with the combined ADU images or NULL if
 unsuccessfull.
 @note     A SPH and CPL error is raised in case there was a problem.

 This is a second step in pre-processing of the zimpol rawdata:
 This function combines 2 detector segments (ADU) for an each frame from the
 frameset (exframes) into a single image and it treams away overscan areas.

 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_combine_adus(cpl_frameset* exframes) {
    cpl_frame* curframe = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_image* image = NULL;
    cpl_image* image_adu1 = NULL;
    cpl_image* image_adu2 = NULL;
    cpl_image* image_adus = NULL;
    cpl_image* image_adu1_ovsc = NULL;
    cpl_image* image_adu2_ovsc = NULL;
    cpl_frame* frame = NULL;
    int camera_id = 0;
    int bin_factor_x = 0; // it must be read from the header
    int bin_factor_y = 0; // it must be read from the header
    int adu1_x, adu1_y, adu1_nx, adu1_ny;
    int adu1_prscx, adu1_prscy, adu1_ovscx, adu1_ovscy;
    int adu2_x, adu2_y, adu2_nx, adu2_ny;
    int adu2_prscx, adu2_prscy, adu2_ovscx, adu2_ovscy;
    int llx, lly, urx, ury;
    //double                  adu1_ovsc_mean = 0.0;
    //double                  adu2_ovsc_mean = 0.0;
    //double                  adu1_ovsc_rms = 0.0;
    //double                  adu2_ovsc_rms = 0.0;

    double sigma = 0.0; //might be used later as parameter for this function if needed

    sph_master_frame* adu1_ovsc_mframe = NULL;
    sph_master_frame* adu2_ovsc_mframe = NULL;
    int llx_ovsc_adu1, lly_ovsc_adu1, urx_ovsc_adu1, ury_ovsc_adu1;
    int llx_ovsc_adu2, lly_ovsc_adu2, urx_ovsc_adu2, ury_ovsc_adu2;

    cpl_error_reset();
    if (!exframes)
        return NULL;

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(exframes);
    while (curframe) {
        int xsize_adu1, ysize_adu1, xsize_adu2, ysize_adu2;

        ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe ), 0, ".*ESO.*", 0 );
        pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);

        camera_id = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID);

        if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {

            // adu-1
            adu1_x = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_X);
            adu1_y = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU1_Y);
            adu1_nx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_NX);
            adu1_ny = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_NY);
            adu1_prscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_PRSCX);
            adu1_prscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_PRSCY);
            adu1_ovscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_OVSCX);
            adu1_ovscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU1_OVSCY);

            bin_factor_x = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_BIN_X);
            bin_factor_y = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_BIN_Y);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Camera-1, ADU-1 sizes (original keywords): \n"
                            "adu1_x = %d\n"
                            "adu1_y = %d\n"
                            "adu1_nx = %d\n"
                            "adu1_ny = %d\n"
                            "adu1_prscx = %d\n"
                            "adu1_prscy = %d\n"
                            "adu1_ovscx = %d\n"
                            "adu1_ovscy = %d\n", adu1_x, adu1_y, adu1_nx,
                    adu1_ny, adu1_prscx, adu1_prscy, adu1_ovscx, adu1_ovscy);

            if (bin_factor_x <= 0)
                bin_factor_x = 2; //set default binning factor to 2
            if (bin_factor_y <= 0)
                bin_factor_y = 2; //set default binning factor to 2

            //re-calculate keywords to the old pre-processing "coord system" used in the sph_zpl_util pre-processing steps
            //first is to apply binning factor
            adu1_prscx = adu1_prscx / bin_factor_x;
            adu1_prscy = adu1_prscy / bin_factor_y;
            adu1_ovscx = adu1_ovscx / bin_factor_x;
            adu1_ovscy = adu1_ovscy / bin_factor_y;
            adu1_nx = adu1_nx / bin_factor_x;
            adu1_ny = adu1_ny / bin_factor_y;
            //second the meaning of the adu1_nx, in the old specifications it includes the prescan and overscan regions
            adu1_nx = adu1_nx + adu1_prscx + adu1_ovscx;

            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, 0);
            llx = adu1_x + adu1_prscx;
            lly = adu1_y + adu1_prscy;
            urx = adu1_x + (adu1_nx - 1) - adu1_ovscx;
            ury = adu1_y + (adu1_ny - 1) - adu1_ovscy;
            image_adu1 = cpl_image_extract(image, llx, lly, urx, ury);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO, "Camera-1, ADU-1 extract coords: \n"
                            "llx = %d\n"
                            "lly = %d\n"
                            "urx = %d\n"
                            "ury = %d\n", llx, lly, urx, ury);

            // extract over-scan image for the adu1 & calculate mean with statistics (rms)
            // use master frame for that
            llx_ovsc_adu1 = urx + 1;
            lly_ovsc_adu1 = lly;
            urx_ovsc_adu1 = urx + adu1_ovscx;
            ury_ovsc_adu1 = ury;
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Camera-1, ADU-1 ovescan  extract coords : \n"
                            "llx_ovsc_adu1         (urx + 1 ) = %d\n"
                            "lly_ovsc_adu1              (lly) = %d\n"
                            "urx_ovsc_adu1 (urx + adu1_ovscx) = %d\n"
                            "ury_ovsc_adu1              (ury) = %d\n",
                    llx_ovsc_adu1, lly_ovsc_adu1, urx_ovsc_adu1, ury_ovsc_adu1);

            image_adu1_ovsc = cpl_image_extract(image, llx_ovsc_adu1,
                    lly_ovsc_adu1, urx_ovsc_adu1, ury_ovsc_adu1);
            adu1_ovsc_mframe = sph_master_frame_new_from_cpl_image(
                    image_adu1_ovsc);
            sph_master_frame_quality_check(adu1_ovsc_mframe);
            if (sigma > 0.0) {
                sph_master_frame_mask_sigma(adu1_ovsc_mframe, sigma);
                sph_master_frame_quality_check(adu1_ovsc_mframe);
            }
            cpl_propertylist_update_double(
                    pl,
                    SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN,
                    cpl_propertylist_get_double(adu1_ovsc_mframe->qclist,
                            SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME));
            cpl_propertylist_update_double(
                    pl,
                    SPH_ZPL_KEYWORD_ADU1_OVSC_RMS,
                    cpl_propertylist_get_double(adu1_ovsc_mframe->qclist,
                            SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME));

            cpl_image_delete(image_adu1_ovsc);
            image_adu1_ovsc = NULL;
            sph_master_frame_delete(adu1_ovsc_mframe);
            adu1_ovsc_mframe = NULL;
            //adu1_ovsc_mean = cpl_image_get_mean( image_adu1_ovsc );
            //adu1_ovsc_rms = cpl_image_get_stdev( image_adu1_ovsc );
            //bad_mask = cpl_mask_threshold_image_create ( image_adu1_ovsc , adu1_ovsc_mean - sigma * rms, adu1_ovsc_mean + sigma * rms);
            //cpl_ensure_code(bad_mask,cpl_error_get_code());
            //cpl_mask_not(bad_mask);

            // adu-2
            adu2_x = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_X);
            adu2_y = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM1_ADU2_Y);
            adu2_nx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_NX);
            adu2_ny = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_NY);
            adu2_prscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_PRSCX);
            adu2_prscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_PRSCY);
            adu2_ovscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_OVSCX);
            adu2_ovscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM1_ADU2_OVSCY);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Camera-1, ADU-2 sizes (original keywords): \n"
                            "adu2_x = %d\n"
                            "adu2_y = %d\n"
                            "adu2_nx = %d\n"
                            "adu2_ny = %d\n"
                            "adu2_prscx = %d\n"
                            "adu2_prscy = %d\n"
                            "adu2_ovscx = %d\n"
                            "adu2_ovscy = %d\n", adu2_x, adu2_y, adu2_nx,
                    adu2_ny, adu2_prscx, adu2_prscy, adu2_ovscx, adu2_ovscy);

            //re-calculate keywords to the old pre-processing "coord system" used in the sph_zpl_util pre-processing steps
            //first is to apply binning factor
            adu2_prscx = adu2_prscx / bin_factor_x;
            adu2_prscy = adu2_prscy / bin_factor_y;
            adu2_ovscx = adu2_ovscx / bin_factor_x;
            adu2_ovscy = adu2_ovscy / bin_factor_y;
            adu2_nx = adu2_nx / bin_factor_x;
            adu2_ny = adu2_ny / bin_factor_y;
            //second the adu2_nx, in the old specifications it includes the prescan and overscan regions
            adu2_nx = adu2_nx + adu2_prscx + adu2_ovscx;
            //third the adu2_x, in the old coord sys it is a begin point for the second adu
            adu2_x = adu2_nx + 1;

            //forth switch the adu2_prscx with the adu_ovscx by applying to the adu2

            llx = adu2_x + adu2_ovscx;
            lly = adu2_y + adu2_prscy;
            urx = adu2_x + (adu2_nx - 1) - adu2_prscx;
            ury = adu2_y + (adu2_ny - 1) - adu2_ovscy;
            image_adu2 = cpl_image_extract(image, llx, lly, urx, ury);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO, "Camera-1, ADU-2 extract coords: \n"
                            "llx = %d\n"
                            "lly = %d\n"
                            "urx = %d\n"
                            "ury = %d\n", llx, lly, urx, ury);

            // extract over-scan image for the adu2 & calculate mean with statistics (rms)
            // use master frame for that
            llx_ovsc_adu2 = urx_ovsc_adu1 + 1;
            lly_ovsc_adu2 = lly;
            urx_ovsc_adu2 = llx - 1;
            ury_ovsc_adu2 = ury;
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Camera-1, ADU-1 ovescan  extract coords : \n"
                            "llx_ovsc_adu2  (urx_ovsc_adu1 + 1) = %d\n"
                            "lly_ovsc_adu2                (lly) = %d\n"
                            "urx_ovsc_adu2            (llx - 1) = %d\n"
                            "ury_ovsc_adu2                (ury) = %d\n",
                    llx_ovsc_adu2, lly_ovsc_adu2, urx_ovsc_adu2, ury_ovsc_adu2);

            image_adu2_ovsc = cpl_image_extract(image, llx_ovsc_adu2,
                    lly_ovsc_adu2, urx_ovsc_adu2, ury_ovsc_adu2);
            adu2_ovsc_mframe = sph_master_frame_new_from_cpl_image(
                    image_adu2_ovsc);
            sph_master_frame_quality_check(adu2_ovsc_mframe);
            if (sigma > 0.0) {
                sph_master_frame_mask_sigma(adu2_ovsc_mframe, sigma);
                sph_master_frame_quality_check(adu2_ovsc_mframe);
            }
            cpl_propertylist_update_double(
                    pl,
                    SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN,
                    cpl_propertylist_get_double(adu2_ovsc_mframe->qclist,
                            SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME));
            cpl_propertylist_update_double(
                    pl,
                    SPH_ZPL_KEYWORD_ADU2_OVSC_RMS,
                    cpl_propertylist_get_double(adu2_ovsc_mframe->qclist,
                            SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME));

            cpl_image_delete(image_adu2_ovsc);
            image_adu2_ovsc = NULL;
            sph_master_frame_delete(adu2_ovsc_mframe);
            adu2_ovsc_mframe = NULL;

        } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {

            // adu-1
            adu1_x = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_X);
            adu1_y = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU1_Y);
            adu1_nx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_NX);
            adu1_ny = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_NY);
            adu1_prscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_PRSCX);
            adu1_prscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_PRSCY);
            adu1_ovscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_OVSCX);
            adu1_ovscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU1_OVSCY);

            bin_factor_x = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_BIN_X);
            bin_factor_y = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_BIN_Y);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Camera-2, ADU-1 sizes (original keywords): \n"
                            "adu1_x = %d\n"
                            "adu1_y = %d\n"
                            "adu1_nx = %d\n"
                            "adu1_ny = %d\n"
                            "adu1_prscx = %d\n"
                            "adu1_prscy = %d\n"
                            "adu1_ovscx = %d\n"
                            "adu1_ovscy = %d\n", adu1_x, adu1_y, adu1_nx,
                    adu1_ny, adu1_prscx, adu1_prscy, adu1_ovscx, adu1_ovscy);

            if (bin_factor_x <= 0)
                bin_factor_x = 2; //set default binning factor to 2
            if (bin_factor_y <= 0)
                bin_factor_y = 2; //set default binning factor to 2

            //re-calculate keywords to the old pre-processing "coord system" used in the sph_zpl_util pre-processing steps
            //first is to apply binning factor
            adu1_prscx = adu1_prscx / bin_factor_x;
            adu1_prscy = adu1_prscy / bin_factor_y;
            adu1_ovscx = adu1_ovscx / bin_factor_x;
            adu1_ovscy = adu1_ovscy / bin_factor_y;
            adu1_nx = adu1_nx / bin_factor_x;
            adu1_ny = adu1_ny / bin_factor_y;
            //second the meaning of the adu1_nx, in the old specifications it includes the prescan and overscan regions
            adu1_nx = adu1_nx + adu1_prscx + adu1_ovscx;

            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, 0);
            llx = adu1_x + adu1_prscx;
            lly = adu1_y + adu1_prscy;
            urx = adu1_x + (adu1_nx - 1) - adu1_ovscx;
            ury = adu1_y + (adu1_ny - 1) - adu1_ovscy;
            image_adu1 = cpl_image_extract(image, llx, lly, urx, ury);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO, "Camera-2, ADU-1 extract coords: \n"
                            "llx = %d\n"
                            "lly = %d\n"
                            "urx = %d\n"
                            "ury = %d\n", llx, lly, urx, ury);

            // extract over-scan image for the adu1 & calculate mean with statistics (rms)
            // use master frame for that
            llx_ovsc_adu1 = urx + 1;
            lly_ovsc_adu1 = lly;
            urx_ovsc_adu1 = urx + adu1_ovscx;
            ury_ovsc_adu1 = ury;
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Camera-2, ADU-1 ovescan  extract coords : \n"
                            "llx_ovsc_adu1         (urx + 1 ) = %d\n"
                            "lly_ovsc_adu1              (lly) = %d\n"
                            "urx_ovsc_adu1 (urx + adu1_ovscx) = %d\n"
                            "ury_ovsc_adu1              (ury) = %d\n",
                    llx_ovsc_adu1, lly_ovsc_adu1, urx_ovsc_adu1, ury_ovsc_adu1);

            image_adu1_ovsc = cpl_image_extract(image, llx_ovsc_adu1,
                    lly_ovsc_adu1, urx_ovsc_adu1, ury_ovsc_adu1);
            adu1_ovsc_mframe = sph_master_frame_new_from_cpl_image(
                    image_adu1_ovsc);
            sph_master_frame_quality_check(adu1_ovsc_mframe);
            if (sigma > 0.0) {
                sph_master_frame_mask_sigma(adu1_ovsc_mframe, sigma);
                sph_master_frame_quality_check(adu1_ovsc_mframe);
            }
            cpl_propertylist_update_double(
                    pl,
                    SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN,
                    cpl_propertylist_get_double(adu1_ovsc_mframe->qclist,
                            SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME));
            cpl_propertylist_update_double(
                    pl,
                    SPH_ZPL_KEYWORD_ADU1_OVSC_RMS,
                    cpl_propertylist_get_double(adu1_ovsc_mframe->qclist,
                            SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME));

            cpl_image_delete(image_adu1_ovsc);
            image_adu1_ovsc = NULL;
            sph_master_frame_delete(adu1_ovsc_mframe);
            adu1_ovsc_mframe = NULL;

            // adu-2
            adu2_x = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_X);
            adu2_y = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAM2_ADU2_Y);
            adu2_nx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_NX);
            adu2_ny = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_NY);
            adu2_prscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_PRSCX);
            adu2_prscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_PRSCY);
            adu2_ovscx = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_OVSCX);
            adu2_ovscy = cpl_propertylist_get_int(pl,
                    SPH_ZPL_KEYWORD_CAM2_ADU2_OVSCY);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Camera-2, ADU-2 sizes (original keywords): \n"
                            "adu2_x = %d\n"
                            "adu2_y = %d\n"
                            "adu2_nx = %d\n"
                            "adu2_ny = %d\n"
                            "adu2_prscx = %d\n"
                            "adu2_prscy = %d\n"
                            "adu2_ovscx = %d\n"
                            "adu2_ovscy = %d\n", adu2_x, adu2_y, adu2_nx,
                    adu2_ny, adu2_prscx, adu2_prscy, adu2_ovscx, adu2_ovscy);

            //re-calculate keywords to the old pre-processing "coord system" used in the sph_zpl_util pre-processing steps
            //first is to apply binning factor
            adu2_prscx = adu2_prscx / bin_factor_x;
            adu2_prscy = adu2_prscy / bin_factor_y;
            adu2_ovscx = adu2_ovscx / bin_factor_x;
            adu2_ovscy = adu2_ovscy / bin_factor_y;
            adu2_nx = adu2_nx / bin_factor_x;
            adu2_ny = adu2_ny / bin_factor_y;
            //second the adu2_nx, in the old specifications it includes the prescan and overscan regions
            adu2_nx = adu2_nx + adu2_prscx + adu2_ovscx;
            //third the adu2_x, in the old coord sys it is a begin point for the second adu
            adu2_x = adu2_nx + 1;

            //forth switch the adu2_prscx with the adu_ovscx by applying to the adu2

            llx = adu2_x + adu2_ovscx;
            lly = adu2_y + adu2_prscy;
            urx = adu2_x + (adu2_nx - 1) - adu2_prscx;
            ury = adu2_y + (adu2_ny - 1) - adu2_ovscy;
            image_adu2 = cpl_image_extract(image, llx, lly, urx, ury);

            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO, "Camera-2, ADU-2 extract coords: \n"
                            "llx = %d\n"
                            "lly = %d\n"
                            "urx = %d\n"
                            "ury = %d\n", llx, lly, urx, ury);

            // extract over-scan image for the adu2 & calculate mean with statistics (rms)
            // use master frame for that
            llx_ovsc_adu2 = urx_ovsc_adu1 + 1;
            lly_ovsc_adu2 = lly;
            urx_ovsc_adu2 = llx - 1;
            ury_ovsc_adu2 = ury;
            sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                    SPH_ERROR_INFO,
                    "Camera-2, ADU-1 ovescan  extract coords : \n"
                            "llx_ovsc_adu2  (urx_ovsc_adu1 + 1) = %d\n"
                            "lly_ovsc_adu2                (lly) = %d\n"
                            "urx_ovsc_adu2            (llx - 1) = %d\n"
                            "ury_ovsc_adu2                (ury) = %d\n",
                    llx_ovsc_adu2, lly_ovsc_adu2, urx_ovsc_adu2, ury_ovsc_adu2);

            image_adu2_ovsc = cpl_image_extract(image, llx_ovsc_adu2,
                    lly_ovsc_adu2, urx_ovsc_adu2, ury_ovsc_adu2);
            adu2_ovsc_mframe = sph_master_frame_new_from_cpl_image(
                    image_adu2_ovsc);
            sph_master_frame_quality_check(adu2_ovsc_mframe);
            if (sigma > 0.0) {
                sph_master_frame_mask_sigma(adu2_ovsc_mframe, sigma);
                sph_master_frame_quality_check(adu2_ovsc_mframe);
            }
            cpl_propertylist_update_double(
                    pl,
                    SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN,
                    cpl_propertylist_get_double(adu2_ovsc_mframe->qclist,
                            SPH_COMMON_KEYWORD_QC_MEANMASTERFRAME));
            cpl_propertylist_update_double(
                    pl,
                    SPH_ZPL_KEYWORD_ADU2_OVSC_RMS,
                    cpl_propertylist_get_double(adu2_ovsc_mframe->qclist,
                            SPH_COMMON_KEYWORD_QC_RMSMASTERFRAME));
            cpl_image_delete(image_adu2_ovsc);
            image_adu2_ovsc = NULL;
            sph_master_frame_delete(adu2_ovsc_mframe);
            adu2_ovsc_mframe = NULL;

        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Wrong camera ID: %d in the header of the fitsfile: %s.",
                    camera_id, cpl_frame_get_filename(curframe));
            cpl_propertylist_delete(pl);
            cpl_frameset_delete(results);
            return NULL;
        }

        //cpl_image_save( image_adu1, fname+"extract1.fits", CPL_TYPE_FLOAT, pl, 2 )
        //cpl_image_save( image_adu2, fname+"extract2.fits", CPL_TYPE_FLOAT, pl, 2 )
        xsize_adu1 = cpl_image_get_size_x(image_adu1);
        ysize_adu1 = cpl_image_get_size_y(image_adu1);
        xsize_adu2 = cpl_image_get_size_x(image_adu2);
        ysize_adu2 = cpl_image_get_size_y(image_adu2);
        if (xsize_adu1 != xsize_adu2 || ysize_adu1 != ysize_adu2) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "Problems with the size of ADUs: \n"
                            "xsize_adu1 = %d\n"
                            "ysize_adu1 = %d\n"
                            "xsize_adu2 = %d\n"
                            "ysize_adu2 = %d\n", xsize_adu1, ysize_adu1,
                    xsize_adu2, ysize_adu2);
            cpl_propertylist_delete(pl);
            cpl_frameset_delete(results);
            return NULL;
        }
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "ADU sizes: \n"
                        "xsize_adu1 = %d\n"
                        "ysize_adu1 = %d\n"
                        "xsize_adu2 = %d\n"
                        "ysize_adu2 = %d\n", xsize_adu1, ysize_adu1, xsize_adu2,
                ysize_adu2);

        image_adus = cpl_image_new(xsize_adu1 + xsize_adu2, ysize_adu1,
                CPL_TYPE_INT);
        cpl_image_copy(image_adus, image_adu1, 1, 1);
        cpl_image_copy(image_adus, image_adu2, xsize_adu1 + 1, 1);
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)), "NONE",
                CPL_FRAME_GROUP_NONE);
        /* Update the header if required */
        sph_utils_update_header(pl);
        cpl_image_save(image_adus, cpl_frame_get_filename(frame),
                CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE);
        cpl_frameset_insert(results, frame);

        // delete images to free memory
        cpl_image_delete(image);
        cpl_image_delete(image_adu1);
        cpl_image_delete(image_adu2);
        cpl_image_delete(image_adus);
        cpl_propertylist_delete(pl);

        curframe = cpl_frameset_get_next(exframes);
    }
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        cpl_frameset_delete(results);
        results = NULL;
    }
    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create trimmed images from the adus frames
 @return   a pointer to the cpl_frameset with the treamed images or NULL if
 unsuccessfull.
 @note     A SPH and CPL error is raised in case there was a problem.

 This is a third step in pre-processing of the zimpol rawdata:
 This function junk rows with regard to the phase (ZERO, PI) to create
 a "trimmed" image

 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_junk_rows(cpl_frameset* adusframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl;
    const char* phase;
    int lly_cut, ury_cut;
    cpl_image* image;
    cpl_image* image_new;

    cpl_error_reset();
    if (!adusframes)
        return NULL;

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(adusframes);
    while (curframe) {
        ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe ), 0, ".*ESO.*", 0 );
        pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);

        phase = cpl_propertylist_get_string(pl, SPH_ZPL_KEYWORD_PHASE);

        if (!strcmp(phase, SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO)) {
            lly_cut = 2; //2; // 1; // bottom row(s) to be cut for phase = ZERO (in PYTHON) + 1
            ury_cut = 0; //2; //1; // upper row(s) to be cut for phase = ZERO
            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, 0);
            image_new = cpl_image_extract(image, 1, 1 + lly_cut,
                    cpl_image_get_size_x(image),
                    cpl_image_get_size_y(image) - ury_cut);
        } else if (!strcmp(phase, SPH_ZPL_KEYWORD_VALUE_PHASE_PI)) {
            lly_cut = 1; //1; // 0; // bottom row(s) to be cut for phase = PI
            ury_cut = 1; //3; //2; // upper row(s) to be cut for phase = PI (in PYTHON)
            image = cpl_image_load(cpl_frame_get_filename(curframe),
                    CPL_TYPE_INT, 0, 0);
            image_new = cpl_image_extract(image, 1, 1 + lly_cut,
                    cpl_image_get_size_x(image),
                    cpl_image_get_size_y(image) - ury_cut);
        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Not expected phase value (string: ZERO/PI): %s", phase);
            cpl_propertylist_delete(pl);
            cpl_frameset_delete(results);
            results = NULL;
            return NULL;
        }

        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)), "NONE",
                CPL_FRAME_GROUP_NONE);
        /* Update the header if required */
        sph_utils_update_header(pl);
        cpl_image_save(image_new, cpl_frame_get_filename(frame),
                CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE);

        cpl_image_delete(image);
        cpl_image_delete(image_new);
        cpl_propertylist_delete(pl);
        cpl_frameset_insert(results, frame);
        curframe = cpl_frameset_get_next(adusframes);
    }
    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        cpl_frameset_delete(results);
        return NULL;
    }
    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Split "trimmed" images of the frames into even and odd sub-images
 @return   a pointer to the cpl_frameset with even and odd sub-images or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This is a forth-A step in pre-processing of the zimpol rawdata:
 This function  splits "trimmed" images of the frames into even and
 odd sub-images.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_split_image(cpl_frameset* jrframes,
		const cpl_boolean swap_odd_even_rows) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_image* image = NULL;
    cpl_image* image_odd = NULL;
    cpl_image* image_even = NULL;
    int nx, ny, pis_rejected, ynew;
    double val;

    cpl_error_reset();

    if (!jrframes)
        return NULL;

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(jrframes);
    while (curframe) {
        ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe ), 0, ".*ESO.*", 0 );
        pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);

        //ftmp = fopen(sph_filemanager_get_tmp_filename_const("propertylist"), "w");
        //cpl_propertylist_dump(pl, ftmp);
        //fclose(ftmp);

        //pl = cpl_propertylist_new();
        if (pl == NULL) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "property list is NULL. CPL error message: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));
        }
        image = cpl_image_load(cpl_frame_get_filename(curframe), CPL_TYPE_INT,
                0, 0);
        nx = cpl_image_get_size_x(image);
        ny = cpl_image_get_size_y(image);
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "nx= %d, ny=%d", nx, ny);
        if (ny % 2 != 0) {
            sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING, "ny = %d not an even number.", ny);
            cpl_image_delete(image);
            return NULL;
        }
        image_odd = cpl_image_new(nx, ny / 2, CPL_TYPE_INT);
        image_even = cpl_image_new(nx, ny / 2, CPL_TYPE_INT);
        for (int xx = 0; xx < nx; ++xx) {
            for (int yy = 0; yy < ny; ++yy) {
                val = cpl_image_get(image, xx + 1, yy + 1, &pis_rejected);
                ynew = yy / 2;
                if (yy % 2 == 0) {
                    cpl_image_set(image_odd, xx + 1, ynew + 1, val); //with regard to the yy it is even
                                                                     //but cpl_image starts from 1,1 pixels (fits convention)
                } else {
                    cpl_image_set(image_even, xx + 1, ynew + 1, val); //with regard to the yy it is odd
                                                                      //but cpl_image starts from 1,1 pixels (fits convention)
                }
                if (cpl_error_get_code() != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__,
                            __LINE__, SPH_ERROR_WARNING,
                            "%s: nx= %d, ny=%d, xx=%d, yy=%d, ynew=%d",
                            cpl_error_get_message_default(cpl_error_get_code()),
                            nx, ny, xx, yy, ynew);
                    /* temporarily commented
                     sph_error_raise(SPH_ERROR_ERROR,
                     __FILE__, __func__, __LINE__,
                     SPH_ERROR_ERROR, "CPL error has occured: %d \n"
                     "Info message: %s \n"
                     "nx= %d, ny=%d, xx=%d, yy=%d, ynew=%d", cpl_error_get_code(),
                     cpl_error_get_message_default(cpl_error_get_code()),
                     nx, ny, xx, yy, ynew);
                     */
                    cpl_error_reset(); //temporarily solution to avoid error arising.
                    //return NULL; //temporarily commented
                }
            }
        }

        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)), "NONE",
                CPL_FRAME_GROUP_NONE);

        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Problem with appending propertylist. CPL error message: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));
        }

        if(swap_odd_even_rows){
        	cpl_image * tmp = image_odd;
        	image_odd = image_even;
        	image_even = tmp;
        }

        /* Update the header if required */
        sph_utils_update_header(pl);

        if (cpl_propertylist_get_int(
                pl,
                SPH_ZPL_KEYWORD_CAMERA_ID) == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {

            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD);
            //ftmp = fopen(sph_filemanager_get_tmp_filename_const("propertylist"), "w");
            //cpl_propertylist_dump(pl, ftmp);
            //fclose(ftmp);
            cpl_image_save((const cpl_image*) image_odd,
                    (const char*) cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, (const cpl_propertylist*) pl,
                    CPL_IO_CREATE); // 2 means create a new fits file
            cpl_image_delete(image);
            cpl_image_delete(image_odd);
            cpl_propertylist_update_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN);
            cpl_image_save((const cpl_image*) image_even,
                    (const char*) cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, (const cpl_propertylist*) pl,
                    CPL_IO_EXTEND); // 4 means CPL_IO_EXTEND Mode (signify the creation of extention in fits file)
            cpl_image_delete(image_even);
            cpl_propertylist_delete(pl);
        } else if (cpl_propertylist_get_int(
                pl,
                SPH_ZPL_KEYWORD_CAMERA_ID) == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {

            /* subframes for the CAMERA2 must be swapped, i.e. we save first even and then odd     */
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD);
            cpl_image_save((const cpl_image*) image_even,
                    (const char*) cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, (const cpl_propertylist*) pl,
                    CPL_IO_CREATE); // 2 means create a new fits file
            cpl_image_delete(image);
            cpl_image_delete(image_even);
            cpl_propertylist_update_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN);
            cpl_image_save((const cpl_image*) image_odd,
                    (const char*) cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, (const cpl_propertylist*) pl,
                    CPL_IO_EXTEND); // 4 means CPL_IO_EXTEND Mode (signify the creation of extention in fits file)
            cpl_image_delete(image_odd);
            cpl_propertylist_delete(pl);
        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "CAMERA: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));

        }

        image = NULL;
        image_odd = NULL;
        image_even = NULL;
        pl = NULL;

        cpl_frameset_insert(results, frame);

        curframe = cpl_frameset_get_next(jrframes);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d \n"
                        "Error message: %s", cpl_error_get_code(),
                cpl_error_get_message_default(cpl_error_get_code()));
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Split "trimmed" images of the frames into even and odd sub-images in imaging mode
 @return   a pointer to the cpl_frameset with even and odd sub-images or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This is a forth-A step in pre-processing of the zimpol rawdata:
 This function  splits "trimmed" images of the frames into even and
 odd sub-images in imaging mode. This function is identical to the sph_zpl_common_preproc_split_image
 (polarimetric mode) except it doesn't make swap of the odd and even for the camera-2 as it
 is done in the case of the polarimetric mode. It does make swap for both cameras in the case of
 the "SnapShot Imaging" mode.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_split_image_imaging(
        cpl_frameset* jrframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_image* image = NULL;
    cpl_image* image_odd = NULL;
    cpl_image* image_even = NULL;
    int nx, ny, pis_rejected, ynew, camera_id;
    double val;
    const char* det_read_curname;

    cpl_error_reset();

    if (!jrframes)
        return NULL;

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(jrframes);
    while (curframe) {
        ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe ), 0, ".*ESO.*", 0 );
        pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);

        //ftmp = fopen(sph_filemanager_get_tmp_filename_const("propertylist"), "w");
        //cpl_propertylist_dump(pl, ftmp);
        //fclose(ftmp);

        //pl = cpl_propertylist_new();
        if (pl == NULL) {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "property list is NULL. CPL error message: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));
        }

        det_read_curname = cpl_propertylist_get_string(pl,
                SPH_ZPL_KEYWORD_DET_READ_CURNAME);
        if (!det_read_curname) {
            det_read_curname = "Keyword is not defined";
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_INFO,
                    "det_read_curname = %s => StandardImaging mode is set up as default", det_read_curname);
        }

        camera_id = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID);

        image = cpl_image_load(cpl_frame_get_filename(curframe), CPL_TYPE_INT,
                0, 0);
        nx = cpl_image_get_size_x(image);
        ny = cpl_image_get_size_y(image);
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "nx= %d, ny=%d", nx, ny);
        if (ny % 2 != 0) {
            sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__, __LINE__,
                    SPH_ERROR_WARNING, "ny = %d not an even number.", ny);
            cpl_image_delete(image);
            return NULL;
        }
        image_odd = cpl_image_new(nx, ny / 2, CPL_TYPE_INT);
        image_even = cpl_image_new(nx, ny / 2, CPL_TYPE_INT);
        for (int xx = 0; xx < nx; ++xx) {
            for (int yy = 0; yy < ny; ++yy) {
                val = cpl_image_get(image, xx + 1, yy + 1, &pis_rejected);
                ynew = yy / 2;
                if (yy % 2 == 0) {
                    cpl_image_set(image_odd, xx + 1, ynew + 1, val); //with regard to the yy it is even
                                                                     //but cpl_image starts from 1,1 pixels (fits convention)
                } else {
                    cpl_image_set(image_even, xx + 1, ynew + 1, val); //with regard to the yy it is odd
                                                                      //but cpl_image starts from 1,1 pixels (fits convention)
                }
                if (cpl_error_get_code() != CPL_ERROR_NONE) {
                    sph_error_raise(SPH_ERROR_WARNING, __FILE__, __func__,
                            __LINE__, SPH_ERROR_WARNING,
                            "%s: nx= %d, ny=%d, xx=%d, yy=%d, ynew=%d",
                            cpl_error_get_message_default(cpl_error_get_code()),
                            nx, ny, xx, yy, ynew);
                    /* temporarily commented
                     sph_error_raise(SPH_ERROR_ERROR,
                     __FILE__, __func__, __LINE__,
                     SPH_ERROR_ERROR, "CPL error has occured: %d \n"
                     "Info message: %s \n"
                     "nx= %d, ny=%d, xx=%d, yy=%d, ynew=%d", cpl_error_get_code(),
                     cpl_error_get_message_default(cpl_error_get_code()),
                     nx, ny, xx, yy, ynew);
                     */
                    cpl_error_reset(); //temporarily solution to avoid error arising.
                    //return NULL; //temporarily commented
                }
            }
        }

        //Adjust images orientation with respect to the usual "North is up and East is to the right" convention:
        // 1. An image flip (up -down) for both cameras
        cpl_image_flip( image_odd, IMAGE_FLIP_HORIZONTAL );
        cpl_image_flip( image_even, IMAGE_FLIP_HORIZONTAL );
        // 2. An image flip (left -right) only for the camera 2
        if ( camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID ){
            cpl_image_flip( image_odd, IMAGE_FLIP_VERTICAL );
            cpl_image_flip( image_even, IMAGE_FLIP_VERTICAL);
        }

        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)), "NONE",
                CPL_FRAME_GROUP_NONE);

        if (cpl_error_get_code() != CPL_ERROR_NONE) {
            sph_error_raise(
                    SPH_ERROR_ERROR,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Problem with appending propertylist. CPL error message: %s",
                    cpl_error_get_message_default(cpl_error_get_code()));
        }

        /* Update the header if required */
        sph_utils_update_header(pl);

        if (!strcmp(det_read_curname,
                SPH_ZPL_KEYWORD_VALUE_DET_READ_SNAPSHOT_IMAGING)) {
            SPH_INFO_MSG( "SnapShotImaging detector mode...");

            // swap extentions if the detector mode is "SnapShotImaging"
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD);
            cpl_image_save(image_even, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE); // 2 means create a new fits file
            cpl_image_delete(image);
            cpl_image_delete(image_even);
            cpl_propertylist_update_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN);
            cpl_image_save(image_odd, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_EXTEND); // 4 means CPL_IO_EXTEND Mode (signify the creation of extention in fits file)
            cpl_image_delete(image_odd);
            cpl_propertylist_delete(pl);

        } else {
            if (!strcmp(det_read_curname,
                    SPH_ZPL_KEYWORD_VALUE_DET_READ_STANDARD_IMAGING)) {
                SPH_INFO_MSG( "StandardImaging detector mode...");
            } else {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_INFO,
                        "det_read_curname = %s is neither StandardImaging nor SnapShotImaging"
                        " => StandardImaging mode is set up as default", det_read_curname);
            }
            cpl_propertylist_append_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_ODD);
            cpl_image_save(image_odd, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_CREATE); // 2 means create a new fits file
            cpl_image_delete(image);
            cpl_image_delete(image_odd);
            cpl_propertylist_update_string(pl, SPH_ZPL_KEYWORD_SPLIT,
                    SPH_ZPL_KEYWORD_VALUE_SPLIT_EVEN);
            cpl_image_save(image_even, cpl_frame_get_filename(frame),
                    CPL_BPP_32_SIGNED, pl, CPL_IO_EXTEND); // 4 means CPL_IO_EXTEND Mode (signify the creation of extention in fits file)
            cpl_image_delete(image_even);
            cpl_propertylist_delete(pl);

        }

        image = NULL;
        image_odd = NULL;
        image_even = NULL;
        pl = NULL;

        cpl_frameset_insert(results, frame);

        curframe = cpl_frameset_get_next(jrframes);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d \n"
                        "Error message: %s", cpl_error_get_code(),
                cpl_error_get_message_default(cpl_error_get_code()));
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create zimpol expose frames
 @return   a pointer to the cpl_frameset with zimpol exposure frame or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This is a forth-B  step in pre-processing of the zimpol rawdata:
 This function combines frames of two different phases (ZERO, PI) into
 four extension fits file. The input frames are separate frames with even and
 odd images (split-images) for each phase.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_create_zpl_exposure(
        cpl_frameset* splitframes) {
    cpl_frame* curframe_pair = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* frames_ccd = NULL;
    cpl_frameset* results = NULL;
    cpl_frameset* single_frames = NULL;
    cpl_frameset* paired_frames = NULL;
    cpl_propertylist* pl_pair_odd = NULL;
    cpl_propertylist* pl_pair_even = NULL;
    cpl_image* image_pair_odd = NULL;
    cpl_image* image_pair_even = NULL;
    int io;
    int camera_id;
    int n_paired_frames;
    int save_nx, save_ny;

    cpl_error_reset();
    if (!splitframes)
        return NULL;

    results = cpl_frameset_new();
    for (int ccdID = 1; ccdID < 3; ccdID++) {
        if (ccdID == 1) {
            camera_id = SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID; //ccdID = 1 -> CAM1_ID
        } else {
            camera_id = SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID; //ccdID = 2 -> CAM2_ID
        }
        frames_ccd = sph_zpl_common_preproc_get_frames_ccd(splitframes,
                camera_id);

        if (frames_ccd == NULL) {
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO, "No frames for the ccdID: %d, escape from the loops ", ccdID );
            continue;
        }

        // A frameset of all paired frames
        paired_frames = cpl_frameset_new();
        // A frameset of all single frames
        single_frames = cpl_frameset_new();
        // Find the paired frames - any leftovers go in single_frames
        sph_zpl_common_preproc_get_frame_pairs(frames_ccd, paired_frames, single_frames);

        // Iterate over the paired frames
        curframe_pair = cpl_frameset_get_first(paired_frames);
        n_paired_frames = 0;
        while (curframe_pair) {
          frame = sph_filemanager_create_temp_frame(
                  sph_filemanager_get_basename( cpl_frame_get_filename(curframe_pair)),
                  "NONE", CPL_FRAME_GROUP_NONE);
          io = 0;
          // Paired frames are guaranteed to be consecutive in the frameset at this point
          for (int i=0; i<2; i++) {
              image_pair_odd = cpl_image_load(
                      cpl_frame_get_filename(curframe_pair),
                      CPL_TYPE_INT, 0, 0);
              image_pair_even = cpl_image_load(
                      cpl_frame_get_filename(curframe_pair),
                      CPL_TYPE_INT, 0, 1);

              //Adjust images orientation with respect to the usual "North is up and East is to the right" convention:
              // 1. An image flip (up -down) for both cameras
              cpl_image_flip( image_pair_odd, IMAGE_FLIP_HORIZONTAL );
              cpl_image_flip( image_pair_even, IMAGE_FLIP_HORIZONTAL );
              // 2. An image flip (left -right) only for the camera 2
              if ( camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID ){
                  cpl_image_flip( image_pair_odd, IMAGE_FLIP_VERTICAL );
                  cpl_image_flip( image_pair_even, IMAGE_FLIP_VERTICAL);
              }

              ////pl_pair_odd = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe_pair ), 0, ".*ESO.*", 0 );
              pl_pair_odd = cpl_propertylist_load(
                      cpl_frame_get_filename(curframe_pair), 0);
              ////pl_pair_even = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe_pair ), 1, ".*ESO.*", 0 );
              pl_pair_even = cpl_propertylist_load(
                      cpl_frame_get_filename(curframe_pair), 1);
              /* Update the headers if required */
              sph_utils_update_header(pl_pair_odd);
              sph_utils_update_header(pl_pair_even);
              cpl_image_save(image_pair_odd,
                      cpl_frame_get_filename(frame),
                      CPL_BPP_32_SIGNED, pl_pair_odd, 2 + io);
              cpl_image_save(image_pair_even,
                      cpl_frame_get_filename(frame),
                      CPL_BPP_32_SIGNED, pl_pair_even, 4);
              cpl_image_delete(image_pair_odd);
              cpl_image_delete(image_pair_even);
              cpl_propertylist_delete(pl_pair_odd);
              cpl_propertylist_delete(pl_pair_even);
              io = 2;
              curframe_pair = cpl_frameset_get_next(paired_frames);
          } // end for

          n_paired_frames++;
          cpl_frameset_insert(results, frame);
        }

        cpl_frameset_delete(paired_frames);

	cpl_msg_info(cpl_func, "Found %d paired frames", n_paired_frames);

        // If no paired frames were found, then process the single frames
        if (n_paired_frames == 0) {
	    cpl_msg_info(cpl_func, "No paired frames, so processing any single frames");
            curframe_pair = cpl_frameset_get_first(single_frames);
            while (curframe_pair) {
                frame = sph_filemanager_create_temp_frame(
                        sph_filemanager_get_basename( cpl_frame_get_filename(curframe_pair)),
                        "NONE", CPL_FRAME_GROUP_NONE);
                image_pair_odd = cpl_image_load(
                        cpl_frame_get_filename(curframe_pair),
                        CPL_TYPE_INT, 0, 0);
                image_pair_even = cpl_image_load(
                        cpl_frame_get_filename(curframe_pair),
                        CPL_TYPE_INT, 0, 1);

                //Adjust images orientation with respect to the usual "North is up and East is to the right" convention:
                // 1. An image flip (up -down) for both cameras
                cpl_image_flip( image_pair_odd, IMAGE_FLIP_HORIZONTAL );
                cpl_image_flip( image_pair_even, IMAGE_FLIP_HORIZONTAL );
                // 2. An image flip (left -right) only for the camera 2
                if ( camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID ){
                    cpl_image_flip( image_pair_odd, IMAGE_FLIP_VERTICAL );
                    cpl_image_flip( image_pair_even, IMAGE_FLIP_VERTICAL);
                }

                ////pl_pair_odd = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe_pair ), 0, ".*ESO.*", 0 );
                pl_pair_odd = cpl_propertylist_load(
                        cpl_frame_get_filename(curframe_pair), 0);
                ////pl_pair_even = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe_pair ), 1, ".*ESO.*", 0 );
                pl_pair_even = cpl_propertylist_load(
                        cpl_frame_get_filename(curframe_pair), 1);
                // Add header to indicate this is a single frame
                cpl_propertylist_append_bool(pl_pair_odd, SPH_ZPL_KEYWORD_SINGLE, 1);
                cpl_propertylist_append_bool(pl_pair_even, SPH_ZPL_KEYWORD_SINGLE, 1);
                /* Update the headers if required */
                sph_utils_update_header(pl_pair_odd);
                sph_utils_update_header(pl_pair_even);
                cpl_image_save(image_pair_odd,
                        cpl_frame_get_filename(frame),
                        CPL_BPP_32_SIGNED, pl_pair_odd, 2);
                cpl_image_save(image_pair_even,
                        cpl_frame_get_filename(frame),
                        CPL_BPP_32_SIGNED, pl_pair_even, 4);
                save_nx = cpl_image_get_size_x(image_pair_odd);
                save_ny = cpl_image_get_size_y(image_pair_odd);
                cpl_image_delete(image_pair_odd);
                cpl_image_delete(image_pair_even);

                // Pad out the file with empty extensions so that later processing
                // steps still recognise it as polarimetry data.
                cpl_propertylist_append_bool(pl_pair_odd, SPH_ZPL_KEYWORD_SINGLE, 1);
                cpl_propertylist_append_string(pl_pair_odd, SPH_ZPL_KEYWORD_PHASE, SPH_ZPL_KEYWORD_VALUE_PHASE_PI);
                cpl_propertylist_append_bool(pl_pair_even, SPH_ZPL_KEYWORD_SINGLE, 1);
                cpl_propertylist_append_string(pl_pair_even, SPH_ZPL_KEYWORD_PHASE, SPH_ZPL_KEYWORD_VALUE_PHASE_PI);
                cpl_msg_info(cpl_func, "Adding empty extensions to %s", cpl_frame_get_filename(frame));
                /* Update the headers if required */
                sph_utils_update_header(pl_pair_odd);
                sph_utils_update_header(pl_pair_even);
                cpl_image_save(cpl_image_new(save_nx, save_ny, CPL_TYPE_INT),
                        cpl_frame_get_filename(frame),
                        CPL_BPP_32_SIGNED, pl_pair_odd, 4);
                cpl_image_save(cpl_image_new(save_nx, save_ny, CPL_TYPE_INT),
                        cpl_frame_get_filename(frame),
                        CPL_BPP_32_SIGNED, pl_pair_even, 4);

                cpl_propertylist_delete(pl_pair_odd);
                cpl_propertylist_delete(pl_pair_even);

                cpl_frameset_insert(results, frame);

                curframe_pair = cpl_frameset_get_next(single_frames);
            } // single_frames
        }

        if ( frames_ccd ) cpl_frameset_delete(frames_ccd);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Extract frames for a given CCD ID from the input frames
 @return   a pointer to the cpl_frameset with the same CCD ID or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_get_frames_ccd(
        cpl_frameset* inframes, int ccdID) {
    cpl_frame* curframe = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    int camera_id;

    cpl_error_reset();

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first(inframes);
    while (curframe) {
        ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename ( curframe ), 0, ".*ESO.*", 0 );
        pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);
        camera_id = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_CAMERA_ID);
        if (ccdID == camera_id) {
            cpl_frameset_insert(results, cpl_frame_duplicate(curframe));
        }
        cpl_propertylist_delete(pl);
        curframe = cpl_frameset_get_next(inframes);
    }

    if ( cpl_frameset_is_empty(results ) ) {
        SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO, "No frames found for the ccdID: %d", ccdID );
        if ( results ) {
            cpl_frameset_delete( results );
            results = NULL;
        }
    }
    if ( cpl_error_get_code() != CPL_ERROR_NONE ) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        if ( results) {
            cpl_frameset_delete(results);
            results = NULL;
        }
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Extract two frames with the same unique frame ID
 @return   a pointer to the cpl_frameset with 2 corresponding frames or with
 ini-frame if a pair was not found, or NULL if error is raised.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This function extracts two frames based on the before generated ID
 from a given frameset. The frame ID (id_frame_ini) is defined from
 the first frame of the given frameset. As soon as a corresponding
 frame is found the result (frameset with 2 frames) will be returned.
 If the corresponding frame is not found the frameset with
 one ini-frame will be returned.
 */
/*----------------------------------------------------------------------------*/
static void sph_zpl_common_preproc_get_frame_pairs(
        cpl_frameset* inframes, cpl_frameset* paired_frames, cpl_frameset* single_frames) {
    cpl_frameset* dup_inframes = NULL;
    cpl_frameset* frames_pair = NULL;
    cpl_frameset* frames_phase = NULL;
    cpl_frame* firstframe = NULL;
    cpl_frame* curframe = NULL;
    cpl_propertylist* pl = NULL;
    int id_frame_ini, id_frame;
    int paired = FALSE;

    cpl_error_reset();
    if (!inframes)
        return;

    dup_inframes = cpl_frameset_duplicate(inframes);

    firstframe = cpl_frameset_get_first(dup_inframes);
    while (firstframe) {
      frames_pair = cpl_frameset_new();
      ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename ( firstframe ), 0, ".*ESO.*", 0 );
      pl = cpl_propertylist_load(cpl_frame_get_filename(firstframe), 0);
      id_frame_ini = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_FRAME_ID);
      cpl_propertylist_delete(pl);
      cpl_frameset_insert(frames_pair, cpl_frame_duplicate(firstframe));
      curframe = cpl_frameset_get_next(dup_inframes);
      paired = FALSE;
      while (curframe) {
          ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename ( curframe ), 0, ".*ESO.*", 0 );
          pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);
          id_frame = cpl_propertylist_get_int(pl, SPH_ZPL_KEYWORD_FRAME_ID);
          //printf( "id_frame_ini = %d, id_frame = %d\n", id_frame_ini, id_frame);
          if (id_frame == id_frame_ini) {
              cpl_frameset_insert(frames_pair, cpl_frame_duplicate(curframe));
              cpl_propertylist_delete(pl);
              paired = TRUE;
              break;
          }
          cpl_propertylist_delete(pl);
          curframe = cpl_frameset_get_next(dup_inframes);
      } // end while


      frames_phase = sph_zpl_common_preproc_sort_frames_phase(frames_pair);
      cpl_frameset_delete(frames_pair);
      frames_pair = NULL;

      if (cpl_error_get_code() != CPL_ERROR_NONE) {
          sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                  SPH_ERROR_ERROR, "CPL error has occured: %d",
                  cpl_error_get_code());
          cpl_frameset_delete(frames_phase);
          frames_phase = NULL;
      }

      // Remove the first frame from the input frameset
      cpl_frameset_erase_frame(dup_inframes, firstframe);
      if (paired) {
        // Keep track of the paired frames
        cpl_frameset_join(paired_frames, frames_phase);
        // Remove the paired frame from the input frameset
        cpl_frameset_erase_frame(dup_inframes, curframe);
      } else {
        // Keep track of the single frames
        cpl_frameset_join(single_frames, frames_phase);
      }

      // The input frameset only contains unprocessed frames, so
      // we always take the first frame.
      firstframe = cpl_frameset_get_first(dup_inframes);
    }

    cpl_msg_info(cpl_func, "paired_frames frameset size: %lld", cpl_frameset_get_size(paired_frames));
    cpl_msg_info(cpl_func, "single_frames frameset size: %lld", cpl_frameset_get_size(single_frames));
}

/*----------------------------------------------------------------------------*/
/**
 @brief    sorts pair-frames in the order of phase.
 @return   a pointer to the cpl_frameset with 2 ordered frames or
 NULL if error is raised.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This function sorts pair-frames (with the same frame ID) in the order
 of phase, i.e. first the frame with phase = ZERO and then with the
 phase = PI are put into a frameset.
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_sort_frames_phase(
        cpl_frameset* inframes) {
    cpl_frame* frame_first = NULL;
    cpl_frame* frame_second = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl_first = NULL;
    cpl_propertylist* pl_second = NULL;
    const char* phase_first;
    const char* phase_second;

    cpl_error_reset();
    if (!inframes)
        return NULL;

    results = cpl_frameset_new();

    if (cpl_frameset_get_size(inframes) == 1) {
        cpl_msg_info(cpl_func, "Handling single frame");
        frame_first = cpl_frameset_get_first(inframes);
        pl_first = cpl_propertylist_load(cpl_frame_get_filename(frame_first),
                0);
        phase_first = cpl_propertylist_get_string(pl_first,
                SPH_ZPL_KEYWORD_PHASE);

        // A single frame should always have phase ZERO, but we check anyway
        if (!strcmp(phase_first, SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO)) {
            cpl_msg_info(cpl_func, "single frame has phase ZERO");
            cpl_frameset_insert(results, cpl_frame_duplicate(frame_first));
        } else if (!strcmp(phase_first, SPH_ZPL_KEYWORD_VALUE_PHASE_PI)) {
            cpl_msg_info(cpl_func, "single frame has phase PI");
            cpl_frameset_insert(results, cpl_frame_duplicate(frame_first));
        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "Phases ID is not correct for this single frame: \n"
                            " phase_first: %s", phase_first);
            cpl_propertylist_delete(pl_first);
            cpl_frameset_delete(results);
            results = NULL;
            return NULL;
        }

        cpl_propertylist_delete(pl_first);
        pl_first = NULL;

    } else if (cpl_frameset_get_size(inframes) == 2) {
        cpl_msg_info(cpl_func, "Handling frame pair");
        frame_first = cpl_frameset_get_first(inframes);
        ////pl_first = cpl_propertylist_load_regexp( cpl_frame_get_filename( frame_first ), 0, ".*ESO.*", 0 );
        pl_first = cpl_propertylist_load(cpl_frame_get_filename(frame_first),
                0);

        frame_second = cpl_frameset_get_next(inframes);
        ////pl_second = cpl_propertylist_load_regexp( cpl_frame_get_filename( frame_second ), 0, ".*ESO.*", 0 );
        pl_second = cpl_propertylist_load(cpl_frame_get_filename(frame_second),
                0);

        if (cpl_propertylist_get_int(pl_first, SPH_ZPL_KEYWORD_FRAME_ID)
                == cpl_propertylist_get_int(pl_second, SPH_ZPL_KEYWORD_FRAME_ID)) {
            phase_first = cpl_propertylist_get_string(pl_first,
                    SPH_ZPL_KEYWORD_PHASE);
            phase_second = cpl_propertylist_get_string(pl_second,
                    SPH_ZPL_KEYWORD_PHASE);

            if (!strcmp(phase_first, SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO)
                    && !strcmp(phase_second, SPH_ZPL_KEYWORD_VALUE_PHASE_PI)) {
                cpl_propertylist_delete(pl_first);
                cpl_propertylist_delete(pl_second);
                cpl_frameset_delete(results);
                results = NULL;
                return cpl_frameset_duplicate(inframes);

            } else if (!strcmp(phase_first, SPH_ZPL_KEYWORD_VALUE_PHASE_PI)
                    && !strcmp(phase_second, SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO)) {
                cpl_frameset_insert(results, cpl_frame_duplicate(frame_second));
                cpl_frameset_insert(results, cpl_frame_duplicate(frame_first));

            } else {
                sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Phases IDs are not correct for this frame-pair: \n"
                                " phase_first: %s\n"
                                "phase_second: %s", phase_first, phase_second);
                cpl_propertylist_delete(pl_first);
                cpl_propertylist_delete(pl_second);
                cpl_frameset_delete(results);
                results = NULL;
                return NULL;
            }

        } else {
            sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR, "IDs of frame are not the same.");
            cpl_propertylist_delete(pl_first);
            cpl_propertylist_delete(pl_second);
            cpl_frameset_delete(results);
            results = NULL;
            return NULL;
        }

        cpl_propertylist_delete(pl_first);
        pl_first = NULL;
        cpl_propertylist_delete(pl_second);
        pl_second = NULL;

    } else {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Not expected number of frames (2): %d ",
                (int) cpl_frameset_get_size(inframes));
        cpl_frameset_delete(results);
        results = NULL;
        return NULL;
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        cpl_frameset_delete(results);
        results = NULL;
        if (pl_first) {
          cpl_propertylist_delete(pl_first);
        }
        if (pl_second) {
          cpl_propertylist_delete(pl_second);
        }
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create cube fits-files for the zimpol camera-1,2
 @return   a pointer to the cpl_frameset of two cube fits-files or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This function create two cubes of the pre-processed zimpol raw data.
 Input cpl_frameset is framesets of separate zimpol exposure fits files for
 camera-1,2
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_cube_zpl_exposures(
        cpl_frameset* zexpframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* frames_ccd = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* pl_copy = NULL;
    //cpl_imagelist*       imagelist       = NULL;
    cpl_image* image = NULL;
    cpl_table* ovsc_table = NULL;
    int ccdID;
    int io, n_ext, extn;
    int single_frame = FALSE;
    int tabsize = 0;

    cpl_error_reset();
    if (!zexpframes)
        return NULL;

    results = cpl_frameset_new();
    for (ccdID = 1; ccdID < 3; ccdID++) {
        frames_ccd = sph_zpl_common_preproc_get_frames_ccd(zexpframes, ccdID);
        if (frames_ccd == NULL) {
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO, "No frames for the ccdID: %d, escape from the loops ", ccdID );
            continue;
        }

        curframe = cpl_frameset_get_first(frames_ccd);
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)),
                SPH_ZPL_TAG_PREPROC_CALIB, CPL_FRAME_GROUP_NONE);
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Name of the pre-processed file: %s",
                cpl_frame_get_filename(frame));

        ovsc_table = cpl_table_new(0);
        cpl_table_new_column(ovsc_table, ADU1_ZERO_OVSC_MEAN, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU1_ZERO_OVSC_RMS, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU2_ZERO_OVSC_MEAN, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU2_ZERO_OVSC_RMS, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU1_PI_OVSC_MEAN, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU1_PI_OVSC_RMS, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU2_PI_OVSC_MEAN, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU2_PI_OVSC_RMS, CPL_TYPE_DOUBLE);

        io = 0;
        while (curframe) {
            n_ext = cpl_fits_count_extensions(cpl_frame_get_filename(curframe));
	    pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), 0);
            // Check if this is a single frame exposure
            single_frame = sph_utils_is_single_frame(pl);

            for (extn = 0; extn <= n_ext; extn++) {
                ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe ), extn, ".*ESO.*", 0);
                pl = cpl_propertylist_load(cpl_frame_get_filename(curframe), extn);

                if (extn == 0) {
                    if (!strcmp(
                            cpl_propertylist_get_string(pl,
                                    SPH_ZPL_KEYWORD_PHASE),
                            SPH_ZPL_KEYWORD_VALUE_PHASE_ZERO)) {
                        SPH_INFO_MSG(
                                "Inserting overscan values of phase ZERO into cpl_table.");
                    } else {
                        SPH_ERROR_RAISE_ERR(
                                SPH_ERROR_INCONSISTENT_INPUT,
                                "Extension is not consistent with phase (0, ZERO):"
                                "(extn = %d, phase = %s)", extn, cpl_propertylist_get_string( pl, SPH_ZPL_KEYWORD_PHASE));
                    }
                    tabsize = cpl_table_get_nrow(ovsc_table);
                    cpl_table_set_size(ovsc_table, tabsize + 1);

                    //note that putting the values of the adu2 into adu1 and vise versa for cam2 is
                    //done here in order to simplify subtraction of the overscans
                    //because cam2 rotated of the 180 degrees
                    if (ccdID == 2) {
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_ZERO_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_ZERO_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_RMS));

                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_ZERO_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_ZERO_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_RMS));
                    } else {
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_ZERO_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_ZERO_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_RMS));

                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_ZERO_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_ZERO_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_RMS));
                    }
                }

                //note that putting the values of the adu2 into adu1 and vise versa for cam2 is
                //done here in order to simplify subtraction of the overscans
                //because cam2 rotated of the 180 degrees
                if (extn == 2) {
                    if (!strcmp(
                            cpl_propertylist_get_string(pl,
                                    SPH_ZPL_KEYWORD_PHASE),
                            SPH_ZPL_KEYWORD_VALUE_PHASE_PI)) {
                        SPH_INFO_MSG(
                                "Inserting overscan values of phase PI into cpl_table.");
                    } else {
                        SPH_ERROR_RAISE_ERR(
                                SPH_ERROR_INCONSISTENT_INPUT,
                                "Extension is not consistent with phase (0, PI):"
                                "(extn = %d, phase = %s)", extn, cpl_propertylist_get_string( pl, SPH_ZPL_KEYWORD_PHASE));
                    }

                    if ( ccdID == 2 ){
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_PI_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_PI_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_RMS));

                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_PI_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_PI_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_RMS));

                    } else {
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_PI_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_PI_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_RMS));

                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_PI_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_PI_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_RMS));
                    }
                }

                image = cpl_image_load(cpl_frame_get_filename(curframe),
                        CPL_TYPE_INT, 0, extn);
                if (io == 0) {
                    pl_copy = cpl_propertylist_new();
                    cpl_propertylist_copy_property_regexp(pl_copy, pl, ".*ZPL",
                            1); //remove temporally created zpl keywords
                    cpl_propertylist_append_string(pl_copy,
                            SPH_COMMON_KEYWORD_SPH_TYPE,
                            SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP);
                    if (single_frame) {
                      // Keep the header that shows whether this cube came from a single frame
                      // (i.e. does not contain valid PI phase data)
                      cpl_propertylist_append_bool(pl_copy, SPH_ZPL_KEYWORD_SINGLE, 1);
                    }
                    sph_cube_append_image(cpl_frame_get_filename(frame), image,
                            pl_copy, extn);
                    cpl_propertylist_delete(pl_copy);
                    io = 1;
                } else {
                    sph_cube_append_image(cpl_frame_get_filename(frame), image,
                            NULL, extn);
                }

                cpl_image_delete(image);
                cpl_propertylist_delete(pl);
                pl = NULL;
            }

            curframe = cpl_frameset_get_next(frames_ccd);
        }SPH_INFO_MSG("Finalising cube...");
        sph_cube_finalise_file(cpl_frame_get_filename(frame));
        cpl_frameset_delete(frames_ccd);
        cpl_frameset_insert(results, frame);
        cpl_table_save(ovsc_table, NULL, NULL, cpl_frame_get_filename(frame),
                CPL_IO_EXTEND);
        cpl_table_delete(ovsc_table);

    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        cpl_frameset_delete(results);
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Create cube fits-files for the zimpol data exposure imaging camera-1,2
 @return   a pointer to the cpl_frameset of two cube fits-files or NULL
 if unsuccessfull.
 @note     A SPH and/or CPL error is raised in case there was a problem.

 This function create two cubes of the pre-processed zimpol imaging raw data.
 Input cpl_frameset is framesets of separate zimpol exposure fits files for
 camera-1,2
 */
/*----------------------------------------------------------------------------*/
static cpl_frameset* sph_zpl_common_preproc_cube_zpl_exposures_imaging(
        cpl_frameset* zexpframes) {
    cpl_frame* curframe = NULL;
    cpl_frame* frame = NULL;
    cpl_frameset* frames_ccd = NULL;
    cpl_frameset* results = NULL;
    cpl_propertylist* pl = NULL;
    cpl_propertylist* pl_copy = NULL;
    //cpl_imagelist*        imagelist       = NULL;
    cpl_image* image = NULL;
    cpl_table* ovsc_table = NULL;
    int ccdID, io, extn;
    int tabsize = 0;

    cpl_error_reset();
    if (!zexpframes)
        return NULL;

    results = cpl_frameset_new();
    for (ccdID = 1; ccdID < 3; ccdID++) {
        frames_ccd = sph_zpl_common_preproc_get_frames_ccd(zexpframes, ccdID);

        if (frames_ccd == NULL) {
            SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO, "No frames for the ccdID: %d, escape from the loops ", ccdID );
            continue;
        }

        curframe = cpl_frameset_get_first(frames_ccd);
        frame = sph_filemanager_create_temp_frame(
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)),
                SPH_ZPL_TAG_PREPROC_IMAGING_CALIB, CPL_FRAME_GROUP_NONE);
        sph_error_raise(SPH_ERROR_INFO, __FILE__, __func__, __LINE__,
                SPH_ERROR_INFO, "Name of the pre-processed file: %s",
                cpl_frame_get_filename(frame));

        ovsc_table = cpl_table_new(0);
        cpl_table_new_column(ovsc_table, ADU1_OVSC_MEAN, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU1_OVSC_RMS, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU2_OVSC_MEAN, CPL_TYPE_DOUBLE);
        cpl_table_new_column(ovsc_table, ADU2_OVSC_RMS, CPL_TYPE_DOUBLE);

        io = 0;
        while (curframe) {
            for (extn = 0; extn < 2; extn++) {
                ////pl = cpl_propertylist_load_regexp( cpl_frame_get_filename( curframe ), extn, ".*ESO.*", 0);
                pl = cpl_propertylist_load(cpl_frame_get_filename(curframe),
                        extn);
                if (extn == 0) {
                    // write the mean and rms of the over-scan ADU1, ADU2 image region into the cpl_table (imaging mode)
                    tabsize = cpl_table_get_nrow(ovsc_table);
                    cpl_table_set_size(ovsc_table, tabsize + 1);

                    if ( ccdID == 2 ){
                        //note that putting the values of the adu2 into adu1 and vise versa for cam2 is
                        //done here in order to simplify subtraction of the overscans
                        //because cam2 rotated of the 180 degrees
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_RMS));

                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_RMS));

                    } else {
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU1_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU1_OVSC_RMS));

                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_OVSC_MEAN,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_MEAN));
                        cpl_table_set_double(
                                ovsc_table,
                                ADU2_OVSC_RMS,
                                tabsize,
                                cpl_propertylist_get_double(pl,
                                        SPH_ZPL_KEYWORD_ADU2_OVSC_RMS));
                    }
                }

                image = cpl_image_load(cpl_frame_get_filename(curframe),
                        CPL_TYPE_INT, 0, extn);
                if (io == 0) {
                    pl_copy = cpl_propertylist_new();
                    cpl_propertylist_copy_property_regexp(pl_copy, pl, ".*ZPL",
                            1); //remove temporally created zpl keywords
                    cpl_propertylist_append_string(
                            pl_copy,
                            SPH_COMMON_KEYWORD_SPH_TYPE,
                            SPH_COMMON_KEYWORD_VALUE_SPH_TYPE_PREPROC_ZPL_EXP_IMG);
                    sph_cube_append_image(cpl_frame_get_filename(frame), image,
                            pl_copy, extn);
                    cpl_propertylist_delete(pl_copy);
                    io = 1;
                } else {
                    sph_cube_append_image(cpl_frame_get_filename(frame), image,
                            NULL, extn);
                }
                cpl_image_delete(image);
                cpl_propertylist_delete(pl);
                pl = NULL;
            }

            curframe = cpl_frameset_get_next(frames_ccd);
        }SPH_INFO_MSG("Finalising cube...");
        sph_cube_finalise_file(cpl_frame_get_filename(frame));
        cpl_frameset_insert(results, frame);
        cpl_frameset_delete(frames_ccd);
        cpl_table_save(ovsc_table, NULL, NULL, cpl_frame_get_filename(frame),
                CPL_IO_EXTEND);
        cpl_table_delete(ovsc_table);
    }

    if (cpl_error_get_code() != CPL_ERROR_NONE) {
        sph_error_raise(SPH_ERROR_ERROR, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "CPL error has occured: %d",
                cpl_error_get_code());
        cpl_frameset_delete(results);
        return NULL;
    }

    return results;
}

/*----------------------------------------------------------------------------*/
/**
 @brief    Computes and writes the bias & RON into the passed-in header
 @return   Nothing
 @note     Assumes the passed-in frameset contains raw frames in either the
 old or the new format.

 This function computes and writes the bias & RON of the rawframes into
 the header.
 */
/*----------------------------------------------------------------------------*/
void sph_zpl_common_preproc_update_header_with_raw_stats(
        cpl_frameset *rawset, cpl_propertylist *pl){

    // exit early if inputs fail some basic checks
    if (!rawset || !pl || 1 != cpl_frameset_get_size(rawset)) return;
    cpl_frame *rawframe = cpl_frameset_get_first(rawset);
    int format = sph_zpl_utils_check_format_frame(rawframe);
    if (format != 0) return;  // only support old format for now
    int cam = cpl_propertylist_get_int(pl, SPH_COMMOM_KEYWORD_CHIP_INDEX);
    const char *raw_fname = cpl_frame_get_filename(rawframe);
    cpl_propertylist *camera_pl = cpl_propertylist_load(raw_fname, cam);
    if (!camera_pl) return;
    if (!cpl_propertylist_has(camera_pl, "NAXIS3")) return;
    cpl_size nplanes = cpl_propertylist_get_long_long(camera_pl, "NAXIS3");
    if (nplanes < 1) return;
    cpl_propertylist_delete(camera_pl);

    // could do something like this if we want to support both formats
    //cpl_frameset* (*extract)(const cpl_frame *inframe);
    //if (0 == format) extract = sph_zpl_common_preproc_extract_frames;
    //if (1 == format) extract = sph_zpl_common_preproc_extract_frames_newformat;
    //cpl_frameset *extracted = extract(rawframe);

    // compute the bias
    sph_master_frame *bias_comb =
        sph_framecombination_new_master_frame_from_cpl_frame_multi_ext(
            rawframe, cam, SPH_COLL_ALG_MEAN, NULL);
    cpl_image *biasimg = sph_master_frame_extract_image(bias_comb, 0);
    double bias = cpl_image_get_mean_window(biasimg, 60, 200, 260, 900);
    cpl_image_delete(biasimg);
    sph_master_frame_delete(bias_comb);

    // compute the ron
    cpl_imagelist *slices = cpl_imagelist_load_window(raw_fname,
        CPL_TYPE_DOUBLE, cam, 300, 100, 850, 900);  // ints fail in combine
    if (cpl_imagelist_get_size(slices) != nplanes) return;  // sanity check
    cpl_vector *ron_vect = cpl_vector_new(nplanes);
    cpl_vector_fill(ron_vect, -1.0);
    nplanes = fmin(5, nplanes);
    for (cpl_size i=0; i<nplanes; i++) cpl_vector_set(ron_vect, i, 0.0);
    cpl_imagelist_erase(slices, ron_vect);
    sph_master_frame *ron_comb = sph_framecombination_master_frame_new_collapse(
        slices, SPH_COLL_ALG_MEAN, NULL);
    double ron = sph_master_frame_get_mean_rms(ron_comb, NULL);
    sph_master_frame_delete(ron_comb);
    cpl_vector_delete(ron_vect);
    cpl_imagelist_delete(slices);

    // update the header
    cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_BIAS, bias);
    cpl_propertylist_update_double(pl, SPH_COMMON_KEYWORD_QC_RON, ron);
}

