/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_zpl_science_p23.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_exposure.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_zpl_utils.h"
#include "sph_zpl_framegroups.h"
#include "sph_quad_image.h"
#include "sph_triple_image.h"
#include "sph_double_image.h"
#include "sph_master_frame.h"
#include "sph_keywordnode.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_fits.h"
#include "sph_common_keywords.h"
#include "sph_common_science.h"
#include "sph_filemanager.h"
#include "sph_transform.h"
#include "sph_zpl_subtract_dark_scaled.h"

#include <string.h>
#include <math.h>

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_science_p23_run Reduce science frames of the Q and U observations for the P2 and P3 polarization modes.
 *
 * This module provides the algorithm implementation for the creation of the
 * reduced polarimetric science observations (in P2 and P3 modes) and the creation of the corresponding
 * mueller matrix elements (not implemented yet)
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_science_p23_run.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

#define  SPH_ZPL_STOKES_Q                    0
#define  SPH_ZPL_STOKES_U                    1
#define  SPH_ZPL_STOKES_NONE                2

#define  SPH_SAVE_NONDFS                    0

/*-------------------------------------------------------------------------------
 * The Structure Definition
 *
 * This structure contains the members of the sph_zpl_science_p23_cunit that are
 * to reduce data for the one zimpol channel (camera). This structure is used as
 * an interface between a sphere "standard" recipe structure and re-designed
 * zimpol "_run" function which treats both zimpol channel.
 *
 * ------------------------------------------------------------------------------
 */
typedef struct _sph_zpl_science_p23_cunit {
    cpl_frameset* inframes; /* The recipe input frames */

    cpl_frameset* current_raw_frameset; /* The frameset for one product */

    cpl_parameterlist* inparams; /* The recipe input parameters */

    cpl_parameterlist* framecomb_parameterlist; /* The recipe input parameters */

    char* science_p23_outfilename; /* The parameter of zpl.science_p23.outfilename_q_cam1 */

    char* science_p23_plus_outfilename; /* The parameter of zpl.science_p23_plus_q_cam1.outfilename */

    char* science_p23_minus_outfilename; /* The parameter of zpl.science_p23_minus_q_cam1.outfilename */

    short subtract_overscan; /* The parameter of zpl.science_p23.subtract_overscan */

    short keep_intermediate; /* The parameter of zpl.science_p23.keep_intermediate */

    short save_interprod; /* The parameter of zpl.science_p23.save_interprod */

    int coll_alg; /* The parameter of zpl.science_p23.coll_alg */

    double filter_radius; /* The parameter of zpl.science_p23.filter_radius */

    double center_xoffset; /* The parameter of zpl.science_p23.center_xoffset_cam1 or _cam1 */

    double center_yoffset; /* The parameter of zpl.science_p23.center_yoffset_cam1 or cam2 */

    short star_center_iframe; /* The parameter of zpl.science_p23.star_center_iframe */

    cpl_frameset* rawframes; /* The ZPL_SCIENCE_P23_RAW frames */

    cpl_frameset* preproc_frames; /* The ZPL_SCIENCE_P23_PREPROC_CAM1 frames */

    cpl_frame* master_bias_frame; /* The ZPL_MASTER_BIAS_CAM1 frames */

    cpl_frame* master_dark_frame; /* The ZPL_MASTER_DARK_CAM1 frames */

    cpl_frame* intensity_flat_frame; /* The ZPL_INT_FLAT_FIELD_CAM1 frames */

    cpl_frame* intensity_flat_frame_master; /* The ZPL_INT_FLAT_FIELD_MASTER_CAM1 frames */

    cpl_frame* polarization_flat_frame; /* The ZPL_POL_FLAT_FIELD_CAM1 frames */

    cpl_frame* modem_efficiency_frame; /* SPH_ZPL_TAG_MODEM_EFF_CALIB_CAM1 or _CAM2 */

    cpl_frameset* fctable_frames; /* The ZPL_CENTER_TABLE frames */

    cpl_frame*  star_center_frame; /* The ZPL_STAR_CENTER_POL frames */

    sph_double_image* science_di; /* The ZPL_SCIENCE_P23_REDUCED product */

    sph_double_image* science_di_plus; /* The ZPL_SCIENCE_P23_PLUS_REDUCED product */

    sph_double_image* science_di_minus; /* The ZPL_SCIENCE_P23_MINUS_REDUCED product */

    char* eso_pro_catg;

    char* eso_pro_catg_plus;

    char* eso_pro_catg_minus;

    sph_zpl_framegroups* fgs;

    sph_zpl_framegroups*  fgsraw;

//  sph_triple_image*               science_ti                            ; /* The ZPL_SCIENCE_XMATRIX_ELEMENTS product */

} sph_zpl_science_p23_cunit;

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static sph_zpl_science_p23_cunit* _sph_zpl_science_p23_cunit_create(
        sph_zpl_science_p23* self, int camera, short int stoke)
    CPL_ATTR_ALLOC;
static sph_error_code _sph_zpl_science_p23_cunit_delete(
        sph_zpl_science_p23_cunit* cunit);
static cpl_error_code _sph_zpl_science_p23_cunit_run(
        sph_zpl_science_p23_cunit* cunit);
//static cpl_error_code _sph_zpl_science_p23_create_fctable_frames ()

static cpl_frame* sph_zpl_science_p23_calibrate_transform_frame__(
        const char* filename, cpl_frameset* inframes,
        const sph_fctable* fctable,
        const sph_quad_image* master_bias_quadimage,
        const sph_quad_image* master_dark_quadimage,
        const sph_quad_image* intensity_flat_quadimage,
        const sph_master_frame* intensity_flat_masterframe,
        const sph_double_image* polarization_flat_doubleimage,
        const sph_master_frame* modem_efficiency_masterframe)
    CPL_ATTR_ALLOC;

static cpl_frameset* sph_zpl_science_p23_calibrate_frames__(
        sph_zpl_science_p23_cunit* self, const cpl_frameset* inframes,
        const sph_quad_image* master_bias_quadimage,
        const sph_quad_image* master_dark_quadimage,
        const sph_quad_image* intensity_flat_quadimage,
        const sph_master_frame* intensity_flat_masterframe,
        const sph_double_image* polarization_flat_doubleimage,
        const sph_master_frame* modem_efficiency_masterframe)
    CPL_ATTR_ALLOC;


/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_science_p23 recipe
 (polarimetric mode). The error  code returned is always a cpl error code
 (to allow maximal compatibility with  esorex, gasgano, etc.) even if during
 recipe execution an error in the SPHERE API is the cause. In this case
 (and if the underlying error is not a cpl error)  the cpl error code is set
 to the cpl_error_code that matches the failure reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code sph_zpl_science_p23_run(sph_zpl_science_p23* self) {
    sph_zpl_science_p23_cunit* cunit_q = NULL;
    sph_zpl_science_p23_cunit* cunit_u = NULL;
    cpl_error_code recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_science_p23_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return cpl_error_get_code();
    }

/*
    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL) {

        if (self->rawframes) {
            self->preproc_frames_cam1 = cpl_frameset_new();
            self->preproc_frames_cam2 = cpl_frameset_new();
            SPH_INFO_MSG("Start pre-processing...")
            //recipe_error = sph_zpl_science_p23_preproc( self );
            recipe_error = sph_zpl_common_preproc_recipe(self->rawframes,
                    self->outfilename_cam1, self->outfilename_cam2,
                    self->preproc_frames_cam1, self->preproc_frames_cam2);
        } else {
            SPH_ERR(
                    "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
            return sph_error_get_last_code();
        }
        if (recipe_error) {
            SPH_ERR("Pre-processing step is failed.");
            cpl_frameset_delete(self->preproc_frames_cam1);
            self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2);
            self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }

    }
*/

    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL && self->preproc_frames == NULL) {

         if (self->rawframes) {
             self->preproc_frames_cam1 = cpl_frameset_new();
             self->preproc_frames_cam2 = cpl_frameset_new();
             SPH_INFO_MSG("Start pre-processing...")
             //recipe_error = sph_zpl_master_dark_preproc( self );
             recipe_error = sph_zpl_common_preproc_recipe(self->rawframes,
                     self->outfilename_cam1, self->outfilename_cam2,
                     self->preproc_frames_cam1, self->preproc_frames_cam2);
         } else {
             SPH_ERR(
                     "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
             return sph_error_get_last_code();
         }
         if (recipe_error) {
             SPH_ERR("Pre-processing step is failed.");
             cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
             cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
             return (int) recipe_error;
         }


         if ( sph_zpl_utils_check_format( self->rawframes ) == 1) {
             SPH_INFO_MSG("New style format of the raw data...")
             if ( cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 ) ) {
                 self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam2 );
                 cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                 cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
             } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                 self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam1 );
                 cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                 cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
             } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                 SPH_WARNING("Rawdata set contains fits files from camera-1 and camera-2. Please, make sure that you provide calibrations for both cameras."
                         "The products will be created individually for each camera (off-line pipeline extended support!")
             } else {
                 SPH_ERR("Pre-processing step is failed.");
                 cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
                 cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
                 return (int) recipe_error;
             }
         } else if ( sph_zpl_utils_check_format( self->rawframes ) == 0) {
             SPH_INFO_MSG("Old style format of the raw data...")
         } else {
             SPH_ERR("Not supported raw data format or new format and old format are mixed in the input dataset. "
                     "Please, make sure that you use either old or new style of the rawdata format! Stop recipe...");
             cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
             cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
             return (int) recipe_error;
         }
     }

    sph_error_reset();
    SPH_RAISE_CPL_RESET; //reset of the error system after pre-processing

    SPH_INFO_MSG("*********** [AFTER PREPROC] ***********");
    //reduce data for the camera-1
    //create cunit structure
    if (self->preproc_frames_cam1) {
        SPH_INFO_MSG("Create cunit structures for the camera-1...");

        SPH_INFO_MSG("Camera-1 stokes Q...")
        cunit_q = _sph_zpl_science_p23_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID, SPH_ZPL_STOKES_Q);
        if (cunit_q) {

            recipe_error = _sph_zpl_science_p23_cunit_run(cunit_q);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-1 for the stokes Q is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_q->science_di)
                    self->science_di_q_cam1 = sph_double_image_duplicate(
                            cunit_q->science_di);
                if (cunit_q->science_di_minus)
                    self->science_di_q_minus_cam1 = sph_double_image_duplicate(
                            cunit_q->science_di_minus);
                if (cunit_q->science_di_plus)
                    self->science_di_q_plus_cam1 = sph_double_image_duplicate(
                            cunit_q->science_di_plus);
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_q structure for the camera-1 (stokes Q) is failed...trying for the stokes U")
        }

        SPH_INFO_MSG("Camera-1 stokes U...")
        //reset an error log system to reduce data for the U frames of the camera-1
        sph_error_reset();
        SPH_RAISE_CPL_RESET;
        cunit_u = _sph_zpl_science_p23_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID, SPH_ZPL_STOKES_U);
        if (cunit_u) {

            recipe_error = _sph_zpl_science_p23_cunit_run(cunit_u);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-1 for the stokes U is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_u->science_di)
                    self->science_di_u_cam1 = sph_double_image_duplicate(
                            cunit_u->science_di);
                if (cunit_u->science_di_minus)
                    self->science_di_u_minus_cam1 = sph_double_image_duplicate(
                            cunit_u->science_di_minus);
                if (cunit_u->science_di_plus)
                    self->science_di_u_plus_cam1 = sph_double_image_duplicate(
                            cunit_u->science_di_plus);
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_u structure for the camera-1 (stokes U) is failed!")
        }

        if (cunit_u == NULL && cunit_q == NULL) {
            SPH_INFO_MSG(
                    "I will try to create cunit, considering all input frames from the camera-1 as Qplus frames...");
            //reset an error log system to set up all frames as Q frames of the camera-1
            sph_error_reset();
            SPH_RAISE_CPL_RESET;
            cunit_q = _sph_zpl_science_p23_cunit_create(self,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID, SPH_ZPL_STOKES_NONE);

            if (cunit_q) {

                recipe_error = _sph_zpl_science_p23_cunit_run(cunit_q);
                if (recipe_error != CPL_ERROR_NONE) {
                    SPH_ERR(
                            "reduction for the camera-1 for the stokes Q is failed anyway!");
                } else {
                    //put the output product in the main self structure
                    if (cunit_q->science_di)
                        self->science_di_q_cam1 = sph_double_image_duplicate(
                                cunit_q->science_di);
                    if (cunit_q->science_di_minus)
                        self->science_di_q_minus_cam1 =
                                sph_double_image_duplicate(
                                        cunit_q->science_di_minus);
                    if (cunit_q->science_di_plus)
                        self->science_di_q_plus_cam1 =
                                sph_double_image_duplicate(
                                        cunit_q->science_di_plus);
                }
            } else {
                SPH_WARNING(
                        "Creation of the cunit_q structure for the camera-1 (stokes Q) is failed anyway...trying camera-2")
            }
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-1 -> nothing to reduce..trying for the camera-2");
    }
    _sph_zpl_science_p23_cunit_delete(cunit_q);
    cunit_q = NULL;
    _sph_zpl_science_p23_cunit_delete(cunit_u);
    cunit_u = NULL;

    //reset an error log system to reduce data for camera-2
    sph_error_reset();
    SPH_RAISE_CPL_RESET;

    //reduce data for the camera-2
    //create cunit structure
    if (self->preproc_frames_cam2) {
        SPH_INFO_MSG("Create cunit structure for the camera-2...");

        SPH_INFO_MSG("Camera-2 stokes Q...")
        cunit_q = _sph_zpl_science_p23_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID, SPH_ZPL_STOKES_Q);

        if (cunit_q) {
            recipe_error = _sph_zpl_science_p23_cunit_run(cunit_q);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-2 is for the stokes Q is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_q->science_di)
                    self->science_di_q_cam2 = sph_double_image_duplicate(
                            cunit_q->science_di);
                if (cunit_q->science_di_minus)
                    self->science_di_q_minus_cam2 = sph_double_image_duplicate(
                            cunit_q->science_di_minus);
                if (cunit_q->science_di_plus)
                    self->science_di_q_plus_cam2 = sph_double_image_duplicate(
                            cunit_q->science_di_plus);

            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_q structure for the camera-2 (stokes Q) is failed...trying for the stokes U")
        }

        SPH_INFO_MSG("Camera-2 stokes U...")
        //reset an error log system to reduce data for U frames of the camera-2
        sph_error_reset();
        SPH_RAISE_CPL_RESET;
        cunit_u = _sph_zpl_science_p23_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID, SPH_ZPL_STOKES_U);

        if (cunit_u) {
            recipe_error = _sph_zpl_science_p23_cunit_run(cunit_u);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-2 is for the stokes U is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit_u->science_di)
                    self->science_di_u_cam2 = sph_double_image_duplicate(
                            cunit_u->science_di);
                if (cunit_u->science_di_minus)
                    self->science_di_u_minus_cam2 = sph_double_image_duplicate(
                            cunit_u->science_di_minus);
                if (cunit_u->science_di_plus)
                    self->science_di_u_plus_cam2 = sph_double_image_duplicate(
                            cunit_u->science_di_plus);
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit_u structure for the camera-2 (stokes U) is failed!")
        }

        if (cunit_q == NULL && cunit_u == NULL) {
            SPH_INFO_MSG(
                    "I will try to create cunit, considering all input frames from the camera-2 as Qplus frames...");
            //reset an error log system to set up all frames as Q frames of the camera-2
            sph_error_reset();
            SPH_RAISE_CPL_RESET;

            cunit_q = _sph_zpl_science_p23_cunit_create(self,
                    SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID, SPH_ZPL_STOKES_NONE);
            if (cunit_q) {

                recipe_error = _sph_zpl_science_p23_cunit_run(cunit_q);
                if (recipe_error != CPL_ERROR_NONE) {
                    SPH_ERR(
                            "reduction for the camera-2 for the stokes Q is failed anyway!");
                } else {
                    //put the output product in the main self structure
                    if (cunit_q->science_di)
                        self->science_di_q_cam2 = sph_double_image_duplicate(
                                cunit_q->science_di);
                    if (cunit_q->science_di_minus)
                        self->science_di_q_minus_cam2 =
                                sph_double_image_duplicate(
                                        cunit_q->science_di_minus);
                    if (cunit_q->science_di_plus)
                        self->science_di_q_plus_cam2 =
                                sph_double_image_duplicate(
                                        cunit_q->science_di_plus);
                }
            } else {
                SPH_WARNING(
                        "Creation of the cunit_q structure for the camera-1 (stokes Q) is failed anyway...trying camera-2")
            }
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-2 -> nothing to reduce.");
    }

    _sph_zpl_science_p23_cunit_delete(cunit_q);
    _sph_zpl_science_p23_cunit_delete(cunit_u);

    //reset the error log system to avoid a recipe abort in the case if some products were not created
    sph_error_reset();
    SPH_RAISE_CPL_RESET;

    //post-condistions
    if ((self->science_di_q_cam1 == NULL && self->science_di_q_cam2 == NULL
            && self->science_di_q_minus_cam1 == NULL
            && self->science_di_q_minus_cam2 == NULL
            && self->science_di_q_plus_cam1 == NULL
            && self->science_di_q_plus_cam2 == NULL
            && self->science_di_u_cam1 == NULL && self->science_di_u_cam2
            && self->science_di_u_minus_cam1 == NULL
            && self->science_di_u_minus_cam2
            && self->science_di_u_plus_cam1 == NULL
            && self->science_di_u_plus_cam2)) {
        SPH_ERR(" No reduced frame outputs have been created !!!");
    }

    if (!self->keep_intermediate && self->rawframes != NULL) {
        SPH_INFO_MSG("Unliking intermediate data!");
        if (self->preproc_frames_cam1) sph_utils_frames_unlink(self->preproc_frames_cam1);
        if (self->preproc_frames_cam2) sph_utils_frames_unlink(self->preproc_frames_cam2);
        //if (self->preproc_frames) sph_utils_frames_unlink(self->preproc_frames);
    }

    SPH_INFO_MSG("sph_zpl_science_p23_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

sph_zpl_science_p23_cunit* _sph_zpl_science_p23_cunit_create(
        sph_zpl_science_p23* self, int camera_id, short int stoke) {
    sph_zpl_science_p23_cunit* result = NULL;
//    sph_zpl_framegroups*            fgs            = NULL;
    sph_error_code rerr = CPL_ERROR_NONE;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return NULL;
    }

    result = cpl_calloc(1, sizeof(sph_zpl_science_p23_cunit));

    if (result == NULL) {
        SPH_ERR( "Could not allocate the structure for run unit.");
        return result;
    }

    //copy and duplicate some elements of the initial "sph_zpl_science_p23" structure into the cunit structure
    result->inframes = self->inframes;
    result->inparams = self->inparams;
    result->framecomb_parameterlist = self->framecomb_parameterlist;
    result->subtract_overscan = self->subtract_overscan;
    result->save_interprod = self->save_interprod;
    result->keep_intermediate = self->keep_intermediate;
    result->coll_alg = self->coll_alg;
    result->rawframes = self->rawframes;
    result->star_center_iframe = self->star_center_iframe;

    result->fctable_frames = self->fctable_frames;

    if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {

        result->center_xoffset = self->center_xoffset_cam1;
        result->center_yoffset = self->center_yoffset_cam1;


        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam1);
        if (stoke == SPH_ZPL_STOKES_Q) {
            result->fgs = sph_zpl_framegroups_create_Q(result->preproc_frames);
            if (!result->fgs) {
                SPH_WARNING(
                        "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create returns NULL!");
                _sph_zpl_science_p23_cunit_delete(result);
                result = NULL;
                return result;
            } else {
                SPH_INFO_MSG("CAM1 Q stokes: fgs is not NULL");
                result->science_p23_outfilename = cpl_strdup(
                        self->science_p23_outfilename_q_cam1);
                result->science_p23_plus_outfilename = cpl_strdup(
                        self->science_p23_plus_q_outfilename_cam1);
                result->science_p23_minus_outfilename = cpl_strdup(
                        self->science_p23_minus_q_outfilename_cam1);
                result->eso_pro_catg = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_Q_CALIB_CAM1); //TAG for the double image science product CAM1
                result->eso_pro_catg_plus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_QPLUS_CALIB_CAM1); //TAG for the double image science plus product CAM1
                result->eso_pro_catg_minus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_QMINUS_CALIB_CAM1); //TAG for the double image science plus product CAM1

                //group raw frames on QPLUS and QMINUS, which is needed for save_dfs
                result->fgsraw = sph_zpl_framegroups_create_Q(self->rawframes);
                if (!result->fgsraw) {
                     SPH_WARNING(
                             "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create_Q raw returns NULL!");
                     _sph_zpl_science_p23_cunit_delete(result);
                     result = NULL;
                     return result;
                }

            }
        } else if (stoke == SPH_ZPL_STOKES_U) {
            result->fgs = sph_zpl_framegroups_create_U(result->preproc_frames);
            if (!result->fgs) {
                SPH_WARNING(
                        "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create returns NULL!");
                _sph_zpl_science_p23_cunit_delete(result);
                result = NULL;
                return result;
            } else {
                SPH_INFO_MSG("CAM1 U stokes: fgs is not NULL");
                result->science_p23_outfilename = cpl_strdup(
                        self->science_p23_outfilename_u_cam1);
                result->science_p23_plus_outfilename = cpl_strdup(
                        self->science_p23_plus_u_outfilename_cam1);
                result->science_p23_minus_outfilename = cpl_strdup(
                        self->science_p23_minus_u_outfilename_cam1);
                result->eso_pro_catg = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_U_CALIB_CAM1); //TAG for the double image science product CAM1
                result->eso_pro_catg_plus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_UPLUS_CALIB_CAM1); //TAG for the double image science plus product CAM1
                result->eso_pro_catg_minus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_UMINUS_CALIB_CAM1); //TAG for the double image science plus product CAM1

                //group raw frames on UPLUS and UMINUS, which is needed for save_dfs
                result->fgsraw = sph_zpl_framegroups_create_U(self->rawframes);
                if (!result->fgsraw) {
                     SPH_WARNING(
                             "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create_U raw returns NULL!");
                     _sph_zpl_science_p23_cunit_delete(result);
                     result = NULL;
                     return result;
                }

            }
        } else { //set all to the Q stokes (usually if stoke == SPH_ZPL_STOKES_NONE )
            SPH_INFO_MSG("CAM1: Setting all frames as Qplus frames.")
            result->fgs = sph_zpl_framegroups_new();
            rerr = sph_zpl_framegroups_set_framesplus(result->fgs,
                    self->preproc_frames_cam1);
            if (rerr != CPL_ERROR_NONE) {
                SPH_ERR(
                        "Can't set plus frames into the framegroups, sph_zpl_framegroups_frames returns NULL!");
                _sph_zpl_science_p23_cunit_delete(result);
                result = NULL;
                return result;
            }
            result->science_p23_outfilename = cpl_strdup(
                    self->science_p23_outfilename_q_cam1);
            result->science_p23_plus_outfilename = cpl_strdup(
                    self->science_p23_plus_q_outfilename_cam1);
            result->science_p23_minus_outfilename = cpl_strdup(
                    self->science_p23_minus_q_outfilename_cam1);
            result->eso_pro_catg = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P23_Q_CALIB_CAM1); //TAG for the double image science product CAM1
            result->eso_pro_catg_plus = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P23_QPLUS_CALIB_CAM1); //TAG for the double image science plus product CAM1
            result->eso_pro_catg_minus = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P23_QMINUS_CALIB_CAM1); //TAG for the double image science plus product CAM1


            //group raw frames on QPLUS and QMINUS, which is needed for save_dfs
            result->fgsraw = sph_zpl_framegroups_create_Q(self->rawframes);
            if (!result->fgsraw) {
                 SPH_WARNING(
                         "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create_U raw returns NULL!");
                 _sph_zpl_science_p23_cunit_delete(result);
                 result = NULL;
                 return result;
            }


        }

        if (self->master_bias_frame_cam1) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam1);
        }
        if (self->master_dark_frame_cam1) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam1);
        }
        if (self->intensity_flat_frame_cam1) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame_cam1);
        }
        if (self->intensity_flat_frame_master_cam1) {
            result->intensity_flat_frame_master = cpl_frame_duplicate(
                    self->intensity_flat_frame_master_cam1);
        }
        if (self->polarization_flat_frame_cam1) {
            result->polarization_flat_frame = cpl_frame_duplicate(
                    self->polarization_flat_frame_cam1);
        }
        if (self->modem_efficiency_frame_cam1) {
            result->modem_efficiency_frame = cpl_frame_duplicate(
                    self->modem_efficiency_frame_cam1);
        }
        if (self->star_center_frame_cam1) {
            result->star_center_frame = cpl_frame_duplicate(
                    self->star_center_frame_cam1);
        }


    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {

        result->center_xoffset = self->center_xoffset_cam1;
        result->center_yoffset = self->center_yoffset_cam1;


        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam2);
        if (stoke == SPH_ZPL_STOKES_Q) {
            result->fgs = sph_zpl_framegroups_create_Q(result->preproc_frames);
            if (!result->fgs) {
                SPH_WARNING(
                        "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create returns NULL!");
                _sph_zpl_science_p23_cunit_delete(result);
                result = NULL;
                return result;
            } else {
                SPH_INFO_MSG("CAM2: Q stokes: fgs is not NULL");
                result->science_p23_outfilename = cpl_strdup(
                        self->science_p23_outfilename_q_cam2);
                result->science_p23_plus_outfilename = cpl_strdup(
                        self->science_p23_plus_q_outfilename_cam2);
                result->science_p23_minus_outfilename = cpl_strdup(
                        self->science_p23_minus_q_outfilename_cam2);
                result->eso_pro_catg = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_Q_CALIB_CAM2); //TAG for the double image science product CAM1
                result->eso_pro_catg_plus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_QPLUS_CALIB_CAM2); //TAG for the double image science plus product CAM1
                result->eso_pro_catg_minus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_QMINUS_CALIB_CAM2); //TAG for the double image science plus product CAM1


                //group raw frames on UPLUS and UMINUS, which is needed for save_dfs
                result->fgsraw = sph_zpl_framegroups_create_Q(self->rawframes);
                if (!result->fgsraw) {
                     SPH_WARNING(
                             "Neither Qplus nor Qminus frames found, sph_zpl_framegroups_create_Q raw returns NULL!");
                     _sph_zpl_science_p23_cunit_delete(result);
                     result = NULL;
                     return result;
                }

            }
        } else if (stoke == SPH_ZPL_STOKES_U) {
            result->fgs = sph_zpl_framegroups_create_U(result->preproc_frames);
            if (!result->fgs) {
                SPH_WARNING(
                        "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create returns NULL!");
                _sph_zpl_science_p23_cunit_delete(result);
                result = NULL;
                return result;
            } else {
                SPH_INFO_MSG("CAM2 U stokes: fgs is not NULL");
                result->science_p23_outfilename = cpl_strdup(
                        self->science_p23_outfilename_u_cam2);
                result->science_p23_plus_outfilename = cpl_strdup(
                        self->science_p23_plus_u_outfilename_cam2);
                result->science_p23_minus_outfilename = cpl_strdup(
                        self->science_p23_minus_u_outfilename_cam2);
                result->eso_pro_catg = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_U_CALIB_CAM2); //TAG for the double image science product CAM1
                result->eso_pro_catg_plus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_UPLUS_CALIB_CAM2); //TAG for the double image science plus product CAM1
                result->eso_pro_catg_minus = cpl_strdup(
                        SPH_ZPL_TAG_SCIENCE_P23_UMINUS_CALIB_CAM2); //TAG for the double image science plus product CAM1

                //group raw frames on UPLUS and UMINUS, which is needed for save_dfs
                result->fgsraw = sph_zpl_framegroups_create_U(self->rawframes);

                if (!result->fgsraw) {
                     SPH_WARNING(
                             "Neither Uplus nor Uminus frames found, sph_zpl_framegroups_create_U raw returns NULL!");
                     _sph_zpl_science_p23_cunit_delete(result);
                     result = NULL;
                     return result;
                }
            }
        } else { //set all to the Q stokes
            result->fgs = sph_zpl_framegroups_new();
            rerr = sph_zpl_framegroups_set_framesplus(result->fgs,
                    self->preproc_frames_cam2);
            if (rerr != CPL_ERROR_NONE) {
                SPH_ERR(
                        "Can't set plus frames into the framegroups, sph_zpl_framegroups_frames returns NULL!");
                _sph_zpl_science_p23_cunit_delete(result);
                result = NULL;
                return result;
            }SPH_INFO_MSG("CAM2: Setting all frames as Qplus frames.")
            result->science_p23_outfilename = cpl_strdup(
                    self->science_p23_outfilename_q_cam2);
            result->science_p23_plus_outfilename = cpl_strdup(
                    self->science_p23_plus_q_outfilename_cam2);
            result->science_p23_minus_outfilename = cpl_strdup(
                    self->science_p23_minus_q_outfilename_cam2);
            result->eso_pro_catg = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P23_Q_CALIB_CAM2); //TAG for the double image science product CAM1
            result->eso_pro_catg_plus = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P23_QPLUS_CALIB_CAM2); //TAG for the double image science plus product CAM1
            result->eso_pro_catg_minus = cpl_strdup(
                    SPH_ZPL_TAG_SCIENCE_P23_QMINUS_CALIB_CAM2); //TAG for the double image science plus product CAM1
        }

        if (self->master_bias_frame_cam2) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam2);
        }
        if (self->master_dark_frame_cam2) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam2);
        }
        if (self->intensity_flat_frame_cam2) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame_cam2);
        }
        if (self->intensity_flat_frame_master_cam2) {
            result->intensity_flat_frame_master = cpl_frame_duplicate(
                    self->intensity_flat_frame_master_cam2);
        }
        if (self->polarization_flat_frame_cam2) {
            result->polarization_flat_frame = cpl_frame_duplicate(
                    self->polarization_flat_frame_cam2);
        }
        if (self->modem_efficiency_frame_cam2) {
            result->modem_efficiency_frame = cpl_frame_duplicate(
                    self->modem_efficiency_frame_cam2);
        }
        if (self->star_center_frame_cam2) {
            result->star_center_frame = cpl_frame_duplicate(
                    self->star_center_frame_cam2);
        }


    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Wrong camera id = %d (it must be 1 or 2)",
                camera_id);
        _sph_zpl_science_p23_cunit_delete(result);
        result = NULL;
    }

    if (sph_error_get_last_code() != CPL_ERROR_NONE) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Post condition error check shows the following error %d, please, check ",
                sph_error_get_last_code());
        _sph_zpl_science_p23_cunit_delete(result);
        result = NULL;
    }

    return result;

}

sph_error_code _sph_zpl_science_p23_cunit_delete(
        sph_zpl_science_p23_cunit* cunit) {
    sph_error_code rerr = CPL_ERROR_NONE;

    if (cunit == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return rerr;
    }

    /* Code to delete recipe pointers GENERATED DO NOT EDIT */

    if (cunit->preproc_frames != NULL) {
        cpl_frameset_delete(cunit->preproc_frames);
        cunit->preproc_frames = NULL;
    }
    if (cunit->science_di) {
        sph_double_image_delete(cunit->science_di);
        cunit->science_di = NULL;
    }
    if (cunit->science_di_plus) {
        sph_double_image_delete(cunit->science_di_plus);
        cunit->science_di_plus = NULL;
    }
    if (cunit->science_di_minus) {
        sph_double_image_delete(cunit->science_di_minus);
        cunit->science_di_minus = NULL;
    }

    if (cunit->science_p23_outfilename)
        cpl_free(cunit->science_p23_outfilename);
    if (cunit->science_p23_plus_outfilename)
        cpl_free(cunit->science_p23_plus_outfilename);
    if (cunit->science_p23_minus_outfilename)
        cpl_free(cunit->science_p23_minus_outfilename);
    if (cunit->eso_pro_catg)
        cpl_free(cunit->eso_pro_catg);
    if (cunit->eso_pro_catg_plus)
        cpl_free(cunit->eso_pro_catg_plus);
    if (cunit->eso_pro_catg_minus)
        cpl_free(cunit->eso_pro_catg_minus);
    if (cunit->fgs)
        sph_zpl_framegroups_delete(cunit->fgs);
    if (cunit->fgsraw)
        sph_zpl_framegroups_delete(cunit->fgsraw);

    if (cunit->preproc_frames)
        cpl_frameset_delete(cunit->preproc_frames);

    if (cunit->master_bias_frame)
        cpl_frame_delete(cunit->master_bias_frame);
    if (cunit->master_dark_frame)
        cpl_frame_delete(cunit->master_dark_frame);
    if (cunit->intensity_flat_frame)
        cpl_frame_delete(cunit->intensity_flat_frame);
    if (cunit->intensity_flat_frame_master)
        cpl_frame_delete(cunit->intensity_flat_frame_master);
    if (cunit->polarization_flat_frame)
        cpl_frame_delete(cunit->polarization_flat_frame);
    if (cunit->modem_efficiency_frame)
        cpl_frame_delete(cunit->modem_efficiency_frame);

    if (cunit->star_center_frame)
        cpl_frame_delete(cunit->star_center_frame);

    if (cunit->current_raw_frameset) {
        cpl_frameset_delete(cunit->current_raw_frameset);
        cunit->current_raw_frameset = NULL;
    }
    sph_polygon_free_all();
    cpl_free(cunit);
    return rerr;
}

static void sph_zpl_science_p23_delete__(sph_quad_image* master_bias_quadimage,
        sph_quad_image* master_dark_quadimage,
        sph_quad_image* intensity_flat_quadimage,
        sph_master_frame* intensity_flat_masterframe,
        sph_double_image* polarization_flat_doubleimage,
        sph_master_frame* modem_efficiency_masterframe) {
    if (master_bias_quadimage) {
        sph_quad_image_delete(master_bias_quadimage);

    }
    if (master_dark_quadimage) {
        sph_quad_image_delete(master_dark_quadimage);

    }
    if (intensity_flat_quadimage) {
        sph_quad_image_delete(intensity_flat_quadimage);

    }
    if (intensity_flat_masterframe) {
        sph_master_frame_delete(intensity_flat_masterframe);

    }
    if (polarization_flat_doubleimage) {
        sph_double_image_delete(polarization_flat_doubleimage);

    }
    if (modem_efficiency_masterframe) {
        sph_master_frame_delete(modem_efficiency_masterframe);

    }
}

static cpl_frame* sph_zpl_science_p23_calibrate_transform_frame__(
        const char* filename, cpl_frameset* inframes,
        const sph_fctable* fctable,
        const sph_quad_image* master_bias_quadimage,
        const sph_quad_image* master_dark_quadimage,
        const sph_quad_image* intensity_flat_quadimage,
        const sph_master_frame* intensity_flat_masterframe,
        const sph_double_image* polarization_flat_doubleimage,
        const sph_master_frame* modem_efficiency_masterframe) {

    cpl_msg_info(__func__, "Into calibrate_transform_frame");
    cpl_msg_info(__func__, "Have an outfilename of %s", filename);

    sph_zpl_exposure* zplexp = NULL;
    sph_cube* dicube = NULL;
    cpl_frame* result = NULL;

    int iplane = 0;
    int rerr = CPL_ERROR_NONE;
    sph_transform* transform = NULL;

    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(fctable, CPL_ERROR_NULL_INPUT, NULL);

    transform = sph_transform_new_default();
    if (!transform) {
        SPH_ERR("sph_transform_new_dafault() constructor returns NULL...");
        return NULL;
    }
    dicube = sph_cube_new(filename);
    if (!dicube) {
        SPH_ERR("sph_cube_new constructor returns NULL...");
            sph_transform_delete(transform);
        return NULL;
    }

    zplexp = sph_common_science_get_next_zpl_exposure(inframes);
    if (!zplexp) {
        SPH_ERR(
                "no zplexp can be loaded, sph_common_science_get_next_zpl_exposure returns NULL...");
            sph_transform_delete(transform);
            sph_cube_delete(dicube);
        return NULL;
    }

    while (zplexp) {
        sph_double_image* stokes = NULL;
        sph_quad_image* curquadimage = 
            sph_quad_image_new_from_zpl_exposures(zplexp);

        if (!curquadimage) {
            SPH_ERR(
                    "can't create quad image from the zplexp, sph_quad_image_new_from_zpl_exposures constructor returns NULL...");
            sph_transform_delete(transform);
            sph_cube_delete(dicube);
            sph_zpl_exposure_delete(zplexp);
            return NULL;

        }

        if (master_bias_quadimage) {
            rerr = sph_quad_image_subtract_quad_image(curquadimage,
                    master_bias_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting master bias:\n"
                                "cpl error code is: %d", cpl_error_get_code());

                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Subtracted master bias.")
        }

        if (master_dark_quadimage) {
            rerr = sph_zpl_subtract_dark_quad_image_scaled(curquadimage,
                    master_dark_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting dark:\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Subtracted master dark.")
        }

        if (intensity_flat_masterframe) {
            rerr = sph_quad_image_divide_master_frame(curquadimage,
                    intensity_flat_masterframe);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(
                        SPH_ERROR_GENERAL,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying (dividing) intensity flat (master frame):\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Applied intensity flat field (master frame).")
        } else if (intensity_flat_quadimage) {
            rerr = sph_quad_image_divide_quad_image(curquadimage,
                    intensity_flat_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(
                        SPH_ERROR_GENERAL,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying (dividing) intensity flat (quad image):\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Applied intensity flat field (quad image).")

        }

        stokes =
                sph_zpl_utils_calculate_stokes_param_double_image_from_quad_image(
                        curquadimage);
        if (!stokes) {
            SPH_ERR("stokes parameters is NULL");
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            return NULL;
        }SPH_INFO_MSG("Calculated stokes.");

        if (polarization_flat_doubleimage) {
            rerr = sph_zpl_utils_apply_polarization_flat(stokes,
                    polarization_flat_doubleimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying polarization flat:\n"
                                "sph error code is: %d",
                        sph_error_get_last_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }SPH_INFO_MSG("Applied polarization flat.");

        }

        if (modem_efficiency_masterframe) {
            rerr = sph_zpl_utils_apply_modem_efficiency(stokes,
                    modem_efficiency_masterframe);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying modem efficiency:\n"
                                "sph error code is: %d",
                        sph_error_get_last_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }SPH_INFO_MSG("Applied modem effieciency.");
        }

        //quality check (calculate statistical moments for the basic polarization)
        rerr = sph_double_image_quality_check(stokes);
        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;
        }

        //if stokes double image is not squared, interpolate it in y direction
        if (!sph_double_image_is_squared(stokes)) {
            SPH_INFO_MSG(
                    "double image is not squared, it will be linear interpolated in y-direction, "
                    "the total flux will be preserved (a small flux lost is due to the border lines)!")
            rerr = sph_double_image_interpolate_y_zpl_polarimetry( stokes, CPL_TRUE );
            if (rerr) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "sph_doble_image_interpolate_y returns cpl error.\n"
                                "cpl error code is: %d", rerr);
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }
        }
        cpl_msg_info(__func__, "Past the double image is not squared block");

        //check if it is still not squared
        if (!sph_double_image_is_squared(stokes)) {
            SPH_ERR("doubleimage is still not squared, "
                    "check the size of the input data...stop processing.\n");

            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;
        }

        cpl_msg_info(cpl_func, "sph_transform_apply_to_doubleimage(%d:%d:%d): "
                     "%g", (int)transform->method,
                     (int)SPH_TRANSFORM_METHOD_FFT,
                     (int)SPH_TRANSFORM_METHOD_CPL_WARP,
                     sph_fctable_get_angle(fctable, iplane));
        rerr = sph_transform_apply_to_doubleimage(transform, stokes, NULL,
                sph_fctable_get_centre_x_element(fctable, iplane),
                sph_fctable_get_centre_y_element(fctable, iplane),
                sph_fctable_get_angle(fctable, iplane), 1.0);

        if (rerr) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_transform_apply_to_doubleimage function returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
           return NULL;
        }

        sph_quad_image_delete(curquadimage);
        sph_zpl_exposure_delete(zplexp);

        SPH_INFO_MSG(
                "append the transformed stokes double image to the cube...")
        rerr = sph_cube_append_double_image(dicube, stokes, 1);

        sph_double_image_delete(stokes);

        if (rerr) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "can't append the transformed stokes double image to the cube: "
                            "sph_cube_append_double_image function returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            return NULL;
        }

        zplexp = sph_common_science_get_next_zpl_exposure(inframes);
        iplane++;
    }

    rerr = sph_cube_finalise_file(dicube->filename);
    if (rerr) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "can't finalize output cube for the filename %s: sph_cube_finalise_file function returns cpl error.\n"
                        "cpl error code is: %d", dicube->filename, rerr);
        sph_cube_delete(dicube);
        sph_transform_delete(transform);
        return NULL;
    }

    sph_cube_delete(dicube);

    result = cpl_frame_new();
    cpl_frame_set_filename(result, filename);
    cpl_frame_set_tag(result, SPH_ZPL_TAG_SCIENCE_P23_CALIB_TMP);
    sph_transform_delete(transform);

    return result;

}

static cpl_frameset* sph_zpl_science_p23_calibrate_frames__(
        sph_zpl_science_p23_cunit* self,
        const cpl_frameset* inframes,
        const sph_quad_image* master_bias_quadimage,
        const sph_quad_image* master_dark_quadimage,
        const sph_quad_image* intensity_flat_quadimage,
        const sph_master_frame* intensity_flat_masterframe,
        const sph_double_image* polarization_flat_doubleimage,
        const sph_master_frame* modem_efficiency_masterframe) {

    cpl_frameset*     results        = NULL;
    cpl_propertylist* pl_star_center = NULL;
    const cpl_frame*  curframe       = cpl_frameset_get_first_const(inframes);
    const cpl_size    nframe         = cpl_frameset_get_size(inframes);
    cpl_size          iframe         = 0;

    cpl_image* image = cpl_image_load(cpl_frame_get_filename(curframe),
                                      CPL_TYPE_DOUBLE, 0, 0);
    const cpl_size nx = image ? cpl_image_get_size_x(image) : 1024;


    SPH_INFO_MSG("Calibrate cubes...");

    if (image != NULL) {
        cpl_image_delete( image ); image = NULL;
    } else {
        (void)cpl_error_set_where(cpl_func);
    }

    cpl_ensure(self     != NULL, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(inframes != NULL, CPL_ERROR_NULL_INPUT, NULL);

    results = cpl_frameset_new();

    if ( self->star_center_frame) {
        const char * filename = cpl_frame_get_filename(self->star_center_frame);
        pl_star_center = cpl_propertylist_load(filename, 0);
        if (pl_star_center == NULL)
            (void)cpl_error_set_where(cpl_func);
    }

    //curframe = cpl_frameset_get_first( ovsc_subtract_frames );

    while (curframe) {
        const char * filename = cpl_frame_get_filename( curframe );
        char* outfilename = NULL;
        char* procatg = NULL ;

        cpl_frameset* curframes = cpl_frameset_new();
        cpl_frameset* ovsc_subtract_frames = NULL;
        cpl_frame* calframe = NULL;
        sph_fctable* fctable = NULL;

        /* Set up a middle of the image */
        const double clx = (double)nx / 2.0;
        const double cly = clx;
        const double crx = clx;
        const double cry = clx;

        iframe++;
        cpl_frameset_insert(curframes, cpl_frame_duplicate(curframe));

        cpl_msg_info(cpl_func, "Processing frame %d/%d: %s",
                     (int)iframe, (int)nframe, filename);

        if (self->fctable_frames) {
            fctable = sph_fctable_find_fctable(curframe, self->fctable_frames);
            cpl_msg_info(cpl_func, fctable == NULL
                         ? "Current frame has no input field center table"
                         : "Current frame has an input field center table");
        } else {
            cpl_msg_info(cpl_func, "No input field center table");
        }

        if (fctable == NULL) {
            cpl_propertylist* plist =
                cpl_propertylist_load_regexp(filename, 0,
                                             SPH_ZPL_KEYWORD_INS4_DROT2_POSANG,
                                             CPL_FALSE);
            const cpl_boolean has_drot2 =
                cpl_propertylist_has(plist, SPH_ZPL_KEYWORD_INS4_DROT2_POSANG);
            const double drot2 = has_drot2
                ? cpl_propertylist_get_double(plist,
                                              SPH_ZPL_KEYWORD_INS4_DROT2_POSANG)
                : 0.0;

            cpl_propertylist_delete(plist);
            fctable = sph_fctable_create_fctable_new(curframe, SPH_FCTAB_DOUBLE);

            if (fctable == NULL) {
                SPH_ERR("fctable has null pointer");
                cpl_frameset_delete(results);
                cpl_frameset_delete(curframes);
                return NULL;
            }
            /* Modify the field center table with the rotation angle */
            sph_fctable_add_angle_scalar( fctable, drot2 );

            cpl_msg_info(cpl_func, "Created %d-fctable for the current frame. "
                         "DROT2: %g%s", sph_fctable_get_size(fctable),
                         drot2, has_drot2 ? "" : " (default)");
   
            cpl_msg_info(cpl_func, "Applying offset from the center of image: "
                         "(%g, %g)", self->center_xoffset, self->center_yoffset);
            sph_fctable_add_center_offset_double_mode ( fctable,
                                                        self->center_xoffset,
                                                        self->center_yoffset,
                                                        self->center_xoffset,
                                                        self->center_yoffset);

            if ( pl_star_center ) {
                if ( cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_XCOORD) &&
                     cpl_propertylist_has( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_YCOORD) ){

                    SPH_INFO_MSG("Applying star center calibration...");
                    if ( !self->star_center_iframe ){
                        SPH_INFO_MSG("Center coordinates of IFRAME and PFRAME are used to de-rotate double image.");
                        sph_fctable_add_center_offset_double_mode ( fctable,
                                                                    - clx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                                                                    - cly + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD),
                                                                    - crx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_XCOORD),
                                                                    - cry + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_PFRAME_YCOORD) );
                    } else {
                        SPH_INFO_MSG("Center coordinates of IFRAME  are used to de-rotate double image.");
                        sph_fctable_add_center_offset_double_mode ( fctable,
                                                                    - clx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                                                                    - cly + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD),
                                                                    - crx + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_XCOORD),
                                                                    - cry + cpl_propertylist_get_double( pl_star_center, SPH_ZPL_STAR_CENTER_IFRAME_YCOORD) );
                    }
                }
            }

            if (self->save_interprod) {
                char* fctable_name = 
                    sph_fctable_construct_filename(curframe, 1);
                sph_fctable_save_ascii(fctable, fctable_name);
                cpl_free(fctable_name);
            }
        }

        if (self->subtract_overscan) {
            SPH_INFO_MSG("Subtract overscans from rawframes...")
            ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(curframes);
            if (!ovsc_subtract_frames) {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_WARNING,
                        "Overscans could not be subtracted, trying to proceed further "
                        "without subtraction...")
                ovsc_subtract_frames = cpl_frameset_duplicate(curframes);
                sph_error_reset(); //reset is needed because sph_zpl_utils_subtract_overscans raises error when it returns NULL
            }
        } else {
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_WARNING,
                    "Overscans could not be subtracted, trying to proceed further "
                    "without subtraction...")
            ovsc_subtract_frames = cpl_frameset_duplicate(curframes);
        }

        // PIPE-7143 - here's where we want to change the filenames
        cpl_msg_info(__func__, "Creating filename for the intermediate products...");
//        outfilename = cpl_sprintf( "%s_%s", self->science_p23_outfilename,
//                                   sph_filemanager_get_basename( filename));
        cpl_propertylist* curframe_pl = cpl_propertylist_load_regexp(
                cpl_frame_get_filename(curframe), 0,
                cpl_sprintf("^.*(%s|%s).*$",
                            SPH_ZPL_KEYWORD_POL_STOKES_NAME,
                            SPH_ZPL_KEYWORD_DET_CHIP_INDEX
                            ),
                            0
                            );
//        cpl_propertylist_dump(curframe_pl, stdout);
        if (cpl_propertylist_get_size(curframe_pl) != (cpl_size)2) {
            cpl_msg_warning(__func__, "Unable to determine STOKES NAME and CAM for %s - "
                                      "resorting to generic PRO CATG",
                                      cpl_frame_get_filename(curframe));
            procatg = cpl_sprintf("ZPL_SCIENCE_P23_CALIBRATED%s", "");
        } else {
            const char* stokes_name = cpl_propertylist_get_string(curframe_pl, SPH_ZPL_KEYWORD_POL_STOKES_NAME);
            procatg = cpl_sprintf("ZPL_SCIENCE_P23_CALIBRATED_%s_CAM%d",
                    stokes_name,
                    cpl_propertylist_get_int(curframe_pl, SPH_ZPL_KEYWORD_DET_CHIP_INDEX)
                    );
            sph_utils_uppercase(procatg + 27); /* Convert from stokes_name */
        }
        cpl_propertylist_delete(curframe_pl); curframe_pl = NULL ;
        char base_fn[1024] ;
        char ext_fn[256] ;
        sph_filemanager_split(filename, base_fn, ext_fn);
//        cpl_msg_info(__func__, "Split into %s and %s", base_fn, ext_fn);
        outfilename = cpl_sprintf("%s_%s", procatg, base_fn);
//        cpl_msg_info(__func__, "outfilename base is %s", outfilename_base);
        SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                              "Intermediate outfilename = %s", outfilename);
//        getchar();

        char* outfilename_tmp = cpl_sprintf("tmp_%s", outfilename);
        calframe = sph_zpl_science_p23_calibrate_transform_frame__(
                outfilename_tmp, ovsc_subtract_frames, fctable,
                master_bias_quadimage, master_dark_quadimage,
                intensity_flat_quadimage, intensity_flat_masterframe,
                polarization_flat_doubleimage, modem_efficiency_masterframe );
        cpl_msg_info(__func__, "Completed calibrate_transform_frame");

        // PIPE-7143 outfilename points to our interprod cube, so we can apply
        // PRO.CATG to it here
        if (self->save_interprod) {
            //            cpl_property* pcatg_prop = cpl_property_new(SPH_COMMON_KEYWORD_PRO_CATG,
            //                                                        CPL_TYPE_STRING);
            //            cpl_property_set_string(pcatg_prop, procatg);
            //            sph_fits_update_property(outfilename,
            //                                     pcatg_prop,
            //                                     0);
            //            cpl_property_delete(pcatg_prop); pcatg_prop = NULL ;

            cpl_propertylist* applist = cpl_propertylist_new();
            cpl_propertylist_append_string(applist,
                                           SPH_COMMON_KEYWORD_PRO_CATG,
                                           procatg);

            //            cpl_msg_info(__func__, "self->inframes:");
            //            cpl_frameset_dump(self->inframes, stdout);
            //            cpl_msg_info(__func__, "self->inparams:");
            //            cpl_parameterlist_dump(self->inparams, stdout);
            //             Must set CPL_FRAME_GROUP_CALIB for all frames
            cpl_frameset_iterator* it = cpl_frameset_iterator_new(inframes);
            cpl_frame* fr = cpl_frameset_iterator_get(it);
            while (fr) {
                cpl_frame_set_group(fr, CPL_FRAME_GROUP_CALIB);
                cpl_frameset_iterator_advance(it, 1);
                fr = cpl_frameset_iterator_get(it);
            }
            cpl_frameset_iterator_delete(it); it = NULL;
            //            cpl_msg_info(__func__, "inframes:");
            //            cpl_frameset_dump(inframes, stdout);
            // Now, see if we can save a DFS product here
            //            cpl_msg_info(__func__, "first:");
            //            cpl_msg_info(__func__, "applist:");
            //            cpl_propertylist_dump(applist, stdout);
            cpl_errorstate prestate = cpl_errorstate_get();
            cpl_error_code rerr = cpl_dfs_save_propertylist(
                    self->inframes, NULL,
                    self->inparams, inframes,
                    NULL, SPH_RECIPE_NAME_ZPL_SCIENCE_P1,
                    applist, NULL,
                    SPH_PIPELINE_NAME_ZIMPOL,
                    outfilename
                    );
            if (rerr != CPL_ERROR_NONE) {
                cpl_msg_warning(__func__, "DFS FAIL: %d, %s @ L%i, %s",
                                rerr, cpl_error_get_file(),
                                cpl_error_get_line(),
                                cpl_error_get_message());
                cpl_msg_warning(__func__, "Previous error state:");
                cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
                //                getchar();
            }

            // Now need to append the different panels to the DFS product
            cpl_imagelist* cube_im ;
            for (cpl_size i=0; i <= cpl_frame_get_nextensions(calframe); i++) {
                cube_im = cpl_imagelist_load(cpl_frame_get_filename(calframe),
                                             CPL_TYPE_UNSPECIFIED,
                                             i);
                prestate = cpl_errorstate_get();
                rerr = cpl_imagelist_save(cube_im, outfilename,
                                          CPL_TYPE_UNSPECIFIED, NULL,
                                          CPL_IO_EXTEND);
                if (rerr != CPL_ERROR_NONE) {
                    cpl_msg_warning(__func__, "IMSAVE FAIL: %d, %s @ L%i, %s",
                                    rerr, cpl_error_get_file(),
                                    cpl_error_get_line(),
                                    cpl_error_get_message());
                    cpl_errorstate_dump(prestate, CPL_FALSE, NULL);
                    //                    getchar();
                }
            }
            cpl_imagelist_delete(cube_im) ; cube_im = NULL ;

            cpl_propertylist_delete(applist) ; applist = NULL;
        }

        cpl_free(outfilename); outfilename = NULL;
        cpl_free(outfilename_tmp); outfilename_tmp = NULL;
        cpl_free(procatg); procatg = NULL ;
        sph_fctable_delete(fctable); fctable = NULL;
        cpl_frameset_delete(curframes); curframes = NULL;

        if (!self->keep_intermediate) {
            sph_utils_frames_unlink(ovsc_subtract_frames);
        }
        cpl_frameset_delete(ovsc_subtract_frames); ovsc_subtract_frames = NULL;

        if (!calframe) {
            SPH_ERR("can't calibrate the given frame(plane), the sph_zpl_science_p23_calibrate_transform_frame returns NULL");
            cpl_frameset_delete(results);
            results = NULL;
            curframe = NULL;
        } else {
            cpl_frameset_insert(results, calframe);
            //curframe = cpl_frameset_get_next(ovsc_subtract_frames);
            curframe = cpl_frameset_get_next_const(inframes);
        }

    }

    (void)cpl_error_set_where(cpl_func);
    return results;

}

cpl_error_code _sph_zpl_science_p23_cunit_run(sph_zpl_science_p23_cunit* self) {
    int recipe_error = CPL_ERROR_NONE;
    sph_quad_image* master_bias_quadimage = NULL;
    sph_quad_image* master_dark_quadimage = NULL;
    sph_quad_image* intensity_flat_quadimage = NULL;
    sph_master_frame* intensity_flat_masterframe = NULL;
    sph_double_image* polarization_flat_doubleimage = NULL;
    sph_master_frame* modem_efficiency_masterframe = NULL;
    sph_master_frame* imframe = NULL;
    sph_master_frame* pmframe = NULL;

    cpl_frameset* calframes = NULL;
    cpl_propertylist* pl = NULL;

    cpl_propertylist* pl_wcs = NULL;
    double xpix = -9999.0;
    double ypix = -9999.0;


    SPH_INFO_MSG("Starting static _sph_zpl_science_p23_cunit_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (!self->preproc_frames) {
        SPH_ERR("no preproc frames are set up!")
        return sph_error_get_last_code();
    }
    if (!self->fgs) {
        SPH_ERR("no framegroups found in the cunit!")
        return sph_error_get_last_code();
    }

    //if master bias exists load it
    if (self->master_bias_frame) {
        SPH_INFO_MSG("Loading master bias...\n");
        master_bias_quadimage = sph_quad_image_load(
                cpl_frame_get_filename(self->master_bias_frame), 0);
    }

    //if master dark exists load it
    if (self->master_dark_frame) {
        SPH_INFO_MSG("Loading master dark...\n");
        master_dark_quadimage = sph_quad_image_load(
                cpl_frame_get_filename(self->master_dark_frame), 0);
    }

    if (self->intensity_flat_frame_master) {
        //if intensity flat field master frame exists load
        SPH_INFO_MSG("Loading intensity flat field masterframe...\n");
        intensity_flat_masterframe = sph_master_frame_load_(
                self->intensity_flat_frame_master, 0);
    } else {
        //if intensity flat field quad image exists load
        if (self->intensity_flat_frame) {
            SPH_INFO_MSG("Loading intensity flat field quad image...\n");
            intensity_flat_quadimage = sph_quad_image_load(
                    cpl_frame_get_filename(self->intensity_flat_frame), 0);
        }
    }

    //if polarization flat field exists load it
    if (self->polarization_flat_frame) {
        SPH_INFO_MSG("Loading polarization flat field...\n");
        polarization_flat_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->polarization_flat_frame), 0);
    }

    //if modulation/demodulation efficiency exists load it
    if (self->modem_efficiency_frame) {
        SPH_INFO_MSG("Loading modulation/demodulation efficiency...\n");
        modem_efficiency_masterframe = sph_master_frame_load_(
                self->modem_efficiency_frame, 0);
    }

    if (self->fgs->framesplus != NULL && self->fgs->framesminus == NULL) {
        SPH_WARNING(
                "Only Plus frames were found, the performance can be reduced!");
        SPH_ERROR_RAISE_WARNING(
                CPL_ERROR_ILLEGAL_INPUT,
                "It is usually needed two polarization stokes string value (Qplus/Uplus) and (Qminus/Uminus) "
                "in the header input. Make sure the %s keyword with the value = %s(%s) is presented in the "
                "input frames.", SPH_ZPL_KEYWORD_POL_STOKES_NAME, SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS, SPH_ZPL_KEYWORD_VALUE_POL_STOKES_UMINUS);

        //calibrate and transform
        calframes = sph_zpl_science_p23_calibrate_frames__(self,
                self->fgs->framesplus,
                master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);

        if (!calframes) {
            SPH_ERR("Can't calibrate given frames...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine iframe and pframe
        imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!imframe) {
            SPH_ERR(
                    "Can't combine calframes, imframe from sph_common_science_combine is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                // PIPE-7143 - if no longer needed, saved as DFS
//                if ( !self->save_interprod ) {
                    sph_utils_frames_unlink(calframes);
//                }
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT + 4,
                -1);
        if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                // PIPE-7143 - if no longer needed, saved as DFS
//                if ( !self->save_interprod ) {
                    sph_utils_frames_unlink(calframes);
//                }
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }
        // PIPE-7143 - if no longer needed, saved as DFS
//        if ( !self->save_interprod ) {
            sph_utils_frames_unlink(calframes);
//        }
        cpl_frameset_delete(calframes);
        calframes = NULL;

        //create double image from iframe and pframe master frames
        //input master frames are not duplicated, so don't delete them after creation of the double image
        self->science_di_plus = sph_double_image_new_from_master_frames(imframe,
                pmframe);
        if (!self->science_di_plus) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe master frames, self->science_di_plus is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            if (pmframe) {
                sph_master_frame_delete(pmframe);
                pmframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di_plus);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di_plus->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di_plus->iframe->image)/2.;


        if (self->rawframes) {
            SPH_INFO_MSG(
                    "Save science polarimetry P2 as a dfs product: header info taken from the first raw frame.");
            pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->fgsraw->framesplus);
            //create WCS keywords and add to pl
            pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesplus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

            recipe_error = sph_double_image_save_dfs(self->science_di_plus,
                    self->science_p23_plus_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesplus),
                    self->inparams,
                    self->eso_pro_catg_plus, //SPH_ZPL_TAG_SCIENCE_P23_CALIB,
                    SPH_RECIPE_NAME_ZPL_SCIENCE_P23, SPH_PIPELINE_NAME_ZIMPOL,
                    pl);
            if ( pl ) {
                cpl_propertylist_delete(pl);
                pl = NULL;
             }

        } else if (self->preproc_frames) {
            SPH_INFO_MSG(
                     "Save science polarimetry P2 as a dfs product: header info taken from the pre-processed frame.");
             pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                     (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesplus),
                     (const double) xpix, (const double) ypix);

            recipe_error = sph_double_image_save_dfs(self->science_di_plus,
                    self->science_p23_plus_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesplus),
                    self->inparams,
                    self->eso_pro_catg_plus, //SPH_ZPL_TAG_SCIENCE_P23_CALIB,
                    SPH_RECIPE_NAME_ZPL_SCIENCE_P23, SPH_PIPELINE_NAME_ZIMPOL,
                    pl_wcs);
            if ( pl_wcs ) {
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
        }

        sph_zpl_science_p23_delete__(master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);

        SPH_INFO_MSG("sph_zpl_science_p23...End");
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

    } else if (self->fgs->framesplus == NULL && self->fgs->framesminus != NULL) {
        SPH_WARNING(
                "Only Minus frames were found, the performance can be reduced!");
        SPH_ERROR_RAISE_WARNING(
                CPL_ERROR_ILLEGAL_INPUT,
                "It is usually needed two polarization stokes string value (Qplus/Uplus) and (Qminus/Uminus) "
                "in the header input. Make sure the %s keyword with the value = %s(%s) is presented in the "
                "input frames.", SPH_ZPL_KEYWORD_POL_STOKES_NAME, SPH_ZPL_KEYWORD_VALUE_POL_STOKES_QMINUS, SPH_ZPL_KEYWORD_VALUE_POL_STOKES_UMINUS);

        //calibrate and transform MINUS frames(cubes)
        calframes = sph_zpl_science_p23_calibrate_frames__(self,
                self->fgs->framesminus,
                master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);
        if (!calframes) {
            SPH_ERR("Can't calibrate given MINUS frames...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine MINUS frames(cubes)
        imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!imframe) {
            SPH_ERR("Can't combine calframes, imframe is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                // PIPE-7143 - if no longer needed, saved as DFS
//                if ( !self->save_interprod ) {
                    sph_utils_frames_unlink(calframes);
//                }
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                // PIPE-7143 - if no longer needed, saved as DFS
//                if ( !self->save_interprod ) {
                    sph_utils_frames_unlink(calframes);
//                }
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        if (calframes) {
//            if ( !self->save_interprod ) {
                sph_utils_frames_unlink(calframes);
//            }
            cpl_frameset_delete(calframes);
            calframes = NULL;
        }

        //create calibrated and transformed double image for MINUS frames
        self->science_di_minus = sph_double_image_new_from_master_frames(
                imframe, pmframe);
        if (!self->science_di_minus) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe of the MINUS master frames, self->science_di_minus is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            if (pmframe) {
                sph_master_frame_delete(pmframe);
                pmframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //MINUS: calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di_minus);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }


        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di_minus->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di_minus->iframe->image)/2.;


        if (self->rawframes) {
            SPH_INFO_MSG(
                    "Save science polarimetry P2 as a dfs product: header info taken from the first raw frame.");

            pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->fgsraw->framesminus);
            //create WCS keywords and add to pl
            pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesminus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }


            recipe_error = sph_double_image_save_dfs(self->science_di_minus,
                    self->science_p23_minus_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesminus), self->inparams,
                    self->eso_pro_catg_minus, SPH_RECIPE_NAME_ZPL_SCIENCE_P23,
                    SPH_PIPELINE_NAME_ZIMPOL, pl);

            if ( pl ) {
                cpl_propertylist_delete(pl);
                pl = NULL;
             }

        } else if (self->preproc_frames) {
            SPH_INFO_MSG(
                    "Save science polarimetry P2 as a dfs product: header info taken from the pre-processed frame.");
            pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesminus),
                    (const double) xpix, (const double) ypix);

            recipe_error = sph_double_image_save_dfs(self->science_di_minus,
                    self->science_p23_minus_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesminus),
                    self->inparams, self->eso_pro_catg_minus,
                    SPH_RECIPE_NAME_ZPL_SCIENCE_P23, SPH_PIPELINE_NAME_ZIMPOL,
                    pl_wcs);
            if ( pl_wcs ) {
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }


        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di_minus product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        SPH_INFO_MSG("sph_zpl_science_p23...End");
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE

    } else {
        SPH_INFO_MSG("PLUS and MINUS frames found!");
        //calibrate and transform PLUS frames(cubes)
        calframes = sph_zpl_science_p23_calibrate_frames__(self,
                self->fgs->framesplus,
                master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);
        if (!calframes) {
            SPH_ERR("Can't calibrate given PLUS frames...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine PLUS frames(cubes): imframe
        imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!imframe) {
            SPH_ERR("Can't combine calframes, imframe is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                // PIPE-7143 - if no longer needed, saved as DFS
//                if ( !self->save_interprod ) {
                    sph_utils_frames_unlink(calframes);
//                }
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine PLUS frames(cubes): pmframe
        pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                // PIPE-7143 - if no longer needed, saved as DFS
//                if ( !self->save_interprod ) {
                    sph_utils_frames_unlink(calframes);
//                }
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        if (calframes) {
            // PIPE-7143 - if no longer needed, saved as DFS
//            if ( !self->save_interprod ) {
                sph_utils_frames_unlink(calframes);
//            }
            cpl_frameset_delete(calframes);
            calframes = NULL;
        }

        //create calibrated and transfrmed double image for PLUS frames
        self->science_di_plus = sph_double_image_new_from_master_frames(imframe,
                pmframe);
        if (!self->science_di_plus) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe PLUS master frames, self->science_di_plus is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            if (pmframe) {
                sph_master_frame_delete(pmframe);
                pmframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //calibrate and transform MINUS frames(cubes)
        calframes = sph_zpl_science_p23_calibrate_frames__(self,
                self->fgs->framesminus,
                master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);
        if (!calframes) {
            SPH_ERR("Can't calibrate given MINUS frames...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        //combine MINUS frames(cubes)
        imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!imframe) {
            SPH_ERR("Can't combine calframes, imframe is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                // PIPE-7143 - if no longer needed, saved as DFS
//                if ( !self->save_interprod ) {
                    sph_utils_frames_unlink(calframes);
//                }
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT, -1);
        if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (calframes) {
                // PIPE-7143 - if no longer needed, saved as DFS
//                if ( !self->save_interprod ) {
                    sph_utils_frames_unlink(calframes);
//                }
                cpl_frameset_delete(calframes);
                calframes = NULL;
            }
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        if (calframes) {
            // PIPE-7143 - if no longer needed, saved as DFS
//            if ( !self->save_interprod ) {
                sph_utils_frames_unlink(calframes);
//            }
            cpl_frameset_delete(calframes);
            calframes = NULL;
        }

        //create calibrated and transformed double image for MINUS frames
        self->science_di_minus = sph_double_image_new_from_master_frames(
                imframe, pmframe);
        if (!self->science_di_minus) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe of the MINUS master frames, self->science_di_minus is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            if (imframe) {
                sph_master_frame_delete(imframe);
                imframe = NULL;
            }
            if (pmframe) {
                sph_master_frame_delete(pmframe);
                pmframe = NULL;
            }
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        if (!(self->science_di_minus && self->science_di_plus)) {
            SPH_ERROR_RAISE_ERR(
                    SPH_ERROR_GENERAL,
                    " sciece_di_minus and/or self->science_di_plus have(has) null pointer(s)!\n");
            SPH_RAISE_CPL;
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return (int) cpl_error_get_code();
        }

        //PLUS: calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di_plus);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di_plus->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di_plus->iframe->image)/2.;


        if (self->rawframes) {
            SPH_INFO_MSG(
                    "Save science polarimetry P2 as a dfs product: header info taken from the first raw frame.");
            pl = sph_zpl_utils_get_camera_header(self->fgs->framesplus,
                    self->fgsraw->framesplus);
            //create WCS keywords and add to pl
            pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesplus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }


            recipe_error = sph_double_image_save_dfs(self->science_di_plus,
                    self->science_p23_plus_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesplus), self->inparams,
                    self->eso_pro_catg_plus, SPH_RECIPE_NAME_ZPL_SCIENCE_P23,
                    SPH_PIPELINE_NAME_ZIMPOL, pl);
            if ( pl ) {
                cpl_propertylist_delete(pl);
                pl = NULL;
            }

        } else if (self->preproc_frames) {
            SPH_INFO_MSG(
                    "Save science polarimetry P1 as a dfs product: header info taken from the pre-processed frame.");
            pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesplus),
                    (const double) xpix, (const double) ypix);

            recipe_error = sph_double_image_save_dfs(self->science_di_plus,
                    self->science_p23_plus_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesplus),
                    self->inparams, self->eso_pro_catg_plus,
                    SPH_RECIPE_NAME_ZPL_SCIENCE_P23, SPH_PIPELINE_NAME_ZIMPOL,
                    pl);
            if ( pl_wcs ) {
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }
        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di_plus product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        //MINUS: calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di_minus);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di_minus->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di_minus->iframe->image)/2.;

        if (self->rawframes) {
            SPH_INFO_MSG(
                    "Save science polarimetry P2 as a dfs product: header info taken from the first raw frame.");
            pl = sph_zpl_utils_get_camera_header(self->fgs->framesminus,
                    self->fgsraw->framesminus);
            pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesminus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

            recipe_error = sph_double_image_save_dfs(self->science_di_minus,
                    self->science_p23_minus_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesminus), self->inparams,
                    self->eso_pro_catg_minus, SPH_RECIPE_NAME_ZPL_SCIENCE_P23,
                    SPH_PIPELINE_NAME_ZIMPOL, pl);
            if ( pl ) {
                cpl_propertylist_delete(pl);
                pl = NULL;
            }

        } else if (self->preproc_frames) {
            SPH_INFO_MSG(
                     "Save science polarimetry P2 as a dfs product: header info taken from the pre-processed frame.");
             pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                     (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesminus),
                     (const double) xpix, (const double) ypix);

            recipe_error = sph_double_image_save_dfs(self->science_di_minus,
                    self->science_p23_minus_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesminus),
                    self->inparams, self->eso_pro_catg_minus,
                    SPH_RECIPE_NAME_ZPL_SCIENCE_P23, SPH_PIPELINE_NAME_ZIMPOL,
                    pl_wcs);
            if ( pl_wcs ) {
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di_minus product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        //combine +Q(+U) and -Q(+U) double images:
        // I = [I(+Q) + I(-Q)]/2.  (the same for +U and -U)
        // P = [P(+Q) - P(-Q)]/2.  (the same for +U and -U)
        //first, duplicate science_di_plus into science_di and then combine it with the science_di_minus
        self->science_di = sph_double_image_duplicate(self->science_di_plus);
        if (!self->science_di) {
            SPH_ERR(
                    "Can't duplicate science_di_plus to science_di, self->science_di is NULL...");
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
        }

        recipe_error = sph_double_image_combine_double_image(self->science_di,
                self->science_di_minus);
        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't combine plus and minus product science product: sph_double_image_combine_double_image returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        //calculate quality parameters and save it as a double image
        recipe_error = sph_double_image_quality_check(self->science_di);
        if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_science_p23_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
        }

        // reference pixels for WCS
        xpix = cpl_image_get_size_x(self->science_di->iframe->image)/2.;
        ypix = cpl_image_get_size_y(self->science_di->iframe->image)/2.;

        if (self->rawframes) {
            SPH_INFO_MSG(
                    "Save science polarimetry P2 as a dfs product: header info taken from the first raw frame.");
            pl = sph_zpl_utils_get_camera_header(self->preproc_frames,
                    self->fgsraw->framesplus);
            //create WCS keywords and add to pl
            pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgsraw->framesplus),
                    (const double) xpix, (const double) ypix);
            if ( pl_wcs && pl){
                cpl_propertylist_append( pl, pl_wcs );
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

            recipe_error = sph_double_image_save_dfs(self->science_di,
                    self->science_p23_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgsraw->framesplus), self->inparams,
                    self->eso_pro_catg, SPH_RECIPE_NAME_ZPL_SCIENCE_P23,
                    SPH_PIPELINE_NAME_ZIMPOL, pl);
            if ( pl ) {
                cpl_propertylist_delete(pl);
                pl = NULL;
            }

        } else if (self->preproc_frames) {
            SPH_INFO_MSG(
                    "Save science polarimetry P2 as a dfs product: header info taken from the pre-processed frame.");
            pl_wcs = sph_zpl_utils_astrometry_create_wcs_pl(
                    (const cpl_frame*) cpl_frameset_get_first(self->fgs->framesplus),
                    (const double) xpix, (const double) ypix);

            recipe_error = sph_double_image_save_dfs(self->science_di,
                    self->science_p23_outfilename, self->inframes,
                    cpl_frameset_get_first(self->fgs->framesplus),
                    self->inparams, self->eso_pro_catg,
                    SPH_RECIPE_NAME_ZPL_SCIENCE_P23, SPH_PIPELINE_NAME_ZIMPOL,
                    pl_wcs);
            if ( pl_wcs ) {
                cpl_propertylist_delete( pl_wcs ); pl_wcs = NULL;
            }

        } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
        }

        if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save science_di product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
        }

        sph_zpl_science_p23_delete__(master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);

        SPH_INFO_MSG("sph_zpl_science_p23_run...End");
        SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

}

/**@}*/

