/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <libgen.h>
#include <cpl.h>
#include <math.h>

#include "sph_zpl_star_center_pol.h"
#include "sph_zpl_common_preproc.h"
#include "sph_zpl_exposure.h"
#include "sph_zpl_keywords.h"
#include "sph_zpl_tags.h"
#include "sph_zpl_utils.h"
#include "sph_zpl_framegroups.h"
#include "sph_quad_image.h"
#include "sph_triple_image.h"
#include "sph_double_image.h"
#include "sph_master_frame.h"
#include "sph_keywordnode.h"
#include "sph_error.h"
#include "sph_utils.h"
#include "sph_common_keywords.h"
#include "sph_common_science.h"
#include "sph_filemanager.h"
#include "sph_transform.h"
#include "sph_zpl_subtract_dark_scaled.h"

#include "sph_zpl_star_center_common.h"

/*-----------------------------------------------------------------------------
 Error Codes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_star_center_pol_run    Star center Polarimetry Recipe
 *
 * This module finds the center coordinates from the star center calibration
 * frames, polarization modes
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_star_center_pol_run.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*-------------------------------------------------------------------------------
 * The Structure Definition
 *
 * This structure contains the members of the sph_zpl_star_center_pol_cunit that are
 * to reduce data for the one zimpol channel (camera). This structure is used as
 * an interface between a sphere "standard" recipe structure and re-designed
 * zimpol "_run" function which treats both zimpol channel.
 *
 * ------------------------------------------------------------------------------
 */
typedef struct _sph_zpl_star_center_pol_cunit {
    cpl_frameset*                   inframes                              ; /* The recipe input frames */

    cpl_frameset*                   current_raw_frameset                  ; /* The frameset for one product */

    cpl_parameterlist*        inparams                              ; /* The recipe input parameters */

    cpl_parameterlist*        framecomb_parameterlist               ; /* The recipe input parameters */

    char*                     star_center_outfilename       ; /* The parameter of zpl.star_center.outfilename */

    short                           subtract_overscan                     ; /* The parameter of zpl.star_center.subtract_overscan */

    short                           keep_intermediate                     ; /* The parameter of zpl.star_center.keep_intermediate */

    short                           save_interprod                        ; /* The parameter of zpl.star_center.save_interprod */

    int                             coll_alg                              ; /* The parameter of zpl.star_center.coll_alg */

    double                          filter_radius                         ; /* The parameter of zpl.star_center.filter_radius */

    double                          sigma                                 ; /* The parameter of zpl.star_center.sigma */

    int                             unsharp_window                        ; /* The parameter of zpl.star_center.unsharp_window */

    cpl_frameset*                   rawframes                             ; /* The ZPL_STAR_CENTER_POL_RAW frames */

    cpl_frameset*                   preproc_frames                        ; /* The ZPL_STAR_CENTER_POL_PREPROC frames */

    cpl_frame*                      master_bias_frame                     ; /* The ZPL_MASTER_BIAS frames */

    cpl_frame*                      master_dark_frame                     ; /* The ZPL_MASTER_DARK frames */

    cpl_frame*                      intensity_flat_frame                  ; /* The ZPL_INT_FLAT_FIELD frames */

    cpl_frame*                      intensity_flat_frame_master           ; /* The ZPL_INT_FLAT_FIELD_MASTER frames */

    cpl_frame*                      polarization_flat_frame               ; /* The ZPL_POL_FLAT_FIELD frames */

    cpl_frame*                      modem_efficiency_frame                ; /* The ZPL_MODEM_EFF frames */

    sph_double_image*               star_center_di                        ; /* The ZPL_STAR_CENTER_POL product */

    char* eso_pro_catg; /* Here the ZPL_STAR_CENTER_CALIB_CAM1 or _CAM2 will be written */

} sph_zpl_star_center_pol_cunit;

/*-----------------------------------------------------------------------------
 Private function prototypes
 -----------------------------------------------------------------------------*/
static sph_zpl_star_center_pol_cunit* _sph_zpl_star_center_pol_cunit_create(
        sph_zpl_star_center_pol* self, int camera);
static void _sph_zpl_star_center_pol_cunit_delete(
        sph_zpl_star_center_pol_cunit* cunit);
static cpl_error_code _sph_zpl_star_center_pol_cunit_run(
        sph_zpl_star_center_pol_cunit* cunit);
//static cpl_error_code _sph_zpl_star_center_pol_create_fctable_frames ()

/*----------------------------------------------------------------------------*/
/**
 @brief    Interpret the command line options and execute the data processing
 @param    frameset   the frames list
 @param    parlist    the parameters list

 @return   the cpl error code of the operation.

 This is the main recipe function for the sph_zpl_star_center_pol recipe
 (polarimetric mode). The error  code returned is always a cpl error code
 (to allow maximal compatibility with  esorex, gasgano, etc.) even if during
 recipe execution an error in the SPHERE API is the cause. In this case
 (and if the underlying error is not a cpl error)  the cpl error code is set
 to the cpl_error_code that matches the failure reason best.
 The error from the SPHERE API is still written in the log as usual
 with the more informative and accurate sph_error_code.

 */
/*----------------------------------------------------------------------------*/

cpl_error_code sph_zpl_star_center_pol_run(sph_zpl_star_center_pol* self) {
    sph_zpl_star_center_pol_cunit* cunit = NULL;
    cpl_error_code recipe_error = CPL_ERROR_NONE;

    SPH_INFO_MSG("Starting sph_zpl_star_center_pol_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return cpl_error_get_code();
    }

    if (self->preproc_frames_cam1 == NULL && self->preproc_frames_cam2 == NULL && self->preproc_frames == NULL) {

        if (self->rawframes) {
            self->preproc_frames_cam1 = cpl_frameset_new();
            self->preproc_frames_cam2 = cpl_frameset_new();
            SPH_INFO_MSG("Start pre-processing...")
            //recipe_error = sph_zpl_master_dark_preproc( self );
            recipe_error = sph_zpl_common_preproc_recipe(self->rawframes,
                    self->outfilename_cam1, self->outfilename_cam2,
                    self->preproc_frames_cam1, self->preproc_frames_cam2);
        } else {
            SPH_ERR(
                    "There are neither raw frames nor pre-processed ones! Please verify that the tags are set correctly!");
            return sph_error_get_last_code();
        }
        if (recipe_error) {
            SPH_ERR("Pre-processing step is failed.");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }


        if ( sph_zpl_utils_check_format( self->rawframes ) == 1) {
            SPH_INFO_MSG("New style format of the raw data...")
            if ( cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 ) ) {
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam2 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                self->preproc_frames = cpl_frameset_duplicate( self->preproc_frames_cam1 );
                cpl_frameset_delete( self-> preproc_frames_cam1 ); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete( self-> preproc_frames_cam2 ); self->preproc_frames_cam2 = NULL;
            } else if ( !cpl_frameset_is_empty( self->preproc_frames_cam1 ) && !cpl_frameset_is_empty( self->preproc_frames_cam2 )){
                SPH_WARNING("Rawdata set contains fits files from camera-1 and camera-2. Please, make sure that you provide calibrations for both cameras."
                        "The products will be created individually for each camera (off-line pipeline extended support!")
            } else {
                SPH_ERR("Pre-processing step is failed.");
                cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
                cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
                return (int) recipe_error;
            }
        } else if ( sph_zpl_utils_check_format( self->rawframes ) == 0) {
            SPH_INFO_MSG("Old style format of the raw data...")
        } else {
            SPH_ERR("Not supported raw data format or new format and old format are mixed in the input dataset. "
                    "Please, make sure that you use either old or new style of the rawdata format! Stop recipe...");
            cpl_frameset_delete(self->preproc_frames_cam1); self->preproc_frames_cam1 = NULL;
            cpl_frameset_delete(self->preproc_frames_cam2); self->preproc_frames_cam2 = NULL;
            return (int) recipe_error;
        }
    }
    sph_error_reset();
    SPH_RAISE_CPL_RESET; //reset of the error system after pre-processing

    SPH_INFO_MSG("*********** [AFTER PREPROC] ***********");
    //reduce data for the camera-1
    //create cunit structure
    if (self->preproc_frames_cam1) {
        SPH_INFO_MSG("Create cunit structures for the camera-1...");

        SPH_INFO_MSG("Camera-1...")
        cunit = _sph_zpl_star_center_pol_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID);
        if (cunit) {

            recipe_error = _sph_zpl_star_center_pol_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-1 is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit->star_center_di)
                    self->star_center_di_cam1 = sph_double_image_duplicate(
                            cunit->star_center_di );
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit structure for the camera-1 is failed...trying for the camera-2")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-1 -> nothing to reduce..trying for the camera-2");
    }

    if (cunit) {
        _sph_zpl_star_center_pol_cunit_delete(cunit);
        cunit = NULL;
    }

    //reset an error log system to reduce data for camera-2
    sph_error_reset();
    SPH_RAISE_CPL_RESET;

    //reduce data for the camera-2
    //create cunit structure
    if (self->preproc_frames_cam2) {
        SPH_INFO_MSG("Create cunit structure for the camera-2...");

        cunit = _sph_zpl_star_center_pol_cunit_create(self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID);

        if ( cunit ) {
            recipe_error = _sph_zpl_star_center_pol_cunit_run( cunit );
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the camera-2 is is failed!");
            } else {
                //put the output product in the main self structure
                if ( cunit->star_center_di )
                    self->star_center_di_cam2 = sph_double_image_duplicate( cunit->star_center_di );
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit structure for the camera-2 is failed...")
        }

    } else {
        SPH_WARNING(
                "No pre-processed data found for the camera-2 -> nothing to reduce.");
    }

    if ( cunit ) {
        _sph_zpl_star_center_pol_cunit_delete( cunit );
        cunit = NULL;
    }


    //reset the error log system to avoid a recipe abort in the case if some products were not created
    //for the new style format data (indifferent to the cameras))
    sph_error_reset();
    SPH_RAISE_CPL_RESET;

    if (self->preproc_frames) {
        SPH_INFO_MSG("Create cunit structure for the new style format...");

        SPH_INFO_MSG("Indifferent camera stokes ...")
        cunit = _sph_zpl_star_center_pol_cunit_create( self,
                SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID );

        if (cunit) {

            recipe_error = _sph_zpl_star_center_pol_cunit_run(cunit);
            if (recipe_error != CPL_ERROR_NONE) {
                SPH_ERR(
                        "reduction for the newstyle pre-processed data is failed!");
            } else {
                //put the output product in the main self structure
                if (cunit -> star_center_di )
                    self->star_center_di = sph_double_image_duplicate( cunit->star_center_di );
            }
        } else {
            SPH_WARNING(
                    "Creation of the cunit structure for the new style format  is failed...")
        }
    } else {
        SPH_WARNING(
                "No pre-processed data found for the new style format -> nothing to reduce.");
    }

    if ( cunit ) {
        _sph_zpl_star_center_pol_cunit_delete( cunit );
        cunit = NULL;
    }
    //reset an error log system
    sph_error_reset();
    SPH_RAISE_CPL_RESET;

    //post-condistions
    if (self->star_center_di_cam1 == NULL && self->star_center_di_cam2 == NULL &&
         self->star_center_di ) {
        SPH_ERR(" No reduced frame outputs have been created !!!");
    }

    if (!self->keep_intermediate && self->rawframes != NULL) {
        SPH_INFO_MSG("Unliking intermediate data!");
        if (self->preproc_frames_cam1) sph_utils_frames_unlink(self->preproc_frames_cam1);
        if (self->preproc_frames_cam2) sph_utils_frames_unlink(self->preproc_frames_cam2);
        if (self->preproc_frames) sph_utils_frames_unlink(self->preproc_frames);
    }

    SPH_INFO_MSG("sph_zpl_star_center_pol_run...End");
    SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
}

sph_zpl_star_center_pol_cunit* _sph_zpl_star_center_pol_cunit_create(
        sph_zpl_star_center_pol* self, int camera_id ) {
    sph_zpl_star_center_pol_cunit* result = NULL;

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    if (self == NULL) {
        sph_error_raise(CPL_ERROR_NULL_INPUT, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Null input pointer.");
        return NULL;
    }

    result = cpl_calloc(1, sizeof(sph_zpl_star_center_pol_cunit));

    if (result == NULL) {
        SPH_ERR( "Could not allocate the structure for run unit.");
        return result;
    }

    //copy and duplicate some elements of the initial "sph_zpl_star_center_pol" structure into the cunit structure
    result->inframes = self->inframes;
    result->inparams = self->inparams;
    result->framecomb_parameterlist = self->framecomb_parameterlist;
    result->subtract_overscan = self->subtract_overscan;
    result->save_interprod = self->save_interprod;
    result->keep_intermediate = self->keep_intermediate;
    result->coll_alg = self->coll_alg;
    result->sigma = self->sigma;
    result->unsharp_window = self->unsharp_window;

    result->rawframes = self->rawframes;

    if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA1_ID) {

        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam1);

       result->star_center_outfilename = cpl_strdup(
                   self->star_center_outfilename_cam1);
            result->eso_pro_catg = cpl_strdup(
                    SPH_ZPL_TAG_STAR_CENTER_POL_CALIB_CAM1); //TAG for the double image star center product CAM1

        if (self->master_bias_frame_cam1) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam1);
        }
        if (self->master_dark_frame_cam1) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam1);
        }
        if (self->intensity_flat_frame_cam1) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame_cam1);
        }
        if (self->intensity_flat_frame_master_cam1) {
            result->intensity_flat_frame_master = cpl_frame_duplicate(
                    self->intensity_flat_frame_master_cam1);
        }
        if (self->polarization_flat_frame_cam1) {
            result->polarization_flat_frame = cpl_frame_duplicate(
                    self->polarization_flat_frame_cam1);
        }
        if (self->modem_efficiency_frame_cam1) {
            result->modem_efficiency_frame = cpl_frame_duplicate(
                    self->modem_efficiency_frame_cam1);
        }

    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA2_ID) {

        result->preproc_frames = cpl_frameset_duplicate(
                self->preproc_frames_cam2);

        result->star_center_outfilename = cpl_strdup(
                    self->star_center_outfilename_cam2);
             result->eso_pro_catg = cpl_strdup(
                     SPH_ZPL_TAG_STAR_CENTER_POL_CALIB_CAM2); //TAG for the double image star center product CAM1


        if (self->master_bias_frame_cam2) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame_cam2);
        }
        if (self->master_dark_frame_cam2) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame_cam2);
        }
        if (self->intensity_flat_frame_cam2) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame_cam2);
        }
        if (self->intensity_flat_frame_master_cam2) {
            result->intensity_flat_frame_master = cpl_frame_duplicate(
                    self->intensity_flat_frame_master_cam2);
        }
        if (self->polarization_flat_frame_cam2) {
            result->polarization_flat_frame = cpl_frame_duplicate(
                    self->polarization_flat_frame_cam2);
        }
        if (self->modem_efficiency_frame_cam2) {
            result->modem_efficiency_frame = cpl_frame_duplicate(
                    self->modem_efficiency_frame_cam2);
        }

    } else if (camera_id == SPH_ZPL_KEYWORD_VALUE_CAMERA_INDEFERENT_ID) {
        result->preproc_frames = cpl_frameset_duplicate(
                        self->preproc_frames);
        result->star_center_outfilename = cpl_strdup(
                    self->star_center_outfilename);
        result->eso_pro_catg = cpl_strdup(
                     SPH_ZPL_TAG_STAR_CENTER_POL_CALIB); //TAG for the double image star center product

        if (self->master_bias_frame) {
            result->master_bias_frame = cpl_frame_duplicate(
                    self->master_bias_frame);
        }
        if (self->master_dark_frame) {
            result->master_dark_frame = cpl_frame_duplicate(
                    self->master_dark_frame);
        }
        if (self->intensity_flat_frame) {
            result->intensity_flat_frame = cpl_frame_duplicate(
                    self->intensity_flat_frame);
        }
        if (self->intensity_flat_frame_master) {
            result->intensity_flat_frame_master = cpl_frame_duplicate(
                    self->intensity_flat_frame_master);
        }
        if (self->polarization_flat_frame) {
            result->polarization_flat_frame = cpl_frame_duplicate(
                    self->polarization_flat_frame);
        }
        if (self->modem_efficiency_frame) {
            result->modem_efficiency_frame = cpl_frame_duplicate(
                    self->modem_efficiency_frame);
        }

    } else {
        sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                SPH_ERROR_ERROR, "Wrong camera id = %d (it must be 1 or 2 for oldstyle format, 0 for newstyle format)",
                camera_id);
        _sph_zpl_star_center_pol_cunit_delete(result);
        result = NULL;
    }

    if (sph_error_get_last_code() != CPL_ERROR_NONE) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "Post condition error check shows the following error %d, please, check ",
                sph_error_get_last_code());
        _sph_zpl_star_center_pol_cunit_delete(result);
        result = NULL;
    }

    return result;

}

void _sph_zpl_star_center_pol_cunit_delete(sph_zpl_star_center_pol_cunit* cunit) {

    if (cunit != NULL) {
        cpl_frameset_delete(cunit->preproc_frames);
        sph_double_image_delete(cunit->star_center_di);
        cpl_free(cunit->star_center_outfilename);

        cpl_free(cunit->eso_pro_catg);

        cpl_frame_delete(cunit->master_bias_frame);
        cpl_frame_delete(cunit->master_dark_frame);
        cpl_frame_delete(cunit->intensity_flat_frame);
        cpl_frame_delete(cunit->intensity_flat_frame_master);
        cpl_frame_delete(cunit->polarization_flat_frame);
        cpl_frame_delete(cunit->modem_efficiency_frame);
        cpl_frameset_delete(cunit->current_raw_frameset);
        sph_polygon_free_all();
        cpl_free(cunit);
    }
}

static void sph_zpl_star_center_pol_delete__(sph_quad_image* master_bias_quadimage,
        sph_quad_image* master_dark_quadimage,
        sph_quad_image* intensity_flat_quadimage,
        sph_master_frame* intensity_flat_masterframe,
        sph_double_image* polarization_flat_doubleimage,
        sph_master_frame* modem_efficiency_masterframe) {
    if (master_bias_quadimage) {
        sph_quad_image_delete(master_bias_quadimage);

    }
    if (master_dark_quadimage) {
        sph_quad_image_delete(master_dark_quadimage);

    }
    if (intensity_flat_quadimage) {
        sph_quad_image_delete(intensity_flat_quadimage);

    }
    if (intensity_flat_masterframe) {
        sph_master_frame_delete(intensity_flat_masterframe);

    }
    if (polarization_flat_doubleimage) {
        sph_double_image_delete(polarization_flat_doubleimage);

    }
    if (modem_efficiency_masterframe) {
        sph_master_frame_delete(modem_efficiency_masterframe);

    }
}

static cpl_frame* sph_zpl_star_center_pol_calibrate_transform_frame__(
        const char* filename, cpl_frameset* inframes, sph_fctable* fctable,
        sph_quad_image* master_bias_quadimage,
        sph_quad_image* master_dark_quadimage,
        sph_quad_image* intensity_flat_quadimage,
        sph_master_frame* intensity_flat_masterframe,
        sph_double_image* polarization_flat_doubleimage,
        sph_master_frame* modem_efficiency_masterframe ) {

    sph_zpl_exposure* zplexp = NULL;
    sph_cube* dicube = NULL;
    cpl_frame* result = NULL;

    int iplane = 0;
    int rerr = CPL_ERROR_NONE;
    sph_transform* transform = NULL;

    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(filename, CPL_ERROR_NULL_INPUT, NULL);
    cpl_ensure(fctable, CPL_ERROR_NULL_INPUT, NULL);

    transform = sph_transform_new_default();
    if (!transform) {
        SPH_ERR("sph_transform_new_dafault() constructor returns NULL...");
        return NULL;
    }
    dicube = sph_cube_new(filename);
    if (!dicube) {
        SPH_ERR("sph_cube_new constructor returns NULL...");
        sph_transform_delete(transform);
        return NULL;
    }

    zplexp = sph_common_science_get_next_zpl_exposure(inframes);
    if (!zplexp) {
        SPH_ERR(
                "no zplexp can be loaded, sph_common_science_get_next_zpl_exposure returns NULL...");
        sph_transform_delete(transform);
        sph_cube_delete(dicube);
        return NULL;
    }

    while (zplexp) {
        sph_quad_image* curquadimage = NULL;
        sph_double_image* stokes = NULL;

        curquadimage = sph_quad_image_new_from_zpl_exposures(zplexp);
        if (!curquadimage) {
            SPH_ERR(
                    "can't create quad image from the zplexp, sph_quad_image_new_from_zpl_exposures constructor returns NULL...");
            sph_transform_delete(transform);
            sph_cube_delete(dicube);
            sph_zpl_exposure_delete(zplexp);
            return NULL;
        }

        if (master_bias_quadimage) {
            rerr = sph_quad_image_subtract_quad_image(curquadimage,
                    master_bias_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting master bias:\n"
                                "cpl error code is: %d", cpl_error_get_code());

                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Subtracted master bias.")
        }

        if (master_dark_quadimage) {
            rerr = sph_zpl_subtract_dark_quad_image_scaled(curquadimage,
                    master_dark_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by subtracting dark:\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Subtracted master dark.")
        }

        if (intensity_flat_masterframe) {
            rerr = sph_quad_image_divide_master_frame(curquadimage,
                    intensity_flat_masterframe);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(
                        SPH_ERROR_GENERAL,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying (dividing) intensity flat (master frame):\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Applied intensity flat field (master frame).")
        } else if (intensity_flat_quadimage) {
            rerr = sph_quad_image_divide_quad_image(curquadimage,
                    intensity_flat_quadimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(
                        SPH_ERROR_GENERAL,
                        __FILE__,
                        __func__,
                        __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying (dividing) intensity flat (quad image):\n"
                                "cpl error code is: %d", cpl_error_get_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                return NULL;
            }
            SPH_INFO_MSG("Applied intensity flat field (quad image).")

        }

        stokes =
                sph_zpl_utils_calculate_stokes_param_double_image_from_quad_image(
                        curquadimage);
        if (!stokes) {
            SPH_ERR("stokes parameters is NULL");
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            return NULL;
        }SPH_INFO_MSG("Calculated stokes.");

        if (polarization_flat_doubleimage) {
            rerr = sph_zpl_utils_apply_polarization_flat(stokes,
                    polarization_flat_doubleimage);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying polarization flat:\n"
                                "sph error code is: %d",
                        sph_error_get_last_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }SPH_INFO_MSG("Applied polarization flat.");

        }

        if (modem_efficiency_masterframe) {
            rerr = sph_zpl_utils_apply_modem_efficiency(stokes,
                    modem_efficiency_masterframe);
            if (rerr != CPL_ERROR_NONE) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "Error is raised by applying modem efficiency:\n"
                                "sph error code is: %d",
                        sph_error_get_last_code());
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }SPH_INFO_MSG("Applied modem effieciency.");
        }

        //quality check (calculate statistical moments for the basic polarization)
        rerr = sph_double_image_quality_check(stokes);
        if (rerr) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;
        }

        //if stokes double image is not squared expand it by factor 2 in y direction
        if (!sph_double_image_is_squared(stokes)) {
            SPH_INFO_MSG(
                    "double image is not squared, it will be linear interpolated in y-direction, "
                    "the total flux will be preserved (a small flux lost is due to the border lines)!")
            rerr = sph_double_image_interpolate_y_zpl_polarimetry( stokes, CPL_TRUE );
            if (rerr) {
                sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                        SPH_ERROR_ERROR,
                        "sph_doble_image_interpolate_y returns cpl error.\n"
                                "cpl error code is: %d", rerr);
                sph_zpl_exposure_delete(zplexp);
                sph_quad_image_delete(curquadimage);
                sph_cube_delete(dicube);
                sph_transform_delete(transform);
                sph_double_image_delete(stokes);
                return NULL;
            }
        }

        //check if it is squared
        if (!sph_double_image_is_squared(stokes)) {
            SPH_ERR("doubleimage is still not squared, "
                    "check the size of the input data...stop processing.\n");

            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;

        }

        SPH_INFO_MSG("sph_transform_apply_to_doubleimage...")
        rerr = sph_transform_apply_to_doubleimage(transform, stokes, NULL,
                sph_fctable_get_centre_x_element(fctable, iplane),
                sph_fctable_get_centre_y_element(fctable, iplane),
                sph_fctable_get_angle(fctable, iplane), 1.0);

        if (rerr) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_transform_apply_to_doubleimage function returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;
        }

        SPH_INFO_MSG(
                "append the transformed stokes double image to the cube...")
        rerr = sph_cube_append_double_image(dicube, stokes, 1);
        if (rerr) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "can't append the transformed stokes double image to the cube: "
                            "sph_cube_append_double_image function returns cpl error.\n"
                            "cpl error code is: %d", rerr);
            sph_zpl_exposure_delete(zplexp);
            sph_quad_image_delete(curquadimage);
            sph_cube_delete(dicube);
            sph_transform_delete(transform);
            sph_double_image_delete(stokes);
            return NULL;
        }

        sph_double_image_delete(stokes);
        sph_quad_image_delete(curquadimage);
        sph_zpl_exposure_delete(zplexp);
        zplexp = sph_common_science_get_next_zpl_exposure(inframes);
        iplane++;
    }

    rerr = sph_cube_finalise_file(dicube->filename);
    if (rerr) {
        sph_error_raise(
                SPH_ERROR_GENERAL,
                __FILE__,
                __func__,
                __LINE__,
                SPH_ERROR_ERROR,
                "can't finalize output cube for the filename %s: sph_cube_finalise_file function returns cpl error.\n"
                        "cpl error code is: %d", dicube->filename, rerr);
        sph_cube_delete(dicube);
        sph_transform_delete(transform);
        return NULL;
    }

    sph_cube_delete(dicube);

    result = cpl_frame_new();
    cpl_frame_set_filename(result, filename);
    cpl_frame_set_tag(result, SPH_ZPL_TAG_STAR_CENTER_POL_CALIB_TMP);
    sph_transform_delete(transform);

    return result;

}

static cpl_frameset* sph_zpl_star_center_pol_calibrate_frames__(
        sph_zpl_star_center_pol_cunit* self, cpl_frameset* inframes,
        sph_quad_image* master_bias_quadimage,
        sph_quad_image* master_dark_quadimage,
        sph_quad_image* intensity_flat_quadimage,
        sph_master_frame* intensity_flat_masterframe,
        sph_double_image* polarization_flat_doubleimage,
        sph_master_frame* modem_efficiency_masterframe) {

    cpl_frameset* results = NULL;
    const cpl_frame* curframe = NULL;

    cpl_ensure(inframes, CPL_ERROR_NULL_INPUT, NULL);

    SPH_INFO_MSG("Calibrate cubes...")

    results = cpl_frameset_new();
    curframe = cpl_frameset_get_first_const(inframes);
    //curframe = cpl_frameset_get_first( ovsc_subtract_frames );

    while (curframe) {
        cpl_frameset* curframes = NULL;
        cpl_frameset* ovsc_subtract_frames = NULL;
        cpl_frame* calframe = NULL;
        char outfilename[612];
        char outfilename_tmp[612];
        char* fctable_name = NULL;
        sph_fctable* fctable = NULL;
        curframes = cpl_frameset_new();
        cpl_frameset_insert(curframes, cpl_frame_duplicate(curframe));

        SPH_ERROR_RAISE_INFO(
                SPH_ERROR_WARNING,
                "The filename of the current frame: %s", cpl_frame_get_filename( curframe ));

        if (fctable == NULL) {
            SPH_INFO_MSG("Creating fctable for the current frame");
            fctable = sph_fctable_create_fctable(curframe, SPH_FCTAB_SINGLE);
            sph_fctable_fill_angle(fctable, 0.0, 0.0);

            if (self->save_interprod) {
                fctable_name = sph_fctable_construct_filename(curframe, 1);
                sph_fctable_save_ascii(fctable, fctable_name);
                cpl_free(fctable_name);
            }
        }

        if (fctable == NULL) {
            SPH_ERR("fctable has null pointer");
            cpl_frameset_delete(results);
            cpl_frameset_delete(curframes);
            return NULL;
        }

        SPH_INFO_MSG("Creating filename for the intermediate products...");
        (void)strcpy(outfilename, self->star_center_outfilename);
        sprintf(outfilename_tmp, "_%s",
                sph_filemanager_get_basename( cpl_frame_get_filename(curframe)));
        strcat(outfilename, outfilename_tmp);
        SPH_ERROR_RAISE_INFO( SPH_ERROR_INFO,
                "Intermediate outfilename = %s", outfilename)

        if (self->subtract_overscan) {
            SPH_INFO_MSG("Subtract overscans from rawframes...")
            ovsc_subtract_frames = sph_zpl_utils_subtract_overscans(curframes);
            if (!ovsc_subtract_frames) {
                SPH_ERROR_RAISE_WARNING(
                        SPH_ERROR_WARNING,
                        "Overscans could not be subtracted, trying to proceed further "
                        "without subtraction...")
                ovsc_subtract_frames = cpl_frameset_duplicate(curframes);
                sph_error_reset(); //reset is needed because sph_zpl_utils_subtract_overscans raises error when it returns NULL
            }
        } else {
            SPH_ERROR_RAISE_WARNING(
                    SPH_ERROR_WARNING,
                    "Overscans could not be subtracted, trying to proceed further "
                    "without subtraction...")
            ovsc_subtract_frames = cpl_frameset_duplicate(curframes);
        }

        calframe = sph_zpl_star_center_pol_calibrate_transform_frame__(
                (const char*) outfilename, ovsc_subtract_frames, fctable,
                master_bias_quadimage, master_dark_quadimage,
                intensity_flat_quadimage, intensity_flat_masterframe,
                polarization_flat_doubleimage, modem_efficiency_masterframe );
        if (!calframe) {
            SPH_ERR(
                    "can't calibrate the given frame(plane), the sph_zpl_star_center_pol_calibrate_transform_frame returns NULL");
            cpl_frameset_delete(results);
            cpl_frameset_delete(curframes);
            if (ovsc_subtract_frames) {
                if (!self->keep_intermediate)  sph_utils_frames_unlink(ovsc_subtract_frames);
                cpl_frameset_delete(ovsc_subtract_frames);
            }
            sph_fctable_delete(fctable);
            return NULL;
        }

        cpl_frameset_insert(results, calframe);
        if (!self->keep_intermediate) {
            sph_utils_frames_unlink(ovsc_subtract_frames);
        }
        cpl_frameset_delete(ovsc_subtract_frames);
        cpl_frameset_delete(curframes);
        sph_fctable_delete(fctable);

        //curframe = cpl_frameset_get_next(ovsc_subtract_frames);
        curframe = cpl_frameset_get_next_const(inframes);
    }

    SPH_ERROR_CHECK_STATE_ONERR_RETURN_NULL;
    return results;

}

cpl_error_code _sph_zpl_star_center_pol_cunit_run(sph_zpl_star_center_pol_cunit* self) {
    int recipe_error = CPL_ERROR_NONE;
    sph_quad_image* master_bias_quadimage = NULL;
    sph_quad_image* master_dark_quadimage = NULL;
    sph_quad_image* intensity_flat_quadimage = NULL;
    sph_master_frame* intensity_flat_masterframe = NULL;
    sph_double_image* polarization_flat_doubleimage = NULL;
    sph_master_frame* modem_efficiency_masterframe = NULL;
    sph_master_frame* imframe = NULL;
    sph_master_frame* pmframe = NULL;

    cpl_frameset* calframes = NULL;
    cpl_propertylist* pl_cent = NULL;

    SPH_INFO_MSG("Starting static _sph_zpl_star_center_pol_cunit_run...");
    SPH_ERROR_CHECK_STATE_ONERR_RETURN_ERRCODE

    if (!self->preproc_frames) {
        SPH_ERR("no preproc frames are set up!")
        return sph_error_get_last_code();
    }

    //if master bias exists load it
    if (self->master_bias_frame) {
        SPH_INFO_MSG("Loading master bias...\n");
        master_bias_quadimage = sph_quad_image_load(
                cpl_frame_get_filename(self->master_bias_frame), 0);
    }

    //if master dark exists load it
    if (self->master_dark_frame) {
        SPH_INFO_MSG("Loading master dark...\n");
        master_dark_quadimage = sph_quad_image_load(
                cpl_frame_get_filename(self->master_dark_frame), 0);
    }

    if (self->intensity_flat_frame_master) {
        //if intensity flat field master frame exists load
        SPH_INFO_MSG("Loading intensity flat field masterframe...\n");
        intensity_flat_masterframe = sph_master_frame_load_
            (self->intensity_flat_frame_master, 0);
    } else {
        //if intensity flat field quad image exists load
        if (self->intensity_flat_frame) {
            SPH_INFO_MSG("Loading intensity flat field quad image...\n");
            intensity_flat_quadimage = sph_quad_image_load(
                    cpl_frame_get_filename(self->intensity_flat_frame), 0);
        }
    }

    //if polarization flat field exists load it
    if (self->polarization_flat_frame) {
        SPH_INFO_MSG("Loading polarization flat field...\n");
        polarization_flat_doubleimage = sph_double_image_load(
                cpl_frame_get_filename(self->polarization_flat_frame), 0);
    }

    //if modulation/demodulation efficiency exists load it
    if (self->modem_efficiency_frame) {
        SPH_INFO_MSG("Loading modulation/demodulation efficiency...\n");
        modem_efficiency_masterframe = sph_master_frame_load_(
                self->modem_efficiency_frame, 0);
    }

    //calibrate and transform
    calframes = sph_zpl_star_center_pol_calibrate_frames__(self,
                self->preproc_frames, master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);

    if (!calframes) {
            SPH_ERR("Can't calibrate given frames...");
            sph_zpl_star_center_pol_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
     }

    //combine iframe and pframe
    imframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT, SPH_MASTER_FRAME_BPIX_EXT, -1);
    if (!imframe) {
            SPH_ERR(
                    "Can't combine calframes, imframe from sph_common_science_combine is NULL...");
            sph_zpl_star_center_pol_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            cpl_frameset_delete(calframes);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    pmframe = sph_common_science_combine(calframes, self->coll_alg,
                SPH_MASTER_FRAME_IMAGE_EXT + 4, SPH_MASTER_FRAME_BPIX_EXT + 4,
                -1);
    if (!pmframe) {
            SPH_ERR("Can't combine calframes, pmframe is NULL...");
            sph_zpl_star_center_pol_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            cpl_frameset_delete(calframes);
            sph_master_frame_delete(imframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    if ( !self->save_interprod ) {
            sph_utils_frames_unlink( calframes );
    }

    cpl_frameset_delete(calframes);

    //create double image from iframe and pframe master frames
    //input master frames are not duplicated, so don't delete them after creation of the double image
    self->star_center_di = sph_double_image_new_from_master_frames(imframe, pmframe);
    if (!self->star_center_di) {
            SPH_ERR(
                    "Can't create double image from iframe and pframe master frames, self->star_center_di is NULL...");
            sph_zpl_star_center_pol_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            sph_master_frame_delete(imframe);
            sph_master_frame_delete(pmframe);
            SPH_ERROR_CHECK_STATE_RETURN_ERRCODE
    }

    //calculate quality parameters and save it as a double image
    recipe_error = sph_double_image_quality_check(self->star_center_di);
    if (recipe_error) {
            sph_error_raise(SPH_ERROR_GENERAL, __FILE__, __func__, __LINE__,
                    SPH_ERROR_ERROR,
                    "sph_double_image_quality_check returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
            sph_zpl_star_center_pol_delete__(master_bias_quadimage,
                    master_dark_quadimage, intensity_flat_quadimage,
                    intensity_flat_masterframe, polarization_flat_doubleimage,
                    modem_efficiency_masterframe);
            return recipe_error;
    }


    SPH_INFO_MSG("Saving...");
    if (self->rawframes) {
        cpl_propertylist* pl = NULL;
            SPH_INFO_MSG(
                    "Save star center calibration (polarimetry)  as a dfs product: header info taken from the first raw frame.");
            pl = sph_zpl_utils_get_camera_header(self->preproc_frames, self->rawframes );

            pl_cent = sph_zpl_star_center_common_get_center_double_image(
                    self->star_center_di, self->sigma, self->unsharp_window);

            //get star center of the double image and appent it to the pl
            if (pl_cent) {
            	cpl_propertylist_append( pl, (const cpl_propertylist*) pl_cent);
            } else {
            	SPH_ERR("get star center of the double image failed!");

            	cpl_error_reset();
            }

            recipe_error = sph_double_image_save_dfs(self->star_center_di,
                    self->star_center_outfilename, self->inframes,
                    cpl_frameset_get_first(self->rawframes),
                    self->inparams,
                    self->eso_pro_catg, //SPH_ZPL_TAG_STAR_CENTER_POL_CALIB, _CAM1 or _CAM2
                    SPH_RECIPE_NAME_ZPL_STAR_CENTER_POL, SPH_PIPELINE_NAME_ZIMPOL,
                    pl);
            cpl_propertylist_delete(pl);
    } else if (self->preproc_frames) {
        cpl_propertylist* pl = NULL;
            SPH_INFO_MSG(
                    "Save star center calibration (polarimetry) as a dfs product: header info taken from the pre-processed frame.");
            pl = sph_zpl_star_center_common_get_center_double_image(
                     self->star_center_di, self->sigma, self->unsharp_window);

            recipe_error = sph_double_image_save_dfs(self->star_center_di,
                    self->star_center_outfilename, self->inframes,
                    cpl_frameset_get_first(self->preproc_frames),
                    self->inparams,
                    self->eso_pro_catg, //SPH_ZPL_TAG_STAR_CENTER_CALIB, _CAM1 or _CAM2
                    SPH_RECIPE_NAME_ZPL_STAR_CENTER_POL, SPH_PIPELINE_NAME_ZIMPOL,
                    pl);
            cpl_propertylist_delete( pl );

     } else {
            SPH_ERR("Neither raw frames nor pre-processed frames provided!");
            recipe_error = sph_error_get_last_code();
     }

     if (recipe_error) {
            sph_error_raise(
                    SPH_ERROR_GENERAL,
                    __FILE__,
                    __func__,
                    __LINE__,
                    SPH_ERROR_ERROR,
                    "Couldn't save star_center_di product: sph_double_image_save_dfs returns cpl error.\n"
                            "cpl error code is: %d", recipe_error);
     }

     sph_zpl_star_center_pol_delete__(master_bias_quadimage,
                master_dark_quadimage, intensity_flat_quadimage,
                intensity_flat_masterframe, polarization_flat_doubleimage,
                modem_efficiency_masterframe);

     SPH_INFO_MSG("sph_zpl_star_center_pol...End");
     SPH_ERROR_CHECK_STATE_RETURN_ERRCODE


}

/**@}*/

