/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2020 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * $Author: $
 * $Date: $
 * $Revision: $
 * $Name: $
 */

/*-----------------------------------------------------------------------------
 Includes
 -----------------------------------------------------------------------------*/
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "sph_zpl_subtract_dark_scaled.h"
#include "sph_common_keywords.h"

#include <cpl.h>
#include <math.h>

static const double MAX_RATIO = 10.0; // Maximum DIT ratio before a warning

/*----------------------------------------------------------------------------*/
/**
 * @defgroup sph_zpl_subtract_dark_scaled Dark Subtraction Recipe
 *
 * This module provides the implementation of the scaled dark subtraction
 * recipe
 *
 * @par Synopsis:
 * @code
 *   #include "sph_zpl_subtract_dark_scaled.h"
 * @endcode
 */
/*----------------------------------------------------------------------------*/
/**@{*/

/*----------------------------------------------------------------------------*/
/**
 * @brief Subtract a scaled dark frame from a master_frame
 * @param    self      The master image to subtract from
 * @param    mdark     The dark image to subtract
 * @return   CPL_ERROR_NONE or the relevant error code on error
 *
 * This function subtracts the dark frame mdark from self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_zpl_subtract_dark_double_image_scaled(sph_double_image* self,
                                            const sph_double_image* mdark) {
    double dit_dark = 0.0;
    double dit_self = 0.0;
    double dit_ratio = 1.0;
    cpl_boolean dits_ok = CPL_FALSE;
    sph_double_image* tmpdark = NULL;
    const sph_double_image* usedark;

    cpl_ensure_code(self != NULL,  CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mdark != NULL, CPL_ERROR_NULL_INPUT);

    if (self->properties == NULL) {
        cpl_msg_warning(cpl_func, "Raw-frame has no propertylist");
    } else if (mdark->properties == NULL) {
        cpl_msg_warning(cpl_func, "Dark-frame has no propertylist");
    } else if (!cpl_propertylist_has(self->properties,
                                     SPH_COMMON_KEYWORD_DIT)) {
        cpl_msg_warning(cpl_func, "Raw-frame has no " SPH_COMMON_KEYWORD_DIT
                        " card");
    } else if (!cpl_propertylist_has(mdark->properties,
                                     SPH_COMMON_KEYWORD_DIT)) {
        cpl_msg_warning(cpl_func, "Dark-frame has no " SPH_COMMON_KEYWORD_DIT
                        " card");
    } else {

        dit_self = cpl_propertylist_get_double(self->properties,
                                               SPH_COMMON_KEYWORD_DIT);
        dit_dark = cpl_propertylist_get_double(mdark->properties,
                                               SPH_COMMON_KEYWORD_DIT);

        if (dit_self <= 0.0 || dit_dark <= 0.0) {
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "Non-positive DIT ("
                                         SPH_COMMON_KEYWORD_DIT
                                         ") Raw-DIT=%f. dark-DIT=%f.",
                                         dit_dark, dit_self);
        }
        dits_ok = CPL_TRUE;

        if (dit_self != dit_dark) {
            /* Need to scale the dark */
            dit_ratio = dit_self / dit_dark;
            cpl_msg_info(cpl_func, "Multiplying dark by DIT-ratio: %g/%g "
                         "= %g", dit_self, dit_dark, dit_ratio);

            if (dit_ratio > MAX_RATIO) {
                cpl_msg_warning(cpl_func, "Excessive DIT-ratio of %f (> %f) "
                                "Dark DIT is %f and raw frame DIT is %f "
                                "Please consider a different dark with a "
                                "more appropriate DIT. "
                                "Subtracting scaled dark anyway.", dit_ratio,
                                MAX_RATIO, dit_dark, dit_self);
            }

            tmpdark = sph_double_image_duplicate(mdark);
            sph_double_image_multiply_double(tmpdark, dit_ratio);
        } else {
            cpl_msg_info(cpl_func, "Subtracting dark with matching DIT: %g",
                         dit_self);
        }
    }

    if (!dits_ok) {
        cpl_msg_warning(cpl_func, "Could not obtain DIT ("
                        SPH_COMMON_KEYWORD_DIT ") from dark and/or raw "
                        "frame, so a scaling of the dark is not possible. "
                        "Subtracting dark as it is");
    }

    usedark = tmpdark ? tmpdark : mdark;
    sph_double_image_subtract_double_image(self, usedark);
    
    sph_double_image_delete(tmpdark);

    return cpl_error_get_code() ?
        cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;

}

/*----------------------------------------------------------------------------*/
/**
 * @brief Subtract a scaled dark frame from a master_frame
 * @param    self      The master image to subtract from
 * @param    mdark     The dark image to subtract
 * @return   CPL_ERROR_NONE or the relevant error code on error
 *
 * This function subtracts the dark frame mdark from self. Self is changed
 * in place.
 *
 */
/*----------------------------------------------------------------------------*/
sph_error_code sph_zpl_subtract_dark_quad_image_scaled(sph_quad_image* self,
                                            const sph_quad_image* mdark) {
    double dit_dark = 0.0;
    double dit_self = 0.0;
    double dit_ratio = 1.0;
    cpl_boolean dits_ok = CPL_FALSE;
    sph_quad_image* tmpdark = NULL;
    const sph_quad_image* usedark;

    cpl_ensure_code(self != NULL,  CPL_ERROR_NULL_INPUT);
    cpl_ensure_code(mdark != NULL, CPL_ERROR_NULL_INPUT);

    if (self->properties == NULL) {
        cpl_msg_warning(cpl_func, "Raw-frame has no propertylist");
    } else if (mdark->properties == NULL) {
        cpl_msg_warning(cpl_func, "Dark-frame has no propertylist");
    } else if (!cpl_propertylist_has(self->properties,
                                     SPH_COMMON_KEYWORD_DIT)) {
        cpl_msg_warning(cpl_func, "Raw-frame has no " SPH_COMMON_KEYWORD_DIT
                        " card");
    } else if (!cpl_propertylist_has(mdark->properties,
                                     SPH_COMMON_KEYWORD_DIT)) {
        cpl_msg_warning(cpl_func, "Dark-frame has no " SPH_COMMON_KEYWORD_DIT
                        " card");
    } else {

        dit_self = cpl_propertylist_get_double(self->properties,
                                               SPH_COMMON_KEYWORD_DIT);
        dit_dark = cpl_propertylist_get_double(mdark->properties,
                                               SPH_COMMON_KEYWORD_DIT);

        if (dit_self <= 0.0 || dit_dark <= 0.0) {
            return cpl_error_set_message(cpl_func, CPL_ERROR_ILLEGAL_INPUT,
                                         "Non-positive DIT ("
                                         SPH_COMMON_KEYWORD_DIT
                                         ") Raw-DIT=%f. dark-DIT=%f.",
                                         dit_dark, dit_self);
        }
        dits_ok = CPL_TRUE;

        if (dit_self != dit_dark) {
            /* Need to scale the dark */
            dit_ratio = dit_self / dit_dark;
            cpl_msg_info(cpl_func, "Multiplying dark by DIT-ratio: %g/%g "
                         "= %g", dit_self, dit_dark, dit_ratio);

            if (dit_ratio > MAX_RATIO) {
                cpl_msg_warning(cpl_func, "Excessive DIT-ratio of %f (> %f) "
                                "Dark DIT is %f and raw frame DIT is %f "
                                "Please consider a different dark with a "
                                "more appropriate DIT. "
                                "Subtracting scaled dark anyway.", dit_ratio,
                                MAX_RATIO, dit_dark, dit_self);
            }

            tmpdark = sph_quad_image_duplicate(mdark);
            sph_quad_image_multiply_double(tmpdark, dit_ratio);
        } else {
            cpl_msg_info(cpl_func, "Subtracting dark with matching DIT: %g",
                         dit_self);
        }
    }

    if (!dits_ok) {
        cpl_msg_warning(cpl_func, "Could not obtain DIT ("
                        SPH_COMMON_KEYWORD_DIT ") from dark and/or raw "
                        "frame, so a scaling of the dark is not possible. "
                        "Subtracting dark as it is");
    }

    usedark = tmpdark ? tmpdark : mdark;
    sph_quad_image_subtract_quad_image(self, usedark);
    
    sph_quad_image_delete(tmpdark);

    return cpl_error_get_code() ?
        cpl_error_set_where(cpl_func) : CPL_ERROR_NONE;
}

/**@}*/
