/* $Id: $
 * This file is part of the SPHERE Pipeline
 * Copyright (C) 2007-2010 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*-------------------------------------------------------------------------*/
/**
   @file    sph_dictionary.c
   @author    N. Devillard
   @date    Sep 2007
   @version    $Revision: 1.27 $
   @brief    Implements a sph_dictionary for string variables.

   This module implements a simple sph_dictionary object, i.e. a list
   of string/string associations. This object is useful to store e.g.
   informations retrieved from a configuration file (ini files).
*/
/*--------------------------------------------------------------------------*/

/*
    $Id: sph_dictionary.c,v 1.27 2007-11-23 21:39:18 ndevilla Exp $
    $Revision: 1.27 $
*/
/*---------------------------------------------------------------------------
                                   Includes
 ---------------------------------------------------------------------------*/
#include "sph_dictionary.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/*-----------------------------------------------------------------------------
 Defines
 -----------------------------------------------------------------------------*/

/** Maximum value size for integers and doubles. */
#define MAXVALSZ    1024

/** Minimal allocated number of entries in a sph_dictionary */
#define DICTMINSZ    128

/*---------------------------------------------------------------------------
                              Private function declarations
 ---------------------------------------------------------------------------*/

static void * mem_double(void * ptr, int size) CPL_ATTR_ALLOC;

/*---------------------------------------------------------------------------
                              Private functions
 ---------------------------------------------------------------------------*/

/* Doubles the allocated size associated to a pointer */
/* 'size' is the current allocated size. */
static void * mem_double(void * ptr, int size)
{
    void * newptr = cpl_calloc(2*size, 1);

    if (newptr != NULL) {
        (void)memcpy(newptr, ptr, size);
        cpl_free(ptr);
    }
    return newptr;
}

/*---------------------------------------------------------------------------
                              Function codes
 ---------------------------------------------------------------------------*/
/*-------------------------------------------------------------------------*/
/**
  @brief    Compute the hash key for a string.
  @param    key        Character string to use for key.
  @return    1 unsigned int on at least 32 bits.

  This hash function has been taken from an Article in Dr Dobbs Journal.
  This is normally a collision-free function, distributing keys evenly.
  The key is stored anyway in the struct so that collision can be avoided
  by comparing the key itself in last resort.
 */
/*--------------------------------------------------------------------------*/
unsigned sph_dictionary_hash(const char * key)
{
    const int len = key ? strlen(key) : 0;
    unsigned  hash = 0;

    for (int i = 0; i < len; i++) {
        hash += (unsigned)key[i];
        hash += (hash<<10);
        hash ^= (hash>>6);
    }
    hash += (hash <<3);
    hash ^= (hash >>11);
    hash += (hash <<15);
    return hash;
}

/*-------------------------------------------------------------------------*/
/**
  @brief    Create a new sph_dictionary object.
  @param    size    Optional initial size of the sph_dictionary.
  @return    1 newly allocated sph_dictionary objet.

  This function allocates a new sph_dictionary object of given size and returns
  it. If you do not know in advance (roughly) the number of entries in the
  sph_dictionary, give size=0.
 */
/*--------------------------------------------------------------------------*/
sph_dictionary * sph_dictionary_new(int size)
{
    sph_dictionary    *    d ;

    /* If no size was specified, allocate space for DICTMINSZ */
    if (size<DICTMINSZ) size=DICTMINSZ ;

    d = (sph_dictionary *)cpl_calloc(1, sizeof(sph_dictionary));

    if (d != NULL) {
        d->size = size ;
        d->val  = (char **)cpl_calloc(size, sizeof(char*));
        d->key  = (char **)cpl_calloc(size, sizeof(char*));
        d->hash = (unsigned int *)cpl_calloc(size, sizeof(unsigned));
    }

    return d ;
}

/*-------------------------------------------------------------------------*/
/**
  @brief    Delete a sph_dictionary object
  @param    d    sph_dictionary object to deallocate.
  @return    void

  Deallocate a sph_dictionary object and all memory associated to it.
 */
/*--------------------------------------------------------------------------*/
void sph_dictionary_del(sph_dictionary * d)
{

    if (d != NULL) {
        for (int i=0 ; i < d->size ; i++) {
            cpl_free(d->key[i]);
            cpl_free(d->val[i]);
        }
        cpl_free(d->val);
        cpl_free(d->key);
        cpl_free(d->hash);
        cpl_free(d);
    }
}

/*-------------------------------------------------------------------------*/
/**
  @brief    Get a value from a sph_dictionary.
  @param    d        sph_dictionary object to search.
  @param    key        Key to look for in the sph_dictionary.
  @param    def     Default value to return if key not found.
  @return    1 pointer to internally allocated character string.

  This function locates a key in a sph_dictionary and returns a pointer to its
  value, or the passed 'def' pointer if no such key can be found in
  sph_dictionary. The returned character pointer points to data internal to the
  sph_dictionary object, you should not try to free it or modify it.
 */
/*--------------------------------------------------------------------------*/
const char * sph_dictionary_get(const sph_dictionary * d,
                                const char * key,
                                const char * def)
{
    unsigned    hash = sph_dictionary_hash(key);

    for (int i=0 ; i<d->size ; i++) {
        if (d->key[i]==NULL)
            continue ;
        /* Compare hash */
        if (hash==d->hash[i]) {
            /* Compare string, to avoid hash collisions */
            if (!strcmp(key, d->key[i])) {
                return d->val[i] ;
            }
        }
    }
    return def ;
}

/*-------------------------------------------------------------------------*/
/**
  @brief    Set a value in a sph_dictionary.
  @param    d       sph_dictionary object to modify.
  @param    key     Key to modify or add.
  @param    val     Value to add.
  @return   int     0 if Ok, anything else otherwise

  If the given key is found in the sph_dictionary, the associated value is
  replaced by the provided one. If the key cannot be found in the
  sph_dictionary, it is added to it.

  It is Ok to provide a NULL value for val, but NULL values for the sph_dictionary
  or the key are considered as errors: the function will return immediately
  in such a case.

  Notice that if you sph_dictionary_set a variable to NULL, a call to
  sph_dictionary_get will return a NULL value: the variable will be found, and
  its value (NULL) is returned. In other words, setting the variable
  content to NULL is equivalent to deleting the variable from the
  sph_dictionary. It is not possible (in this implementation) to have a key in
  the sph_dictionary without value.

  This function returns non-zero in case of failure.
 */
/*--------------------------------------------------------------------------*/
int sph_dictionary_set(sph_dictionary * d, const char * key, const char * val)
{
    int            i ;
    unsigned    hash ;

    if (d==NULL || key==NULL) return -1 ;
    
    /* Compute hash for this key */
    hash = sph_dictionary_hash(key) ;
    /* Find if value is already in sph_dictionary */
    if (d->n>0) {
        for (i=0 ; i<d->size ; i++) {
            if (d->key[i]==NULL)
                continue ;
            if (hash==d->hash[i]) { /* Same hash value */
                if (!strcmp(key, d->key[i])) {     /* Same key */
                    /* Found a value: modify and return */
                    if (d->val[i]!=NULL)
                        cpl_free(d->val[i]);
                    d->val[i] = cpl_strdup(val);
                    /* Value has been modified: return */
                    return 0 ;
                }
            }
        }
    }
    /* Add a new value */
    /* See if sph_dictionary needs to grow */
    if (d->n==d->size) {

        /* Reached maximum size: reallocate sph_dictionary */
        d->val  = (char **)mem_double(d->val,  d->size * sizeof(char*)) ;
        d->key  = (char **)mem_double(d->key,  d->size * sizeof(char*)) ;
        d->hash = (unsigned int *)mem_double(d->hash, d->size * sizeof(unsigned)) ;
        if ((d->val==NULL) || (d->key==NULL) || (d->hash==NULL)) {
            /* Cannot grow sph_dictionary */
            return -1 ;
        }
        /* Double size */
        d->size *= 2 ;
    }

    /* Insert key in the first empty slot */
    for (i=0 ; i<d->size ; i++) {
        if (d->key[i]==NULL) {
            /* Add key here */
            break ;
        }
    }
    /* Copy key */
    d->key[i]  = cpl_strdup(key);
    d->val[i]  = cpl_strdup(val);
    d->hash[i] = hash;
    d->n ++ ;
    return 0 ;
}

/*-------------------------------------------------------------------------*/
/**
  @brief    Delete a key in a sph_dictionary
  @param    d        sph_dictionary object to modify.
  @param    key        Key to remove.
  @return   void

  This function deletes a key in a sph_dictionary. Nothing is done if the
  key cannot be found.
 */
/*--------------------------------------------------------------------------*/
void sph_dictionary_unset(sph_dictionary * d, const char * key)
{
    unsigned    hash ;
    int            i ;

    if (key == NULL) {
        return;
    }

    hash = sph_dictionary_hash(key);
    for (i=0 ; i<d->size ; i++) {
        if (d->key[i]==NULL)
            continue ;
        /* Compare hash */
        if (hash==d->hash[i]) {
            /* Compare string, to avoid hash collisions */
            if (!strcmp(key, d->key[i])) {
                /* Found key */
                break ;
            }
        }
    }
    if (i>=d->size)
        /* Key not found */
        return ;

    cpl_free(d->key[i]);
    d->key[i] = NULL ;
    if (d->val[i]!=NULL) {
        cpl_free(d->val[i]);
        d->val[i] = NULL ;
    }
    d->hash[i] = 0 ;
    d->n -- ;
    return ;
}

/*-------------------------------------------------------------------------*/
/**
  @brief    Dump a sph_dictionary to an opened file pointer.
  @param    d    Dictionary to dump
  @param    f    Opened file pointer.
  @return    void

  Dumps a sph_dictionary onto an opened file pointer. Key pairs are printed out
  as @c [Key]=[Value], one per line. It is Ok to provide stdout or stderr as
  output file pointers.
 */
/*--------------------------------------------------------------------------*/
void sph_dictionary_dump(const sph_dictionary * d, FILE * out)
{
    int        i ;

    if (d==NULL || out==NULL) return ;
    if (d->n<1) {
        fprintf(out, "empty sph_dictionary\n");
        return ;
    }
    for (i=0 ; i<d->size ; i++) {
        if (d->key[i]) {
            fprintf(out, "%20s\t[%s]\n",
                    d->key[i],
                    d->val[i] ? d->val[i] : "UNDEF");
        }
    }
    return ;
}
